/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.fasten.vulnerabilitystatementsprocessor;

import eu.fasten.analyzer.vulnerabilitystatementsprocessor.VulnerabilityStatementsProcessor;
import eu.fasten.analyzer.vulnerabilitystatementsprocessor.db.MetadataUtility;
import eu.fasten.core.data.Constants;
import eu.fasten.core.data.vulnerability.Patch;
import eu.fasten.core.data.vulnerability.Purl;
import eu.fasten.core.data.vulnerability.Severity;
import eu.fasten.core.data.vulnerability.Vulnerability;
import org.jooq.DSLContext;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

import java.text.ParseException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Map;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.when;

public class VulnerabilityStatementsProcessorTest {
    private VulnerabilityStatementsProcessor.VulnerabilityStatementsKafkaPlugin kafkaPlugin;
    private DSLContext context;

    @BeforeEach
    public void setUp() {
        context = Mockito.mock(DSLContext.class);
        kafkaPlugin = new VulnerabilityStatementsProcessor.VulnerabilityStatementsKafkaPlugin();
        kafkaPlugin.setTopics(Collections.singletonList("fasten.vulnerability-statements-plugin.out"));
        kafkaPlugin.setDBConnection(new HashMap<>(Map.of(Constants.mvnForge, context)));
    }

    @Test
    public void injectIntoDBVulnerabilityStatement() {
        var v = new Vulnerability();
        v.setId("CVE-TEST");
        v.setDescription("Mock Vulnerability");
        v.setSeverity(Severity.HIGH);
        v.setScoreCVSS2(7.5);
        v.setScoreCVSS3(5.0);
        v.setPublishedDate("12-11-2019");
        v.setLastModifiedDate("08-31-2020");
        var lastVulnerablePurl = Purl.of("pkg:maven/org.testing/mock@1.0.1");
        v.addPurl(lastVulnerablePurl.toString());
        var patch = new Patch();
        patch.setPatchDate("04-04-2004");
        patch.setNewChangedLineNumbers(Collections.singletonList(33));
        patch.setOriginalChangedLineNumbers(Collections.singletonList(33));
        patch.setFileName("/src/main/java/net/GenericClass.java");
        v.setPatches(new HashSet<>(Collections.singletonList(patch)));

        var metadataUtility = Mockito.mock(MetadataUtility.class);
        kafkaPlugin.setMetadataUtility(metadataUtility);
        // MOCKITO WHEN
        var pkgIds = new HashMap<String, Long>(); pkgIds.put("org.testing:mock", 1L);
        when(metadataUtility.getPackageIds(Mockito.any(DSLContext.class), Mockito.any(LinkedHashSet.class))).thenReturn(pkgIds);
        when(metadataUtility.getPackageVersionId(Mockito.eq(lastVulnerablePurl), Mockito.any(DSLContext.class), Mockito.eq(1L))).thenReturn(1L);
        when(metadataUtility.getPackageVersionIds(v.getValidatedPurls(), context, pkgIds)).thenReturn(Collections.singletonList(1L));

        var fasten_uri = "/java/net/GenericClass.%3Cclinit%3E()%2Fjava.lang%2FVoidType";
        var fastenUris = new HashSet<>(Collections.singletonList(fasten_uri));
        var pkgVersionIds = new HashSet<>(Collections.singletonList(1L));
        when(metadataUtility.getFastenUrisForPatch(patch.getFileName(),
                patch.getNewChangedLineNumbers(),
                1L, context)).thenReturn(fastenUris);
        doNothing().when(metadataUtility).injectPackageVersionVulnerability(Mockito.any(Vulnerability.class), Mockito.anyLong(), Mockito.any(DSLContext.class));
        when(metadataUtility.getCallableIdsForFastenUri(fasten_uri, pkgVersionIds, context)).thenReturn(new HashSet<>(Collections.singletonList(42L)));
        when(metadataUtility.injectCallableVulnerability(Mockito.any(Vulnerability.class), Mockito.anyLong(), Mockito.any(DSLContext.class))).thenReturn(1);
        when(metadataUtility.getFullFastenUri(fasten_uri, 42L)).thenReturn("full_fasten_uri");

        // CALL
        kafkaPlugin.updateVulnerability(v, context);

        // MOCKITO VERIFY
        Mockito.verify(metadataUtility).getPackageIds(context, v.getValidatedPurls());
        Mockito.verify(metadataUtility).getPackageVersionId(lastVulnerablePurl, context, 1L);
        Mockito.verify(metadataUtility).getPackageVersionIds(v.getValidatedPurls(), context, pkgIds);
        Mockito.verify(metadataUtility).getFastenUrisForPatch(patch.getFileName(),
                patch.getNewChangedLineNumbers(),
                1L, context);
        Mockito.verify(metadataUtility).getCallableIdsForFastenUri(fasten_uri, pkgVersionIds, context);
        // VERIFY injection
        Mockito.verify(metadataUtility).injectPackageVersionVulnerability(v, 1L, context);
        Mockito.verify(metadataUtility).injectCallableVulnerability(v, 42L, context);
        Mockito.verify(metadataUtility).getFullFastenUri(fasten_uri, 42L);
    }

    @Test
    public void writePatchDateTest() throws ParseException {
        var v = new Vulnerability();
        var patch = new Patch();
        patch.setPatchDate("2015-05-15");
        patch.setNewChangedLineNumbers(Collections.singletonList(42));
        patch.setFileName("/src/main/java/net/GenericClass.java");

        var patch2 = new Patch();
        patch2.setPatchDate("2004-09-28");
        patch2.setNewChangedLineNumbers(Collections.singletonList(55));
        patch2.setFileName("/src/main/java/net/NiceClass.java");

        v.setPatches(new HashSet<>(Arrays.asList(patch, patch2)));
        v.updatePatchDate();

        assertEquals(v.getPatchDate(), "2015-05-15");
    }

    @Test
    public void cleanFileExtensions() {
        var v = new Vulnerability();
        var patch = new Patch();
        patch.setPatchDate("2015-05-15");
        patch.setNewChangedLineNumbers(Collections.singletonList(42));
        patch.setFileName("/src/main/java/net/GenericClass.java");

        var patch2 = new Patch();
        patch2.setPatchDate("2004-09-28");
        patch2.setNewChangedLineNumbers(Collections.singletonList(55));
        patch2.setFileName("/src/main/java/net/CHANGELOG.md");

        v.setPatches(new HashSet<>(Arrays.asList(patch, patch2)));

        v.filterUnsupportedPatches(VulnerabilityStatementsProcessor.VulnerabilityStatementsKafkaPlugin.extensions);

        assertEquals(new HashSet<>(Collections.singletonList(patch)), v.getPatches());
    }

    @Test
    public void splitSearch() {
        var v = new Vulnerability();
        v.setId("CVE-TEST");
        v.setDescription("Mock Vulnerability");
        v.setSeverity(Severity.HIGH);
        v.setScoreCVSS2(7.5);
        v.setScoreCVSS3(5.0);
        v.setPublishedDate("12-11-2019");
        v.setLastModifiedDate("08-31-2020");
        var lastVulnerablePurl = Purl.of("pkg:maven/org.testing/mock@1.0.1");
        v.addPurl(lastVulnerablePurl.toString());
        var firstPatchedPurl = Purl.of("pkg:maven/org.testing/mock@1.0.2");
        v.addFirstPatchedPurl(firstPatchedPurl.toString());
        var patch = new Patch();
        patch.setPatchDate("04-04-2004");
        patch.setNewChangedLineNumbers(Collections.singletonList(36));
        patch.setOriginalChangedLineNumbers(Collections.singletonList(33));
        patch.setFileName("/src/main/java/net/GenericClass.java");
        v.setPatches(new HashSet<>(Collections.singletonList(patch)));

        var metadataUtility = Mockito.mock(MetadataUtility.class);
        kafkaPlugin.setMetadataUtility(metadataUtility);

        // MOCKITO WHEN
        var pkgIds = new HashMap<String, Long>(); pkgIds.put("org.testing:mock", 1L);
        when(metadataUtility.getPackageIds(Mockito.any(DSLContext.class), Mockito.any(LinkedHashSet.class))).thenReturn(pkgIds);
        when(metadataUtility.getPackageVersionId(Mockito.eq(lastVulnerablePurl), Mockito.any(DSLContext.class), Mockito.eq(1L))).thenReturn(1L);
        when(metadataUtility.getPackageVersionId(Mockito.eq(firstPatchedPurl), Mockito.any(DSLContext.class), Mockito.eq(1L))).thenReturn(2L);
        when(metadataUtility.getPackageVersionIds(v.getValidatedPurls(), context, pkgIds)).thenReturn(Collections.singletonList(1L));
        when(metadataUtility.getPackageVersionIds(v.getValidatedFirstPatchedPurls(), context, pkgIds)).thenReturn(Collections.singletonList(2L));

        var fasten_uri = "/java/net/GenericClass.%3Cclinit%3E()%2Fjava.lang%2FVoidType";
        var fastenUris = new HashSet<>(Collections.singletonList(fasten_uri));
        var fastenUrisEmpty = new HashSet<String>();
        var pkgVersionVulnIds = new HashSet<>(Collections.singletonList(1L));
        when(metadataUtility.getFastenUrisForPatch(patch.getFileName(),
                patch.getOriginalChangedLineNumbers(),
                1L, context)).thenReturn(fastenUrisEmpty);
        when(metadataUtility.getFastenUrisForPatch(patch.getFileName(),
                patch.getNewChangedLineNumbers(),
                2L, context)).thenReturn(fastenUris);
        doNothing().when(metadataUtility).injectPackageVersionVulnerability(Mockito.any(Vulnerability.class), Mockito.anyLong(), Mockito.any(DSLContext.class));
        when(metadataUtility.getCallableIdsForFastenUri(fasten_uri, pkgVersionVulnIds, context)).thenReturn(new HashSet<>(Collections.singletonList(42L)));
        when(metadataUtility.injectCallableVulnerability(Mockito.any(Vulnerability.class), Mockito.anyLong(), Mockito.any(DSLContext.class))).thenReturn(1);
        when(metadataUtility.getFullFastenUri(fasten_uri, 42L)).thenReturn("full_fasten_uri");

        // CALL
        kafkaPlugin.updateVulnerability(v, context);

        // MOCKITO VERIFY
        Mockito.verify(metadataUtility).getPackageIds(context, v.getValidatedPurls());
        Mockito.verify(metadataUtility).getPackageVersionIds(v.getValidatedPurls(), context, pkgIds);
        Mockito.verify(metadataUtility).getPackageVersionId(lastVulnerablePurl, context, 1L);
        Mockito.verify(metadataUtility).getPackageVersionId(firstPatchedPurl, context, 1L);
        Mockito.verify(metadataUtility).getFastenUrisForPatch(patch.getFileName(),
                patch.getOriginalChangedLineNumbers(),
                1L, context);
        Mockito.verify(metadataUtility).getFastenUrisForPatch(patch.getFileName(),
                patch.getNewChangedLineNumbers(),
                2L, context);
        Mockito.verify(metadataUtility).getCallableIdsForFastenUri(fasten_uri, pkgVersionVulnIds, context);
        // VERIFY injection
        Mockito.verify(metadataUtility).injectPackageVersionVulnerability(v, 1L, context);
        Mockito.verify(metadataUtility).injectCallableVulnerability(v, 42L, context);
        Mockito.verify(metadataUtility).getFullFastenUri(fasten_uri, 42L);
    }
}
