/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.fasten.vulnerabilitystatementsprocessor;

import eu.fasten.core.data.vulnerability.Purl;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.maven.artifact.versioning.ComparableVersion;
import org.junit.jupiter.api.Test;

import java.util.ArrayList;

import static org.junit.jupiter.api.Assertions.assertEquals;

public class PurlTest {

    String purlDebian = "pkg:deb/debian/curl@7.50.3-1?arch=i386&distro=jessie";
    String purlPython = "pkg:pypi/django@1.11.1";
    String purlMaven  = "pkg:maven/org.apache.xmlgraphics/batik-anim@1.9.1";

    @Test
    public void testPurlDebian() {
        // Expected
        Purl ppE = new Purl();
        ppE.setType("deb");
        ppE.setNamespace("debian");
        ppE.setName("curl");
        ppE.setVersion(new ComparableVersion("7.50.3-1"));
        var qE = new ArrayList<Pair<String, String>>();
        qE.add(Pair.of("arch", "i386"));
        qE.add(Pair.of("distro", "jessie"));
        ppE.setQualifiers(qE);

        // Actual
        Purl pp = Purl.of(purlDebian);

        assertEquals(ppE, pp);
        assert pp != null;
        assertEquals(purlDebian, pp.toString());
        assertEquals("pkg:deb/debian/curl", pp.getBasePurl());
        assertEquals("curl", pp.getPackageName());
    }

    @Test
    public void testPurlPython() {
        // Expected
        Purl ppE = new Purl();
        ppE.setType("pypi");
        ppE.setNamespace("");
        ppE.setName("django");
        ppE.setVersion(new ComparableVersion("1.11.1"));

        // Actual
        var pp = Purl.of(purlPython);

        assertEquals(ppE, pp);
        assert pp != null;
        assertEquals(purlPython, pp.toString());
        assertEquals("pkg:pypi/django", pp.getBasePurl());
        assertEquals("django", pp.getPackageName());
    }

    @Test
    public void testPurlMaven() {
        // Expected
        Purl ppE = new Purl();
        ppE.setType("maven");
        ppE.setNamespace("org.apache.xmlgraphics");
        ppE.setName("batik-anim");
        ppE.setVersion(new ComparableVersion("1.9.1"));

        // Actual
        var pp = Purl.of(purlMaven);

        assertEquals(ppE, pp);
        assert pp != null;
        assertEquals(purlMaven, pp.toString());
        assertEquals("pkg:maven/org.apache.xmlgraphics/batik-anim", pp.getBasePurl());
        assertEquals("org.apache.xmlgraphics:batik-anim", pp.getPackageName());
    }

}
