/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.fasten.analyzer.vulnerabilitystatementsprocessor.db;

import com.google.gson.Gson;
import eu.fasten.analyzer.vulnerabilitystatementsprocessor.utils.Cache;
import eu.fasten.core.data.metadatadb.codegen.Keys;
import eu.fasten.core.data.metadatadb.codegen.tables.Callables;
import eu.fasten.core.data.metadatadb.codegen.tables.Files;
import eu.fasten.core.data.metadatadb.codegen.tables.ModuleContents;
import eu.fasten.core.data.metadatadb.codegen.tables.Modules;
import eu.fasten.core.data.metadatadb.codegen.tables.PackageVersions;
import eu.fasten.core.data.metadatadb.codegen.tables.Packages;
import eu.fasten.core.data.metadatadb.codegen.tables.Vulnerabilities;
import eu.fasten.core.data.metadatadb.codegen.tables.VulnerabilitiesPurls;
import eu.fasten.core.data.metadatadb.codegen.tables.VulnerabilitiesXCallables;
import eu.fasten.core.data.metadatadb.codegen.tables.VulnerabilitiesXPackageVersions;
import eu.fasten.core.data.vulnerability.Purl;
import eu.fasten.core.data.vulnerability.Vulnerability;
import org.jooq.DSLContext;
import org.jooq.JSONB;
import org.jooq.Record1;
import org.jooq.Record2;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import static eu.fasten.core.utils.FastenUriUtils.generateFullFastenUri;

public class MetadataUtility {

    private final Gson gson = new Gson();
    private final Cache cache = new Cache();
    private final Logger logger = LoggerFactory.getLogger(MetadataUtility.class.getName());

    public MetadataUtility() {
    }

    /**
     * Looks for Unique Base PURLs and find the corresponding package ID.
     *
     * @param context - DSL Context
     * @param purls   - Purls of the vulnerability
     * @return Map (packageName --> pkgId)
     */
    public HashMap<String, Long> getPackageIds(DSLContext context, LinkedHashSet<Purl> purls) {
        var pkgIds = new HashMap<String, Long>();

        var packageNames = purls.stream().map(Purl::getPackageName).
            distinct().collect(Collectors.toList());
        packageNames.forEach(packageName -> {
            var pkgId = getPackageId(packageName, context);
            if (pkgId > 0)
                pkgIds.put(packageName, pkgId);
            cache.pkgIdToName.put(pkgId, packageName);
        });
        return pkgIds;
    }

    /**
     * Get a list of packageVersion IDs from the PURLs in the vulnerability.
     * @param purls   - List of PURLS
     * @param context - DSL Context
     * @param pkgIds  - List of package version IDs
     */
    public List<Long> getPackageVersionIds(LinkedHashSet<Purl> purls,
                                           DSLContext context,
                                           HashMap<String, Long> pkgIds) {
        var pkgVersionIds = new ArrayList<Long>();
        for (var purl : purls) {
            var pkgId = pkgIds.get(purl.getPackageName());
            var pkgVersionId = getPackageVersionId(purl, context, pkgId);
            if (pkgVersionId > 0L) {
                pkgVersionIds.add(pkgVersionId);
                cache.pkgVsnToName.put(purl.getVersion(), cache.pkgIdToName.get(pkgId));
                cache.pkgNameToForge.put(cache.pkgIdToName.get(pkgId), purl.getType());
                cache.pkgVsnIdToVsn.put(pkgVersionId, purl.getVersion());
            }
        }
        return pkgVersionIds;
    }

    /**
     * Given a file and a package version id, looks for each hunk in the file.
     * The fasten_uri of each callable that is found at the linenumber indicated in the hunk
     * is then stored and returned.
     *
     * @param filename     - filename that was changed
     * @param lineNumbers  - lines to look for callables in the file
     * @param pkgVersionId - Long - ID of the package version
     * @return List of fastenURIs there fall under those lines
     */
    public HashSet<String> getFastenUrisForPatch(String filename,
                                                 List<Integer> lineNumbers,
                                                 Long pkgVersionId,
                                                 DSLContext context) {
        var fastenURIs = new HashSet<String>();

        // Get the fileID
        var fileId = getFileId(pkgVersionId, filename, context);
        if (fileId < 0) return fastenURIs;

        // There could be more moduleIds for the same file
        var moduleIds = getModuleIds(fileId, context);
        logger.info("Found " + moduleIds.size() + " module(s) corresponding to file with id " + fileId);

        for (Long moduleId : moduleIds) {
            for (Integer lineNumber : lineNumbers) {
                var fastenUrisInModule = getFastenUrisInModuleLines(moduleId, lineNumber, context);
                if (fastenUrisInModule.size() > 0) {
                    fastenURIs.addAll(fastenUrisInModule);
                }
            }
        }
        return fastenURIs;
    }

    /**
     * Retrieves fasten_uris in the module that include line_number.
     *
     * @param moduleId   - Long ID of the module
     * @param lineNumber - Indicates the line of the file where the change took place
     * @param context    - DSLContext
     * @return - List of fasten_uris
     */
    public List<String> getFastenUrisInModuleLines(Long moduleId, Integer lineNumber, DSLContext context) {
        var fasten_uris = context.select(Callables.CALLABLES.FASTEN_URI)
                .from(Callables.CALLABLES)
                .where(Callables.CALLABLES.MODULE_ID.equal(moduleId))
                .and(Callables.CALLABLES.LINE_START.le(lineNumber))
                .and(Callables.CALLABLES.LINE_END.ge(lineNumber))
                .fetch();

        return fasten_uris.stream().map(Record1::component1).collect(Collectors.toList());
    }

    /**
     * Retrieves the package_version_id given the purl of the package version.
     *
     * @param purlObj - purl object go bank the information
     * @return negative if it cannot be found
     */
    public Long getPackageVersionId(Purl purlObj, DSLContext context, Long pkgId) {
        logger.info("Looking for package_version_id of " + purlObj.toString());
        assert pkgId > 0L;

        var pkgVersionRecord = context.select(PackageVersions.PACKAGE_VERSIONS.ID)
                .from(PackageVersions.PACKAGE_VERSIONS)
                .where(PackageVersions.PACKAGE_VERSIONS.PACKAGE_ID.equal(pkgId))
                .and(PackageVersions.PACKAGE_VERSIONS.VERSION.equal(purlObj.getVersion().toString()))
                .fetchOne();
        return pkgVersionRecord != null ? pkgVersionRecord.component1() : -1L;
    }

    /**
     * Retrieve the fileId of the file that was patched.
     *
     * @param packageVersionId - Long pkg version ID
     * @param filepath         - path to the file
     * @return -1 if the file cannot be found
     */
    public Long getFileId(Long packageVersionId, String filepath, DSLContext context) {
        var fileRecords = context.select(Files.FILES.ID, Files.FILES.PATH)
                .from(Files.FILES)
                .where(Files.FILES.PACKAGE_VERSION_ID.equal(packageVersionId))
                .fetch();
        var file = fileRecords.stream()
                .filter(fr -> (!fr.component2().equals("") && filepath.contains(fr.component2())))
                .map(Record2::component1).collect(Collectors.toList());
        return file.size() > 0 ? file.get(0) : -1L;
    }

    /**
     * Gets the moduleId that corresponds to the file.
     *
     * @param fileId - Long fileId
     * @return list of module Ids
     */
    public List<Long> getModuleIds(Long fileId, DSLContext context) {
        var moduleRecords = context.select(ModuleContents.MODULE_CONTENTS.MODULE_ID)
                .from(ModuleContents.MODULE_CONTENTS)
                .where(ModuleContents.MODULE_CONTENTS.FILE_ID.equal(fileId))
                .fetch();
        return moduleRecords.stream().map(Record1::component1).collect(Collectors.toList());
    }

    /**
     * Returns package ID given PURL and CONTEXT (forge).
     *
     * @param pkgName - Package name
     * @param context - DSLContext
     * @return -1L if not existing
     */
    public Long getPackageId(String pkgName, DSLContext context) {
        var pkgRecord = context.select(Packages.PACKAGES.ID)
                .from(Packages.PACKAGES)
                .where(Packages.PACKAGES.PACKAGE_NAME.equal(pkgName))
                .fetchOne();
        return pkgRecord != null ? pkgRecord.component1() : -1L;
    }

    /**
     * Gets ids of callables with the fasten_uri, whose module belongs to a vulnerable package.
     *
     * @param fastenUri     - String
     * @param pkgVersionIds - List of Long vulnerable package versions
     * @param context       - DSL context
     * @return - List of ids of the callables
     */
    public HashSet<Long> getCallableIdsForFastenUri(String fastenUri, HashSet<Long> pkgVersionIds,
                                                 DSLContext context) {
        var callInfo = context.select(Callables.CALLABLES.ID, Modules.MODULES.PACKAGE_VERSION_ID)
                .from(Callables.CALLABLES)
                .join(Modules.MODULES).on(Modules.MODULES.ID.eq(Callables.CALLABLES.MODULE_ID))
                .where(Callables.CALLABLES.FASTEN_URI.equal(fastenUri))
                .and(Modules.MODULES.PACKAGE_VERSION_ID.in(pkgVersionIds))
                .fetch();

        var callIds = new HashSet<>(callInfo.map(Record2::value1));
//        callIds.addAll(callIdsSubProj);
        callInfo.forEach(r -> cache.callIdToPkgVsn.put(r.value1(), cache.pkgVsnIdToVsn.get(r.value2())));
        return callIds;
    }

    /**
     * Builds a full fasten uri using information in the Cache
     * @param partialFastenUri - partial fasten_uri of the callable
     * @param callId - callable ID - Long
     * @return String - full_fasten_uri
     */
    public String getFullFastenUri(String partialFastenUri, Long callId) {
        var pkgVersion = cache.callIdToPkgVsn.get(callId);
        var pkgName = cache.pkgVsnToName.get(pkgVersion);
        var pkgForge = cache.pkgNameToForge.get(pkgName);
        return generateFullFastenUri(pkgForge, pkgName, pkgVersion.toString(), partialFastenUri);
    }

    //////////////////////////
    //    DATABASE INJECT   //
    //////////////////////////

    /**
     * Injects information at package level.
     *
     * @param v            - Vulnerability Object
     * @param pkgVersionId - ID of the package version
     * @param context      - DSLContext to use to inject the data
     */
    public void injectPackageVersionVulnerability(Vulnerability v, Long pkgVersionId, DSLContext context) {
        logger.info("Injecting vulnerability " + v.getId() + " into PACKAGE VERSION with ID: " + pkgVersionId);
        // Step 1: Get metadata JSONObject of the pkgVersionId
        var metadata = getPackageVersionMetaData(pkgVersionId, context);

        // Step 2: Check if vulnerabilities exist, else create a map
        if (!metadata.has("vulnerabilities"))
            metadata.put("vulnerabilities", new JSONObject());

        // Step 3: Get the JSON of the vulnerability and put it in the metadata
        var vulnJson = gson.toJson(v.getPackageVersionMetadata());
        ((JSONObject) metadata.get("vulnerabilities")).put(v.getId(), new JSONObject(vulnJson));

        // Step 4: Update the value of the metadata in the DB
        updatePackageVersionMetaData(pkgVersionId, metadata, context);
    }

    /**
     * Injects information at callable level.
     *
     * @param v          - Vulnerability Object
     * @param callableId - ID of the callable
     * @param context    - DSLContext to use to inject the data
     * @return int       - result of update call
     */
    public int injectCallableVulnerability(Vulnerability v, Long callableId, DSLContext context) {
        logger.info("Injecting vulnerability " + v.getId() + " into CALLABLE with ID: " + callableId);
        // Step 1: Get metadata JSONObject of the callableId
        var metadata = getCallableMetaData(callableId, context);

        // Step 2: Check if vulnerabilities exist, else create a map
        if (!metadata.has("vulnerabilities"))
            metadata.put("vulnerabilities", new JSONObject());

        // Step 3: Get the JSON of the vulnerability and put it in the metadata
        var vulnJson = gson.toJson(v.getCallableMetadata());
        ((JSONObject) metadata.get("vulnerabilities")).put(v.getId(), new JSONObject(vulnJson));

        // Step 4: Update the value of the metadata in the DB
        return updateCallableMetaData(callableId, metadata, context);
    }

    /**
     * Inserts a vulnerability record into the 'vulnerabilities', updating existing records.
     *
     * @param v  The Vulnerability data object
     * @param context  The database context
     * @return ID of the new record
     */
    public Long insertVulnerability(Vulnerability v, DSLContext context) {
        var vJson = JSONB.valueOf(v.toJson());
        var resultRecord = context.insertInto(Vulnerabilities.VULNERABILITIES,
                Vulnerabilities.VULNERABILITIES.EXTERNAL_ID,
                Vulnerabilities.VULNERABILITIES.STATEMENT)
            .values(v.getId(), vJson)
            .onConflictOnConstraint(Keys.UNIQUE_VULNERABILITIES).doUpdate()
            .set(Vulnerabilities.VULNERABILITIES.STATEMENT, vJson)
            .returning(Vulnerabilities.VULNERABILITIES.ID).fetchOne();

        Long vId = null;
        if (resultRecord != null) {
            vId = resultRecord.getValue(Vulnerabilities.VULNERABILITIES.ID);

        }
        return vId;
    }

    /**
     * Inserts purls related to the vulnerability, ignoring existing records.
     *
     * @param vulnerabilityId The CVE or similar identifier for the vulnerability
     * @param purls  The purls of package-versions related to the vulnerability
     * @param context  The database context
     */
    public void insertPurls(Long vulnerabilityId, LinkedHashSet<Purl> purls, DSLContext context) {
        purls.forEach(p -> context.insertInto(VulnerabilitiesPurls.VULNERABILITIES_PURLS,
                        VulnerabilitiesPurls.VULNERABILITIES_PURLS.VULNERABILITY_ID,
                        VulnerabilitiesPurls.VULNERABILITIES_PURLS.PURL,
                        VulnerabilitiesPurls.VULNERABILITIES_PURLS.FORGE,
                        VulnerabilitiesPurls.VULNERABILITIES_PURLS.PACKAGE_NAME,
                        VulnerabilitiesPurls.VULNERABILITIES_PURLS.PACKAGE_VERSION
                )
                .values(vulnerabilityId, p.toString(), p.getEcosystem(), p.getPackageName(), p.getVersion().toString())
                .execute());
    }

    /**
     * Link vulnerability to package-versions
     *
     * @param vulnerabilityId The CVE or similar identifier for the vulnerability
     * @param packageVersionIds The package-version ids to link to
     * @param context  The database context
     */
    public void insertVulnerabilityToPackageVersions(Long vulnerabilityId, List<Long> packageVersionIds, DSLContext context) {
        packageVersionIds.forEach(pkgVerId ->
                context.insertInto(VulnerabilitiesXPackageVersions.VULNERABILITIES_X_PACKAGE_VERSIONS,
                        VulnerabilitiesXPackageVersions.VULNERABILITIES_X_PACKAGE_VERSIONS.VULNERABILITY_ID,
                        VulnerabilitiesXPackageVersions.VULNERABILITIES_X_PACKAGE_VERSIONS.PACKAGE_VERSION_ID)
                        .values(vulnerabilityId, pkgVerId)
                        .onConflictOnConstraint(Keys.UNIQUE_VULN_X_PKG_VER)
                        .doNothing()
                        .execute());
    }

    /**
     * Link vulnerability to callables
     *
     * @param vulnerabilityId The CVE or similar identifier for the vulnerability
     * @param callableIds The package-version ids to link to
     * @param context  The database context
     */
    public void insertVulnerabilityToCallables(Long vulnerabilityId, Set<Long> callableIds, DSLContext context) {
        callableIds.forEach(callableId ->
                context.insertInto(VulnerabilitiesXCallables.VULNERABILITIES_X_CALLABLES,
                        VulnerabilitiesXCallables.VULNERABILITIES_X_CALLABLES.VULNERABILITY_ID,
                        VulnerabilitiesXCallables.VULNERABILITIES_X_CALLABLES.CALLABLE_ID)
                        .values(vulnerabilityId, callableId)
                        .onConflictOnConstraint(Keys.UNIQUE_VULN_X_CALLABLE)
                        .doNothing()
                        .execute());
    }

    /**
     * Returns vulnerability ID given PURL components.
     *
     * @param forge - forge/ecosystem, eg. mvn, debian, PyPI
     * @param packageName - name of the package, eg. log4j:log4j
     * @param version - version the package, eg. 1.2.17
     * @return empty list if no vulnerabilties were found to match
     */
    public List<Long> getVulnerabilitiesForPurl(String forge, String packageName, String version, DSLContext context) {
        var pkgRecords = context.select(VulnerabilitiesPurls.VULNERABILITIES_PURLS.VULNERABILITY_ID)
                .from(VulnerabilitiesPurls.VULNERABILITIES_PURLS)
                .where(VulnerabilitiesPurls.VULNERABILITIES_PURLS.FORGE.equal(forge))
                .and(VulnerabilitiesPurls.VULNERABILITIES_PURLS.PACKAGE_NAME.equal(packageName))
                .and(VulnerabilitiesPurls.VULNERABILITIES_PURLS.PACKAGE_VERSION.equal(version))
                .fetchArray();
        return Arrays.stream(pkgRecords).map(Record1::component1).collect(Collectors.toList());
    }

    /**
     * Returns vulnerability statement given id
     *
     * @param vulnerabilityId - id of the vulnerability
     * @return Statement for the vulnerability (JSON as String)
     */
    public String getVulnerabilityStatement(Long vulnerabilityId, DSLContext context) {
        var pkgRecord = context.select(Vulnerabilities.VULNERABILITIES.STATEMENT)
                .from(Vulnerabilities.VULNERABILITIES)
                .where(Vulnerabilities.VULNERABILITIES.ID.equal(vulnerabilityId))
                .fetchOne();
        if (pkgRecord != null) {
            return String.valueOf(pkgRecord.component1());
        }
        else {
            return null;
        }
    }

    /**
     * Returns vulnerability statement given id
     *
     * @param externalId - external id of the vulnerability, eg. CVE-2019-1234
     * @return Id of the vulnerability in the DB
     */
    public Long getVulnerabilityId(String externalId, DSLContext context) {
        var pkgRecord = context.select(Vulnerabilities.VULNERABILITIES.ID)
                .from(Vulnerabilities.VULNERABILITIES)
                .where(Vulnerabilities.VULNERABILITIES.EXTERNAL_ID.equal(externalId))
                .fetchOne();
        if (pkgRecord != null) {
            return pkgRecord.component1();
        }
        else {
            return -1L;
        }
    }

    public void deleteVulnerabilityData(String externalVulnerabilityId, Long vulnerabilityId, DSLContext context) {
        context.deleteFrom(VulnerabilitiesPurls.VULNERABILITIES_PURLS)
                .where(VulnerabilitiesPurls.VULNERABILITIES_PURLS.VULNERABILITY_ID.equal(vulnerabilityId))
                .execute();

        deletePackageVersionVulnerabilityData(externalVulnerabilityId, vulnerabilityId, context);
        deleteCallableVulnerabilityData(externalVulnerabilityId, vulnerabilityId, context);

        context.deleteFrom(Vulnerabilities.VULNERABILITIES)
                .where(Vulnerabilities.VULNERABILITIES.ID.equal(vulnerabilityId))
                .and(Vulnerabilities.VULNERABILITIES.EXTERNAL_ID.equal(externalVulnerabilityId))
                .execute();
    }

    private void deletePackageVersionVulnerabilityData(String externalVulnerabilityId, Long vulnerabilityId, DSLContext context) {
        var packageVersionIds = context.select(VulnerabilitiesXPackageVersions.VULNERABILITIES_X_PACKAGE_VERSIONS.PACKAGE_VERSION_ID)
                .from(VulnerabilitiesXPackageVersions.VULNERABILITIES_X_PACKAGE_VERSIONS)
                .where(VulnerabilitiesXPackageVersions.VULNERABILITIES_X_PACKAGE_VERSIONS.VULNERABILITY_ID.equal(vulnerabilityId))
                .fetchArray();

        Arrays.stream(packageVersionIds).forEach(idRecord -> {
            var id = idRecord.component1();
            var metadata = getPackageVersionMetaData(id, context);
            if(metadata.has("vulnerabilities")) {
                var result = metadata.getJSONObject("vulnerabilities").remove(externalVulnerabilityId);
                if(result != null ) {
                    updatePackageVersionMetaData(id, metadata, context);
                }
            }
        });

        context.deleteFrom(VulnerabilitiesXPackageVersions.VULNERABILITIES_X_PACKAGE_VERSIONS)
                .where(VulnerabilitiesXPackageVersions.VULNERABILITIES_X_PACKAGE_VERSIONS.VULNERABILITY_ID.equal(vulnerabilityId))
                .execute();
    }

    private void deleteCallableVulnerabilityData(String externalVulnerabilityId, Long vulnerabilityId, DSLContext context) {
        var callableIds = context.select(VulnerabilitiesXCallables.VULNERABILITIES_X_CALLABLES.CALLABLE_ID)
                .from(VulnerabilitiesXCallables.VULNERABILITIES_X_CALLABLES)
                .where(VulnerabilitiesXCallables.VULNERABILITIES_X_CALLABLES.VULNERABILITY_ID.equal(vulnerabilityId))
                .fetchArray();

        Arrays.stream(callableIds).forEach(idRecord -> {
            var id = idRecord.component1();
            var metadata = getCallableMetaData(id, context);
            if(metadata.has("vulnerabilities")) {
                var result = metadata.getJSONObject("vulnerabilities").remove(externalVulnerabilityId);
                if(result != null) {
                    updateCallableMetaData(id, metadata, context);
                }
            }
        });

        context.deleteFrom(VulnerabilitiesXCallables.VULNERABILITIES_X_CALLABLES)
                .where(VulnerabilitiesXCallables.VULNERABILITIES_X_CALLABLES.VULNERABILITY_ID.equal(vulnerabilityId))
                .execute();
    }

    private JSONObject getPackageVersionMetaData(Long id, DSLContext context) {
        var metadataRecord = Objects.requireNonNull(context.select(PackageVersions.PACKAGE_VERSIONS.METADATA)
                        .from(PackageVersions.PACKAGE_VERSIONS)
                        .where(PackageVersions.PACKAGE_VERSIONS.ID.equal(id))
                        .fetchOne()).component1();

        return new JSONObject(metadataRecord.data());
    }

    private int updatePackageVersionMetaData(Long id, JSONObject newMetaData, DSLContext context) {
        return context.update(PackageVersions.PACKAGE_VERSIONS)
                .set(PackageVersions.PACKAGE_VERSIONS.METADATA, JSONB.valueOf(newMetaData.toString()))
                .where(PackageVersions.PACKAGE_VERSIONS.ID.equal(id))
                .execute();
    }

    private JSONObject getCallableMetaData(Long id, DSLContext context) {
        var metadataRecord = Objects.requireNonNull(context.select(Callables.CALLABLES.METADATA)
                        .from(Callables.CALLABLES)
                        .where(Callables.CALLABLES.ID.equal(id))
                        .fetchOne()).component1();

        return new JSONObject(metadataRecord.data());
    }

    private int updateCallableMetaData(Long id, JSONObject newMetaData, DSLContext context) {
        return context.update(Callables.CALLABLES)
                .set(Callables.CALLABLES.METADATA, JSONB.valueOf(newMetaData.toString()))
                .where(Callables.CALLABLES.ID.equal(id))
                .execute();
    }
}
