/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.fasten.analyzer.vulnerabilitystatementsprocessor;

import com.google.gson.Gson;
import eu.fasten.analyzer.vulnerabilitystatementsprocessor.db.MetadataUtility;
import eu.fasten.analyzer.vulnerabilitystatementsprocessor.utils.FastenApiClient;
import eu.fasten.core.data.vulnerability.Purl;
import eu.fasten.core.data.vulnerability.Vulnerability;
import eu.fasten.core.plugins.DBConnector;
import eu.fasten.core.plugins.KafkaPlugin;
import org.apache.maven.artifact.versioning.ComparableVersion;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.pf4j.Extension;
import org.pf4j.Plugin;
import org.pf4j.PluginWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import static eu.fasten.core.data.vulnerability.Vulnerability.sortPurls;

public class VulnerabilityStatementsProcessor extends Plugin {

    public VulnerabilityStatementsProcessor(PluginWrapper wrapper) {
        super(wrapper);
    }

    @Extension
    public static class VulnerabilityStatementsKafkaPlugin implements KafkaPlugin, DBConnector {
        private List<String> consumeTopics = new LinkedList<>(Collections.singletonList("fasten.Vulnerabilities.out"));
        private final String baseOutputPath = File.separator + "vulnerabilities" + File.separator + "processed_statements";
        private String outputPath;
        private static Map<String, DSLContext> contexts;
        private Exception pluginError = null;
        private Vulnerability lastProcessedVulnerability = null;
        private final Gson gson = new Gson();
        private MetadataUtility metadataUtility;
        public final static String[] extensions = new String[]{".java", ".py", ".pyw", ".c", ".cpp", ".h"};
        private final static Logger logger = LoggerFactory.getLogger("VulnerabilityStatementsKafkaPlugin");
        private FastenApiClient fastenApiClient = new FastenApiClient();
        private Vulnerability vulnerability = null;
        private Long vulnerabilityId = -1L;

        @Override
        public void setDBConnection(Map<String, DSLContext> dslContexts) {
            contexts = dslContexts;
        }

        @Override
        public Optional<List<String>> consumeTopic() {
            return Optional.of(consumeTopics);
        }

        @Override
        public void setTopics(List<String> consumeTopics) {
            this.consumeTopics = consumeTopics;
        }

        @Override
        public void consume(String record) {
            this.pluginError = null;
            var v = updateVulnerability(record);
            outputPath = baseOutputPath + File.separator + v.getId() + ".json";
            lastProcessedVulnerability = v;
        }

        @Override
        public Optional<String> produce() {
            if (lastProcessedVulnerability == null) {
                return Optional.empty();
            } else {
                return Optional.of(gson.toJson(lastProcessedVulnerability));
            }
        }

        @Override
        public String getOutputPath() {
            return outputPath;
        }

        @Override
        public String name() {
            return "Vulnerability Statements Processor";
        }

        @Override
        public String description() {
            return "Reads vulnerabilities from " + consumeTopics.toString() +
                    " and injects them in the database.";
        }

        @Override
        public String version() {
            return "0.0.1";
        }

        @Override
        public void start() {
        }

        @Override
        public void stop() {
        }

        @Override
        public Exception getPluginError() {
            return pluginError;
        }

        public void setPluginError(Exception throwable) {
            this.pluginError = throwable;
        }

        @Override
        public void freeResource() {
        }

        public void setMetadataUtility() {
            metadataUtility = new MetadataUtility();
        }

        public void setMetadataUtility(MetadataUtility metadataUtility) {
            this.metadataUtility = metadataUtility;
        }

        public void setFastenApiClient(FastenApiClient client) {
            this.fastenApiClient = client;
        }

        public Vulnerability updateVulnerability(String record) {
            var v = gson.fromJson(record, Vulnerability.class);
            logger.info("Read vulnerability statement " + v.getId() + " from " + consumeTopics.toString());
            contexts.keySet().forEach(ecosystem -> getDbContext(ecosystem).
                    transaction(configuration -> {
                        logger.info(v.getId() + ": Updating vulnerability in the database for ecosystem: " + "\"" + ecosystem + "\".");
                        var vulnerabilityForEcosystem = gson.fromJson(record, Vulnerability.class);
                        restrictByEcosystem(vulnerabilityForEcosystem, ecosystem);
                        updateVulnerability(vulnerabilityForEcosystem, DSL.using(configuration));
                    }));
            return v;
        }

        public void updateVulnerability(Vulnerability v, DSLContext dbContext) {
            if(metadataUtility == null) setMetadataUtility();
            this.vulnerability = v;
            updateVulnerability(dbContext);
        }

        private void updateVulnerability(DSLContext dbContext)  {
            updateVulnerabilityPatchData();

            this.vulnerabilityId = metadataUtility.getVulnerabilityId(vulnerability.getId(), dbContext);
            if(vulnerabilityId != -1) {
                metadataUtility.deleteVulnerabilityData(vulnerability.getId(), vulnerabilityId, dbContext);
            }
            if(!vulnerability.getValidatedPurls().isEmpty()) {
                logger.info(vulnerability.getId() + ": Inserting vulnerability and purls.");
                vulnerabilityId = metadataUtility.insertVulnerability(vulnerability, dbContext);
                metadataUtility.insertPurls(vulnerabilityId, vulnerability.getValidatedPurls(), dbContext);
                updatePackageVersions(null, dbContext);
            }
            else {
                logger.info(vulnerability.getId() + ": No relevant purls for ecosystem, skipping insertion.");
            }
        }

        private void updateVulnerabilityPatchData() {
            vulnerability.filterUnsupportedPatches(extensions);
            try {
                vulnerability.updatePatchDate();
            } catch (ParseException e) {
                logger.warn(vulnerability.getId() + ": Error updating patch date, continuing: " + e);
            }
        }

        public void updateNewPackageVersion(String ecosystem, String packageName, String version) {
            if(metadataUtility == null) setMetadataUtility();

            getDbContext(ecosystem).transaction(configuration -> {
                var transactionContext = DSL.using(configuration);
                var vulnerabilities = metadataUtility.getVulnerabilitiesForPurl(ecosystem, packageName, version, transactionContext);
                logger.info("Found the following vulnerabilities for " + packageName + ":" + version + ": " + vulnerabilities.toString());
                vulnerabilities.forEach(vulnerabilityId -> {
                    var jsonStatement = metadataUtility.getVulnerabilityStatement(vulnerabilityId, transactionContext);
                    this.vulnerability = gson.fromJson(jsonStatement, Vulnerability.class);
                    this.vulnerabilityId = vulnerabilityId;
                    restrictByEcosystem(this.vulnerability, ecosystem);
                    restrictByPackageName(this.vulnerability, packageName);
                    updatePackageVersions(version, transactionContext);
                });
            });
        }

        private void updatePackageVersions(String version, DSLContext dbContext) {
            var purls = vulnerability.getValidatedPurls();
            var pkgIds = metadataUtility.getPackageIds(dbContext, purls);
            if (pkgIds.size() > 0) {
                for(var pkgId : pkgIds.entrySet()) {
                    var purlsForPackage = getPurlsForPackage(purls, pkgId.getKey());
                    var purlsToUpdate = purlsForPackage;
                    if(version != null) {
                        purlsToUpdate = getPurlsForVersion(purlsForPackage, new ComparableVersion(version));
                    }
                    var vulnerablePackageVersionIds = updateVulnerablePackageVersions(purlsToUpdate, pkgId.getValue(), dbContext);
                    updateVulnerableCallables(purlsForPackage, pkgId.getValue(), vulnerablePackageVersionIds, dbContext);

                    if (vulnerablePackageVersionIds.size() == 0) {
                        logger.info(vulnerability.getId() + ": No matching package-versions for package + " + pkgId.getKey() + " found in database.");
                    }
                }
            }
            else {
                logger.info(vulnerability.getId() + ": No matching packages found in database.");
            }
        }

        private List<Long> updateVulnerablePackageVersions(LinkedHashSet<Purl> purls, Long pkgId, DSLContext dbContext) {
            var vulnerablePackageVersionIds = new ArrayList<Long>();
            if(purls.size() > 0) {
                var packageName = purls.stream().findFirst().orElseThrow().getPackageName();
                var pkgIdMap = new HashMap<String, Long>();
                pkgIdMap.put(packageName, pkgId);
                vulnerablePackageVersionIds.addAll(metadataUtility.getPackageVersionIds(purls, dbContext, pkgIdMap));
            }
            logger.info(vulnerability.getId() + ": Inserting data for " + vulnerablePackageVersionIds.size() + " vulnerable package-versions into the database.");
            vulnerablePackageVersionIds.forEach(id -> metadataUtility.injectPackageVersionVulnerability(vulnerability, id, dbContext));
            metadataUtility.insertVulnerabilityToPackageVersions(vulnerabilityId, vulnerablePackageVersionIds, dbContext);

            return vulnerablePackageVersionIds;
        }

        private void restrictByEcosystem(Vulnerability v, String ecosystem) {
            var purls = getPurlsForEcosystem(v.getValidatedPurls(), ecosystem);
            v.setPurls(new LinkedHashSet<>(purls.stream()
                    .filter(p -> p.getEcosystem().equals(ecosystem))
                    .map(Purl::toString).
                    collect(Collectors.toList())));

            var firstPatchedPurls = getPurlsForEcosystem(v.getValidatedFirstPatchedPurls(), ecosystem);
            v.setFirstPatchedPurls(new LinkedHashSet<>(firstPatchedPurls.stream()
                    .filter(p -> p.getEcosystem().equals(ecosystem))
                    .map(Purl::toString)
                    .collect(Collectors.toList())));
        }

        private void restrictByPackageName(Vulnerability v, String packageName) {
            var purls = getPurlsForPackage(v.getValidatedPurls(), packageName);
            v.setPurls(new LinkedHashSet<>(purls.stream()
                    .filter(p -> p.getPackageName().equals(packageName))
                    .map(Purl::toString)
                    .collect(Collectors.toList())));

            var firstPatchedPurls = getPurlsForPackage(v.getValidatedFirstPatchedPurls(), packageName);
            v.setFirstPatchedPurls(new LinkedHashSet<>(firstPatchedPurls.stream()
                    .filter(p -> p.getPackageName().equals(packageName))
                    .map(Purl::toString)
                    .collect(Collectors.toList())));
        }

        private LinkedHashSet<Purl> getPurlsForEcosystem(LinkedHashSet<Purl> purls, String ecosystem) {
            return sortPurls(purls.stream()
                    .filter(p -> ecosystem.equals(p.getEcosystem()))
                    .collect(Collectors.toCollection(LinkedHashSet::new)));
        }

        private LinkedHashSet<Purl> getPurlsForPackage(LinkedHashSet<Purl> purls, String packageName) {
            return sortPurls(purls.stream()
                .filter(p -> packageName.equals(p.getPackageName()))
                .collect(Collectors.toCollection(LinkedHashSet::new)));
        }

        private LinkedHashSet<Purl> getPurlsForVersion(LinkedHashSet<Purl> purls, ComparableVersion version) {
            return sortPurls(purls.stream()
                    .filter(p -> version.equals(p.getVersion()))
                    .collect(Collectors.toCollection(LinkedHashSet::new)));
        }

        private void updateVulnerableCallables(LinkedHashSet<Purl> purls, Long pkgId, List<Long> vulnerablePackageVersionIds, DSLContext dbContext) {
            var vulnerableFastenUris = new HashSet<String>();
            vulnerableFastenUris.addAll(findFastenUrisInLastVulnerableVersion(purls, pkgId, dbContext));
            vulnerableFastenUris.addAll(findFastenUrisInFirstPatchedVersions(pkgId, dbContext));
            var vulnerableCallables = findVulnerableCallableIds(vulnerableFastenUris, vulnerablePackageVersionIds, dbContext);
            var vulnerableCallableIds = vulnerableCallables.keySet();
            vulnerability.setFastenUris(new HashSet<>(vulnerableCallables.values()));

            logger.info(vulnerability.getId() + ": Inserting data for " + vulnerableCallableIds.size() + " vulnerable callables into the database.");
            vulnerableCallableIds.forEach(id -> metadataUtility.injectCallableVulnerability(vulnerability, id, dbContext));
            metadataUtility.insertVulnerabilityToCallables(vulnerabilityId, vulnerableCallableIds, dbContext);
        }

        private DSLContext getDbContext(String forge) {
            var context = forge.equals("") ? null : contexts.get(forge);
            if (context == null) {
                throw new UnsupportedOperationException(vulnerability.getId() + ": Malformed ecosystem data or unsupported ecosystem \"" + forge + "\"");
            }
            return context;
        }

        private HashSet<String> findFastenUrisInLastVulnerableVersion(LinkedHashSet<Purl> purls, Long pkgId, DSLContext dbContext) {
            var lastVulnerablePurl = findLast(purls);
            var lastVulnVersionId = findIdForPurl(lastVulnerablePurl, pkgId, dbContext);
            var fastenUris = new HashSet<String>();
            if(lastVulnVersionId != -1L) {
                vulnerability.getPatches().forEach(p -> {
                    logger.info(vulnerability.getId() + ": Searching for callables in " + p.getFileName() + " for package-version: " + lastVulnVersionId);
                    fastenUris.addAll(metadataUtility.getFastenUrisForPatch(p.getFileName(),
                            p.getOriginalChangedLineNumbers(),
                            lastVulnVersionId, dbContext));
                });
            }
            else {
                logger.info(vulnerability.getId() + ": could not find package-version in DB for last vulnerable purl: " + lastVulnerablePurl + ", skipping callable search.");
                if(fastenApiClient.getApiUrl() != null) {
                    requestPurlIngestion(lastVulnerablePurl);
                }
            }
            return fastenUris;
        }

        private void requestPurlIngestion(Purl p) {
            try {
                logger.info(vulnerability.getId() + ": calling FASTEN API to ingest package-version for purl: " + p);
                var response = fastenApiClient.requestPackageVersion(p.getEcosystem(), p.getPackageName(), p.getVersion().toString());
                var code = response.statusCode();
                if(code != 201) {
                    logger.warn(vulnerability.getId() + ": unexpected response " + code + " to ingest request for purl: " + p);
                }
            } catch (Exception e) {
                logger.warn(vulnerability.getId() + ": there was a problem calling FASTEN API for purl " + p + " : " + e);
            }
        }

        private Purl findLast(LinkedHashSet<Purl> purls) {
            Purl[] purlsArray = new Purl[purls.size()];
            assert purlsArray.length > 0;
            purlsArray = purls.toArray(purlsArray);
            return purlsArray[purlsArray.length - 1];
        }

        private HashSet<String> findFastenUrisInFirstPatchedVersions(Long pkgId, DSLContext dbContext) {
            var fastenUris = new HashSet<String>();
            vulnerability.getValidatedFirstPatchedPurls().forEach(purl -> {
                var firstPatchedVersionId = findIdForPurl(purl, pkgId, dbContext);
                if(firstPatchedVersionId != -1L) {
                    vulnerability.getPatches().forEach(patch -> {
                        logger.info(vulnerability.getId() + ": Searching for callables in " + patch.getFileName() + " for package-version: " + firstPatchedVersionId);
                        fastenUris.addAll(metadataUtility.getFastenUrisForPatch(patch.getFileName(),
                                patch.getNewChangedLineNumbers(),
                                firstPatchedVersionId, dbContext));
                    });
                }
            });

            return fastenUris;
        }

        private Long findIdForPurl(Purl purl, Long pkgId, DSLContext dbContext) {
            if (purl != null && pkgId != -1) {
                return metadataUtility.getPackageVersionId(purl, dbContext, pkgId);
            }
            return -1L;
        }

        private HashMap<Long, String> findVulnerableCallableIds(HashSet<String> vulnerableFastenUris, List<Long> vulnPkgVersionIds, DSLContext dbContext) {
            var vulnerableCallables = new HashMap<Long, String>();
            vulnerableFastenUris.forEach(uri -> {
                var callIds = metadataUtility.getCallableIdsForFastenUri(uri, new HashSet<>(vulnPkgVersionIds), dbContext);
                callIds.forEach(id -> vulnerableCallables.put(id, metadataUtility.getFullFastenUri(uri, id)));
            });
            return vulnerableCallables;
        }
    }
}
