/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.fasten.analyzer.vulnerabilitystatementsprocessor;

import eu.fasten.core.dbconnectors.PostgresConnector;
import org.jooq.DSLContext;
import org.json.JSONArray;
import org.json.JSONTokener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import picocli.CommandLine;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;

@CommandLine.Command(name = "Vulnerability Statements Processor")
public class Main implements Runnable {

    private static final Logger logger = LoggerFactory.getLogger(Main.class);

    @CommandLine.Option(names = {"-f", "--file"},
            paramLabel = "Vulnerability statements",
            description = "Path to file with JSON array of vulnerability statements")
    String inputFile;

    @CommandLine.Option(names = {"-d", "--database"},
        paramLabel = "Database URLs",
        description = "Key-value pairs of database URLs, eg. \n" +
            "mvn=jdbc:postgresql://localhost/fasten_java,\n" +
            "PyPI=jdbc:postgresql://localhost/fasten_python,\n" +
            "debian=jdbc:postgresql://localhost/fasten_c\n",
        split = ",")
    Map<String, String> dbUrls;

    @CommandLine.Option(names = {"-u", "--user"},
            paramLabel = "Database user",
            description = "Database user name",
            defaultValue = "fasten")
    String dbUser;

    public static void main(String[] args) {
        final int exitCode = new CommandLine(new Main()).execute(args);
        System.exit(exitCode);
    }

    @Override
    public void run() {
        var kafkaPlugin = new VulnerabilityStatementsProcessor.VulnerabilityStatementsKafkaPlugin();
        try {
            var contexts = new HashMap<String, DSLContext>();
            for(var forge : dbUrls.keySet()) {
                contexts.put(forge,
                    PostgresConnector.getDSLContext(dbUrls.get(forge), dbUser, true));
            }
            kafkaPlugin.setDBConnection(contexts);
            var reader = new FileReader(inputFile);
            final var vulnsJson = new JSONArray(new JSONTokener(reader));
            vulnsJson.forEach(v -> kafkaPlugin.consume(v.toString()));
            kafkaPlugin.produce().ifPresent(System.out::println);

        } catch (SQLException e) {
            logger.error("Error connecting to databases: ", e);
        }
        catch (FileNotFoundException e) {
            logger.error("Error reading input file " + inputFile + ": ", e);
        }
    }
}
