/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.fasten.analyzer.vulnerabilitypackageslistener;

import eu.fasten.analyzer.vulnerabilitystatementsprocessor.VulnerabilityStatementsProcessor;
import eu.fasten.analyzer.vulnerabilitystatementsprocessor.utils.FastenApiClient;
import eu.fasten.core.data.Constants;
import eu.fasten.core.plugins.DBConnector;
import eu.fasten.core.plugins.KafkaPlugin;
import org.jooq.DSLContext;
import org.json.JSONObject;
import org.pf4j.Extension;
import org.pf4j.Plugin;
import org.pf4j.PluginWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public class VulnerabilityPackagesListener extends Plugin {

    public VulnerabilityPackagesListener(PluginWrapper wrapper) {
        super(wrapper);
    }

    @Extension
    public static class VulnerabilityPackagesKafkaPlugin implements KafkaPlugin, DBConnector {

        private static Map<String, DSLContext> contexts;
        private List<String> consumeTopics = new LinkedList<>(Collections.singletonList("fasten.CallableIndexFastenPlugin.out"));
        private Exception pluginError = null;
        private final static Logger logger = LoggerFactory.getLogger("VulnerabilityPackagesKafkaPlugin");
        private final FastenApiClient fastenApiClient = new FastenApiClient();
        private static final VulnerabilityStatementsProcessor.VulnerabilityStatementsKafkaPlugin statementsProcessor = new VulnerabilityStatementsProcessor.VulnerabilityStatementsKafkaPlugin();

        @Override
        public void setDBConnection(Map<String, DSLContext> dslContexts) {
            contexts = dslContexts;
            statementsProcessor.setDBConnection(contexts);
        }

        @Override
        public String name() {
            return "Vulnerability Packages Listener";
        }

        @Override
        public String description() {
            return "Listens to package updates from the (graph) database and triggers re-processing of vulnerabilities";
        }

        @Override
        public String version() {
            return "0.0.1";
        }

        @Override
        public void start() {
        }

        @Override
        public void stop() {
        }

        @Override
        public void setPluginError(Exception throwable) {
            this.pluginError = throwable;
        }

        @Override
        public Exception getPluginError() {
            return pluginError;
        }

        @Override
        public void freeResource() {
        }

        @Override
        public Optional<List<String>> consumeTopic() {
            return Optional.of(consumeTopics);
        }

        @Override
        public void setTopics(List<String> consumeTopics) {
            this.consumeTopics = consumeTopics;
        }

        @Override
        public void consume(String record) {
            setPluginError(null);
            statementsProcessor.setDBConnection(contexts);
            statementsProcessor.setFastenApiClient(fastenApiClient);
            var jsonRecord = new JSONObject(record);
            var payload = findPayload(jsonRecord);
            if (payload != null) {
                var ecosystem = payload.getString("forge");
                var packageName = getPackageName(payload);
                var version = payload.getString("version");
                logger.info("Processing package update for forge \"" + ecosystem + "\": " + packageName + ":" + version);
                statementsProcessor.updateNewPackageVersion(ecosystem, packageName, version);
            } else {
                logger.error("Could not parse payload in message: " + record);
            }
        }

        private String getPackageName(JSONObject payload) {
            var ecosystem = payload.getString("forge");
            switch(ecosystem) {
                case Constants.mvnForge: return payload.getString("groupId") + Constants.mvnCoordinateSeparator + payload.getString("artifactId");
                case Constants.debianForge:
                case Constants.pypiForge:
                    return payload.getString("product");
                default: throw new UnsupportedOperationException("Unsupported forge: \"" + ecosystem + "\"");
            }
        }

        static JSONObject findPayload(JSONObject json) {
            for (var key : json.keySet()) {
                if (key.equals("payload")) {
                    var candidatePayload = json.getJSONObject(key);
                    if (candidatePayload.has("forge")) {
                        return candidatePayload;
                    }
                } else {
                    var other = json.get(key);
                    if (other instanceof JSONObject) {
                        var otherPayload = findPayload((JSONObject) other);
                        if(otherPayload != null) return otherPayload;
                    }
                }
            }
            return null;
        }

        @Override
        public Optional<String> produce() {

            return Optional.empty();
        }

        @Override
        public String getOutputPath() {
            return null;
        }
    }
}
