package eu.fasten.analyzer.restapiplugin.api;

import eu.fasten.analyzer.restapiplugin.KnowledgeBaseConnector;
import eu.fasten.analyzer.restapiplugin.LazyIngestionProvider;
import eu.fasten.analyzer.restapiplugin.RestApplication;
import eu.fasten.core.data.Constants;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.MediaType;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import java.io.IOException;
import java.util.List;

@RestController
public class VulnerabilityApi {

    private static final Logger LOG = LoggerFactory.getLogger(VulnerabilityApi.class);

    private LazyIngestionProvider ingestion = new LazyIngestionProvider();

    public void setLazyIngestionProvider(LazyIngestionProvider ingestion) {
        this.ingestion = ingestion;
    }

    @GetMapping(value = "/vulnerabilities", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<String> getAllVulnerabilities(@RequestParam(required = false, defaultValue = "0") int offset,
                                                 @RequestParam(required = false, defaultValue = RestApplication.DEFAULT_PAGE_SIZE) int limit,
                                                 @RequestParam(required = false, defaultValue = "") List<String> attributes) {

        var statements = KnowledgeBaseConnector.kbDao.getAllVulnerabilities(offset, limit);
        JSONArray vulnerabilities = new JSONArray(statements);
        var result = getStatementsAttributes(vulnerabilities, attributes).toString();
        return Responses.ok(result);
    }

    @GetMapping(value = "/vulnerabilities/{external_id}", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<String > getVulnerability(@PathVariable("external_id") String externalId,
                                             @RequestParam(required = false, defaultValue = "") List<String> attributes) {

        var statement = KnowledgeBaseConnector.kbDao.getVulnerability(externalId);
        if (statement == null) {
            return new ResponseEntity<>("Vulnerability not found", HttpStatus.NOT_FOUND);
        }
        JSONObject vulnerability = new JSONObject(statement);
        JSONObject needed = getStatementAttributes(vulnerability.getJSONObject("statement"), attributes);
        vulnerability.put("statement", needed);
        var result = vulnerability.toString();
        return Responses.ok(result);
    }

    @GetMapping(value = "/vulnerabilities/{external_id}/purls", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<String> getPurls(@RequestParam(required = false, defaultValue = "0") int offset,
                                    @RequestParam(required = false, defaultValue = RestApplication.DEFAULT_PAGE_SIZE) int limit,
                                    @PathVariable("external_id") String externalId) {

        var result = KnowledgeBaseConnector.kbDao.getPurls(externalId, offset, limit);
        if (result == null) {
            return new ResponseEntity<>("Vulnerability not found", HttpStatus.NOT_FOUND);
        }
        return Responses.ok(result);
    }

    @GetMapping(value = "/vulnerabilities/{external_id}/callables", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<String> getVulnerableCallables(@RequestParam(required = false, defaultValue = "0") int offset,
                                                  @RequestParam(required = false, defaultValue = "1000") int limit,
                                                  @PathVariable("external_id") String externalId) {

        var uriMap = KnowledgeBaseConnector.kbDao.getVulnerableCallables(externalId, offset, limit);
        if (uriMap == null) {
            return new ResponseEntity<>("Vulnerability not found", HttpStatus.NOT_FOUND);
        }
        var json = new JSONObject();
        uriMap.forEach((key, value) -> json.put(String.valueOf(key), value));
        var result = json.toString();
        return Responses.ok(result);
    }

    @PostMapping(value = "/vulnerabilities/coordinates", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<String> getCoordinatesVulnerabilities(@RequestBody List<String> coordinates,
                                                         @RequestParam(required = false, defaultValue = "") List<String> attributes) {
        JSONArray jsonArray = new JSONArray();
        LOG.info("Received a list of coordinates");
        coordinates.stream().map(c -> {
            String packageName, packageVersion;
            switch (KnowledgeBaseConnector.forge) {
                case Constants.mvnForge: {
                    packageName = c.split(Constants.mvnCoordinateSeparator)[0]
                        + Constants.mvnCoordinateSeparator
                        + c.split(Constants.mvnCoordinateSeparator)[1];
                    packageVersion = c.split(Constants.mvnCoordinateSeparator)[2];
                    break;
                }
                default: {
                    packageName = c.split(Constants.mvnCoordinateSeparator)[0];
                    packageVersion = c.split(Constants.mvnCoordinateSeparator)[1];
                }
            }
            var vulnerabilities = "";
            if (!KnowledgeBaseConnector.kbDao.assertPackageExistence(packageName, packageVersion)) {
                try {
                    ingestion.ingestArtifactWithDependencies(packageName, packageVersion);
                    LOG.info("Coordinate " + c + " not found, but should be processed soon. Try again later");
                } catch (Exception e) {
                    LOG.info("Cannot find coordinate " + c + ": " +e.getMessage());
                }
            } else {
                var statements = KnowledgeBaseConnector.kbDao
                        .getPackageVersionVulnerabilities(packageName, packageVersion, false);
                JSONArray vul = new JSONArray(statements);
                vulnerabilities = getStatementsAttributes(vul, attributes).toString();
            }
            var json = new JSONObject();
            json.put(c, vulnerabilities);
            return json;
        }).forEach(jsonArray::put);                                                
        var result = jsonArray.toString();
        return Responses.ok(result);
    }

    @GetMapping(value = "/packages/{pkg}/{pkg_ver}/vulnerabilities", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<String> getPackageVersionVulnerabilities(@PathVariable("pkg") String packageName,
                                                            @PathVariable("pkg_ver") String packageVersion,
                                                            @RequestParam(required = false, defaultValue = "") List<String> attributes) {
        
        if (!KnowledgeBaseConnector.kbDao.assertPackageExistence(packageName, packageVersion)) {
            try {
                ingestion.ingestArtifactWithDependencies(packageName, packageVersion);
            } catch (IllegalArgumentException | IOException e) {
                return new ResponseEntity<>(e.getMessage(), HttpStatus.BAD_REQUEST);
            }
            return Responses.lazyIngestion();
        }
        var statements = KnowledgeBaseConnector.kbDao.getPackageVersionVulnerabilities(packageName, packageVersion, true);
        JSONArray vulnerabilities = new JSONArray(statements);
        var result = getStatementsAttributes(vulnerabilities, attributes).toString();
        return Responses.ok(result);
    }

    @PostMapping(value = "/vulnerabilities/purls", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<String> getPurlVulnerabilities(@RequestBody String purl,
                                                  @RequestParam(required = false, defaultValue = "") List<String> attributes) {
        if (!KnowledgeBaseConnector.kbDao.assertPurlExistence(purl)) return new ResponseEntity<>("", HttpStatus.OK);
        var statements = KnowledgeBaseConnector.kbDao.getPurlVulnerabilities(purl, true);
        JSONArray vulnerabilities = new JSONArray((statements));
        var result = getStatementsAttributes(vulnerabilities, attributes)
                .toString().replace("\\/", "/");
        return Responses.ok(result);
    }

    private JSONObject getStatementAttributes(JSONObject statement, List<String> attributes) {
    var neededInfo = new JSONObject();
    if (attributes.size() > 0) {
        for (var attribute : statement.keySet()) {
            if (attributes.contains(attribute)) {
                neededInfo.put(attribute, statement.get(attribute));
            }
        }
    } else {
        neededInfo = statement;
    }
    return neededInfo;
}

    private JSONArray getStatementsAttributes(JSONArray statements, List<String> attributes) {
        JSONArray vulnInfo = new JSONArray();
        statements.forEach(v -> {
            JSONObject vJson = (JSONObject) v;
            JSONObject needed = getStatementAttributes(vJson.getJSONObject("statement"), attributes);
            vJson.put("statement", needed);
            vulnInfo.put(vJson);
        });
        return vulnInfo;
    }
}