/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.fasten.analyzer.parallelvulnerabilitycacheinvalidationplugin;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;

import java.sql.Timestamp;
import java.util.Collections;
import java.util.Optional;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

import dev.c0ps.maven.data.ResolvedRevision;
import dev.c0ps.maven.data.Scope;
import dev.c0ps.maven.resolution.MavenResolver;
import it.unimi.dsi.fastutil.objects.ObjectLinkedOpenHashSet;

public class ParallelVulnerabilityCacheInvalidationPluginTest {

    private ParallelVulnerabilityCacheInvalidationPlugin.ParallelVulnerabilityCacheInvalidationExtension pluginExtension;

    @BeforeEach
    public void setUp() {
        pluginExtension = new ParallelVulnerabilityCacheInvalidationPlugin.ParallelVulnerabilityCacheInvalidationExtension();
        pluginExtension.setTopics(Collections.singletonList("fasten.VulnerabilityConsumerExtension.out"));
    }

    @Test
    public void nullGraphResolverFailureTest() {
        pluginExtension.consume("{}");
        assertNotNull(pluginExtension.getPluginError());
    }

    @Test
    public void consumeJsonErrorTest() {
        pluginExtension.loadGraphResolver(Mockito.mock(MavenResolver.class));
        pluginExtension.consume("{\"input\":\"\"}");
        assertNotNull(pluginExtension.getPluginError());
    }

    @Test
    public void consumeTest() {
        var json ="{\n" +
                "\t\"input\": {\n" +
                "\t\t\"purls\": [\n" +
                "\t\t\t\"pkg:maven/org.wso2.carbon.apimgt.org/wso2.carbon.apimgt.api@6.5.213\"\n" +
                "\t\t]\n" +
                "\t}\n" +
                "}";
        var mockResolver = Mockito.mock(MavenResolver.class);
        var dependantsMock = new ObjectLinkedOpenHashSet<>(new ResolvedRevision[]{new ResolvedRevision(0, "groupId", "artifactId", "version", Timestamp.valueOf("2018-09-01 09:01:15"), Scope.COMPILE)});
        Mockito
                .when(mockResolver.resolveDependents("org.wso2.carbon.apimgt.org", "wso2.carbon.apimgt.api", "6.5.213"))
                .thenReturn(dependantsMock);
        pluginExtension.loadGraphResolver(mockResolver);
        pluginExtension.consume(json);
        assertNull(pluginExtension.getPluginError());
    }

    @Test
    public void consumerTopicsTest() {
        var topics = Optional.of(Collections.singletonList("fasten.VulnerabilityConsumerExtension.out"));
        assertEquals(topics, pluginExtension.consumeTopic());
    }

    @Test
    public void consumerTopicChangeTest() {
        var topics1 = Optional.of(Collections.singletonList("fasten.VulnerabilityConsumerExtension.out"));
        assertEquals(topics1, pluginExtension.consumeTopic());
        var differentTopic = Collections.singletonList("DifferentKafkaTopic");
        pluginExtension.setTopics(differentTopic);
        assertEquals(Optional.of(differentTopic), pluginExtension.consumeTopic());
    }

}
