/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.fasten.analyzer.parallelvulnerabilitycacheinvalidationplugin;

import java.io.File;
import java.sql.Timestamp;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

import org.jooq.DSLContext;
import org.json.JSONArray;
import org.json.JSONObject;
import org.pf4j.Extension;
import org.pf4j.Plugin;
import org.pf4j.PluginWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import dev.c0ps.maven.data.Revision;
import dev.c0ps.maven.resolution.IMavenResolver;
import dev.c0ps.maven.resolution.MavenResolver;
import eu.fasten.core.json.ToJson;
import eu.fasten.core.maven.resolution.MavenResolverIO;
import eu.fasten.core.plugins.DataWriter;
import eu.fasten.core.plugins.DependencyGraphUser;
import eu.fasten.core.plugins.KafkaPlugin;
import eu.fasten.core.utils.FileUtils;
import it.unimi.dsi.fastutil.objects.ObjectLinkedOpenHashSet;

public class ParallelVulnerabilityCacheInvalidationPlugin extends Plugin {
    public ParallelVulnerabilityCacheInvalidationPlugin(PluginWrapper wrapper) {
        super(wrapper);
    }

    @Extension
    public static class ParallelVulnerabilityCacheInvalidationExtension implements KafkaPlugin, DependencyGraphUser, DataWriter {

        private final Logger logger = LoggerFactory.getLogger(ParallelVulnerabilityCacheInvalidationExtension.class.getName());

        private List<String> consumeTopics = new LinkedList<>(Collections.singletonList("fasten.VulnerabilityConsumerExtension.out"));

        private static IMavenResolver graphResolver;
        private static String baseDir;
        private Exception pluginError = null;
        private ObjectLinkedOpenHashSet<Revision> depSet;

        /**
         * The helper method that creates a graph resolver. It first creates a Database Context from Knowledge Base and then uses it to build dependency graph in the graph resolver.
         *
         * @param dbContext    - Connection to the database
         * @param depGraphPath - the directory where the dependency graph can be found.
         */
        public void loadGraphResolver(DSLContext dbContext, String depGraphPath) {
            logger.info("Building Dependency Graph from " + depGraphPath + "...");
            try {
                var graphResolver = new MavenResolverIO(dbContext, new File(depGraphPath)).loadResolver();
                ParallelVulnerabilityCacheInvalidationExtension.graphResolver = graphResolver;
            } catch (Exception e) {
                var err = "Couldn't build the dependency graph";
                logger.error(err, e);
                this.setPluginError(new RuntimeException(err, e));
            }
            logger.info("...Dependency Graph has been successfully built.");
        }

        /**
         * The helper method that creates a graph resolver. It is overloaded method that allows to load graph resolver from the mocked instance. Currently, used for testing purposes.
         *
         * @param mockResolver - mocked instance of GraphMavenResolver.
         */
        public void loadGraphResolver(MavenResolver mockResolver) {
            logger.info("Loaded mock graph resolver");
            ParallelVulnerabilityCacheInvalidationExtension.graphResolver = mockResolver;
        }

        @Override
        public void setBaseDir(String baseDir) {
            ParallelVulnerabilityCacheInvalidationExtension.baseDir = baseDir;
        }

        @Override
        public String name() {
            return "Vulnerability Cache Invalidation Plugin";
        }

        @Override
        public String description() {
            return "Vulnerability Cache Invalidation Plugin. " + "Consumes list of updated product from Kafka" + " topic and invalidates cache of vulnerable paths"
                    + " for all its transitive dependants.";
        }

        @Override
        public String version() {
            return "0.0.1";
        }

        @Override
        public void start() {}

        @Override
        public void stop() {
            ParallelVulnerabilityCacheInvalidationExtension.graphResolver = null;
        }

        public void setPluginError(Exception throwable) {
            this.pluginError = throwable;
        }

        @Override
        public Exception getPluginError() {
            return this.pluginError;
        }

        @Override
        public void freeResource() {}

        @Override
        public Optional<List<String>> consumeTopic() {
            return Optional.of(consumeTopics);
        }

        @Override
        public void setTopics(List<String> consumeTopics) {
            this.consumeTopics = consumeTopics;
        }

        @Override
        public void consume(String record) {
            this.depSet = new ObjectLinkedOpenHashSet<>();
            this.setPluginError(null);

            // Parse JSON object from kafka topic of GraphDBExtension.
            // Although it doesn't have output payload, the plugin serializes the graph for its input.
            // And we can use the input copy from this topic and the serialized graph to process our caching.
            var json = new JSONObject(record);

            if (json.has("input")) {
                if (json.get("input").toString().isEmpty()) {
                    logger.error("Empty input");
                    setPluginError(new RuntimeException("Empty input"));
                    return;
                }
                json = json.getJSONObject("input");
            } else {
                logger.error("Input is absent");
                setPluginError(new RuntimeException("Input is absent"));
                return;
            }

            var purls = json.getJSONArray("purls");
            for (int i = 0; i < purls.length(); i++) {
                var purl = purls.getString(i);
                if (!purl.startsWith("pkg:maven/")) {
                    continue;
                }
                purl = purl.replace("pkg:maven/", "");

                var groupId = purl.split("/")[0];
                var artifactId = purl.split("/")[1].split("@")[0];
                var version = purl.split("/")[1].split("@")[1];

                try {
                    // Resolve the set of transitive dependants for this product.
                    this.depSet.addAll(ParallelVulnerabilityCacheInvalidationExtension.graphResolver.resolveDependents(groupId, artifactId, version));
                    this.depSet.add(new Revision(groupId, artifactId, version, new Timestamp(-1)));
                } catch (RuntimeException e) {
                    logger.error("Failed to resolve dependents for {}", purl, e);
                    setPluginError(e);
                    return;
                }
            }
            try {
                // Go over the set and invalidate the cache for each dependant.
                for (Revision revision : depSet) {
                    var firstLetter = revision.getGroupId().substring(0, 1);
                    var outputPath = baseDir + File.separator + firstLetter + File.separator + revision.getGroupId() + File.separator + revision.getArtifactId() + File.separator
                            + revision.version.toString() + ".json";

                    File outputFile = new File(outputPath);
                    if (!outputFile.exists() || outputFile.length() == 0) {
                        continue;
                    }
                    FileUtils.forceDeleteFile(outputFile);
                }
            } catch (RuntimeException e) {
                logger.error("Error deleting files", e);
                setPluginError(e);
            }
        }

        @Override
        public Optional<String> produce() {
            if (this.depSet == null) {
                return Optional.empty();
            } else {
                var jsonDepSet = new JSONArray();
                depSet.stream().map(ToJson::map).forEach(jsonDepSet::put);
                var json = new JSONObject();
                json.put("artifacts", jsonDepSet);
                return Optional.of(json.toString());
            }
        }

        @Override
        public String getOutputPath() {
            return null;
        }
    }

}
