/*
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package aws

import (
	"context"
	"errors"
	"os"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ssm"

	//nolint
	. "github.com/onsi/ginkgo/v2"

	// nolint
	. "github.com/onsi/gomega"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"

	"github.com/external-secrets/external-secrets-e2e/framework"
	"github.com/external-secrets/external-secrets-e2e/framework/log"
	awscommon "github.com/external-secrets/external-secrets-e2e/suites/provider/cases/aws"
	esv1beta1 "github.com/external-secrets/external-secrets/apis/externalsecrets/v1beta1"
	esmetav1 "github.com/external-secrets/external-secrets/apis/meta/v1"
)

type Provider struct {
	ServiceAccountName      string
	ServiceAccountNamespace string

	region    string
	client    *ssm.SSM
	framework *framework.Framework
}

func NewProvider(f *framework.Framework, kid, sak, st, region, saName, saNamespace string) *Provider {
	sess, err := session.NewSessionWithOptions(session.Options{
		Config: aws.Config{
			Credentials: credentials.NewStaticCredentials(kid, sak, st),
			Region:      aws.String(region),
		},
	})
	if err != nil {
		Fail(err.Error())
	}
	sm := ssm.New(sess)
	prov := &Provider{
		ServiceAccountName:      saName,
		ServiceAccountNamespace: saNamespace,
		region:                  region,
		client:                  sm,
		framework:               f,
	}

	BeforeEach(func() {
		awscommon.SetupStaticStore(f, awscommon.AccessOpts{KID: kid, SAK: sak, ST: st, Region: region}, esv1beta1.AWSServiceParameterStore)
		awscommon.CreateReferentStaticStore(f, awscommon.AccessOpts{KID: kid, SAK: sak, ST: st, Region: region}, esv1beta1.AWSServiceParameterStore)
		prov.SetupReferencedIRSAStore()
		prov.SetupMountedIRSAStore()
	})

	AfterEach(func() {
		prov.TeardownReferencedIRSAStore()
		prov.TeardownMountedIRSAStore()
	})

	return prov
}

func NewFromEnv(f *framework.Framework) *Provider {
	kid := os.Getenv("AWS_ACCESS_KEY_ID")
	sak := os.Getenv("AWS_SECRET_ACCESS_KEY")
	st := os.Getenv("AWS_SESSION_TOKEN")
	region := os.Getenv("AWS_REGION")
	saName := os.Getenv("AWS_SA_NAME")
	saNamespace := os.Getenv("AWS_SA_NAMESPACE")
	return NewProvider(f, kid, sak, st, region, saName, saNamespace)
}

// CreateSecret creates a secret at the provider.
func (s *Provider) CreateSecret(key string, val framework.SecretEntry) {
	pmTags := make([]*ssm.Tag, 0)
	for k, v := range val.Tags {
		pmTags = append(pmTags, &ssm.Tag{
			Key:   aws.String(k),
			Value: aws.String(v),
		})
	}
	// tags and overwrite can't be used together
	// overwrite is needed for the version test
	overwrite := false
	if len(val.Tags) == 0 {
		overwrite = true
	}
	_, err := s.client.PutParameter(&ssm.PutParameterInput{
		Name:      aws.String(key),
		Value:     aws.String(val.Value),
		DataType:  aws.String("text"),
		Type:      aws.String("String"),
		Overwrite: aws.Bool(overwrite),
		Tags:      pmTags,
	})
	Expect(err).ToNot(HaveOccurred())
}

// DeleteSecret deletes a secret at the provider.
func (s *Provider) DeleteSecret(key string) {
	_, err := s.client.DeleteParameter(&ssm.DeleteParameterInput{
		Name: aws.String(key),
	})
	var nf *ssm.ParameterNotFound
	if errors.As(err, &nf) {
		return
	}
	Expect(err).ToNot(HaveOccurred())
}

// MountedIRSAStore is a SecretStore without auth config
// ESO relies on the pod-mounted ServiceAccount when using this store.
func (s *Provider) SetupMountedIRSAStore() {
	secretStore := &esv1beta1.SecretStore{
		ObjectMeta: metav1.ObjectMeta{
			Name:      awscommon.MountedIRSAStoreName(s.framework),
			Namespace: s.framework.Namespace.Name,
		},
		Spec: esv1beta1.SecretStoreSpec{
			Provider: &esv1beta1.SecretStoreProvider{
				AWS: &esv1beta1.AWSProvider{
					Service: esv1beta1.AWSServiceParameterStore,
					Region:  s.region,
					Auth:    esv1beta1.AWSAuth{},
				},
			},
		},
	}
	err := s.framework.CRClient.Create(context.Background(), secretStore)
	Expect(err).ToNot(HaveOccurred())
}

func (s *Provider) TeardownMountedIRSAStore() {
	s.framework.CRClient.Delete(context.Background(), &esv1beta1.ClusterSecretStore{
		ObjectMeta: metav1.ObjectMeta{
			Name: awscommon.MountedIRSAStoreName(s.framework),
		},
	})
}

// ReferncedIRSAStore is a ClusterStore
// that references a (IRSA-) ServiceAccount in the default namespace.
func (s *Provider) SetupReferencedIRSAStore() {
	log.Logf("creating IRSA ClusterSecretStore %s", s.framework.Namespace.Name)
	secretStore := &esv1beta1.ClusterSecretStore{
		ObjectMeta: metav1.ObjectMeta{
			Name: awscommon.ReferencedIRSAStoreName(s.framework),
		},
	}
	_, err := controllerutil.CreateOrUpdate(context.Background(), s.framework.CRClient, secretStore, func() error {
		secretStore.Spec.Provider = &esv1beta1.SecretStoreProvider{
			AWS: &esv1beta1.AWSProvider{
				Service: esv1beta1.AWSServiceParameterStore,
				Region:  s.region,
				Auth: esv1beta1.AWSAuth{
					JWTAuth: &esv1beta1.AWSJWTAuth{
						ServiceAccountRef: &esmetav1.ServiceAccountSelector{
							Name:      s.ServiceAccountName,
							Namespace: &s.ServiceAccountNamespace,
						},
					},
				},
			},
		}
		return nil
	})
	Expect(err).ToNot(HaveOccurred())
}

func (s *Provider) TeardownReferencedIRSAStore() {
	s.framework.CRClient.Delete(context.Background(), &esv1beta1.ClusterSecretStore{
		ObjectMeta: metav1.ObjectMeta{
			Name: awscommon.ReferencedIRSAStoreName(s.framework),
		},
	})
}
