package wireguard

import (
	"errors"
	"github.com/eventure/hide.client.linux/rest"
	"github.com/vishvananda/netlink"
	"golang.zx2c4.com/wireguard/wgctrl"
	"golang.zx2c4.com/wireguard/wgctrl/wgtypes"
	"log"
	"net"
	"time"
)

type Config struct {
	Name					string				`yaml:"name,omitempty"`							// Interface name to use for the created WireGuard interface
	ListenPort				int					`yaml:"listenPort,omitempty"`					// Local UDP listen/bind port - 0 for automatic
	Mark					int					`yaml:"mark,omitempty"`							// Firewall mark for the traffic generated by the wireguard module
	RPDBPriority			int					`yaml:"rpdbPriority,omitempty"`					// Priority of installed RPDB rules
	PrivateKey				string				`yaml:"privateKey,omitempty"`					// Explicitly specified private key
	RoutingTable			int					`yaml:"routingTable,omitempty"`					// Routing table number to operate on when managing wireguard routes
	LeakProtection			bool				`yaml:"leakProtection,omitempty"`				// Enable or disable leak protection ( loopback routes )
	ResolvConfBackupFile	string				`yaml:"resolvConfBackupFile,omitempty"`			// Name of the resolv.conf backup file
	DpdTimeout				time.Duration		`yaml:"dpdTimeout,omitempty"`					// DPD timeout
	SplitTunnel				string				`yaml:"splitTunnel,omitempty"`					// A comma separated list of networks (CIDRs) for which to bypass the wireguard tunnel ( Split-Tunneling )
	IPv4					bool				`yaml:"IPv4,omitempty"`							// Add routes and rules for IPv4 protocol family
	IPv6					bool				`yaml:"IPv6,omitempty"`							// Add routes and rules for IPv6 protocol family
}

func ( c *Config ) Check() ( err error ) {
	if len( c.Name ) == 0 { err = errors.New( "missing wireGuard interface name" ); return }
	if c.DpdTimeout == 0 { err = errors.New( "dpd timeout not set" ); return }
	if c.DpdTimeout > time.Minute { err = errors.New( "dpd timeout above 1 minute" ); return }
	return
}

type Link struct {
	*Config
	
	wireguardLink	netlink.Link
	mtu				int
	wgClient		*wgctrl.Client
	
	privateKey		wgtypes.Key
	
	peer			wgtypes.PeerConfig
	
	ips				[]net.IP
	routes			[]*netlink.Route																														// Unfortunately, the netlink library can't list routes in the routing tables ...
	gatewayRoutes	[]*netlink.Route																														// ... other than "main"
	loopbackRoutes	[]*netlink.Route
	
	rule			*netlink.Rule																															// Use just one rule when diverting traffic to our routing table
	rule6			*netlink.Rule
	
	resolvConf		[]byte																																	// resolv.conf backup
	
	stack			[]func() error
}

func New( config *Config ) *Link { if config == nil { config = &Config{} }; return &Link{ Config: config } }
func (l *Link) PublicKey() wgtypes.Key { return l.privateKey.PublicKey() }

// Open the wireguard link, i.e. create or open an existing wireguard interface
func ( l *Link ) Open() ( err error ) {
	if err = l.Config.Check(); err != nil { log.Println( "Link: [ERR] Bad WireGuard configuration:", err.Error() ); return }
	if err = l.handlePrivateKey(); err != nil { return }																									// Check the private key first
	if l.wgClient, err = wgctrl.New(); err != nil { log.Println( "Link: [ERR] Wireguard control client failed:", err ); return }							// Create a wireguard control client
	if err = l.ipLinkUp(); err != nil { return }																											// Bring the networking interface UP
	if err = l.wgLinkUp(); err != nil { return }																											// Configure the wireguard private key and listen port
	return
}

// Close the wireguard interface
func ( l *Link ) Close() { _ = l.ipLinkDown() }

// Up adds a wireguard peer and routes it
func ( l *Link ) Up( response *rest.ConnectResponse ) ( err error ) {
	defer func() { if err != nil { l.Down() } }()
	// Avoid fragmentation if possible, set a small MTU
	// On IPv4, DS-Lite carrier connection takes MTU down as low as 1452 bytes
	// On IPv6, we can't go below 1280 as the lowest Internet IPv6 MTU is 1280 bytes
	// IPv4 header is 20 bytes, IPv6 header is 40 bytes and UDP header is 8 bytes
	// Wireguard overhead is 32 bytes
	if response.Endpoint.IP.To4() == nil { l.mtu = 1360 - 80 } else { l.mtu = 1452 - 60 }																	// Calculate MTU according to the carrier connection protocol
	if err = l.ipLinkSetMtu(); err != nil { return }																										// Set the wireguard interface MTU
	if err = l.wgAddPeer( response.PublicKey, response.PresharedKey, response.Endpoint, response.PersistentKeepaliveInterval ); err != nil { return }		// Add a wireguard peer
	l.stack = append( l.stack, l.wgRemovePeer )
	if err = l.ipAddrsAdd( response.AllowedIps ); err != nil { l.Down(); return }																			// Add the IP addresses to the wireguard device
	l.stack = append( l.stack, l.ipAddrsDel )
	if err = l.ipRoutesAdd( response ); err != nil { l.Down(); return }																						// Add the default routes over the wireguard interface
	l.stack = append( l.stack, l.ipRoutesRemove )
	if err = l.dnsSet( response.DNS ); err != nil { l.Down(); return }																						// Set the DNS
	l.stack = append( l.stack, l.dnsRestore )
	log.Println( "Link: Up" )
	return
}

// Down undoes Up, removes the wireguard peer and un-routes it
func ( l *Link ) Down() {
	if rxBytes, txBytes, err := l.Acct(); err == nil { log.Println( "Link: Received", rxBytes, "bytes, transmitted", txBytes, "bytes" ) }
	for i := len( l.stack )-1; i >= 0; i-- { _ = l.stack[i]() }
	log.Println( "Link: Down" )
	return
}