/*
	This file is part of solidity.

	solidity is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	solidity is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with solidity.  If not, see <http://www.gnu.org/licenses/>.
*/
// SPDX-License-Identifier: GPL-3.0
/**
 * @date 2017
 * Abstract assembly interface, subclasses of which are to be used with the generic
 * bytecode generator.
 */

#pragma once

#include <libyul/ASTForward.h>

#include <libsolutil/Common.h>
#include <libsolutil/CommonData.h>
#include <libsolutil/Numeric.h>
#include <liblangutil/EVMVersion.h>

#include <functional>
#include <memory>
#include <optional>

namespace solidity::langutil
{
struct SourceLocation;
}

namespace solidity::evmasm
{
enum class Instruction: uint8_t;
}

namespace solidity::yul
{
struct Identifier;

///
/// Assembly class that abstracts both the libevmasm assembly and the new Yul assembly.
///
class AbstractAssembly
{
public:
	using LabelID = size_t;
	using SubID = size_t;
	using ContainerID = uint8_t;
	using FunctionID = uint16_t;
	enum class JumpType { Ordinary, IntoFunction, OutOfFunction };

	virtual ~AbstractAssembly() = default;

	/// Set a new source location valid starting from the next instruction.
	virtual void setSourceLocation(langutil::SourceLocation const& _location) = 0;
	/// Retrieve the current height of the stack. This does not have to be zero
	/// at the beginning.
	virtual int stackHeight() const = 0;
	virtual void setStackHeight(int height) = 0;
	/// Append an EVM instruction.
	virtual void appendInstruction(evmasm::Instruction _instruction) = 0;
	/// Append a constant.
	virtual void appendConstant(u256 const& _constant) = 0;
	/// Append a label.
	virtual void appendLabel(LabelID _labelId) = 0;
	/// Append a label reference.
	virtual void appendLabelReference(LabelID _labelId) = 0;
	/// Generate a new unique label.
	virtual LabelID newLabelId() = 0;
	/// Returns a label identified by the given name. Creates it if it does not yet exist.
	virtual LabelID namedLabel(std::string const& _name, size_t _params, size_t _returns, std::optional<size_t> _sourceID) = 0;
	/// Append a reference to a to-be-linked symbol.
	/// Currently, we assume that the value is always a 20 byte number.
	virtual void appendLinkerSymbol(std::string const& _name) = 0;

	/// Append raw bytes that stay untouched by the optimizer.
	virtual void appendVerbatim(bytes _data, size_t _arguments, size_t _returnVariables) = 0;

	/// Append a jump instruction.
	/// @param _stackDiffAfter the stack adjustment after this instruction.
	/// This is helpful to stack height analysis if there is no continuing control flow.
	virtual void appendJump(int _stackDiffAfter, JumpType _jumpType = JumpType::Ordinary) = 0;

	/// Append a jump-to-immediate operation.
	/// @param _stackDiffAfter the stack adjustment after this instruction.
	virtual void appendJumpTo(LabelID _labelId, int _stackDiffAfter = 0, JumpType _jumpType = JumpType::Ordinary) = 0;
	/// Append a jump-to-if-immediate operation.
	virtual void appendJumpToIf(LabelID _labelId, JumpType _jumpType = JumpType::Ordinary) = 0;

	/// Append the assembled size as a constant.
	virtual void appendAssemblySize() = 0;
	/// Creates a new sub-assembly, which can be referenced using dataSize and dataOffset.
	virtual std::pair<std::shared_ptr<AbstractAssembly>, SubID> createSubAssembly(bool _creation, std::string _name = "") = 0;

	/// Registers a new function with given signature and returns its ID.
	/// The function is initially empty and its body must be filled with instructions.
	virtual FunctionID registerFunction(uint8_t _args, uint8_t _rets) = 0;
	/// Selects a function as a target for newly appended instructions.
	/// May only be called after the main code section is already filled and
	/// must not be called when another function is already selected.
	/// Filling the same function more than once is not allowed.
	/// @a endFunction() must be called at the end to finalize the function.
	virtual void beginFunction(FunctionID _functionID) = 0;
	/// Finalizes the process of filling a function body and switches back to the main code section.
	/// Must not be called if no function is selected.
	/// Must be called after filling the main yul section
	virtual void endFunction() = 0;
	/// Appends function call to a function under given ID
	virtual void appendFunctionCall(FunctionID _functionID) = 0;
	/// Appends an instruction that returns values from the current function.
	/// Only allowed inside a function body.
	virtual void appendFunctionReturn() = 0;

	/// Appends the offset of the given sub-assembly or data.
	virtual void appendDataOffset(std::vector<SubID> const& _subPath) = 0;
	/// Appends the size of the given sub-assembly or data.
	virtual void appendDataSize(std::vector<SubID> const& _subPath) = 0;
	/// Appends the given data to the assembly and returns its ID.
	virtual SubID appendData(bytes const& _data) = 0;

	/// Appends loading an immutable variable.
	virtual void appendImmutable(std::string const& _identifier) = 0;
	/// Appends an assignment to an immutable variable.
	virtual void appendImmutableAssignment(std::string const& _identifier) = 0;

	/// Appends an operation that loads 32 bytes of data from a known offset relative to the start of the static_aux_data area of the EOF data section.
	/// Note that static_aux_data is only a part or the data section.
	/// It is preceded by the pre_deploy_data, whose size is not determined before the bytecode is assembled, and which cannot be accessed using this function.
	/// The function is meant to allow indexing into static_aux_data in a way that's independent of the size of pre_deploy_data.
	virtual void appendAuxDataLoadN(uint16_t _offset) = 0;

	/// Appends EOF contract creation instruction which takes creation code from subcontainer with _containerID.
	virtual void appendEOFCreate(ContainerID _containerID) = 0;
	/// Appends EOF contract return instruction which returns a subcontainer ID (_containerID) with auxiliary data filled in.
	virtual void appendReturnContract(ContainerID _containerID) = 0;
	/// Appends data to the very end of the bytecode. Repeated calls concatenate.
	/// EOF auxiliary data in data section and the auxiliary data are different things.
	virtual void appendToAuxiliaryData(bytes const& _data) = 0;

	/// Mark this assembly as invalid. Any attempt to request bytecode from it should throw.
	virtual void markAsInvalid() = 0;

	/// @returns the EVM version the assembly targets.
	virtual langutil::EVMVersion evmVersion() const = 0;
};

enum class IdentifierContext { LValue, RValue, VariableDeclaration, NonExternal };

/// Object that is used to resolve references and generate code for access to identifiers external
/// to inline assembly (not used in standalone assembly mode).
struct ExternalIdentifierAccess
{
	using Resolver = std::function<bool(Identifier const&, IdentifierContext, bool /*_crossesFunctionBoundary*/)>;
	/// Resolve an external reference given by the identifier in the given context.
	/// @returns the size of the value (number of stack slots) or size_t(-1) if not found.
	Resolver resolve;
	using CodeGenerator = std::function<void(Identifier const&, IdentifierContext, yul::AbstractAssembly&)>;
	/// Generate code for retrieving the value (rvalue context) or storing the value (lvalue context)
	/// of an identifier. The code should be appended to the assembly. In rvalue context, the value is supposed
	/// to be put onto the stack, in lvalue context, the value is assumed to be at the top of the stack.
	CodeGenerator generateCode;
};

}
