// Aleth: Ethereum C++ client, tools and libraries.
// Copyright 2015-2019 Aleth Authors.
// Licensed under the GNU General Public License, Version 3.

#include <test/tools/libtesteth/Stats.h>
#include <test/tools/libtesteth/Options.h>
#include <iterator>
#include <numeric>
#include <fstream>

namespace dev
{
namespace test
{

Stats& Stats::get()
{
	static Stats instance;
	return instance;
}

void Stats::suiteStarted(std::string const& _name)
{
	m_currentSuite = _name;
}

void Stats::testStarted(std::string const& _name)
{
	m_currentTest = _name;
	m_tp = clock::now();
}

void Stats::testFinished(int64_t _gasUsed)
{
	m_stats.push_back({clock::now() - m_tp, _gasUsed, m_currentSuite + "/" + m_currentTest});
}

std::ostream& operator<<(std::ostream& out, Stats::clock::duration const& d)
{
	return out << std::setw(10) << std::right << std::chrono::duration_cast<std::chrono::microseconds>(d).count() << " us";
}

Stats::~Stats()
{
	if (m_stats.empty())
		return;

	std::sort(m_stats.begin(), m_stats.end(), [](Stats::Item const& a, Stats::Item const& b){
		return a.duration < b.duration;
	});

	auto& out = std::cout;
	auto itr = m_stats.begin();
	auto min = *itr;
	auto max = *m_stats.rbegin();
	std::advance(itr, m_stats.size() / 2);
	auto med = *itr;
	auto tot = std::accumulate(m_stats.begin(), m_stats.end(), clock::duration{}, [](clock::duration const& a, Stats::Item const& v)
	{
		return a + v.duration;
	});

	out << "\nSTATS:\n\n" << std::setfill(' ');

	if (Options::get().statsOutFile == "out")
	{
		for (auto&& s: m_stats)
		{
			auto usecs = std::chrono::duration_cast<std::chrono::microseconds>(s.duration).count();
			out << "  " << std::setw(40) << std::left << s.name.substr(0, 40) << s.duration;
			if (s.gasUsed >= 0)
			{
				auto gasRate = uint64_t(double(s.gasUsed) * 1000 / usecs);
				out << "\t" << std::setw(10) << std::right  << gasRate << " gas/ms\n";
			}
			else
				out << "\tOOG\n";
		}
		out << "\n";
	}
	else if (!Options::get().statsOutFile.empty())
	{
		// Output stats to file
		std::ofstream file{Options::get().statsOutFile};
		for (auto&& s: m_stats)
		{
			auto usecs = std::chrono::duration_cast<std::chrono::microseconds>(s.duration).count();
			file << s.name << "\t" << usecs;
			if (s.gasUsed >= 0)
			{
				auto gasRate = s.gasUsed / usecs;
				file << "\t" << gasRate << " gas/us\n";
			}
			else
				file << "\tOOG\n";
		}
	}

	out	<< "  tot: " << tot << "\n"
		<< "  avg: " << (tot / m_stats.size()) << "\n\n"
		<< "  min: " << min.duration << " (" << min.name << ")\n"
		<< "  med: " << med.duration << " (" << med.name << ")\n"
		<< "  max: " << max.duration << " (" << max.name << ")\n";
}

}
}
