#include <fs/fd.h>
#include <net/socket.h>
#include <vector.h>
#include <cpu.h>
#include <sched/sched.h>
#include <errno.h>
#include <bitmap.h>
#include <string.h>
#include <fs/vfs.h>
#include <debug.h>
#include <time.h>
#include <mm/pmm.h>
#include <fs/cdev.h>
#include <events/io.h> 

int dirfd_lookup_vfs(int dirfd, const char *path, struct vfs_node **ret) {
	bool relative = *path != '/' ? true : false;

	if(relative) {
		if(dirfd == AT_FDCWD) {
			*ret = *CURRENT_TASK->cwd;
		} else {
			struct fd_handle *dir_handle = fd_translate(dirfd);
			if(dir_handle == NULL) {
				set_errno(EBADF);
				return -1;
			}
			*ret = dir_handle->file_handle->vfs_node;
		}
	} else {
		*ret = vfs_root;
	}

	if(*ret == NULL) {
		*ret = vfs_root;
	}

	return 0;
}

int user_lookup_at(int dirfd, const char *path, int lookup_flags, mode_t mode, struct vfs_node **ret) {
	if(*path == '/' && *(path + 1) == '\0') {
		*ret = vfs_root;
		return 0;
	}

	struct vfs_node *parent;
	if(dirfd_lookup_vfs(dirfd, path, &parent) == -1) {
		return -1;
	}

	bool symlink_follow = (lookup_flags & AT_SYMLINK_NOFOLLOW) == AT_SYMLINK_NOFOLLOW ? true : false;
	bool effective_ids = (lookup_flags & AT_EACCESS) == AT_EACCESS ? true : false;

	VECTOR(const char*) subpath_list = { 0 };

	char *str = alloc(strlen(path));
	strcpy(str, path);

	while(*str == '/') *str++ = 0;

	while(*str) {
		const char *subpath = str;

		while(*str && *str != '/') str++;
		while(*str == '/') *str++ = 0;

		VECTOR_PUSH(subpath_list, subpath);
	}

	size_t i = 0;
	for(; i < (subpath_list.length - 1); i++) {
		parent = vfs_search_relative(parent, subpath_list.data[i], true);
		if(parent == NULL) {
			set_errno(ENOENT);
			return -1;
		}
	}

	struct vfs_node *vfs_node;
	vfs_node = vfs_search_relative(parent, subpath_list.data[i], symlink_follow);
	if(vfs_node == NULL) {
		set_errno(ENOENT);
		return -1;
	}

	i = 0;
	for(; i < (subpath_list.length - 1); i++) {
		if(stat_has_access(parent->stat, CURRENT_TASK->effective_uid, CURRENT_TASK->effective_gid, X_OK) == -1) {
			set_errno(EACCES);
			return -1;
		}
	}

	if(stat_has_access(parent->stat, CURRENT_TASK->effective_uid, CURRENT_TASK->effective_gid, X_OK) == -1) {
		set_errno(EACCES);
		return -1;
	}

	uid_t uid = effective_ids ? CURRENT_TASK->effective_uid : CURRENT_TASK->real_uid;
	gid_t gid = effective_ids ? CURRENT_TASK->effective_gid : CURRENT_TASK->real_gid;

	struct stat *stat = vfs_node->stat;

	if(stat_has_access(stat, uid, gid, mode) == -1) {
		set_errno(EACCES);
		return -1;
	}

	*ret = vfs_node;

	return 0;
}

static ssize_t file_read(struct fd_handle *fd_handle, void *buffer, size_t cnt, off_t offset) {
	struct file_handle *file = fd_handle->file_handle;

	node_lock(file->vfs_node);
	struct stat *stat = file->stat;

	if(offset > stat->st_size) {
		node_unlock(file->vfs_node);
		return 0;
	}

	ssize_t ret = file->ops->read(file, buffer, cnt, offset);

	if(offset + cnt > stat->st_size) {
		cnt = stat->st_size - offset;
	}

	node_unlock(file->vfs_node);

	return ret;
}

static ssize_t file_write(struct fd_handle *fd_handle, const void *buffer, size_t cnt, off_t offset) {
	struct file_handle *file = fd_handle->file_handle;

	node_lock(file->vfs_node);
	struct stat *stat = file->stat;

	ssize_t ret = file->ops->write(file, buffer, cnt, offset);

	if(offset + cnt > stat->st_size) {
		stat->st_size = offset + cnt;
		stat->st_blocks = DIV_ROUNDUP(stat->st_size, stat->st_blksize);
	}

	node_unlock(file->vfs_node);

	return ret;
}

int stat_has_access(struct stat *stat, uid_t uid, gid_t gid, int mode) {
	if(uid == 0) {
		return 0;
	}

	mode_t mask_uid = 0, mask_gid = 0, mask_oth = 0;

	if(mode & R_OK) { mask_uid |= S_IRUSR; mask_gid |= S_IRGRP; mask_oth |= S_IROTH; }
	if(mode & W_OK) { mask_uid |= S_IWUSR; mask_gid |= S_IWGRP; mask_oth |= S_IWOTH; }
	if(mode & X_OK) { mask_uid |= S_IXUSR; mask_gid |= S_IXGRP; mask_oth |= S_IXOTH; }

	if(stat->st_uid == uid) {
		if((stat->st_mode & mask_uid) == mask_uid) {
			return 0;
		}
		return -1;
	} else if(stat->st_gid == gid) {
		if((stat->st_mode & mask_gid) == mask_gid) {
			return 0;
		}
		return -1;
	} else {
		if((stat->st_mode & mask_oth) == mask_oth) {
			return 0;
		}
		return -1;
	}
}

int stat_update_time(struct stat *stat, int flags) {
	if((flags & STAT_ACCESS) == STAT_ACCESS) stat->st_atim = clock_realtime;
	if((flags & STAT_MOD) == STAT_MOD) stat->st_mtim = clock_realtime;
	if((flags & STAT_STATUS) == STAT_STATUS) stat->st_ctim = clock_realtime;

	return 0;
}

struct fd_handle *fd_translate_unlocked(int index) {
	struct task *current_task = CURRENT_TASK;
	return hash_table_search(&current_task->fd_table->fd_list, &index, sizeof(index));
}

struct fd_handle *fd_translate(int index) {
	struct task *current_task = CURRENT_TASK;
	if(current_task == NULL) {
		return NULL;
	}

	spinlock_irqsave(&current_task->fd_table->fd_lock);
	struct fd_handle *handle = fd_translate_unlocked(index);
	spinrelease_irqsave(&current_task->fd_table->fd_lock);

	return handle;
}

off_t fd_seek(int fd, off_t offset, int whence) {
	struct fd_handle *fd_handle = fd_translate(fd);
	if(fd_handle == NULL) {
		set_errno(EBADF);
		return -1;
	}

	struct stat *stat = fd_handle->file_handle->stat;
	if(S_ISFIFO(stat->st_mode) || S_ISSOCK(stat->st_mode)) {
		set_errno(ESPIPE);
		return -1;
	}

	file_lock(fd_handle->file_handle);
	switch(whence) {
		case SEEK_SET:
			fd_handle->file_handle->position = offset;
			break;
		case SEEK_CUR:
			fd_handle->file_handle->position += offset;
			break;
		case SEEK_END:
			fd_handle->file_handle->position = stat->st_size + offset;
			break;
		default:
			file_unlock(fd_handle->file_handle);
			set_errno(EINVAL);
			return -1;
	}

	off_t pos =  fd_handle->file_handle->position;
	file_unlock(fd_handle->file_handle);
	return pos;
}

ssize_t fd_write(int fd, const void *buf, size_t count) {
	struct fd_handle *fd_handle = fd_translate(fd);
	if(fd_handle == NULL) {
		set_errno(EBADF);
		return -1;
	}

	file_lock(fd_handle->file_handle);
	struct stat *stat = fd_handle->file_handle->stat;

	if(fd_handle->file_handle->ops->write == NULL) {
		file_unlock(fd_handle->file_handle);
		set_errno(ENODEV);
		return -1;
	}

//	if((fd_handle->file_handle->flags & O_ACCMODE) != O_WRONLY
//		&& (fd_handle->file_handle->flags & O_ACCMODE) != O_RDWR) {
//		file_unlock(fd_handle->file_handle);
//		set_errno(EBADF);
//		return -1;
//	}

	if ((fd_handle->file_handle->flags & O_APPEND) && !(S_ISFIFO(stat->st_mode))) {
		if(!S_ISSOCK(stat->st_mode)) fd_handle->file_handle->position = stat->st_size;
	}

	ssize_t ret;
	off_t off = fd_handle->file_handle->position;

	file_unlock(fd_handle->file_handle);

	if(S_ISCHR(stat->st_mode) || S_ISSOCK(stat->st_mode)) {
		ret = fd_handle->file_handle->ops->write(fd_handle->file_handle, buf, count, off);
	} else {
		ret = file_write(fd_handle, buf, count, off);
	}

	file_lock(fd_handle->file_handle);

	if(ret != -1) {
		stat_update_time(stat, STAT_MOD | STAT_STATUS);

		if(!S_ISSOCK(stat->st_mode)) {
			fd_handle->file_handle->position += ret;
			fd_handle->file_handle->status |= POLLIN;
			waitq_arise(fd_handle->file_handle->trigger, CURRENT_TASK);
		}
	}

	file_unlock(fd_handle->file_handle);

	return ret;
}

ssize_t fd_read(int fd, void *buf, size_t count) {
	struct fd_handle *fd_handle = fd_translate(fd);
	if(fd_handle == NULL) {
		set_errno(EBADF);
		return -1;
	}

	file_lock(fd_handle->file_handle);
	struct stat *stat = fd_handle->file_handle->stat;
	if(S_ISDIR(stat->st_mode)) {
		file_unlock(fd_handle->file_handle);
		set_errno(EISDIR);
		return -1;
	}

	if(fd_handle->file_handle->ops->read == NULL) {
		file_unlock(fd_handle->file_handle);
		set_errno(ENODEV);
		return -1;
	}

//	if((fd_handle->file_handle->flags & O_ACCMODE) != O_RDONLY
//		&& (fd_handle->file_handle->flags & O_ACCMODE) != O_RDWR) {
//		file_unlock(fd_handle->file_handle);
//		set_errno(EBADF);
//		return -1;
//	}

	ssize_t ret;
	off_t off = fd_handle->file_handle->position;

	file_unlock(fd_handle->file_handle);

	if(S_ISCHR(stat->st_mode)) {
		ret = fd_handle->file_handle->ops->read(fd_handle->file_handle, buf, count, off);
	} else {
		ret = file_read(fd_handle, buf, count, off);
	}

	file_lock(fd_handle->file_handle);

	if(ret != -1 && !S_ISSOCK(stat->st_mode)) {
		fd_handle->file_handle->position += ret;
	}

	stat_update_time(stat, STAT_ACCESS);

	file_unlock(fd_handle->file_handle);

	return ret;
}

ssize_t pipe_read(struct file_handle *file, void *buf, size_t cnt, off_t offset) {
	struct stat *stat = file->stat;
	const void *out = file->pipe->buffer;

	if(offset > stat->st_size) {
		for(;;) {
			if((file->status & POLLIN) == POLLIN) {
				file->status &= ~POLLIN;
				break;
			}

			int ret = waitq_block(&file->waitq, NULL);
			if(ret == -1) {
				return -1;
			}
		}
	}

	stat_update_time(stat, STAT_ACCESS);

	if(offset + cnt > stat->st_size) {
		cnt = stat->st_size - offset;
	}

	memcpy8(buf, out + offset, cnt);
	offset += cnt;

	return cnt;
}

ssize_t pipe_write(struct file_handle *file, const void *buf, size_t cnt, off_t offset) {
	struct stat *stat = file->stat;
	void *out = file->pipe->buffer;

	if(offset >= PIPE_BUFFER_SIZE) {
		set_errno(EINVAL);
		return -1;
	}

	if(offset + cnt > PIPE_BUFFER_SIZE) {
		cnt = stat->st_size - offset;
	}

	if(offset > stat->st_size) {
		file->status |= POLLIN;
		waitq_arise(file->trigger, CURRENT_TASK);
	}

	stat_update_time(stat, STAT_MOD);

	if(offset + cnt > stat->st_size) {
		stat->st_size += offset + cnt - stat->st_size;
	}

	memcpy8(out + offset, buf, cnt);

	return cnt;
}

int fd_unlinkat(int dirfd, const char *path, int flags) {
	if(strlen(path) > MAX_PATH_LENGTH) {
		set_errno(ENAMETOOLONG);
		return -1;
	}

	struct vfs_node *vfs_node;
	if(user_lookup_at(dirfd, path, flags, W_OK, &vfs_node) == -1) {
		return -1;
	}

	if(S_ISDIR(vfs_node->stat->st_mode)) {
		set_errno(EISDIR);
		return -1;
	}

	vfs_node->fops->unlink(vfs_node);
	vfs_unlink(vfs_node);

	return 0;
}

int fd_openat(int dirfd, const char *path, int flags, mode_t mode) {
	if(strlen(path) > MAX_PATH_LENGTH) {
		set_errno(ENAMETOOLONG);
		return -1;
	}

	mode &= (S_IRWXU | S_IRWXG | S_IRWXO | S_ISVTX | S_ISUID | S_ISGID);
	if((flags & O_ACCMODE) == 0)
		flags |= O_RDONLY;

	int access_mode = 0;
	if((flags & O_ACCMODE) == O_RDONLY) {
		access_mode = R_OK;
	} else if((flags & O_ACCMODE) == O_WRONLY) {
		access_mode = W_OK;
	} else if((flags & O_ACCMODE) == O_RDWR) {
		access_mode = R_OK | W_OK;
	} else {
		set_errno(EINVAL);
		return -1;
	}

	if((flags & O_TRUNC) && !(access_mode & W_OK)) {
		set_errno(EINVAL);
		return -1;
	}

	bool symfollow = (flags & AT_SYMLINK_NOFOLLOW) == AT_SYMLINK_NOFOLLOW ? false : true;

	struct vfs_node *dir;
	if(dirfd_lookup_vfs(dirfd, path, &dir) == -1) {
		return -1;
	}

	struct vfs_node *vfs_node = vfs_search_absolute(dir, path, symfollow);

	if(flags & O_CREAT && vfs_node == NULL) {
		int cutoff = find_last_char(path, '/');

		struct vfs_node *parent;
		char *name;

		if(cutoff == -1) {
			name = alloc(strlen(path) + 1);
			strcpy(name, path);
			parent = dir;
		} else {
			name = alloc(strlen(path + cutoff) + 1);
			strcpy(name, path + cutoff + 1);

			char *dirpath = alloc(cutoff + 1);
			strncpy(dirpath, path, cutoff);

			parent = vfs_search_absolute(dir, dirpath, symfollow);
			if(parent == NULL) {
				set_errno(ENOTDIR);
				return -1;
			}
		}

		if(stat_has_access(parent->stat, CURRENT_TASK->effective_uid, CURRENT_TASK->effective_gid, W_OK | X_OK) == -1) {
			set_errno(EACCES);
			return -1;
		}

		struct stat *stat = alloc(sizeof(struct stat));
		stat_init(stat);
		stat->st_mode = S_IFREG | (mode & ~(*CURRENT_TASK->umask));
		stat->st_uid = CURRENT_TASK->effective_uid;

		stat_update_time(stat, STAT_ACCESS | STAT_MOD | STAT_STATUS);

		vfs_node = vfs_create(parent, name, stat);

		if(parent->stat->st_mode & S_ISGID) {
			vfs_node->stat->st_gid = parent->stat->st_gid;
		} else {
			vfs_node->stat->st_gid = CURRENT_TASK->effective_gid;
		}
	} else if((flags & O_CREAT) && (flags & O_EXCL)) {
		set_errno(EEXIST);
		return -1;
	} else if(vfs_node == NULL) {
		set_errno(ENOENT);
		return -1;
	}

	if(!(flags & O_DIRECTORY) && S_ISDIR(vfs_node->stat->st_mode)) {
		set_errno(EISDIR);
		return -1;
	}

	if(stat_has_access(vfs_node->stat, CURRENT_TASK->effective_uid, CURRENT_TASK->effective_gid, access_mode) == -1) {
		set_errno(EACCES);
		return -1;
	}

	if((flags & O_TRUNC) && vfs_node->filesystem->truncate) {
		vfs_truncate(vfs_node, 0);
		stat_update_time(vfs_node->stat, STAT_MOD | STAT_STATUS);
	}

	struct file_ops *fops = vfs_node->fops;
	struct file_handle *new_file_handle = alloc(sizeof(struct file_handle));
	file_init(new_file_handle);
	new_file_handle->vfs_node = vfs_node;
	new_file_handle->ops = fops;
	new_file_handle->flags = flags & ~O_CLOEXEC;
	new_file_handle->stat = vfs_node->stat;
	new_file_handle->trigger = EVENT_DEFAULT_TRIGGER(&new_file_handle->waitq);

	if(S_ISCHR(vfs_node->stat->st_mode)) {
		if(cdev_open(vfs_node, new_file_handle, flags) == -1) {
			file_put(new_file_handle);
			return -1;
		}
	} else {
		if(fops->open) {
			if(fops->open(vfs_node, new_file_handle, flags) == -1) {
				file_put(new_file_handle);
				return -1;
			}
		}
	}

	stat_update_time(vfs_node->stat, STAT_ACCESS);

	struct fd_handle *new_fd_handle = alloc(sizeof(struct fd_handle));
	fd_init(new_fd_handle);
	new_fd_handle->fd_number = bitmap_alloc(&CURRENT_TASK->fd_table->fd_bitmap);
	new_fd_handle->file_handle = new_file_handle;
	new_fd_handle->flags = (flags & O_CLOEXEC) ? FD_CLOEXEC : 0;

	struct task *current_task = CURRENT_TASK;
	if(current_task == NULL) {
		set_errno(ENOENT);
		return -1;
	}

	hash_table_push(&current_task->fd_table->fd_list, &new_fd_handle->fd_number, new_fd_handle, sizeof(new_fd_handle->fd_number));

	return new_fd_handle->fd_number;
}

static void fd_close_unlocked(struct fd_handle *handle) {
	struct task *current_task = CURRENT_TASK;

	if(handle->file_handle == NULL) {
		return;
	}

	if(handle->file_handle->ops->close) {
		handle->file_handle->ops->close(handle->file_handle->vfs_node, handle->file_handle);
	}

	file_put(handle->file_handle);
	hash_table_delete(&current_task->fd_table->fd_list, &handle->fd_number, sizeof(handle->fd_number));
	bitmap_free(&current_task->fd_table->fd_bitmap, handle->fd_number);
	free(handle);
}

int fd_close(int fd) {
	struct task *current_task = CURRENT_TASK;

	spinlock_irqsave(&current_task->fd_table->fd_lock);
	struct fd_handle *fd_handle = fd_translate_unlocked(fd);
	if(fd_handle == NULL) {
		spinrelease_irqsave(&current_task->fd_table->fd_lock);
		set_errno(EBADF);
		return -1;
	}

	if(current_task == NULL) {
		spinrelease_irqsave(&current_task->fd_table->fd_lock);
		set_errno(ENOENT);
		return -1;
	}

	fd_close_unlocked(fd_handle);
	spinrelease_irqsave(&current_task->fd_table->fd_lock);

	return 0;
}

int fd_stat(int fd, void *buffer) {
	struct fd_handle *fd_handle = fd_translate(fd);
	if(fd_handle == NULL) {
		set_errno(EBADF);
		return -1;
	}

	struct stat *stat = buffer;
	*stat = *fd_handle->file_handle->stat;
	return 0;
}

int fd_statat(int dirfd, const char *path, void *buffer, int flags) {
	if(!strlen(path) && !(flags & AT_EMPTY_PATH)) {
		set_errno(ENOENT);
		return -1;
	}

	bool symfollow = (flags & AT_SYMLINK_NOFOLLOW) == AT_SYMLINK_NOFOLLOW ? false : true;
	struct vfs_node *vfs_node;

	if(flags & AT_EMPTY_PATH) {
		struct fd_handle *fd_handle = fd_translate(dirfd);
		if(fd_handle == NULL) {
			set_errno(EBADF);
			return -1;
		}

		vfs_node = fd_handle->file_handle->vfs_node;
	} else {
		int relative = *path == '/' ? 0 : 1;
		struct vfs_node *dir;

		if(!relative) {
			dir = vfs_root;
		} else {
			if(dirfd == AT_FDCWD) {
				dir = *CURRENT_TASK->cwd;
			} else {
				struct fd_handle *fd_handle = fd_translate(dirfd);
				if(fd_handle == NULL) {
					set_errno(EBADF);
					return -1;
				}

				if(!S_ISDIR(fd_handle->file_handle->stat->st_mode)) {
					set_errno(EBADF);
					return -1;
				}

				dir = fd_handle->file_handle->vfs_node;
			}
		}

		vfs_node = vfs_search_absolute(dir, path, symfollow);
		if(vfs_node == NULL) {
			set_errno(ENOENT);
			return -1;
		}
	}

	struct stat *stat = buffer;
	*stat = *vfs_node->stat;

	return 0;
}

int fd_dup(int fd, bool clear_cloexec) {
	struct task *current_task = CURRENT_TASK;
	spinlock_irqsave(&current_task->fd_table->fd_lock);

	struct fd_handle *fd_handle = fd_translate_unlocked(fd);
	if(fd_handle == NULL) {
		spinrelease_irqsave(&current_task->fd_table->fd_lock);
		set_errno(EBADF);
		return -1;
	}

	struct fd_handle *handle = alloc(sizeof(struct fd_handle));
	*handle = *fd_handle;
	handle->fd_number = bitmap_alloc(&current_task->fd_table->fd_bitmap);

	if (clear_cloexec)
		handle->flags &= ~FD_CLOEXEC;
	else
		handle->flags |= FD_CLOEXEC;

	file_get(handle->file_handle);
	hash_table_push(&current_task->fd_table->fd_list, &handle->fd_number, handle, sizeof(handle->fd_number));
	spinrelease_irqsave(&current_task->fd_table->fd_lock);

	return handle->fd_number;
}

int fd_dup2(int oldfd, int newfd) {
	struct task *current_task = CURRENT_TASK;
	spinlock_irqsave(&current_task->fd_table->fd_lock);

	struct fd_handle *oldfd_handle = fd_translate_unlocked(oldfd), *new_handle;;
	if(oldfd_handle == NULL) {
		spinrelease_irqsave(&current_task->fd_table->fd_lock);
		set_errno(EBADF);
		return -1;
	}

	if(oldfd == newfd) {
		spinrelease_irqsave(&current_task->fd_table->fd_lock);
		return newfd;
	}

	new_handle = alloc(sizeof(struct fd_handle));
	*new_handle = *oldfd_handle;
	new_handle->fd_number = newfd;
	new_handle->flags &= ~FD_CLOEXEC;
	file_get(new_handle->file_handle);

	if(BIT_TEST(current_task->fd_table->fd_bitmap.data, newfd)) {
		fd_close_unlocked(fd_translate_unlocked(newfd));
	}

	BIT_SET(current_task->fd_table->fd_bitmap.data, newfd);

	hash_table_push(&current_task->fd_table->fd_list, &new_handle->fd_number, new_handle, sizeof(new_handle->fd_number));

	spinrelease_irqsave(&current_task->fd_table->fd_lock);

	return new_handle->fd_number;
}

int fd_generate_dirent(struct fd_handle *dir_handle, struct vfs_node *node, struct dirent *entry) {
	if(!S_ISDIR(dir_handle->file_handle->stat->st_mode)) {
		set_errno(ENOTDIR);
		return -1;
	}

	strcpy(entry->d_name, node->name);
	entry->d_ino = node->stat->st_ino;
	entry->d_off = 0;
	entry->d_reclen = sizeof(struct dirent);

	switch(node->stat->st_mode & S_IFMT) {
		case S_IFCHR:
			entry->d_type = DT_CHR;
			break;
		case S_IFBLK:
			entry->d_type = DT_BLK;
			break;
		case S_IFDIR:
			entry->d_type = DT_DIR;
			break;
		case S_IFLNK:
			entry->d_type = DT_LNK;
			break;
		case S_IFIFO:
			entry->d_type = DT_FIFO;
			break;
		case S_IFREG:
			entry->d_type = DT_REG;
			break;
		case S_IFSOCK:
			entry->d_type = DT_SOCK;
			break;
		default:
			entry->d_type = DT_UNKNOWN;
	}

	return 0;
}


int fd_fchownat(int fd, const char *path, uid_t uid, gid_t gid, int flag) {
	struct vfs_node *node;

	// Restricted chown: only root may change the owner.
	if(CURRENT_TASK->effective_uid != 0) {
		set_errno(EPERM);
		return -1;
	}

	if(uid == -1 && gid == -1)
		return 0;

	if(!(flag & AT_EMPTY_PATH) && !strlen(path)) {
		set_errno(EINVAL);
		return -1;
	}

	if(flag & AT_EMPTY_PATH) {
		struct fd_handle *handle = fd_translate(fd);
		if(!handle) {
			set_errno(EBADF);
			return -1;
		}

		node = handle->file_handle->vfs_node;
	} else {
		// We are only interested in the node and we are superuser, so with a mode of 0
		// we can get away with it.
		if(user_lookup_at(fd, path, flag & AT_SYMLINK_NOFOLLOW, 0, &node) == -1) {
			return -1;
		}
	}

	if(uid != -1)
		node->stat->st_uid = uid;
	if(gid != -1)
		node->stat->st_gid = gid;

	stat_update_time(node->stat, STAT_STATUS);

	return 0;
}

int fd_poll(struct pollfd *fds, nfds_t nfds, struct timespec *timespec) {
	struct waitq waitq = { 0 };

	if(timespec) {
		waitq_set_timer(&waitq, timespec);
	}

	VECTOR(struct file_handle*) handle_list = { 0 };

	for(size_t i = 0; i < nfds; i++) {
		struct pollfd *pollfd = &fds[i];

		struct fd_handle *fd_handle = fd_translate(pollfd->fd);
		if(fd_handle == NULL) {
			set_errno(EBADF);
			return -1;
		}

		struct file_handle *file_handle = fd_handle->file_handle;

		print("polling fd %x for events %x %s\n", pollfd->fd, pollfd->events, vfs_absolute_path(file_handle->vfs_node));

		waitq_add(&waitq, file_handle->trigger);
		VECTOR_PUSH(handle_list, file_handle);
	}

	int ret = 0;

	for(;;) {
		if(waitq.timer_trigger && waitq.timer_trigger->fired) {
			break;
		}

		for(size_t i = 0; i < handle_list.length; i++) {
			struct file_handle *handle = handle_list.data[i];

			print("Checking on %d %d %x\n", i, handle->status, handle); 

			if(handle->status & fds[i].events) {
				fds[i].revents = handle->status & fds[i].events;
				ret++;
			}
		}

		if(ret) {
			break;
		}

		if(waitq_block(&waitq, NULL) == -1) {
			ret = -1;
			break;
		}
	}

	for(size_t i = 0; i < handle_list.length; i++) {
		struct file_handle *handle = handle_list.data[i];
		waitq_remove(&waitq, handle->trigger);
	}

	return ret;
}

void syscall_dup2(struct registers *regs) {
	int oldfd = regs->rdi;
	int newfd = regs->rsi;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] dup2: oldfd {%x}, newfd {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, oldfd, newfd);
#endif

	regs->rax = fd_dup2(oldfd, newfd);
}

void syscall_dup(struct registers *regs) {
	int fd = regs->rdi;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] dup: fd {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fd);
#endif

	regs->rax = fd_dup(fd, true);
}

void syscall_stat(struct registers *regs) {
	int fd = regs->rdi;
	void *buf = (void*)regs->rsi;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] stat: fd {%x}, buf {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fd, (uintptr_t)buf);
#endif

	regs->rax = fd_stat(fd, buf);
}

void syscall_statat(struct registers *regs) {
	int dirfd = regs->rdi;
	const char *path = (void*)regs->rsi;
	void *buf = (void*)regs->rdx;
	int flags = regs->r10;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] statat: dirfd {%x}, path {%s}, buf {%x}, flags {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, dirfd, path, (uintptr_t)buf, flags);
#endif

	regs->rax = fd_statat(dirfd, path, buf, flags);
}

void syscall_write(struct registers *regs) {
	int fd = regs->rdi;
	const void *buf = (const void*)regs->rsi;
	size_t cnt = regs->rdx;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] write: fd {%x}, buf {%x}, cnt {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fd, (uintptr_t)buf, cnt);
#endif

	regs->rax = fd_write(fd, buf, cnt);
}

void syscall_read(struct registers *regs) {
	int fd = regs->rdi;
	void *buf = (void*)regs->rsi;
	size_t cnt = regs->rdx;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] read: fd {%x}, buf {%x}, cnt {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fd, (uintptr_t)buf, cnt);
#endif

	regs->rax = fd_read(fd, buf, cnt);
}

void syscall_seek(struct registers *regs) {
	int fd = regs->rdi;
	off_t offset = regs->rsi;
	int whence = regs->rdx;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] seek: fd {%x}, offset {%x}, whence {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fd, offset, whence);
#endif

	regs->rax = fd_seek(fd, offset, whence);
}

void syscall_unlinkat(struct registers *regs) {
	int dirfd = regs->rdi; 
	const char *pathname = (const char *)regs->rsi; 
	int flags = regs->rdx;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] unlinkat: dirfd {%x}, pathname {%s}, flags {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, dirfd, pathname, flags);
#endif

	regs->rax = fd_unlinkat(dirfd, pathname, flags);
}

void syscall_openat(struct registers *regs) {
	int dirfd = regs->rdi;
	const char *pathname = (const char*)regs->rsi;
	int flags = regs->rdx;
	mode_t mode = regs->r10;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] open: dirfd {%x}, pathname {%s}, flags {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, dirfd, pathname, flags);
#endif

	regs->rax = fd_openat(dirfd, pathname, flags, mode);

//	print("sycall return: [pid %x, tid %x] open: %x %x\n", CORE_LOCAL->pid, CORE_LOCAL->tid, regs->rax, regs->rax == -1 ? get_errno() : 0);
}

void syscall_close(struct registers *regs) {
	int fd = regs->rdi;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] close: fd {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fd);
#endif

	regs->rax = fd_close(fd);
}

void syscall_fcntl(struct registers *regs) {
#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] fcntl: fd {%x}, cmd {%x}, data {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, regs->rdi, regs->rsi, regs->rdx);
#endif

	struct fd_handle *fd_handle = fd_translate(regs->rdi);
	if(fd_handle == NULL) {
		set_errno(EBADF);
		regs->rax = -1;
		return;
	}

	switch(regs->rsi) {
		case F_DUPFD:
			regs->rax = fd_dup(regs->rdi, true);
			break;
		case F_DUPFD_CLOEXEC:
			regs->rax = fd_dup(regs->rdi, false);
			break;
		case F_GETFD:
			fd_lock(fd_handle);
			regs->rax = fd_handle->flags;
			fd_unlock(fd_handle);
			break;
		case F_SETFD:
			fd_lock(fd_handle);
			fd_handle->flags = regs->rdx;
			regs->rax = 0;
			fd_unlock(fd_handle);
			break;
		case F_GETFL:
			file_lock(fd_handle->file_handle);
			regs->rax = fd_handle->file_handle->flags;
			file_unlock(fd_handle->file_handle);
			break;
		case F_SETFL: {
			/*if (regs->rdx & O_ACCMODE) {
				// It is disallowed to change the access mode.
				set_errno(EINVAL);
				regs->rax = -1;
				break;
			}*/
			file_lock(fd_handle->file_handle);
			fd_handle->file_handle->flags = regs->rdx;
			regs->rax = 0;
			file_unlock(fd_handle->file_handle);
			break;
		}
		default:
			print("fnctl unknown command %x\n", regs->rsi);
			set_errno(EINVAL);
			regs->rax = -1;
	}
}

void syscall_readdir(struct registers *regs) {
	int fd = regs->rdi;
	struct dirent *buf = (void*)regs->rsi;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] readdir: fd {%x}, buf {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fd, (uintptr_t)buf);
#endif

	struct fd_handle *dir_handle = fd_translate(fd);
	if(dir_handle == NULL) {
		set_errno(EBADF);
		regs->rax = -1;
		return;
	}

	struct vfs_node *dir = dir_handle->file_handle->vfs_node;

	if(!S_ISDIR(dir->stat->st_mode)) {
		set_errno(ENOTDIR);
		regs->rax = -1;
		return;
	}

	if(dir->refresh) {
		vfs_refresh(dir);
		dir->refresh = 0;
	}

	if(dir->mountpoint) {
		dir = dir->mountpoint;
	}

	if((dir->children.length >= dir_handle->file_handle->current_dirent) && dir->children.length != dir_handle->file_handle->dirent_list.length) {
		VECTOR_CLEAR(dir_handle->file_handle->dirent_list);
		dir_handle->file_handle->current_dirent = 0;
	}

	if(!dir_handle->file_handle->dirent_list.length) {
		for(size_t i = 0; i < dir->children.length; i++) {
			struct vfs_node *node = vfs_get_node(dir, i);
			if(node == NULL) {
				regs->rax = -1;
				return;
			}

			struct dirent *entry = alloc(sizeof(struct dirent));

			int ret = fd_generate_dirent(dir_handle, node, entry);
			if(ret == -1) {
				regs->rax = -1;
				return;
			}

			VECTOR_PUSH(dir_handle->file_handle->dirent_list, entry);
		}
	}

	if(dir_handle->file_handle->current_dirent >= dir_handle->file_handle->dirent_list.length) {
		set_errno(0);
		regs->rax = -1;
		return;
	}

	*buf = *dir_handle->file_handle->dirent_list.data[dir_handle->file_handle->current_dirent];
	dir_handle->file_handle->current_dirent++;

	regs->rax = 0;
}

void syscall_getcwd(struct registers *regs) {
	char *buf = (void*)regs->rdi;
	size_t size = regs->rsi;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] getcwd: buf {%x}, size {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, buf, size);
#endif

	const char *path = vfs_absolute_path(*CURRENT_TASK->cwd);
	if(strlen(path) <= size) {
		memcpy8((void*)buf, (void*)path, strlen(path));
	} else {
		set_errno(ERANGE);
		regs->rax = 0;
		return;
	}

	regs->rax = (uintptr_t)buf;
}

void syscall_chdir(struct registers *regs) {
	const char *path = (const char*)regs->rdi;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] chdir: path {%s}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, path);
#endif

	struct vfs_node *node;
	if (user_lookup_at(AT_FDCWD, path, 0, X_OK, &node) == -1) {
		regs->rax = -1;
		return;
	}

	if(!S_ISDIR(node->stat->st_mode)) {
		set_errno(ENOTDIR);
		regs->rax = -1;
		return;
	}

	*CURRENT_TASK->cwd = node;

	regs->rax = 0;
}

void syscall_mkdirat(struct registers *regs) {
	int dirfd = regs->rdi;
	const char *pathname = (void*)regs->rsi;
	mode_t mode = regs->rdx;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] mkdirat: dirfd {%x}, pathname {%s}, mode {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, dirfd, pathname, mode);
#endif

	struct vfs_node *dir;
	if(dirfd_lookup_vfs(dirfd, pathname, &dir) == -1) {
		regs->rax = -1;
		return;
	}

	char *pathname_copy = alloc(strlen(pathname));
	strcpy(pathname_copy, pathname);

	while(*pathname_copy == '/') pathname_copy++;

	int cutoff = find_last_char(pathname_copy, '/');
	const char *name; 
	const char *dirpath;

	if(cutoff == -1) {
		name = pathname_copy;
		dirpath = "./";
	} else {
		name = pathname_copy + cutoff + 1;
		dirpath = pathname_copy;
		pathname_copy[cutoff] = '\0';
	}

	struct vfs_node *pathname_parent;
	if(user_lookup_at(dirfd, dirpath, AT_SYMLINK_NOFOLLOW, X_OK, &pathname_parent) == -1) {
		regs->rax = -1;
		return;
	}

	struct vfs_node *dir_node = vfs_search_relative(pathname_parent, name, false);
	if(dir_node) {
		set_errno(EEXIST);
		regs->rax = -1; 
		return;
	}

	struct stat *stat = alloc(sizeof(struct stat)); 
	stat_init(stat);
	stat->st_mode = S_IFDIR | (mode & ~(*CURRENT_TASK->umask));
	stat->st_uid = CURRENT_TASK->effective_uid;

	dir_node = vfs_create(pathname_parent, name, stat);

	if(pathname_parent->stat->st_mode & S_ISGID) {
		dir_node->stat->st_gid = pathname_parent->stat->st_gid;
	} else {
		dir_node->stat->st_gid = CURRENT_TASK->effective_gid;
	}

	regs->rax = 0;
}

void syscall_pipe(struct registers *regs) {
	int *fd_pair = (int*)regs->rdi;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] pipe: fd pair {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fd_pair);
#endif

	fd_pair[0] = bitmap_alloc(&CURRENT_TASK->fd_table->fd_bitmap);
	fd_pair[1] = bitmap_alloc(&CURRENT_TASK->fd_table->fd_bitmap);

	print("Returning %d:%d\n", fd_pair[0], fd_pair[1]);

	struct fd_handle *read_fd_handle = alloc(sizeof(struct fd_handle));
	struct fd_handle *write_fd_handle = alloc(sizeof(struct fd_handle));
	struct file_handle *read_file_handle = alloc(sizeof(struct file_handle));
	struct file_handle *write_file_handle = alloc(sizeof(struct file_handle));

	fd_init(read_fd_handle);
	fd_init(write_fd_handle);
	file_init(read_file_handle);
	file_init(write_file_handle);

	struct pipe *pipe = alloc(sizeof(struct pipe));
	*pipe = (struct pipe) {
		.read = read_file_handle,
		.write = write_file_handle,
		.buffer = (void*)(pmm_alloc(DIV_ROUNDUP(PIPE_BUFFER_SIZE, PAGE_SIZE), 1) + HIGH_VMA)
	};

	struct file_ops *read_ops = alloc(sizeof(struct file_ops));
	read_ops->read = pipe_read;

	struct file_ops *write_ops = alloc(sizeof(struct file_ops));
	write_ops->write = pipe_write;

	struct stat *pipe_stat = alloc(sizeof(struct stat));
	stat_init(pipe_stat);
	pipe_stat->st_mode = S_IFIFO | S_IWUSR | S_IRUSR;

	// Do we want to support full duplex pipes? If so,
	// make both ends readable and writable.
	read_fd_handle->fd_number = fd_pair[0];
	read_fd_handle->file_handle = read_file_handle;
	read_file_handle->ops = read_ops;
	read_file_handle->pipe = pipe;
	read_file_handle->flags = O_RDONLY;
	read_file_handle->stat = pipe_stat;
	read_file_handle->trigger = EVENT_DEFAULT_TRIGGER(&read_file_handle->waitq);

	write_fd_handle->fd_number = fd_pair[1];
	write_fd_handle->file_handle = write_file_handle;
	write_file_handle->ops = write_ops;
	write_file_handle->pipe = pipe;
	write_file_handle->flags = O_WRONLY;
	write_file_handle->stat = pipe_stat;
	write_file_handle->trigger = EVENT_DEFAULT_TRIGGER(&write_file_handle->waitq);

	stat_update_time(pipe_stat, STAT_ACCESS | STAT_MOD | STAT_STATUS);

	spinlock_irqsave(&CURRENT_TASK->fd_table->fd_lock);
	hash_table_push(&CURRENT_TASK->fd_table->fd_list, &read_fd_handle->fd_number, read_fd_handle, sizeof(read_fd_handle->fd_number));
	hash_table_push(&CURRENT_TASK->fd_table->fd_list, &write_fd_handle->fd_number, write_fd_handle, sizeof(write_fd_handle->fd_number));
	spinrelease_irqsave(&CURRENT_TASK->fd_table->fd_lock);

	regs->rax = 0;
}

void syscall_faccessat(struct registers *regs) {
	int dirfd = regs->rdi;
	const char *path = (const char*)regs->rsi;
	int mode = regs->rdx;
	int flags = regs->r10;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] faccessat: dirfd {%x}, path {%s}, mode {%x}, flags {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, dirfd, path, mode, flags);
#endif

	if (!(mode & F_OK) && !(mode & (R_OK | W_OK | X_OK))) {
		regs->rax = -1;
		return;
	}

	int lookup_flags = 0;

	if(mode == F_OK) mode = 0;
	if((flags & AT_SYMLINK_NOFOLLOW) == AT_SYMLINK_NOFOLLOW) lookup_flags |= AT_SYMLINK_NOFOLLOW;
	if((flags & AT_EMPTY_PATH) == AT_EMPTY_PATH) lookup_flags |= AT_EMPTY_PATH;

	struct vfs_node *node;
	if(user_lookup_at(dirfd, path, lookup_flags, mode, &node) == -1) {
		regs->rax = -1;
		return;
	}

	regs->rax = 0;
}

void syscall_readlinkat(struct registers *regs) {
	const char *pathname = (void*)regs->rdi;
	int dirfd = regs->rsi;
	char *buf = (void*)regs->rdx;
	size_t bufsize = regs->r10;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] readlinkat: pathname {%s}, dirfd {%x}, buf {%x}, bufsize {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, pathname, dirfd, buf, bufsize);
#endif

	struct vfs_node *parent; 
	if(dirfd_lookup_vfs(dirfd, pathname, &parent) == -1) { 
		regs->rax = -1; 
		return;
	}

	struct vfs_node *vfs_node = vfs_search_absolute(parent, pathname, false);
	if(vfs_node == NULL) {
		set_errno(EINVAL);
		regs->rax = -1; 
		return;
	}

	if(!S_ISLNK(vfs_node->stat->st_mode) || vfs_node->symlink == NULL) {
		set_errno(EINVAL);
		regs->rax = -1;
		return;
	}

	int pathlength = strlen(vfs_node->symlink) > bufsize ? bufsize : strlen(vfs_node->symlink);
	strncpy(buf, vfs_node->symlink, pathlength);

	regs->rax = pathlength;
}

void syscall_symlinkat(struct registers *regs) {
	const char *target = (const char*)regs->rdi;
	int newdirfd = regs->rsi;
	const char *linkpath = (const char*)regs->rdx;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] symlinkat: target {%s}, newdirfd {%x}, linkpath {%s}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, target, newdirfd, linkpath);
#endif

	char *linkpath_copy = alloc(strlen(linkpath));
	strcpy(linkpath_copy, linkpath);

	while(*linkpath_copy == '/') linkpath_copy++;

	int cutoff = find_last_char(linkpath_copy, '/');
	const char *name; 
	const char *dirpath;

	if(cutoff == -1) {
		name = linkpath_copy;
		dirpath = "./";
	} else {
		name = linkpath_copy + cutoff + 1;
		dirpath = linkpath_copy;
		linkpath_copy[cutoff] = '\0';
	}

	struct vfs_node *linkpath_parent;
	if(user_lookup_at(newdirfd, dirpath, AT_SYMLINK_NOFOLLOW, X_OK, &linkpath_parent) == -1) {
		regs->rax = -1;
		return;
	}

	struct vfs_node *linkpath_node = vfs_search_relative(linkpath_parent, name, false);
	if(linkpath_node) { 
		set_errno(EEXIST); 
		regs->rax = -1 ; 
		return; 
	}

	struct stat *stat = alloc(sizeof(struct stat));
	stat_init(stat); 
	stat->st_mode = S_IFLNK;
	stat->st_uid = CURRENT_TASK->effective_uid;

	linkpath_node = vfs_create(linkpath_parent, name, stat);

	if(linkpath_parent->stat->st_mode & S_ISGID) {
		linkpath_node->stat->st_gid = linkpath_parent->stat->st_gid; 
	} else {
		linkpath_node->stat->st_gid = CURRENT_TASK->effective_gid;
	}

	char *path = alloc(strlen(target));
	strcpy(path, target);

	linkpath_node->symlink = path;

	regs->rax = 0;
}

void syscall_ioctl(struct registers *regs) {
	int fd = regs->rdi;
	uint64_t req = regs->rsi;
	void *args = (void*)regs->rdx;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] ioctl: fd {%x}, req {%x}, args {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fd, req, args);
#endif

	struct fd_handle *fd_handle = fd_translate(fd);
	if(fd_handle == NULL) {
		set_errno(EBADF);
		regs->rax = -1;
		return;
	}

	if(fd_handle->file_handle->ops->ioctl == NULL) {
		set_errno(ENOTTY);
		regs->rax = -1;
		return;
	}

	regs->rax = fd_handle->file_handle->ops->ioctl(fd_handle->file_handle, req, args);
}

void syscall_umask(struct registers *regs) {
	mode_t mask = regs->rdi & 0777;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] umask: mask {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, mask);
#endif

	regs->rax = *CURRENT_TASK->umask;
	*CURRENT_TASK->umask = mask;
}

static int stat_chmod(struct stat *stat, mode_t mode) {
	if(CURRENT_TASK->effective_uid != stat->st_uid
		&& CURRENT_TASK->effective_uid != 0) {
		set_errno(EPERM);
		return -1;
	}

	stat->st_mode |= (mode & ( S_IRWXU | S_IRWXG | S_IRWXO | S_ISUID | S_ISGID | S_ISVTX));

	stat_update_time(stat, STAT_STATUS);

	return 0;
}

void syscall_fchmod(struct registers *regs) {
	int fd = regs->rdi;
	mode_t mode = regs->rsi;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] fchmod: fd {%x}, mode {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fd, mode);
#endif

	struct fd_handle *handle = fd_translate(fd);
	if(handle == NULL) {
		set_errno(EBADF);
		regs->rax = -1;
		return;
	}

	regs->rax = stat_chmod(handle->file_handle->stat, mode);
}

void syscall_fchmodat(struct registers *regs) {
	int fd = regs->rdi;
	const char *path = (const char*) regs->rsi;
	mode_t mode = regs->rdx;
	int flags = regs->r10;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] fchmodat: fd {%x}, path {%s}, mode {%x}, flags {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fd, path, mode, flags);
#endif

	struct vfs_node *file;
	if(user_lookup_at(fd, path, flags, 0, &file) == -1) {
		regs->rax = -1;
		return;
	}

	regs->rax = stat_chmod(file->stat, mode);
}

void syscall_fchownat(struct registers *regs) {
	int fd = regs->rdi;
	const char *path = (const char*) regs->rsi;
	uid_t uid = regs->rdx;
	gid_t gid = regs->r10;
	int flag = regs->r8;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] fchownat: fd {%x}, path {%s}, uid {%x}, gid {%x}, flag {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fd, path, uid, gid, flag);
#endif

	regs->rax = fd_fchownat(fd, path, uid, gid, flag);
}

void syscall_poll(struct registers *regs) {
	struct pollfd *fds = (void*)regs->rdi;
	nfds_t nfds = regs->rsi;
	int timeout = regs->rdx;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] poll: fds {%x}, nfds {%x}, timeout {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fds, nfds, timeout);
#endif

	if(timeout == 0) {
		regs->rax = 0;
		return;
	}

	struct timespec timespec = timespec_convert_ms(timeout);

	regs->rax = fd_poll(fds, nfds, &timespec);
}

void syscall_ppoll(struct registers *regs) {
	struct pollfd *fds = (void*)regs->rdi;
	nfds_t nfds = regs->rsi;
	struct timespec *timespec = (void*)regs->rdx;
	sigset_t *sigmask = (void*)regs->r10;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] ppoll: fds {%x}, nfds {%x}, timespec {%x}, sigmask {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, fds, nfds, timespec, sigmask);
#endif

	sigset_t original_mask;

	sigprocmask(SIG_SETMASK, sigmask, &original_mask);
	uint64_t ret = fd_poll(fds, nfds, timespec);
	sigprocmask(SIG_SETMASK, &original_mask, NULL);

	regs->rax = ret;
}

void syscall_utimensat(struct registers *regs) {
	int dirfd = regs->rdi;
	const char *path = (void*)regs->rsi;
	const struct timespec *timespec = (void*)regs->rdx;
	int flags = regs->r10;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] utimensat: dirfd {%x}, path {%s}, timespec {%x}, flags {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, dirfd, path, timespec, flags);
#endif

	struct vfs_node *vfs_node;

	if(!path) {
		if(dirfd == AT_FDCWD) {
			set_errno(EFAULT);
			regs->rax = -1;
			return;
		}

		struct fd_handle *handle = fd_translate(dirfd);
		if(handle == NULL) {
			set_errno(EBADF);
			regs->rax = -1;
			return;
		}

		vfs_node = handle->file_handle->vfs_node;
	}

	if(path && user_lookup_at(dirfd, path, flags, W_OK, &vfs_node) == -1) {
		regs->rax = -1;
		return;
	} 

	struct timespec atime;
	struct timespec mtime;

	if(!timespec) {
		atime = clock_realtime; 
		mtime = clock_realtime;
	} else {
		atime = timespec[0];
		mtime = timespec[1];

		if(atime.tv_nsec == UTIME_NOW) atime = clock_realtime;
		if(mtime.tv_nsec == UTIME_NOW) mtime = clock_realtime;
		if(atime.tv_nsec == UTIME_OMIT) atime = vfs_node->stat->st_atim;
		if(mtime.tv_nsec == UTIME_OMIT) mtime = vfs_node->stat->st_mtim;
	}

	vfs_node->stat->st_atim = atime;
	vfs_node->stat->st_mtim = mtime;

	regs->rax = 0;
}

void syscall_renameat(struct registers *regs) {
	int old_dirfd = regs->rdi;
	const char *old_path = (void*)regs->rsi;
	int new_dirfd = regs->rdx;
	const char *new_path = (void*)regs->r10;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] renameat: olddirfd {%x}, oldpath {%s}, newdirfd {%x}, newpath {%s}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, old_dirfd, old_path, new_dirfd, new_path);
#endif

	struct vfs_node *vfs_node_old;
	if(user_lookup_at(old_dirfd, old_path, AT_SYMLINK_FOLLOW, R_OK, &vfs_node_old) == -1) {
		regs->rax = -1; 
		return;
	}

	struct vfs_node *vfs_node_new;
	if(user_lookup_at(new_dirfd, new_path, AT_SYMLINK_FOLLOW, W_OK, &vfs_node_new) == -1) {
		int newfd = fd_openat(new_dirfd, new_path, O_CREAT, vfs_node_old->stat->st_mode);

		struct fd_handle *fd_handle = fd_translate_unlocked(newfd);
		if(fd_handle == NULL) {
			set_errno(EBADF);
			regs->rax = -1;
			return;
		}

		vfs_node_new = fd_handle->file_handle->vfs_node;
	}

	if(vfs_move(vfs_node_old, vfs_node_new, 0) == -1) {
		regs->rax = -1;
		return;
	}

	regs->rax = 0;
}

void syscall_linkat(struct registers *regs) {
	int olddirfd = regs->rdi;
	const char *oldpath = (void*)regs->rsi;
	int newdirfd = regs->rdx;
	const char *newpath = (void*)regs->r10;
	int flags = regs->r8;

#if defined(SYSCALL_DEBUG_FD) || defined(SYSCALL_DEBUG_ALL)
	print("syscall: [pid %x, tid %x] linkat: olddirfd {%x}, oldpath {%s}, newdirfd {%x}, newpath {%s}, flags {%x}\n", CORE_LOCAL->pid, CORE_LOCAL->tid, olddirfd, oldpath, newdirfd, newpath, flags);
#endif
	
	char *newpath_copy = alloc(strlen(newpath));
	strcpy(newpath_copy, newpath);

	while(*newpath_copy == '/') newpath_copy++;

	int cutoff = find_last_char(newpath_copy, '/');
	const char *name; 
	const char *dirpath;

	if(cutoff == -1) {
		name = newpath_copy;
		dirpath = "./";
	} else {
		name = newpath_copy + cutoff + 1;
		dirpath = newpath_copy;
		newpath_copy[cutoff] = '\0';
	}

	struct vfs_node *newpath_parent;
	if(user_lookup_at(newdirfd, dirpath, AT_SYMLINK_FOLLOW, X_OK, &newpath_parent) == -1) {
		regs->rax = -1;
		return;
	}

	struct vfs_node *newpath_node = vfs_search_relative(newpath_parent, name, false);
	if(newpath_node) { 
		set_errno(EEXIST); 
		regs->rax = -1; 
		return; 
	}

	struct vfs_node *oldpath_node;
	if(user_lookup_at(olddirfd, oldpath, AT_SYMLINK_FOLLOW, X_OK, &oldpath_node) == -1) {
		regs->rax = -1;
		return;
	}

	struct stat *stat = alloc(sizeof(struct stat));
	*stat = *oldpath_node->stat;
	stat_init(stat);

	stat->st_uid = CURRENT_TASK->effective_uid;

	newpath_node = vfs_create(newpath_parent, name, stat);

	if(newpath_node->stat->st_mode & S_ISGID) {
		newpath_node->stat->st_gid = newpath_parent->stat->st_gid;
	} else {
		newpath_node->stat->st_gid = newpath_parent->stat->st_gid;
	}

	regs->rax = 0;
}
