---
title: operator-linebreak - Rules
layout: doc
edit_link: https://github.com/eslint/eslint/edit/master/docs/rules/operator-linebreak.md
rule_type: layout
---
<!-- Note: No pull requests accepted for this file. See README.md in the root directory for details. -->

# enforce consistent linebreak style for operators (operator-linebreak)

(fixable) The `--fix` option on the [command line](../user-guide/command-line-interface#fixing-problems) can automatically fix some of the problems reported by this rule.

When a statement is too long to fit on a single line, line breaks are generally inserted next to the operators separating expressions. The first style coming to mind would be to place the operator at the end of the line, following the English punctuation rules.

```js
var fullHeight = borderTop +
                 innerHeight +
                 borderBottom;
```

Some developers find that placing operators at the beginning of the line makes the code more readable.

```js
var fullHeight = borderTop
               + innerHeight
               + borderBottom;
```

## Rule Details

This rule enforces a consistent linebreak style for operators.

## Options

This rule has two options, a string option and an object option.

String option:

* `"after"` requires linebreaks to be placed after the operator
* `"before"` requires linebreaks to be placed before the operator
* `"none"` disallows linebreaks on either side of the operator

Object option:

* `"overrides"` overrides the global setting for specified operators

The default configuration is `"after", { "overrides": { "?": "before", ":": "before" } }`

### after

Examples of **incorrect** code for this rule with the `"after"` option:

```js
/*eslint operator-linebreak: ["error", "after"]*/

foo = 1
+
2;

foo = 1
    + 2;

foo
    = 5;

if (someCondition
    || otherCondition) {
}

answer = everything
  ? 42
  : foo;

class Foo {
    a
        = 1;
    [b]
        = 2;
    [c
    ]
        = 3;
}
```

Examples of **correct** code for this rule with the `"after"` option:

```js
/*eslint operator-linebreak: ["error", "after"]*/

foo = 1 + 2;

foo = 1 +
      2;

foo =
    5;

if (someCondition ||
    otherCondition) {
}

answer = everything ?
  42 :
  foo;

class Foo {
    a =
        1;
    [b] =
        2;
    [c
    ] =
        3;
    d = 4;
}
```

### before

Examples of **incorrect** code for this rule with the `"before"` option:

```js
/*eslint operator-linebreak: ["error", "before"]*/

foo = 1 +
      2;

foo =
    5;

if (someCondition ||
    otherCondition) {
}

answer = everything ?
  42 :
  foo;

class Foo {
    a =
        1;
    [b] =
        2;
    [c
    ] =
        3;
}
```

Examples of **correct** code for this rule with the `"before"` option:

```js
/*eslint operator-linebreak: ["error", "before"]*/

foo = 1 + 2;

foo = 1
    + 2;

foo
    = 5;

if (someCondition
    || otherCondition) {
}

answer = everything
  ? 42
  : foo;

class Foo {
    a
        = 1;
    [b]
        = 2;
    [c
    ]
        = 3;
    d = 4;
}
```

### none

Examples of **incorrect** code for this rule with the `"none"` option:

```js
/*eslint operator-linebreak: ["error", "none"]*/

foo = 1 +
      2;

foo = 1
    + 2;

if (someCondition ||
    otherCondition) {
}

if (someCondition
    || otherCondition) {
}

answer = everything
  ? 42
  : foo;

answer = everything ?
  42 :
  foo;

class Foo {
    a =
        1;
    [b] =
        2;
    [c
    ] =
        3;
    d
        = 4;
    [e]
        = 5;
    [f
    ]
        = 6;
}
```

Examples of **correct** code for this rule with the `"none"` option:

```js
/*eslint operator-linebreak: ["error", "none"]*/

foo = 1 + 2;

foo = 5;

if (someCondition || otherCondition) {
}

answer = everything ? 42 : foo;

class Foo {
    a = 1;
    [b] = 2;
    [c
    ] = 3;
    d = 4;
    [e] = 5;
    [f
    ] = 6;
}
```

### overrides

Examples of additional **incorrect** code for this rule with the `{ "overrides": { "+=": "before" } }` option:

```js
/*eslint operator-linebreak: ["error", "after", { "overrides": { "+=": "before" } }]*/

var thing = 'thing';
thing +=
  's';
```

Examples of additional **correct** code for this rule with the `{ "overrides": { "+=": "before" } }` option:

```js
/*eslint operator-linebreak: ["error", "after", { "overrides": { "+=": "before" } }]*/

var thing = 'thing';
thing
  += 's';
```

Examples of additional **correct** code for this rule with the `{ "overrides": { "?": "ignore", ":": "ignore" } }` option:

```js
/*eslint operator-linebreak: ["error", "after", { "overrides": { "?": "ignore", ":": "ignore" } }]*/

answer = everything ?
  42
  : foo;

answer = everything
  ?
  42
  :
  foo;
```

Examples of **incorrect** code for this rule with the default `"after", { "overrides": { "?": "before", ":": "before" } }` option:

```js
/*eslint operator-linebreak: ["error", "after", { "overrides": { "?": "before", ":": "before" } }]*/

foo = 1
+
2;

foo = 1
    + 2;

foo
    = 5;

if (someCondition
    || otherCondition) {
}

answer = everything ?
  42 :
  foo;
```

Examples of **correct** code for this rule with the default `"after", { "overrides": { "?": "before", ":": "before" } }` option:

```js
/*eslint operator-linebreak: ["error", "after", { "overrides": { "?": "before", ":": "before" } }]*/

foo = 1 + 2;

foo = 1 +
      2;

foo =
    5;

if (someCondition ||
    otherCondition) {
}

answer = everything
  ? 42
  : foo;
```

## When Not To Use It

If your project will not be using a common operator line break style, turn this rule off.

## Related Rules

* [comma-style](comma-style)

## Version

This rule was introduced in ESLint 0.19.0.

## Resources

* [Rule source](https://github.com/eslint/eslint/tree/master/lib/rules/operator-linebreak.js)
* [Documentation source](https://github.com/eslint/eslint/tree/master/docs/rules/operator-linebreak.md)
