---
title: require-unicode-regexp - Rules
layout: doc
edit_link: https://github.com/eslint/eslint/edit/master/docs/rules/require-unicode-regexp.md
rule_type: suggestion
---
<!-- Note: No pull requests accepted for this file. See README.md in the root directory for details. -->

# Enforce the use of `u` flag on RegExp (require-unicode-regexp)

RegExp `u` flag has two effects:

1. **Make the regular expression handling UTF-16 surrogate pairs correctly.**

    Especially, character range syntax gets the correct behavior.

    ```js
    /^[👍]$/.test("👍") //→ false
    /^[👍]$/u.test("👍") //→ true
    ```

2. **Make the regular expression throwing syntax errors early as disabling [Annex B extensions](https://www.ecma-international.org/ecma-262/6.0/#sec-regular-expressions-patterns).**

    Because of historical reason, JavaScript regular expressions are tolerant of syntax errors. For example, `/\w{1, 2/` is a syntax error, but JavaScript doesn't throw the error. It matches strings such as `"a{1, 2"` instead. Such a recovering logic is defined in Annex B.

    The `u` flag disables the recovering logic Annex B defined. As a result, you can find errors early. This is similar to [the strict mode](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Strict_mode).

Therefore, the `u` flag lets us work better with regular expressions.

## Rule Details

This rule aims to enforce the use of `u` flag on regular expressions.

Examples of **incorrect** code for this rule:

```js
/*eslint require-unicode-regexp: error */

const a = /aaa/
const b = /bbb/gi
const c = new RegExp("ccc")
const d = new RegExp("ddd", "gi")
```

Examples of **correct** code for this rule:

```js
/*eslint require-unicode-regexp: error */

const a = /aaa/u
const b = /bbb/giu
const c = new RegExp("ccc", "u")
const d = new RegExp("ddd", "giu")

// This rule ignores RegExp calls if the flags could not be evaluated to a static value.
function f(flags) {
    return new RegExp("eee", flags)
}
```

## When Not To Use It

If you don't want to notify regular expressions with no `u` flag, then it's safe to disable this rule.

## Version

This rule was introduced in ESLint 5.3.0.

## Resources

* [Rule source](https://github.com/eslint/eslint/tree/master/lib/rules/require-unicode-regexp.js)
* [Documentation source](https://github.com/eslint/eslint/tree/master/docs/rules/require-unicode-regexp.md)
