# lumicore :: Swing

Implement `SwingUI` to start a swing application.

```java

public class ExampleMainController extends AbstractSwingUI implements SwingUI, LabelsDisplayer {
	private final JFrame frm;

	@Autowired
	private ContentPane contentPane;

	private final Application application;

	public ExampleMainController(Accounting application) {
		this.application = application;
		this.frm = new JFrame(application.getApplicationName());
		frm.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
	}

	@PostConstruct
	public void init() {
		JMenuBar menuBar = new JMenuBar();
		menuBar.add(DialogUtils.getLanguageSelectionMenu(application));
		frm.setJMenuBar(menuBar);
		frm.setContentPane(contentPane);
	}

	@Override
	public final void showUI() {
		frm.pack();
		frm.setLocationRelativeTo(null);
		this.frm.setVisible(true);
	}

	@Override
	protected Component getMainFrame() {
		return frm;
	}

	@Override
	public void loadLabels() {}

	@Override
	public void setOrientation(boolean rtl) {
		frm.applyComponentOrientation(rtl ? ComponentOrientation.RIGHT_TO_LEFT : ComponentOrientation.LEFT_TO_RIGHT);
	}
}
```

make UI elements that display text implement `LabelsDisplayer` and load all labels in the `loadLabels()` method to follow the dynamic I18N pattern. These `LabelsDisplayer`s are then discovered after the component creation and injection. Make sure that your `SwingUI` implementation class is located in the base package of all UI components. By default `LabelsDisplayer`s outside the UI package will not be discovered. To add more packages to the `LabelsDisplayer`s scan use `LabelsDisplayerPool.registerUiPackage("com.example.other.ui.package")` or annotate the `SwingUI` implementation class with `@LabelsScan(scope = {"some.ui.package", "some.other.ui.package"})`.

for I18N create a constants class like the following:

```java
@LabelsController
public class L {
	private static final LabelsManager LABELS = new DefaultLabelsManager("app-labels");

	public static final String APPLICATION_NAME = "APPLICATION_NAME";
	public static final String MAIN_FRAME_TITLE = "MAIN_FRAME_TITLE";
	public static final String MENU_NEW_WORK = "MENU_NEW_WORK";
	/*
	 * New Items can be initialized with null and then be edited and translated in DevTools.
	 */
	public static final String NEW_ITEM = null;
	

	public static String getLabel(String key, Object... args) {
		return LABELS.getLabel(key, args);
	}

	private L() {}
}

```

Labels can then be loaded like this:

```java
	@Override
	public void loadLabels() {
		mainFrame.setTitle(L.getLabel(L.MAIN_FRAME_TITLE));
	}
```

## Acknowledgments

This project includes and exports the latest/last version of the swingx library as obtained from [github.com/arotenberg/swingx](https://github.com/arotenberg/swingx).