/**
 * @file  fontmin subset
 * @author junmer
 */

/* eslint-env node */
/* global before */

import { expect } from 'chai';

import * as fs from 'fs';
import path from 'path';
import * as url from 'url';
import clean from 'gulp-clean';
import isTtf from 'is-ttf';
import Fontmin from '../index.js';
import fonteditorCore from 'fonteditor-core';
import { b2ab } from 'b3b';

var fontName = 'SentyBrush';
var fontDir = url.fileURLToPath(new URL('../fonts', import.meta.url));
var srcPath = path.resolve(fontDir, fontName + '.ttf');
var destPath = path.resolve(fontDir, 'dest_ttf');
// first mined ttf
var mined;

// first min
before(function (done) {

    // clean
    new Fontmin()
        .src(destPath)
        .use(clean())
        .run(afterClean);

    // subset first
    var fontmin = new Fontmin()
        .src(srcPath)
        .use(Fontmin.glyph({
            text: 'abcd   efg',
            // trim: false
        }))
        .dest(destPath);

    function afterClean() {
        fontmin.run(function (err, files, stream) {
            mined = files[0].contents;
            done();
        });
    }


});

describe('subset', function () {

    it('input is\'t ttf shoud be pass', function (done) {

        new Fontmin()
            .src(fontDir + '/*.html')
            .use(Fontmin.glyph({
                text: 'test'
            }))
            .run(function (err, files) {
                var ext = path.extname(files[0].path);
                expect(ext).equal('.html');
                done();
            });

    });

    it('should be ok when unicodes out of subbset', function () {

        // it ttf
        expect(isTtf(mined)).to.be.ok;

    });

    it('dest should be minier ttf', function () {

        var srcFile = fs.readFileSync(srcPath);

        // minier
        expect(mined.length).to.be.below(srcFile.length);

    });

    // it('should has whitespace when trim false', function () {

    //     var ttf = new fonteditorCore.TTFReader().read(b2ab(mined));

    //     // contain whitespace
    //     expect(ttf.cmap).to.contain.any.keys(['32', '160', '202']);

    // });

    it('should has whitespace when mixed text and whitespace', function () {

        var ttf = new fonteditorCore.TTFReader().read(b2ab(mined));

        // contain whitespace
        expect(ttf.cmap).to.contain.any.keys(['32']);

    });

    it('should support empty text', function (done) {

        new Fontmin()
            .src(srcPath)
            .use(Fontmin.glyph({
                text: ''
            }))
            .run(done);

    });

     it('should support UTF-16-encoded text', function (done) {

        new Fontmin()
            .src(srcPath)
            .use(Fontmin.glyph({
                text: '🐴'
            }))
            .run(done);

    });

    it('should support use plugin function', function (done) {

        new Fontmin()
            .src(srcPath)
            .use(Fontmin.glyph({
                text: 'test',
                use: function (ttf) {
                    expect(ttf).to.have.any.keys('ttf');
                }
            }))
            .run(done);

    });

    it('should pass use plugin not function', function (done) {

        new Fontmin()
            .src(srcPath)
            .use(Fontmin.glyph({
                use: false
            }))
            .run(done);

    });

    it('subset of non-existent character shoud be ttf', function (done) {

        var destTtf = path.resolve(destPath, fontName + '.ttf');

        var fontmin = new Fontmin()
            .src(destTtf)
            .use(Fontmin.glyph({
                text: '字体里是没有中文字符的',
                basicText: true
            }));

        fontmin.run(function (err, files, stream) {

            var twiceMined = files[0].contents;

            // it ttf
            expect(isTtf(twiceMined)).to.be.ok;

            done();
        });


    });


    it('字符串中无空格', function (done) {
        console.log(path.resolve(fontDir, 'HYWenHei-85W-1.ttf'))
        var fontmin = new Fontmin()
            .src(path.resolve(fontDir, 'HYWenHei-85W-1.ttf'))
            .use(Fontmin.glyph({
                text: '天地玄黄',
                basicText: true
            }));

        fontmin.run(function (err, files, stream) {

            var twiceMined = files[0].contents;

            // it ttf
            expect(isTtf(twiceMined)).to.be.ok;

            var font = fonteditorCore.Font.create(twiceMined, {type: 'ttf'});
            expect(font.get().glyf.length).to.be.eq(5, 'glyf length');
            done();
        });
    });

    it('中文空格和英文空格', function (done) {
        console.log(path.resolve(fontDir, 'HYWenHei-85W-1.ttf'))
        var fontmin = new Fontmin()
            .src(path.resolve(fontDir, 'HYWenHei-85W-1.ttf'))
            .use(Fontmin.glyph({
                text: ' 天地玄黄 \u3000 ',
                basicText: true
            }));

        fontmin.run(function (err, files, stream) {

            var twiceMined = files[0].contents;

            // it ttf
            expect(isTtf(twiceMined)).to.be.ok;

            var font = fonteditorCore.Font.create(twiceMined, {type: 'ttf'});
            expect(font.get().glyf.length).to.be.eq(7, 'glyf length');
            expect(font.get().glyf.some(g => g.unicode && g.unicode.includes(0x3000))).to.be.ok;
            expect(font.get().glyf.some(g => g.unicode && g.unicode.includes(0x20))).to.be.ok;
            done();
        });


    });
});
