# 拦截器

easyopen在1.3.1版本开始支持拦截器。

easyopen拦截器实现原理跟springmvc拦截器类似，拦截器作用在api方法上，即有@Api注解的方法。

拦截器定义如下：

```java
/**
 * 拦截器，原理同springmvc拦截器
 * @author tanghc
 *
 */
public interface ApiInterceptor {
    /**
     * 预处理回调方法，在方法调用前执行
     * @param request
     * @param response
     * @param serviceObj service类
     * @param argu 方法参数
     * @return
     * @throws Exception
     */
    boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object serviceObj, Object argu)
            throws Exception;

    /**
     * 接口方法执行完后调用此方法。
     * @param request
     * @param response
     * @param serviceObj service类
     * @param argu 参数
     * @param result 方法返回结果
     * @throws Exception
     */
    void postHandle(HttpServletRequest request, HttpServletResponse response, Object serviceObj, Object argu,
            Object result) throws Exception;

    /**
     * 结果包装完成后执行
     * @param request
     * @param response
     * @param serviceObj service类
     * @param argu 参数
     * @param result 最终结果，被包装过
     * @param e 
     * @throws Exception
     */
    void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object serviceObj, Object argu,
            Object result, Exception e) throws Exception;

    /**
     * 匹配拦截器
     * @param apiMeta 接口信息
     * @return
     */
    boolean match(ApiMeta apiMeta);
}
```

同样也提供一个适配器ApiInterceptorAdapter

- 拦截器执行流程：

跟springmvc拦截器执行流程一样

1. preHandle 如果返回false，则不调用api方法，接着逆序调用afterCompletion，需要通过response自定义返回
2. 如果preHandle 返回true，继续进行下一个preHandle
3. preHandle执行完毕后，逆序执行postHandle
4. 最后逆序调用afterCompletion

- 正常流程：

```
ApiInterceptor1.preHandle
ApiInterceptor2.preHandle

apiMethod.invoke() // api方法调用

ApiInterceptor2.postHandle
ApiInterceptor1.postHandle

ApiInterceptor2.afterCompletion
ApiInterceptor1.afterCompletion
```

## 配置拦截器

新建一个日志处理拦截器，继承ApiInterceptorAdapter，重写父类中的方法

```java
public class LogInterceptor extends ApiInterceptorAdapter {

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object serviceObj, Object argu)
            throws Exception {
        
        System.out.println("======preHandle======");
        System.out.println("IP:" + RequestUtil.getClientIP(request));
        System.out.println("接口类：" + serviceObj.getClass().getName());
        if(argu != null) {
            System.out.println("参数类：" + argu.getClass().getName());
        }
        
        
        return true;
    }

    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response, Object serviceObj, Object argu,
            Object result) throws Exception {
        System.out.println("======postHandle======");
        System.out.println("接口类：" + serviceObj.getClass().getName());
        if(argu != null) {
            System.out.println("参数类：" + argu.getClass().getName());
        }
        System.out.println("结果：" + JSON.toJSONString(result));
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object serviceObj,
            Object argu, Object result, Exception e) throws Exception {
        System.out.println("======afterCompletion======");
        System.out.println("接口类：" + serviceObj.getClass().getName());
        System.out.println("参数类：" + argu.getClass().getName());
        System.out.println("最终结果：" + JSON.toJSONString(result));
        System.out.println("e:" + e);
    }
    
}
```

在apiConfgi中添加拦截器:

```java
@Override
    protected void initApiConfig(ApiConfig apiConfig) {
        ...

        // 配置拦截器
        apiConfig.setInterceptors(
                new ApiInterceptor[] { new LogInterceptor()});

       ...
    }
```

## 拦截范围

默认拦截所有接口，如果要拦截指定接口，可重写boolean match()方法：

```
// 只拦截goods.get接口
@Override
    public boolean match(ApiMeta apiMeta) {
        return apiMeta.getName().equals("goods.get");
    }
```