<?php

namespace Drupal\druio_notification\Entity;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\user\UserInterface;

/**
 * Defines the Druio notification entity.
 *
 * @ingroup druio_notification
 *
 * @ContentEntityType(
 *   id = "druio_notification",
 *   label = @Translation("Druio notification"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\druio_notification\DruioNotificationListBuilder",
 *     "views_data" = "Drupal\druio_notification\Entity\DruioNotificationViewsData",
 *     "form" = {
 *       "default" = "Drupal\druio_notification\Form\DruioNotificationForm",
 *       "add" = "Drupal\druio_notification\Form\DruioNotificationForm",
 *       "edit" = "Drupal\druio_notification\Form\DruioNotificationForm",
 *       "delete" = "Drupal\druio_notification\Form\DruioNotificationDeleteForm",
 *     },
 *     "access" = "Drupal\druio_notification\DruioNotificationAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\druio_notification\DruioNotificationHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "druio_notification",
 *   admin_permission = "administer druio notification entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "subject",
 *     "uuid" = "uuid",
 *     "uid" = "user_id",
 *     "langcode" = "langcode",
 *   },
 *   links = {
 *     "canonical" = "/admin/druio/notification/{druio_notification}",
 *     "add-form" = "/admin/druio/notification/add",
 *     "edit-form" = "/admin/druio/notification/{druio_notification}/edit",
 *     "delete-form" = "/admin/druio/notification/{druio_notification}/delete",
 *     "collection" = "/admin/druio/notification",
 *   },
 *   field_ui_base_route = "druio_notification.settings"
 * )
 */
class DruioNotification extends ContentEntityBase implements DruioNotificationInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('user_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function markAsRead() {
    $this->set('is_read', TRUE);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreated() {
    return $this->get('created');
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('User'))
      ->setDescription(t('The user ID for which the notification is intended.'))
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 0,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['subject'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Subject'))
      ->setDescription(t('The subject of notification.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 10,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['message'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('Message'))
      ->setDisplayOptions('view', [
        'type' => 'text_default',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'text_textfield',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['is_read'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Is notification read'))
      ->setDefaultValue(FALSE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    return $fields;
  }

}
