//          Copyright Brian Schott (Hackerpilot) 2014.
// Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//          http://www.boost.org/LICENSE_1_0.txt)

module dscanner.analysis.enumarrayliteral;

import dparse.ast;
import dparse.lexer;
import dscanner.analysis.base;
import std.algorithm : find, map;
import dsymbol.scope_ : Scope;

void doNothing(string, size_t, size_t, string, bool)
{
}

final class EnumArrayLiteralCheck : BaseAnalyzer
{
	alias visit = BaseAnalyzer.visit;

	mixin AnalyzerInfo!"enum_array_literal_check";

	this(BaseAnalyzerArguments args)
	{
		super(args);
	}

	bool looking;

	mixin visitTemplate!ClassDeclaration;
	mixin visitTemplate!InterfaceDeclaration;
	mixin visitTemplate!UnionDeclaration;
	mixin visitTemplate!StructDeclaration;

	override void visit(const AutoDeclaration autoDec)
	{
		auto enumToken = autoDec.storageClasses.find!(a => a.token == tok!"enum");
		if (enumToken.length)
		{
			foreach (part; autoDec.parts)
			{
				if (part.initializer is null)
					continue;
				if (part.initializer.nonVoidInitializer is null)
					continue;
				if (part.initializer.nonVoidInitializer.arrayInitializer is null)
					continue;
				addErrorMessage(part.initializer.nonVoidInitializer,
						KEY,
						"This enum may lead to unnecessary allocation at run-time."
						~ " Use 'static immutable "
						~ part.identifier.text ~ " = [ ...' instead.",
						[
							AutoFix.replacement(enumToken[0].token, "static immutable")
						]);
			}
		}
		autoDec.accept(this);
	}

	private enum string KEY = "dscanner.performance.enum_array_literal";
}

unittest
{
	import dscanner.analysis.config : Check, disabledConfig, StaticAnalysisConfig;
	import dscanner.analysis.helpers : assertAnalyzerWarnings, assertAutoFix;
	import std.stdio : stderr;

	StaticAnalysisConfig sac = disabledConfig();
	sac.enum_array_literal_check = Check.enabled;
	assertAnalyzerWarnings(q{
		enum x = [1, 2, 3]; /+
		         ^^^^^^^^^ [warn]: This enum may lead to unnecessary allocation at run-time. Use 'static immutable x = [ ...' instead. +/
	}c, sac);

	assertAutoFix(q{
		enum x = [1, 2, 3]; // fix
	}c, q{
		static immutable x = [1, 2, 3]; // fix
	}c, sac);

	stderr.writeln("Unittest for EnumArrayLiteralCheck passed.");
}
