package simpledb;

import java.util.*;
import java.io.*;


/**
 * Page is the interface used to represent pages that are resident in the
 * BufferPool.  Typically, DbFiles will read and write pages from disk.
 * <p>
 * Pages may be "dirty", indicating that they have been modified since they
 * were last written out to disk.
 *
 * For recovery purposes, pages MUST have a single constructor of the form:
 *     Page(PageId id, byte[] data)
 */
public interface Page {

    /**
     * Return the id of this page.  The id is a unique identifier for a page
     * that can be used to look up the page on disk or determine if the page
     * is resident in the buffer pool.
     *
     * @return the id of this page
     */
    public PageId id();

    /**
     * Returns true if this page is dirty.
     *
     * @return The id of the transaction that last dirtied this page, or null
     */
    public TransactionId isDirty();

  /**
   * Set the dirty state of this page as dirtied by a particular transaction
   */
    public void markDirty(boolean dirty, TransactionId tid);
    
    /**
     * Increment pin_count (pinning). It happens every time a page is requested but not released.
     */    
    public void pin();
    
    /**
     * Decrement pin_count (unpinning). It happens when the page is released.
     */    
    public void unpin();
    
    /**
     * Return value of pin_count
     */    
    public int pin_count();
    

  /**
   * Generates a byte array representing the contents of this page.
   * Used to serialize this page to disk.
   * <p>
   * The invariant here is that it should be possible to pass the byte array
   * generated by getPageData to the Page constructor and have it produce
   * an identical Page object.
   *
   * @return A byte array correspond to the bytes of this page.
   */

    public byte[] getPageData();

    /** Provide a representation of this page before any modifications were made
        to it.  Used by recovery.
    */
    public Page getBeforeImage();

    public Iterator<Tuple> iterator();
}
