package simpledb;
import java.util.*;
import java.io.*;

/**
 * The Join operator implements the relational join operation.
 */
public class Join extends AbstractDbIterator {

    private JoinPredicate _predicate;
    private DbIterator _outerRelation;
    private DbIterator _innerRelation;
    private Iterator<Tuple> _outerPage=null;
    private Iterator<Tuple> _innerPage=null;

    private Tuple _outerRecent=null;
    private Tuple _innerRecent=null;

    private int _joinType = 0;
    private int _numMatches =0;
    private int _numComp=0;
    
    public static final int SNL = 0;
    public static final int PNL = 1;    
    public static final int BNL = 2;    
    public static final int SMJ = 3;    
    public static final int HJ = 4;    
    /**
     * Constructor.  Accepts to children to join and the predicate
     * to join them on
     *
     * @param p The predicate to use to join the children
     * @param child1 Iterator for the left(outer) relation to join
     * @param child2 Iterator for the right(inner) relation to join
     */
    public Join(JoinPredicate p, DbIterator child1, DbIterator child2) {
	//IMPLEMENT THIS
    	this._predicate = p;
    	this._outerRelation = child1;
    	this._innerRelation = child2;
    }

    public void setJoinAlgorithm(int joinAlgo){
    	_joinType = joinAlgo;
    }
    /**
     * @see simpledb.TupleDesc#combine(TupleDesc, TupleDesc) for possible implementation logic.
     */
    public TupleDesc getTupleDesc() {
		//IMPLEMENT THIS
    	return TupleDesc.combine(this._outerRecent.getTupleDesc(), this._innerRecent.getTupleDesc());
    }

    public void open()
        throws DbException, NoSuchElementException, TransactionAbortedException, IOException {
		//IMPLEMENT THIS
    	this._outerRelation.open();
    	this._innerRelation.open();
    }

    public void close() {
		//IMPLEMENT THIS
    	this._outerRelation.close();
    	this._innerRelation.close();
    }

    public void rewind() throws DbException, TransactionAbortedException, IOException {
//IMPLEMENT THIS
    	this._outerRelation.rewind();
    	this._innerRelation.rewind();
    	this._outerRecent = null;
    	this._innerRecent = null;
    }

    /**
     * Returns the next tuple generated by the join, or null if there are no more tuples.
     * Logically, this is the next tuple in r1 cross r2 that satisfies the join
     * predicate.  There are many possible implementations; the simplest is a
     * nested loops join.
     * <p>
     * Note that the tuples returned from this particular implementation of
     * Join are simply the concatenation of joining tuples from the left and
     * right relation. Therefore, there will be two copies of the join attribute
     * in the results.  (Removing such duplicate columns can be done with an
     * additional projection operator if needed.)
     * <p>
     * For example, if one tuple is {1,2,3} and the other tuple is {1,5,6},
     * joined on equality of the first column, then this returns {1,2,3,1,5,6}.
     *
     * @return The next matching tuple.
     * @throws IOException 
     * @throws NoSuchElementException 
     * @see JoinPredicate#filter
     */
    protected Tuple readNext() throws TransactionAbortedException, DbException, NoSuchElementException, IOException {
	switch(_joinType){
	case SNL: return SNL_readNext();
	case PNL: return PNL_readNext();
	case BNL: return BNL_readNext();
	case SMJ: return SMJ_readNext();
	case HJ: return HJ_readNext();
	default: return SNL_readNext();
	}
    }

    protected Tuple SNL_readNext() throws TransactionAbortedException, DbException, NoSuchElementException, IOException {
    	//IMPLEMENT THIS 
    	TupleDesc newTupleDesc; 
    	
    	// Initial
    	if (this._outerRecent == null) { 
			if (this._outerRelation.hasNext()){
				this._outerRecent = this._outerRelation.next();
			}
		}
    	// Initial
    	
    	//OR condition was added later as Single Match TestCase was failing
    	//It failed because single relation and hasNext was always false. So while loop did not execute.
    	while(this._outerRelation.hasNext() || this._outerRecent != null){
    		while (this._innerRelation.hasNext()){
    			this._innerRecent = this._innerRelation.next();
    			if (this._predicate.filter(this._outerRecent, this._innerRecent)){
    				this._numMatches++;
    				newTupleDesc = TupleDesc.combine(this._outerRecent.getTupleDesc(), this._innerRecent.getTupleDesc());
    				return joinTuple(this._outerRecent, this._innerRecent, newTupleDesc);
    			}
    			else{
    				return null;
    			}
    		}
    		this._innerRelation.rewind();
    		this._outerRecent = this._outerRelation.next();
    	}
    	return null;
    }

    private boolean initOnce = true;
    private Tuple pnlInner = null;
    
    protected Tuple PNL_readNext() throws TransactionAbortedException, DbException, NoSuchElementException, IOException {
    	//IMPLEMENT THIS (EXTRA CREDIT ONLY)
TupleDesc newTupleDesc;
    	
    	//initialize 
    	if(this.initOnce == true){
    		this._outerRecent = this._outerRelation.next(); //get tuple
			PageId outerPageID = this._outerRecent.getRecordID()._pageid; //get pageid of our tuple
			this._outerPage = Database.getBufferPool().getPage(null, outerPageID, Permissions.READ_ONLY).iterator(); //get page iterator
    		
			this._innerRecent = this._innerRelation.next();
			PageId innerPageID = this._innerRecent.getRecordID()._pageid;
			this._innerPage = Database.getBufferPool().getPage(null, innerPageID, Permissions.READ_ONLY).iterator();
			
			this.initOnce = false;
		}
    	//initialize
    	
		while(this._outerPage.hasNext()){	
    		while(this._innerPage.hasNext()){
    			this._innerRecent = this._innerPage.next();
    			if(this._innerRelation.hasNext())
    				this.pnlInner = this._innerRelation.next();
    			if (this._predicate.filter(this._outerRecent, this._innerRecent)){
					this._numMatches++;
					newTupleDesc = TupleDesc.combine(this._outerRecent.getTupleDesc(), this._innerRecent.getTupleDesc());
					return joinTuple(this._outerRecent, this._innerRecent, newTupleDesc);
				}
    		}
    		//get new inner page once first page is read
    		if(this._innerRelation.hasNext()){
    			this._innerRecent = this._innerRelation.next();
	    		if(this._innerRecent == null)
    	    		return null;
	    		PageId innerPageID = this._innerRecent.getRecordID()._pageid;
				this._innerPage = Database.getBufferPool().getPage(null, innerPageID, Permissions.READ_ONLY).iterator();
    		}
				
	    	//get new outer page after ops on one outer page is over
    		else if(this._outerRelation.hasNext()){
    			this._outerRecent = this._outerRelation.next(); //get tuple
    	    	if(this._outerRecent == null)
    	    		return null;
    			PageId outerPageID = this._outerRecent.getRecordID()._pageid; //get pageid of our tuple
    			this._outerPage = Database.getBufferPool().getPage(null, outerPageID, Permissions.READ_ONLY).iterator(); //get page iterator
    			this._innerRelation.rewind();
    		}
    		else{
    			return null;
    		}
		}
    	return null;
    }


    protected Tuple BNL_readNext() throws TransactionAbortedException, DbException {
	//no need to implement this
	return null;
    }

    private Tuple tempTuple = null;
    private boolean flag = true;
    private Tuple startPointer = null;

    protected Tuple SMJ_readNext() throws TransactionAbortedException, DbException, NoSuchElementException, IOException {
    	//IMPLEMENT THIS. YOU CAN ASSUME THE JOIN PREDICATE IS ALWAYS =
    	
    	TupleDesc newTupleDesc; 
    	Field leftVal = null, rightVal = null;
    	//initial
    	if(this.flag == true){
    		this._outerRecent = this._outerRelation.next();
    		this._innerRecent = this._innerRelation.next();
    		this.flag = false;
    	}
    	//initial
    	
    	while(this._innerRelation.hasNext() || this._outerRelation.hasNext()){
    		//get values
    		leftVal =  this._predicate.getLeftField(this._outerRecent);
    		rightVal =  this._predicate.getRightField(this._innerRecent);
    		//get values
    		
    		while(this._innerRelation.hasNext()){
    			if(leftVal.compare(Predicate.Op.GREATER_THAN, rightVal)){
    				Tuple tempTuple = this._innerRelation.next();
    				//check if new inner tuple has different value as StartPointer
            		//if yes then update the StartPointer
            		if(!this._predicate.getRightField(this._innerRecent).compare(Predicate.Op.EQUALS, this._predicate.getRightField(tempTuple))){
            			this.startPointer = tempTuple;
            			rightVal =  this._predicate.getRightField(this.startPointer);
            		}
    				this._innerRecent = tempTuple;
        		}
        		else if(rightVal.compare(Predicate.Op.GREATER_THAN, leftVal)){
        			if(this._outerRelation.hasNext() ==  false){
        				return null;
        			}
        			Tuple tempTuple = this._outerRelation.next();
        			
        			//check if new outer tuple has same value as StartPointer
            		//if yes then move the inner relation to the StartPointer
            		if(this._predicate.getRightField(this._outerRecent).compare(Predicate.Op.EQUALS, this._predicate.getRightField(tempTuple))){
            			this._innerRelation.rewind();
            			while(!this._innerRecent.equals(this.startPointer)){
            				this._innerRecent = this._innerRelation.next();
            			}
            			rightVal =  this._predicate.getRightField(this._innerRecent);
            		}
        			if(this._outerRelation.hasNext())
        				this._outerRecent = tempTuple;
        		}
        		else{
        			this._numMatches++;
        			//Start pointer is not SET or New right tuple value is not equal to old start partition value
        			if(this.startPointer == null || !rightVal.compare(Predicate.Op.EQUALS, this._predicate.getRightField(this.startPointer))){
        				this.startPointer = this._innerRecent;
        			}
        			tempTuple = this._innerRecent;
        			this._innerRecent = this._innerRelation.next();
        			newTupleDesc = TupleDesc.combine(tempTuple.getTupleDesc(), this._outerRecent.getTupleDesc());
    				return joinTuple(this._outerRecent, tempTuple, newTupleDesc);
    			}
    		}
    		if(this._outerRelation.hasNext() ==  false){
    			this._numMatches++;
    			return null;
    		}
    		this._outerRecent = this._outerRelation.next();
    		this._innerRelation.rewind();
    	}
    	
    	//single match
    	if(this._numMatches == 0 && this._innerRecent != null && this._outerRecent != null){
    		//get values
    		leftVal =  this._predicate.getLeftField(this._outerRecent);
    		rightVal =  this._predicate.getRightField(this._innerRecent);
    		//get values
    		if(leftVal.compare(Predicate.Op.EQUALS, rightVal)){
    			this._numMatches++;
    			newTupleDesc = TupleDesc.combine(this._innerRecent.getTupleDesc(), this._outerRecent.getTupleDesc());
				return joinTuple(this._outerRecent, this._innerRecent, newTupleDesc);
    		}
    	}
    	//single match
    	
    	//cross product offset required
    	if(this._numMatches != 1)
    		this._numMatches++;
    	return null;
    }

    protected Tuple HJ_readNext() throws TransactionAbortedException, DbException {
	//no need to implement this
	return null;
    }


    private Tuple joinTuple(Tuple outer, Tuple inner, TupleDesc tupledesc){
		//IMPLEMENT THIS
    	Tuple resultTuple = new Tuple(tupledesc);
    	int outerSize = outer.getTupleDesc().numFields();
		int innerSize = inner.getTupleDesc().numFields();
		for (int i = 0; i < outerSize; i++)
		{
			resultTuple.setField(i, outer.getField(i));
		}
		for (int i = 0; i < innerSize; i++)
		{
			resultTuple.setField(outerSize + i, inner.getField(i));
		}
		return resultTuple;
    }

    public int getNumMatches(){
	return _numMatches;
    }
    public int getNumComp(){
	return _numComp;
    }
}
