import { useSuspenseQuery } from '@suspensive/react-query';
import { Suspense, useMemo, useState } from 'react';
import { useParams } from 'react-router-dom';
import { createColumnHelper, Table, TableSkeleton } from 'ui-components';

import { DFLink } from '@/components/DFLink';
import { TruncatedText } from '@/components/TruncatedText';
import { getSeverityColorMap } from '@/constants/charts';
import { ConnectorHeader } from '@/features/onboard/components/ConnectorHeader';
import { queries } from '@/queries';
import { useTheme } from '@/theme/ThemeContext';
import { ScanTypeEnum } from '@/types/common';
import { SeverityEnum } from '@/utils/enum';

const DEFAULT_PAGE_SIZE = 10;

type SummaryData = {
  accountName: string;
  accountType: string;
  total: number;
  critical: number;
  high: number;
  medium: number;
  low: number;
  unknown: number;
};

const useGetScanSummary = () => {
  const params = useParams();
  return useSuspenseQuery({
    ...queries.onboard.vulnerabilityScanSummary({
      scanType: ScanTypeEnum.VulnerabilityScan,
      bulkScanId: params.bulkScanId ?? '',
    }),
  });
};

const SummaryTable = () => {
  const { mode } = useTheme();
  const { data } = useGetScanSummary();
  const [pageSize, setPageSize] = useState(DEFAULT_PAGE_SIZE);
  const columnHelper = createColumnHelper<SummaryData>();

  const columns = useMemo(() => {
    const columns = [
      columnHelper.accessor('accountType', {
        cell: (info) => <TruncatedText text={info.getValue()} />,
        header: () => 'Type',
        minSize: 50,
        size: 60,
        maxSize: 60,
      }),
      columnHelper.accessor('accountName', {
        cell: (info) => <TruncatedText text={info.getValue()} />,
        header: () => 'Name',
        minSize: 100,
        size: 120,
        maxSize: 250,
      }),
      columnHelper.accessor('total', {
        cell: (info) => (
          <div className="flex items-center justify-end tabular-nums">
            <span className="truncate">{info.getValue()}</span>
          </div>
        ),
        header: () => (
          <div className="text-right">
            <TruncatedText text="Total" />
          </div>
        ),
        minSize: 80,
        size: 80,
        maxSize: 80,
      }),
      columnHelper.accessor(SeverityEnum.Critical, {
        cell: (info) => {
          return (
            <div className="flex items-center gap-x-2 tabular-nums">
              <div
                className="w-3 h-3 rounded-full"
                style={{
                  backgroundColor: getSeverityColorMap(mode)[SeverityEnum.Critical],
                }}
              ></div>
              <span>{info.getValue() ?? 0}</span>
            </div>
          );
        },
        header: () => <TruncatedText text="Critical" />,
        minSize: 80,
        size: 80,
        maxSize: 80,
        enableResizing: false,
      }),
      columnHelper.accessor(SeverityEnum.High, {
        cell: (info) => {
          return (
            <div className="flex items-center gap-x-2 tabular-nums">
              <div
                className="w-3 h-3 rounded-full shrink-0"
                style={{
                  backgroundColor: getSeverityColorMap(mode)[SeverityEnum.High],
                }}
              ></div>
              <span>{info.getValue() ?? 0}</span>
            </div>
          );
        },
        header: () => <TruncatedText text="High" />,
        minSize: 80,
        size: 80,
        maxSize: 80,
        enableResizing: false,
      }),
      columnHelper.accessor(SeverityEnum.Medium, {
        cell: (info) => {
          return (
            <div className="flex items-center gap-x-2 tabular-nums">
              <div
                className="w-3 h-3 rounded-full shrink-0"
                style={{
                  backgroundColor: getSeverityColorMap(mode)[SeverityEnum.Medium],
                }}
              ></div>
              <span>{info.getValue() ?? 0}</span>
            </div>
          );
        },
        header: () => <TruncatedText text="Medium" />,
        minSize: 80,
        size: 80,
        maxSize: 80,
        enableResizing: false,
      }),
      columnHelper.accessor(SeverityEnum.Low, {
        cell: (info) => {
          return (
            <div className="flex items-center gap-x-2 tabular-nums">
              <div
                className="w-3 h-3 rounded-full shrink-0"
                style={{
                  backgroundColor: getSeverityColorMap(mode)[SeverityEnum.Low],
                }}
              ></div>
              <span>{info.getValue() ?? 0}</span>
            </div>
          );
        },
        header: () => <TruncatedText text="Low" />,
        minSize: 80,
        size: 80,
        maxSize: 80,
        enableResizing: false,
      }),
      columnHelper.accessor(SeverityEnum.Unknown, {
        cell: (info) => {
          return (
            <div className="flex items-center gap-x-2 tabular-nums">
              <div
                className="w-3 h-3 rounded-full shrink-0"
                style={{
                  backgroundColor: getSeverityColorMap(mode)[SeverityEnum.Unknown],
                }}
              ></div>
              <span>{info.getValue() ?? 0}</span>
            </div>
          );
        },
        header: () => <TruncatedText text="Unknown" />,
        minSize: 80,
        size: 80,
        maxSize: 80,
        enableResizing: false,
      }),
    ];

    return columns;
  }, []);

  return (
    <Table
      size="default"
      data={data ?? []}
      columns={columns}
      enableColumnResizing
      enableSorting
      enablePageResize
      pageSize={pageSize}
      enablePagination
      onPageResize={(newSize) => {
        setPageSize(newSize);
      }}
    />
  );
};

const VulnerabilityScanSummary = () => {
  return (
    <div className="flex flex-col">
      <ConnectorHeader
        title={'Vulnerability Scan Result Summary'}
        description={'Summary of vulnerability scan result'}
      />
      <DFLink to={'/vulnerability'} unstyled>
        <div className="dark:text-accent-accent text-text-link hover:text-bg-hover-1 text-p4">
          Go to vulnerability dashboard to view details scan result
        </div>
      </DFLink>

      <div className="flex flex-col gap-4 mt-4">
        <Suspense
          fallback={
            <div className="w-full">
              <TableSkeleton columns={8} rows={DEFAULT_PAGE_SIZE} />
            </div>
          }
        >
          <SummaryTable />
        </Suspense>
      </div>
    </div>
  );
};

export const module = {
  element: <VulnerabilityScanSummary />,
};
