import { useMemo, useState } from 'react';
import { useLocation, useRouteError } from 'react-router-dom';
import { cn } from 'tailwind-preset';

import { AppHeader } from '@/components/AppHeader';
import { DF404Error, FourZeroFourAuthenticated } from '@/components/error/404';
import { getSideNavigationState, SideNavigation } from '@/components/SideNavigation';
import { OnboardAppHeader } from '@/features/onboard/components/OnBoardAppHeader';
import storage from '@/utils/storage';

const ErrorComponent = ({ maintenance }: { maintenance: boolean }) => {
  return (
    <div className="flex flex-col h-full items-center sm:pt-10 2xl:pt-20">
      {!maintenance ? (
        <h1 className="sm:text-[80px] 2xl:text-[100px] text-chart-orange dark:text-chart-orange font-black leading-none">
          ERROR
        </h1>
      ) : null}
      <div className="mt-6 flex flex-col gap-y-14 items-center">
        <h4
          className={cn(
            'font-semibold text-text-text-and-icon flex flex-col text-center',
            {
              'sm:text-base 2xl:text-xl': !maintenance,
              'sm:text-xl 2xl:text-2xl': maintenance,
            },
          )}
        >
          <span>
            {maintenance
              ? 'Maintenance in progress...'
              : 'An error has occurred, please refresh the page.'}
          </span>
          <span>
            {maintenance
              ? 'Please try again after some time.'
              : 'If problem persists, please contact deepfence.'}
          </span>
        </h4>
        {!maintenance ? (
          <div className="sm:w-[375px] 2xl:w-[455px]">
            <SVG500 />
          </div>
        ) : null}
      </div>
    </div>
  );
};
export const FiveZeroZero = () => {
  const [sideNavExpanded, setSideNavExpanded] = useState(
    getSideNavigationState() === 'open' ? true : false,
  );
  const isAuth = useMemo(() => storage.getAuth(), []);
  const location = useLocation();
  const error = useRouteError();
  console.error(error);

  if (location.pathname.startsWith('/onboard')) {
    const errorComponent =
      error instanceof DF404Error ? (
        <FourZeroFourAuthenticated />
      ) : (
        <ErrorComponent maintenance={((error as Error)?.cause as any)?.status === 503} />
      );
    return (
      <div className="min-h-screen isolate dark:bg-bg-page">
        <div className="pt-[56px] h-screen">{errorComponent}</div>
        <OnboardAppHeader />
      </div>
    );
  }

  return (
    <div className="min-h-screen isolate dark:bg-bg-page">
      {isAuth ? (
        <>
          <AppHeader />
          <SideNavigation
            expanded={sideNavExpanded}
            onExpandedChange={(state) => setSideNavExpanded(state)}
          />
          <main
            className={cn('pt-[56px] h-screen overflow-auto transition-[margin-left]', {
              'ml-[61px]': !sideNavExpanded,
              'ml-[240px]': sideNavExpanded,
            })}
          >
            {error instanceof DF404Error ? (
              <FourZeroFourAuthenticated />
            ) : (
              <ErrorComponent
                maintenance={((error as Error)?.cause as any)?.status === 503}
              />
            )}
          </main>
        </>
      ) : (
        <div className="h-screen">
          <ErrorComponent
            maintenance={((error as Error)?.cause as any)?.status === 503}
          />
        </div>
      )}
    </div>
  );
};

const SVG500 = () => {
  return (
    <svg
      width="100%"
      height="100%"
      viewBox="0 0 447 365"
      fill="none"
      xmlns="http://www.w3.org/2000/svg"
    >
      <path
        d="M223.5 364.014C346.493 364.014 446.199 315.557 446.199 255.782C446.199 196.008 346.493 147.551 223.5 147.551C100.507 147.551 0.800781 196.008 0.800781 255.782C0.800781 315.557 100.507 364.014 223.5 364.014Z"
        fill="#16253B"
      />
      <path
        d="M129.025 308.589C128.569 308.597 128.125 308.443 127.772 308.155C127.419 307.867 127.178 307.464 127.094 307.016L126.66 304.679C125.657 304.502 124.667 304.26 123.696 303.955H123.628L119.516 305.886C119.212 306.02 118.883 306.089 118.55 306.089C118.141 306.093 117.739 305.99 117.382 305.789L113.289 303.424C113.041 303.293 112.831 303.1 112.68 302.863C112.53 302.627 112.443 302.355 112.428 302.075C112.413 301.795 112.472 301.516 112.597 301.265C112.723 301.014 112.911 300.8 113.144 300.644L115.625 298.868C115.399 298.559 115.193 298.236 115.007 297.902L109.785 297.574C109.312 297.542 108.868 297.332 108.544 296.986C108.22 296.641 108.038 296.185 108.037 295.711V292.679C108.039 292.204 108.221 291.746 108.547 291.4C108.874 291.054 109.319 290.845 109.794 290.816L115.007 290.488C115.193 290.154 115.399 289.831 115.625 289.522L113.144 287.746C112.916 287.586 112.731 287.371 112.606 287.122C112.481 286.873 112.421 286.596 112.43 286.317C112.446 286.038 112.534 285.767 112.684 285.53C112.835 285.294 113.043 285.1 113.289 284.966L117.382 282.6C117.739 282.4 118.141 282.297 118.55 282.301C118.883 282.296 119.213 282.365 119.516 282.504L123.628 284.435H123.677C124.662 284.127 125.664 283.879 126.679 283.691L127.113 281.365C127.204 280.919 127.446 280.519 127.798 280.23C128.149 279.942 128.59 279.783 129.044 279.782H135.068C135.526 279.774 135.971 279.929 136.324 280.219C136.678 280.509 136.917 280.915 136.999 281.365L137.443 283.691C138.445 283.877 139.435 284.122 140.407 284.425H140.475L144.587 282.494C144.89 282.355 145.22 282.286 145.553 282.292C145.962 282.287 146.364 282.391 146.721 282.591L150.814 284.956C151.06 285.09 151.268 285.284 151.419 285.521C151.569 285.757 151.657 286.028 151.673 286.308C151.681 286.586 151.619 286.862 151.495 287.111C151.37 287.36 151.186 287.575 150.959 287.736L148.478 289.513C148.704 289.821 148.91 290.144 149.096 290.478L154.309 290.806C154.784 290.836 155.229 291.045 155.556 291.391C155.882 291.737 156.064 292.194 156.066 292.67V295.72C156.064 296.196 155.882 296.653 155.556 296.999C155.229 297.345 154.784 297.554 154.309 297.584L149.096 297.912C148.91 298.246 148.704 298.569 148.478 298.877L150.959 300.654C151.186 300.815 151.37 301.03 151.495 301.279C151.619 301.528 151.681 301.804 151.673 302.082C151.656 302.362 151.568 302.632 151.417 302.869C151.267 303.105 151.06 303.299 150.814 303.434L146.721 305.799C146.364 305.999 145.962 306.103 145.553 306.098C145.22 306.099 144.891 306.03 144.587 305.896L140.475 303.965C139.493 304.27 138.494 304.516 137.482 304.699L137.038 307.035C136.954 307.483 136.713 307.887 136.36 308.175C136.007 308.463 135.563 308.616 135.107 308.608L129.025 308.589ZM132.037 289.87C129.86 289.818 127.705 290.327 125.781 291.347C124.381 292.158 123.58 293.191 123.58 294.195C123.58 295.199 124.381 296.232 125.781 297.043C127.729 298.02 129.877 298.529 132.056 298.529C134.235 298.529 136.384 298.02 138.331 297.043C139.731 296.232 140.533 295.199 140.533 294.195C140.533 293.191 139.731 292.158 138.331 291.347C136.396 290.32 134.227 289.811 132.037 289.87Z"
        fill="#0C1A33"
      />
      <path
        d="M175.848 281.663C175.558 281.659 175.272 281.598 175.007 281.482C174.742 281.365 174.502 281.198 174.303 280.988L172.671 279.279H171.465C171.059 279.279 170.663 279.279 170.268 279.279L168.53 281.103C168.345 281.287 168.125 281.431 167.882 281.528C167.64 281.624 167.381 281.67 167.12 281.663C166.945 281.66 166.77 281.637 166.599 281.596L162.544 280.63C162.285 280.569 162.043 280.451 161.836 280.284C161.629 280.117 161.462 279.905 161.347 279.665C161.249 279.458 161.198 279.232 161.198 279.004C161.198 278.775 161.249 278.549 161.347 278.342L161.994 277.068C161.646 276.875 161.318 276.672 161.029 276.46L157.167 277.097C157.071 277.107 156.974 277.107 156.877 277.097C156.557 277.09 156.244 277.009 155.96 276.86C155.677 276.71 155.433 276.498 155.246 276.238L153.711 274.104C153.506 273.823 153.389 273.487 153.373 273.139C153.368 272.801 153.473 272.471 153.671 272.198C153.87 271.925 154.151 271.723 154.474 271.623L157.428 270.658C157.423 270.613 157.423 270.568 157.428 270.523L154.58 269.615C154.253 269.527 153.961 269.342 153.744 269.083C153.526 268.824 153.393 268.504 153.363 268.167V268.061C153.361 267.741 153.463 267.43 153.653 267.173L155.323 264.836C155.509 264.598 155.748 264.407 156.021 264.277C156.295 264.148 156.595 264.085 156.897 264.093C157.009 264.083 157.122 264.083 157.235 264.093L161.009 264.73C161.28 264.547 161.579 264.363 161.888 264.19L162.004 264.122L161.405 262.935C161.295 262.729 161.232 262.501 161.222 262.268V262.066V261.96C161.26 261.637 161.396 261.333 161.612 261.09C161.827 260.846 162.112 260.675 162.428 260.598L166.667 259.633C166.802 259.597 166.942 259.58 167.082 259.585C167.37 259.588 167.654 259.65 167.918 259.766C168.182 259.882 168.419 260.05 168.617 260.26L170.258 261.969C170.663 261.969 171.078 261.969 171.484 261.969C171.889 261.969 172.266 261.969 172.652 261.969L174.39 260.145C174.575 259.961 174.795 259.817 175.037 259.72C175.28 259.624 175.539 259.578 175.799 259.585C175.972 259.583 176.144 259.605 176.311 259.652L180.375 260.618C180.743 260.69 181.075 260.885 181.317 261.171C181.559 261.457 181.697 261.817 181.708 262.191C181.717 262.422 181.667 262.651 181.563 262.857L180.906 264.161L181.041 264.238C181.333 264.4 181.617 264.578 181.891 264.769L185.753 264.132C185.846 264.122 185.94 264.122 186.033 264.132C186.353 264.138 186.667 264.218 186.952 264.366C187.236 264.513 187.483 264.724 187.674 264.981L189.209 267.124C189.42 267.403 189.538 267.741 189.547 268.09V268.264V268.379C189.504 268.667 189.381 268.936 189.194 269.158C189.006 269.379 188.761 269.544 188.485 269.634L185.511 270.6V270.735L188.407 271.633C188.651 271.703 188.875 271.829 189.063 272C189.25 272.17 189.396 272.381 189.489 272.617C189.582 272.853 189.62 273.107 189.6 273.36C189.58 273.613 189.502 273.857 189.373 274.075C188.88 274.838 188.311 275.552 187.79 276.286L187.703 276.412C187.517 276.65 187.278 276.842 187.004 276.971C186.731 277.1 186.431 277.163 186.129 277.155H185.782L182.016 276.527C181.698 276.74 181.37 276.943 181.051 277.136L181.65 278.323C181.764 278.549 181.824 278.799 181.824 279.052C181.824 279.305 181.764 279.555 181.65 279.781C181.548 280.002 181.397 280.196 181.207 280.348C181.018 280.501 180.796 280.607 180.559 280.659L176.35 281.673C176.182 281.691 176.014 281.688 175.848 281.663ZM165.576 271.44C165.792 271.756 166.063 272.031 166.377 272.251C166.53 272.368 166.692 272.474 166.86 272.569C167.5 272.928 168.19 273.192 168.906 273.351C169.735 273.547 170.584 273.644 171.436 273.641C173.022 273.673 174.592 273.308 176.002 272.579C176.172 272.482 176.336 272.375 176.494 272.26C176.777 272.065 177.026 271.824 177.228 271.546C177.412 271.336 177.523 271.072 177.545 270.794C177.568 270.516 177.501 270.238 177.354 270.001L177.276 269.886L177.18 269.741L177.074 269.615C176.228 268.85 175.19 268.329 174.071 268.109C173.242 267.914 172.393 267.817 171.542 267.82C169.012 267.82 166.628 268.679 165.749 269.914C165.596 270.096 165.495 270.316 165.456 270.55C165.416 270.785 165.441 271.025 165.527 271.247V271.353L165.634 271.536L165.576 271.44Z"
        fill="#0C1A33"
      />
      <path
        d="M231.049 299.756L265.687 279.753C265.912 279.678 266.108 279.534 266.246 279.342C266.385 279.149 266.46 278.918 266.46 278.681C266.46 278.444 266.385 278.213 266.246 278.021C266.108 277.828 265.912 277.684 265.687 277.609L257.192 272.705C256.615 272.415 255.979 272.264 255.333 272.264C254.688 272.264 254.051 272.415 253.475 272.705L218.846 292.679C218.621 292.754 218.425 292.898 218.287 293.09C218.148 293.283 218.073 293.514 218.073 293.751C218.073 293.988 218.148 294.219 218.287 294.411C218.425 294.604 218.621 294.748 218.846 294.823L223.084 297.275L179.641 322.375C178.618 322.964 179.052 323.167 179.525 323.447C179.998 323.727 180.365 323.968 181.388 323.379L224.831 298.279L227.332 299.727C227.906 300.022 228.541 300.178 229.187 300.183C229.832 300.188 230.47 300.042 231.049 299.756Z"
        fill="#0C1A33"
      />
      <path
        d="M217.987 261.37C215.293 261.421 212.632 260.773 210.264 259.487C210.023 259.343 209.791 259.198 209.569 259.043L208.295 258.146L219.194 256.543L220.159 254.487L217.215 252.759L206.827 254.284L207.165 253.319C207.785 251.828 208.919 250.609 210.361 249.882C212.708 248.642 215.332 248.017 217.987 248.067C219.16 248.066 220.331 248.179 221.482 248.405L257.018 227.89C256.583 226.989 256.477 225.964 256.719 224.994C256.979 224.16 257.405 223.387 257.973 222.723C258.541 222.06 259.237 221.518 260.021 221.132C262.373 219.893 265 219.269 267.657 219.317C270.355 219.281 273.017 219.946 275.38 221.248C275.631 221.393 275.853 221.537 276.076 221.692L277.36 222.59L266.46 224.183L265.495 226.249L268.439 227.967L278.827 226.451L278.489 227.417C277.867 228.904 276.733 230.119 275.294 230.844C272.946 232.084 270.322 232.708 267.667 232.659C266.494 232.662 265.324 232.553 264.172 232.331L228.636 252.845C229.078 253.744 229.184 254.771 228.935 255.742C228.671 256.568 228.242 257.331 227.675 257.987C227.107 258.642 226.413 259.176 225.633 259.555C223.281 260.802 220.649 261.427 217.987 261.37Z"
        fill="#0C1A33"
      />
      <path
        d="M355.779 308.975C386.192 308.975 410.846 294.742 410.846 277.184C410.846 259.627 386.192 245.394 355.779 245.394C325.367 245.394 300.713 259.627 300.713 277.184C300.713 294.742 325.367 308.975 355.779 308.975Z"
        fill="black"
      />
      <path
        d="M120.269 199.228L180.182 233.818L236.06 201.448L176.147 166.867L120.269 199.228Z"
        fill="#0C1A33"
      />
      <path
        d="M97.5248 283.024L178.31 236.386L107.623 195.578L26.8477 242.217L97.5248 283.024Z"
        fill="#0C1A33"
      />
      <path
        d="M180.172 231.25L123.455 198.494V196.022L175.586 165.921L232.304 198.667V201.148L180.172 231.25Z"
        fill="#489CFF"
      />
      <path
        d="M123.455 196.022L180.172 228.769L232.304 198.667L175.586 165.921L123.455 196.022Z"
        fill="#489CFF"
      />
      <path
        opacity="0.15"
        d="M123.455 196.022L180.172 228.769L232.304 198.667L175.586 165.921L123.455 196.022Z"
        fill="black"
      />
      <path
        d="M194.228 209.653C194.817 209.953 195.469 210.109 196.13 210.109C196.791 210.109 197.443 209.953 198.032 209.653C198.26 209.574 198.458 209.427 198.599 209.23C198.739 209.034 198.814 208.798 198.814 208.557C198.814 208.316 198.739 208.08 198.599 207.884C198.458 207.687 198.26 207.54 198.032 207.461C197.443 207.161 196.791 207.005 196.13 207.005C195.469 207.005 194.817 207.161 194.228 207.461C194 207.54 193.802 207.687 193.661 207.884C193.521 208.08 193.446 208.316 193.446 208.557C193.446 208.798 193.521 209.034 193.661 209.23C193.802 209.427 194 209.574 194.228 209.653Z"
        fill="#37474F"
      />
      <path
        opacity="0.3"
        d="M180.172 228.768V231.249L123.455 198.493V196.021L180.172 228.768Z"
        fill="black"
      />
      <path
        d="M106.049 1.99515L169.244 38.4777C169.859 38.8792 170.37 39.4195 170.738 40.0551C171.105 40.6907 171.318 41.4037 171.359 42.1366V234.068C171.319 234.803 171.107 235.518 170.74 236.155C170.373 236.793 169.86 237.335 169.244 237.737L100.894 277.232C100.239 277.565 99.5144 277.739 98.7796 277.739C98.0447 277.739 97.3203 277.565 96.6654 277.232L33.47 240.71C32.8598 240.307 32.3531 239.765 31.9908 239.13C31.6285 238.494 31.4209 237.782 31.3848 237.052V45.1197C31.4242 44.3851 31.6362 43.6702 32.0035 43.0328C32.3708 42.3954 32.8832 41.8536 33.499 41.4512L101.859 1.99515C102.509 1.66829 103.227 1.49805 103.954 1.49805C104.682 1.49805 105.399 1.66829 106.049 1.99515Z"
        fill="#489CFF"
      />
      <path
        d="M31.3848 45.1109V237.052C31.4272 237.785 31.6405 238.497 32.0076 239.133C32.3748 239.768 32.8855 240.309 33.499 240.711L96.6654 277.232C97.3131 277.585 98.0422 277.762 98.7796 277.744V80.8888C98.0428 80.9007 97.3151 80.7246 96.6654 80.3771L33.47 43.8945L32.8908 43.4118L32.1378 42.7939C31.6743 43.4803 31.4133 44.2832 31.3848 45.1109Z"
        fill="#489CFF"
      />
      <path
        opacity="0.3"
        d="M31.3848 45.1109V237.052C31.4272 237.785 31.6405 238.497 32.0076 239.133C32.3748 239.768 32.8855 240.309 33.499 240.711L96.6654 277.232C97.3131 277.585 98.0422 277.762 98.7796 277.744V80.8888C98.0428 80.9007 97.3151 80.7246 96.6654 80.3771L33.47 43.8945L32.8908 43.4118L32.1378 42.7939C31.6743 43.4803 31.4133 44.2832 31.3848 45.1109Z"
        fill="black"
      />
      <path
        d="M33.4699 41.4521L101.859 1.99602C102.512 1.66292 103.235 1.48926 103.969 1.48926C104.702 1.48926 105.425 1.66292 106.078 1.99602L169.273 38.4786C169.527 38.5671 169.746 38.7322 169.901 38.951C170.057 39.1698 170.14 39.4315 170.14 39.6999C170.14 39.9682 170.057 40.2299 169.901 40.4487C169.746 40.6675 169.527 40.8326 169.273 40.9211L100.894 80.3868C100.237 80.7155 99.5135 80.8867 98.7795 80.8867C98.0455 80.8867 97.3216 80.7155 96.6653 80.3868L33.4699 43.8945C33.2166 43.806 32.9971 43.6409 32.8418 43.4221C32.6865 43.2033 32.603 42.9416 32.603 42.6733C32.603 42.405 32.6865 42.1433 32.8418 41.9245C32.9971 41.7057 33.2166 41.5406 33.4699 41.4521Z"
        fill="#489CFF"
      />
      <path
        opacity="0.15"
        d="M33.4699 41.4521L101.859 1.99602C102.512 1.66292 103.235 1.48926 103.969 1.48926C104.702 1.48926 105.425 1.66292 106.078 1.99602L169.273 38.4786C169.527 38.5671 169.746 38.7322 169.901 38.951C170.057 39.1698 170.14 39.4315 170.14 39.6999C170.14 39.9682 170.057 40.2299 169.901 40.4487C169.746 40.6675 169.527 40.8326 169.273 40.9211L100.894 80.3868C100.237 80.7155 99.5135 80.8867 98.7795 80.8867C98.0455 80.8867 97.3216 80.7155 96.6653 80.3868L33.4699 43.8945C33.2166 43.806 32.9971 43.6409 32.8418 43.4221C32.6865 43.2033 32.603 42.9416 32.603 42.6733C32.603 42.405 32.6865 42.1433 32.8418 41.9245C32.9971 41.7057 33.2166 41.5406 33.4699 41.4521Z"
        fill="black"
      />
      <path
        d="M163.77 171.637L107.053 204.383V264.576L163.77 231.83V171.637Z"
        fill="#455A64"
      />
      <path
        d="M135.899 214.637C135.804 214.594 135.7 214.578 135.597 214.588C135.494 214.598 135.395 214.635 135.31 214.695L112.237 227.998C111.992 228.151 111.79 228.364 111.648 228.616L110.104 227.718C110.242 227.466 110.441 227.254 110.683 227.1L133.273 214.115C133.528 213.982 133.811 213.913 134.099 213.913C134.386 213.913 134.669 213.982 134.924 214.115L135.899 214.637Z"
        fill="#263238"
      />
      <path
        d="M111.407 229.407V255.251C111.407 255.782 111.783 255.994 112.237 255.734L135.31 242.479C135.551 242.322 135.751 242.111 135.895 241.863C136.039 241.615 136.123 241.336 136.14 241.05V215.177C136.14 214.646 135.764 214.434 135.31 214.694L112.237 227.997C111.998 228.152 111.799 228.359 111.655 228.604C111.511 228.849 111.426 229.124 111.407 229.407Z"
        fill="#37474F"
      />
      <path
        d="M111.629 255.781L110.663 255.23C110.424 255.073 110.225 254.862 110.083 254.614C109.94 254.366 109.858 254.088 109.843 253.802V228.508C109.849 228.218 109.933 227.935 110.084 227.688L111.629 228.585C111.482 228.835 111.399 229.117 111.387 229.406V255.25C111.382 255.351 111.401 255.453 111.443 255.545C111.485 255.637 111.549 255.718 111.629 255.781Z"
        fill="#263238"
      />
      <path
        d="M113.251 253.407C113.184 253.406 113.118 253.389 113.058 253.359C112.987 253.31 112.93 253.243 112.892 253.165C112.855 253.087 112.839 253.001 112.846 252.915V229.928C112.864 229.733 112.927 229.545 113.029 229.378C113.131 229.211 113.27 229.069 113.435 228.963L134.007 217.137C134.082 217.093 134.167 217.069 134.254 217.069C134.34 217.069 134.425 217.093 134.5 217.137C134.571 217.186 134.628 217.252 134.665 217.33C134.703 217.408 134.719 217.495 134.712 217.581V240.548C134.703 240.751 134.646 240.949 134.545 241.125C134.445 241.301 134.303 241.451 134.133 241.562L113.56 253.388C113.46 253.42 113.354 253.427 113.251 253.407ZM134.297 217.262C134.236 217.265 134.177 217.281 134.123 217.311L113.541 229.146C113.407 229.235 113.296 229.354 113.215 229.493C113.135 229.632 113.088 229.788 113.077 229.948V252.915C113.077 253.031 113.077 253.117 113.164 253.146C113.202 253.168 113.246 253.18 113.29 253.18C113.334 253.18 113.377 253.168 113.415 253.146L133.988 241.32C134.122 241.231 134.233 241.113 134.314 240.974C134.394 240.835 134.441 240.679 134.451 240.519V217.513C134.451 217.398 134.451 217.311 134.365 217.282C134.345 217.268 134.321 217.262 134.297 217.262Z"
        fill="#489CFF"
      />
      <path
        d="M114.873 250.385C114.835 250.397 114.795 250.397 114.757 250.385C114.715 250.361 114.681 250.326 114.659 250.283C114.636 250.24 114.627 250.192 114.631 250.144V230.623C114.642 230.528 114.673 230.436 114.721 230.353C114.769 230.27 114.834 230.197 114.911 230.141L132.491 220.014C132.534 219.988 132.582 219.975 132.631 219.975C132.681 219.975 132.729 219.988 132.771 220.014C132.813 220.041 132.846 220.078 132.868 220.122C132.89 220.166 132.9 220.215 132.897 220.265V239.785C132.893 239.884 132.865 239.98 132.817 240.065C132.768 240.151 132.699 240.224 132.617 240.277L115.056 250.366C114.998 250.391 114.934 250.398 114.873 250.385ZM132.636 220.226L115.047 230.363C114.967 230.433 114.912 230.528 114.892 230.633V250.202L132.443 240.113C132.49 240.082 132.529 240.04 132.557 239.992C132.585 239.944 132.603 239.889 132.607 239.833V220.313C132.62 220.315 132.634 220.312 132.645 220.305C132.656 220.298 132.665 220.288 132.669 220.275C132.674 220.263 132.674 220.249 132.669 220.237C132.665 220.225 132.657 220.214 132.646 220.207L132.636 220.226Z"
        fill="#489CFF"
      />
      <path
        d="M116.06 224.994C115.998 224.992 115.939 224.965 115.896 224.92C115.853 224.875 115.829 224.815 115.829 224.753V219.25C115.829 219.189 115.853 219.13 115.896 219.086C115.94 219.043 115.999 219.019 116.06 219.019C116.122 219.019 116.181 219.043 116.224 219.086C116.268 219.13 116.292 219.189 116.292 219.25V224.753C116.292 224.815 116.268 224.875 116.225 224.92C116.182 224.965 116.123 224.992 116.06 224.994Z"
        fill="#489CFF"
      />
      <path
        d="M119.912 222.783C119.881 222.783 119.851 222.777 119.823 222.766C119.795 222.754 119.77 222.737 119.748 222.716C119.727 222.694 119.709 222.669 119.698 222.64C119.686 222.612 119.68 222.582 119.68 222.552V217.039C119.68 217.009 119.686 216.979 119.698 216.951C119.709 216.923 119.727 216.897 119.748 216.875C119.77 216.854 119.795 216.837 119.823 216.825C119.851 216.814 119.881 216.808 119.912 216.808C119.973 216.808 120.032 216.832 120.076 216.875C120.119 216.919 120.144 216.978 120.144 217.039V222.552C120.144 222.613 120.119 222.672 120.076 222.716C120.032 222.759 119.973 222.783 119.912 222.783Z"
        fill="#489CFF"
      />
      <path
        d="M123.764 220.572C123.733 220.574 123.702 220.569 123.673 220.558C123.643 220.547 123.617 220.53 123.594 220.508C123.572 220.486 123.554 220.46 123.541 220.432C123.529 220.403 123.523 220.372 123.523 220.341V214.838C123.534 214.783 123.564 214.734 123.607 214.699C123.65 214.664 123.704 214.645 123.759 214.645C123.815 214.645 123.869 214.664 123.912 214.699C123.955 214.734 123.985 214.783 123.996 214.838V220.341C123.996 220.402 123.972 220.461 123.928 220.505C123.885 220.548 123.826 220.572 123.764 220.572Z"
        fill="#489CFF"
      />
      <path
        d="M127.606 218.343C127.545 218.343 127.486 218.318 127.442 218.275C127.399 218.231 127.375 218.172 127.375 218.111V212.599C127.386 212.544 127.415 212.495 127.458 212.46C127.501 212.425 127.555 212.405 127.611 212.405C127.667 212.405 127.721 212.425 127.764 212.46C127.807 212.495 127.836 212.544 127.848 212.599V218.111C127.845 218.173 127.819 218.232 127.774 218.276C127.729 218.319 127.669 218.343 127.606 218.343Z"
        fill="#489CFF"
      />
      <path
        d="M131.458 216.16C131.397 216.16 131.338 216.135 131.294 216.092C131.251 216.048 131.227 215.99 131.227 215.928V210.425C131.227 210.395 131.233 210.365 131.244 210.337C131.256 210.309 131.273 210.283 131.294 210.261C131.316 210.24 131.341 210.223 131.37 210.211C131.398 210.2 131.428 210.194 131.458 210.194C131.489 210.192 131.52 210.197 131.549 210.208C131.577 210.22 131.603 210.237 131.625 210.258C131.647 210.28 131.664 210.306 131.675 210.335C131.686 210.364 131.691 210.394 131.69 210.425V215.928C131.691 215.959 131.686 215.99 131.675 216.018C131.664 216.047 131.647 216.073 131.625 216.095C131.603 216.117 131.577 216.134 131.549 216.145C131.52 216.156 131.489 216.161 131.458 216.16Z"
        fill="#489CFF"
      />
      <path
        d="M135.899 194.305C135.805 194.261 135.7 194.243 135.597 194.253C135.493 194.263 135.394 194.301 135.31 194.363L112.237 207.637C111.995 207.789 111.793 207.998 111.648 208.245L110.104 207.357C110.239 207.104 110.439 206.891 110.683 206.739L133.273 193.745C133.529 193.617 133.812 193.55 134.099 193.55C134.385 193.55 134.668 193.617 134.924 193.745L135.899 194.305Z"
        fill="#263238"
      />
      <path
        d="M111.407 209.066V222.099C111.407 222.63 111.783 222.843 112.237 222.582L135.31 209.308C135.551 209.151 135.751 208.94 135.895 208.692C136.039 208.444 136.123 208.166 136.14 207.879V194.836C136.14 194.315 135.764 194.103 135.31 194.363L112.237 207.638C111.995 207.793 111.794 208.004 111.65 208.252C111.505 208.5 111.422 208.78 111.407 209.066Z"
        fill="#37474F"
      />
      <path
        d="M111.629 222.629L110.663 222.079C110.424 221.922 110.225 221.711 110.083 221.462C109.94 221.214 109.858 220.936 109.843 220.65V208.167C109.851 207.881 109.934 207.601 110.084 207.356L111.629 208.245C111.483 208.494 111.4 208.776 111.387 209.065V222.108C111.382 222.208 111.402 222.307 111.444 222.398C111.486 222.489 111.549 222.568 111.629 222.629Z"
        fill="#263238"
      />
      <path
        d="M113.251 220.273C113.184 220.271 113.118 220.255 113.058 220.224C112.987 220.176 112.93 220.109 112.892 220.031C112.855 219.953 112.839 219.866 112.846 219.78V209.653C112.864 209.458 112.927 209.27 113.029 209.103C113.131 208.936 113.27 208.794 113.435 208.688L134.007 196.852C134.077 196.8 134.161 196.77 134.248 196.765C134.336 196.761 134.422 196.782 134.497 196.826C134.572 196.87 134.633 196.936 134.671 197.014C134.709 197.093 134.724 197.181 134.712 197.267V207.414C134.703 207.616 134.646 207.814 134.545 207.991C134.445 208.167 134.303 208.317 134.133 208.427L113.56 220.253C113.46 220.286 113.354 220.293 113.251 220.273ZM134.297 196.939C134.235 196.94 134.174 196.96 134.123 196.997L113.541 208.813C113.407 208.902 113.296 209.021 113.215 209.16C113.135 209.299 113.088 209.454 113.077 209.615V219.761C113.077 219.877 113.077 219.964 113.164 219.993C113.202 220.015 113.246 220.026 113.29 220.026C113.334 220.026 113.377 220.015 113.415 219.993L133.988 208.167C134.122 208.078 134.233 207.959 134.314 207.82C134.394 207.681 134.441 207.525 134.451 207.365V197.18C134.457 197.136 134.452 197.091 134.437 197.049C134.422 197.007 134.397 196.969 134.365 196.939H134.297Z"
        fill="#489CFF"
      />
      <path
        d="M114.873 217.242C114.835 217.253 114.795 217.253 114.757 217.242C114.715 217.218 114.681 217.182 114.659 217.139C114.636 217.097 114.627 217.048 114.631 217V210.329C114.643 210.233 114.674 210.14 114.722 210.055C114.77 209.971 114.834 209.896 114.911 209.837L132.491 199.71C132.534 199.686 132.582 199.674 132.631 199.674C132.68 199.674 132.729 199.686 132.771 199.71C132.813 199.735 132.846 199.77 132.868 199.813C132.89 199.856 132.9 199.903 132.897 199.951V206.709C132.893 206.808 132.865 206.904 132.817 206.989C132.768 207.075 132.699 207.148 132.617 207.201L115.056 217.29C114.992 217.293 114.928 217.276 114.873 217.242ZM132.636 199.864L115.047 209.991C114.968 210.067 114.914 210.165 114.892 210.271V216.981L132.443 206.893C132.49 206.861 132.529 206.82 132.557 206.771C132.585 206.723 132.603 206.668 132.607 206.613V199.855C132.607 199.855 132.646 199.874 132.646 199.864H132.636Z"
        fill="#489CFF"
      />
      <path
        d="M140.562 191.649C140.164 191.908 139.833 192.257 139.595 192.668C139.357 193.079 139.219 193.54 139.191 194.014C139.191 194.883 139.809 195.231 140.562 194.796C140.957 194.537 141.286 194.189 141.522 193.78C141.758 193.371 141.896 192.912 141.924 192.441C141.924 191.572 141.306 191.214 140.562 191.649Z"
        fill="#489CFF"
      />
      <path
        d="M140.562 214.819C140.164 215.078 139.833 215.427 139.595 215.838C139.357 216.249 139.219 216.71 139.191 217.184C139.191 218.053 139.809 218.4 140.562 217.966C140.957 217.707 141.286 217.359 141.522 216.95C141.758 216.541 141.896 216.082 141.924 215.61C141.924 214.742 141.306 214.384 140.562 214.819Z"
        fill="#489CFF"
      />
      <path
        d="M140.562 221.056C140.165 221.313 139.833 221.661 139.595 222.07C139.357 222.479 139.219 222.939 139.191 223.412C139.191 224.281 139.809 224.628 140.562 224.194C140.958 223.935 141.288 223.588 141.524 223.178C141.76 222.769 141.897 222.31 141.924 221.838C141.924 220.969 141.306 220.622 140.562 221.056Z"
        fill="#489CFF"
      />
      <path
        d="M140.562 227.283C140.164 227.542 139.833 227.891 139.595 228.302C139.357 228.713 139.219 229.174 139.191 229.648C139.191 230.507 139.809 230.864 140.562 230.43C140.957 230.17 141.286 229.822 141.522 229.413C141.758 229.004 141.896 228.546 141.924 228.074C141.924 227.205 141.306 226.848 140.562 227.283Z"
        fill="#489CFF"
      />
      <path
        d="M140.562 233.519C140.165 233.776 139.833 234.124 139.595 234.533C139.357 234.942 139.219 235.402 139.191 235.875C139.191 236.744 139.809 237.091 140.562 236.657C140.957 236.397 141.286 236.049 141.522 235.64C141.758 235.231 141.896 234.773 141.924 234.301C141.924 233.432 141.306 233.085 140.562 233.519Z"
        fill="#489CFF"
      />
      <path
        d="M140.562 197.277C140.164 197.536 139.833 197.885 139.595 198.296C139.357 198.707 139.219 199.168 139.191 199.642C139.191 200.511 139.809 200.859 140.562 200.424C140.957 200.165 141.286 199.817 141.522 199.408C141.758 198.999 141.896 198.54 141.924 198.068C141.924 197.2 141.306 196.842 140.562 197.277Z"
        fill="#489CFF"
      />
      <path
        d="M140.562 202.896C140.164 203.155 139.833 203.504 139.595 203.915C139.357 204.326 139.219 204.787 139.191 205.261C139.191 206.13 139.809 206.478 140.562 206.043C140.957 205.784 141.286 205.436 141.522 205.027C141.758 204.618 141.896 204.159 141.924 203.688C141.924 202.828 141.306 202.471 140.562 202.896Z"
        fill="#489CFF"
      />
      <path
        d="M155.459 194.595L155.324 193.33C155.323 193.309 155.318 193.289 155.308 193.271C155.298 193.252 155.285 193.237 155.268 193.224C155.251 193.212 155.232 193.204 155.211 193.201C155.191 193.197 155.17 193.199 155.15 193.204C154.944 193.25 154.733 193.273 154.522 193.272C154.473 193.27 154.425 193.28 154.381 193.302C154.337 193.324 154.299 193.357 154.271 193.398L153.991 193.813L155.478 194.662C155.466 194.642 155.459 194.618 155.459 194.595ZM156.53 185.732C156.536 185.832 156.536 185.932 156.53 186.032C156.5 186.804 156.284 187.559 155.901 188.231C155.518 188.903 154.979 189.473 154.329 189.893L154.059 190.038C154.194 191.216 155.121 191.65 156.241 191.003C156.884 190.588 157.419 190.025 157.801 189.362C158.184 188.699 158.404 187.955 158.442 187.19C158.461 185.838 157.621 185.259 156.53 185.732ZM161.3 184.612L159.803 183.744L159.581 184.197C159.56 184.241 159.549 184.289 159.549 184.337C159.549 184.386 159.56 184.434 159.581 184.477C159.689 184.66 159.776 184.854 159.842 185.056C159.847 185.076 159.856 185.095 159.869 185.11C159.882 185.126 159.899 185.138 159.917 185.147C159.936 185.155 159.956 185.159 159.976 185.158C159.997 185.158 160.017 185.153 160.035 185.143L161.193 184.632C161.208 184.625 161.225 184.621 161.242 184.621C161.258 184.621 161.275 184.625 161.29 184.632L161.3 184.612ZM158.403 181.311L157.563 180.818H157.438L156.752 181.851L155.884 181.349L155.256 180.982H155.189L154.291 181.504L153.885 181.735H153.837V181.793C153.834 181.819 153.834 181.845 153.837 181.871L153.711 183.328C153.714 183.344 153.714 183.361 153.711 183.377C153.715 183.393 153.715 183.409 153.711 183.425C153.686 183.489 153.65 183.547 153.605 183.599C153.344 183.836 153.099 184.091 152.872 184.361C152.861 184.38 152.845 184.397 152.827 184.408C152.808 184.42 152.787 184.427 152.765 184.429C152.718 184.443 152.668 184.443 152.621 184.429L151.781 184.139H151.636H151.588L151.192 184.825L150.777 185.549L150.632 185.8C150.619 185.841 150.619 185.885 150.632 185.925C150.64 185.973 150.664 186.018 150.699 186.051L151.375 186.63L151.423 186.698C151.432 186.733 151.432 186.769 151.423 186.804C151.429 186.829 151.429 186.856 151.423 186.881C151.298 187.212 151.195 187.55 151.115 187.895C151.099 187.954 151.073 188.009 151.037 188.059L150.95 188.155L149.763 188.986C149.763 188.986 149.705 189.044 149.676 189.082C149.662 189.136 149.662 189.192 149.676 189.246V190.752C149.669 190.768 149.666 190.784 149.666 190.801C149.666 190.817 149.669 190.834 149.676 190.849L151.192 191.718L150.641 192.828C150.623 192.867 150.62 192.911 150.633 192.952C150.645 192.993 150.672 193.028 150.709 193.05L152.543 194.112H152.621C152.596 194.092 152.573 194.069 152.553 194.044C152.541 194.019 152.535 193.991 152.535 193.962C152.535 193.934 152.541 193.906 152.553 193.88L153.325 192.326C153.331 192.304 153.331 192.281 153.325 192.258C153.353 192.19 153.353 192.114 153.325 192.046C153.223 191.862 153.139 191.668 153.074 191.467C153.069 191.447 153.06 191.429 153.047 191.413C153.034 191.397 153.017 191.385 152.999 191.377C152.98 191.368 152.96 191.364 152.94 191.365C152.919 191.365 152.899 191.371 152.881 191.38L152.292 191.641L151.713 191.901C151.699 191.911 151.682 191.916 151.665 191.916C151.648 191.916 151.631 191.911 151.617 191.901C151.6 191.892 151.586 191.877 151.578 191.86C151.569 191.843 151.566 191.824 151.568 191.805V190.299C151.571 190.244 151.587 190.191 151.617 190.144C151.617 190.144 151.617 190.067 151.703 190.048L152.891 189.208C152.93 189.183 152.963 189.15 152.987 189.111C153.022 189.066 153.045 189.013 153.055 188.957C153.136 188.612 153.239 188.274 153.364 187.943C153.369 187.914 153.369 187.885 153.364 187.856C153.361 187.791 153.333 187.728 153.287 187.682L152.611 187.103C152.593 187.089 152.579 187.072 152.569 187.052C152.559 187.032 152.554 187.01 152.553 186.987C152.539 186.947 152.539 186.902 152.553 186.862L152.833 186.389L153.161 185.8L153.509 185.201C153.546 185.15 153.601 185.116 153.663 185.105H153.75L154.58 185.394C154.63 185.414 154.685 185.414 154.735 185.394C154.775 185.381 154.812 185.358 154.841 185.327C155.065 185.054 155.307 184.796 155.565 184.554C155.613 184.509 155.65 184.453 155.671 184.39C155.68 184.355 155.68 184.319 155.671 184.284L155.806 182.836C155.802 182.81 155.802 182.784 155.806 182.759C155.832 182.691 155.88 182.633 155.942 182.595L156.183 182.45L156.704 182.151L157.245 181.842C157.258 181.835 157.273 181.831 157.288 181.831C157.303 181.831 157.318 181.835 157.332 181.842C157.332 181.842 157.38 181.89 157.39 181.929L157.457 182.595L157.515 183.193C157.516 183.214 157.521 183.234 157.531 183.253C157.541 183.271 157.554 183.287 157.571 183.299C157.588 183.311 157.607 183.319 157.628 183.322C157.648 183.326 157.669 183.325 157.689 183.319C157.895 183.273 158.105 183.251 158.316 183.251C158.349 183.255 158.383 183.252 158.414 183.242C158.446 183.232 158.475 183.215 158.5 183.193C158.529 183.178 158.553 183.155 158.567 183.126L159.533 181.678C159.556 181.642 159.593 181.618 159.634 181.609C159.676 181.6 159.719 181.607 159.755 181.629L158.403 181.311Z"
        fill="#263238"
      />
      <path
        opacity="0.1"
        d="M155.478 183.068C155.473 183.094 155.473 183.12 155.478 183.145L155.342 184.594C155.351 184.628 155.351 184.665 155.342 184.7L153.412 183.59V183.541C153.414 183.525 153.414 183.509 153.412 183.493L153.537 182.035C153.535 182.01 153.535 181.984 153.537 181.958L155.478 183.068Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M154.358 185.703C154.308 185.722 154.253 185.722 154.203 185.703L153.373 185.413H153.286L151.355 184.322H151.413L152.253 184.612C152.301 184.626 152.351 184.626 152.398 184.612L154.358 185.703Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M152.968 188.155L151.037 187.054C151.037 187.054 151.037 186.996 151.037 186.958L150.988 186.89L150.313 186.311C150.277 186.278 150.253 186.234 150.245 186.186L152.176 187.296C152.177 187.318 152.182 187.34 152.192 187.36C152.202 187.38 152.216 187.398 152.234 187.412L152.91 187.991C152.947 188.037 152.968 188.095 152.968 188.155Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M152.958 192.557C152.964 192.579 152.964 192.602 152.958 192.625L152.186 194.179C152.174 194.205 152.167 194.233 152.167 194.261C152.167 194.289 152.174 194.317 152.186 194.343L150.361 193.281C150.325 193.259 150.297 193.224 150.285 193.183C150.272 193.142 150.276 193.098 150.294 193.059L150.844 191.949L151.249 192.181C151.264 192.19 151.28 192.195 151.298 192.195C151.315 192.195 151.332 192.19 151.346 192.181L151.925 191.92L152.958 192.557Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M155.458 194.594C155.453 194.616 155.453 194.639 155.458 194.662L153.972 193.812L154.252 193.397C154.279 193.356 154.317 193.324 154.361 193.302C154.405 193.28 154.454 193.269 154.503 193.271C154.714 193.272 154.924 193.25 155.13 193.204C155.15 193.198 155.171 193.197 155.191 193.2C155.212 193.204 155.231 193.212 155.248 193.224C155.265 193.236 155.279 193.252 155.288 193.27C155.298 193.288 155.303 193.309 155.304 193.329L155.458 194.594Z"
        fill="black"
      />
      <path
        opacity="0.1"
        d="M159.436 181.928C159.4 181.906 159.356 181.899 159.315 181.908C159.274 181.917 159.237 181.941 159.214 181.976L158.248 183.425C158.234 183.454 158.21 183.477 158.181 183.492L157.64 183.183L157.138 182.894L157.071 182.227C157.071 182.227 157.071 182.15 157.013 182.141L156.626 181.909L157.312 180.876C157.347 180.867 157.383 180.867 157.418 180.876L158.258 181.368L159.436 181.928Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M152.581 189.381C152.557 189.42 152.524 189.453 152.485 189.477L151.297 190.317C151.297 190.317 151.24 190.375 151.211 190.414L149.28 189.304C149.28 189.304 149.328 189.226 149.367 189.207L150.554 188.377L150.641 188.28L152.581 189.381Z"
        fill="black"
      />
      <path
        d="M161.212 184.622L160.054 185.143C160.036 185.153 160.016 185.158 159.995 185.158C159.975 185.159 159.955 185.155 159.936 185.146C159.918 185.138 159.901 185.126 159.888 185.11C159.875 185.094 159.866 185.076 159.861 185.056C159.795 184.854 159.708 184.66 159.6 184.477C159.579 184.433 159.567 184.386 159.567 184.337C159.567 184.289 159.579 184.241 159.6 184.197L160.372 182.643C160.391 182.603 160.394 182.557 160.382 182.514C160.369 182.472 160.342 182.435 160.305 182.411L159.465 181.928C159.427 181.906 159.382 181.898 159.339 181.907C159.296 181.916 159.258 181.941 159.233 181.977L158.268 183.425C158.24 183.465 158.202 183.498 158.158 183.52C158.114 183.542 158.066 183.552 158.017 183.55C157.806 183.55 157.595 183.573 157.389 183.618C157.369 183.624 157.348 183.625 157.328 183.622C157.307 183.618 157.288 183.61 157.271 183.598C157.254 183.586 157.241 183.57 157.231 183.552C157.221 183.533 157.216 183.513 157.215 183.492L157.09 182.228C157.09 182.21 157.085 182.192 157.075 182.177C157.066 182.162 157.052 182.149 157.036 182.141C157.02 182.133 157.002 182.13 156.985 182.132C156.967 182.133 156.95 182.14 156.935 182.15L155.632 182.903C155.589 182.929 155.553 182.964 155.526 183.006C155.499 183.048 155.483 183.095 155.478 183.145L155.342 184.593C155.33 184.696 155.282 184.791 155.207 184.863C154.949 185.104 154.707 185.362 154.483 185.636C154.45 185.669 154.408 185.694 154.362 185.706C154.316 185.718 154.268 185.717 154.223 185.703L153.392 185.414C153.348 185.4 153.301 185.401 153.257 185.416C153.213 185.432 153.176 185.462 153.151 185.5L152.186 187.161C152.16 187.196 152.146 187.238 152.146 187.282C152.146 187.325 152.16 187.367 152.186 187.402L152.861 187.981C152.895 188.015 152.92 188.056 152.931 188.102C152.943 188.148 152.942 188.197 152.929 188.242C152.804 188.573 152.701 188.912 152.62 189.256C152.6 189.358 152.541 189.448 152.456 189.507L151.268 190.347C151.228 190.376 151.196 190.413 151.172 190.457C151.149 190.5 151.135 190.548 151.133 190.598V192.104C151.131 192.122 151.133 192.14 151.141 192.157C151.148 192.173 151.16 192.187 151.175 192.198C151.19 192.208 151.208 192.213 151.226 192.214C151.244 192.214 151.262 192.21 151.278 192.2L152.446 191.679C152.464 191.67 152.484 191.664 152.505 191.664C152.525 191.663 152.545 191.667 152.564 191.676C152.582 191.684 152.599 191.697 152.612 191.712C152.625 191.728 152.634 191.746 152.639 191.766C152.704 191.967 152.788 192.161 152.89 192.345C152.914 192.388 152.926 192.436 152.926 192.485C152.926 192.534 152.914 192.582 152.89 192.625L152.118 194.179C152.099 194.22 152.096 194.265 152.108 194.308C152.121 194.351 152.148 194.387 152.186 194.411L153.025 194.894C153.057 194.922 153.099 194.938 153.141 194.938C153.184 194.938 153.225 194.922 153.257 194.894L154.223 193.446C154.25 193.405 154.288 193.372 154.332 193.35C154.376 193.329 154.425 193.318 154.474 193.32C154.685 193.321 154.895 193.298 155.101 193.253C155.121 193.247 155.142 193.245 155.162 193.249C155.183 193.252 155.202 193.26 155.219 193.272C155.236 193.285 155.249 193.3 155.259 193.319C155.269 193.337 155.274 193.357 155.275 193.378L155.41 194.643C155.41 194.73 155.478 194.768 155.564 194.72L156.868 193.967C156.908 193.939 156.943 193.903 156.97 193.862C156.996 193.821 157.014 193.774 157.022 193.726L157.148 192.278C157.161 192.175 157.208 192.079 157.283 192.007C157.544 191.769 157.786 191.51 158.007 191.235C158.04 191.201 158.082 191.177 158.128 191.165C158.174 191.153 158.222 191.154 158.268 191.167L159.108 191.457C159.15 191.47 159.196 191.469 159.238 191.453C159.28 191.437 159.316 191.408 159.339 191.37L160.305 189.71C160.332 189.675 160.347 189.633 160.347 189.589C160.347 189.545 160.332 189.502 160.305 189.468L159.629 188.889C159.595 188.856 159.571 188.814 159.559 188.768C159.547 188.722 159.548 188.674 159.561 188.628C159.686 188.297 159.789 187.959 159.87 187.615C159.894 187.514 159.952 187.426 160.034 187.364L161.222 186.524C161.262 186.495 161.295 186.457 161.318 186.414C161.341 186.37 161.355 186.322 161.357 186.273V184.767C161.376 184.747 161.387 184.721 161.387 184.694C161.387 184.667 161.376 184.641 161.357 184.622C161.338 184.603 161.312 184.592 161.284 184.592C161.257 184.592 161.231 184.603 161.212 184.622ZM156.26 190.955C155.043 191.66 154.058 191.09 154.058 189.681C154.097 188.916 154.316 188.171 154.699 187.508C155.082 186.845 155.617 186.283 156.26 185.867C157.476 185.162 158.461 185.732 158.461 187.142C158.422 187.906 158.203 188.651 157.82 189.314C157.438 189.977 156.903 190.539 156.26 190.955Z"
        fill="#37474F"
      />
      <path
        d="M152.243 197.703H152.186L151.877 197.529L150.998 197.018H150.921H150.805L149.984 197.095H149.907L149.463 198.282C149.45 198.32 149.426 198.353 149.396 198.379L149.289 198.456L149.019 198.591L148.169 198.099C148.141 198.091 148.111 198.091 148.083 198.099L147.445 198.775L147.185 199.045C147.18 199.064 147.18 199.084 147.185 199.103V199.161C147.181 199.177 147.181 199.193 147.185 199.209L147.436 200.059C147.436 200.059 147.436 200.126 147.436 200.165C147.44 200.197 147.44 200.229 147.436 200.261C147.281 200.49 147.142 200.728 147.021 200.976C146.999 201.017 146.965 201.051 146.924 201.072C146.896 201.096 146.863 201.112 146.828 201.121L146.055 201.314H145.997C145.981 201.31 145.965 201.31 145.949 201.314C145.945 201.323 145.943 201.333 145.943 201.343C145.943 201.353 145.945 201.363 145.949 201.372L145.659 202.337L145.563 202.684C145.563 202.733 145.563 202.762 145.563 202.781L146.075 203.08H146.142L146.422 203.235C146.422 203.351 146.422 203.466 146.422 203.582C146.425 203.619 146.421 203.656 146.409 203.691C146.398 203.725 146.379 203.758 146.355 203.785L145.573 204.75C145.567 204.786 145.567 204.821 145.573 204.857C145.566 204.872 145.562 204.888 145.562 204.905C145.562 204.922 145.566 204.938 145.573 204.953L145.939 205.764L147.494 206.672H147.571V206.594L147.388 206.199L147.233 205.851C147.223 205.816 147.223 205.78 147.233 205.745C147.238 205.711 147.255 205.68 147.281 205.658L148.063 204.693C148.069 204.661 148.069 204.628 148.063 204.596C148.071 204.558 148.071 204.518 148.063 204.48C148.044 204.255 148.044 204.029 148.063 203.804C148.063 203.789 148.06 203.774 148.054 203.76C148.049 203.746 148.04 203.733 148.029 203.722C148.019 203.712 148.006 203.703 147.992 203.697C147.978 203.692 147.963 203.689 147.947 203.689H147.243H147.185C147.177 203.675 147.173 203.66 147.173 203.645C147.173 203.63 147.177 203.615 147.185 203.602L147.359 203.013L147.552 202.337V202.279C147.588 202.234 147.64 202.203 147.696 202.192L148.469 201.999C148.506 201.995 148.54 201.978 148.565 201.951C148.609 201.932 148.643 201.898 148.662 201.854C148.787 201.609 148.926 201.37 149.077 201.14C149.082 201.108 149.082 201.075 149.077 201.043C149.086 201.008 149.086 200.972 149.077 200.937L148.826 200.088C148.816 200.056 148.816 200.022 148.826 199.991C148.833 199.953 148.854 199.919 148.884 199.894L149.492 199.248L149.782 198.939C149.782 198.939 149.878 198.881 149.907 198.939L150.322 199.576C150.322 199.576 150.429 199.576 150.496 199.576C150.659 199.464 150.83 199.364 151.008 199.277C151.044 199.265 151.077 199.245 151.104 199.219C151.139 199.191 151.166 199.154 151.182 199.112L151.626 197.915V197.857L151.925 197.78L152.581 197.713H152.649L152.243 197.703ZM148.768 206.392C148.775 206.362 148.771 206.331 148.756 206.305C148.742 206.278 148.719 206.258 148.691 206.247C148.562 206.193 148.442 206.122 148.334 206.034C148.305 206.015 148.271 206.005 148.237 206.005C148.203 206.005 148.169 206.015 148.141 206.034L147.31 206.797C147.279 206.805 147.245 206.805 147.214 206.797L148.034 207.27L148.546 207.569C148.54 207.557 148.537 207.544 148.537 207.531C148.537 207.517 148.54 207.504 148.546 207.492L148.768 206.392ZM152.437 197.78C152.443 197.794 152.447 197.809 152.447 197.824C152.447 197.839 152.443 197.854 152.437 197.867L152.176 199.035C152.172 199.066 152.178 199.097 152.194 199.123C152.21 199.149 152.234 199.169 152.263 199.18C152.389 199.238 152.509 199.309 152.62 199.392C152.62 199.392 152.736 199.392 152.803 199.392L153.634 198.62C153.682 198.62 153.73 198.572 153.759 198.62L152.437 197.78ZM151.346 206.469L150.931 205.832C150.931 205.783 150.824 205.783 150.757 205.832L150.506 205.986L151.365 206.488L151.346 206.469ZM152.012 201.15C151.959 201.015 151.871 200.896 151.757 200.807C151.644 200.717 151.508 200.658 151.365 200.638H151.268C151.068 200.636 150.87 200.679 150.689 200.763C150.694 200.837 150.694 200.911 150.689 200.985C150.659 201.557 150.494 202.113 150.207 202.608C149.92 203.104 149.52 203.523 149.038 203.833L148.855 203.94C148.864 204.086 148.9 204.23 148.961 204.364C149.021 204.52 149.127 204.654 149.265 204.747C149.403 204.84 149.567 204.889 149.733 204.886H150.052C150.463 204.756 150.831 204.516 151.114 204.191C151.53 203.815 151.838 203.335 152.005 202.8C152.171 202.265 152.19 201.695 152.06 201.15H152.012ZM153.305 201.15C153.305 201.265 153.305 201.372 153.305 201.478C153.305 201.494 153.309 201.51 153.315 201.524C153.321 201.539 153.331 201.552 153.343 201.563C153.355 201.574 153.368 201.582 153.384 201.588C153.399 201.593 153.415 201.595 153.431 201.594H154.126H154.184L153.305 201.15Z"
        fill="#263238"
      />
      <path
        opacity="0.15"
        d="M151.346 206.469L150.486 205.967L150.737 205.813C150.805 205.764 150.882 205.764 150.911 205.813L151.346 206.469Z"
        fill="black"
      />
      <path
        opacity="0.1"
        d="M148.508 207.608C148.502 207.62 148.498 207.633 148.498 207.647C148.498 207.66 148.502 207.673 148.508 207.685L147.996 207.386L147.175 206.913C147.207 206.921 147.24 206.921 147.272 206.913L148.102 206.15C148.131 206.131 148.164 206.121 148.199 206.121C148.233 206.121 148.267 206.131 148.295 206.15C148.403 206.238 148.524 206.309 148.652 206.363C148.68 206.374 148.704 206.394 148.718 206.421C148.732 206.447 148.736 206.478 148.73 206.508L148.508 207.608Z"
        fill="black"
      />
      <path
        opacity="0.1"
        d="M147.678 204.827C147.664 204.859 147.644 204.889 147.62 204.914L146.828 205.879C146.819 205.91 146.819 205.944 146.828 205.975L145.177 205.01C145.173 204.972 145.173 204.933 145.177 204.894L145.959 203.929C145.959 203.929 145.959 203.871 146.007 203.852L147.678 204.827Z"
        fill="black"
      />
      <path
        opacity="0.05"
        d="M147.686 204.026C147.686 204.257 147.686 204.479 147.686 204.702C147.695 204.74 147.695 204.779 147.686 204.817L146.026 203.852C146.039 203.811 146.039 203.767 146.026 203.726C146.026 203.611 146.026 203.495 146.026 203.379L146.798 203.833H146.856H147.561C147.586 203.82 147.616 203.817 147.643 203.824C147.671 203.832 147.695 203.849 147.71 203.873C147.726 203.897 147.732 203.926 147.727 203.954C147.723 203.982 147.708 204.008 147.686 204.026Z"
        fill="black"
      />
      <path
        opacity="0.1"
        d="M148.199 202.076C148.164 202.087 148.127 202.087 148.092 202.076L147.32 202.269C147.291 202.275 147.263 202.286 147.238 202.303C147.213 202.319 147.192 202.341 147.175 202.366L145.524 201.401C145.524 201.401 145.524 201.401 145.573 201.401H145.631L146.403 201.217C146.439 201.205 146.472 201.185 146.499 201.159L148.199 202.076Z"
        fill="black"
      />
      <path
        opacity="0.1"
        d="M148.769 201.053C148.778 201.087 148.778 201.124 148.769 201.159L147.118 200.193C147.128 200.155 147.128 200.115 147.118 200.078L146.867 199.228C146.867 199.228 146.867 199.228 146.867 199.18L148.527 200.145C148.523 200.177 148.523 200.21 148.527 200.242L148.769 201.053Z"
        fill="black"
      />
      <path
        opacity="0.1"
        d="M151.404 197.954V198.012L150.969 199.199C150.95 199.244 150.92 199.284 150.882 199.315L149.222 198.349L149.289 198.263L149.743 197.065L151.404 197.954Z"
        fill="black"
      />
      <path
        d="M154.165 201.651H154.107H153.412C153.396 201.652 153.38 201.65 153.365 201.645C153.349 201.64 153.336 201.631 153.324 201.62C153.312 201.609 153.302 201.596 153.296 201.582C153.29 201.567 153.286 201.551 153.286 201.535C153.286 201.429 153.286 201.322 153.286 201.207C153.286 201.091 153.286 200.975 153.286 200.859C153.288 200.785 153.316 200.713 153.364 200.656L154.155 199.691C154.171 199.661 154.179 199.628 154.179 199.594C154.179 199.561 154.171 199.528 154.155 199.498L153.788 198.677C153.788 198.677 153.711 198.677 153.663 198.677L152.833 199.45C152.765 199.498 152.688 199.517 152.649 199.45C152.538 199.366 152.418 199.295 152.292 199.237C152.263 199.226 152.239 199.206 152.223 199.18C152.207 199.154 152.201 199.123 152.205 199.092L152.466 197.924C152.473 197.911 152.476 197.896 152.476 197.881C152.476 197.866 152.473 197.851 152.466 197.837C152.441 197.828 152.413 197.828 152.389 197.837L151.732 197.905H151.568C151.517 197.914 151.47 197.937 151.433 197.972V198.03L150.989 199.228C150.973 199.269 150.946 199.306 150.912 199.334C150.884 199.36 150.851 199.38 150.815 199.392C150.637 199.479 150.466 199.579 150.303 199.691C150.236 199.691 150.159 199.691 150.13 199.691L149.714 199.054C149.714 199.054 149.637 199.054 149.589 199.054L149.299 199.363L148.691 200.01C148.661 200.036 148.638 200.069 148.624 200.106C148.614 200.137 148.614 200.171 148.624 200.203L148.875 201.052C148.884 201.087 148.884 201.124 148.875 201.158C148.88 201.19 148.88 201.223 148.875 201.255C148.723 201.485 148.585 201.724 148.459 201.969C148.441 202.013 148.406 202.047 148.363 202.066C148.338 202.093 148.303 202.11 148.266 202.114L147.494 202.307C147.437 202.319 147.386 202.349 147.349 202.394V202.452L147.156 203.128L146.982 203.717C146.975 203.73 146.971 203.745 146.971 203.76C146.971 203.775 146.975 203.79 146.982 203.804H147.04H147.745C147.776 203.804 147.805 203.816 147.827 203.837C147.849 203.859 147.861 203.889 147.861 203.919C147.842 204.144 147.842 204.37 147.861 204.595C147.869 204.633 147.869 204.673 147.861 204.711C147.866 204.743 147.866 204.776 147.861 204.808L147.079 205.773C147.056 205.799 147.04 205.83 147.032 205.864C147.024 205.897 147.023 205.932 147.031 205.966L147.185 206.314V206.362L147.233 206.468L147.301 206.603V206.729V206.787C147.333 206.794 147.366 206.794 147.397 206.787L148.228 206.024C148.256 206.005 148.29 205.995 148.324 205.995C148.359 205.995 148.392 206.005 148.421 206.024C148.529 206.111 148.649 206.183 148.778 206.236C148.806 206.247 148.829 206.268 148.844 206.294C148.858 206.321 148.862 206.352 148.855 206.381L148.595 207.559C148.589 207.571 148.586 207.584 148.586 207.598C148.586 207.611 148.589 207.624 148.595 207.636C148.619 207.646 148.647 207.646 148.672 207.636L149.492 207.559C149.53 207.553 149.566 207.538 149.598 207.517C149.63 207.495 149.656 207.467 149.676 207.433L150.12 206.246C150.135 206.207 150.159 206.171 150.189 206.141C150.219 206.111 150.254 206.088 150.294 206.072L150.554 205.927L150.805 205.773C150.873 205.725 150.95 205.725 150.979 205.773L151.394 206.41C151.414 206.424 151.438 206.432 151.462 206.432C151.486 206.432 151.51 206.424 151.529 206.41L152.427 205.445C152.455 205.421 152.475 205.391 152.487 205.357C152.499 205.323 152.502 205.287 152.495 205.252L152.244 204.402C152.237 204.331 152.254 204.26 152.292 204.199C152.443 203.969 152.582 203.73 152.707 203.485C152.726 203.447 152.753 203.415 152.787 203.39C152.82 203.365 152.859 203.348 152.9 203.34L153.673 203.157C153.747 203.133 153.81 203.081 153.846 203.012L154.233 201.699C154.227 201.685 154.218 201.674 154.206 201.665C154.194 201.656 154.18 201.651 154.165 201.651ZM150.062 204.837H149.743C149.577 204.84 149.413 204.791 149.275 204.698C149.137 204.605 149.031 204.471 148.971 204.315C148.91 204.181 148.874 204.037 148.865 203.89C148.86 203.81 148.86 203.729 148.865 203.649C148.918 202.778 149.277 201.954 149.879 201.322C150.061 201.124 150.273 200.955 150.506 200.82L150.709 200.714C150.89 200.63 151.088 200.587 151.288 200.589H151.385C151.528 200.609 151.663 200.668 151.777 200.757C151.891 200.847 151.978 200.966 152.031 201.1C152.159 201.633 152.145 202.191 151.989 202.716C151.833 203.241 151.541 203.716 151.143 204.093C150.86 204.436 150.485 204.69 150.062 204.827V204.837Z"
        fill="#37474F"
      />
      <path
        d="M151.423 219.985C149.934 220.473 148.685 221.508 147.929 222.881C147.505 223.733 147.3 224.676 147.332 225.627C147.364 226.577 147.632 227.505 148.112 228.326C147.477 229.335 147.142 230.503 147.147 231.695C147.221 233.166 147.727 234.582 148.603 235.765C149.479 236.948 150.685 237.847 152.07 238.347L152.35 238.444L153.19 237.961C152.201 237.6 151.293 237.048 150.517 236.335C149.742 235.623 149.115 234.765 148.672 233.81C148.396 233.099 148.266 232.341 148.291 231.579C148.316 230.818 148.495 230.069 148.817 229.378C149.269 229.964 149.809 230.476 150.419 230.894C152.099 231.995 155.372 232.825 156.907 231.058C157.319 230.546 157.533 229.902 157.51 229.244C157.487 228.587 157.228 227.959 156.781 227.477C156.295 226.928 155.701 226.485 155.037 226.174C154.373 225.864 153.652 225.693 152.92 225.671C152.17 225.66 151.426 225.804 150.735 226.095C150.043 226.386 149.42 226.817 148.904 227.361C148.572 226.575 148.436 225.72 148.508 224.87C148.621 223.945 149.003 223.074 149.607 222.366C150.212 221.657 151.011 221.141 151.906 220.883C153.296 223.045 156.366 225.015 158.731 223.171C161.097 221.327 157.322 219.657 155.739 219.512C154.645 219.399 153.54 219.461 152.466 219.695C152.217 219.084 152.144 218.415 152.254 217.765C152.469 216.76 153.042 215.868 153.866 215.255C154.749 214.522 155.843 214.09 156.988 214.021C158.134 213.951 159.272 214.248 160.237 214.868C160.309 214.927 160.401 214.955 160.494 214.946C160.586 214.937 160.671 214.892 160.73 214.82C160.789 214.748 160.817 214.656 160.808 214.564C160.799 214.472 160.753 214.387 160.681 214.328C156.887 210.852 149.435 215.11 151.423 219.985ZM150.825 227.148C151.366 226.849 151.969 226.679 152.587 226.65C153.205 226.622 153.821 226.736 154.387 226.984C155.352 227.419 156.511 228.336 156.318 229.552C156.048 231.666 153.2 231.078 151.867 230.518C150.835 230.065 149.963 229.312 149.367 228.355C149.764 227.856 150.26 227.445 150.825 227.148ZM153.721 220.477C154.716 220.335 155.73 220.407 156.694 220.69C157.148 220.835 158.442 221.269 158.374 221.925C158.374 222.331 157.747 222.64 157.409 222.775C156.913 222.969 156.367 222.996 155.854 222.852C154.777 222.511 153.836 221.835 153.171 220.921L152.949 220.603C153.194 220.547 153.442 220.505 153.692 220.477H153.721Z"
        fill="#489CFF"
      />
      <path
        opacity="0.7"
        d="M151.423 219.985C149.934 220.473 148.685 221.508 147.929 222.881C147.505 223.733 147.3 224.676 147.332 225.627C147.364 226.577 147.632 227.505 148.112 228.326C147.477 229.335 147.142 230.503 147.147 231.695C147.221 233.166 147.727 234.582 148.603 235.765C149.479 236.948 150.685 237.847 152.07 238.347L152.35 238.444L153.19 237.961C152.201 237.6 151.293 237.048 150.517 236.335C149.742 235.623 149.115 234.765 148.672 233.81C148.396 233.099 148.266 232.341 148.291 231.579C148.316 230.818 148.495 230.069 148.817 229.378C149.269 229.964 149.809 230.476 150.419 230.894C152.099 231.995 155.372 232.825 156.907 231.058C157.319 230.546 157.533 229.902 157.51 229.244C157.487 228.587 157.228 227.959 156.781 227.477C156.295 226.928 155.701 226.485 155.037 226.174C154.373 225.864 153.652 225.693 152.92 225.671C152.17 225.66 151.426 225.804 150.735 226.095C150.043 226.386 149.42 226.817 148.904 227.361C148.572 226.575 148.436 225.72 148.508 224.87C148.621 223.945 149.003 223.074 149.607 222.366C150.212 221.657 151.011 221.141 151.906 220.883C153.296 223.045 156.366 225.015 158.731 223.171C161.097 221.327 157.322 219.657 155.739 219.512C154.645 219.399 153.54 219.461 152.466 219.695C152.217 219.084 152.144 218.415 152.254 217.765C152.469 216.76 153.042 215.868 153.866 215.255C154.749 214.522 155.843 214.09 156.988 214.021C158.134 213.951 159.272 214.248 160.237 214.868C160.309 214.927 160.401 214.955 160.494 214.946C160.586 214.937 160.671 214.892 160.73 214.82C160.789 214.748 160.817 214.656 160.808 214.564C160.799 214.472 160.753 214.387 160.681 214.328C156.887 210.852 149.435 215.11 151.423 219.985ZM150.825 227.148C151.366 226.849 151.969 226.679 152.587 226.65C153.205 226.622 153.821 226.736 154.387 226.984C155.352 227.419 156.511 228.336 156.318 229.552C156.048 231.666 153.2 231.078 151.867 230.518C150.835 230.065 149.963 229.312 149.367 228.355C149.764 227.856 150.26 227.445 150.825 227.148ZM153.721 220.477C154.716 220.335 155.73 220.407 156.694 220.69C157.148 220.835 158.442 221.269 158.374 221.925C158.374 222.331 157.747 222.64 157.409 222.775C156.913 222.969 156.367 222.996 155.854 222.852C154.777 222.511 153.836 221.835 153.171 220.921L152.949 220.603C153.194 220.547 153.442 220.505 153.692 220.477H153.721Z"
        fill="#183867"
      />
      <path
        d="M148.459 230.315C148.54 229.983 148.653 229.659 148.797 229.349C148.511 229.04 148.279 228.684 148.112 228.297C147.909 228.623 147.735 228.965 147.59 229.32C147.853 229.674 148.144 230.007 148.459 230.315Z"
        fill="#489CFF"
      />
      <path
        d="M150.709 221.441C151.083 221.215 151.477 221.024 151.887 220.872C151.706 220.59 151.55 220.293 151.423 219.983C150.966 220.141 150.526 220.344 150.11 220.592C150.288 220.89 150.488 221.174 150.709 221.441Z"
        fill="#489CFF"
      />
      <path
        d="M152.331 238.433L153.17 237.95C152.814 237.802 152.466 237.635 152.128 237.448L151.23 237.96C151.495 238.103 151.769 238.229 152.051 238.336L152.331 238.433Z"
        fill="#489CFF"
      />
      <path
        d="M174.709 222.853C173.463 221.904 171.905 221.459 170.345 221.607C169.405 221.747 168.513 222.116 167.748 222.681C166.984 223.247 166.37 223.992 165.962 224.851C165.205 224.877 164.459 225.04 163.761 225.334C163.368 225.507 162.99 225.714 162.632 225.952C161.577 226.747 160.757 227.813 160.258 229.037C159.76 230.261 159.603 231.597 159.803 232.903C159.803 233.202 159.89 233.492 159.948 233.781L160.006 233.994L160.749 233.569V233.337C160.52 231.622 160.888 229.88 161.792 228.404C162.299 227.649 162.986 227.033 163.79 226.608C164.331 226.318 164.914 226.116 165.518 226.01C165.289 226.72 165.182 227.464 165.2 228.211C165.2 230.219 166.416 233.385 168.743 233.646C169.403 233.699 170.06 233.511 170.592 233.117C171.125 232.723 171.496 232.15 171.639 231.503C171.819 230.79 171.849 230.048 171.726 229.323C171.603 228.598 171.33 227.907 170.924 227.294C170.51 226.67 169.971 226.139 169.341 225.734C168.711 225.328 168.005 225.058 167.266 224.938C167.72 224.22 168.343 223.623 169.081 223.2C169.907 222.769 170.841 222.59 171.768 222.687C172.695 222.783 173.574 223.149 174.294 223.741C173.328 226.116 173.425 229.746 176.282 230.644C179.14 231.541 178.368 227.496 177.605 226.116C177.073 225.153 176.395 224.277 175.597 223.519C175.96 222.958 176.476 222.513 177.084 222.235C178.03 221.842 179.087 221.808 180.057 222.138C181.161 222.452 182.138 223.108 182.845 224.012C183.553 224.916 183.956 226.021 183.996 227.168C183.992 227.214 183.997 227.26 184.011 227.303C184.025 227.347 184.048 227.387 184.077 227.422C184.107 227.457 184.143 227.486 184.184 227.507C184.225 227.528 184.269 227.541 184.315 227.545C184.36 227.549 184.406 227.543 184.45 227.529C184.493 227.515 184.534 227.493 184.569 227.463C184.604 227.434 184.633 227.398 184.654 227.357C184.675 227.316 184.687 227.272 184.691 227.226C185.338 222.206 177.595 218.451 174.709 222.853ZM168.472 226.425C169.025 226.7 169.507 227.1 169.881 227.592C170.254 228.085 170.509 228.656 170.625 229.263C170.828 230.325 170.722 231.783 169.66 232.343C167.729 233.308 166.628 230.634 166.329 229.215C166.123 228.109 166.251 226.966 166.696 225.932C167.315 225.977 167.919 226.144 168.472 226.425ZM175.587 224.996C176.267 225.736 176.784 226.611 177.103 227.564C177.238 228.018 177.624 229.34 177.026 229.649C176.678 229.842 176.06 229.543 175.79 229.35C175.338 229.059 174.998 228.623 174.825 228.114C174.493 227.031 174.52 225.87 174.902 224.803C174.935 224.675 174.977 224.549 175.027 224.426C175.23 224.62 175.423 224.803 175.587 224.996Z"
        fill="#489CFF"
      />
      <path
        opacity="0.7"
        d="M174.709 222.853C173.463 221.904 171.905 221.459 170.345 221.607C169.405 221.747 168.513 222.116 167.748 222.681C166.984 223.247 166.37 223.992 165.962 224.851C165.205 224.877 164.459 225.04 163.761 225.334C163.368 225.507 162.99 225.714 162.632 225.952C161.577 226.747 160.757 227.813 160.258 229.037C159.76 230.261 159.603 231.597 159.803 232.903C159.803 233.202 159.89 233.492 159.948 233.781L160.006 233.994L160.749 233.569V233.337C160.52 231.622 160.888 229.88 161.792 228.404C162.299 227.649 162.986 227.033 163.79 226.608C164.331 226.318 164.914 226.116 165.518 226.01C165.289 226.72 165.182 227.464 165.2 228.211C165.2 230.219 166.416 233.385 168.743 233.646C169.403 233.699 170.06 233.511 170.592 233.117C171.125 232.723 171.496 232.15 171.639 231.503C171.819 230.79 171.849 230.048 171.726 229.323C171.603 228.598 171.33 227.907 170.924 227.294C170.51 226.67 169.971 226.139 169.341 225.734C168.711 225.328 168.005 225.058 167.266 224.938C167.72 224.22 168.343 223.623 169.081 223.2C169.907 222.769 170.841 222.59 171.768 222.687C172.695 222.783 173.574 223.149 174.294 223.741C173.328 226.116 173.425 229.746 176.282 230.644C179.14 231.541 178.368 227.496 177.605 226.116C177.073 225.153 176.395 224.277 175.597 223.519C175.96 222.958 176.476 222.513 177.084 222.235C178.03 221.842 179.087 221.808 180.057 222.138C181.161 222.452 182.138 223.108 182.845 224.012C183.553 224.916 183.956 226.021 183.996 227.168C183.992 227.214 183.997 227.26 184.011 227.303C184.025 227.347 184.048 227.387 184.077 227.422C184.107 227.457 184.143 227.486 184.184 227.507C184.225 227.528 184.269 227.541 184.315 227.545C184.36 227.549 184.406 227.543 184.45 227.529C184.493 227.515 184.534 227.493 184.569 227.463C184.604 227.434 184.633 227.398 184.654 227.357C184.675 227.316 184.687 227.272 184.691 227.226C185.338 222.206 177.595 218.451 174.709 222.853ZM168.472 226.425C169.025 226.7 169.507 227.1 169.881 227.592C170.254 228.085 170.509 228.656 170.625 229.263C170.828 230.325 170.722 231.783 169.66 232.343C167.729 233.308 166.628 230.634 166.329 229.215C166.123 228.109 166.251 226.966 166.696 225.932C167.315 225.977 167.919 226.144 168.472 226.425ZM175.587 224.996C176.267 225.736 176.784 226.611 177.103 227.564C177.238 228.018 177.624 229.34 177.026 229.649C176.678 229.842 176.06 229.543 175.79 229.35C175.338 229.059 174.998 228.623 174.825 228.114C174.493 227.031 174.52 225.87 174.902 224.803C174.935 224.675 174.977 224.549 175.027 224.426C175.23 224.62 175.423 224.803 175.587 224.996Z"
        fill="#183867"
      />
      <path
        d="M164.523 226.269C164.837 226.153 165.16 226.065 165.489 226.008C165.586 225.6 165.745 225.209 165.962 224.85C165.578 224.871 165.197 224.926 164.823 225.014C164.689 225.423 164.589 225.843 164.523 226.269Z"
        fill="#489CFF"
      />
      <path
        d="M173.096 223.092C173.499 223.264 173.881 223.481 174.235 223.739C174.37 223.428 174.528 223.128 174.708 222.841C174.311 222.566 173.887 222.33 173.444 222.137C173.299 222.444 173.183 222.764 173.096 223.092Z"
        fill="#489CFF"
      />
      <path
        d="M160.005 233.991L159.948 233.779C159.89 233.489 159.841 233.2 159.803 232.9L160.72 233.335V233.567L160.005 233.991Z"
        fill="#489CFF"
      />
      <path
        d="M161.656 126.629L109.157 156.942C108.544 157.344 108.034 157.885 107.668 158.521C107.303 159.156 107.092 159.869 107.053 160.601V178.616C107.053 179.967 108.018 180.546 109.167 179.842L161.656 149.538C162.269 149.135 162.78 148.595 163.147 147.959C163.514 147.324 163.728 146.611 163.77 145.879V127.855C163.77 126.513 162.824 125.963 161.656 126.629Z"
        fill="#455A64"
      />
      <path
        d="M159.784 134.575C159.732 135.286 159.514 135.974 159.146 136.585C158.779 137.195 158.273 137.711 157.669 138.089L113.106 163.817C111.938 164.483 110.992 164.01 110.992 162.746C111.044 162.035 111.262 161.347 111.63 160.736C111.997 160.126 112.503 159.61 113.106 159.232L157.669 133.504C158.818 132.837 159.784 133.311 159.784 134.575Z"
        fill="#489CFF"
      />
      <path
        d="M159.783 143.738C159.732 144.448 159.513 145.137 159.146 145.747C158.779 146.358 158.273 146.873 157.669 147.252L113.125 172.97C111.957 173.646 111.011 173.173 111.011 171.908C111.063 171.198 111.281 170.509 111.649 169.899C112.016 169.288 112.522 168.773 113.125 168.394L157.688 142.666C158.818 141.99 159.783 142.473 159.783 143.738Z"
        fill="#263238"
      />
      <path
        d="M157.689 142.666L138.776 153.585V158.161L157.689 147.252C158.292 146.873 158.798 146.358 159.165 145.747C159.533 145.137 159.751 144.448 159.803 143.738C159.784 142.473 158.818 141.99 157.689 142.666Z"
        fill="#489CFF"
      />
      <path
        d="M88.8653 84.2197C86.6738 82.955 84.8975 84.1039 84.8975 86.807V183.714C84.9482 185.139 85.3338 186.532 86.0232 187.78C86.7125 189.027 87.6862 190.095 88.8653 190.897C91.0567 192.171 92.8331 191.013 92.8331 188.319V91.4023C92.7801 89.9781 92.3935 88.5862 91.7043 87.3386C91.0151 86.0911 90.0427 85.0228 88.8653 84.2197Z"
        fill="#489CFF"
      />
      <path
        opacity="0.5"
        d="M88.8653 84.2197C86.6738 82.955 84.8975 84.1039 84.8975 86.807V183.714C84.9482 185.139 85.3338 186.532 86.0232 187.78C86.7125 189.027 87.6862 190.095 88.8653 190.897C91.0567 192.171 92.8331 191.013 92.8331 188.319V91.4023C92.7801 89.9781 92.3935 88.5862 91.7043 87.3386C91.0151 86.0911 90.0427 85.0228 88.8653 84.2197Z"
        fill="black"
      />
      <path
        d="M88.8648 188.581C88.2775 188.18 87.7924 187.648 87.4486 187.026C87.1048 186.404 86.912 185.709 86.8857 184.999V87.7926C86.8857 86.441 87.7739 85.8618 88.8648 86.4893C89.4451 86.8952 89.922 87.4316 90.2574 88.0553C90.5927 88.679 90.7771 89.3727 90.7956 90.0806V187.287C90.8439 188.639 89.9557 189.218 88.8648 188.581Z"
        fill="#37474F"
      />
      <path
        d="M88.8648 188.58C88.2775 188.18 87.7924 187.648 87.4486 187.025C87.1048 186.403 86.912 185.709 86.8857 184.999V137.501L90.8439 139.789V187.287C90.8439 188.638 89.9557 189.218 88.8648 188.58Z"
        fill="#489CFF"
      />
      <path
        d="M76.9712 77.394C74.7798 76.1197 73.0034 77.2782 73.0034 79.9717V176.831C73.0564 178.255 73.443 179.647 74.1322 180.894C74.8213 182.142 75.7938 183.21 76.9712 184.013C79.153 185.278 80.9294 184.119 80.9294 181.426V84.5284C80.8716 83.1129 80.4835 81.7306 79.7963 80.4919C79.109 79.2531 78.1416 78.1922 76.9712 77.394Z"
        fill="#489CFF"
      />
      <path
        opacity="0.5"
        d="M76.9712 77.394C74.7798 76.1197 73.0034 77.2782 73.0034 79.9717V176.831C73.0564 178.255 73.443 179.647 74.1322 180.894C74.8213 182.142 75.7938 183.21 76.9712 184.013C79.153 185.278 80.9294 184.119 80.9294 181.426V84.5284C80.8716 83.1129 80.4835 81.7306 79.7963 80.4919C79.109 79.2531 78.1416 78.1922 76.9712 77.394Z"
        fill="black"
      />
      <path
        d="M76.9712 181.715C76.3807 181.314 75.8929 180.781 75.5474 180.157C75.2019 179.533 75.0083 178.836 74.9824 178.123V80.9168C74.9824 79.5653 75.8706 78.986 76.9712 79.6232C77.5496 80.0287 78.0253 80.5636 78.3605 81.1854C78.6957 81.8072 78.8811 82.4987 78.902 83.2048V180.411C78.9502 181.763 78.0621 182.342 76.9712 181.715Z"
        fill="#489CFF"
      />
      <path
        d="M65.0674 70.4824C62.876 69.2177 61.0996 70.3666 61.0996 73.0697V169.977C61.1503 171.402 61.536 172.794 62.2253 174.042C62.9147 175.29 63.8883 176.358 65.0674 177.16C67.2589 178.434 69.0352 177.275 69.0352 174.582V77.665C68.9823 76.2408 68.5956 74.8489 67.9065 73.6013C67.2173 72.3538 66.2449 71.2855 65.0674 70.4824Z"
        fill="#489CFF"
      />
      <path
        opacity="0.5"
        d="M65.0674 70.4824C62.876 69.2177 61.0996 70.3666 61.0996 73.0697V169.977C61.1503 171.402 61.536 172.794 62.2253 174.042C62.9147 175.29 63.8883 176.358 65.0674 177.16C67.2589 178.434 69.0352 177.275 69.0352 174.582V77.665C68.9823 76.2408 68.5956 74.8489 67.9065 73.6013C67.2173 72.3538 66.2449 71.2855 65.0674 70.4824Z"
        fill="black"
      />
      <path
        d="M65.068 174.842C64.4896 174.437 64.0138 173.902 63.6786 173.28C63.3434 172.658 63.158 171.967 63.1372 171.261V74.0543C63.1372 72.7027 64.0254 72.1235 65.068 72.751C65.6674 73.1458 66.1646 73.6772 66.5187 74.3015C66.8728 74.9258 67.0737 75.6253 67.105 76.3423V173.549C67.0471 174.9 66.1396 175.48 65.068 174.842Z"
        fill="#37474F"
      />
      <path
        d="M65.068 174.842C64.4896 174.436 64.0138 173.901 63.6786 173.28C63.3434 172.658 63.158 171.966 63.1372 171.26V100.67L67.105 102.968V173.548C67.0471 174.9 66.1396 175.479 65.068 174.842Z"
        fill="#489CFF"
      />
      <path
        d="M53.1739 63.6177C50.9824 62.3433 49.2061 63.5018 49.2061 66.1953V163.112C49.259 164.536 49.6457 165.928 50.3348 167.176C51.024 168.423 51.9964 169.492 53.1739 170.295C55.3557 171.559 57.132 170.401 57.132 167.707V70.7906C57.0817 69.3682 56.6971 67.9777 56.0095 66.7315C55.3218 65.4853 54.3504 64.4186 53.1739 63.6177Z"
        fill="#489CFF"
      />
      <path
        opacity="0.5"
        d="M53.1739 63.6177C50.9824 62.3433 49.2061 63.5018 49.2061 66.1953V163.112C49.259 164.536 49.6457 165.928 50.3348 167.176C51.024 168.423 51.9964 169.492 53.1739 170.295C55.3557 171.559 57.132 170.401 57.132 167.707V70.7906C57.0817 69.3682 56.6971 67.9777 56.0095 66.7315C55.3218 65.4853 54.3504 64.4186 53.1739 63.6177Z"
        fill="black"
      />
      <path
        d="M53.1738 167.977C52.5834 167.577 52.0956 167.043 51.75 166.419C51.4045 165.795 51.2109 165.099 51.1851 164.386V67.1795C51.1851 65.828 52.0732 65.2487 53.1738 65.8859C53.7611 66.2862 54.2463 66.8188 54.59 67.4409C54.9338 68.0631 55.1266 68.7572 55.1529 69.4675V166.674C55.1529 168.026 54.2647 168.605 53.1738 167.977Z"
        fill="#37474F"
      />
      <path
        d="M53.1738 167.978C52.5834 167.578 52.0956 167.044 51.75 166.42C51.4045 165.796 51.2109 165.1 51.1851 164.387V146.662L55.1529 148.95V166.675C55.1529 168.026 54.2647 168.606 53.1738 167.978Z"
        fill="#489CFF"
      />
      <path
        d="M41.2701 56.7434C39.0786 55.4691 37.3022 56.6276 37.3022 59.3307V156.238C37.3529 157.663 37.7386 159.055 38.428 160.303C39.1173 161.551 40.091 162.619 41.2701 163.421C43.4615 164.695 45.2379 163.536 45.2379 160.843V63.926C45.1849 62.5018 44.7983 61.1099 44.1091 59.8623C43.4199 58.6148 42.4475 57.5465 41.2701 56.7434Z"
        fill="#489CFF"
      />
      <path
        opacity="0.5"
        d="M41.2701 56.7434C39.0786 55.4691 37.3022 56.6276 37.3022 59.3307V156.238C37.3529 157.663 37.7386 159.055 38.428 160.303C39.1173 161.551 40.091 162.619 41.2701 163.421C43.4615 164.695 45.2379 163.536 45.2379 160.843V63.926C45.1849 62.5018 44.7983 61.1099 44.1091 59.8623C43.4199 58.6148 42.4475 57.5465 41.2701 56.7434Z"
        fill="black"
      />
      <path
        d="M41.2701 161.104C40.6827 160.703 40.1976 160.171 39.8539 159.549C39.5101 158.926 39.3173 158.232 39.291 157.522V60.3059C39.291 58.9639 40.1792 58.3751 41.2701 59.0122C41.8605 59.4124 42.3483 59.9461 42.6938 60.5701C43.0394 61.1941 43.2329 61.8907 43.2588 62.6035V159.81C43.2588 161.152 42.3707 161.741 41.2701 161.104Z"
        fill="#37474F"
      />
      <path
        d="M41.2701 161.103C40.6827 160.702 40.1976 160.17 39.8539 159.548C39.5101 158.925 39.3173 158.231 39.291 157.521V105.438L43.2588 107.735V159.809C43.2588 161.151 42.3707 161.74 41.2701 161.103Z"
        fill="#489CFF"
      />
      <path
        d="M161.656 53.5683L109.157 83.8723C108.543 84.2731 108.032 84.8135 107.666 85.4494C107.301 86.0852 107.09 86.7986 107.053 87.5312V147.869C107.053 149.211 108.018 149.8 109.167 149.085L161.656 118.781C162.269 118.379 162.78 117.838 163.147 117.203C163.514 116.567 163.728 115.855 163.77 115.122V54.7847C163.77 53.4331 162.824 52.8925 161.656 53.5683Z"
        fill="#37474F"
      />
      <path
        d="M107.652 85.5332C107.296 86.1401 107.099 86.8279 107.083 87.5316V147.869C107.083 149.211 108.048 149.8 109.197 149.086L161.686 118.782C162.333 118.371 162.862 117.798 163.221 117.121L107.652 85.5332Z"
        fill="#263238"
      />
      <path
        d="M109.157 83.8714C108.543 84.2722 108.032 84.8126 107.666 85.4485C107.301 86.0843 107.09 86.7977 107.053 87.5303V147.868L159.542 117.564C160.155 117.159 160.664 116.617 161.03 115.98C161.395 115.343 161.606 114.629 161.646 113.895V53.5674L109.157 83.8714Z"
        fill="#455A64"
      />
      <path
        d="M126.448 99.7333C126.168 99.7369 125.897 99.6294 125.695 99.434L118.715 92.6762C118.613 92.5773 118.53 92.4591 118.473 92.3285C118.416 92.1978 118.386 92.0572 118.383 91.9147C118.38 91.7721 118.406 91.6305 118.458 91.4977C118.51 91.365 118.588 91.2439 118.686 91.1412C118.785 91.0385 118.903 90.9563 119.034 90.8992C119.165 90.8422 119.305 90.8115 119.448 90.8088C119.59 90.8061 119.732 90.8315 119.865 90.8836C119.998 90.9356 120.119 91.0133 120.221 91.1122L127.201 97.8218C127.356 97.9726 127.463 98.1663 127.508 98.378C127.553 98.5897 127.534 98.8099 127.453 99.0107C127.372 99.2115 127.233 99.3837 127.055 99.5055C126.876 99.6273 126.665 99.6931 126.448 99.6946V99.7333Z"
        fill="#489CFF"
      />
      <path
        d="M119.468 103.759C119.308 103.761 119.149 103.724 119.005 103.653C118.744 103.529 118.543 103.308 118.445 103.036C118.348 102.765 118.361 102.466 118.484 102.205L125.463 87.4338C125.524 87.3047 125.61 87.189 125.716 87.0933C125.822 86.9976 125.946 86.9238 126.081 86.8762C126.216 86.8286 126.358 86.8082 126.501 86.8161C126.643 86.824 126.783 86.86 126.912 86.9222C127.172 87.0456 127.373 87.2673 127.471 87.5386C127.569 87.81 127.555 88.109 127.433 88.3703L120.453 103.141C120.364 103.326 120.224 103.483 120.05 103.592C119.876 103.701 119.674 103.759 119.468 103.759Z"
        fill="#489CFF"
      />
      <path
        d="M149.233 86.5736C148.952 86.5752 148.682 86.4679 148.479 86.2743L141.5 79.5648C141.396 79.4652 141.314 79.3463 141.256 79.2149C141.199 79.0834 141.168 78.9419 141.165 78.7984C141.163 78.655 141.188 78.5124 141.241 78.3789C141.293 78.2454 141.371 78.1234 141.471 78.0201C141.57 77.9168 141.689 77.8341 141.821 77.7767C141.952 77.7193 142.094 77.6884 142.237 77.6857C142.38 77.683 142.523 77.7086 142.657 77.761C142.79 77.8134 142.912 77.8916 143.015 77.9912L149.986 84.7007C150.188 84.9066 150.302 85.184 150.302 85.473C150.302 85.762 150.188 86.0395 149.986 86.2454C149.888 86.3473 149.771 86.4289 149.642 86.4852C149.513 86.5416 149.374 86.5716 149.233 86.5736Z"
        fill="#489CFF"
      />
      <path
        d="M142.252 90.6099C142.092 90.6069 141.934 90.5708 141.789 90.5037C141.53 90.3784 141.33 90.1566 141.233 89.8859C141.136 89.6153 141.148 89.3172 141.268 89.0556L148.247 74.2849C148.397 74.0702 148.619 73.9165 148.873 73.852C149.127 73.7875 149.396 73.8164 149.63 73.9335C149.864 74.0507 150.048 74.2481 150.149 74.4899C150.25 74.7317 150.26 75.0016 150.178 75.2503L143.237 89.9438C143.155 90.1382 143.018 90.3045 142.843 90.4228C142.669 90.541 142.463 90.606 142.252 90.6099Z"
        fill="#489CFF"
      />
      <path
        d="M119.16 120.498C118.965 120.5 118.772 120.451 118.603 120.354C118.433 120.257 118.292 120.117 118.194 119.948C118.051 119.697 118.014 119.4 118.09 119.121C118.166 118.842 118.349 118.605 118.6 118.461L148.981 100.92C149.231 100.776 149.528 100.737 149.806 100.811C150.085 100.885 150.323 101.067 150.468 101.316C150.611 101.568 150.649 101.867 150.573 102.147C150.497 102.428 150.314 102.667 150.062 102.812L119.681 120.344C119.526 120.444 119.345 120.498 119.16 120.498Z"
        fill="#489CFF"
      />
      <path
        d="M140.331 120.991C139.795 120.995 139.269 120.855 138.805 120.586C138.216 120.185 137.744 119.635 137.437 118.993C137.13 118.35 137 117.637 137.058 116.927V108.441C137.061 108.153 137.176 107.877 137.38 107.673C137.584 107.469 137.86 107.353 138.149 107.35C138.437 107.353 138.713 107.469 138.918 107.673C139.122 107.877 139.237 108.153 139.24 108.441V116.975C139.195 117.301 139.232 117.633 139.347 117.941C139.461 118.249 139.65 118.524 139.896 118.742C140.147 118.832 140.416 118.858 140.679 118.818C140.942 118.777 141.191 118.672 141.402 118.51C142.367 117.842 143.162 116.958 143.727 115.929C144.291 114.901 144.609 113.755 144.656 112.583V104.106C144.641 103.955 144.659 103.803 144.707 103.659C144.755 103.515 144.833 103.382 144.935 103.27C145.037 103.157 145.162 103.068 145.301 103.006C145.44 102.945 145.59 102.913 145.742 102.913C145.894 102.913 146.044 102.945 146.183 103.006C146.322 103.068 146.446 103.157 146.549 103.27C146.651 103.382 146.729 103.515 146.777 103.659C146.825 103.803 146.842 103.955 146.828 104.106V112.583C146.781 114.136 146.363 115.655 145.609 117.014C144.854 118.373 143.786 119.531 142.493 120.393C141.838 120.78 141.092 120.987 140.331 120.991Z"
        fill="#489CFF"
      />
      <path
        d="M141.943 115.528C141.654 115.528 141.377 115.413 141.172 115.208C140.967 115.004 140.853 114.726 140.853 114.437V106.25C140.853 105.961 140.967 105.683 141.172 105.479C141.377 105.274 141.654 105.159 141.943 105.159C142.233 105.159 142.51 105.274 142.715 105.479C142.919 105.683 143.034 105.961 143.034 106.25V114.437C143.036 114.58 143.008 114.723 142.954 114.856C142.9 114.989 142.819 115.109 142.718 115.211C142.616 115.313 142.495 115.393 142.362 115.447C142.23 115.502 142.087 115.529 141.943 115.528Z"
        fill="#489CFF"
      />
      <path
        d="M46.0201 240.933C46.2986 241.105 46.5318 241.341 46.6998 241.621C46.8678 241.902 46.9658 242.219 46.9855 242.545C46.9855 243.144 46.5704 243.385 46.0201 243.086C45.7442 242.912 45.5132 242.675 45.3455 242.395C45.1778 242.115 45.0782 241.799 45.0547 241.474C45.0836 240.875 45.5084 240.634 46.0201 240.933Z"
        fill="#37474F"
      />
      <path
        d="M53.8209 245.442C54.0994 245.613 54.3326 245.849 54.5006 246.13C54.6686 246.411 54.7666 246.728 54.7863 247.054C54.7863 247.643 54.3712 247.884 53.8209 247.595C53.5421 247.421 53.3091 247.183 53.1411 246.901C52.9731 246.619 52.8752 246.301 52.8555 245.973C52.8844 245.374 53.2996 245.143 53.8209 245.442Z"
        fill="#37474F"
      />
      <path
        d="M61.4761 249.864C61.7539 250.034 61.9867 250.268 62.1548 250.547C62.3228 250.826 62.4212 251.142 62.4415 251.467C62.4415 252.065 62.0264 252.307 61.4761 252.007C61.1987 251.835 60.9664 251.598 60.7985 251.318C60.6306 251.037 60.532 250.721 60.5107 250.395C60.5494 249.806 60.9645 249.565 61.4761 249.864Z"
        fill="#37474F"
      />
      <path
        d="M69.2091 254.324C69.4852 254.496 69.7167 254.731 69.8846 255.01C70.0524 255.288 70.1517 255.603 70.1745 255.927C70.1745 256.516 69.7593 256.757 69.2091 256.468C68.9329 256.296 68.7014 256.061 68.5335 255.782C68.3657 255.504 68.2664 255.189 68.2437 254.865C68.2823 254.266 68.6974 254.064 69.2091 254.324Z"
        fill="#37474F"
      />
      <path
        d="M36.9261 195.096L44.9004 199.643C45.5478 200.069 46.0873 200.64 46.476 201.31C46.8648 201.98 47.0922 202.732 47.1401 203.505V208.052C47.1401 209.481 46.1747 210.07 44.9004 209.346L36.9261 204.828C36.2764 204.404 35.7346 203.834 35.3441 203.163C34.9536 202.493 34.7251 201.74 34.6768 200.966V196.419C34.6768 194.961 35.6904 194.382 36.9261 195.096Z"
        fill="#37474F"
      />
      <path
        d="M39.1079 200.224L40.1409 199.625C40.3608 199.524 40.6037 199.483 40.8447 199.507C41.0857 199.53 41.3159 199.618 41.5118 199.76C42.0765 200.135 42.5454 200.636 42.8808 201.225C43.2162 201.813 43.4086 202.472 43.4426 203.149C43.4689 203.39 43.43 203.634 43.3298 203.854C43.2296 204.075 43.0718 204.265 42.873 204.404L41.9076 204.954L41.7338 204.665C41.5363 204.627 41.3469 204.555 41.1739 204.452C40.6094 204.08 40.1404 203.58 39.8049 202.993C39.4695 202.406 39.277 201.749 39.2431 201.073C39.2409 200.904 39.2604 200.735 39.301 200.571L39.1079 200.224Z"
        fill="#489CFF"
      />
      <path
        opacity="0.2"
        d="M39.1079 200.224L40.1409 199.625C40.3608 199.524 40.6037 199.483 40.8447 199.507C41.0857 199.53 41.3159 199.618 41.5118 199.76C42.0765 200.135 42.5454 200.636 42.8808 201.225C43.2162 201.813 43.4086 202.472 43.4426 203.149C43.4689 203.39 43.43 203.634 43.3298 203.854C43.2296 204.075 43.0718 204.265 42.873 204.404L41.9076 204.954L41.7338 204.665C41.5363 204.627 41.3469 204.555 41.1739 204.452C40.6094 204.08 40.1404 203.58 39.8049 202.993C39.4695 202.406 39.277 201.749 39.2431 201.073C39.2409 200.904 39.2604 200.735 39.301 200.571L39.1079 200.224Z"
        fill="black"
      />
      <path
        d="M40.4795 200.33C41.0436 200.704 41.5122 201.206 41.8475 201.794C42.1828 202.383 42.3755 203.042 42.4103 203.718C42.4103 204.954 41.5317 205.466 40.4795 204.838C39.9155 204.465 39.4469 203.966 39.1115 203.379C38.7762 202.792 38.5834 202.134 38.5487 201.459C38.5293 200.214 39.4079 199.712 40.4795 200.33Z"
        fill="#489CFF"
      />
      <path
        d="M36.9259 173.212L44.9002 177.797C45.5492 178.221 46.0898 178.791 46.4788 179.462C46.8677 180.132 47.0942 180.885 47.1399 181.659V186.196C47.1399 187.635 46.1745 188.214 44.9002 187.499L36.9742 182.943C36.3239 182.519 35.7818 181.949 35.3912 181.279C35.0006 180.608 34.7724 179.856 34.7248 179.081V174.534C34.6765 173.076 35.6902 172.497 36.9259 173.212Z"
        fill="#455A64"
      />
      <path
        d="M39.1079 178.337L40.1409 177.738C40.3618 177.639 40.6051 177.6 40.8459 177.626C41.0866 177.651 41.3164 177.74 41.5118 177.883C42.0768 178.254 42.5462 178.754 42.8817 179.341C43.2172 179.928 43.4094 180.586 43.4426 181.262C43.4677 181.504 43.4282 181.749 43.3281 181.971C43.2281 182.193 43.0711 182.385 42.873 182.527L41.9076 183.077L41.7338 182.787C41.5372 182.741 41.3485 182.666 41.1739 182.565C40.6088 182.194 40.1394 181.694 39.8039 181.107C39.4684 180.52 39.2762 179.862 39.2431 179.186C39.2416 179.017 39.261 178.849 39.301 178.684L39.1079 178.337Z"
        fill="#489CFF"
      />
      <path
        opacity="0.2"
        d="M39.1079 178.337L40.1409 177.738C40.3618 177.639 40.6051 177.6 40.8459 177.626C41.0866 177.651 41.3164 177.74 41.5118 177.883C42.0768 178.254 42.5462 178.754 42.8817 179.341C43.2172 179.928 43.4094 180.586 43.4426 181.262C43.4677 181.504 43.4282 181.749 43.3281 181.971C43.2281 182.193 43.0711 182.385 42.873 182.527L41.9076 183.077L41.7338 182.787C41.5372 182.741 41.3485 182.666 41.1739 182.565C40.6088 182.194 40.1394 181.694 39.8039 181.107C39.4684 180.52 39.2762 179.862 39.2431 179.186C39.2416 179.017 39.261 178.849 39.301 178.684L39.1079 178.337Z"
        fill="black"
      />
      <path
        d="M40.4795 178.454C41.044 178.826 41.513 179.326 41.8484 179.913C42.1838 180.5 42.3763 181.158 42.4103 181.833C42.4103 183.078 41.5317 183.58 40.4795 182.962C39.9163 182.587 39.4486 182.085 39.1134 181.496C38.7783 180.908 38.5849 180.25 38.5487 179.574C38.5293 178.329 39.4079 177.798 40.4795 178.454Z"
        fill="#489CFF"
      />
      <path
        d="M161.298 156.625L124.613 177.816C123.896 178.285 123.3 178.917 122.872 179.66C122.444 180.403 122.197 181.236 122.151 182.092V187.084C122.151 188.647 123.252 189.285 124.613 188.503L161.298 167.312C162.017 166.843 162.615 166.211 163.044 165.468C163.473 164.725 163.722 163.892 163.77 163.035V158.054C163.77 156.48 162.679 155.843 161.298 156.625Z"
        fill="#455A64"
      />
      <path
        d="M161.164 160.854V163.316C161.155 163.488 161.106 163.655 161.021 163.804C160.935 163.954 160.815 164.081 160.671 164.175L160.121 164.494C159.841 164.648 159.619 164.493 159.619 164.214V161.79C159.629 161.618 159.679 161.45 159.766 161.3C159.854 161.151 159.975 161.024 160.121 160.931L160.671 160.613C160.942 160.42 161.164 160.535 161.164 160.854Z"
        fill="#489CFF"
      />
      <path
        d="M158.673 162.292V164.706C158.665 164.877 158.616 165.045 158.53 165.194C158.444 165.344 158.325 165.47 158.181 165.565L157.621 165.883C157.35 166.038 157.128 165.883 157.128 165.603V163.18C157.138 163.009 157.188 162.842 157.273 162.693C157.359 162.544 157.478 162.416 157.621 162.321L158.181 162.002C158.451 161.848 158.673 161.973 158.673 162.292Z"
        fill="#489CFF"
      />
      <path
        d="M156.182 163.72V166.143C156.172 166.315 156.122 166.482 156.037 166.631C155.951 166.78 155.832 166.907 155.689 167.002L155.13 167.321C154.859 167.476 154.637 167.321 154.637 167.031V164.618C154.646 164.446 154.695 164.279 154.78 164.129C154.866 163.98 154.986 163.853 155.13 163.759L155.689 163.44C155.921 163.315 156.182 163.411 156.182 163.72Z"
        fill="#489CFF"
      />
      <path
        d="M153.692 165.161V167.584C153.684 167.755 153.635 167.921 153.549 168.069C153.463 168.216 153.344 168.341 153.2 168.434L152.64 168.762C152.37 168.916 152.147 168.762 152.147 168.472V166.049C152.157 165.879 152.207 165.713 152.293 165.566C152.378 165.418 152.497 165.293 152.64 165.199L153.2 164.881C153.47 164.717 153.692 164.852 153.692 165.161Z"
        fill="#489CFF"
      />
      <path
        d="M151.191 166.598V169.022C151.184 169.192 151.135 169.359 151.049 169.507C150.963 169.655 150.843 169.78 150.699 169.871L150.149 170.19C149.878 170.354 149.656 170.19 149.656 169.91V167.487C149.666 167.316 149.716 167.151 149.801 167.003C149.887 166.856 150.006 166.73 150.149 166.637L150.699 166.318C150.979 166.154 151.191 166.289 151.191 166.598Z"
        fill="#37474F"
      />
      <path
        d="M148.701 168.036V170.459C148.691 170.63 148.641 170.795 148.555 170.943C148.47 171.09 148.351 171.216 148.208 171.309L147.658 171.627C147.378 171.792 147.166 171.627 147.166 171.347V168.924C147.173 168.754 147.222 168.587 147.308 168.439C147.393 168.291 147.514 168.166 147.658 168.075L148.208 167.747C148.478 167.592 148.701 167.718 148.701 168.036Z"
        fill="#37474F"
      />
      <path
        d="M146.21 169.475V171.888C146.201 172.06 146.152 172.228 146.067 172.377C145.981 172.526 145.861 172.653 145.717 172.747L145.157 173.066C144.887 173.22 144.665 173.066 144.665 172.786V170.363C144.675 170.191 144.725 170.024 144.81 169.875C144.896 169.726 145.014 169.599 145.157 169.504L145.717 169.185C145.988 169.031 146.21 169.156 146.21 169.475Z"
        fill="#489CFF"
      />
      <path
        d="M143.719 170.914V173.328C143.71 173.499 143.661 173.667 143.576 173.816C143.49 173.966 143.371 174.093 143.227 174.187L142.667 174.505C142.396 174.66 142.174 174.505 142.174 174.216V171.802C142.184 171.631 142.234 171.464 142.319 171.315C142.405 171.166 142.524 171.038 142.667 170.943L143.227 170.624C143.497 170.47 143.719 170.595 143.719 170.914Z"
        fill="#489CFF"
      />
      <path
        d="M141.228 172.342V174.765C141.218 174.936 141.169 175.103 141.083 175.252C140.998 175.402 140.879 175.529 140.736 175.624L140.176 175.943C139.906 176.097 139.684 175.943 139.684 175.653V173.24C139.692 173.068 139.741 172.9 139.827 172.751C139.912 172.602 140.032 172.475 140.176 172.38L140.736 172.062C141.006 171.907 141.228 172.004 141.228 172.342Z"
        fill="#489CFF"
      />
      <path
        d="M138.738 173.781V176.204C138.729 176.376 138.679 176.543 138.591 176.691C138.504 176.839 138.382 176.963 138.235 177.054L137.685 177.382C137.415 177.536 137.193 177.382 137.193 177.092V174.669C137.203 174.499 137.253 174.333 137.338 174.186C137.424 174.038 137.542 173.913 137.685 173.82L138.235 173.501C138.544 173.337 138.738 173.472 138.738 173.781Z"
        fill="#37474F"
      />
      <path
        d="M136.237 175.22V177.643C136.227 177.814 136.177 177.979 136.092 178.127C136.006 178.274 135.888 178.4 135.745 178.493L135.194 178.812C134.924 178.976 134.702 178.812 134.702 178.532V176.108C134.712 175.938 134.762 175.773 134.847 175.625C134.933 175.478 135.052 175.352 135.194 175.259L135.745 174.94C136.015 174.776 136.237 174.902 136.237 175.22Z"
        fill="#489CFF"
      />
      <path
        d="M133.718 176.658V179.081C133.708 179.252 133.658 179.417 133.572 179.565C133.487 179.712 133.368 179.838 133.225 179.931L132.675 180.25C132.395 180.414 132.183 180.25 132.183 179.97V177.546C132.19 177.376 132.239 177.209 132.325 177.061C132.41 176.913 132.531 176.788 132.675 176.697L133.225 176.369C133.525 176.214 133.718 176.34 133.718 176.658Z"
        fill="#489CFF"
      />
      <path
        d="M131.256 178.096V180.51C131.247 180.682 131.198 180.849 131.113 180.998C131.027 181.148 130.907 181.275 130.763 181.369L130.203 181.688C129.933 181.842 129.711 181.688 129.711 181.408V178.984C129.721 178.813 129.771 178.646 129.856 178.497C129.941 178.348 130.06 178.221 130.203 178.125L130.763 177.807C131.034 177.652 131.256 177.797 131.256 178.096Z"
        fill="#37474F"
      />
      <path
        d="M128.765 179.534V181.948C128.757 182.12 128.708 182.287 128.622 182.436C128.537 182.586 128.417 182.713 128.273 182.807L127.713 183.125C127.443 183.28 127.221 183.126 127.221 182.836V180.422C127.231 180.251 127.28 180.084 127.366 179.935C127.451 179.786 127.57 179.658 127.713 179.563L128.273 179.245C128.543 179.09 128.765 179.216 128.765 179.534Z"
        fill="#489CFF"
      />
      <path
        d="M126.274 180.963V183.386C126.264 183.557 126.214 183.724 126.129 183.873C126.044 184.022 125.925 184.15 125.782 184.245L125.222 184.564C124.952 184.718 124.729 184.564 124.729 184.274V181.86C124.738 181.689 124.787 181.521 124.873 181.372C124.958 181.222 125.078 181.096 125.222 181.001L125.782 180.683C126.052 180.528 126.274 180.692 126.274 180.963Z"
        fill="#37474F"
      />
      <path
        d="M118.416 185.819L117.393 185.221C117.173 185.119 116.93 185.079 116.689 185.102C116.448 185.126 116.218 185.214 116.022 185.356C115.458 185.731 114.989 186.232 114.654 186.821C114.318 187.409 114.126 188.068 114.091 188.745C114.067 188.985 114.106 189.229 114.207 189.449C114.307 189.669 114.463 189.859 114.661 190L115.626 190.54L115.8 190.251C115.996 190.209 116.185 190.138 116.36 190.038C116.923 189.665 117.39 189.164 117.726 188.578C118.061 187.991 118.254 187.334 118.29 186.659C118.288 186.491 118.268 186.322 118.233 186.157L118.416 185.819Z"
        fill="#455A64"
      />
      <path
        d="M116.997 185.925C116.434 186.301 115.966 186.803 115.631 187.391C115.296 187.98 115.102 188.638 115.066 189.314C115.066 190.559 115.944 191.061 116.997 190.434C117.562 190.062 118.031 189.563 118.367 188.975C118.702 188.388 118.894 187.73 118.928 187.055C118.947 185.809 118.078 185.307 116.997 185.925Z"
        fill="#489CFF"
      />
      <path
        opacity="0.8"
        d="M116.997 185.925C116.434 186.301 115.966 186.803 115.631 187.391C115.296 187.98 115.102 188.638 115.066 189.314C115.066 190.559 115.944 191.061 116.997 190.434C117.562 190.062 118.031 189.563 118.367 188.975C118.702 188.388 118.894 187.73 118.928 187.055C118.947 185.809 118.078 185.307 116.997 185.925Z"
        fill="#183867"
      />
      <path
        d="M109.949 190.704L108.916 190.115C108.695 190.018 108.452 189.98 108.211 190.006C107.971 190.031 107.741 190.119 107.545 190.26C106.98 190.632 106.511 191.131 106.175 191.718C105.84 192.305 105.648 192.964 105.614 193.639C105.587 193.881 105.626 194.127 105.726 194.349C105.826 194.572 105.984 194.763 106.184 194.904L107.149 195.454L107.323 195.164C107.521 195.121 107.71 195.046 107.883 194.942C108.448 194.571 108.918 194.071 109.253 193.484C109.589 192.897 109.781 192.239 109.814 191.563C109.816 191.397 109.796 191.232 109.756 191.071L109.949 190.704Z"
        fill="#455A64"
      />
      <path
        d="M108.521 190.821C107.957 191.194 107.488 191.694 107.153 192.281C106.817 192.867 106.625 193.525 106.59 194.2C106.59 195.445 107.468 195.947 108.521 195.33C109.085 194.957 109.553 194.457 109.889 193.87C110.224 193.283 110.417 192.626 110.451 191.951C110.48 190.705 109.583 190.194 108.521 190.821Z"
        fill="#489CFF"
      />
      <path
        opacity="0.8"
        d="M108.521 190.821C107.957 191.194 107.488 191.694 107.153 192.281C106.817 192.867 106.625 193.525 106.59 194.2C106.59 195.445 107.468 195.947 108.521 195.33C109.085 194.957 109.553 194.457 109.889 193.87C110.224 193.283 110.417 192.626 110.451 191.951C110.48 190.705 109.583 190.194 108.521 190.821Z"
        fill="#183867"
      />
      <path
        d="M53.4631 180.982L91.8765 203.187C92.2386 203.423 92.54 203.74 92.7564 204.114C92.9727 204.488 93.098 204.908 93.1219 205.34V215.959C93.1219 216.751 92.562 217.069 91.8765 216.673L53.4631 194.469C53.0949 194.231 52.7894 193.907 52.5726 193.526C52.3559 193.144 52.2341 192.716 52.2178 192.278V181.658C52.2178 180.905 52.7681 180.587 53.4631 180.982Z"
        fill="#455A64"
      />
      <path
        d="M54.5547 185.654C54.5836 186.072 54.7116 186.478 54.9282 186.837C55.1447 187.196 55.4435 187.499 55.8001 187.72L89.5409 207.192C90.2263 207.588 90.7863 207.298 90.7863 206.555C90.7551 206.137 90.6261 205.732 90.4098 205.373C90.1935 205.015 89.8958 204.712 89.5409 204.489L55.8001 185.026C55.105 184.63 54.5547 184.91 54.5547 185.654Z"
        fill="#489CFF"
      />
      <path
        d="M54.5547 191.052C54.5836 191.47 54.7116 191.876 54.9282 192.235C55.1447 192.594 55.4435 192.897 55.8001 193.118L89.5409 212.551C90.2263 212.947 90.7863 212.667 90.7863 211.924C90.7551 211.506 90.6261 211.101 90.4098 210.742C90.1935 210.383 89.8958 210.08 89.5409 209.858L55.8001 190.424C55.105 190.028 54.5547 190.347 54.5547 191.052Z"
        fill="#263238"
      />
      <path
        d="M55.8001 190.424L74.442 201.169V203.863L55.8001 193.118C55.4435 192.897 55.1447 192.594 54.9282 192.235C54.7116 191.876 54.5836 191.47 54.5547 191.052C54.5547 190.347 55.105 190.028 55.8001 190.424Z"
        fill="#489CFF"
      />
      <path
        d="M36.9259 173.212L44.9002 177.797C45.5492 178.221 46.0898 178.791 46.4788 179.462C46.8677 180.132 47.0942 180.885 47.1399 181.659V186.196C47.1399 187.635 46.1745 188.214 44.9002 187.499L36.9742 182.943C36.3239 182.519 35.7818 181.949 35.3912 181.279C35.0006 180.608 34.7724 179.856 34.7248 179.081V174.534C34.6765 173.076 35.6902 172.497 36.9259 173.212Z"
        fill="#37474F"
      />
      <path
        d="M39.1079 178.337L40.1409 177.738C40.3618 177.639 40.6051 177.6 40.8459 177.626C41.0866 177.651 41.3164 177.74 41.5118 177.883C42.0768 178.254 42.5462 178.754 42.8817 179.341C43.2172 179.928 43.4094 180.586 43.4426 181.262C43.4677 181.504 43.4282 181.749 43.3281 181.971C43.2281 182.193 43.0711 182.385 42.873 182.527L41.9076 183.077L41.7338 182.787C41.5372 182.741 41.3485 182.666 41.1739 182.565C40.6088 182.194 40.1394 181.694 39.8039 181.107C39.4684 180.52 39.2762 179.862 39.2431 179.186C39.2416 179.017 39.261 178.849 39.301 178.684L39.1079 178.337Z"
        fill="#489CFF"
      />
      <path
        opacity="0.2"
        d="M39.1079 178.337L40.1409 177.738C40.3618 177.639 40.6051 177.6 40.8459 177.626C41.0866 177.651 41.3164 177.74 41.5118 177.883C42.0768 178.254 42.5462 178.754 42.8817 179.341C43.2172 179.928 43.4094 180.586 43.4426 181.262C43.4677 181.504 43.4282 181.749 43.3281 181.971C43.2281 182.193 43.0711 182.385 42.873 182.527L41.9076 183.077L41.7338 182.787C41.5372 182.741 41.3485 182.666 41.1739 182.565C40.6088 182.194 40.1394 181.694 39.8039 181.107C39.4684 180.52 39.2762 179.862 39.2431 179.186C39.2416 179.017 39.261 178.849 39.301 178.684L39.1079 178.337Z"
        fill="black"
      />
      <path
        d="M40.4795 178.454C41.044 178.826 41.513 179.326 41.8484 179.913C42.1838 180.5 42.3763 181.158 42.4103 181.833C42.4103 183.078 41.5317 183.58 40.4795 182.962C39.9163 182.587 39.4486 182.085 39.1134 181.496C38.7783 180.908 38.5849 180.25 38.5487 179.574C38.5293 178.329 39.4079 177.798 40.4795 178.454Z"
        fill="#489CFF"
      />
      <path
        d="M53.4631 180.982L91.8765 203.187C92.2386 203.423 92.54 203.74 92.7564 204.114C92.9727 204.488 93.098 204.908 93.1219 205.34V215.959C93.1219 216.751 92.562 217.069 91.8765 216.673L53.4631 194.469C53.0949 194.231 52.7894 193.907 52.5726 193.526C52.3559 193.144 52.2341 192.716 52.2178 192.278V181.658C52.2178 180.905 52.7681 180.587 53.4631 180.982Z"
        fill="#37474F"
      />
      <path
        d="M54.5547 185.654C54.5836 186.072 54.7116 186.478 54.9282 186.837C55.1447 187.196 55.4435 187.499 55.8001 187.72L89.5409 207.192C90.2263 207.588 90.7863 207.298 90.7863 206.555C90.7551 206.137 90.6261 205.732 90.4098 205.373C90.1935 205.015 89.8958 204.712 89.5409 204.489L55.8001 185.026C55.105 184.63 54.5547 184.91 54.5547 185.654Z"
        fill="#489CFF"
      />
      <path
        d="M54.5547 191.052C54.5836 191.47 54.7116 191.876 54.9282 192.235C55.1447 192.594 55.4435 192.897 55.8001 193.118L89.5409 212.551C90.2263 212.947 90.7863 212.667 90.7863 211.924C90.7551 211.506 90.6261 211.101 90.4098 210.742C90.1935 210.383 89.8958 210.08 89.5409 209.858L55.8001 190.424C55.105 190.028 54.5547 190.347 54.5547 191.052Z"
        fill="#263238"
      />
      <path
        d="M55.8001 190.424L74.442 201.169V203.863L55.8001 193.118C55.4435 192.897 55.1447 192.594 54.9282 192.235C54.7116 191.876 54.5836 191.47 54.5547 191.052C54.5547 190.347 55.105 190.028 55.8001 190.424Z"
        fill="#489CFF"
      />
      <path
        d="M53.4631 202.897L91.8765 225.101C92.2377 225.34 92.5383 225.66 92.7545 226.035C92.9706 226.411 93.0964 226.831 93.1219 227.264V237.883C93.1219 238.675 92.562 238.993 91.8765 238.597L53.4631 216.412C53.0993 216.176 52.7967 215.857 52.5802 215.481C52.3637 215.105 52.2395 214.683 52.2178 214.25V203.631C52.2178 202.897 52.7681 202.53 53.4631 202.897Z"
        fill="#37474F"
      />
      <path
        d="M54.5547 207.599C54.5851 208.016 54.7138 208.42 54.9303 208.777C55.1467 209.134 55.4448 209.435 55.8001 209.655L89.5409 229.118C90.2263 229.523 90.7863 229.234 90.7863 228.49C90.7551 228.073 90.6261 227.668 90.4098 227.309C90.1935 226.95 89.8958 226.647 89.5409 226.424L55.8001 206.972C55.105 206.576 54.5547 206.856 54.5547 207.599Z"
        fill="#489CFF"
      />
      <path
        d="M54.5547 212.996C54.5836 213.414 54.7116 213.82 54.9282 214.179C55.1447 214.538 55.4435 214.841 55.8001 215.062L89.5409 234.524C90.2263 234.92 90.7863 234.64 90.7863 233.897C90.7551 233.479 90.6261 233.074 90.4098 232.715C90.1935 232.356 89.8958 232.053 89.5409 231.831L55.8001 212.368C55.105 211.972 54.5547 212.252 54.5547 212.996Z"
        fill="#263238"
      />
      <path
        d="M55.8001 212.368L79.1146 225.807V228.5L55.8001 215.062C55.4435 214.841 55.1447 214.538 54.9282 214.179C54.7116 213.82 54.5836 213.414 54.5547 212.996C54.5547 212.252 55.105 211.972 55.8001 212.368Z"
        fill="#489CFF"
      />
      <path
        d="M36.9269 217.107L44.9011 221.654C45.5496 222.079 46.0899 222.649 46.4788 223.319C46.8677 223.99 47.0944 224.742 47.1409 225.516V230.063C47.1409 231.492 46.1755 232.071 44.9011 231.356L36.9752 226.809C36.3264 226.384 35.7855 225.814 35.3951 225.144C35.0047 224.474 34.7756 223.722 34.7258 222.948V218.401C34.6775 216.972 35.6912 216.412 36.9269 217.107Z"
        fill="#455A64"
      />
      <path
        d="M39.1074 222.204L40.1404 221.606C40.3603 221.504 40.6032 221.463 40.8442 221.487C41.0852 221.511 41.3154 221.598 41.5113 221.741C42.0758 222.113 42.5448 222.613 42.8802 223.2C43.2157 223.787 43.4081 224.444 43.4421 225.12C43.4692 225.362 43.4307 225.608 43.3305 225.83C43.2302 226.053 43.0721 226.244 42.8725 226.384L41.9071 226.935L41.7333 226.645C41.5371 226.603 41.3484 226.531 41.1734 226.433C40.6087 226.059 40.1397 225.557 39.8044 224.968C39.469 224.38 39.2766 223.721 39.2426 223.044C39.2406 222.878 39.2601 222.713 39.3005 222.552L39.1074 222.204Z"
        fill="#489CFF"
      />
      <path
        opacity="0.2"
        d="M39.1074 222.204L40.1404 221.606C40.3603 221.504 40.6032 221.463 40.8442 221.487C41.0852 221.511 41.3154 221.598 41.5113 221.741C42.0758 222.113 42.5448 222.613 42.8802 223.2C43.2157 223.787 43.4081 224.444 43.4421 225.12C43.4692 225.362 43.4307 225.608 43.3305 225.83C43.2302 226.053 43.0721 226.244 42.8725 226.384L41.9071 226.935L41.7333 226.645C41.5371 226.603 41.3484 226.531 41.1734 226.433C40.6087 226.059 40.1397 225.557 39.8044 224.968C39.469 224.38 39.2766 223.721 39.2426 223.044C39.2406 222.878 39.2601 222.713 39.3005 222.552L39.1074 222.204Z"
        fill="black"
      />
      <path
        d="M40.4785 222.34C41.043 222.712 41.512 223.212 41.8474 223.799C42.1829 224.386 42.3753 225.044 42.4093 225.719C42.4093 226.964 41.5308 227.466 40.4785 226.848C39.9145 226.476 39.4459 225.976 39.1106 225.389C38.7752 224.802 38.5824 224.145 38.5477 223.469C38.5284 222.205 39.4069 221.722 40.4785 222.34Z"
        fill="#489CFF"
      />
      <path
        d="M36.9269 217.107L44.9011 221.654C45.5496 222.079 46.0899 222.649 46.4788 223.319C46.8677 223.99 47.0944 224.742 47.1409 225.516V230.063C47.1409 231.492 46.1755 232.071 44.9011 231.356L36.9752 226.809C36.3264 226.384 35.7855 225.814 35.3951 225.144C35.0047 224.474 34.7756 223.722 34.7258 222.948V218.401C34.6775 216.972 35.6912 216.412 36.9269 217.107Z"
        fill="#37474F"
      />
      <path
        d="M39.1074 222.204L40.1404 221.606C40.3603 221.504 40.6032 221.463 40.8442 221.487C41.0852 221.511 41.3154 221.598 41.5113 221.741C42.0758 222.113 42.5448 222.613 42.8802 223.2C43.2157 223.787 43.4081 224.444 43.4421 225.12C43.4692 225.362 43.4307 225.608 43.3305 225.83C43.2302 226.053 43.0721 226.244 42.8725 226.384L41.9071 226.935L41.7333 226.645C41.5371 226.603 41.3484 226.531 41.1734 226.433C40.6087 226.059 40.1397 225.557 39.8044 224.968C39.469 224.38 39.2766 223.721 39.2426 223.044C39.2406 222.878 39.2601 222.713 39.3005 222.552L39.1074 222.204Z"
        fill="#489CFF"
      />
      <path
        opacity="0.2"
        d="M39.1074 222.204L40.1404 221.606C40.3603 221.504 40.6032 221.463 40.8442 221.487C41.0852 221.511 41.3154 221.598 41.5113 221.741C42.0758 222.113 42.5448 222.613 42.8802 223.2C43.2157 223.787 43.4081 224.444 43.4421 225.12C43.4692 225.362 43.4307 225.608 43.3305 225.83C43.2302 226.053 43.0721 226.244 42.8725 226.384L41.9071 226.935L41.7333 226.645C41.5371 226.603 41.3484 226.531 41.1734 226.433C40.6087 226.059 40.1397 225.557 39.8044 224.968C39.469 224.38 39.2766 223.721 39.2426 223.044C39.2406 222.878 39.2601 222.713 39.3005 222.552L39.1074 222.204Z"
        fill="black"
      />
      <path
        d="M40.4785 222.34C41.043 222.712 41.512 223.212 41.8474 223.799C42.1829 224.386 42.3753 225.044 42.4093 225.719C42.4093 226.964 41.5308 227.466 40.4785 226.848C39.9145 226.476 39.4459 225.976 39.1106 225.389C38.7752 224.802 38.5824 224.145 38.5477 223.469C38.5284 222.205 39.4069 221.722 40.4785 222.34Z"
        fill="#489CFF"
      />
      <path
        d="M53.4637 224.878L91.8771 247.082C92.2481 247.313 92.559 247.628 92.7839 248.003C93.0089 248.377 93.1415 248.799 93.1708 249.235V259.855C93.1708 260.656 92.6108 260.975 91.9254 260.579L53.512 238.374C53.1489 238.14 52.8466 237.822 52.6301 237.448C52.4136 237.074 52.289 236.653 52.2666 236.222V225.602C52.2184 224.801 52.7686 224.473 53.4637 224.878Z"
        fill="#37474F"
      />
      <path
        d="M54.5542 229.552C54.5831 229.97 54.7111 230.376 54.9277 230.735C55.1442 231.094 55.4431 231.397 55.7996 231.618L89.5404 251.08C90.2259 251.476 90.7858 251.196 90.7858 250.453C90.7546 250.035 90.6256 249.63 90.4093 249.272C90.193 248.913 89.8953 248.61 89.5404 248.387L55.7996 228.963C55.1045 228.519 54.5542 228.809 54.5542 229.552Z"
        fill="#489CFF"
      />
      <path
        d="M54.5542 234.938C54.5832 235.358 54.7111 235.765 54.9276 236.125C55.144 236.486 55.4429 236.791 55.7996 237.014L89.5404 256.476C90.2259 256.872 90.7858 256.592 90.7858 255.849C90.7545 255.429 90.6256 255.023 90.4094 254.663C90.1931 254.303 89.8956 253.998 89.5404 253.773L55.7996 234.31C55.1045 233.915 54.5542 234.195 54.5542 234.938Z"
        fill="#263238"
      />
      <path
        d="M55.7996 234.31L74.4415 245.055V247.759L55.7996 237.014C55.4429 236.791 55.144 236.486 54.9276 236.125C54.7111 235.765 54.5832 235.358 54.5542 234.938C54.5542 234.195 55.1045 233.915 55.7996 234.31Z"
        fill="#489CFF"
      />
      <path
        opacity="0.1"
        d="M153.305 43.0359L102.071 72.6159C101.661 72.8207 101.208 72.9273 100.749 72.9273C100.29 72.9273 99.8369 72.8207 99.4261 72.6159L48.1921 43.0359C48.032 42.9813 47.8929 42.8779 47.7945 42.7402C47.696 42.6026 47.6431 42.4376 47.6431 42.2684C47.6431 42.0992 47.696 41.9342 47.7945 41.7966C47.8929 41.6589 48.032 41.5556 48.1921 41.5009L99.4261 12.9443C99.8369 12.7394 100.29 12.6328 100.749 12.6328C101.208 12.6328 101.661 12.7394 102.071 12.9443L153.305 41.5009C153.465 41.5556 153.604 41.6589 153.703 41.7966C153.801 41.9342 153.854 42.0992 153.854 42.2684C153.854 42.4376 153.801 42.6026 153.703 42.7402C153.604 42.8779 153.465 42.9813 153.305 43.0359Z"
        fill="black"
      />
      <path
        d="M102.013 70.9931L151.037 42.6875C151.406 42.4482 151.715 42.1251 151.936 41.7443C152.157 41.3635 152.286 40.9358 152.311 40.496V38.8452C152.287 38.4039 152.159 37.9746 151.937 37.5922C151.716 37.2097 151.407 36.8849 151.037 36.6441L102.013 8.33845C101.622 8.13846 101.188 8.03418 100.749 8.03418C100.309 8.03418 99.8756 8.13846 99.484 8.33845L50.4609 36.6441C50.0903 36.8849 49.7818 37.2097 49.5603 37.5922C49.3389 37.9746 49.2108 38.4039 49.1865 38.8452V40.496C49.2116 40.9358 49.34 41.3635 49.5614 41.7443C49.7829 42.1251 50.091 42.4482 50.4609 42.6875L99.484 70.9931C99.8756 71.1931 100.309 71.2974 100.749 71.2974C101.188 71.2974 101.622 71.1931 102.013 70.9931Z"
        fill="#489CFF"
      />
      <path
        opacity="0.5"
        d="M102.013 70.9931L151.037 42.6875C151.406 42.4482 151.715 42.1251 151.936 41.7443C152.157 41.3635 152.286 40.9358 152.311 40.496V38.8452C152.287 38.4039 152.159 37.9746 151.937 37.5922C151.716 37.2097 151.407 36.8849 151.037 36.6441L102.013 8.33845C101.622 8.13846 101.188 8.03418 100.749 8.03418C100.309 8.03418 99.8756 8.13846 99.484 8.33845L50.4609 36.6441C50.0903 36.8849 49.7818 37.2097 49.5603 37.5922C49.3389 37.9746 49.2108 38.4039 49.1865 38.8452V40.496C49.2116 40.9358 49.34 41.3635 49.5614 41.7443C49.7829 42.1251 50.091 42.4482 50.4609 42.6875L99.484 70.9931C99.8756 71.1931 100.309 71.2974 100.749 71.2974C101.188 71.2974 101.622 71.1931 102.013 70.9931Z"
        fill="#183867"
      />
      <path
        opacity="0.05"
        d="M102.013 70.9931L151.037 42.6875C151.406 42.4482 151.715 42.1251 151.936 41.7443C152.157 41.3635 152.286 40.9358 152.311 40.496V38.8452C152.287 38.4039 152.159 37.9746 151.937 37.5922C151.716 37.2097 151.407 36.8849 151.037 36.6441L102.013 8.33845C101.622 8.13846 101.188 8.03418 100.749 8.03418C100.309 8.03418 99.8756 8.13846 99.484 8.33845L50.4609 36.6441C50.0903 36.8849 49.7818 37.2097 49.5603 37.5922C49.3389 37.9746 49.2108 38.4039 49.1865 38.8452V40.496C49.2116 40.9358 49.34 41.3635 49.5614 41.7443C49.7829 42.1251 50.091 42.4482 50.4609 42.6875L99.484 70.9931C99.8756 71.1931 100.309 71.2974 100.749 71.2974C101.188 71.2974 101.622 71.1931 102.013 70.9931Z"
        fill="black"
      />
      <path
        d="M151.037 38.1112L102.013 66.4071C101.622 66.6071 101.188 66.7114 100.749 66.7114C100.309 66.7114 99.8756 66.6071 99.484 66.4071L50.4609 38.1112C50.3063 38.0604 50.1718 37.9621 50.0764 37.8303C49.981 37.6986 49.9297 37.5401 49.9297 37.3775C49.9297 37.2148 49.981 37.0563 50.0764 36.9246C50.1718 36.7928 50.3063 36.6945 50.4609 36.6437L99.484 9.32283C99.8756 9.12283 100.309 9.01855 100.749 9.01855C101.188 9.01855 101.622 9.12283 102.013 9.32283L151.037 36.6437C151.191 36.6945 151.326 36.7928 151.421 36.9246C151.516 37.0563 151.568 37.2148 151.568 37.3775C151.568 37.5401 151.516 37.6986 151.421 37.8303C151.326 37.9621 151.191 38.0604 151.037 38.1112Z"
        fill="#489CFF"
      />
      <path
        opacity="0.7"
        d="M151.037 38.1112L102.013 66.4071C101.622 66.6071 101.188 66.7114 100.749 66.7114C100.309 66.7114 99.8756 66.6071 99.484 66.4071L50.4609 38.1112C50.3063 38.0604 50.1718 37.9621 50.0764 37.8303C49.981 37.6986 49.9297 37.5401 49.9297 37.3775C49.9297 37.2148 49.981 37.0563 50.0764 36.9246C50.1718 36.7928 50.3063 36.6945 50.4609 36.6437L99.484 9.32283C99.8756 9.12283 100.309 9.01855 100.749 9.01855C101.188 9.01855 101.622 9.12283 102.013 9.32283L151.037 36.6437C151.191 36.6945 151.326 36.7928 151.421 36.9246C151.516 37.0563 151.568 37.2148 151.568 37.3775C151.568 37.5401 151.516 37.6986 151.421 37.8303C151.326 37.9621 151.191 38.0604 151.037 38.1112Z"
        fill="#183867"
      />
      <path
        d="M100.749 66.7158V71.2918C101.189 71.3017 101.625 71.1987 102.014 70.9925L151.037 42.6869C151.407 42.4461 151.716 42.1213 151.937 41.7388C152.159 41.3564 152.287 40.9271 152.311 40.4858V38.835C152.289 38.4438 152.186 38.0615 152.008 37.7124C151.831 37.3634 151.582 37.0552 151.278 36.8076C151.732 37.2034 151.655 37.773 151.037 38.1013L102.014 66.4069C101.627 66.6206 101.191 66.7272 100.749 66.7158Z"
        fill="#489CFF"
      />
      <path
        opacity="0.6"
        d="M100.749 66.7158V71.2918C101.189 71.3017 101.625 71.1987 102.014 70.9925L151.037 42.6869C151.407 42.4461 151.716 42.1213 151.937 41.7388C152.159 41.3564 152.287 40.9271 152.311 40.4858V38.835C152.289 38.4438 152.186 38.0615 152.008 37.7124C151.831 37.3634 151.582 37.0552 151.278 36.8076C151.732 37.2034 151.655 37.773 151.037 38.1013L102.014 66.4069C101.627 66.6206 101.191 66.7272 100.749 66.7158Z"
        fill="#183867"
      />
      <path
        d="M102.013 57.2554L135.175 38.1115C135.545 37.8708 135.852 37.5457 136.072 37.1631C136.292 36.7805 136.418 36.3511 136.44 35.9104V34.2595C136.417 33.8203 136.291 33.3926 136.071 33.0116C135.851 32.6306 135.544 32.3073 135.175 32.0681L113.936 19.7881C113.116 19.2728 112.391 18.6196 111.793 17.8573L101.637 4.17757C101.392 3.86201 101.043 3.64398 100.652 3.56259C100.261 3.48121 99.8536 3.54179 99.5032 3.73348L66.3223 22.9064C65.9537 23.1462 65.647 23.4697 65.4273 23.8505C65.2075 24.2314 65.0808 24.6588 65.0576 25.0979V35.8814C65.0816 36.3216 65.2085 36.7501 65.4282 37.1324C65.6478 37.5147 65.9541 37.8401 66.3223 38.0825L99.4839 57.2265C99.8732 57.431 100.305 57.5402 100.745 57.5453C101.185 57.5503 101.619 57.451 102.013 57.2554Z"
        fill="#263238"
      />
      <path
        d="M99.4844 52.6698L78.2456 40.3995C77.8491 40.1635 77.4772 39.8887 77.1354 39.5789C76.7558 39.2503 76.4098 38.8849 76.1024 38.488L65.927 24.8082C65.8065 24.6625 65.7201 24.4917 65.6743 24.3082C65.6286 24.1247 65.6245 23.9333 65.6625 23.748C65.7005 23.5628 65.7795 23.3884 65.8938 23.2377C66.0081 23.0871 66.1547 22.9639 66.3228 22.8774L99.4844 3.73348C99.8348 3.54179 100.242 3.48121 100.633 3.56259C101.024 3.64398 101.373 3.86201 101.618 4.17757L111.774 17.8573C112.302 18.6069 112.955 19.26 113.705 19.7881L135.137 31.9908C135.293 32.0388 135.429 32.1355 135.526 32.2667C135.623 32.3979 135.675 32.5567 135.675 32.7197C135.675 32.8828 135.623 33.0416 135.526 33.1727C135.429 33.3039 135.293 33.4006 135.137 33.4486L101.975 52.5926C101.595 52.7983 101.171 52.9124 100.739 52.9258C100.307 52.9392 99.877 52.8516 99.4844 52.6698Z"
        fill="#455A64"
      />
      <path
        d="M66.323 22.9067L99.4846 3.76278C99.8349 3.5711 100.242 3.51051 100.633 3.59189C101.024 3.67327 101.373 3.8913 101.618 4.20687L111.774 17.8866C112.086 18.2801 112.435 18.6422 112.817 18.9679C112.817 18.9679 77.0969 39.5793 77.0969 39.5696C76.7177 39.2442 76.3717 38.882 76.0639 38.4883L65.9271 24.8086C65.812 24.6634 65.7301 24.4946 65.6874 24.3143C65.6447 24.134 65.6421 23.9464 65.6799 23.765C65.7176 23.5836 65.7948 23.4126 65.9059 23.2643C66.0171 23.116 66.1594 22.9939 66.323 22.9067Z"
        fill="#37474F"
      />
      <path
        d="M127.134 32.3959C126.673 32.1626 126.164 32.041 125.647 32.041C125.13 32.041 124.621 32.1626 124.16 32.3959C123.33 32.8786 123.33 33.6412 124.16 34.1143C124.621 34.3476 125.13 34.4691 125.647 34.4691C126.164 34.4691 126.673 34.3476 127.134 34.1143C127.313 34.0533 127.469 33.9376 127.579 33.7835C127.69 33.6294 127.749 33.4446 127.749 33.2551C127.749 33.0655 127.69 32.8807 127.579 32.7266C127.469 32.5725 127.313 32.4569 127.134 32.3959Z"
        fill="#489CFF"
      />
      <path
        d="M121.167 35.8822C120.706 35.6489 120.197 35.5273 119.68 35.5273C119.164 35.5273 118.654 35.6489 118.194 35.8822C118.014 35.9432 117.858 36.0589 117.748 36.2129C117.637 36.367 117.578 36.5518 117.578 36.7414C117.578 36.9309 117.637 37.1157 117.748 37.2698C117.858 37.4239 118.014 37.5396 118.194 37.6006C118.654 37.8339 119.164 37.9554 119.68 37.9554C120.197 37.9554 120.706 37.8339 121.167 37.6006C121.346 37.5396 121.502 37.4239 121.613 37.2698C121.723 37.1157 121.782 36.9309 121.782 36.7414C121.782 36.5518 121.723 36.367 121.613 36.2129C121.502 36.0589 121.346 35.9432 121.167 35.8822Z"
        fill="#489CFF"
      />
      <path
        d="M115.23 39.2699C114.769 39.0366 114.259 38.915 113.743 38.915C113.226 38.915 112.717 39.0366 112.256 39.2699C112.077 39.3309 111.921 39.4465 111.81 39.6006C111.7 39.7547 111.641 39.9395 111.641 40.1291C111.641 40.3186 111.7 40.5034 111.81 40.6575C111.921 40.8116 112.077 40.9273 112.256 40.9883C112.717 41.2216 113.226 41.3431 113.743 41.3431C114.259 41.3431 114.769 41.2216 115.23 40.9883C115.409 40.9273 115.565 40.8116 115.675 40.6575C115.786 40.5034 115.845 40.3186 115.845 40.1291C115.845 39.9395 115.786 39.7547 115.675 39.6006C115.565 39.4465 115.409 39.3309 115.23 39.2699Z"
        fill="#263238"
      />
      <path
        d="M109.283 42.7064C108.822 42.4731 108.313 42.3516 107.796 42.3516C107.279 42.3516 106.77 42.4731 106.309 42.7064C106.13 42.7674 105.974 42.8831 105.864 43.0372C105.753 43.1913 105.694 43.3761 105.694 43.5656C105.694 43.7552 105.753 43.94 105.864 44.094C105.974 44.2481 106.13 44.3638 106.309 44.4248C106.77 44.6581 107.279 44.7796 107.796 44.7796C108.313 44.7796 108.822 44.6581 109.283 44.4248C109.462 44.3638 109.618 44.2481 109.728 44.094C109.839 43.94 109.898 43.7552 109.898 43.5656C109.898 43.3761 109.839 43.1913 109.728 43.0372C109.618 42.8831 109.462 42.7674 109.283 42.7064Z"
        fill="#263238"
      />
      <path
        d="M122.673 29.8288C122.212 29.5909 121.701 29.4668 121.182 29.4668C120.663 29.4668 120.151 29.5909 119.69 29.8288C119.51 29.8872 119.352 30.0013 119.24 30.1548C119.129 30.3084 119.068 30.4934 119.068 30.6832C119.068 30.8731 119.129 31.0581 119.24 31.2116C119.352 31.3651 119.51 31.4793 119.69 31.5376C120.151 31.7755 120.663 31.8997 121.182 31.8997C121.701 31.8997 122.212 31.7755 122.673 31.5376C122.854 31.4793 123.012 31.3651 123.123 31.2116C123.235 31.0581 123.295 30.8731 123.295 30.6832C123.295 30.4934 123.235 30.3084 123.123 30.1548C123.012 30.0013 122.854 29.8872 122.673 29.8288Z"
        fill="#489CFF"
      />
      <path
        d="M116.717 33.2552C116.256 33.0219 115.747 32.9004 115.23 32.9004C114.714 32.9004 114.204 33.0219 113.743 33.2552C113.564 33.3162 113.408 33.4319 113.298 33.586C113.187 33.7401 113.128 33.9249 113.128 34.1144C113.128 34.304 113.187 34.4888 113.298 34.6429C113.408 34.797 113.564 34.9126 113.743 34.9736C114.204 35.2069 114.714 35.3285 115.23 35.3285C115.747 35.3285 116.256 35.2069 116.717 34.9736C116.896 34.9126 117.052 34.797 117.163 34.6429C117.273 34.4888 117.332 34.304 117.332 34.1144C117.332 33.9249 117.273 33.7401 117.163 33.586C117.052 33.4319 116.896 33.3162 116.717 33.2552Z"
        fill="#263238"
      />
      <path
        d="M109.283 38.7688C110.446 38.7688 111.388 38.2241 111.388 37.5523C111.388 36.8805 110.446 36.3359 109.283 36.3359C108.121 36.3359 107.179 36.8805 107.179 37.5523C107.179 38.2241 108.121 38.7688 109.283 38.7688Z"
        fill="#489CFF"
      />
      <path
        d="M104.823 40.1292C104.362 39.896 103.853 39.7744 103.336 39.7744C102.82 39.7744 102.31 39.896 101.849 40.1292C101.67 40.1902 101.514 40.3059 101.404 40.46C101.293 40.6141 101.234 40.7989 101.234 40.9884C101.234 41.178 101.293 41.3628 101.404 41.5169C101.514 41.671 101.67 41.7867 101.849 41.8477C102.31 42.0809 102.82 42.2025 103.336 42.2025C103.853 42.2025 104.362 42.0809 104.823 41.8477C105.002 41.7867 105.158 41.671 105.268 41.5169C105.379 41.3628 105.438 41.178 105.438 40.9884C105.438 40.7989 105.379 40.6141 105.268 40.46C105.158 40.3059 105.002 40.1902 104.823 40.1292Z"
        fill="#263238"
      />
      <path
        d="M118.203 27.2503C117.742 27.0171 117.233 26.8955 116.716 26.8955C116.2 26.8955 115.691 27.0171 115.23 27.2503C115.05 27.3113 114.894 27.427 114.784 27.5811C114.674 27.7352 114.614 27.92 114.614 28.1096C114.614 28.2991 114.674 28.4839 114.784 28.638C114.894 28.7921 115.05 28.9078 115.23 28.9688C115.691 29.202 116.2 29.3236 116.716 29.3236C117.233 29.3236 117.742 29.202 118.203 28.9688C118.383 28.9078 118.538 28.7921 118.649 28.638C118.759 28.4839 118.819 28.2991 118.819 28.1096C118.819 27.92 118.759 27.7352 118.649 27.5811C118.538 27.427 118.383 27.3113 118.203 27.2503Z"
        fill="#263238"
      />
      <path
        d="M112.257 30.6872C111.797 30.4493 111.287 30.3252 110.77 30.3252C110.252 30.3252 109.743 30.4493 109.283 30.6872C109.102 30.7455 108.945 30.8597 108.833 31.0132C108.721 31.1667 108.661 31.3517 108.661 31.5416C108.661 31.7314 108.721 31.9164 108.833 32.07C108.945 32.2235 109.102 32.3376 109.283 32.3959C109.743 32.6338 110.252 32.7579 110.77 32.7579C111.287 32.7579 111.797 32.6338 112.257 32.3959C112.437 32.3376 112.595 32.2235 112.706 32.07C112.818 31.9164 112.878 31.7314 112.878 31.5416C112.878 31.3517 112.818 31.1667 112.706 31.0132C112.595 30.8597 112.437 30.7455 112.257 30.6872Z"
        fill="#489CFF"
      />
      <path
        d="M106.31 34.1156C105.849 33.8823 105.339 33.7607 104.823 33.7607C104.306 33.7607 103.797 33.8823 103.336 34.1156C103.157 34.1766 103.001 34.2922 102.89 34.4463C102.78 34.6004 102.721 34.7852 102.721 34.9748C102.721 35.1643 102.78 35.3491 102.89 35.5032C103.001 35.6573 103.157 35.773 103.336 35.834C103.797 36.0673 104.306 36.1888 104.823 36.1888C105.339 36.1888 105.849 36.0673 106.31 35.834C106.489 35.773 106.645 35.6573 106.755 35.5032C106.866 35.3491 106.925 35.1643 106.925 34.9748C106.925 34.7852 106.866 34.6004 106.755 34.4463C106.645 34.2922 106.489 34.1766 106.31 34.1156Z"
        fill="#263238"
      />
      <path
        d="M103.298 46.0668C102.837 45.8335 102.328 45.7119 101.811 45.7119C101.295 45.7119 100.785 45.8335 100.324 46.0668C100.145 46.1277 99.9891 46.2434 99.8787 46.3975C99.7683 46.5516 99.709 46.7364 99.709 46.926C99.709 47.1155 99.7683 47.3003 99.8787 47.4544C99.9891 47.6085 100.145 47.7242 100.324 47.7852C100.785 48.0184 101.295 48.14 101.811 48.14C102.328 48.14 102.837 48.0184 103.298 47.7852C103.477 47.7242 103.633 47.6085 103.744 47.4544C103.854 47.3003 103.913 47.1155 103.913 46.926C103.913 46.7364 103.854 46.5516 103.744 46.3975C103.633 46.2434 103.477 46.1277 103.298 46.0668Z"
        fill="#489CFF"
      />
      <path
        d="M97.3502 45.5647C98.5125 45.5647 99.4548 45.0201 99.4548 44.3483C99.4548 43.6764 98.5125 43.1318 97.3502 43.1318C96.1879 43.1318 95.2456 43.6764 95.2456 44.3483C95.2456 45.0201 96.1879 45.5647 97.3502 45.5647Z"
        fill="#263238"
      />
      <path
        d="M100.363 37.5511C99.9019 37.3178 99.3927 37.1963 98.8761 37.1963C98.3596 37.1963 97.8503 37.3178 97.3894 37.5511C97.2099 37.6121 97.054 37.7278 96.9437 37.8819C96.8333 38.036 96.7739 38.2208 96.7739 38.4103C96.7739 38.5999 96.8333 38.7847 96.9437 38.9388C97.054 39.0929 97.2099 39.2085 97.3894 39.2695C97.8503 39.5028 98.3596 39.6244 98.8761 39.6244C99.3927 39.6244 99.9019 39.5028 100.363 39.2695C100.542 39.2085 100.698 39.0929 100.809 38.9388C100.919 38.7847 100.978 38.5999 100.978 38.4103C100.978 38.2208 100.919 38.036 100.809 37.8819C100.698 37.7278 100.542 37.6121 100.363 37.5511Z"
        fill="#263238"
      />
      <path
        d="M113.743 24.6742C113.282 24.4409 112.773 24.3193 112.256 24.3193C111.74 24.3193 111.231 24.4409 110.77 24.6742C110.59 24.7352 110.434 24.8508 110.324 25.0049C110.214 25.159 110.154 25.3438 110.154 25.5334C110.154 25.7229 110.214 25.9077 110.324 26.0618C110.434 26.2159 110.59 26.3316 110.77 26.3926C111.231 26.6259 111.74 26.7474 112.256 26.7474C112.773 26.7474 113.282 26.6259 113.743 26.3926C113.923 26.3316 114.079 26.2159 114.189 26.0618C114.299 25.9077 114.359 25.7229 114.359 25.5334C114.359 25.3438 114.299 25.159 114.189 25.0049C114.079 24.8508 113.923 24.7352 113.743 24.6742Z"
        fill="#489CFF"
      />
      <path
        d="M107.796 28.1585C107.336 27.9253 106.826 27.8037 106.31 27.8037C105.793 27.8037 105.284 27.9253 104.823 28.1585C104.644 28.2195 104.488 28.3352 104.377 28.4893C104.267 28.6434 104.208 28.8282 104.208 29.0178C104.208 29.2073 104.267 29.3921 104.377 29.5462C104.488 29.7003 104.644 29.816 104.823 29.877C105.284 30.1102 105.793 30.2318 106.31 30.2318C106.826 30.2318 107.336 30.1102 107.796 29.877C107.976 29.816 108.132 29.7003 108.242 29.5462C108.353 29.3921 108.412 29.2073 108.412 29.0178C108.412 28.8282 108.353 28.6434 108.242 28.4893C108.132 28.3352 107.976 28.2195 107.796 28.1585Z"
        fill="#263238"
      />
      <path
        d="M101.859 31.5472C101.398 31.3139 100.889 31.1924 100.372 31.1924C99.8556 31.1924 99.3463 31.3139 98.8855 31.5472C98.706 31.6082 98.5501 31.7239 98.4398 31.878C98.3294 32.0321 98.27 32.2169 98.27 32.4064C98.27 32.596 98.3294 32.7808 98.4398 32.9349C98.5501 33.089 98.706 33.2046 98.8855 33.2656C99.3463 33.4989 99.8556 33.6205 100.372 33.6205C100.889 33.6205 101.398 33.4989 101.859 33.2656C102.038 33.2046 102.194 33.089 102.305 32.9349C102.415 32.7808 102.474 32.596 102.474 32.4064C102.474 32.2169 102.415 32.0321 102.305 31.878C102.194 31.7239 102.038 31.6082 101.859 31.5472Z"
        fill="#489CFF"
      />
      <path
        d="M95.9023 34.975C95.4414 34.7417 94.9321 34.6201 94.4156 34.6201C93.899 34.6201 93.3897 34.7417 92.9288 34.975C92.0986 35.4577 92.0986 36.2203 92.9288 36.6934C93.3897 36.9267 93.899 37.0482 94.4156 37.0482C94.9321 37.0482 95.4414 36.9267 95.9023 36.6934C96.0818 36.6324 96.2376 36.5167 96.348 36.3626C96.4584 36.2085 96.5177 36.0237 96.5177 35.8342C96.5177 35.6446 96.4584 35.4598 96.348 35.3057C96.2376 35.1516 96.0818 35.036 95.9023 34.975Z"
        fill="#263238"
      />
      <path
        d="M94.377 40.9203C93.9161 40.687 93.4068 40.5654 92.8903 40.5654C92.3737 40.5654 91.8644 40.687 91.4035 40.9203C91.2241 40.9813 91.0682 41.0969 90.9578 41.251C90.8474 41.4051 90.7881 41.5899 90.7881 41.7795C90.7881 41.969 90.8474 42.1538 90.9578 42.3079C91.0682 42.462 91.2241 42.5777 91.4035 42.6387C91.8644 42.872 92.3737 42.9935 92.8903 42.9935C93.4068 42.9935 93.9161 42.872 94.377 42.6387C94.5565 42.5777 94.7123 42.462 94.8227 42.3079C94.9331 42.1538 94.9924 41.969 94.9924 41.7795C94.9924 41.5899 94.9331 41.4051 94.8227 41.251C94.7123 41.0969 94.5565 40.9813 94.377 40.9203Z"
        fill="#489CFF"
      />
      <path
        d="M89.917 38.3431C89.4561 38.1098 88.9468 37.9883 88.4303 37.9883C87.9137 37.9883 87.4045 38.1098 86.9436 38.3431C86.7641 38.4041 86.6082 38.5198 86.4979 38.6739C86.3875 38.828 86.3281 39.0128 86.3281 39.2023C86.3281 39.3919 86.3875 39.5767 86.4979 39.7308C86.6082 39.8849 86.7641 40.0005 86.9436 40.0615C87.4045 40.2948 87.9137 40.4164 88.4303 40.4164C88.9468 40.4164 89.4561 40.2948 89.917 40.0615C90.0965 40.0005 90.2524 39.8849 90.3627 39.7308C90.4731 39.5767 90.5325 39.3919 90.5325 39.2023C90.5325 39.0128 90.4731 38.828 90.3627 38.6739C90.2524 38.5198 90.0965 38.4041 89.917 38.3431Z"
        fill="#263238"
      />
      <path
        d="M100.681 52.978V57.5443C101.143 57.5668 101.603 57.4669 102.013 57.2547L135.175 38.1011C135.545 37.8631 135.853 37.54 136.073 37.1588C136.293 36.7775 136.419 36.3493 136.44 35.9096V34.2588C136.42 33.8673 136.317 33.4843 136.139 33.135C135.961 32.7857 135.712 32.4778 135.407 32.2314C135.87 32.6273 135.793 33.1968 135.175 33.5251L102.013 52.669C101.606 52.8919 101.145 52.9988 100.681 52.978Z"
        fill="#37474F"
      />
      <path
        d="M107.391 17.3075C107.227 17.0967 106.993 16.9509 106.731 16.8961C106.47 16.8413 106.197 16.881 105.962 17.0083L77.0771 33.6325C76.9694 33.6928 76.876 33.7756 76.8032 33.8753C76.7304 33.9749 76.6799 34.0891 76.6551 34.21C76.6304 34.3309 76.632 34.4558 76.6598 34.576C76.6877 34.6963 76.7411 34.8091 76.8165 34.9068C76.9819 35.1161 77.2155 35.2607 77.4766 35.3154C77.7377 35.3701 78.0098 35.3314 78.2453 35.2061L107.092 18.5432C107.198 18.4869 107.291 18.4088 107.364 18.3139C107.438 18.219 107.491 18.1095 107.519 17.9927C107.547 17.8759 107.551 17.7545 107.528 17.6363C107.506 17.5182 107.459 17.4062 107.391 17.3075Z"
        fill="#263238"
      />
      <path
        d="M107.391 17.3075C107.227 17.0967 106.993 16.9509 106.731 16.8961C106.47 16.8413 106.197 16.881 105.962 17.0083L84.4819 29.3558L85.679 30.968L107.121 18.5819C107.23 18.5236 107.326 18.442 107.401 18.3429C107.475 18.2437 107.528 18.1293 107.553 18.0078C107.579 17.8863 107.578 17.7606 107.55 17.6396C107.522 17.5187 107.468 17.4053 107.391 17.3075Z"
        fill="#489CFF"
      />
      <path
        d="M105.006 14.1121C104.843 13.8996 104.609 13.7525 104.347 13.6976C104.085 13.6427 103.812 13.6836 103.577 13.8128L74.6926 30.437C74.5839 30.4951 74.4894 30.5764 74.4156 30.6751C74.3418 30.7738 74.2907 30.8876 74.2658 31.0083C74.2409 31.129 74.2429 31.2536 74.2717 31.3735C74.3004 31.4933 74.3551 31.6054 74.432 31.7017C74.595 31.9142 74.8288 32.0612 75.0909 32.1161C75.353 32.1711 75.6262 32.1302 75.8608 32.001L104.755 15.3864C104.863 15.3264 104.957 15.2435 105.03 15.1436C105.103 15.0436 105.153 14.9289 105.177 14.8076C105.201 14.6862 105.198 14.5611 105.168 14.441C105.139 14.3208 105.084 14.2086 105.006 14.1121Z"
        fill="#263238"
      />
      <path
        d="M105.007 14.1121C104.844 13.8996 104.61 13.7525 104.348 13.6976C104.085 13.6427 103.812 13.6836 103.578 13.8128L96.4048 17.9544L97.6019 19.5569L104.765 15.425C104.879 15.3651 104.977 15.2801 105.054 15.1766C105.13 15.073 105.181 14.9535 105.205 14.8271C105.228 14.7007 105.222 14.5707 105.188 14.4468C105.153 14.323 105.091 14.2085 105.007 14.1121Z"
        fill="#489CFF"
      />
      <path
        d="M102.622 10.907C102.459 10.6945 102.225 10.5474 101.963 10.4925C101.701 10.4376 101.428 10.4785 101.193 10.6077L72.3081 27.1933C72.2004 27.2536 72.107 27.3364 72.0341 27.4361C71.9613 27.5357 71.9108 27.6499 71.8861 27.7708C71.8613 27.8918 71.863 28.0166 71.8908 28.1368C71.9186 28.2571 71.9721 28.3699 72.0474 28.4677C72.2118 28.6786 72.4462 28.8237 72.7082 28.8768C72.9703 28.9299 73.2427 28.8875 73.4762 28.7573L102.322 12.1041C102.423 12.0474 102.511 11.9708 102.581 11.879C102.651 11.7871 102.702 11.682 102.73 11.5699C102.758 11.4579 102.763 11.3413 102.744 11.2273C102.725 11.1133 102.684 11.0043 102.622 10.907Z"
        fill="#263238"
      />
      <path
        d="M102.622 10.907C102.459 10.6945 102.225 10.5474 101.963 10.4925C101.701 10.4376 101.428 10.4785 101.193 10.6077L86.1621 19.2963L87.3785 20.8217L102.4 12.133C102.499 12.0701 102.583 11.9874 102.648 11.8903C102.714 11.7931 102.758 11.6836 102.779 11.5685C102.8 11.4534 102.797 11.3352 102.77 11.2214C102.743 11.1076 102.692 11.0005 102.622 10.907Z"
        fill="#489CFF"
      />
      <path
        d="M100.238 7.70295C100.073 7.49205 99.8388 7.34693 99.5767 7.29381C99.3146 7.24069 99.0422 7.28309 98.8087 7.41333L69.9239 24.0279C69.8149 24.087 69.7202 24.1693 69.6465 24.2688C69.5727 24.3684 69.5216 24.4829 69.4967 24.6044C69.4719 24.7258 69.4739 24.8512 69.5027 24.9717C69.5314 25.0923 69.5862 25.2051 69.6632 25.3022C69.8262 25.5147 70.06 25.6618 70.3221 25.7167C70.5843 25.7716 70.8574 25.7307 71.092 25.6015L99.9286 8.97728C100.041 8.92261 100.141 8.84366 100.219 8.7461C100.298 8.64854 100.354 8.53479 100.384 8.41299C100.413 8.29119 100.415 8.16436 100.39 8.04161C100.365 7.91885 100.313 7.8032 100.238 7.70295Z"
        fill="#263238"
      />
      <path
        d="M100.237 7.70295C100.073 7.49205 99.8383 7.34693 99.5762 7.29381C99.3142 7.24069 99.0418 7.28309 98.8082 7.41333L72.5879 22.4929L73.785 24.0955L99.9281 8.97728C100.041 8.92261 100.14 8.84366 100.219 8.7461C100.298 8.64854 100.354 8.53479 100.383 8.41299C100.413 8.29119 100.415 8.16436 100.39 8.04161C100.364 7.91885 100.312 7.8032 100.237 7.70295Z"
        fill="#489CFF"
      />
      <path
        d="M109.969 288.54V295.819C109.971 295.952 110.022 296.081 110.113 296.179C110.204 296.277 110.328 296.338 110.461 296.35L114.255 296.581V301.93C114.259 301.987 114.277 302.043 114.31 302.09C114.342 302.138 114.387 302.176 114.439 302.2L118.445 304.507C118.6 304.587 118.773 304.624 118.947 304.614C119.077 304.619 119.206 304.592 119.324 304.536L123.378 302.663C123.685 302.54 124.024 302.52 124.344 302.605C125.374 302.928 126.426 303.179 127.491 303.359C127.634 303.386 127.765 303.455 127.87 303.556C127.974 303.657 128.047 303.786 128.08 303.928L128.582 306.593C128.615 306.733 128.693 306.858 128.804 306.95C128.91 307.037 129.043 307.084 129.18 307.085H135.06C135.18 307.082 135.296 307.037 135.388 306.959C135.42 306.945 135.446 306.921 135.465 306.892C135.496 306.869 135.525 306.843 135.552 306.815V306.757C135.552 306.699 135.639 306.66 135.649 306.593L136.16 303.928C136.191 303.787 136.262 303.658 136.365 303.557C136.468 303.456 136.598 303.387 136.74 303.359C137.805 303.18 138.856 302.929 139.887 302.605C140.206 302.52 140.545 302.54 140.852 302.663L144.907 304.536C144.99 304.552 145.075 304.552 145.158 304.536H145.303C145.437 304.531 145.568 304.498 145.689 304.44L149.686 302.132C149.745 302.1 149.794 302.054 149.83 301.997C149.836 301.962 149.836 301.926 149.83 301.891V299.613V296.543L153.557 296.301H153.624C153.658 296.287 153.69 296.271 153.721 296.253C153.815 296.236 153.902 296.192 153.972 296.128C154.013 296.074 154.04 296.011 154.049 295.944C154.049 295.886 154.126 295.838 154.126 295.77V292.797V288.984V286.01C154.124 285.875 154.072 285.745 153.979 285.647C153.886 285.549 153.759 285.489 153.624 285.479L149.83 285.238V279.899C149.83 279.839 149.812 279.78 149.78 279.729C149.747 279.677 149.701 279.636 149.647 279.61L145.65 277.302C145.51 277.241 145.359 277.209 145.206 277.209C145.053 277.209 144.902 277.241 144.762 277.302L140.707 279.175C140.402 279.305 140.061 279.325 139.742 279.233C138.711 278.913 137.66 278.665 136.595 278.49C136.454 278.459 136.325 278.389 136.223 278.288C136.12 278.188 136.049 278.06 136.016 277.92L135.649 275.304C135.619 275.168 135.545 275.046 135.438 274.957C135.332 274.868 135.199 274.817 135.06 274.812H129.18C129.04 274.816 128.906 274.867 128.798 274.956C128.69 275.045 128.614 275.167 128.582 275.304L128.08 277.978C128.045 278.119 127.971 278.247 127.867 278.347C127.763 278.448 127.633 278.517 127.491 278.548C126.426 278.724 125.375 278.972 124.344 279.291C124.025 279.382 123.684 279.362 123.378 279.233L119.324 277.36C119.186 277.298 119.036 277.266 118.884 277.266C118.733 277.266 118.583 277.298 118.445 277.36L114.448 279.668C114.393 279.694 114.346 279.735 114.312 279.786C114.278 279.837 114.258 279.896 114.255 279.957V285.286L110.461 285.528C110.328 285.539 110.204 285.6 110.113 285.698C110.022 285.796 109.971 285.925 109.969 286.058V288.433C109.964 288.469 109.964 288.504 109.969 288.54ZM125.328 290.47C127.423 289.409 129.738 288.855 132.086 288.855C134.435 288.855 136.75 289.409 138.844 290.47C139.127 290.631 139.388 290.795 139.626 290.963C139.376 291.152 139.115 291.326 138.844 291.484C136.75 292.547 134.435 293.102 132.086 293.102C129.738 293.102 127.422 292.547 125.328 291.484C125.057 291.33 124.796 291.159 124.547 290.972C124.797 290.777 125.058 290.597 125.328 290.432V290.47Z"
        fill="#489CFF"
      />
      <path
        opacity="0.3"
        d="M109.959 288.54V295.819C109.961 295.952 110.013 296.079 110.104 296.175C110.195 296.272 110.319 296.331 110.451 296.34L114.245 296.582V301.93C114.249 301.987 114.268 302.043 114.3 302.09C114.332 302.138 114.377 302.176 114.429 302.2L118.435 304.508C118.59 304.588 118.763 304.624 118.937 304.614C119.067 304.619 119.196 304.592 119.314 304.537L123.368 302.664C123.492 302.613 123.622 302.581 123.755 302.567C123.933 302.532 124.117 302.532 124.295 302.567C125.326 302.889 126.377 303.141 127.442 303.32C127.532 303.333 127.618 303.366 127.693 303.417C127.858 303.517 127.979 303.675 128.031 303.861L128.533 306.525C128.565 306.663 128.643 306.786 128.755 306.873C128.862 306.96 128.995 307.007 129.132 307.008H135.117C135.273 306.97 135.41 306.878 135.504 306.747V306.689C135.504 306.631 135.59 306.593 135.6 306.525L136.112 303.861C136.143 303.709 136.224 303.573 136.343 303.475C136.442 303.387 136.562 303.327 136.691 303.301C137.756 303.123 138.808 302.871 139.838 302.548C140.042 302.508 140.252 302.508 140.456 302.548C140.562 302.557 140.665 302.58 140.765 302.615L144.82 304.488C144.903 304.504 144.988 304.504 145.071 304.488C145.285 304.523 145.504 304.489 145.698 304.392L149.695 302.084L149.782 302.026L149.849 301.94C149.858 301.901 149.858 301.862 149.849 301.824V296.475L150.545 296.427L153.576 296.234H153.643C153.778 296.227 153.904 296.169 153.997 296.072C154.09 295.975 154.143 295.847 154.145 295.713V288.955C154.143 289.089 154.09 289.218 153.997 289.314C153.904 289.411 153.778 289.469 153.643 289.476L148.314 289.814C148.166 289.834 148.024 289.888 147.9 289.971C147.776 290.055 147.674 290.166 147.6 290.297C147.286 290.866 146.903 291.394 146.461 291.87C146.416 291.914 146.382 291.967 146.361 292.026C146.34 292.085 146.332 292.148 146.338 292.21C146.345 292.272 146.365 292.332 146.398 292.385C146.431 292.439 146.476 292.484 146.528 292.517L149.695 294.796C149.746 294.82 149.789 294.858 149.82 294.906C149.85 294.954 149.866 295.009 149.866 295.066C149.866 295.123 149.85 295.178 149.82 295.226C149.789 295.274 149.746 295.312 149.695 295.336L145.698 297.644C145.559 297.708 145.407 297.741 145.254 297.741C145.101 297.741 144.949 297.708 144.81 297.644L140.755 295.771C140.448 295.646 140.109 295.626 139.79 295.713C138.76 296.037 137.708 296.289 136.643 296.466C136.502 296.496 136.373 296.566 136.271 296.667C136.168 296.768 136.097 296.895 136.063 297.035L135.552 299.7C135.522 299.836 135.448 299.957 135.341 300.047C135.235 300.136 135.102 300.187 134.963 300.192H129.084C128.944 300.187 128.809 300.136 128.701 300.047C128.593 299.958 128.517 299.836 128.485 299.7L127.983 297.035C127.947 296.895 127.873 296.768 127.769 296.667C127.665 296.567 127.535 296.497 127.394 296.466C126.329 296.287 125.278 296.035 124.247 295.713C123.927 295.627 123.589 295.647 123.282 295.771L119.227 297.644C119.089 297.708 118.939 297.742 118.788 297.742C118.636 297.742 118.486 297.708 118.348 297.644L114.409 295.413C114.358 295.389 114.315 295.351 114.285 295.303C114.255 295.255 114.239 295.2 114.239 295.143C114.239 295.087 114.255 295.031 114.285 294.983C114.315 294.935 114.358 294.897 114.409 294.873L117.576 292.594C117.627 292.558 117.669 292.511 117.7 292.457C117.731 292.402 117.75 292.342 117.756 292.28C117.761 292.217 117.753 292.154 117.732 292.095C117.711 292.037 117.678 291.983 117.634 291.938C117.193 291.467 116.813 290.941 116.504 290.374C116.43 290.243 116.326 290.13 116.2 290.046C116.074 289.963 115.93 289.909 115.78 289.891L110.451 289.563C110.318 289.551 110.194 289.49 110.103 289.392C110.012 289.294 109.961 289.166 109.959 289.032V288.434C109.954 288.469 109.954 288.505 109.959 288.54Z"
        fill="black"
      />
      <path
        opacity="0.3"
        d="M114.265 279.957V285.286L115.8 285.19C115.95 285.171 116.094 285.118 116.219 285.034C116.345 284.951 116.449 284.838 116.524 284.707C116.835 284.141 117.214 283.616 117.653 283.143C117.697 283.098 117.731 283.044 117.752 282.985C117.773 282.926 117.781 282.864 117.775 282.801C117.77 282.739 117.751 282.679 117.72 282.624C117.689 282.57 117.646 282.523 117.596 282.486L114.41 280.208C114.366 280.181 114.328 280.144 114.3 280.101C114.272 280.057 114.253 280.008 114.246 279.957H114.265Z"
        fill="black"
      />
      <path
        opacity="0.4"
        d="M124.556 290.933L124.894 290.692L125.29 290.45C127.384 289.389 129.699 288.835 132.048 288.835C134.396 288.835 136.711 289.389 138.805 290.45L139.163 290.663C139.307 290.759 139.452 290.846 139.587 290.952C142.484 288.78 142.252 285.652 138.805 283.664C136.713 282.596 134.397 282.039 132.048 282.039C129.698 282.039 127.382 282.596 125.29 283.664C121.882 285.633 121.621 288.819 124.556 290.933Z"
        fill="black"
      />
      <path
        opacity="0.3"
        d="M146.519 283.143C146.961 283.616 147.344 284.141 147.658 284.707C147.732 284.837 147.834 284.948 147.958 285.032C148.082 285.116 148.224 285.17 148.372 285.189L149.907 285.286V279.947C149.9 280 149.882 280.05 149.853 280.095C149.825 280.14 149.788 280.178 149.743 280.208L146.577 282.486C146.525 282.522 146.481 282.568 146.449 282.623C146.417 282.677 146.397 282.738 146.392 282.801C146.386 282.864 146.395 282.927 146.417 282.986C146.439 283.045 146.474 283.099 146.519 283.143Z"
        fill="black"
      />
      <path
        d="M144.839 277.322L140.784 279.194C140.479 279.324 140.138 279.344 139.819 279.252C138.788 278.932 137.737 278.684 136.672 278.509C136.531 278.478 136.402 278.409 136.3 278.308C136.197 278.207 136.125 278.079 136.092 277.939L135.648 275.304C135.619 275.168 135.545 275.046 135.438 274.957C135.331 274.868 135.198 274.817 135.059 274.812H129.18C129.04 274.816 128.906 274.867 128.797 274.956C128.689 275.045 128.613 275.167 128.582 275.304L128.08 277.978C128.045 278.119 127.971 278.247 127.867 278.347C127.763 278.448 127.632 278.517 127.491 278.548C126.426 278.724 125.374 278.972 124.343 279.291C124.024 279.382 123.684 279.362 123.378 279.233L119.323 277.36C119.185 277.298 119.036 277.266 118.884 277.266C118.733 277.266 118.583 277.298 118.445 277.36L114.448 279.668C114.397 279.692 114.354 279.73 114.324 279.778C114.293 279.826 114.277 279.881 114.277 279.938C114.277 279.994 114.293 280.05 114.324 280.098C114.354 280.146 114.397 280.184 114.448 280.208L117.615 282.486C117.665 282.523 117.708 282.57 117.739 282.624C117.77 282.679 117.789 282.739 117.794 282.801C117.8 282.864 117.792 282.927 117.771 282.986C117.75 283.044 117.716 283.098 117.673 283.143C117.233 283.616 116.854 284.141 116.543 284.707C116.469 284.838 116.364 284.951 116.238 285.034C116.113 285.118 115.969 285.171 115.819 285.19L110.49 285.528C110.357 285.539 110.233 285.6 110.142 285.698C110.051 285.796 109.999 285.925 109.998 286.058V289.032C109.999 289.166 110.051 289.294 110.142 289.392C110.233 289.49 110.357 289.551 110.49 289.563L115.819 289.891C115.969 289.909 116.113 289.962 116.238 290.046C116.364 290.13 116.469 290.242 116.543 290.374C116.854 290.943 117.233 291.471 117.673 291.947C117.716 291.991 117.749 292.044 117.77 292.102C117.791 292.161 117.799 292.223 117.794 292.284C117.788 292.346 117.769 292.406 117.738 292.459C117.708 292.513 117.665 292.559 117.615 292.594L114.409 294.873C114.358 294.897 114.315 294.935 114.285 294.983C114.255 295.031 114.239 295.086 114.239 295.143C114.239 295.2 114.255 295.255 114.285 295.303C114.315 295.351 114.358 295.389 114.409 295.413L118.406 297.721C118.544 297.785 118.694 297.819 118.845 297.819C118.997 297.819 119.147 297.785 119.285 297.721L123.339 295.848C123.646 295.724 123.985 295.704 124.305 295.79C125.336 296.112 126.387 296.364 127.452 296.543C127.595 296.571 127.726 296.639 127.831 296.74C127.935 296.842 128.008 296.971 128.041 297.112L128.543 299.777C128.575 299.913 128.651 300.036 128.759 300.125C128.867 300.214 129.001 300.264 129.142 300.269H135.021C135.16 300.264 135.293 300.213 135.399 300.124C135.506 300.034 135.58 299.913 135.61 299.777L136.121 297.112C136.152 296.971 136.223 296.843 136.326 296.741C136.429 296.64 136.559 296.571 136.701 296.543C137.766 296.365 138.817 296.113 139.848 295.79C140.167 295.703 140.506 295.723 140.813 295.848L144.868 297.721C145.007 297.785 145.159 297.818 145.312 297.818C145.465 297.818 145.617 297.785 145.756 297.721L149.753 295.413C149.804 295.389 149.847 295.351 149.877 295.303C149.908 295.255 149.924 295.2 149.924 295.143C149.924 295.086 149.908 295.031 149.877 294.983C149.847 294.935 149.804 294.897 149.753 294.873L146.586 292.594C146.533 292.561 146.488 292.516 146.454 292.462C146.421 292.409 146.401 292.348 146.395 292.285C146.39 292.223 146.399 292.159 146.422 292.101C146.445 292.042 146.482 291.99 146.528 291.947C146.968 291.471 147.347 290.943 147.658 290.374C147.731 290.244 147.834 290.132 147.958 290.048C148.082 289.965 148.224 289.911 148.372 289.891L153.701 289.563C153.836 289.553 153.963 289.493 154.056 289.395C154.149 289.297 154.201 289.167 154.203 289.032V286.058C154.201 285.923 154.149 285.794 154.056 285.695C153.963 285.597 153.836 285.537 153.701 285.528L148.372 285.19C148.224 285.17 148.082 285.116 147.958 285.032C147.834 284.949 147.731 284.837 147.658 284.707C147.347 284.141 146.968 283.616 146.528 283.143C146.481 283.1 146.445 283.047 146.422 282.987C146.399 282.928 146.39 282.864 146.395 282.8C146.401 282.737 146.421 282.676 146.454 282.621C146.487 282.567 146.533 282.521 146.586 282.486L149.753 280.208C149.804 280.184 149.847 280.146 149.877 280.098C149.908 280.05 149.924 279.994 149.924 279.938C149.924 279.881 149.908 279.826 149.877 279.778C149.847 279.73 149.804 279.692 149.753 279.668L145.756 277.36C145.615 277.288 145.46 277.247 145.302 277.241C145.143 277.234 144.986 277.262 144.839 277.322ZM138.844 283.645C142.58 285.798 142.58 289.293 138.844 291.445C136.75 292.509 134.434 293.063 132.086 293.063C129.737 293.063 127.422 292.509 125.328 291.445C121.592 289.293 121.592 285.798 125.328 283.645C127.422 282.582 129.737 282.027 132.086 282.027C134.434 282.027 136.75 282.582 138.844 283.645Z"
        fill="#489CFF"
      />
      <path
        d="M149.907 295.143C149.905 295.089 149.89 295.036 149.863 294.989C149.836 294.942 149.799 294.902 149.753 294.873L148.199 293.724V289.949C148.267 289.921 148.338 289.902 148.411 289.891L153.74 289.554C153.874 289.546 154.001 289.488 154.094 289.392C154.187 289.295 154.24 289.166 154.242 289.032V295.79C154.24 295.924 154.187 296.053 154.094 296.149C154.001 296.246 153.874 296.304 153.74 296.311H153.673L150.641 296.504L149.946 296.553C149.946 296.553 149.917 295.24 149.907 295.143Z"
        fill="#489CFF"
      />
      <path
        opacity="0.2"
        d="M149.907 295.143C149.905 295.089 149.89 295.036 149.863 294.989C149.836 294.942 149.799 294.902 149.753 294.873L148.199 293.724V289.949C148.267 289.921 148.338 289.902 148.411 289.891L153.74 289.554C153.874 289.546 154.001 289.488 154.094 289.392C154.187 289.295 154.24 289.166 154.242 289.032V295.79C154.24 295.924 154.187 296.053 154.094 296.149C154.001 296.246 153.874 296.304 153.74 296.311H153.673L150.641 296.504L149.946 296.553C149.946 296.553 149.917 295.24 149.907 295.143Z"
        fill="black"
      />
      <path
        d="M140.475 295.799C140.578 295.815 140.679 295.844 140.774 295.886L144.829 297.759C144.994 297.826 145.174 297.849 145.35 297.826V304.584C145.261 304.598 145.17 304.598 145.08 304.584C144.997 304.599 144.912 304.599 144.829 304.584L140.774 302.711C140.678 302.676 140.577 302.654 140.475 302.644V295.799Z"
        fill="#489CFF"
      />
      <path
        opacity="0.2"
        d="M140.475 295.799C140.578 295.815 140.679 295.844 140.774 295.886L144.829 297.759C144.994 297.826 145.174 297.849 145.35 297.826V304.584C145.261 304.598 145.17 304.598 145.08 304.584C144.997 304.599 144.912 304.599 144.829 304.584L140.774 302.711C140.678 302.676 140.577 302.654 140.475 302.644V295.799Z"
        fill="black"
      />
      <path
        d="M135.542 300.009C135.582 299.951 135.608 299.885 135.62 299.815L136.131 297.151C136.141 297.078 136.168 297.009 136.209 296.948V303.706C136.169 303.767 136.143 303.837 136.131 303.909L135.62 306.573C135.62 306.631 135.571 306.66 135.542 306.708V300.009Z"
        fill="#489CFF"
      />
      <path
        opacity="0.2"
        d="M135.542 300.009C135.582 299.951 135.608 299.885 135.62 299.815L136.131 297.151C136.141 297.078 136.168 297.009 136.209 296.948V303.706C136.169 303.767 136.143 303.837 136.131 303.909L135.62 306.573C135.62 306.631 135.571 306.66 135.542 306.708V300.009Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M127.974 296.948C128.011 297.011 128.037 297.079 128.051 297.151L128.553 299.815C128.573 299.882 128.599 299.947 128.63 300.009V306.766C128.599 306.704 128.573 306.64 128.553 306.573L128.051 303.909C128.035 303.844 128.009 303.782 127.974 303.725V296.948Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M118.812 297.826C118.988 297.849 119.168 297.826 119.333 297.759L123.388 295.886C123.487 295.844 123.59 295.815 123.696 295.799V302.557C123.59 302.565 123.486 302.591 123.388 302.634L119.333 304.507C119.215 304.563 119.086 304.589 118.956 304.584H118.812V297.826Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M110.471 289.563L115.8 289.891C115.876 289.901 115.95 289.921 116.022 289.949V293.724L114.409 294.872C114.363 294.901 114.323 294.941 114.295 294.988C114.266 295.035 114.249 295.088 114.245 295.143V296.543L110.451 296.301C110.319 296.292 110.195 296.233 110.104 296.136C110.013 296.04 109.961 295.913 109.959 295.78V288.501C109.954 288.465 109.954 288.43 109.959 288.395V288.993C109.955 289.064 109.965 289.135 109.989 289.202C110.012 289.269 110.049 289.331 110.097 289.384C110.144 289.437 110.202 289.48 110.266 289.511C110.33 289.541 110.4 289.559 110.471 289.563Z"
        fill="black"
      />
      <path
        d="M187.944 267.629V267.532V262.705C187.941 262.621 187.911 262.539 187.857 262.473L186.361 260.388C186.293 260.292 186.201 260.214 186.095 260.164C185.988 260.113 185.87 260.09 185.752 260.099L181.833 260.755C181.569 260.764 181.309 260.69 181.089 260.543C180.841 260.365 180.583 260.201 180.317 260.05V256.961C180.317 256.836 180.201 256.72 180.008 256.671L176.031 255.706C175.923 255.677 175.809 255.673 175.699 255.693C175.589 255.713 175.485 255.757 175.394 255.822L173.463 257.82C173.29 257.964 173.068 258.033 172.845 258.013C171.983 257.96 171.119 257.96 170.258 258.013C169.992 258.002 169.741 257.892 169.553 257.704L167.854 255.996C167.772 255.906 167.669 255.837 167.555 255.795C167.441 255.753 167.318 255.739 167.197 255.754L163.065 256.72C163.003 256.731 162.947 256.765 162.908 256.815C162.869 256.865 162.85 256.927 162.853 256.99V260.089C162.519 260.265 162.197 260.461 161.888 260.678C161.681 260.789 161.444 260.829 161.212 260.794L157.418 260.157C157.311 260.134 157.201 260.139 157.097 260.169C156.993 260.199 156.897 260.255 156.819 260.33L155.178 262.618C155.165 262.652 155.165 262.69 155.178 262.725C155.178 263.072 155.178 268.642 155.178 268.671C155.172 268.703 155.172 268.736 155.178 268.768V273.595C155.177 273.686 155.208 273.775 155.265 273.846L156.761 275.931C156.831 276.026 156.923 276.102 157.029 276.152C157.135 276.203 157.252 276.226 157.37 276.221L161.231 275.555C161.5 275.543 161.765 275.622 161.984 275.777C162.226 275.941 162.486 276.105 162.756 276.259V279.358C162.756 279.494 162.872 279.609 163.065 279.658L167.033 280.623C167.143 280.653 167.259 280.658 167.371 280.636C167.483 280.614 167.589 280.567 167.68 280.498L169.611 278.499C169.783 278.356 170.005 278.286 170.229 278.306C171.09 278.354 171.954 278.354 172.816 278.306C172.946 278.307 173.075 278.333 173.194 278.385C173.314 278.436 173.421 278.511 173.511 278.605L175.22 280.391C175.303 280.481 175.405 280.55 175.519 280.594C175.633 280.637 175.755 280.654 175.876 280.642L180.008 279.677C180.061 279.665 180.11 279.637 180.148 279.598C180.185 279.558 180.211 279.509 180.221 279.455C180.221 279.387 180.221 276.308 180.221 276.308C180.552 276.132 180.875 275.938 181.186 275.729C181.388 275.605 181.629 275.561 181.862 275.603L185.656 276.25C185.763 276.272 185.874 276.267 185.979 276.235C186.083 276.203 186.178 276.145 186.254 276.066L187.896 273.778C187.907 273.744 187.907 273.707 187.896 273.672C187.896 273.566 187.896 267.88 187.896 267.716L187.944 267.629ZM165.846 267.754C166.029 267.629 166.222 267.503 166.435 267.378C168.027 266.573 169.786 266.154 171.571 266.154C173.355 266.154 175.114 266.573 176.707 267.378C176.909 267.503 177.102 267.619 177.276 267.745C177.079 267.893 176.873 268.028 176.658 268.15C174.223 269.378 171.428 269.693 168.781 269.038C167.967 268.853 167.184 268.554 166.454 268.15C166.242 268.025 166.029 267.889 165.846 267.754Z"
        fill="#489CFF"
      />
      <path
        opacity="0.3"
        d="M187.944 267.629V267.532V262.705C187.941 262.621 187.911 262.539 187.857 262.473L186.361 260.388C186.293 260.292 186.201 260.214 186.095 260.164C185.988 260.113 185.87 260.09 185.752 260.099L181.833 260.755C181.569 260.764 181.309 260.69 181.089 260.543C180.841 260.365 180.583 260.201 180.317 260.05V256.961C180.317 256.836 180.201 256.72 180.008 256.671L176.031 255.706C175.923 255.677 175.809 255.673 175.699 255.693C175.589 255.713 175.485 255.757 175.394 255.822L173.463 257.82C173.29 257.964 173.068 258.033 172.845 258.013C171.983 257.96 171.119 257.96 170.258 258.013C169.992 258.002 169.741 257.892 169.553 257.704L167.854 255.996C167.772 255.906 167.669 255.837 167.555 255.795C167.441 255.753 167.318 255.739 167.197 255.754L163.065 256.72C163.003 256.731 162.947 256.765 162.908 256.815C162.869 256.865 162.85 256.927 162.853 256.99V260.089C162.519 260.265 162.197 260.461 161.888 260.678C161.681 260.789 161.444 260.829 161.212 260.794L157.418 260.157C157.311 260.134 157.201 260.139 157.097 260.169C156.993 260.199 156.897 260.255 156.819 260.33L155.178 262.618C155.165 262.652 155.165 262.69 155.178 262.725C155.178 263.072 155.178 268.642 155.178 268.671C155.172 268.703 155.172 268.736 155.178 268.768V273.595C155.177 273.686 155.208 273.775 155.265 273.846L156.761 275.931C156.831 276.026 156.923 276.102 157.029 276.152C157.135 276.203 157.252 276.226 157.37 276.221L161.231 275.555C161.5 275.543 161.765 275.622 161.984 275.777C162.226 275.941 162.486 276.105 162.756 276.259V279.358C162.756 279.494 162.872 279.609 163.065 279.658L167.033 280.623C167.143 280.653 167.259 280.658 167.371 280.636C167.483 280.614 167.589 280.567 167.68 280.498L169.611 278.499C169.783 278.356 170.005 278.286 170.229 278.306C171.09 278.354 171.954 278.354 172.816 278.306C172.946 278.307 173.075 278.333 173.194 278.385C173.314 278.436 173.421 278.511 173.511 278.605L175.22 280.391C175.303 280.481 175.405 280.55 175.519 280.594C175.633 280.637 175.755 280.654 175.876 280.642L180.008 279.677C180.061 279.665 180.11 279.637 180.148 279.598C180.185 279.558 180.211 279.509 180.221 279.455C180.221 279.387 180.221 276.308 180.221 276.308C180.552 276.132 180.875 275.938 181.186 275.729C181.388 275.605 181.629 275.561 181.862 275.603L185.656 276.25C185.763 276.272 185.874 276.267 185.979 276.235C186.083 276.203 186.178 276.145 186.254 276.066L187.896 273.778C187.907 273.744 187.907 273.707 187.896 273.672C187.896 273.566 187.896 267.88 187.896 267.716L187.944 267.629ZM165.846 267.754C166.029 267.629 166.222 267.503 166.435 267.378C168.027 266.573 169.786 266.154 171.571 266.154C173.355 266.154 175.114 266.573 176.707 267.378C176.909 267.503 177.102 267.619 177.276 267.745C177.079 267.893 176.873 268.028 176.658 268.15C174.223 269.378 171.428 269.693 168.781 269.038C167.967 268.853 167.184 268.554 166.454 268.15C166.242 268.025 166.029 267.889 165.846 267.754Z"
        fill="black"
      />
      <path
        opacity="0.3"
        d="M163.761 259.606C163.442 259.751 163.143 259.905 162.853 260.07V256.961C162.836 257.014 162.836 257.072 162.853 257.125L163.818 259.056C163.854 259.094 163.881 259.14 163.898 259.19C163.915 259.239 163.92 259.292 163.915 259.344C163.909 259.396 163.893 259.447 163.866 259.492C163.839 259.537 163.804 259.576 163.761 259.606Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M159.214 264.501C159.141 264.929 159.141 265.367 159.214 265.795C159.231 265.889 159.214 265.986 159.167 266.069C159.119 266.153 159.043 266.216 158.953 266.249L155.381 267.378C155.332 267.394 155.289 267.425 155.258 267.466C155.227 267.508 155.209 267.558 155.208 267.61C155.208 267.677 155.208 268.768 155.208 268.768V268.681C155.208 266.654 155.208 263.266 155.208 262.966V262.754C155.208 262.87 155.323 262.976 155.497 263.024L158.866 264.096C158.958 264.121 159.041 264.172 159.103 264.245C159.165 264.317 159.204 264.407 159.214 264.501Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M161.444 269.513V275.479H161.299L157.437 276.135C157.374 276.149 157.308 276.149 157.244 276.135V270.169C157.311 270.184 157.38 270.184 157.447 270.169L161.357 269.513H161.444Z"
        fill="black"
      />
      <path
        opacity="0.25"
        d="M162.824 276.271C162.554 276.116 162.293 275.952 162.052 275.788C161.869 275.669 161.661 275.593 161.444 275.566V269.513C161.665 269.536 161.877 269.612 162.062 269.735C162.565 270.078 163.094 270.381 163.645 270.642C163.877 270.758 164.002 270.961 163.925 271.115L162.853 273.239C162.848 273.271 162.848 273.304 162.853 273.336C162.834 273.49 162.824 276.271 162.824 276.271Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M169.891 272.349V278.315C169.8 278.338 169.717 278.384 169.65 278.45L167.719 280.448C167.624 280.535 167.5 280.583 167.372 280.584V274.617C167.503 274.618 167.631 274.57 167.729 274.482L169.66 272.484C169.728 272.425 169.806 272.379 169.891 272.349Z"
        fill="black"
      />
      <path
        opacity="0.25"
        d="M173.106 272.329V278.296C173.023 278.277 172.938 278.277 172.855 278.296C171.993 278.344 171.129 278.344 170.268 278.296C170.14 278.29 170.012 278.309 169.891 278.353V272.387C170.015 272.344 170.146 272.324 170.277 272.329C171.139 272.382 172.003 272.382 172.865 272.329C172.944 272.312 173.026 272.312 173.106 272.329Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M175.635 274.599V280.565C175.492 280.526 175.362 280.449 175.259 280.343L173.56 278.557C173.434 278.432 173.277 278.342 173.106 278.296V272.33C173.276 272.38 173.431 272.469 173.56 272.591L175.23 274.338C175.335 274.465 175.477 274.556 175.635 274.599Z"
        fill="black"
      />
      <path
        d="M180.26 273.422V276.27V279.417C180.25 279.471 180.224 279.52 180.187 279.56C180.149 279.599 180.1 279.627 180.047 279.639L175.915 280.604C175.824 280.633 175.726 280.633 175.635 280.604V274.638C175.73 274.666 175.83 274.666 175.925 274.638L180.057 273.673C180.116 273.663 180.169 273.633 180.206 273.586C180.244 273.54 180.263 273.481 180.26 273.422Z"
        fill="#489CFF"
      />
      <path
        opacity="0.2"
        d="M180.26 273.422V276.27V279.417C180.25 279.471 180.224 279.52 180.187 279.56C180.149 279.599 180.1 279.627 180.047 279.639L175.915 280.604C175.824 280.633 175.726 280.633 175.635 280.604V274.638C175.73 274.666 175.83 274.666 175.925 274.638L180.057 273.673C180.116 273.663 180.169 273.633 180.206 273.586C180.244 273.54 180.263 273.481 180.26 273.422Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M181.592 269.56V275.526C181.449 275.534 181.309 275.577 181.187 275.651C180.875 275.861 180.553 276.054 180.221 276.231C180.221 276.231 180.221 273.498 180.221 273.334C180.217 273.277 180.201 273.221 180.173 273.17L179.207 271.239C179.181 271.2 179.164 271.155 179.155 271.109C179.147 271.062 179.148 271.014 179.159 270.968C179.17 270.922 179.19 270.879 179.218 270.841C179.247 270.803 179.282 270.771 179.323 270.747C179.967 270.455 180.581 270.103 181.158 269.695C181.287 269.611 181.438 269.564 181.592 269.56Z"
        fill="black"
      />
      <path
        d="M187.944 267.639C187.944 267.841 187.944 273.489 187.944 273.595C187.956 273.63 187.956 273.667 187.944 273.701L186.303 275.989C186.23 276.074 186.132 276.131 186.023 276.154V270.197C186.132 270.169 186.232 270.112 186.313 270.033L187.954 267.735C187.958 267.703 187.955 267.67 187.944 267.639Z"
        fill="#489CFF"
      />
      <path
        d="M187.944 267.639C187.944 267.841 187.944 273.489 187.944 273.595C187.956 273.63 187.956 273.667 187.944 273.701L186.303 275.989C186.23 276.074 186.132 276.131 186.023 276.154V270.197C186.132 270.169 186.232 270.112 186.313 270.033L187.954 267.735C187.958 267.703 187.955 267.67 187.944 267.639Z"
        fill="#489CFF"
      />
      <path
        opacity="0.2"
        d="M187.944 267.639C187.944 267.841 187.944 273.489 187.944 273.595C187.956 273.63 187.956 273.667 187.944 273.701L186.303 275.989C186.23 276.074 186.132 276.131 186.023 276.154V270.197C186.132 270.169 186.232 270.112 186.313 270.033L187.954 267.735C187.958 267.703 187.955 267.67 187.944 267.639Z"
        fill="black"
      />
      <path
        d="M162.853 256.963C162.842 257.014 162.842 257.066 162.853 257.117L163.819 259.048C163.845 259.089 163.862 259.135 163.87 259.182C163.877 259.23 163.875 259.279 163.863 259.326C163.852 259.373 163.831 259.417 163.801 259.455C163.772 259.494 163.735 259.526 163.693 259.55C163.375 259.695 163.075 259.849 162.786 260.013C162.452 260.189 162.129 260.386 161.82 260.602C161.613 260.713 161.376 260.754 161.144 260.718L157.35 260.081C157.244 260.059 157.134 260.063 157.03 260.094C156.925 260.124 156.83 260.179 156.752 260.255L155.111 262.543C155.097 262.577 155.097 262.615 155.111 262.649C155.111 262.755 155.227 262.852 155.4 262.91L158.77 263.981C158.873 264.003 158.966 264.058 159.034 264.138C159.102 264.219 159.142 264.32 159.146 264.425C159.073 264.853 159.073 265.291 159.146 265.719C159.163 265.813 159.146 265.91 159.099 265.993C159.051 266.076 158.975 266.14 158.885 266.173L155.313 267.302C155.265 267.318 155.223 267.35 155.194 267.391C155.164 267.433 155.149 267.483 155.149 267.534C155.154 267.617 155.181 267.696 155.227 267.766L156.723 269.851C156.795 269.946 156.888 270.021 156.996 270.072C157.104 270.122 157.222 270.146 157.341 270.14L161.251 269.484C161.519 269.465 161.785 269.54 162.004 269.696C162.507 270.04 163.036 270.343 163.587 270.604C163.819 270.72 163.944 270.922 163.867 271.077L162.795 273.201C162.718 273.355 162.853 273.519 163.085 273.587L167.053 274.552C167.163 274.583 167.279 274.587 167.391 274.565C167.503 274.543 167.609 274.496 167.7 274.427L169.63 272.428C169.808 272.295 170.026 272.227 170.248 272.235C171.11 272.284 171.974 272.284 172.835 272.235C172.966 272.236 173.094 272.263 173.214 272.314C173.333 272.366 173.441 272.441 173.531 272.535L175.23 274.34C175.312 274.43 175.415 274.499 175.529 274.542C175.642 274.586 175.765 274.603 175.886 274.591L180.018 273.626C180.063 273.615 180.104 273.594 180.139 273.563C180.173 273.533 180.2 273.495 180.216 273.452C180.232 273.409 180.237 273.363 180.232 273.317C180.226 273.272 180.209 273.229 180.182 273.191L179.217 271.26C179.19 271.221 179.171 271.176 179.163 271.128C179.154 271.081 179.156 271.032 179.168 270.986C179.18 270.939 179.202 270.896 179.232 270.858C179.262 270.82 179.299 270.79 179.342 270.768C179.982 270.474 180.592 270.122 181.167 269.716C181.373 269.593 181.617 269.552 181.852 269.6L185.637 270.237C185.745 270.259 185.858 270.254 185.964 270.222C186.07 270.19 186.167 270.132 186.245 270.054L187.886 267.766C187.898 267.734 187.898 267.7 187.886 267.669V267.572C187.824 267.482 187.732 267.417 187.625 267.389L184.247 266.327C184.015 266.25 183.851 266.047 183.88 265.873C183.942 265.448 183.942 265.015 183.88 264.589C183.862 264.494 183.878 264.395 183.926 264.31C183.974 264.226 184.05 264.161 184.14 264.126L187.712 262.996C187.759 262.984 187.8 262.956 187.83 262.918C187.86 262.88 187.876 262.833 187.876 262.784C187.874 262.699 187.843 262.618 187.79 262.552L186.293 260.467C186.225 260.371 186.134 260.293 186.027 260.242C185.921 260.192 185.803 260.169 185.685 260.177L181.765 260.834C181.501 260.843 181.241 260.769 181.022 260.622C180.773 260.444 180.516 260.28 180.25 260.129C179.989 259.984 179.719 259.84 179.439 259.704C179.159 259.569 179.072 259.386 179.149 259.241L180.23 257.108C180.235 257.079 180.235 257.049 180.23 257.021C180.23 256.895 180.115 256.779 179.922 256.731L175.944 255.766C175.836 255.737 175.723 255.733 175.613 255.753C175.502 255.773 175.398 255.817 175.307 255.881L173.376 257.88C173.204 258.023 172.982 258.093 172.758 258.073C171.897 258.02 171.033 258.02 170.171 258.073C169.906 258.062 169.654 257.951 169.466 257.764L167.854 255.997C167.772 255.908 167.669 255.839 167.555 255.797C167.441 255.755 167.318 255.741 167.198 255.756L163.066 256.721C163.008 256.732 162.956 256.762 162.918 256.805C162.879 256.849 162.857 256.905 162.853 256.963ZM177.865 263.093C177.903 263.126 177.938 263.161 177.971 263.199L178.039 263.267L178.184 263.45C178.242 263.518 178.29 263.595 178.338 263.663L178.425 263.807V263.865C178.689 264.312 178.811 264.828 178.774 265.345C178.738 265.862 178.546 266.356 178.222 266.762C177.954 267.133 177.628 267.459 177.257 267.727C177.06 267.875 176.854 268.011 176.639 268.132C174.204 269.36 171.409 269.675 168.761 269.021C167.948 268.836 167.165 268.537 166.435 268.132C166.222 268.007 166.01 267.872 165.827 267.737C165.315 267.376 164.892 266.904 164.591 266.356C164.586 266.331 164.586 266.304 164.591 266.279C164.587 266.25 164.587 266.221 164.591 266.192C164.587 266.163 164.587 266.134 164.591 266.105C164.424 265.687 164.37 265.232 164.435 264.786C164.499 264.341 164.679 263.92 164.958 263.566C166.493 261.433 170.75 260.409 174.438 261.297C175.721 261.582 176.905 262.205 177.865 263.103V263.093Z"
        fill="#489CFF"
      />
      <path
        d="M210.806 249.411C209.985 249.884 208.45 249.865 207.967 250.425V252.433L217.312 251.072L220.846 253.138L219.504 255.995L210.062 255.368V257.376C210.266 257.524 210.479 257.66 210.699 257.781C212.926 258.925 215.39 259.527 217.893 259.538C220.396 259.55 222.866 258.971 225.103 257.849C227.121 256.7 228.135 255.184 228.135 253.669V251.661L227.575 251.748L227.517 251.641L262.879 231.223C266.467 232.007 270.216 231.536 273.499 229.891C274.778 229.259 275.786 228.186 276.337 226.869V224.813L266.992 228.182L263.459 226.116L264.801 223.307L274.242 221.926V219.908C274.242 219.908 263.208 219.175 259.201 221.453C258.182 221.992 257.32 222.788 256.701 223.76L256.17 223.567V225.566C256.17 226.309 256.384 227.036 256.788 227.661L221.444 248.079C217.85 247.288 214.094 247.759 210.806 249.411Z"
        fill="#37474F"
      />
      <path
        d="M262.879 231.174L262.889 229.156L227.517 249.575V251.592L262.879 231.174Z"
        fill="#37474F"
      />
      <path
        d="M256.817 225.652L221.454 246.07C217.866 245.282 214.116 245.752 210.834 247.403C209.553 248.032 208.545 249.106 207.996 250.424L217.341 249.053L220.875 251.129L219.533 253.938L210.091 255.319C210.294 255.454 210.496 255.589 210.728 255.724C212.955 256.866 215.42 257.467 217.923 257.477C220.426 257.487 222.896 256.906 225.132 255.782C228.028 254.112 228.868 251.67 227.546 249.575L262.908 229.156C266.496 229.945 270.246 229.474 273.528 227.824C274.806 227.195 275.814 226.125 276.366 224.812L267.021 226.173L263.487 224.136L264.829 221.337L274.271 219.946L273.634 219.541C271.406 218.401 268.941 217.802 266.439 217.792C263.937 217.782 261.467 218.361 259.23 219.483C256.324 221.115 255.494 223.557 256.817 225.652Z"
        fill="#455A64"
      />
      <path
        d="M264.24 224.541L263.487 224.097L264.829 221.297L274.271 219.907V221.925L264.829 223.305L264.24 224.541Z"
        fill="#263238"
      />
      <path
        d="M220.093 252.702L220.846 251.128L217.312 249.053L207.967 250.424V252.432L217.312 251.07L220.093 252.702Z"
        fill="#263238"
      />
      <path
        d="M232.44 279.165L239.835 291.966L264.771 277.572C265.795 277.06 266.441 275.863 266.441 274.116C266.34 272.302 265.814 270.537 264.905 268.964C263.996 267.391 262.73 266.054 261.209 265.06C260.679 264.676 260.057 264.439 259.405 264.374C258.754 264.31 258.097 264.42 257.502 264.693L232.44 279.165Z"
        fill="#489CFF"
      />
      <path
        d="M230.953 281.781C230.847 283.711 228.057 284.146 228.057 284.146L233.28 293.201C233.28 293.201 235.027 291.058 236.775 291.908C239.352 293.028 241.351 291.637 241.351 288.567C241.251 286.752 240.726 284.985 239.817 283.41C238.908 281.835 237.641 280.497 236.118 279.502C233.473 277.958 231.282 278.991 230.953 281.781Z"
        fill="#489CFF"
      />
      <path
        opacity="0.05"
        d="M230.953 281.781C230.847 283.711 228.057 284.146 228.057 284.146L233.28 293.201C233.28 293.201 235.027 291.058 236.775 291.908C239.352 293.028 241.351 291.637 241.351 288.567C241.251 286.752 240.726 284.985 239.817 283.41C238.908 281.835 237.641 280.497 236.118 279.502C233.473 277.958 231.282 278.991 230.953 281.781Z"
        fill="black"
      />
      <path
        opacity="0.05"
        d="M229.293 283.76C228.894 283.937 228.476 284.066 228.047 284.146L233.27 293.201C233.558 292.873 233.882 292.578 234.236 292.323C234.088 290.621 233.569 288.972 232.715 287.493C231.861 286.013 230.693 284.739 229.293 283.76Z"
        fill="black"
      />
      <path
        d="M222.642 284.784L223.076 285.546C222.921 286.073 222.847 286.619 222.854 287.168C222.955 288.982 223.481 290.747 224.39 292.32C225.299 293.893 226.565 295.23 228.086 296.224C228.556 296.509 229.067 296.717 229.602 296.841L230.037 297.594L231.784 296.629C232.316 296.251 232.738 295.737 233.006 295.142C233.274 294.546 233.379 293.89 233.309 293.241C233.209 291.427 232.685 289.663 231.777 288.09C230.87 286.517 229.606 285.18 228.086 284.185C227.558 283.801 226.937 283.565 226.287 283.5C225.638 283.436 224.982 283.546 224.389 283.818L222.642 284.784Z"
        fill="#489CFF"
      />
      <path
        d="M231.571 294.216C231.472 292.4 230.946 290.633 230.037 289.059C229.128 287.484 227.861 286.145 226.339 285.151C223.443 283.48 221.106 284.842 221.106 288.172C221.207 289.986 221.734 291.751 222.643 293.324C223.552 294.897 224.818 296.234 226.339 297.228C229.225 298.898 231.571 297.508 231.571 294.216Z"
        fill="#489CFF"
      />
      <path
        opacity="0.15"
        d="M231.571 294.216C231.472 292.4 230.946 290.633 230.037 289.059C229.128 287.484 227.861 286.145 226.339 285.151C223.443 283.48 221.106 284.842 221.106 288.172C221.207 289.986 221.734 291.751 222.643 293.324C223.552 294.897 224.818 296.234 226.339 297.228C229.225 298.898 231.571 297.508 231.571 294.216Z"
        fill="black"
      />
      <path
        opacity="0.15"
        d="M224.302 295.577C224.7 295.999 225.132 296.386 225.595 296.735L226.821 293.25L225.344 292.159L224.302 295.577Z"
        fill="black"
      />
      <path
        opacity="0.2"
        d="M262.184 270.476C262.364 270.804 262.409 271.188 262.31 271.549C262.211 271.909 261.975 272.217 261.653 272.407L243.841 282.688C243.514 282.868 243.129 282.913 242.768 282.814C242.408 282.715 242.1 282.479 241.911 282.157C241.731 281.83 241.686 281.445 241.785 281.084C241.884 280.724 242.119 280.416 242.442 280.226L260.186 269.964C260.35 269.864 260.532 269.797 260.722 269.769C260.913 269.74 261.107 269.75 261.293 269.798C261.479 269.846 261.654 269.93 261.807 270.047C261.96 270.163 262.088 270.309 262.184 270.476Z"
        fill="black"
      />
      <path
        opacity="0.2"
        d="M258.97 266.663C259.1 266.741 259.206 266.853 259.278 266.986C259.35 267.12 259.384 267.271 259.377 267.422C259.37 267.573 259.322 267.72 259.239 267.847C259.155 267.973 259.039 268.075 258.902 268.14L240.762 278.615C240.426 278.78 240.05 278.846 239.678 278.807C239.306 278.768 238.952 278.624 238.658 278.393C238.528 278.314 238.422 278.201 238.351 278.067C238.279 277.932 238.245 277.781 238.252 277.63C238.259 277.478 238.307 277.33 238.39 277.203C238.473 277.076 238.589 276.973 238.725 276.906L256.875 266.432C257.209 266.263 257.585 266.195 257.957 266.236C258.329 266.277 258.681 266.426 258.97 266.663Z"
        fill="black"
      />
      <path
        opacity="0.2"
        d="M263.893 275.197C263.891 275.045 263.849 274.896 263.769 274.766C263.69 274.637 263.577 274.531 263.442 274.461C263.307 274.39 263.156 274.358 263.004 274.367C262.852 274.376 262.706 274.425 262.58 274.511L244.43 284.957C244.122 285.167 243.879 285.459 243.728 285.8C243.578 286.142 243.527 286.518 243.581 286.888C243.582 287.04 243.625 287.188 243.705 287.318C243.784 287.448 243.897 287.553 244.032 287.624C244.167 287.694 244.318 287.726 244.47 287.718C244.622 287.709 244.768 287.659 244.894 287.573L263.082 277.127C263.384 276.914 263.62 276.619 263.764 276.278C263.907 275.937 263.952 275.562 263.893 275.197Z"
        fill="black"
      />
      <path
        d="M180.473 315.338V318.824L189.161 315.995C189.161 315.995 190.909 312.79 192.646 311.776L225.808 292.68L226.822 289.784L225.808 289.195L192.675 308.32C192.13 308.598 191.52 308.728 190.908 308.696C190.296 308.664 189.704 308.471 189.19 308.137L180.473 315.338Z"
        fill="#455A64"
      />
      <path
        d="M181.496 315.927V319.412L190.184 316.584C190.184 316.584 191.932 313.379 193.669 312.375L226.792 293.25V289.784L193.669 308.909C193.124 309.189 192.514 309.32 191.902 309.288C191.29 309.256 190.697 309.061 190.184 308.725L181.496 315.927Z"
        fill="#37474F"
      />
      <path
        d="M181.496 319.412L180.473 318.823V315.338L181.496 315.927V319.412Z"
        fill="#263238"
      />
      <path
        d="M180.473 315.337L181.496 315.926L190.214 308.695L189.19 308.106L180.473 315.337Z"
        fill="#455A64"
      />
      <path
        d="M327.338 92.8409C332.976 87.0485 343.181 73.89 343.181 73.89L348.973 53.8965C349.118 53.9544 346.029 53.9641 345.111 53.8965C338.875 54.3116 337.041 54.8619 333.16 59.0904C330.08 62.4596 318.331 78.5143 314.238 83.9109L303.396 70.096C300.799 66.1668 301.224 61.7742 299.873 58.5691C298.521 55.3639 296.349 53.9448 294.602 51.0389C292.854 48.133 292.439 46.2119 291.01 47.7276C289.581 49.2433 291.232 53.2111 291.445 54.1186C291.657 55.026 293.839 57.7388 290.19 56.3293C286.54 54.9198 280.902 48.2489 279.445 46.878C277.987 45.5072 277.224 47.631 277.514 49.3012C277.804 50.9713 276.278 54.7364 276.848 57.4975C277.446 60.3937 277.881 61.7742 280.709 64.6221C283.335 67.277 289.784 70.5208 292.294 74.0059C294.804 77.491 300.191 88.3035 303.483 94.82C307.654 103.103 311.206 105.835 315.918 103.045C320.629 100.255 325.243 94.9937 327.338 92.8409Z"
        fill="#FFA8A7"
      />
      <path
        d="M347.119 53.7035C343.928 53.3982 340.709 53.8377 337.716 54.9875C334.328 56.5129 332.03 59.8145 328.728 64.2168C325.427 68.619 317.144 79.3253 317.144 79.3253C322.085 82.6806 325.655 87.7002 327.203 93.4685L347.38 71.1677L347.119 53.7035Z"
        fill="#A9B6BE"
      />
      <path
        d="M314.247 83.9213C316.282 85.8704 317.636 88.4225 318.109 91.2004C318.109 91.2004 319.325 87.194 315.106 82.6855L314.247 83.9213Z"
        fill="#F28F8F"
      />
      <path
        d="M366.534 53.6358C368.725 54.1668 374.614 54.9198 378.495 56.03C381.391 56.8506 384.287 60.4033 384.847 63.0292C386.778 71.4379 385.06 76.7283 384.345 84.0943C383.631 91.4603 383.583 128.213 383.583 128.213C379.499 136.187 348.558 137.809 333.584 126.871C333.584 126.871 335.419 77.9447 335.419 70.3276C335.419 62.7106 339.647 53.5682 351.329 52.8828L366.534 53.6358Z"
        fill="#D9E4EA"
      />
      <path
        d="M353.308 29.5402C353.319 29.8586 353.424 30.1667 353.61 30.4259C353.795 30.6851 354.053 30.8839 354.35 30.9975C354.648 31.1111 354.973 31.1345 355.284 31.0648C355.595 30.995 355.878 30.8352 356.099 30.6053C356.32 30.3754 356.468 30.0856 356.525 29.772C356.581 29.4585 356.545 29.1351 356.419 28.8423C356.293 28.5494 356.084 28.3002 355.817 28.1255C355.551 27.9509 355.239 27.8587 354.92 27.8604C354.484 27.8705 354.07 28.0529 353.767 28.3676C353.465 28.6824 353.3 29.104 353.308 29.5402Z"
        fill="#263238"
      />
      <path
        d="M354.92 24.0748L358.26 25.9187C358.499 25.462 358.551 24.9308 358.407 24.4362C358.263 23.9415 357.934 23.5215 357.488 23.2639C357.267 23.1428 357.024 23.0677 356.774 23.0431C356.523 23.0185 356.271 23.0449 356.031 23.1207C355.791 23.1965 355.569 23.3201 355.378 23.484C355.187 23.648 355.031 23.849 354.92 24.0748Z"
        fill="#263238"
      />
      <path
        d="M353.714 39.4062L349.968 40.8061C350.044 41.0602 350.173 41.2957 350.344 41.498C350.516 41.7003 350.728 41.865 350.966 41.9819C351.204 42.0987 351.464 42.1652 351.729 42.1772C351.994 42.1892 352.259 42.1464 352.507 42.0515C353.011 41.8524 353.419 41.4656 353.644 40.9723C353.869 40.479 353.894 39.9176 353.714 39.4062Z"
        fill="#F28F8F"
      />
      <path
        d="M340.41 25.4178L343.451 23.0719C343.308 22.8662 343.125 22.6917 342.913 22.5591C342.7 22.4265 342.463 22.3385 342.216 22.3006C341.968 22.2626 341.716 22.2755 341.473 22.3384C341.231 22.4013 341.004 22.5129 340.806 22.6664C340.4 22.9868 340.132 23.451 340.059 23.9631C339.985 24.4751 340.111 24.9959 340.41 25.4178Z"
        fill="#263238"
      />
      <path
        d="M341.095 29.5398C341.107 29.8464 341.208 30.1429 341.387 30.3924C341.565 30.6418 341.813 30.8333 342.1 30.9428C342.387 31.0523 342.699 31.0752 342.999 31.0084C343.298 30.9417 343.571 30.7884 343.784 30.5675C343.997 30.3467 344.14 30.068 344.196 29.7664C344.252 29.4647 344.218 29.1532 344.098 28.8708C343.978 28.5884 343.778 28.3475 343.522 28.1781C343.266 28.0086 342.966 27.9182 342.659 27.918C342.45 27.9217 342.244 27.9667 342.052 28.0502C341.86 28.1338 341.687 28.2543 341.541 28.4049C341.396 28.5555 341.282 28.7333 341.206 28.928C341.129 29.1228 341.092 29.3306 341.095 29.5398Z"
        fill="#263238"
      />
      <path
        d="M349.562 26.8467L349.031 36.6745L343.895 34.9271L349.562 26.8467Z"
        fill="#F28F8F"
      />
      <path
        d="M353.327 27.2802L376.313 25.5811C380.764 28.9503 389.028 36.0364 389.018 37.9961V39.3574C389.014 39.4179 388.997 39.4769 388.97 39.5311C388.159 41.0758 385.224 42.0315 380.658 42.1088C375.242 42.1956 366.833 41.7033 364.622 39.8014C361.301 36.9052 353.327 28.5738 353.327 28.5738L354.486 28.4869L353.327 27.2802Z"
        fill="#489CFF"
      />
      <path
        opacity="0.2"
        d="M353.327 27.2802L376.313 25.5811C380.764 28.9503 389.028 36.0364 389.018 37.9961V39.3574C389.014 39.4179 388.997 39.4769 388.97 39.5311C388.159 41.0758 385.224 42.0315 380.658 42.1088C375.242 42.1956 366.833 41.7033 364.622 39.8014C361.301 36.9052 353.327 28.5738 353.327 28.5738L354.486 28.4869L353.327 27.2802Z"
        fill="black"
      />
      <path
        d="M376.313 26.875C380.957 30.3987 389.829 37.9771 388.97 39.5314C388.111 41.0857 385.224 42.0318 380.658 42.1091C375.242 42.196 366.833 41.7036 364.622 39.8018C361.301 36.9055 353.327 28.5741 353.327 28.5741L376.313 26.875Z"
        fill="#489CFF"
      />
      <path
        opacity="0.4"
        d="M376.313 26.875C380.957 30.3987 389.829 37.9771 388.97 39.5314C388.111 41.0857 385.224 42.0318 380.658 42.1091C375.242 42.196 366.833 41.7036 364.622 39.8018C361.301 36.9055 353.327 28.5741 353.327 28.5741L376.313 26.875Z"
        fill="black"
      />
      <path
        d="M376.313 26.8736C376.39 27.5397 374.044 28.9299 374.044 28.9299L374.701 24.8945L376.313 26.8736Z"
        fill="#489CFF"
      />
      <path
        opacity="0.3"
        d="M376.313 26.8736C376.39 27.5397 374.044 28.9299 374.044 28.9299L374.701 24.8945L376.313 26.8736Z"
        fill="black"
      />
      <path
        d="M370.241 13.4453C370.241 13.4453 373.832 13.9473 375.155 16.6505C376.294 19.006 375.888 24.6923 374.353 30.8902C373.484 34.8789 372.19 38.7627 370.492 42.475C369.894 43.5337 368.913 44.3219 367.75 44.6761L367.933 37.3197L368.204 32.2031C368.204 32.2031 364.912 27.714 364.554 24.0358C364.081 19.2957 365.085 17.3649 365.085 17.3649L370.241 13.4453Z"
        fill="#263238"
      />
      <path
        d="M364.448 20.5617C364.435 24.2009 362.978 27.6862 360.397 30.2513C357.815 32.8165 354.321 34.2517 350.682 34.2414C343.103 34.2414 337.417 27.9953 337.446 20.4169C337.475 12.8384 343.19 6.76606 350.768 6.79503C354.408 6.80781 357.893 8.26509 360.458 10.8466C363.023 13.428 364.459 16.9224 364.448 20.5617Z"
        fill="#263238"
      />
      <path
        d="M368.03 31.4318C369.266 32.5034 370.27 29.7327 371.312 28.6514C372.355 27.5702 375.744 26.1027 377.414 29.6651C379.084 33.2274 375.937 38.3537 373.33 39.3867C372.549 39.6957 371.686 39.7302 370.883 39.4847C370.08 39.2391 369.384 38.728 368.909 38.0351V54.3215C365.047 61.3593 358.115 61.1469 354.428 60.7414C350.74 60.336 349.938 56.5226 352.632 53.4333V48.3939C350.586 48.7143 348.511 48.8146 346.444 48.6932C343.074 48.1622 341.308 45.6136 340.304 41.9354C338.672 36.0657 338.025 26.1993 340.304 14.7979C344.078 11.9017 357.179 12.3554 365.105 17.3852C364.535 27.5219 366.804 30.3602 368.03 31.4318Z"
        fill="#FFA8A7"
      />
      <path
        d="M341.723 9.81708C343.286 8.15082 345.185 6.83471 347.293 5.95547C348.988 5.22949 350.808 4.84247 352.651 4.81629C355.045 4.81629 357.478 4.45909 359.824 4.53633C363.975 4.67148 368.996 6.36094 370.868 10.3963C371.728 12.2499 371.834 14.2 370.173 15.4743C368.661 16.504 366.928 17.1652 365.115 17.4052C362.035 17.9361 358.984 18.6119 355.885 19.027C352.574 19.5875 349.187 19.5385 345.893 18.8822C344.345 18.5319 342.873 17.9039 341.549 17.0286C340.872 16.5747 340.244 16.0505 339.676 15.4647C339.454 15.2427 338.865 14.7793 338.798 14.4993C338.73 14.2193 339.126 13.5339 339.29 13.2829C339.937 12.0206 340.755 10.8542 341.723 9.81708Z"
        fill="#263238"
      />
      <path
        d="M352.661 48.4328C352.661 48.4328 360.201 46.9461 362.836 45.5366C364.386 44.7332 365.658 43.4818 366.486 41.9453C366.083 43.482 365.377 44.9229 364.41 46.1834C362.479 48.6356 352.651 50.4216 352.651 50.4216L352.661 48.4328Z"
        fill="#F28F8F"
      />
      <path
        d="M354.698 28.894C354.709 29.2063 354.812 29.5084 354.994 29.7626C355.176 30.0169 355.429 30.212 355.72 30.3237C356.012 30.4354 356.331 30.4589 356.636 30.391C356.941 30.3232 357.219 30.167 357.436 29.9421C357.653 29.7172 357.799 29.4335 357.856 29.1262C357.913 28.8188 357.878 28.5016 357.756 28.2139C357.634 27.9262 357.43 27.6807 357.17 27.5081C356.909 27.3355 356.603 27.2434 356.291 27.2432C356.078 27.2469 355.868 27.2926 355.672 27.3776C355.477 27.4626 355.3 27.5853 355.152 27.7386C355.005 27.8919 354.888 28.0728 354.81 28.2711C354.732 28.4693 354.694 28.681 354.698 28.894Z"
        fill="#263238"
      />
      <path
        d="M351.068 41.1137C351.08 41.4261 351.183 41.7281 351.364 41.9824C351.546 42.2366 351.799 42.4317 352.091 42.5434C352.383 42.6552 352.701 42.6786 353.006 42.6107C353.311 42.5429 353.589 42.3868 353.806 42.1619C354.024 41.937 354.17 41.6532 354.226 41.3459C354.283 41.0386 354.249 40.7213 354.127 40.4336C354.004 40.1459 353.8 39.9004 353.54 39.7278C353.279 39.5552 352.974 39.4631 352.661 39.4629C352.448 39.4654 352.237 39.5102 352.041 39.5948C351.845 39.6793 351.668 39.802 351.52 39.9555C351.372 40.1091 351.256 40.2905 351.178 40.4894C351.1 40.6882 351.063 40.9004 351.068 41.1137Z"
        fill="#F28F8F"
      />
      <path
        d="M356.87 22.7429L359.93 25.1371C360.239 24.7217 360.378 24.2051 360.321 23.691C360.263 23.177 360.014 22.7038 359.622 22.3664C359.418 22.2062 359.184 22.0886 358.934 22.0207C358.684 21.9527 358.423 21.9358 358.166 21.971C357.909 22.0061 357.662 22.0925 357.44 22.2252C357.217 22.3578 357.023 22.5339 356.87 22.7429Z"
        fill="#263238"
      />
      <path
        d="M340.613 25.1853L343.567 22.6173C343.406 22.4211 343.207 22.2587 342.983 22.1393C342.758 22.02 342.513 21.9461 342.26 21.9219C342.007 21.8977 341.752 21.9236 341.509 21.9982C341.266 22.0728 341.04 22.1946 340.844 22.3566C340.452 22.7093 340.209 23.1989 340.166 23.7249C340.123 24.2509 340.283 24.7734 340.613 25.1853Z"
        fill="#263238"
      />
      <path
        d="M341.617 28.4204C341.628 28.7327 341.731 29.0348 341.913 29.289C342.095 29.5432 342.347 29.7384 342.639 29.8501C342.931 29.9618 343.25 29.9852 343.555 29.9174C343.86 29.8495 344.138 29.6934 344.355 29.4685C344.572 29.2436 344.718 28.9598 344.775 28.6525C344.832 28.3452 344.797 28.0279 344.675 27.7402C344.553 27.4525 344.349 27.2071 344.089 27.0345C343.828 26.8619 343.522 26.7697 343.21 26.7695C342.997 26.7733 342.787 26.819 342.591 26.904C342.396 26.989 342.219 27.1117 342.071 27.265C341.924 27.4183 341.807 27.5992 341.729 27.7975C341.651 27.9957 341.613 28.2074 341.617 28.4204Z"
        fill="#263238"
      />
      <path
        d="M350.344 25.6973L349.745 36.9732L344.397 35.5637L350.344 25.6973Z"
        fill="#F28F8F"
      />
      <path
        d="M377.298 21.2855C377.256 23.2242 376.93 25.1463 376.333 26.991C375.947 26.3912 375.472 25.854 374.923 25.3981C371.852 22.8243 368.348 20.8172 364.574 19.4706C355.885 16.111 345.749 15.6089 337.649 18.4472C337.788 16.7966 338.152 15.1726 338.73 13.6202C341.684 5.89699 347.66 1.89056 357.198 2.88493C368.368 4.0241 377.597 10.222 377.298 21.2855Z"
        fill="#489CFF"
      />
      <path
        d="M356.822 14.48C354.225 13.6015 347.602 13.0898 345.671 13.698C344.775 12.9599 343.991 12.0948 343.345 11.13C342.437 9.69158 342.746 7.88628 343.596 7.50977C344.445 7.13326 344.503 8.26279 347.795 9.89432C347.361 8.83927 347.266 7.67497 347.525 6.56367C348.066 5.23142 349.195 7.62562 351.647 9.17991C354.099 10.7342 356.513 11.5065 356.822 14.48Z"
        fill="#263238"
      />
      <path
        d="M390.708 47.1975C392.87 45.4211 405.98 60.9545 400.497 65.4243C395.013 69.8941 388.313 58.6664 389.221 53.125C390.128 47.5836 390.708 47.1975 390.708 47.1975Z"
        fill="#37474F"
      />
      <path
        d="M401.25 38.4693C397.967 34.4436 389.25 24.7702 387.319 23.0229C385.388 21.2755 384.23 21.5072 383.033 21.092C381.692 20.6112 380.47 19.8473 379.451 18.8523C378.834 18.2125 378.257 17.5354 377.723 16.825C375.85 14.5106 373.655 12.4756 371.206 10.7815C368.619 8.99555 365.781 7.67295 363.136 5.95453C361.552 4.94085 359.834 3.23209 361.63 1.48471C362.174 0.98046 362.839 0.624732 363.56 0.451732C368.05 -0.726061 372.066 0.664117 376.545 1.61021C378.494 2.01648 380.408 2.57491 382.27 3.28036C383.184 3.59556 384.046 4.04452 384.828 4.61262C386.345 5.90848 387.749 7.3294 389.028 8.86039C392.107 12.6351 394.714 13.8033 397.716 16.4195C401.259 19.4798 417.14 30.7075 420.49 34.0478C423.271 36.8185 422.421 39.1065 422.054 39.8981C420.075 44.4452 417.72 46.1346 414.939 49.0115C414.022 49.9769 413.095 50.9423 412.169 51.9077C408.558 55.5666 404.928 59.0131 401.163 62.5272C397.147 66.2729 395.506 68.0686 391.924 71.6792C389.68 74.0128 387.851 76.713 386.518 79.6631C380.04 81.1401 375.618 67.4314 377.935 59.4861C378.648 57.4858 379.964 55.7556 381.7 54.5336C388.304 49.2625 401.25 38.4693 401.25 38.4693Z"
        fill="#FFA8A7"
      />
      <path
        d="M401.249 38.4697C403.438 40.9389 406.29 42.7281 409.465 43.625C409.465 43.625 405.343 44.8607 399.425 39.9854L401.249 38.4697Z"
        fill="#F28F8F"
      />
      <path
        d="M378.495 56.0307C379.972 54.8722 383.564 52.4008 390.707 47.1973C390.707 47.1973 389.742 49.6687 391.876 55.5963C394.009 61.5238 398.074 65.1345 400.497 65.4241C400.497 65.4241 396.529 68.9575 392.918 72.8963C389.308 76.8351 386.305 79.6541 384.345 84.095C384.345 84.095 378.775 83.0137 376.989 73.8038C375.724 67.3356 375.792 59.6606 378.495 56.0307Z"
        fill="#A9B6BE"
      />
      <path
        d="M366.012 291.503C365.665 291.869 365.887 294.534 366.321 295.142C366.756 295.75 369.217 297.778 373.697 297.855C377.993 297.932 381.787 297.15 384.22 295.412C386.653 293.675 387.763 291.86 387.869 289.408C387.975 286.956 387.579 284.581 388.641 282.65C389.607 280.864 390.958 279.097 391.287 278.132C391.664 276.405 391.664 274.617 391.287 272.89L366.012 291.503Z"
        fill="#263238"
      />
      <path
        d="M376.439 270.823C376.426 271.749 376.3 272.67 376.063 273.565C375.51 275.487 374.726 277.335 373.726 279.067C373.24 279.976 372.692 280.85 372.085 281.684C370.869 283.296 369.324 284.647 368.04 286.192C366.653 287.664 365.817 289.571 365.675 291.589C365.675 294.958 369.382 295.904 372.211 296.252C375.276 296.615 378.384 296.285 381.305 295.286C382.843 294.798 384.208 293.875 385.235 292.631C386.263 291.386 386.909 289.871 387.097 288.268C387.203 287.302 387.097 286.337 387.155 285.429C387.412 283.267 388.198 281.201 389.443 279.415C390.171 278.383 390.786 277.276 391.277 276.113C392.04 273.989 391.219 271.682 390.544 269.645C389.964 267.878 389.501 265.783 388.883 265.957C388.883 266.324 388.883 267.106 388.883 267.106C388.69 267.512 388.246 267.801 388.14 268.255C388.039 268.737 387.894 269.209 387.705 269.664C387.43 270.372 386.941 270.977 386.306 271.392C386.19 270.128 386.083 268.863 385.968 267.598C385.981 267.203 385.859 266.814 385.62 266.498C385.311 266.24 384.922 266.097 384.519 266.092C382.14 265.656 379.709 265.584 377.308 265.88C377.123 265.886 376.941 265.931 376.774 266.012C376.608 266.094 376.461 266.21 376.343 266.353C376.214 266.624 376.177 266.928 376.236 267.222C376.366 268.418 376.434 269.62 376.439 270.823Z"
        fill="#489CFF"
      />
      <path
        opacity="0.2"
        d="M376.439 270.823C376.426 271.749 376.3 272.67 376.063 273.565C375.51 275.487 374.726 277.335 373.726 279.067C373.24 279.976 372.692 280.85 372.085 281.684C370.869 283.296 369.324 284.647 368.04 286.192C366.653 287.664 365.817 289.571 365.675 291.589C365.675 294.958 369.382 295.904 372.211 296.252C375.276 296.615 378.384 296.285 381.305 295.286C382.843 294.798 384.208 293.875 385.235 292.631C386.263 291.386 386.909 289.871 387.097 288.268C387.203 287.302 387.097 286.337 387.155 285.429C387.412 283.267 388.198 281.201 389.443 279.415C390.171 278.383 390.786 277.276 391.277 276.113C392.04 273.989 391.219 271.682 390.544 269.645C389.964 267.878 389.501 265.783 388.883 265.957C388.883 266.324 388.883 267.106 388.883 267.106C388.69 267.512 388.246 267.801 388.14 268.255C388.039 268.737 387.894 269.209 387.705 269.664C387.43 270.372 386.941 270.977 386.306 271.392C386.19 270.128 386.083 268.863 385.968 267.598C385.981 267.203 385.859 266.814 385.62 266.498C385.311 266.24 384.922 266.097 384.519 266.092C382.14 265.656 379.709 265.584 377.308 265.88C377.123 265.886 376.941 265.931 376.774 266.012C376.608 266.094 376.461 266.21 376.343 266.353C376.214 266.624 376.177 266.928 376.236 267.222C376.366 268.418 376.434 269.62 376.439 270.823Z"
        fill="black"
      />
      <path
        d="M373.311 279.801C374.517 278.836 377.317 278.633 378.852 278.71C380.417 278.765 381.943 279.21 383.293 280.004C383.49 280.137 383.728 280.192 383.963 280.16C384.198 280.128 384.413 280.011 384.567 279.83C384.655 279.728 384.721 279.609 384.76 279.48C384.799 279.352 384.811 279.216 384.795 279.082C384.778 278.949 384.734 278.82 384.665 278.704C384.596 278.589 384.503 278.489 384.394 278.411C382.909 277.433 381.179 276.894 379.402 276.857C375.618 276.799 374.575 277.504 374.575 277.504C374.575 277.504 372.886 278.701 373.311 279.801Z"
        fill="#455A64"
      />
      <path
        d="M370.627 283.392C372.143 282.378 375.232 282.349 376.777 282.426C378.314 282.502 379.805 282.978 381.102 283.807C381.297 283.941 381.535 283.998 381.77 283.967C382.005 283.937 382.221 283.822 382.376 283.643C382.466 283.541 382.533 283.421 382.573 283.291C382.613 283.161 382.625 283.024 382.609 282.889C382.592 282.755 382.547 282.625 382.477 282.508C382.407 282.392 382.314 282.292 382.202 282.214C380.69 281.188 378.921 280.603 377.095 280.524C373.311 280.466 372.365 281.287 372.365 281.287C371.935 281.484 371.555 281.775 371.254 282.14C370.953 282.504 370.739 282.932 370.627 283.392Z"
        fill="#455A64"
      />
      <path
        d="M380.706 273.169C379.105 272.977 377.482 273.203 375.995 273.826C375.348 274.231 374.962 275.303 375.406 275.554C376.707 274.942 378.14 274.664 379.576 274.743C380.995 274.809 382.392 275.116 383.708 275.65C384.073 275.789 384.431 275.947 384.78 276.123C384.889 276.176 385.008 276.206 385.129 276.212C385.25 276.218 385.371 276.199 385.485 276.157C385.599 276.114 385.703 276.049 385.79 275.965C385.878 275.882 385.948 275.781 385.996 275.67C386.075 275.476 386.08 275.26 386.008 275.064C385.937 274.868 385.795 274.705 385.61 274.608C384.096 273.785 382.424 273.294 380.706 273.169Z"
        fill="#455A64"
      />
      <path
        d="M313.958 278.025C313.687 279.022 313.804 280.084 314.286 280.998C314.75 281.848 319.461 284.184 325.591 283.431C330.143 282.868 334.46 281.094 338.093 278.295C340.362 276.615 342.785 276.19 345.817 275.853C348.848 275.515 351.146 274.453 351.734 272.956C352.323 271.46 351.426 269.095 351.426 269.095L313.958 278.025Z"
        fill="#263238"
      />
      <path
        d="M337.871 258.504C337.318 258.385 336.773 258.23 336.239 258.041C335.949 257.917 335.625 257.896 335.322 257.983C335.146 258.101 335 258.258 334.896 258.443C334.792 258.627 334.733 258.833 334.724 259.045C334.627 259.963 334.449 260.871 334.193 261.758C333.777 262.499 333.223 263.155 332.561 263.689C331.101 265.002 329.505 266.158 327.802 267.135C325.871 268.323 323.94 269.269 321.893 270.321C319.847 271.373 317.066 272.117 315.444 273.323C313.118 275.08 313.021 279.193 315.86 280.66C318.312 281.935 324.645 283.103 331.103 280.892C334.627 279.695 338.382 275.66 343.055 275.042C346.019 274.646 349.977 274.183 351.744 271.817C352.39 270.775 351.821 267.869 351.29 264.422C350.817 261.333 350.382 256.506 349.359 256.825C349.359 256.825 348.992 257.964 348.925 258.089C348.679 258.476 348.404 258.844 348.104 259.19C347.839 259.469 347.558 259.733 347.264 259.981C346.936 260.363 346.584 260.724 346.212 261.063C345.729 261.477 345.127 261.727 344.493 261.777C344.196 261.795 343.899 261.731 343.634 261.594C343.123 261.304 343.055 260.628 342.669 260.223C342.195 259.753 341.63 259.385 341.008 259.142C340.143 258.827 339.235 258.645 338.315 258.601L337.871 258.504Z"
        fill="#489CFF"
      />
      <path
        opacity="0.2"
        d="M337.871 258.504C337.318 258.385 336.773 258.23 336.239 258.041C335.949 257.917 335.625 257.896 335.322 257.983C335.146 258.101 335 258.258 334.896 258.443C334.792 258.627 334.733 258.833 334.724 259.045C334.627 259.963 334.449 260.871 334.193 261.758C333.777 262.499 333.223 263.155 332.561 263.689C331.101 265.002 329.505 266.158 327.802 267.135C325.871 268.323 323.94 269.269 321.893 270.321C319.847 271.373 317.066 272.117 315.444 273.323C313.118 275.08 313.021 279.193 315.86 280.66C318.312 281.935 324.645 283.103 331.103 280.892C334.627 279.695 338.382 275.66 343.055 275.042C346.019 274.646 349.977 274.183 351.744 271.817C352.39 270.775 351.821 267.869 351.29 264.422C350.817 261.333 350.382 256.506 349.359 256.825C349.359 256.825 348.992 257.964 348.925 258.089C348.679 258.476 348.404 258.844 348.104 259.19C347.839 259.469 347.558 259.733 347.264 259.981C346.936 260.363 346.584 260.724 346.212 261.063C345.729 261.477 345.127 261.727 344.493 261.777C344.196 261.795 343.899 261.731 343.634 261.594C343.123 261.304 343.055 260.628 342.669 260.223C342.195 259.753 341.63 259.385 341.008 259.142C340.143 258.827 339.235 258.645 338.315 258.601L337.871 258.504Z"
        fill="black"
      />
      <path
        d="M330.496 265.338C330.682 264.915 330.971 264.544 331.337 264.261C331.703 263.978 332.134 263.79 332.591 263.717C333.932 263.388 336.385 264.257 337.939 265.406C338.087 265.512 338.207 265.653 338.288 265.817C338.369 265.98 338.408 266.161 338.402 266.343C338.396 266.526 338.346 266.704 338.255 266.862C338.164 267.02 338.036 267.153 337.881 267.25C337.688 267.372 337.461 267.432 337.233 267.42C337.005 267.408 336.786 267.325 336.607 267.182C334.887 265.796 332.695 265.134 330.496 265.338Z"
        fill="#455A64"
      />
      <path
        d="M326.421 267.917C326.693 267.483 327.067 267.122 327.509 266.863C327.95 266.604 328.448 266.456 328.96 266.43C328.96 266.43 332.503 266.343 334.81 268.265C334.951 268.37 335.063 268.507 335.139 268.665C335.215 268.823 335.252 268.997 335.246 269.173C335.24 269.348 335.193 269.519 335.107 269.672C335.021 269.825 334.899 269.955 334.752 270.051C334.566 270.169 334.347 270.226 334.127 270.212C333.906 270.198 333.696 270.114 333.526 269.973C332.57 269.123 331.436 268.497 330.206 268.141C328.977 267.786 327.684 267.709 326.421 267.917Z"
        fill="#455A64"
      />
      <path
        d="M321.894 270.292C322.21 269.888 322.611 269.557 323.067 269.322C323.524 269.087 324.026 268.954 324.539 268.931C326.56 268.944 328.524 269.6 330.148 270.804C330.288 270.909 330.401 271.046 330.477 271.204C330.553 271.362 330.589 271.536 330.584 271.712C330.578 271.887 330.53 272.058 330.444 272.211C330.358 272.364 330.237 272.494 330.09 272.59C329.902 272.703 329.684 272.756 329.465 272.742C329.246 272.728 329.036 272.648 328.864 272.512C327.956 271.623 326.853 270.959 325.643 270.574C324.432 270.188 323.148 270.092 321.894 270.292Z"
        fill="#455A64"
      />
      <path
        d="M331.789 125.23C331.625 139.334 332.532 186.91 332.841 194.005C333.111 200.222 334.193 261.41 334.193 261.41C341.317 263.62 351.068 261.332 351.068 261.332C351.068 261.332 356.03 223.17 356.349 216.412C356.528 210.192 356.205 203.968 355.383 197.799L359.032 158.865C359.032 158.865 363.02 188.02 364.941 200.782C367.142 215.408 375.377 271.421 375.377 271.421C381.932 273.97 391.113 270.349 391.113 270.349C391.113 270.349 390.64 226.771 389.983 211.962C389.52 201.738 386.672 196.766 386.363 193.928C386.363 193.928 384.944 140.831 384.789 127.373C384.693 118.356 383.718 108.451 383.718 108.451C383.718 108.451 373.871 109.243 373.002 106.376C372.133 103.508 373.002 75.946 373.369 70.588C373.369 70.588 352.67 65.9831 334.752 69.3813C334.762 69.3427 331.962 111.125 331.789 125.23Z"
        fill="#455A64"
      />
      <path
        d="M373.436 114.079C373.33 123.453 374.981 129.275 377.472 131.302C379.963 133.329 384.867 133.175 384.867 133.175V131.775C384.867 131.775 380.233 132.46 378.157 130.288C376.082 128.116 374.189 124.988 373.436 114.079Z"
        fill="#37474F"
      />
      <path
        d="M341.578 79.3252C341.578 79.3252 352.323 79.3252 362.489 80.0203C362.489 80.0203 362.556 87.869 361.678 90.7266C360.799 93.5842 354.032 97.8609 351.223 97.8513C348.79 97.8513 342.022 93.0243 341.491 90.4949C340.96 87.9656 341.578 79.3252 341.578 79.3252Z"
        fill="#37474F"
      />
      <path
        d="M359.013 158.827L358.048 152.803C355.151 152.368 347.689 147.734 343.287 143.68C346.418 148.417 350.656 152.32 355.634 155.052L355.943 193.475L359.013 158.827Z"
        fill="#37474F"
      />
      <path
        d="M378.495 56.0313C376.565 55.539 375.474 55.9444 374.499 58.3869C373.524 60.8294 372.404 68.765 372.027 76.1889C372.027 76.1889 367.557 76.9999 364.458 75.6869C364.458 75.6869 364.835 65.9364 365.878 60.9066C366.178 59.4225 366.664 57.9823 367.326 56.6202C367.754 55.7426 368.458 55.029 369.329 54.588C370.2 54.1471 371.192 54.003 372.153 54.1777L379.219 55.4714L378.495 56.0313Z"
        fill="#37474F"
      />
      <path
        d="M370.038 72.9055C370.038 73.253 369.935 73.5927 369.742 73.8816C369.549 74.1706 369.274 74.3958 368.953 74.5288C368.632 74.6617 368.279 74.6965 367.938 74.6287C367.597 74.5609 367.284 74.3936 367.039 74.1479C366.793 73.9022 366.625 73.5891 366.558 73.2483C366.49 72.9074 366.525 72.5541 366.658 72.2331C366.791 71.912 367.016 71.6376 367.305 71.4446C367.594 71.2515 367.933 71.1484 368.281 71.1484C368.747 71.1484 369.194 71.3336 369.523 71.6631C369.853 71.9926 370.038 72.4395 370.038 72.9055Z"
        fill="#489CFF"
      />
      <path
        d="M352.632 53.4331C351.088 53.4917 349.593 53.9886 348.321 54.8656C347.049 55.7426 346.053 56.9635 345.449 58.3857C343.991 62.0928 343.982 70.318 343.895 74.4017C341.659 75.2531 339.189 75.2531 336.954 74.4017C336.954 74.4017 336.954 61.4556 339.164 57.1692C341.375 52.8829 345.633 52.6222 347.853 52.4581C349.438 52.3642 351.027 52.3642 352.612 52.4581L352.632 53.4331Z"
        fill="#37474F"
      />
      <path
        d="M342.37 71.5344C342.37 71.8819 342.266 72.2216 342.073 72.5105C341.88 72.7995 341.606 73.0247 341.285 73.1577C340.964 73.2907 340.611 73.3254 340.27 73.2576C339.929 73.1899 339.616 73.0225 339.37 72.7768C339.124 72.5311 338.957 72.218 338.889 71.8772C338.821 71.5363 338.856 71.183 338.989 70.862C339.122 70.5409 339.347 70.2665 339.636 70.0735C339.925 69.8804 340.265 69.7773 340.613 69.7773C341.079 69.7773 341.525 69.9625 341.855 70.292C342.184 70.6215 342.37 71.0684 342.37 71.5344Z"
        fill="#489CFF"
      />
    </svg>
  );
};
