//
// Aspia Project
// Copyright (C) 2016-2024 Dmitry Chapyshev <dmitry@aspia.ru>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <https://www.gnu.org/licenses/>.
//

#include "client/ui/sys_info/sys_info_widget_local_user_groups.h"

#include "base/macros_magic.h"
#include "common/system_info_constants.h"

#include <cmath>
#include <limits>

#include <QMenu>

namespace client {

//--------------------------------------------------------------------------------------------------
SysInfoWidgetLocalUserGroups::SysInfoWidgetLocalUserGroups(QWidget* parent)
    : SysInfoWidget(parent)
{
    ui.setupUi(this);

    connect(ui.action_copy_row, &QAction::triggered, this, [this]()
    {
        copyRow(ui.tree->currentItem());
    });

    connect(ui.action_copy_name, &QAction::triggered, this, [this]()
    {
        copyColumn(ui.tree->currentItem(), 0);
    });

    connect(ui.action_copy_value, &QAction::triggered, this, [this]()
    {
        copyColumn(ui.tree->currentItem(), 1);
    });

    connect(ui.tree, &QTreeWidget::customContextMenuRequested,
            this, &SysInfoWidgetLocalUserGroups::onContextMenu);

    connect(ui.tree, &QTreeWidget::itemDoubleClicked,
            this, [this](QTreeWidgetItem* item, int /* column */)
    {
        copyRow(item);
    });
}

//--------------------------------------------------------------------------------------------------
SysInfoWidgetLocalUserGroups::~SysInfoWidgetLocalUserGroups() = default;

//--------------------------------------------------------------------------------------------------
std::string SysInfoWidgetLocalUserGroups::category() const
{
    return common::kSystemInfo_LocalUserGroups;
}

//--------------------------------------------------------------------------------------------------
void SysInfoWidgetLocalUserGroups::setSystemInfo(const proto::system_info::SystemInfo& system_info)
{
    ui.tree->clear();

    if (!system_info.has_local_user_groups())
    {
        ui.tree->setEnabled(false);
        return;
    }

    const proto::system_info::LocalUserGroups& local_user_groups = system_info.local_user_groups();
    QIcon item_icon(":/img/users.png");

    for (int i = 0; i < local_user_groups.local_user_group_size(); ++i)
    {
        const proto::system_info::LocalUserGroups::LocalUserGroup& local_user_group =
            local_user_groups.local_user_group(i);

        if (local_user_group.name().empty())
            continue;

        QTreeWidgetItem* item = new QTreeWidgetItem();
        item->setIcon(0, item_icon);
        item->setText(0, QString::fromStdString(local_user_group.name()));
        item->setText(1, QString::fromStdString(local_user_group.comment()));

        ui.tree->addTopLevelItem(item);
    }

    ui.tree->resizeColumnToContents(0);
}

//--------------------------------------------------------------------------------------------------
QTreeWidget* SysInfoWidgetLocalUserGroups::treeWidget()
{
    return ui.tree;
}

//--------------------------------------------------------------------------------------------------
void SysInfoWidgetLocalUserGroups::onContextMenu(const QPoint& point)
{
    QTreeWidgetItem* current_item = ui.tree->itemAt(point);
    if (!current_item)
        return;

    ui.tree->setCurrentItem(current_item);

    QMenu menu;
    menu.addAction(ui.action_copy_row);
    menu.addAction(ui.action_copy_name);
    menu.addAction(ui.action_copy_value);

    menu.exec(ui.tree->viewport()->mapToGlobal(point));
}

} // namespace client
