/**
 * Copyright 2011 Pablo Mendes, Max Jakob
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.dbpedia.spotlight.evaluation

import io.Source
import java.lang.System
import java.text.NumberFormat
import java.io.{PrintStream, File}
import scala.collection.mutable.HashMap

/**
 * The idea here is to compare the set of DBpediaResources generated by each of the spotlight methods.
 * We first load up all the gold standards, then evaluate each method against the gold standard.
 *
 * Evaluation is done on a set-basis:
 *  loading entities for *all documents* for a method into a set and comparing it with the gold set.
 *  This ignores repeated entities, and aggregates over all paragraphs, losing track if the entity was found in the correct paragraph.
 *  It's ok for evaluating entity extraction, but not suitable for evaluating some other use cases.
 *
 * Micro vs Macro average
 *  statistics can be calculated for each category and then averaged (macro), or can be calculated over all decisions and then averaged (micro).
 *  The notable difference between these to calculations is that micro-averaging gives equal weight to every entity (it is called a entity-pivoted measure) while macro-averaging gives equal weight to every document (document-pivoted measure).
 *  The two procedures bias the results differently - micro-averaging tends to over-emphasize the performance on the largest documents, while macro-averaging over-emphasizes the performance on the smallest.
 *  (Source http://backpan.perl.org/authors/id/K/KW/KWILLIAMS/Statistics-Contingency-0.03.readme)
 *  (Source http://datamin.ubbcluj.ro/wiki/index.php/Evaluation_methods_in_text_categorization)
 * @author pablomendes
 */

object SetEvaluation {

    val systems = Set(
        "Ontos",
        "OntosHeu",
        "OpenCalais",
        "OpenCalaisHeu",
        "WikiMachine",
        "Alchemy",
        "WMWikify",
        "WMWikify.0.3",
        "WMWikify.0.8",
        "Spotlight",
        "Zemanta",
        "HeadUp"
    )
    val spotlight = Set(
        "SpotlightNoFilter",
//        "SpotlightTop10Score",
//        "SpotlightTop10Prior",
//        "SpotlightTop10Confusion",
//        "SpotlightTop10Confidence",
//        "SpotlightTop10Context",
        "SpotlightRandom"
    )

    def getSystemsResults(baseDir: String):  HashMap[String, Set[String]] = {
        getExternalSystemsResults(baseDir) ++ getSpotlightResults(baseDir)
    }

    def getExternalSystemsResults(baseDir: String) :  HashMap[String, Set[String]] = {
        buildMap(baseDir+"/systems/", systems, "list");
    }
    def getSpotlightResults(baseDir: String) :  HashMap[String, Set[String]] = {
        buildMap(baseDir+"/spotlight/", spotlight, "set");
    }

    def buildMap(baseDir: String, names: Set[String], extension: String = "list") :  HashMap[String, Set[String]] = {
        var resultsByName = new HashMap[String, Set[String]]
        names.foreach( systemName => {
            val f = new File(baseDir+systemName+"."+extension)
            if (f.exists) {
                //println("Found: "+systemName);
                resultsByName += systemName -> slurp(f)
            } else {
                println("Not found: "+f.getAbsolutePath);
            }
        })
        return resultsByName;
    }

    implicit def string2file(filename: String) = new File(filename);

    def slurp(f: File) = {
        Source.fromFile(f).getLines.filter(l => l.trim() != "").toSet
    }

  def getManualGold(baseDir: String) :  Map[String, Set[String]] = {
    val mostrelaxed = slurp(baseDir+"/gold/gold-mostRelaxed-Union.set")
    val relaxed = slurp(baseDir+"/gold/gold-relaxed.set")
    val strict = slurp(baseDir+"/gold/gold-strict.set")
    val strictest = slurp(baseDir+"/gold/gold-strictest-onlyInfobox.set")
    val manualGold = Map("mostrelaxed"->mostrelaxed, "relaxed"->relaxed, "strict"->strict, "strictest"->strictest);
    manualGold;
  }

  def getCucerzanGold(baseDir: String) :  Map[String, Set[String]] = {
    val cucerzan = slurp(baseDir+"/gold/cucerzan.set")
    val cucerzanGold = Map("cucerzan"->cucerzan);
    cucerzanGold
  }

  def getWikifyGold(baseDir: String) :  Map[String, Set[String]] = {
    val wikify = slurp(baseDir+"/gold/WikifyAllInOne.set")
    val wikifyGold = Map("wikify"->wikify);
    wikifyGold
  }

    def getGoldResults(baseDir: String) : Map[String, Set[String]] = {
        val sep = System.getProperty("file.separator")
        val dirs : Array[String] = baseDir.split(sep)
        val goldStandardName = dirs(dirs.size-1)
        goldStandardName match {
            case "cucerzan" =>
                 Map("cucerzan"->slurp(baseDir+"/gold/cucerzan.set"))
            case "manual" =>
                getManualGold(baseDir)
            case "wikify" =>
                Map("wikify"->slurp(baseDir+"/gold/WikifyAllInOne.set"))
            case "grounder" =>
                Map("grounder"->slurp(baseDir+"/gold/g1b_spotlight.set"))
            case "csaw" =>
                Map("csaw"->slurp(baseDir+"/gold/gold.set"))
            case _ => {
                println("There is no loader method configured for a gold standard called "+goldStandardName)
                Map()
            }
        }
    }


    /**
     * Prints the precision / recall points in the chart for each of the systems
     */

  def printPoints(baseDir: String, gold: Map[String, Set[String]], systems: HashMap[String, Set[String]]) {
    System.err.println("Printing points file.");
    for (gk <- gold.keys) {
      var g = gold(gk);
      //System.out.println("Gold size: "+g.size+" ("+gk+")");
      val out = new PrintStream(baseDir+"/"+gk+".points");
      out.println("System, Precision, Recall, F1" );
      for (sk <- systems.keySet.toList.sorted) {
        var s = systems(sk)
        //System.out.println("\tPredicted size: "+s.size);
        val intersection : Double = ((s & g).size)
        //System.out.printf("\t %s & %f & %f \\\\ ", sk, intersection / s.size , intersection / g.size);
        val precision =  intersection / s.size
        val recall = intersection / g.size
        val f1 = 2 * ( precision * recall ) / ( precision + recall )
        out.println(sk+", "+precision+ ", " + recall + ", " + f1);
      }
      out.close();
    }
  }

  def printLine(out: PrintStream, fileName: String, system: String, g: Set[String], support: Int, confidence: Double) {
    System.err.println("Reading: "+fileName);
    val s = Source.fromFile(fileName).getLines.toSet
    //System.out.println("\tPredicted size: "+s.size);
    val intersection : Double = ((s & g).size)
    //System.out.printf("\t %s & %f & %f \\\\ ", sk, intersection / s.size , intersection / g.size);
    val precision =  intersection / s.size
    val recall = intersection / g.size
    val f1 = 2 * ( precision * recall ) / ( precision + recall )
    out.println(system+", "+support+", "+confidence+", "+precision+ ", " + recall + ", " + f1);
  }

    /**
     * Prints the precision and recall values in CSV format for plotting the lines in the chart for the paper
     */
  def printLines(baseDir: String, gold: Map[String, Set[String]]) {
    System.err.println("Printing lines file.");
    for (gk <- gold.keys) {
      var g = gold(gk);
      //System.out.println("Gold size: "+g.size+" ("+gk+")");
      val out1 = new PrintStream(baseDir+"/"+gk+".byScore.lines");
      out1.println("System, Support, Confidence, Precision, Recall, F1" );
      val system = "spotlight"
      for (support <- EvalParams.supportInterval) {   //TODO make it contextualScore and prior
        for (score <- EvalParams.contextualScoreInterval) {
          //val fileName = baseDir+"/spotlight/Spotlight.c"+confidence+"s"+support+".list";
          val fileName = baseDir+"/spotlight/Spotlight.s"+score+"p"+support+".set";
          //val fileName = baseDir+"/spotlight/AnnotationText-Spotlight.c"+confidence+"s"+support+".txt.set";
          printLine(out1, fileName, system, g, support, score);
        }
      }
      out1.close();

        val out2 = new PrintStream(baseDir+"/"+gk+".byConfidence.lines");
        out2.println("System, Support, Confidence, Precision, Recall, F1" );
        for (support <- EvalParams.supportInterval) {
          for (confidence <- EvalParams.confidenceInterval) {
            //val fileName = baseDir+"/spotlight/Spotlight.c"+confidence+"s"+support+".list";
            val fileName = baseDir+"/spotlight/Spotlight.c"+confidence+"s"+support+".set";
            //val fileName = baseDir+"/spotlight/AnnotationText-Spotlight.c"+confidence+"s"+support+".txt.set";
            printLine(out2, fileName, system, g, support, confidence);
          }
        }
        out2.close();

    }
  }

    /**
     * Prints out a latex formatted table for adding to the paper
     */
  def printTable(gold: Map[String, Set[String]], systems: Map[String, Set[String]])
  {

    //      val baseDir: String = args(0)+"/";
    //      if (!new File(baseDir).exists) {
    //        System.err.println("Base directory does not exist. "+baseDir);
    //        exit();
    //      }
    val nf = NumberFormat.getPercentInstance()

    for (gk <- gold.keys) {
      var g = gold(gk);
      System.out.println("Gold size: "+g.size+" ("+gk+")");
      System.out.println("\tSystem & Precision & Recall & F1 \\\\" );
      for (sk <- systems.keySet.toList.sorted) {
        var s = systems(sk)
        //System.out.println("\tPredicted size: "+s.size);
        val intersection : Double = ((s & g).size)
        //System.out.printf("\t %s & %f & %f \\\\ ", sk, intersection / s.size , intersection / g.size);
        val precision =  intersection / s.size
        val recall = intersection / g.size
        val f1 = 2 * ( precision * recall ) / ( precision + recall )
        System.out.println("\t"+sk+" & "+ nf.format(precision) + " & " + nf.format(recall) + " & " + nf.format(f1) + "\\\\" );
      }
    }

    //      val wikiMachineFileName = baseDir+"AnnotationText-WikiMachine.txt";
    //      val text = Source.fromFile(wikiMachineFileName).getLines.toSet
  }

    def run(baseDir: String) {
        val gold = SetEvaluation.getGoldResults(baseDir)
      val systems = SetEvaluation.getSystemsResults(baseDir)

      printPoints(baseDir, gold, systems);
      printLines(baseDir, gold);
    }

   def main(args : Array[String])
  {

    //      val baseDir: String = args(0)+"/";
    //      if (!new File(baseDir).exists) {
    //        System.err.println("Base directory does not exist. "+baseDir);
    //        exit();
    //      }

//    val baseDir = "/home/pablo/eval/cucerzan"
    //val baseDir = "/home/pablo/eval/manual"
//      val baseDir = "/home/pablo/eval/wikify"
      //val baseDir = "/home/pablo/eval/grounder"
      val baseDir = "/home/pablo/eval/csaw"
      run(baseDir)

  }
}