/*
 * Copyright 2011 Pablo Mendes, Max Jakob
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  Check our project website for information on how to acknowledge the authors and how to contribute to the project: http://spotlight.dbpedia.org
 */

package org.dbpedia.spotlight.spot

import xml.{Node, XML}
import org.dbpedia.spotlight.model.{SurfaceForm, SurfaceFormOccurrence, Text}
import scala.collection.JavaConversions._

/**
 * This is a parser disguised as a spotter.
 * It takes as input an XML containing spots generated by another spotting engine.
 * It parses this XML and builds surface form occurrences from it, so that one of our disambiguators can take spots build somewhere else.
 *
 * TODO separate in two classes? one is the parser and the other is the spotter wrapper around the parser
 * TODO generalize code so that other features can be embedded in the XML and inserted in a feature vector in SurfaceFormOccurrence
 *
 * @author pablomendes
 */

class SpotXmlParser extends Spotter {

  var name = "SpotXmlParser"

  /**
   * Extracts a set of surface form occurrences from the text
   */
  def extract(spotsXml: Text): java.util.List[SurfaceFormOccurrence] = {
    val xml = XML.loadString(spotsXml.text)
    val text = (xml \\ "annotation" \ "@text").toString
    val surfaceForms = xml \\"annotation" \ "surfaceForm"
    val occs = surfaceForms.map(buildOcc(_, new Text(text)))
    occs.toList
  }

  def buildOcc(sf: Node, text: Text) = {
    val offset = (sf \ "@offset").toString.toInt
    val name = (sf \ "@name").toString
    new SurfaceFormOccurrence(new SurfaceForm(name), text, offset)
  }

  def getName() = name

  def setName(n: String) {
    name = n;
  }

}

object SpotXmlParser {
  def main(args: Array[String]) {
    val xml = "<annotation text=\"The research, which is published online May 22 in the European Heart Journal, opens up the prospect of treating heart failure patients with their own, human-induced pluripotent stem cells (hiPSCs) to repair their damaged hearts.\">\n<surfaceForm name=\"published\" offset=\"23\"/>\n<surfaceForm name=\"May 22\" offset=\"40\"/>\n<surfaceForm name=\"European\" offset=\"54\"/>\n<surfaceForm name=\"Heart\" offset=\"63\"/>\n<surfaceForm name=\"Journal\" offset=\"69\"/>\n<surfaceForm name=\"prospect\" offset=\"91\"/>\n<surfaceForm name=\"heart failure\" offset=\"112\"/>\n<surfaceForm name=\"patients\" offset=\"126\"/>\n<surfaceForm name=\"human\" offset=\"151\"/>\n<surfaceForm name=\"stem cells\" offset=\"177\"/>\n<surfaceForm name=\"hearts\" offset=\"221\"/>\n</annotation>"
    val spotter = new SpotXmlParser()
    spotter.extract(new Text(xml)).foreach(println)
  }
}