package main

import (
	"io"
	"strings"
	"text/template"

	"github.com/gertd/go-pluralize"
	"github.com/pkg/errors"
)

const (
	// packageTmpl it's the package definition
	packageTmpl = `
	package azurerm
	// Code generated by 'go generate'; DO NOT EDIT
	import (
		"context"

		"github.com/pkg/errors"

		{{ range .AzureAPIs -}}
		{{if .PackageIdentifier}}{{ .PackageIdentifier }}{{ end }}"github.com/Azure/azure-sdk-for-go/services/{{ if .IsPreview }}preview/{{ end }}{{ if .OtherPath }}{{ .OtherPath }}{{ else }}{{ .API }}/mgmt{{ end }}/{{ .APIVersion }}/{{ .API }}"
		{{ end }}
	)
	`

	// functionTmpl it's the implementation of a reader function
	functionTmpl = `
	// {{ .FunctionName }} returns a list of {{ .PluralName }} within a subscription {{ if .Location }}and a location {{ end }}{{ if .ResourceGroup }}and a resource group {{ end }}
	func (ar *AzureReader) {{ .FunctionName }}(ctx context.Context{{ range .ExtraArgsBeforeResourceGroup }},{{ .Name }} {{ .Type }} {{ end }}{{ range .ExtraArgs }},{{ .Name }} {{ .Type }} {{ end }}) ([]{{ .API }}.{{ .ResourceName }}, error) {
		client := {{ .API }}.{{ if .IrregularClientName }}{{ .IrregularClientName }}{{ else }}New{{ .PluralName }}Client{{ end }}WithBaseURI(ar.env.ResourceManagerEndpoint, ar.config.SubscriptionID)
		client.Authorizer = ar.authorizer

		output, err := client.{{ .AzureSDKListFunction }}(ctx{{ if .Location }}, ar.GetLocation(){{ end }}{{ if .Subscription }}, ar.config.SubscriptionID{{ end }}{{ if .ResourceGroup }}{{ range .ExtraArgsBeforeResourceGroup }},{{ .Name }}{{ end }}, ar.GetResourceGroupName(){{ end }}{{ range .ExtraArgs }},{{ .Name }}{{ end }})
		if err != nil {
			return nil, errors.Wrap(err, "unable to list {{ .API }}.{{ .ResourceName }} from Azure APIs")
		}
		{{ if .ReturnsList }}
			return *output.Value, nil

		{{else}}
		resources := make([]{{ .API }}.{{ .ResourceName }}, 0)
		for output.NotDone() {
			{{ if .ReturnsIterator }}
			res = output.Value()
			resources = append(resources, res)
			{{ else }}
			for _, res := range output.Values() {
				resources = append(resources, res)
			}
			{{ end }}
			if err := output.NextWithContext(ctx); err != nil {
				break
			}
		}
		return resources, nil
		{{end}}
	}
	`
)

var (
	fnTmpl  *template.Template
	pkgTmpl *template.Template
)

func init() {
	var err error
	fnTmpl, err = template.New("template").Parse(functionTmpl)
	if err != nil {
		panic(err)
	}
	pkgTmpl, err = template.New("template").Parse(packageTmpl)
	if err != nil {
		panic(err)
	}
}

// Arg can be used to define
// extra args to pass to the generated
// functions
type Arg struct {
	// Name of the arg
	Name string
	// Type of the arg
	Type string
}

// AzureAPI is the definition of one of the Azure APIs
type AzureAPI struct {
	// API is used to determine the
	// Azure API to use
	// ex: compute, network
	API string

	// PackageIdentifier is used if multiple API versions are required
	// allows to diferentiate the different API versions using GO package identifier
	// Note! if used PackageIdentifier should be used as API when defining objects
	PackageIdentifier string

	// OtherPath is used for cases where the
	// Azure API path is not the usual one github.com/Azure/azure-sdk-for-go/services/{{API}}/mgmt/{{APIVersion}}/{{API}}
	// the path would then be github.com/Azure/azure-sdk-for-go/services/{{otherPath}}/{{APIVersion}}/{{API}}
	OtherPath string

	// APIVersion is used to determine the
	// Azure API Version to use
	// ex: 2019-07-01
	APIVersion string

	// IsPreview defines if the API is a preview
	// functionality
	// https://docs.microsoft.com/en-us/azure/search/search-api-preview
	IsPreview bool

	//if api as AddSuffix = true then FunctionName = "List" + API + PluralName instead of "List" + PluralName
	AddAPISufix bool
}

// Function is the definition of one of the functions
type Function struct {
	// Resource is the Azure name of the entity, like
	// VirtualMachine, VirtualNetwork, etc.
	ResourceName string

	// To specify special cases for the function names
	// it can be useful if you `Resource` is `SslCertificate`, which is not `go`
	// compliant, `Name` will be `SSLCertificate`, your Function name will be
	// `ListSSLCertificates`
	FunctionName string

	// PluralName corresponds to the plural of the resource to identify the irregular plural cases or non-existant plural
	// e.g.: ends in s or x change plural to es instead of s
	// Most cases determined automatically, no need to specify
	PluralName string

	// IrregularClientName allows to specify the cases where client name is not the usual New{{PluralName}}}Client
	// Most cases no need to specify
	IrregularClientName string

	// API is used to determine the
	// Azure API to use
	// ex: compute, network
	API string

	// Location is used to determine whether the resource should be filtered by Azure locations or not
	Location bool

	// ResourceGroup is used to determine whether the resource should be filtered by Azure Resource Group or not
	ResourceGroup bool

	// ResourceGroup is used to determine whether the resource should be filtered Azure Subscription
	Subscription bool

	// AzureSDKListFunction is the Azure SKP list function to use
	AzureSDKListFunction string

	// ReturnsIterator should be true is the ListFunction returns an Iterator and not a Page
	ReturnsIterator bool

	// ReturnsList should be true if the ListFunction returns a List and not an Iterator or Page
	ReturnsList bool

	// ExtraArgs should be specified if extra arguments are required for the list function
	ExtraArgs []Arg

	// ExtraArgsBeforeResourceGroup should be specified if extra arguments are required for the list function but before the resource group
	// ex: List(extra_arg_before1, rg, extra_arg1)
	ExtraArgsBeforeResourceGroup []Arg
}

// Execute uses the fnTmpl to interpolate f
// and write the result to w
func (f Function) Execute(w io.Writer) error {
	// If Plural not set determine automatically
	if len(f.PluralName) == 0 {
		f.PluralName = pluralize.NewClient().Plural(f.ResourceName)
	}
	//If no list name defined takes list by default
	if len(f.AzureSDKListFunction) == 0 {
		f.AzureSDKListFunction = "List"
	}
	// Determine the FunctionName to give in the template, if not set
	// By default -> FunctionName = List + PluralName
	// if api AddApiSufix = true -> FunctionName = List + API + PluralName(to avoid equal names of functions)
	if len(f.FunctionName) == 0 {
		f.FunctionName = "List" + f.PluralName
		for _, api := range azureAPIs {
			if api.API == f.API && api.AddAPISufix {
				f.FunctionName = "List" + strings.ToUpper(f.API) + f.PluralName
			}
		}
	}

	if err := fnTmpl.Execute(w, f); err != nil {
		return errors.Wrapf(err, "failed to Execute with Function %+v", f)
	}
	return nil
}
