"use strict";

const clone = require('clone');
const merge = require('../../../util/merge');

/**
 * The Renderable is the base class for any object that can be customized by a
 * [theme set]{@link Defiant.Plugin.ThemeBase}.
 *
 * Renderables have 2 parts: the Definition and the Instance.  To further
 * complicate things, there is a difference between the class definition for
 * each of these and an instantiation of that class.  The complexity arises from
 * the need to be able to programmatically define new Renderable types.
 *
 * The definition is the Renderable class that you see here.  When added to the
 * [theme set]{@link Defiant.Plugin.ThemeBase}, via the
 * [ThemeBase.setRenderable()]{@link Defiant.Plugin.ThemeBase.setRenderable}
 * function, the defintion will be instantiated.  This way, the instantiation
 * can be customized.  Such flexibility is helpful when multiple themes are used
 * on a single site (such as an admin theme and a general site theme).  The
 * instance used in each theme can be customized without affecting the
 * Renderable in the other theme.
 *
 * The Instance is what is instantiated to represent the actual element on the
 * page.  For example, consider a button in a form.  There may be several
 * buttons in the same form, each with its own text and other properties. There
 * will be an instantiated
 * [RenderableInstance]{@link Defiant.Plugin.Theme.RenderableInstance} to
 * represent each one of those buttons individually.
 *
 * To put it all together, a Renderable is a class that describes a type of
 * visible content.  An instance of this Renderable is added to the theme where
 * it can be customized for use in that theme.  If the Renderable instance that
 * is added to the theme set is customized in any way, then those customizations
 * will be applied to any RenderableInstance generated by that theme set.
 *
 * When that Renderable needs to be displayed, then the theme set creates a
 * RenderableInstance, providing it with the specific settings for that
 * occurrence on the page (e.g., CSS ids and classes, the necessary title or
 * body text, etc.).
 *
 * In this way, all Renderables can be customized first by theme, and then by
 * the individual appearance within a request fulfillment.
 * @class
 * @memberOf Defiant.Plugin.Theme
 */
class Renderable {
  /**
   * @param {Defiant.Engine} engine
   *   The app engine.
   * @param {Object} [setup={}]
   *   The setup options.
   * @param {Object} [setup.instanceSetup={}]
   *   Setup options which will be passed to the
   *   [RenderableInstance]{@link Defiant.Plugin.Theme.RenderableInstance} when
   *   it is instantiated.
   * @returns {Defiant.Plugin.Theme.Renderable}
   *   The instantiated Renderable.
   */
  constructor(engine, setup={}) {
    this.engine = engine;
    /**
     * @member {Object} Defiant.Plugin.Theme.Renderable#instanceSetup
     *   Settings to pass to the
     *   [RenderableInstance]{@link Defiant.Plugin.Theme.RenderableInstance}
     *   when it is instantiated by
     *   [Renderable.newInstance()]{@link Defiant.Plugin.Theme.Renderable#newInstance}
     */
    this.instanceSetup = setup.instanceSetup || {};
    [
      /**
       * @member {String} Defiant.Plugin.Theme.Renderable#id
       *   The unique identifier for this Renderable.
       */
      'id',
      /**
       * @member {String} Defiant.Plugin.Theme.Renderable#name
       *   The human-readable name of this Renderable.
       */
      'name',
      /**
       * @member {String} Defiant.Plugin.Theme.Renderable#templateFile
       *   The local file path to the template for this renderable.
       */
      'templateFile',
      /**
       * @member {String[]} Defiant.Plugin.Theme.Renderable#variables
       *   An array of variable names that will be passed to the template
       *   function.
       */
      'variables',
      /**
       * @member {Defiant.Plugin.Theme.RenderableInstance} Defiant.Plugin.Theme.Renderable#Instance
       *   The class that will be instantiated when
       *   [Renderable.newInstance()]{@link Defiant.Plugin.Theme.Renderable#newInstance}
       *   is called.
       */
      'Instance'].map(key => this[key] = setup[key] ? setup[key] : this.constructor[key]);
    if (this.id == undefined) {
      this.id = this.constructor.name;
    }
  }


  /**
   * Create an [Instance]{@link Defiant.Plugin.Theme.RenderableInstance} for
   * this renderable.
   * @param {Defiant.Context} context
   *   The request context.
   * @param {object} [setup={}]
   *   The configuration to be passed to the instance.
   * @returns {Defiant.Plugin.Theme.RenderableInstance}
   *   An instantiated RenderableInstance.
   */
  newInstance(context, setup={}) {
    return new (this.Instance || this.constructor.Instance)(this, merge(clone(this.instanceSetup), setup), context);
  }

  /**
   * This is a work-around to allow for the
   * [Instance.templateFunction]{@link Defiant.Plugin.Theme.Renderable#templateFunction}
   * to be set on some class definitions, and also enable any subclasses find
   * it, which is hard to do with JavaScript's prototypical inheritance.
   *
   * @param {Object} variables
   *   The variables to pass to the template function.
   * @returns {function}
   *   The template function that was compiled from the template.
   */
  templateFunction(variables) {
    // TODO: templateFunction should be the function itself, not just a wrapper.
    return this.constructor.templateFunction(variables);
  }

  /**
   * Compile the template into a function.
   * @static
   *
   * @param {Object} variables
   *   The variables to pass into the template.
   * @param {String} code
   *   The template string to be compiled to a function
   * @param {String} boilerplate
   *   A standard boilerplate code to be put at the top of the template, before
   *   the `code`.
   * @returns {function}
   *   The template compiled into a function.
   */
  static compileTemplate(variables, code, boilerplate) {
    code = ('%>' + code + '<%')
      .replace(/(<%=)(([^%]|%(?!>))*)(%>)/g, (...args) => `<% templateOut += (${args[2]});\n%>`)
      .replace(/(%>)(([^<]|<(?!%))*)(<%)/g, (...args) => `templateOut += ${JSON.stringify(args[2])};\n`);
    boilerplate = boilerplate || '';
    return new Function(`"use strict";
      return function({${variables.join()}}) {
        let templateOut = '';
        ${boilerplate}
        ${code}
        return templateOut;
      };`)();
  }
}

Renderable.Instance = require('./renderableInstance');

/**
 * @member {String} Defiant.Plugin.Theme.Renderable#boilerplate
 *   JavaScript that will be included into the template, before the actual
 *   template code is run.
 */
Renderable.boilerplate = undefined;

/**
 * @member {String[]} Defiant.Plugin.Theme.Renderable#variables
 *   The variables that are expected by the render function.
 */
Renderable.variables = []

module.exports = Renderable;
