---
permalink: /helpers/WebDriver
editLink: false
sidebar: auto
title: WebDriver
---

<!-- Generated by documentation.js. Update this documentation by updating the source code. -->

## WebDriver

**Extends Helper**

WebDriver helper which wraps [webdriverio][1] library to
manipulate browser using Selenium WebDriver or PhantomJS.

WebDriver requires Selenium Server and ChromeDriver/GeckoDriver to be installed. Those tools can be easily installed via NPM. Please check [Testing with WebDriver][2] for more details.

With the release of WebdriverIO version v8.14.0, and onwards, all driver management hassles are now a thing of the past 🙌. Read more [here][3].
One of the significant advantages of this update is that you can now get rid of any driver services you previously had to manage, such as
`wdio-chromedriver-service`, `wdio-geckodriver-service`, `wdio-edgedriver-service`, `wdio-safaridriver-service`, and even `@wdio/selenium-standalone-service`.

For those who require custom driver options, fear not; WebDriver Helper allows you to pass in driver options through custom WebDriver configuration.
If you have a custom grid, use a cloud service, or prefer to run your own driver, there's no need to worry since WebDriver Helper will only start a driver when there are no other connection information settings like hostname or port specified.



## Configuration

This helper should be configured in codecept.conf.js

Type: [object][17]

### Properties

*   `url` **[string][18]** base url of website to be tested.
*   `browser` **[string][18]** Browser in which to perform testing.
*   `basicAuth` **[string][18]?** (optional) the basic authentication to pass to base url. Example: {username: 'username', password: 'password'}
*   `host` **[string][18]?** WebDriver host to connect.
*   `port` **[number][23]?** WebDriver port to connect.
*   `protocol` **[string][18]?** protocol for WebDriver server.
*   `path` **[string][18]?** path to WebDriver server.
*   `restart` **[boolean][33]?** restart browser between tests.
*   `smartWait` **([boolean][33] | [number][23])?** **enables [SmartWait][37]**; wait for additional milliseconds for element to appear. Enable for 5 secs: "smartWait": 5000.
*   `disableScreenshots` **[boolean][33]?** don't save screenshots on failure.
*   `fullPageScreenshots` **[boolean][33]?** (optional - make full page screenshots on failure.
*   `uniqueScreenshotNames` **[boolean][33]?** option to prevent screenshot override if you have scenarios with the same name in different suites.
*   `keepBrowserState` **[boolean][33]?** keep browser state between tests when `restart` is set to false.
*   `keepCookies` **[boolean][33]?** keep cookies between tests when `restart` set to false.
*   `windowSize` **[string][18]?** default window size. Set to `maximize` or a dimension in the format `640x480`.
*   `waitForTimeout` **[number][23]?** sets default wait time in *ms* for all `wait*` functions.
*   `desiredCapabilities` **[object][17]?** Selenium's [desired capabilities][7].
*   `manualStart` **[boolean][33]?** do not start browser before a test, start it manually inside a helper with `this.helpers["WebDriver"]._startBrowser()`.
*   `timeouts` **[object][17]?** [WebDriver timeouts][38] defined as hash.
*   `highlightElement` **[boolean][33]?** highlight the interacting elements. Default: false. Note: only activate under verbose mode (--verbose).
*   `logLevel` **[string][18]?** level of logging verbosity. Default: silent. Options: trace | debug | info | warn | error | silent. More info: [https://webdriver.io/docs/configuration/#loglevel][39]



Example:

```js
{
   helpers: {
     WebDriver : {
       smartWait: 5000,
       browser: "chrome",
       restart: false,
       windowSize: "maximize",
       timeouts: {
         "script": 60000,
         "page load": 10000
       }
     }
   }
}
```

Testing Chrome locally is now more convenient than ever. You can define a browser channel, and WebDriver Helper will take care of downloading the specified browser version for you.
For example:

```js
{
   helpers: {
     WebDriver : {
       smartWait: 5000,
       browser: "chrome",
       browserVersion: '116.0.5793.0', // or 'stable', 'beta', 'dev' or 'canary'
       restart: false,
       windowSize: "maximize",
       timeouts: {
         "script": 60000,
         "page load": 10000
       }
     }
   }
}
```

Example with basic authentication

```js
{
   helpers: {
     WebDriver : {
       smartWait: 5000,
       browser: "chrome",
       basicAuth: {username: 'username', password: 'password'},
       restart: false,
       windowSize: "maximize",
       timeouts: {
         "script": 60000,
         "page load": 10000
       }
     }
   }
}
```

Additional configuration params can be used from [webdriverio
website][4].

### Headless Chrome

```js
{
   helpers: {
     WebDriver : {
       url: "http://localhost",
       browser: "chrome",
       desiredCapabilities: {
         chromeOptions: {
           args: [ "--headless", "--disable-gpu", "--no-sandbox" ]
         }
       }
     }
   }
}
```

### Running with devtools protocol

```js
{
   helpers: {
     WebDriver : {
       url: "http://localhost",
       browser: "chrome",
       desiredCapabilities: {
         chromeOptions: {
           args: [ "--headless", "--disable-gpu", "--no-sandbox" ]
         }
       }
     }
   }
}
```

### Internet Explorer

Additional configuration params can be used from [IE options][5]

```js
{
   helpers: {
     WebDriver : {
       url: "http://localhost",
       browser: "internet explorer",
       desiredCapabilities: {
         ieOptions: {
           "ie.browserCommandLineSwitches": "-private",
           "ie.usePerProcessProxy": true,
           "ie.ensureCleanSession": true,
         }
       }
     }
   }
}
```

### Selenoid Options

[Selenoid][6] is a modern way to run Selenium inside Docker containers.
Selenoid is easy to set up and provides more features than original Selenium Server. Use `selenoidOptions` to set Selenoid capabilities

```js
{
   helpers: {
     WebDriver : {
       url: "http://localhost",
       browser: "chrome",
       desiredCapabilities: {
         selenoidOptions: {
           enableVNC: true,
         }
       }
     }
   }
}
```

### Connect Through proxy

CodeceptJS also provides flexible options when you want to execute tests to Selenium servers through proxy. You will
need to update the `helpers.WebDriver.capabilities.proxy` key.

```js
{
    helpers: {
        WebDriver: {
            capabilities: {
                proxy: {
                    "proxyType": "manual|pac",
                    "proxyAutoconfigUrl": "URL TO PAC FILE",
                    "httpProxy": "PROXY SERVER",
                    "sslProxy": "PROXY SERVER",
                    "ftpProxy": "PROXY SERVER",
                    "socksProxy": "PROXY SERVER",
                    "socksUsername": "USERNAME",
                    "socksPassword": "PASSWORD",
                    "noProxy": "BYPASS ADDRESSES"
                }
            }
        }
    }
}
```

For example,

```js
{
    helpers: {
        WebDriver: {
            capabilities: {
                proxy: {
                    "proxyType": "manual",
                    "httpProxy": "http://corporate.proxy:8080",
                    "socksUsername": "codeceptjs",
                    "socksPassword": "secret",
                    "noProxy": "127.0.0.1,localhost"
                }
            }
        }
    }
}
```

Please refer to [Selenium - Proxy Object][7] for more
information.

### Cloud Providers

WebDriver makes it possible to execute tests against services like `Sauce Labs` `BrowserStack` `TestingBot`
Check out their documentation on [available parameters][8]

Connecting to `BrowserStack` and `Sauce Labs` is simple. All you need to do
is set the `user` and `key` parameters. WebDriver automatically know which
service provider to connect to.

```js
{
    helpers:{
        WebDriver: {
            url: "YOUR_DESIRED_HOST",
            user: "YOUR_BROWSERSTACK_USER",
            key: "YOUR_BROWSERSTACK_KEY",
            capabilities: {
                "browserName": "chrome",

                // only set this if you're using BrowserStackLocal to test a local domain
                // "browserstack.local": true,

                // set this option to tell browserstack to provide addition debugging info
                // "browserstack.debug": true,
            }
        }
    }
}
```

#### SauceLabs

SauceLabs can be configured via wdio service, which should be installed additionally:

    npm i @wdio/sauce-service --save

It is important to make sure it is compatible with current webdriverio version.

Enable `wdio` plugin in plugins list and add `sauce` service:

```js
plugins: {
   wdio: {
      enabled: true,
       services: ['sauce'],
       user: ... ,// saucelabs username
       key: ... // saucelabs api key
       // additional config, from sauce service
   }
}
```

See [complete reference on webdriver.io][9].

> Alternatively, use [codeceptjs-saucehelper][10] for better reporting.

#### BrowserStack

BrowserStack can be configured via wdio service, which should be installed additionally:

    npm i @wdio/browserstack-service --save

It is important to make sure it is compatible with current webdriverio version.

Enable `wdio` plugin in plugins list and add `browserstack` service:

```js
plugins: {
   wdio: {
      enabled: true,
       services: ['browserstack'],
       user: ... ,// browserstack username
       key: ... // browserstack api key
       // additional config, from browserstack service
   }
}
```

See [complete reference on webdriver.io][11].

> Alternatively, use [codeceptjs-bshelper][12] for better reporting.

#### TestingBot

> **Recommended**: use official [TestingBot Helper][13].

Alternatively, TestingBot can be configured via wdio service, which should be installed additionally:

    npm i @wdio/testingbot-service --save

It is important to make sure it is compatible with current webdriverio version.

Enable `wdio` plugin in plugins list and add `testingbot` service:

```js
plugins: {
  wdio: {
      enabled: true,
      services: ['testingbot'],
      user: ... ,// testingbot key
      key: ... // testingbot secret
      // additional config, from testingbot service
  }
}
```

See [complete reference on webdriver.io][14].

#### Applitools

Visual testing via Applitools service

> Use [CodeceptJS Applitools Helper][15] with Applitools wdio service.

### Multiremote Capabilities

This is a work in progress but you can control two browsers at a time right out of the box.
Individual control is something that is planned for a later version.

Here is the [webdriverio docs][16] on the subject

```js
{
    helpers: {
        WebDriver: {
            "multiremote": {
                "MyChrome": {
                    "desiredCapabilities": {
                        "browserName": "chrome"
                     }
                },
                "MyFirefox": {
                   "desiredCapabilities": {
                       "browserName": "firefox"
                   }
                }
            }
        }
    }
}
```

## Access From Helpers

Receive a WebDriver client from a custom helper by accessing `browser` property:

```js
const { WebDriver } = this.helpers;
const browser = WebDriver.browser
```

## Methods

### Parameters

*   `config` &#x20;

### _isShadowLocator

Check if locator is type of "Shadow"

#### Parameters

*   `locator` **[object][17]**&#x20;

### _locate

Get elements by different locator types, including strict locator.
Should be used in custom helpers:

```js
this.helpers['WebDriver']._locate({name: 'password'}).then //...
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `smartWait`   

### _locateCheckable

Find a checkbox by providing human-readable text:

```js
this.helpers['WebDriver']._locateCheckable('I agree with terms and conditions').then // ...
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.

### _locateClickable

Find a clickable element by providing human-readable text:

```js
const els = await this.helpers.WebDriver._locateClickable('Next page');
const els = await this.helpers.WebDriver._locateClickable('Next page', '.pages');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `context` &#x20;

### _locateFields

Find field elements by providing human-readable text:

```js
this.helpers['WebDriver']._locateFields('Your email').then // ...
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.

### _locateShadow

Locate Element within the Shadow Dom

#### Parameters

*   `locator` **[object][17]**&#x20;

### _smartWait

Smart Wait to locate an element

#### Parameters

*   `locator` **[object][17]**&#x20;

### acceptPopup

Accepts the active JavaScript native popup window, as created by window.alert|window.confirm|window.prompt.
Don't confuse popups with modal windows, as created by [various
libraries][19].

### amOnPage

Opens a web page in a browser. Requires relative or absolute url.
If url starts with `/`, opens a web page of a site defined in `url` config parameter.

```js
I.amOnPage('/'); // opens main page of website
I.amOnPage('https://github.com'); // opens github
I.amOnPage('/login'); // opens a login page
```

#### Parameters

*   `url` **[string][18]** url path or global url.

Returns **void** automatically synchronized promise through #recorder

### appendField

Appends text to a input field or textarea.
Field is located by name, label, CSS or XPath

```js
I.appendField('#myTextField', 'appended');
// typing secret
I.appendField('password', secret('123456'));
```

#### Parameters

*   `field` **([string][18] | [object][17])** located by label|name|CSS|XPath|strict locator
*   `value` **[string][18]** text value to append.

Returns **void** automatically synchronized promise through #recorder


This action supports [React locators](https://codecept.io/react#locators)


### attachFile

Appium: not tested

Attaches a file to element located by label, name, CSS or XPath
Path to file is relative current codecept directory (where codecept.conf.ts or codecept.conf.js is located).
File will be uploaded to remote system (if tests are running remotely).

```js
I.attachFile('Avatar', 'data/avatar.jpg');
I.attachFile('form input[name=avatar]', 'data/avatar.jpg');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** field located by label|name|CSS|XPath|strict locator.
*   `pathToFile` **[string][18]** local file path relative to codecept.conf.ts or codecept.conf.js config file.

Returns **void** automatically synchronized promise through #recorder

### blur

Remove focus from a text input, button, etc.
Calls [blur][20] on the element.

Examples:

```js
I.blur('.text-area')
```

```js
//element `#product-tile` is focused
I.see('#add-to-cart-btn');
I.blur('#product-tile')
I.dontSee('#add-to-cart-btn');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** field located by label|name|CSS|XPath|strict locator.
*   `options` **any?** Playwright only: [Additional options][21] for available options object as 2nd argument.

Returns **void** automatically synchronized promise through #recorder

### cancelPopup

Dismisses the active JavaScript popup, as created by `window.alert|window.confirm|window.prompt`.

### checkOption

Appium: not tested
Selects a checkbox or radio button.
Element is located by label or name or CSS or XPath.

The second parameter is a context (CSS or XPath locator) to narrow the search.

```js
I.checkOption('#agree');
I.checkOption('I Agree to Terms and Conditions');
I.checkOption('agree', '//form');
```

#### Parameters

*   `field` **([string][18] | [object][17])** checkbox located by label | name | CSS | XPath | strict locator.
*   `context` **([string][18]? | [object][17])** (optional, `null` by default) element located by CSS | XPath | strict locator. 

Returns **void** automatically synchronized promise through #recorder

### clearCookie

Clears a cookie by name,
if none provided clears all cookies.

```js
I.clearCookie();
I.clearCookie('test');
```

#### Parameters

*   `cookie` **[string][18]?** (optional, `null` by default) cookie name 

### clearField

Clears a `<textarea>` or text `<input>` element's value.

```js
I.clearField('Email');
I.clearField('user[email]');
I.clearField('#email');
```

#### Parameters

*   `field` &#x20;
*   `editable` **([string][18] | [object][17])** field located by label|name|CSS|XPath|strict locator.

Returns **void** automatically synchronized promise through #recorder.

### click

Perform a click on a link or a button, given by a locator.
If a fuzzy locator is given, the page will be searched for a button, link, or image matching the locator string.
For buttons, the "value" attribute, "name" attribute, and inner text are searched. For links, the link text is searched.
For images, the "alt" attribute and inner text of any parent links are searched.

The second parameter is a context (CSS or XPath locator) to narrow the search.

```js
// simple link
I.click('Logout');
// button of form
I.click('Submit');
// CSS button
I.click('#form input[type=submit]');
// XPath
I.click('//form/*[@type=submit]');
// link in context
I.click('Logout', '#nav');
// using strict locator
I.click({css: 'nav a.login'});
```

#### Parameters

*   `locator` **([string][18] | [object][17])** clickable link or button located by text, or any element located by CSS|XPath|strict locator.
*   `context` **([string][18]? | [object][17] | null)** (optional, `null` by default) element to search in CSS|XPath|Strict locator. 

Returns **void** automatically synchronized promise through #recorder


This action supports [React locators](https://codecept.io/react#locators)


### closeCurrentTab

Close current tab.

```js
I.closeCurrentTab();
```

Returns **void** automatically synchronized promise through #recorder

### closeOtherTabs

Close all tabs except for the current one.

```js
I.closeOtherTabs();
```

Returns **void** automatically synchronized promise through #recorder

### defineTimeout

Set [WebDriver timeouts][22] in realtime.

Timeouts are expected to be passed as object:

```js
I.defineTimeout({ script: 5000 });
I.defineTimeout({ implicit: 10000, pageLoad: 10000, script: 5000 });
```

#### Parameters

*   `timeouts` **any** WebDriver timeouts object.

### dontSee

Opposite to `see`. Checks that a text is not present on a page.
Use context parameter to narrow down the search.

```js
I.dontSee('Login'); // assume we are already logged in.
I.dontSee('Login', '.nav'); // no login inside .nav element
```

#### Parameters

*   `text` **[string][18]** which is not present.
*   `context` **([string][18] | [object][17])?** (optional) element located by CSS|XPath|strict locator in which to perfrom search. 

Returns **void** automatically synchronized promise through #recorder


This action supports [React locators](https://codecept.io/react#locators)


### dontSeeCheckboxIsChecked

Appium: not tested
Verifies that the specified checkbox is not checked.

```js
I.dontSeeCheckboxIsChecked('#agree'); // located by ID
I.dontSeeCheckboxIsChecked('I agree to terms'); // located by label
I.dontSeeCheckboxIsChecked('agree'); // located by name
```

#### Parameters

*   `field` **([string][18] | [object][17])** located by label|name|CSS|XPath|strict locator.

Returns **void** automatically synchronized promise through #recorder

### dontSeeCookie

Checks that cookie with given name does not exist.

```js
I.dontSeeCookie('auth'); // no auth cookie
```

#### Parameters

*   `name` **[string][18]** cookie name.

Returns **void** automatically synchronized promise through #recorder

### dontSeeCurrentUrlEquals

Checks that current url is not equal to provided one.
If a relative url provided, a configured url will be prepended to it.

```js
I.dontSeeCurrentUrlEquals('/login'); // relative url are ok
I.dontSeeCurrentUrlEquals('http://mysite.com/login'); // absolute urls are also ok
```

#### Parameters

*   `url` **[string][18]** value to check.

Returns **void** automatically synchronized promise through #recorder

### dontSeeElement

Opposite to `seeElement`. Checks that element is not visible (or in DOM)

```js
I.dontSeeElement('.modal'); // modal is not shown
```

#### Parameters

*   `locator` **([string][18] | [object][17])** located by CSS|XPath|Strict locator.

Returns **void** automatically synchronized promise through #recorder


This action supports [React locators](https://codecept.io/react#locators)


### dontSeeElementInDOM

Opposite to `seeElementInDOM`. Checks that element is not on page.

```js
I.dontSeeElementInDOM('.nav'); // checks that element is not on page visible or not
```

#### Parameters

*   `locator` **([string][18] | [object][17])** located by CSS|XPath|Strict locator.

Returns **void** automatically synchronized promise through #recorder

### dontSeeInCurrentUrl

Checks that current url does not contain a provided fragment.

#### Parameters

*   `url` **[string][18]** value to check.

Returns **void** automatically synchronized promise through #recorder

### dontSeeInField

Checks that value of input field or textarea doesn't equal to given value
Opposite to `seeInField`.

```js
I.dontSeeInField('email', 'user@user.com'); // field by name
I.dontSeeInField({ css: 'form input.email' }, 'user@user.com'); // field by CSS
```

#### Parameters

*   `field` **([string][18] | [object][17])** located by label|name|CSS|XPath|strict locator.
*   `value` **([string][18] | [object][17])** value to check.

Returns **void** automatically synchronized promise through #recorder

### dontSeeInSource

Checks that the current page does not contains the given string in its raw source code.

```js
I.dontSeeInSource('<!--'); // no comments in source
```

#### Parameters

*   `text` &#x20;
*   `value` **[string][18]** to check.

Returns **void** automatically synchronized promise through #recorder

### dontSeeInTitle

Checks that title does not contain text.

```js
I.dontSeeInTitle('Error');
```

#### Parameters

*   `text` **[string][18]** value to check.

Returns **void** automatically synchronized promise through #recorder

### doubleClick

Performs a double-click on an element matched by link|button|label|CSS or XPath.
Context can be specified as second parameter to narrow search.

```js
I.doubleClick('Edit');
I.doubleClick('Edit', '.actions');
I.doubleClick({css: 'button.accept'});
I.doubleClick('.btn.edit');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** clickable link or button located by text, or any element located by CSS|XPath|strict locator.
*   `context` **([string][18]? | [object][17])** (optional, `null` by default) element to search in CSS|XPath|Strict locator. 

Returns **void** automatically synchronized promise through #recorder


This action supports [React locators](https://codecept.io/react#locators)


### dragAndDrop

Appium: not tested
Drag an item to a destination element.

```js
I.dragAndDrop('#dragHandle', '#container');
```

#### Parameters

*   `srcElement` **([string][18] | [object][17])** located by CSS|XPath|strict locator.
*   `destElement` **([string][18] | [object][17])** located by CSS|XPath|strict locator.

Returns **void** automatically synchronized promise through #recorder

### dragSlider

Drag the scrubber of a slider to a given position
For fuzzy locators, fields are matched by label text, the "name" attribute, CSS, and XPath.

```js
I.dragSlider('#slider', 30);
I.dragSlider('#slider', -70);
```

#### Parameters

*   `locator` **([string][18] | [object][17])** located by label|name|CSS|XPath|strict locator.
*   `offsetX` **[number][23]** position to drag. 

Returns **void** automatically synchronized promise through #recorder

### executeAsyncScript

Executes async script on page.
Provided function should execute a passed callback (as first argument) to signal it is finished.

Example: In Vue.js to make components completely rendered we are waiting for [nextTick][24].

```js
I.executeAsyncScript(function(done) {
  Vue.nextTick(done); // waiting for next tick
});
```

By passing value to `done()` function you can return values.
Additional arguments can be passed as well, while `done` function is always last parameter in arguments list.

```js
let val = await I.executeAsyncScript(function(url, done) {
  // in browser context
  $.ajax(url, { success: (data) => done(data); }
}, 'http://ajax.callback.url/');
```

#### Parameters

*   `args` **...any** to be passed to function.
*   `fn` **([string][18] | [function][25])** function to be executed in browser context.

Returns **[Promise][26]<any>** script return value

### executeScript

Wraps [execute][27] command.

Executes sync script on a page.
Pass arguments to function as additional parameters.
Will return execution result to a test.
In this case you should use async function and await to receive results.

Example with jQuery DatePicker:

```js
// change date of jQuery DatePicker
I.executeScript(function() {
  // now we are inside browser context
  $('date').datetimepicker('setDate', new Date());
});
```

Can return values. Don't forget to use `await` to get them.

```js
let date = await I.executeScript(function(el) {
  // only basic types can be returned
  return $(el).datetimepicker('getDate').toString();
}, '#date'); // passing jquery selector
```

#### Parameters

*   `args` **...any** to be passed to function.
*   `fn` **([string][18] | [function][25])** function to be executed in browser context.

Returns **[Promise][26]<any>** script return value

### fillField

Fills a text field or textarea, after clearing its value, with the given string.
Field is located by name, label, CSS, or XPath.

```js
// by label
I.fillField('Email', 'hello@world.com');
// by name
I.fillField('password', secret('123456'));
// by CSS
I.fillField('form#login input[name=username]', 'John');
// or by strict locator
I.fillField({css: 'form#login input[name=username]'}, 'John');
```

#### Parameters

*   `field` **([string][18] | [object][17])** located by label|name|CSS|XPath|strict locator.
*   `value` **([string][18] | [object][17])** text value to fill.

Returns **void** automatically synchronized promise through #recorder


This action supports [React locators](https://codecept.io/react#locators)

{{ custom }}

### focus

Calls [focus][20] on the matching element.

Examples:

```js
I.dontSee('#add-to-cart-btn');
I.focus('#product-tile')
I.see('#add-to-cart-bnt');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** field located by label|name|CSS|XPath|strict locator.
*   `options` **any?** Playwright only: [Additional options][28] for available options object as 2nd argument.

Returns **void** automatically synchronized promise through #recorder

### forceClick

Perform an emulated click on a link or a button, given by a locator.
Unlike normal click instead of sending native event, emulates a click with JavaScript.
This works on hidden, animated or inactive elements as well.

If a fuzzy locator is given, the page will be searched for a button, link, or image matching the locator string.
For buttons, the "value" attribute, "name" attribute, and inner text are searched. For links, the link text is searched.
For images, the "alt" attribute and inner text of any parent links are searched.

The second parameter is a context (CSS or XPath locator) to narrow the search.

```js
// simple link
I.forceClick('Logout');
// button of form
I.forceClick('Submit');
// CSS button
I.forceClick('#form input[type=submit]');
// XPath
I.forceClick('//form/*[@type=submit]');
// link in context
I.forceClick('Logout', '#nav');
// using strict locator
I.forceClick({css: 'nav a.login'});
```

#### Parameters

*   `locator` **([string][18] | [object][17])** clickable link or button located by text, or any element located by CSS|XPath|strict locator.
*   `context` **([string][18]? | [object][17])** (optional, `null` by default) element to search in CSS|XPath|Strict locator. 

Returns **void** automatically synchronized promise through #recorder


This action supports [React locators](https://codecept.io/react#locators)


### forceRightClick

Emulates right click on an element.
Unlike normal click instead of sending native event, emulates a click with JavaScript.
This works on hidden, animated or inactive elements as well.

If a fuzzy locator is given, the page will be searched for a button, link, or image matching the locator string.
For buttons, the "value" attribute, "name" attribute, and inner text are searched. For links, the link text is searched.
For images, the "alt" attribute and inner text of any parent links are searched.

The second parameter is a context (CSS or XPath locator) to narrow the search.

```js
// simple link
I.forceRightClick('Menu');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** clickable link or button located by text, or any element located by CSS|XPath|strict locator.
*   `context` **([string][18]? | [object][17])** (optional, `null` by default) element to search in CSS|XPath|Strict locator. 

Returns **void** automatically synchronized promise through #recorder


This action supports [React locators](https://codecept.io/react#locators)


### grabAllWindowHandles

Get all Window Handles.
Useful for referencing a specific handle when calling `I.switchToWindow(handle)`

```js
const windows = await I.grabAllWindowHandles();
```

Returns **[Promise][26]<[Array][29]<[string][18]>>**&#x20;

### grabAttributeFrom

Retrieves an attribute from an element located by CSS or XPath and returns it to test.
Resumes test execution, so **should be used inside async with `await`** operator.
If more than one element is found - attribute of first element is returned.

```js
let hint = await I.grabAttributeFrom('#tooltip', 'title');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `attr` **[string][18]** attribute name.

Returns **[Promise][26]<[string][18]>** attribute value

### grabAttributeFromAll

Retrieves an array of attributes from elements located by CSS or XPath and returns it to test.
Resumes test execution, so **should be used inside async with `await`** operator.

```js
let hints = await I.grabAttributeFromAll('.tooltip', 'title');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `attr` **[string][18]** attribute name.

Returns **[Promise][26]<[Array][29]<[string][18]>>** attribute value

### grabBrowserLogs

Get JS log from browser. Log buffer is reset after each request.
Resumes test execution, so **should be used inside an async function with `await`** operator.

```js
let logs = await I.grabBrowserLogs();
console.log(JSON.stringify(logs))
```

Returns **([Promise][26]<[Array][29]<[object][17]>> | [undefined][30])** all browser logs

### grabCookie

Gets a cookie object by name.
If none provided gets all cookies.
Resumes test execution, so **should be used inside async function with `await`** operator.

```js
let cookie = await I.grabCookie('auth');
assert(cookie.value, '123456');
```

#### Parameters

*   `name` **[string][18]?** cookie name. 

Returns **any** attribute value

### grabCssPropertyFrom

Grab CSS property for given locator
Resumes test execution, so **should be used inside an async function with `await`** operator.
If more than one element is found - value of first element is returned.

```js
const value = await I.grabCssPropertyFrom('h3', 'font-weight');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `cssProperty` **[string][18]** CSS property name.

Returns **[Promise][26]<[string][18]>** CSS value

### grabCssPropertyFromAll

Grab array of CSS properties for given locator
Resumes test execution, so **should be used inside an async function with `await`** operator.

```js
const values = await I.grabCssPropertyFromAll('h3', 'font-weight');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `cssProperty` **[string][18]** CSS property name.

Returns **[Promise][26]<[Array][29]<[string][18]>>** CSS value

### grabCurrentUrl

Get current URL from browser.
Resumes test execution, so should be used inside an async function.

```js
let url = await I.grabCurrentUrl();
console.log(`Current URL is [${url}]`);
```

Returns **[Promise][26]<[string][18]>** current URL

### grabCurrentWindowHandle

Get the current Window Handle.
Useful for referencing it when calling `I.switchToWindow(handle)`

```js
const window = await I.grabCurrentWindowHandle();
```

Returns **[Promise][26]<[string][18]>**&#x20;

### grabElementBoundingRect

Grab the width, height, location of given locator.
Provide `width` or `height`as second param to get your desired prop.
Resumes test execution, so **should be used inside an async function with `await`** operator.

Returns an object with `x`, `y`, `width`, `height` keys.

```js
const value = await I.grabElementBoundingRect('h3');
// value is like { x: 226.5, y: 89, width: 527, height: 220 }
```

To get only one metric use second parameter:

```js
const width = await I.grabElementBoundingRect('h3', 'width');
// width == 527
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `prop` &#x20;
*   `elementSize` **[string][18]?** x, y, width or height of the given element.

Returns **([Promise][26]<DOMRect> | [Promise][26]<[number][23]>)** Element bounding rectangle

### grabHTMLFrom

Retrieves the innerHTML from an element located by CSS or XPath and returns it to test.
Resumes test execution, so **should be used inside async function with `await`** operator.
If more than one element is found - HTML of first element is returned.

```js
let postHTML = await I.grabHTMLFrom('#post');
```

#### Parameters

*   `locator` &#x20;
*   `element` **([string][18] | [object][17])** located by CSS|XPath|strict locator.

Returns **[Promise][26]<[string][18]>** HTML code for an element

### grabHTMLFromAll

Retrieves all the innerHTML from elements located by CSS or XPath and returns it to test.
Resumes test execution, so **should be used inside async function with `await`** operator.

```js
let postHTMLs = await I.grabHTMLFromAll('.post');
```

#### Parameters

*   `locator` &#x20;
*   `element` **([string][18] | [object][17])** located by CSS|XPath|strict locator.

Returns **[Promise][26]<[Array][29]<[string][18]>>** HTML code for an element

### grabNumberOfOpenTabs

Grab number of open tabs.
Resumes test execution, so **should be used inside async function with `await`** operator.

```js
let tabs = await I.grabNumberOfOpenTabs();
```

Returns **[Promise][26]<[number][23]>** number of open tabs

### grabNumberOfVisibleElements

Grab number of visible elements by locator.
Resumes test execution, so **should be used inside async function with `await`** operator.

```js
let numOfElements = await I.grabNumberOfVisibleElements('p');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** located by CSS|XPath|strict locator.

Returns **[Promise][26]<[number][23]>** number of visible elements

### grabPageScrollPosition

Retrieves a page scroll position and returns it to test.
Resumes test execution, so **should be used inside an async function with `await`** operator.

```js
let { x, y } = await I.grabPageScrollPosition();
```

Returns **[Promise][26]<PageScrollPosition>** scroll position

### grabPopupText

Grab the text within the popup. If no popup is visible then it will return null.

```js
await I.grabPopupText();
```

Returns **[Promise][26]<[string][18]>**&#x20;

### grabSource

Retrieves page source and returns it to test.
Resumes test execution, so **should be used inside async function with `await`** operator.

```js
let pageSource = await I.grabSource();
```

Returns **[Promise][26]<[string][18]>** source code

### grabTextFrom

Retrieves a text from an element located by CSS or XPath and returns it to test.
Resumes test execution, so **should be used inside async with `await`** operator.

```js
let pin = await I.grabTextFrom('#pin');
```

If multiple elements found returns first element.

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.

Returns **[Promise][26]<[string][18]>** attribute value

### grabTextFromAll

Retrieves all texts from an element located by CSS or XPath and returns it to test.
Resumes test execution, so **should be used inside async with `await`** operator.

```js
let pins = await I.grabTextFromAll('#pin li');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.

Returns **[Promise][26]<[Array][29]<[string][18]>>** attribute value

### grabTitle

Retrieves a page title and returns it to test.
Resumes test execution, so **should be used inside async with `await`** operator.

```js
let title = await I.grabTitle();
```

Returns **[Promise][26]<[string][18]>** title

### grabValueFrom

Retrieves a value from a form element located by CSS or XPath and returns it to test.
Resumes test execution, so **should be used inside async function with `await`** operator.
If more than one element is found - value of first element is returned.

```js
let email = await I.grabValueFrom('input[name=email]');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** field located by label|name|CSS|XPath|strict locator.

Returns **[Promise][26]<[string][18]>** attribute value

### grabValueFromAll

Retrieves an array of value from a form located by CSS or XPath and returns it to test.
Resumes test execution, so **should be used inside async function with `await`** operator.

```js
let inputs = await I.grabValueFromAll('//form/input');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** field located by label|name|CSS|XPath|strict locator.

Returns **[Promise][26]<[Array][29]<[string][18]>>** attribute value

### grabWebElements

Grab WebElements for given locator
Resumes test execution, so **should be used inside an async function with `await`** operator.

```js
const webElements = await I.grabWebElements('#button');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.

Returns **[Promise][26]<any>** WebElement of being used Web helper

### moveCursorTo

Moves cursor to element matched by locator.
Extra shift can be set with offsetX and offsetY options.

```js
I.moveCursorTo('.tooltip');
I.moveCursorTo('#submit', 5,5);
```

#### Parameters

*   `locator` **([string][18] | [object][17])** located by CSS|XPath|strict locator.
*   `xOffset` &#x20;
*   `yOffset` &#x20;
*   `offsetX` **[number][23]** (optional, `0` by default) X-axis offset. 
*   `offsetY` **[number][23]** (optional, `0` by default) Y-axis offset. 

Returns **void** automatically synchronized promise through #recorder

### openNewTab

Open new tab and switch to it.

```js
I.openNewTab();
```

#### Parameters

*   `url`   
*   `windowName`   

Returns **void** automatically synchronized promise through #recorder

### pressKey

*Note:* In case a text field or textarea is focused be aware that some browsers do not respect active modifier when combining modifier keys with other keys.

Presses a key in the browser (on a focused element).

*Hint:* For populating text field or textarea, it is recommended to use [`fillField`][31].

```js
I.pressKey('Backspace');
```

To press a key in combination with modifier keys, pass the sequence as an array. All modifier keys (`'Alt'`, `'Control'`, `'Meta'`, `'Shift'`) will be released afterwards.

```js
I.pressKey(['Control', 'Z']);
```

For specifying operation modifier key based on operating system it is suggested to use `'CommandOrControl'`.
This will press `'Command'` (also known as `'Meta'`) on macOS machines and `'Control'` on non-macOS machines.

```js
I.pressKey(['CommandOrControl', 'Z']);
```

Some of the supported key names are:

*   `'AltLeft'` or `'Alt'`
*   `'AltRight'`
*   `'ArrowDown'`
*   `'ArrowLeft'`
*   `'ArrowRight'`
*   `'ArrowUp'`
*   `'Backspace'`
*   `'Clear'`
*   `'ControlLeft'` or `'Control'`
*   `'ControlRight'`
*   `'Command'`
*   `'CommandOrControl'`
*   `'Delete'`
*   `'End'`
*   `'Enter'`
*   `'Escape'`
*   `'F1'` to `'F12'`
*   `'Home'`
*   `'Insert'`
*   `'MetaLeft'` or `'Meta'`
*   `'MetaRight'`
*   `'Numpad0'` to `'Numpad9'`
*   `'NumpadAdd'`
*   `'NumpadDecimal'`
*   `'NumpadDivide'`
*   `'NumpadMultiply'`
*   `'NumpadSubtract'`
*   `'PageDown'`
*   `'PageUp'`
*   `'Pause'`
*   `'Return'`
*   `'ShiftLeft'` or `'Shift'`
*   `'ShiftRight'`
*   `'Space'`
*   `'Tab'`

#### Parameters

*   `key` **([string][18] | [Array][29]<[string][18]>)** key or array of keys to press.

Returns **void** automatically synchronized promise through #recorder

### pressKeyDown

Presses a key in the browser and leaves it in a down state.

To make combinations with modifier key and user operation (e.g. `'Control'` + [`click`][32]).

```js
I.pressKeyDown('Control');
I.click('#element');
I.pressKeyUp('Control');
```

#### Parameters

*   `key` **[string][18]** name of key to press down.

Returns **void** automatically synchronized promise through #recorder

### pressKeyUp

Releases a key in the browser which was previously set to a down state.

To make combinations with modifier key and user operation (e.g. `'Control'` + [`click`][32]).

```js
I.pressKeyDown('Control');
I.click('#element');
I.pressKeyUp('Control');
```

#### Parameters

*   `key` **[string][18]** name of key to release.

Returns **void** automatically synchronized promise through #recorder

### refreshPage

Reload the current page.

```js
I.refreshPage();
```

Returns **void** automatically synchronized promise through #recorder

### resizeWindow

Appium: not tested in web, in apps doesn't work

Resize the current window to provided width and height.
First parameter can be set to `maximize`.

#### Parameters

*   `width` **[number][23]** width in pixels or `maximize`.
*   `height` **[number][23]** height in pixels.

Returns **void** automatically synchronized promise through #recorder

### rightClick

Performs right click on a clickable element matched by semantic locator, CSS or XPath.

```js
// right click element with id el
I.rightClick('#el');
// right click link or button with text "Click me"
I.rightClick('Click me');
// right click button with text "Click me" inside .context
I.rightClick('Click me', '.context');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** clickable element located by CSS|XPath|strict locator.
*   `context` **([string][18]? | [object][17])** (optional, `null` by default) element located by CSS|XPath|strict locator. 

Returns **void** automatically synchronized promise through #recorder


This action supports [React locators](https://codecept.io/react#locators)


### runInWeb

Placeholder for ~ locator only test case write once run on both Appium and WebDriver.

#### Parameters

*   `fn` &#x20;

### runOnAndroid

Placeholder for ~ locator only test case write once run on both Appium and WebDriver.

#### Parameters

*   `caps` **any**&#x20;
*   `fn` **any**&#x20;

### runOnIOS

Placeholder for ~ locator only test case write once run on both Appium and WebDriver.

#### Parameters

*   `caps` **any**&#x20;
*   `fn` **any**&#x20;

### saveElementScreenshot

Saves screenshot of the specified locator to ouput folder (set in codecept.conf.ts or codecept.conf.js).
Filename is relative to output folder.

```js
I.saveElementScreenshot(`#submit`,'debug.png');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `fileName` **[string][18]** file name to save.

Returns **void** automatically synchronized promise through #recorder

### saveScreenshot

Saves a screenshot to ouput folder (set in codecept.conf.ts or codecept.conf.js).
Filename is relative to output folder.
Optionally resize the window to the full available page `scrollHeight` and `scrollWidth` to capture the entire page by passing `true` in as the second argument.

```js
I.saveScreenshot('debug.png');
I.saveScreenshot('debug.png', true) //resizes to available scrollHeight and scrollWidth before taking screenshot
```

#### Parameters

*   `fileName` **[string][18]** file name to save.
*   `fullPage` **[boolean][33]** (optional, `false` by default) flag to enable fullscreen screenshot mode. 

Returns **void** automatically synchronized promise through #recorder

### scrollIntoView

Scroll element into viewport.

```js
I.scrollIntoView('#submit');
I.scrollIntoView('#submit', true);
I.scrollIntoView('#submit', { behavior: "smooth", block: "center", inline: "center" });
```

#### Parameters

*   `locator` **([string][18] | [object][17])** located by CSS|XPath|strict locator.
*   `scrollIntoViewOptions` **(ScrollIntoViewOptions | [boolean][33])** either alignToTop=true|false or scrollIntoViewOptions. See [https://developer.mozilla.org/en-US/docs/Web/API/Element/scrollIntoView][34].

Returns **void** automatically synchronized promise through #recorder

### scrollPageToBottom

Scroll page to the bottom.

```js
I.scrollPageToBottom();
```

Returns **void** automatically synchronized promise through #recorder

### scrollPageToTop

Scroll page to the top.

```js
I.scrollPageToTop();
```

Returns **void** automatically synchronized promise through #recorder

### scrollTo

Scrolls to element matched by locator.
Extra shift can be set with offsetX and offsetY options.

```js
I.scrollTo('footer');
I.scrollTo('#submit', 5, 5);
```

#### Parameters

*   `locator` **([string][18] | [object][17])** located by CSS|XPath|strict locator.
*   `offsetX` **[number][23]** (optional, `0` by default) X-axis offset. 
*   `offsetY` **[number][23]** (optional, `0` by default) Y-axis offset. 

Returns **void** automatically synchronized promise through #recorder

### see

Checks that a page contains a visible text.
Use context parameter to narrow down the search.

```js
I.see('Welcome'); // text welcome on a page
I.see('Welcome', '.content'); // text inside .content div
I.see('Register', {css: 'form.register'}); // use strict locator
```

#### Parameters

*   `text` **[string][18]** expected on page.
*   `context` **([string][18]? | [object][17])** (optional, `null` by default) element located by CSS|Xpath|strict locator in which to search for text. 

Returns **void** automatically synchronized promise through #recorder


This action supports [React locators](https://codecept.io/react#locators)


### seeAttributesOnElements

Checks that all elements with given locator have given attributes.

```js
I.seeAttributesOnElements('//form', { method: "post"});
```

#### Parameters

*   `locator` **([string][18] | [object][17])** located by CSS|XPath|strict locator.
*   `attributes` **[object][17]** attributes and their values to check.

Returns **void** automatically synchronized promise through #recorder

### seeCheckboxIsChecked

Appium: not tested
Verifies that the specified checkbox is checked.

```js
I.seeCheckboxIsChecked('Agree');
I.seeCheckboxIsChecked('#agree'); // I suppose user agreed to terms
I.seeCheckboxIsChecked({css: '#signup_form input[type=checkbox]'});
```

#### Parameters

*   `field` **([string][18] | [object][17])** located by label|name|CSS|XPath|strict locator.

Returns **void** automatically synchronized promise through #recorder

### seeCookie

Checks that cookie with given name exists.

```js
I.seeCookie('Auth');
```

#### Parameters

*   `name` **[string][18]** cookie name.

Returns **void** automatically synchronized promise through #recorder

### seeCssPropertiesOnElements

Checks that all elements with given locator have given CSS properties.

```js
I.seeCssPropertiesOnElements('h3', { 'font-weight': "bold"});
```

#### Parameters

*   `locator` **([string][18] | [object][17])** located by CSS|XPath|strict locator.
*   `cssProperties` **[object][17]** object with CSS properties and their values to check.

Returns **void** automatically synchronized promise through #recorder

### seeCurrentUrlEquals

Checks that current url is equal to provided one.
If a relative url provided, a configured url will be prepended to it.
So both examples will work:

```js
I.seeCurrentUrlEquals('/register');
I.seeCurrentUrlEquals('http://my.site.com/register');
```

#### Parameters

*   `url` **[string][18]** value to check.

Returns **void** automatically synchronized promise through #recorder

### seeElement

Checks that a given Element is visible
Element is located by CSS or XPath.

```js
I.seeElement('#modal');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** located by CSS|XPath|strict locator.

Returns **void** automatically synchronized promise through #recorder


This action supports [React locators](https://codecept.io/react#locators)


### seeElementInDOM

Checks that a given Element is present in the DOM
Element is located by CSS or XPath.

```js
I.seeElementInDOM('#modal');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.

Returns **void** automatically synchronized promise through #recorder

### seeInCurrentUrl

Checks that current url contains a provided fragment.

```js
I.seeInCurrentUrl('/register'); // we are on registration page
```

#### Parameters

*   `url` **[string][18]** a fragment to check

Returns **void** automatically synchronized promise through #recorder

### seeInField

Checks that the given input field or textarea equals to given value.
For fuzzy locators, fields are matched by label text, the "name" attribute, CSS, and XPath.

```js
I.seeInField('Username', 'davert');
I.seeInField({css: 'form textarea'},'Type your comment here');
I.seeInField('form input[type=hidden]','hidden_value');
I.seeInField('#searchform input','Search');
```

#### Parameters

*   `field` **([string][18] | [object][17])** located by label|name|CSS|XPath|strict locator.
*   `value` **([string][18] | [object][17])** value to check.

Returns **void** automatically synchronized promise through #recorder

### seeInPopup

Checks that the active JavaScript popup, as created by `window.alert|window.confirm|window.prompt`, contains the
given string.

#### Parameters

*   `text` **[string][18]** value to check.

### seeInSource

Checks that the current page contains the given string in its raw source code.

```js
I.seeInSource('<h1>Green eggs &amp; ham</h1>');
```

#### Parameters

*   `text` **[string][18]** value to check.

Returns **void** automatically synchronized promise through #recorder

### seeInTitle

Checks that title contains text.

```js
I.seeInTitle('Home Page');
```

#### Parameters

*   `text` **[string][18]** text value to check.

Returns **void** automatically synchronized promise through #recorder

### seeNumberOfElements

Asserts that an element appears a given number of times in the DOM.
Element is located by label or name or CSS or XPath.

```js
I.seeNumberOfElements('#submitBtn', 1);
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `num` **[number][23]** number of elements.

Returns **void** automatically synchronized promise through #recorder


This action supports [React locators](https://codecept.io/react#locators)


### seeNumberOfVisibleElements

Asserts that an element is visible a given number of times.
Element is located by CSS or XPath.

```js
I.seeNumberOfVisibleElements('.buttons', 3);
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `num` **[number][23]** number of elements.

Returns **void** automatically synchronized promise through #recorder


This action supports [React locators](https://codecept.io/react#locators)


### seeTextEquals

Checks that text is equal to provided one.

```js
I.seeTextEquals('text', 'h1');
```

#### Parameters

*   `text` **[string][18]** element value to check.
*   `context` **([string][18] | [object][17])?** element located by CSS|XPath|strict locator. 

Returns **void** automatically synchronized promise through #recorder

### seeTitleEquals

Checks that title is equal to provided one.

```js
I.seeTitleEquals('Test title.');
```

#### Parameters

*   `text` **[string][18]** value to check.

Returns **void** automatically synchronized promise through #recorder

### selectOption

Selects an option in a drop-down select.
Field is searched by label | name | CSS | XPath.
Option is selected by visible text or by value.

```js
I.selectOption('Choose Plan', 'Monthly'); // select by label
I.selectOption('subscription', 'Monthly'); // match option by text
I.selectOption('subscription', '0'); // or by value
I.selectOption('//form/select[@name=account]','Premium');
I.selectOption('form select[name=account]', 'Premium');
I.selectOption({css: 'form select[name=account]'}, 'Premium');
```

Provide an array for the second argument to select multiple options.

```js
I.selectOption('Which OS do you use?', ['Android', 'iOS']);
```

#### Parameters

*   `select` **([string][18] | [object][17])** field located by label|name|CSS|XPath|strict locator.
*   `option` **([string][18] | [Array][29]<any>)** visible text or value of option.

Returns **void** automatically synchronized promise through #recorder

### setCookie

Uses Selenium's JSON [cookie format][35].
Sets cookie(s).

Can be a single cookie object or an array of cookies:

```js
I.setCookie({name: 'auth', value: true});

// as array
I.setCookie([
  {name: 'auth', value: true},
  {name: 'agree', value: true}
]);
```

#### Parameters

*   `cookie` **(Cookie | [Array][29]<Cookie>)** a cookie object or array of cookie objects.

Returns **void** automatically synchronized promise through #recorder

### switchTo

Switches frame or in case of null locator reverts to parent.

```js
I.switchTo('iframe'); // switch to first iframe
I.switchTo(); // switch back to main page
```

#### Parameters

*   `locator` **([string][18]? | [object][17])** (optional, `null` by default) element located by CSS|XPath|strict locator. 

Returns **void** automatically synchronized promise through #recorder

### switchToNextTab

Switch focus to a particular tab by its number. It waits tabs loading and then switch tab.

```js
I.switchToNextTab();
I.switchToNextTab(2);
```

#### Parameters

*   `num` **[number][23]?** (optional) number of tabs to switch forward, default: 1. 
*   `sec` **([number][23] | null)?** (optional) time in seconds to wait. 

Returns **void** automatically synchronized promise through #recorder

### switchToPreviousTab

Switch focus to a particular tab by its number. It waits tabs loading and then switch tab.

```js
I.switchToPreviousTab();
I.switchToPreviousTab(2);
```

#### Parameters

*   `num` **[number][23]?** (optional) number of tabs to switch backward, default: 1. 
*   `sec` **[number][23]??** (optional) time in seconds to wait. 

Returns **void** automatically synchronized promise through #recorder

### switchToWindow

Switch to the window with a specified handle.

```js
const windows = await I.grabAllWindowHandles();
// ... do something
await I.switchToWindow( windows[0] );

const window = await I.grabCurrentWindowHandle();
// ... do something
await I.switchToWindow( window );
```

#### Parameters

*   `window` **[string][18]** name of window handle.

### type

Types out the given text into an active field.
To slow down typing use a second parameter, to set interval between key presses.
*Note:* Should be used when [`fillField`][31] is not an option.

```js
// passing in a string
I.type('Type this out.');

// typing values with a 100ms interval
I.type('4141555311111111', 100);

// passing in an array
I.type(['T', 'E', 'X', 'T']);

// passing a secret
I.type(secret('123456'));
```

#### Parameters

*   `keys` &#x20;
*   `delay` **[number][23]?** (optional) delay in ms between key presses 
*   `key` **([string][18] | [Array][29]<[string][18]>)** or array of keys to type.

Returns **void** automatically synchronized promise through #recorder

### uncheckOption

Appium: not tested
Unselects a checkbox or radio button.
Element is located by label or name or CSS or XPath.

The second parameter is a context (CSS or XPath locator) to narrow the search.

```js
I.uncheckOption('#agree');
I.uncheckOption('I Agree to Terms and Conditions');
I.uncheckOption('agree', '//form');
```

#### Parameters

*   `field` **([string][18] | [object][17])** checkbox located by label | name | CSS | XPath | strict locator.
*   `context` **([string][18]? | [object][17])** (optional, `null` by default) element located by CSS | XPath | strict locator. 

Returns **void** automatically synchronized promise through #recorder

### useWebDriverTo

Use [webdriverio][36] API inside a test.

First argument is a description of an action.
Second argument is async function that gets this helper as parameter.

{ [`browser`][36]) } object from WebDriver API is available.

```js
I.useWebDriverTo('open multiple windows', async ({ browser }) {
   // create new window
   await browser.newWindow('https://webdriver.io');
});
```

#### Parameters

*   `description` **[string][18]** used to show in logs.
*   `fn` **[function][25]** async functuion that executed with WebDriver helper as argument

### wait

Pauses execution for a number of seconds.

```js
I.wait(2); // wait 2 secs
```

#### Parameters

*   `sec` **[number][23]** number of second to wait.

Returns **void** automatically synchronized promise through #recorder

### waitForClickable

Waits for element to be clickable (by default waits for 1sec).
Element can be located by CSS or XPath.

```js
I.waitForClickable('.btn.continue');
I.waitForClickable('.btn.continue', 5); // wait for 5 secs
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `waitTimeout` &#x20;
*   `sec` **[number][23]?** (optional, `1` by default) time in seconds to wait

Returns **void** automatically synchronized promise through #recorder

### waitForCookie

Waits for the specified cookie in the cookies.

```js
I.waitForCookie("token");
```

#### Parameters

*   `name` **[string][18]** expected cookie name.
*   `sec` **[number][23]** (optional, `3` by default) time in seconds to wait 

Returns **void** automatically synchronized promise through #recorder

### waitForDetached

Waits for an element to become not attached to the DOM on a page (by default waits for 1sec).
Element can be located by CSS or XPath.

```js
I.waitForDetached('#popup');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `sec` **[number][23]** (optional, `1` by default) time in seconds to wait 

Returns **void** automatically synchronized promise through #recorder

### waitForElement

Waits for element to be present on page (by default waits for 1sec).
Element can be located by CSS or XPath.

```js
I.waitForElement('.btn.continue');
I.waitForElement('.btn.continue', 5); // wait for 5 secs
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `sec` **[number][23]?** (optional, `1` by default) time in seconds to wait 

Returns **void** automatically synchronized promise through #recorder

### waitForEnabled

Waits for element to become enabled (by default waits for 1sec).
Element can be located by CSS or XPath.

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `sec` **[number][23]** (optional) time in seconds to wait, 1 by default. 

Returns **void** automatically synchronized promise through #recorder

### waitForFunction

Waits for a function to return true (waits for 1 sec by default).
Running in browser context.

```js
I.waitForFunction(fn[, [args[, timeout]])
```

```js
I.waitForFunction(() => window.requests == 0);
I.waitForFunction(() => window.requests == 0, 5); // waits for 5 sec
I.waitForFunction((count) => window.requests == count, [3], 5) // pass args and wait for 5 sec
```

#### Parameters

*   `fn` **([string][18] | [function][25])** to be executed in browser context.
*   `argsOrSec` **([Array][29]<any> | [number][23])?** (optional, `1` by default) arguments for function or seconds. 
*   `sec` **[number][23]?** (optional, `1` by default) time in seconds to wait 

Returns **void** automatically synchronized promise through #recorder

### waitForInvisible

Waits for an element to be removed or become invisible on a page (by default waits for 1sec).
Element can be located by CSS or XPath.

```js
I.waitForInvisible('#popup');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `sec` **[number][23]** (optional, `1` by default) time in seconds to wait 

Returns **void** automatically synchronized promise through #recorder

### waitForNumberOfTabs

Waits for number of tabs.

```js
I.waitForNumberOfTabs(2);
```

#### Parameters

*   `expectedTabs` **[number][23]** expecting the number of tabs.
*   `sec` **[number][23]** number of secs to wait.

Returns **void** automatically synchronized promise through #recorder

### waitForText

Waits for a text to appear (by default waits for 1sec).
Element can be located by CSS or XPath.
Narrow down search results by providing context.

```js
I.waitForText('Thank you, form has been submitted');
I.waitForText('Thank you, form has been submitted', 5, '#modal');
```

#### Parameters

*   `text` **[string][18]** to wait for.
*   `sec` **[number][23]** (optional, `1` by default) time in seconds to wait 
*   `context` **([string][18] | [object][17])?** (optional) element located by CSS|XPath|strict locator. 

Returns **void** automatically synchronized promise through #recorder

### waitForValue

Waits for the specified value to be in value attribute.

```js
I.waitForValue('//input', "GoodValue");
```

#### Parameters

*   `field` **([string][18] | [object][17])** input field.
*   `value` **[string][18]** expected value.
*   `sec` **[number][23]** (optional, `1` by default) time in seconds to wait 

Returns **void** automatically synchronized promise through #recorder

### waitForVisible

Waits for an element to become visible on a page (by default waits for 1sec).
Element can be located by CSS or XPath.

```js
I.waitForVisible('#popup');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `sec` **[number][23]** (optional, `1` by default) time in seconds to wait 

Returns **void** automatically synchronized promise through #recorder

### waitInUrl

Waiting for the part of the URL to match the expected. Useful for SPA to understand that page was changed.

```js
I.waitInUrl('/info', 2);
```

#### Parameters

*   `urlPart` **[string][18]** value to check.
*   `sec` **[number][23]** (optional, `1` by default) time in seconds to wait 

Returns **void** automatically synchronized promise through #recorder

### waitNumberOfVisibleElements

Waits for a specified number of elements on the page.

```js
I.waitNumberOfVisibleElements('a', 3);
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `num` **[number][23]** number of elements.
*   `sec` **[number][23]** (optional, `1` by default) time in seconds to wait 

Returns **void** automatically synchronized promise through #recorder

### waitToHide

Waits for an element to hide (by default waits for 1sec).
Element can be located by CSS or XPath.

```js
I.waitToHide('#popup');
```

#### Parameters

*   `locator` **([string][18] | [object][17])** element located by CSS|XPath|strict locator.
*   `sec` **[number][23]** (optional, `1` by default) time in seconds to wait 

Returns **void** automatically synchronized promise through #recorder

### waitUrlEquals

Waits for the entire URL to match the expected

```js
I.waitUrlEquals('/info', 2);
I.waitUrlEquals('http://127.0.0.1:8000/info');
```

#### Parameters

*   `urlPart` **[string][18]** value to check.
*   `sec` **[number][23]** (optional, `1` by default) time in seconds to wait 

Returns **void** automatically synchronized promise through #recorder

[1]: http://webdriver.io/

[2]: https://codecept.io/webdriver/#testing-with-webdriver

[3]: https://webdriver.io/blog/2023/07/31/driver-management/

[4]: http://webdriver.io/guide/getstarted/configuration.html

[5]: https://seleniumhq.github.io/selenium/docs/api/rb/Selenium/WebDriver/IE/Options.html

[6]: https://aerokube.com/selenoid/latest/

[7]: https://github.com/SeleniumHQ/selenium/wiki/DesiredCapabilities

[8]: http://webdriver.io/guide/usage/cloudservices.html

[9]: https://webdriver.io/docs/sauce-service.html

[10]: https://github.com/puneet0191/codeceptjs-saucehelper/

[11]: https://webdriver.io/docs/browserstack-service.html

[12]: https://github.com/PeterNgTr/codeceptjs-bshelper

[13]: https://github.com/testingbot/codeceptjs-tbhelper

[14]: https://webdriver.io/docs/testingbot-service.html

[15]: https://github.com/PeterNgTr/codeceptjs-applitoolshelper

[16]: http://webdriver.io/guide/usage/multiremote.html

[17]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Object

[18]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String

[19]: http://jster.net/category/windows-modals-popups

[20]: https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/focus

[21]: https://playwright.dev/docs/api/class-locator#locator-blur

[22]: https://webdriver.io/docs/timeouts.html

[23]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Number

[24]: https://vuejs.org/v2/api/#Vue-nextTick

[25]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Statements/function

[26]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Promise

[27]: http://webdriver.io/api/protocol/execute.html

[28]: https://playwright.dev/docs/api/class-locator#locator-focus

[29]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Array

[30]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/undefined

[31]: #fillfield

[32]: #click

[33]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Boolean

[34]: https://developer.mozilla.org/en-US/docs/Web/API/Element/scrollIntoView

[35]: https://code.google.com/p/selenium/wiki/JsonWireProtocol#Cookie_JSON_Object

[36]: https://webdriver.io/docs/api.html

[37]: http://codecept.io/acceptance/#smartwait

[38]: http://webdriver.io/docs/timeouts.html

[39]: https://webdriver.io/docs/configuration/#loglevel
