---
permalink: /helpers/JSONResponse
editLink: false
sidebar: auto
title: JSONResponse
---

<!-- Generated by documentation.js. Update this documentation by updating the source code. -->

## JSONResponse

**Extends Helper**

This helper allows performing assertions on JSON responses paired with following helpers:

*   REST
*   GraphQL
*   Playwright

It can check status codes, response data, response structure.

## Configuration

*   `requestHelper` - a helper which will perform requests. `REST` by default, also `Playwright` or `GraphQL` can be used. Custom helpers must have `onResponse` hook in their config, which will be executed when request is performed.

### Examples

Zero-configuration when paired with REST:

```js
// inside codecept.conf.js
{
  helpers: {
    REST: {
      endpoint: 'http://site.com/api',
    },
    JSONResponse: {}
  }
}
```

Explicitly setting request helper if you use `makeApiRequest` of Playwright to perform requests and not paired REST:

```js
// inside codecept.conf.js
// ...
  helpers: {
    Playwright: {
      url: 'https://localhost',
      browser: 'chromium',
    },
    JSONResponse: {
      requestHelper: 'Playwright',
    }
  }
```

## Access From Helpers

If you plan to add custom assertions it is recommended to create a helper that will retrieve response object from JSONResponse:

```js
// inside custom helper
const response = this.helpers.JSONResponse.response;
```

## Methods

### Parameters

*   `config`   

### dontSeeResponseCodeIs

Checks that response code is not equal to the provided one

```js
I.dontSeeResponseCodeIs(500);
```

#### Parameters

*   `code` **[number][1]**&#x20;

### dontSeeResponseContainsJson

Checks for deep inclusion of a provided json in a response data.

```js
// response.data == { data: { user: 1 } }

I.dontSeeResponseContainsJson({ user: 2 });
```

If an array is received, checks that no element of array contains json:

```js
// response.data == [{ user: 1 }, { user: 3 }]

I.dontSeeResponseContainsJson({ user: 2 });
```

#### Parameters

*   `json` **[object][2]**  

### seeResponseCodeIs

Checks that response code is equal to the provided one

```js
I.seeResponseCodeIs(200);
```

#### Parameters

*   `code` **[number][1]**&#x20;

### seeResponseCodeIsClientError

Checks that the response code is 4xx

### seeResponseCodeIsRedirection

Checks that the response code is 3xx

### seeResponseCodeIsServerError

Checks that the response code is 5xx

### seeResponseCodeIsSuccessful

Checks that the response code is 2xx
Use it instead of seeResponseCodeIs(200) if server can return 204 instead.

```js
I.seeResponseCodeIsSuccessful();
```

### seeResponseContainsJson

Checks for deep inclusion of a provided json in a response data.

```js
// response.data == { user: { name: 'jon', email: 'jon@doe.com' } }

I.seeResponseContainsJson({ user: { email: 'jon@doe.com' } });
```

If an array is received, checks that at least one element contains JSON

```js
// response.data == [{ user: { name: 'jon', email: 'jon@doe.com' } }]

I.seeResponseContainsJson({ user: { email: 'jon@doe.com' } });
```

#### Parameters

*   `json` **[object][2]**  

### seeResponseContainsKeys

Checks for deep inclusion of a provided json in a response data.

```js
// response.data == { user: { name: 'jon', email: 'jon@doe.com' } }

I.seeResponseContainsKeys(['user']);
```

If an array is received, check is performed for each element of array:

```js
// response.data == [{ user: 'jon' }, { user: 'matt'}]

I.seeResponseContainsKeys(['user']);
```

#### Parameters

*   `keys` **[array][3]**  

### seeResponseEquals

Checks that response data equals to expected:

```js
// response.data is { error: 'Not allowed' }

I.seeResponseEquals({ error: 'Not allowed' })
```

#### Parameters

*   `resp` **[object][2]**&#x20;

### seeResponseMatchesJsonSchema

Validates JSON structure of response using [joi library][4].
See [joi API][5] for complete reference on usage.

Use pre-initialized joi instance by passing function callback:

```js
// response.data is { name: 'jon', id: 1 }

I.seeResponseMatchesJsonSchema(joi => {
  return joi.object({
    name: joi.string(),
    id: joi.number()
  })
});

// or pass a valid schema
const joi = require('joi');

I.seeResponseMatchesJsonSchema(joi.object({
  name: joi.string(),
  id: joi.number();
});
```

#### Parameters

*   `fnOrSchema` **any**&#x20;

### seeResponseValidByCallback

Executes a callback function passing in `response` object and assert
Use it to perform custom checks of response data

```js
I.seeResponseValidByCallback(({ data, status }) => {
  assert.strictEqual(status, 200);
  assert('user' in data);
  assert('company' in data);
});
```

#### Parameters

*   `fn` **[function][6]**&#x20;

[1]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Number

[2]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Object

[3]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Array

[4]: https://joi.dev

[5]: https://joi.dev/api/

[6]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Statements/function
