---
permalink: /helpers/GraphQLDataFactory
editLink: false
sidebar: auto
title: GraphQLDataFactory
---

<!-- Generated by documentation.js. Update this documentation by updating the source code. -->

## GraphQLDataFactory

**Extends Helper**

Helper for managing remote data using GraphQL queries.
Uses data generators like [rosie][1] or factory girl to create new record.

By defining a factory you set the rules of how data is generated.
This data will be saved on server via GraphQL queries and deleted in the end of a test.

## Use Case

Acceptance tests interact with a websites using UI and real browser.
There is no way to create data for a specific test other than from user interface.
That makes tests slow and fragile. Instead of testing a single feature you need to follow all creation/removal process.

This helper solves this problem.
If a web application has GraphQL support, it can be used to create and delete test records.
By combining GraphQL with Factories you can easily create records for tests:

```js
I.mutateData('createUser', { name: 'davert', email: 'davert@mail.com' });
let user = await I.mutateData('createUser', { name: 'davert'});
I.mutateMultiple('createPost', 3, {post_id: user.id});
```

To make this work you need

1.  GraphQL endpoint which allows to perform create / delete requests and
2.  define data generation rules

### Setup

Install [Rosie][1] and [Faker][2] libraries.

```sh
npm i rosie @faker-js/faker --save-dev
```

Create a factory file for a resource.

See the example for Users factories:

```js
// tests/factories/users.js

const { Factory } = require('rosie').Factory;
const { faker } = require('@faker-js/faker');

// Used with a constructor function passed to Factory, so that the final build
// object matches the necessary pattern to be sent as the variables object.
module.exports = new Factory((buildObj) => ({
   input: { ...buildObj },
}))
   // 'attr'-id can be left out depending on the GraphQl resolvers
   .attr('name', () => faker.person.findName())
   .attr('email', () => faker.interact.email())
```

For more options see [rosie documentation][1].

Then configure GraphQLDataHelper to match factories and GraphQL schema:

### Configuration

GraphQLDataFactory has following config options:

*   `endpoint`: URL for the GraphQL server.
*   `cleanup` (default: true): should inserted records be deleted up after tests
*   `factories`: list of defined factories
*   `headers`: list of headers
*   `GraphQL`: configuration for GraphQL requests.

See the example:

```js
 GraphQLDataFactory: {
   endpoint: "http://user.com/graphql",
   cleanup: true,
   headers: {
     'Content-Type': 'application/json',
     'Accept': 'application/json',
   },
   factories: {
     createUser: {
       query: 'mutation createUser($input: UserInput!) { createUser(input: $input) { id name }}',
       factory: './factories/users',
       revert: (data) => ({
         query: 'mutation deleteUser($id: ID!) { deleteUser(id: $id) }',
         variables: { id : data.id},
       }),
     },
   }
}
```

It is required to set GraphQL `endpoint` which is the URL to which all the queries go to.
Factory file is expected to be passed via `factory` option.

This Helper uses [GraphQL][3] helper and accepts its configuration in "GraphQL" section.
For instance, to set timeout you should add:

```js
"GraphQLDataFactory": {
   "GraphQL": {
     "timeout": "100000",
  }
}
```

### Factory

Factory contains operations -

*   `operation`: The operation/mutation that needs to be performed for creating a record in the backend.

Each operation must have the following:

*   `query`: The mutation(query) string. It is expected to use variables to send data with the query.
*   `factory`: The path to factory file. The object built by the factory in this file will be passed
    as the 'variables' object to go along with the mutation.
*   `revert`: A function called with the data returned when an item is created. The object returned by
    this function is will be used to later delete the items created. So, make sure RELEVANT DATA IS RETURNED
    when a record is created by a mutation.

### Requests

Requests can be updated on the fly by using `onRequest` function. For instance, you can pass in current session from a cookie.

```js
 onRequest: async (request) => {
    // using global codeceptjs instance
    let cookie = await codeceptjs.container.helpers('WebDriver').grabCookie('session');
    request.headers = { Cookie: `session=${cookie.value}` };
  }
```

### Responses

By default `I.mutateData()` returns a promise with created data as specified in operation query string:

```js
let client = await I.mutateData('createClient');
```

Data of created records are collected and used in the end of a test for the cleanup.

## Methods

### Parameters

*   `config` &#x20;

### _requestCreate

Executes request to create a record to the GraphQL endpoint.
Can be replaced from a custom helper.

#### Parameters

*   `operation` **[string][4]**&#x20;
*   `variables` **any** to be sent along with the query

### _requestDelete

Executes request to delete a record to the GraphQL endpoint.
Can be replaced from a custom helper.

#### Parameters

*   `operation` **[string][4]**&#x20;
*   `data` **any** of the record to be deleted.

### mutateData

Generates a new record using factory, sends a GraphQL mutation to store it.

```js
// create a user
I.mutateData('createUser');
// create user with defined email
// and receive it when inside async function
const user = await I.mutateData('createUser', { email: 'user@user.com'});
```

#### Parameters

*   `operation` **[string][4]** to be performed
*   `params` **any** predefined parameters

### mutateMultiple

Generates bunch of records and sends multiple GraphQL mutation requests to store them.

```js
// create 3 users
I.mutateMultiple('createUser', 3);

// create 3 users of same age
I.mutateMultiple('createUser', 3, { age: 25 });
```

#### Parameters

*   `operation` **[string][4]**&#x20;
*   `times` **[number][5]**&#x20;
*   `params` **any**&#x20;

[1]: https://github.com/rosiejs/rosie

[2]: https://www.npmjs.com/package/faker

[3]: http://codecept.io/helpers/GraphQL/

[4]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/String

[5]: https://developer.mozilla.org/docs/Web/JavaScript/Reference/Global_Objects/Number
