// @ts-check

import { ESLint } from 'eslint';
import fs from 'node:fs/promises';
import path, { dirname } from 'node:path';
import { fileURLToPath } from 'node:url';
import { TEST_FILE_PATTERNS } from '../src/lib/patterns.js';
import {
  configNames,
  getAllEnabledRuleIds,
  importConfig,
  isConfigForTests,
} from './helpers/configs.js';
import { configRulesToMarkdown } from './helpers/format-config.js';
import { configsToMarkdown } from './helpers/format-readme.js';
import {
  findRuleEntry,
  getRulesMetadata,
  ruleLevelFromEntry,
} from './helpers/rules.js';

const currentDir = fileURLToPath(dirname(import.meta.url));
const readmePath = path.join(currentDir, '..', 'README.md');
const docsDir = path.join(currentDir, '..', 'docs');

await generateDocs();

async function generateDocs() {
  const configs = await loadConfigs(configNames);
  const peerDeps = await loadPeerDependencies(configs);

  await fs.mkdir(docsDir, { recursive: true });

  for (const config of configs) {
    await generateConfigDocs(config, configs, peerDeps);
  }

  await generateReadmeDocs(configs, peerDeps);
}

/**
 * @param {string[]} names
 * @returns {Promise<import('./helpers/types').ExportedConfig[]>}
 */
function loadConfigs(names) {
  return Promise.all(
    names.map(async name => ({
      name,
      flatConfig: await importConfig(name),
    })),
  );
}

/**
 * @param {import('./helpers/types').ExportedConfig[]} configs
 * @returns {Promise<import('./helpers/types').PeerDep[]>}
 */
async function loadPeerDependencies(configs) {
  const packageJson = await import('../package.json', {
    with: { type: 'json' },
  }).then(m => m.default);

  /** @type {Record<string, string[]>} */
  const pkgConfigs = {};

  for (const config of configs) {
    const plugins = config.flatConfig.flatMap(({ plugins = {} }) =>
      Object.keys(plugins),
    );

    for (const pkg in packageJson.peerDependencies) {
      if (pkg === 'eslint') {
        continue;
      }
      const alias = pkg.replace(/eslint-plugin-?/, '').replace(/\/$/, '');
      if (
        plugins.includes(pkg) ||
        plugins.includes(alias) ||
        plugins.map(plugin => plugin.replace(/^@/, '')).includes(alias)
      ) {
        pkgConfigs[pkg] ??= [];
        pkgConfigs[pkg].push(config.name);
      }
    }
  }

  return Object.entries(packageJson.peerDependencies).map(([pkg, version]) => ({
    pkg,
    version,
    optional: packageJson.peerDependenciesMeta[pkg]?.optional ?? false,
    usedByConfigs: pkgConfigs[pkg] ?? [],
  }));
}

/**
 * Update auto-generated part of README.md
 * @param {import('./helpers/types').ExportedConfig[]} configs Exported configs
 * @param {import('./helpers/types').PeerDep[]} peerDeps Peer dependencies
 */
async function generateReadmeDocs(configs, peerDeps) {
  const extended = Object.fromEntries(
    configs.map(config => [config.name, getExtendedConfigs(config)]),
  );

  const buffer = await fs.readFile(readmePath);
  const mdPrevious = buffer.toString('utf8');

  const startComment = '<!-- begin autogenerated -->';
  const endComment = '<!-- end autogenerated -->';

  const startIndex = mdPrevious.indexOf(startComment);
  const endIndex = mdPrevious.indexOf(
    endComment,
    startIndex + startComment.length,
  );

  const mdGenerated = configsToMarkdown(configNames, peerDeps, extended);
  const mdGeneratedBlock = [
    startComment,
    mdGenerated.replace(/\n$/, ''),
    endComment,
  ].join('\n\n');

  const mdUpdated =
    startIndex < 0
      ? [mdPrevious, mdGeneratedBlock].join('\n')
      : [
          mdPrevious.slice(0, startIndex),
          mdGeneratedBlock,
          mdPrevious.slice(endIndex + endComment.length),
        ].join('');

  await fs.writeFile(readmePath, mdUpdated);

  console.info(`Updated Markdown docs in ${readmePath}`);
}

/**
 * Generate Markdown file for specified ESLint config.
 * @param {import('./helpers/types').ExportedConfig} config Exported config
 * @param {import('./helpers/types').ExportedConfig[]} allConfigs All exported configs
 * @param {import('./helpers/types').PeerDep[]} peerDeps Peer dependencies
 */
async function generateConfigDocs(config, allConfigs, peerDeps) {
  const extendedConfigs = Object.fromEntries(
    getExtendedConfigs(config).map(otherName => {
      const otherConfig = allConfigs.find(({ name }) => name === otherName);
      const otherRuleIds = getAllEnabledRuleIds(otherConfig?.flatConfig ?? []);
      return [otherName, otherRuleIds];
    }),
  );
  const extendedRuleIds = Object.values(extendedConfigs).flat();

  const ruleIds = getAllEnabledRuleIds(config.flatConfig).filter(
    ruleId => !extendedRuleIds.includes(ruleId),
  );

  const eslint = new ESLint();
  const dummyFile = 'eslint.config.js';
  await eslint.lintFiles(dummyFile);

  const rules = getRulesMetadata(config.flatConfig, ruleIds, eslint, dummyFile);

  const markdown = configRulesToMarkdown(
    config.name,
    ruleIds.map(id => {
      const entry =
        findRuleEntry(
          config.flatConfig.filter(
            ({ name }) =>
              name?.startsWith('code-pushup/') && name.endsWith('/customized'),
          ),
          id,
        ) ??
        findRuleEntry(
          config.flatConfig.filter(({ files }) => files !== TEST_FILE_PATTERNS),
          id,
        );
      if (entry == null) {
        throw new Error(
          `Internal logic error - no entry found for rule ${id} in ${config.name} config`,
        );
      }
      const level = ruleLevelFromEntry(entry);
      if (level === 'off') {
        throw new Error(
          `Internal logic error - rule ${id} turned off in ${config.name} config`,
        );
      }

      const testEntry = findRuleEntry(
        config.flatConfig.filter(({ files }) => files === TEST_FILE_PATTERNS),
        id,
      );
      const testLevel =
        testEntry != null ? ruleLevelFromEntry(testEntry) : null;

      return {
        id,
        meta: rules[id],
        level,
        ...(Array.isArray(entry) &&
          entry.length > 1 && {
            options: entry.slice(1),
          }),
        ...(testLevel &&
          testLevel !== level && {
            testOverride: {
              level: testLevel,
            },
          }),
      };
    }),
    Object.entries(extendedConfigs).map(([alias, rules]) => ({
      alias,
      rulesCount: rules.length,
    })),
    peerDeps,
    {
      hideOverrides: isConfigForTests(config.name),
    },
  );

  const filePath = path.join(docsDir, `${config.name}.md`);
  await fs.writeFile(filePath, markdown);

  console.info(`Generated Markdown docs in ${filePath}`);
}

/**
 * Get all extended code-pushup configs from flat config.
 * @param {import('./helpers/types').ExportedConfig} config Exported config
 */
export function getExtendedConfigs(config) {
  const allExtended = [
    ...new Set(
      config.flatConfig
        .map(cfg => cfg.name)
        .filter(name => name?.startsWith('code-pushup/'))
        .map(name => name?.split('/')[1])
        .filter(name => name != null),
    ),
  ];
  return allExtended.filter(name => name !== config.name).slice(-1);
}
