// Copyright 2019 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package tpch

var (
	maxCols              int
	numColsByQueryNumber = map[int]int{}
	// numExpectedRowsByQueryNumber is a mapping from query number to the
	// number of expected rows the query should return. For all of these
	// queries, only row count is checked (i.e. we won't perform row-by-row
	// check).
	numExpectedRowsByQueryNumber = map[int]int{
		11: 1048,
		16: 18314,
	}
)

func init() {
	for queryNumber, expectedRows := range expectedRowsByQueryNumber {
		numColsByQueryNumber[queryNumber] = len(expectedRows[0])
		numExpectedRowsByQueryNumber[queryNumber] = len(expectedRows)
		if len(expectedRows[0]) > maxCols {
			maxCols = len(expectedRows[0])
		}
	}
}

// expectedRowsByQueryNumber maps a query number to the expected rows for that
// query. Queries 11 and 16 return 1048 and 18314 rows, respectively, so we
// only verify the number of rows and these both are omitted from the map.
var expectedRowsByQueryNumber = map[int][][]string{
	1: {
		{`A`, `F`, `3.7734107e+07`, `5.65865544007299e+10`, `5.375825713486514e+10`, `5.590906522282561e+10`, `25.522005853257337`, `38273.1297346216`, `0.04998529583825443`, `1478493`},
		{`N`, `F`, `991417`, `1.4875047103799965e+09`, `1.413082168054104e+09`, `1.4696492231943603e+09`, `25.516471920522985`, `38284.467760848216`, `0.05009342667419324`, `38854`},
		{`N`, `O`, `7.447604e+07`, `1.1170172969773557e+11`, `1.0611823030761223e+11`, `1.1036704387249208e+11`, `25.50222676958499`, `38249.11798890675`, `0.04999658605362673`, `2920374`},
		{`R`, `F`, `3.7719753e+07`, `5.656804138090447e+10`, `5.374129268460378e+10`, `5.588961911982966e+10`, `25.50579361269077`, `38250.85462610268`, `0.050009405829983596`, `1478870`},
	},
	2: {
		{`9938.53`, `Supplier#000005359`, `UNITED KINGDOM`, `185358`, `Manufacturer#4`, `QKuHYh,vZGiwu2FWEJoLDx04`, `33-429-790-6131`, `uriously regular requests hag`},
		{`9937.84`, `Supplier#000005969`, `ROMANIA`, `108438`, `Manufacturer#1`, `ANDENSOSmk,miq23Xfb5RWt6dvUcvt6Qa`, `29-520-692-3537`, `efully express instructions. regular requests against the slyly fin`},
		{`9936.22`, `Supplier#000005250`, `UNITED KINGDOM`, `249`, `Manufacturer#4`, `B3rqp0xbSEim4Mpy2RH J`, `33-320-228-2957`, `etect about the furiously final accounts. slyly ironic pinto beans sleep inside the furiously`},
		{`9923.77`, `Supplier#000002324`, `GERMANY`, `29821`, `Manufacturer#4`, `y3OD9UywSTOk`, `17-779-299-1839`, `ackages boost blithely. blithely regular deposits c`},
		{`9871.22`, `Supplier#000006373`, `GERMANY`, `43868`, `Manufacturer#5`, `J8fcXWsTqM`, `17-813-485-8637`, `etect blithely bold asymptotes. fluffily ironic platelets wake furiously; blit`},
		{`9870.78`, `Supplier#000001286`, `GERMANY`, `81285`, `Manufacturer#2`, `YKA,E2fjiVd7eUrzp2Ef8j1QxGo2DFnosaTEH`, `17-516-924-4574`, ` regular accounts. furiously unusual courts above the fi`},
		{`9870.78`, `Supplier#000001286`, `GERMANY`, `181285`, `Manufacturer#4`, `YKA,E2fjiVd7eUrzp2Ef8j1QxGo2DFnosaTEH`, `17-516-924-4574`, ` regular accounts. furiously unusual courts above the fi`},
		{`9852.52`, `Supplier#000008973`, `RUSSIA`, `18972`, `Manufacturer#2`, `t5L67YdBYYH6o,Vz24jpDyQ9`, `32-188-594-7038`, `rns wake final foxes. carefully unusual depende`},
		{`9847.83`, `Supplier#000008097`, `RUSSIA`, `130557`, `Manufacturer#2`, `xMe97bpE69NzdwLoX`, `32-375-640-3593`, ` the special excuses. silent sentiments serve carefully final ac`},
		{`9847.57`, `Supplier#000006345`, `FRANCE`, `86344`, `Manufacturer#1`, `VSt3rzk3qG698u6ld8HhOByvrTcSTSvQlDQDag`, `16-886-766-7945`, `ges. slyly regular requests are. ruthless, express excuses cajole blithely across the unu`},
		{`9847.57`, `Supplier#000006345`, `FRANCE`, `173827`, `Manufacturer#2`, `VSt3rzk3qG698u6ld8HhOByvrTcSTSvQlDQDag`, `16-886-766-7945`, `ges. slyly regular requests are. ruthless, express excuses cajole blithely across the unu`},
		{`9836.93`, `Supplier#000007342`, `RUSSIA`, `4841`, `Manufacturer#4`, `JOlK7C1,7xrEZSSOw`, `32-399-414-5385`, `blithely carefully bold theodolites. fur`},
		{`9817.1`, `Supplier#000002352`, `RUSSIA`, `124815`, `Manufacturer#2`, `4LfoHUZjgjEbAKw TgdKcgOc4D4uCYw`, `32-551-831-1437`, `wake carefully alongside of the carefully final ex`},
		{`9817.1`, `Supplier#000002352`, `RUSSIA`, `152351`, `Manufacturer#3`, `4LfoHUZjgjEbAKw TgdKcgOc4D4uCYw`, `32-551-831-1437`, `wake carefully alongside of the carefully final ex`},
		{`9739.86`, `Supplier#000003384`, `FRANCE`, `138357`, `Manufacturer#2`, `o,Z3v4POifevE k9U1b 6J1ucX,I`, `16-494-913-5925`, `s after the furiously bold packages sleep fluffily idly final requests: quickly final`},
		{`9721.95`, `Supplier#000008757`, `UNITED KINGDOM`, `156241`, `Manufacturer#3`, `Atg6GnM4dT2`, `33-821-407-2995`, `eep furiously sauternes; quickl`},
		{`9681.33`, `Supplier#000008406`, `RUSSIA`, `78405`, `Manufacturer#1`, `,qUuXcftUl`, `32-139-873-8571`, `haggle slyly regular excuses. quic`},
		{`9643.55`, `Supplier#000005148`, `ROMANIA`, `107617`, `Manufacturer#1`, `kT4ciVFslx9z4s79p Js825`, `29-252-617-4850`, `final excuses. final ideas boost quickly furiously speci`},
		{`9624.82`, `Supplier#000001816`, `FRANCE`, `34306`, `Manufacturer#3`, `e7vab91vLJPWxxZnewmnDBpDmxYHrb`, `16-392-237-6726`, `e packages are around the special ideas. special, pending foxes us`},
		{`9624.78`, `Supplier#000009658`, `ROMANIA`, `189657`, `Manufacturer#1`, `oE9uBgEfSS4opIcepXyAYM,x`, `29-748-876-2014`, `ronic asymptotes wake bravely final`},
		{`9612.94`, `Supplier#000003228`, `ROMANIA`, `120715`, `Manufacturer#2`, `KDdpNKN3cWu7ZSrbdqp7AfSLxx,qWB`, `29-325-784-8187`, `warhorses. quickly even deposits sublate daringly ironic instructions. slyly blithe t`},
		{`9612.94`, `Supplier#000003228`, `ROMANIA`, `198189`, `Manufacturer#4`, `KDdpNKN3cWu7ZSrbdqp7AfSLxx,qWB`, `29-325-784-8187`, `warhorses. quickly even deposits sublate daringly ironic instructions. slyly blithe t`},
		{`9571.83`, `Supplier#000004305`, `ROMANIA`, `179270`, `Manufacturer#2`, `qNHZ7WmCzygwMPRDO9Ps`, `29-973-481-1831`, `kly carefully express asymptotes. furiou`},
		{`9558.1`, `Supplier#000003532`, `UNITED KINGDOM`, `88515`, `Manufacturer#4`, `EOeuiiOn21OVpTlGguufFDFsbN1p0lhpxHp`, `33-152-301-2164`, ` foxes. quickly even excuses use. slyly special foxes nag bl`},
		{`9492.79`, `Supplier#000005975`, `GERMANY`, `25974`, `Manufacturer#5`, `S6mIiCTx82z7lV`, `17-992-579-4839`, `arefully pending accounts. blithely regular excuses boost carefully carefully ironic p`},
		{`9461.05`, `Supplier#000002536`, `UNITED KINGDOM`, `20033`, `Manufacturer#1`, `8mmGbyzaU 7ZS2wJumTibypncu9pNkDc4FYA`, `33-556-973-5522`, `. slyly regular deposits wake slyly. furiously regular warthogs are.`},
		{`9453.01`, `Supplier#000000802`, `ROMANIA`, `175767`, `Manufacturer#1`, `,6HYXb4uaHITmtMBj4Ak57Pd`, `29-342-882-6463`, `gular frets. permanently special multipliers believe blithely alongs`},
		{`9408.65`, `Supplier#000007772`, `UNITED KINGDOM`, `117771`, `Manufacturer#4`, `AiC5YAH,gdu0i7`, `33-152-491-1126`, `nag against the final requests. furiously unusual packages cajole blit`},
		{`9359.61`, `Supplier#000004856`, `ROMANIA`, `62349`, `Manufacturer#5`, `HYogcF3Jb yh1`, `29-334-870-9731`, `y ironic theodolites. blithely sile`},
		{`9357.45`, `Supplier#000006188`, `UNITED KINGDOM`, `138648`, `Manufacturer#1`, `g801,ssP8wpTk4Hm`, `33-583-607-1633`, `ously always regular packages. fluffily even accounts beneath the furiously final pack`},
		{`9352.04`, `Supplier#000003439`, `GERMANY`, `170921`, `Manufacturer#4`, `qYPDgoiBGhCYxjgC`, `17-128-996-4650`, ` according to the carefully bold ideas`},
		{`9312.97`, `Supplier#000007807`, `RUSSIA`, `90279`, `Manufacturer#5`, `oGYMPCk9XHGB2PBfKRnHA`, `32-673-872-5854`, `ecial packages among the pending, even requests use regula`},
		{`9312.97`, `Supplier#000007807`, `RUSSIA`, `100276`, `Manufacturer#5`, `oGYMPCk9XHGB2PBfKRnHA`, `32-673-872-5854`, `ecial packages among the pending, even requests use regula`},
		{`9280.27`, `Supplier#000007194`, `ROMANIA`, `47193`, `Manufacturer#3`, `zhRUQkBSrFYxIAXTfInj vyGRQjeK`, `29-318-454-2133`, `o beans haggle after the furiously unusual deposits. carefully silent dolphins cajole carefully`},
		{`9274.8`, `Supplier#000008854`, `RUSSIA`, `76346`, `Manufacturer#3`, `1xhLoOUM7I3mZ1mKnerw OSqdbb4QbGa`, `32-524-148-5221`, `y. courts do wake slyly. carefully ironic platelets haggle above the slyly regular the`},
		{`9249.35`, `Supplier#000003973`, `FRANCE`, `26466`, `Manufacturer#1`, `d18GiDsL6Wm2IsGXM,RZf1jCsgZAOjNYVThTRP4`, `16-722-866-1658`, `uests are furiously. regular tithes through the regular, final accounts cajole furiously above the q`},
		{`9249.35`, `Supplier#000003973`, `FRANCE`, `33972`, `Manufacturer#1`, `d18GiDsL6Wm2IsGXM,RZf1jCsgZAOjNYVThTRP4`, `16-722-866-1658`, `uests are furiously. regular tithes through the regular, final accounts cajole furiously above the q`},
		{`9208.7`, `Supplier#000007769`, `ROMANIA`, `40256`, `Manufacturer#5`, `rsimdze 5o9P Ht7xS`, `29-964-424-9649`, `lites was quickly above the furiously ironic requests. slyly even foxes against the blithely bold `},
		{`9201.47`, `Supplier#000009690`, `UNITED KINGDOM`, `67183`, `Manufacturer#5`, `CB BnUTlmi5zdeEl7R7`, `33-121-267-9529`, `e even, even foxes. blithely ironic packages cajole regular packages. slyly final ide`},
		{`9192.1`, `Supplier#000000115`, `UNITED KINGDOM`, `85098`, `Manufacturer#3`, `nJ 2t0f7Ve,wL1,6WzGBJLNBUCKlsV`, `33-597-248-1220`, `es across the carefully express accounts boost caref`},
		{`9189.98`, `Supplier#000001226`, `GERMANY`, `21225`, `Manufacturer#4`, `qsLCqSvLyZfuXIpjz`, `17-725-903-1381`, ` deposits. blithely bold excuses about the slyly bold forges wake `},
		{`9128.97`, `Supplier#000004311`, `RUSSIA`, `146768`, `Manufacturer#5`, `I8IjnXd7NSJRs594RxsRR0`, `32-155-440-7120`, `refully. blithely unusual asymptotes haggle `},
		{`9104.83`, `Supplier#000008520`, `GERMANY`, `150974`, `Manufacturer#4`, `RqRVDgD0ER J9 b41vR2,3`, `17-728-804-1793`, `ly about the blithely ironic depths. slyly final theodolites among the fluffily bold ideas print`},
		{`9101`, `Supplier#000005791`, `ROMANIA`, `128254`, `Manufacturer#5`, `zub2zCV,jhHPPQqi,P2INAjE1zI n66cOEoXFG`, `29-549-251-5384`, `ts. notornis detect blithely above the carefully bold requests. blithely even package`},
		{`9094.57`, `Supplier#000004582`, `RUSSIA`, `39575`, `Manufacturer#1`, `WB0XkCSG3r,mnQ n,h9VIxjjr9ARHFvKgMDf`, `32-587-577-1351`, `jole. regular accounts sleep blithely frets. final pinto beans play furiously past the `},
		{`8996.87`, `Supplier#000004702`, `FRANCE`, `102191`, `Manufacturer#5`, `8XVcQK23akp`, `16-811-269-8946`, `ickly final packages along the express plat`},
		{`8996.14`, `Supplier#000009814`, `ROMANIA`, `139813`, `Manufacturer#2`, `af0O5pg83lPU4IDVmEylXZVqYZQzSDlYLAmR`, `29-995-571-8781`, ` dependencies boost quickly across the furiously pending requests! unusual dolphins play sl`},
		{`8968.42`, `Supplier#000010000`, `ROMANIA`, `119999`, `Manufacturer#5`, `aTGLEusCiL4F PDBdv665XBJhPyCOB0i`, `29-578-432-2146`, `ly regular foxes boost slyly. quickly special waters boost carefully ironi`},
		{`8936.82`, `Supplier#000007043`, `UNITED KINGDOM`, `109512`, `Manufacturer#1`, `FVajceZInZdbJE6Z9XsRUxrUEpiwHDrOXi,1Rz`, `33-784-177-8208`, `efully regular courts. furiousl`},
		{`8929.42`, `Supplier#000008770`, `FRANCE`, `173735`, `Manufacturer#4`, `R7cG26TtXrHAP9 HckhfRi`, `16-242-746-9248`, `cajole furiously unusual requests. quickly stealthy requests are. `},
		{`8920.59`, `Supplier#000003967`, `ROMANIA`, `26460`, `Manufacturer#1`, `eHoAXe62SY9`, `29-194-731-3944`, `aters. express, pending instructions sleep. brave, r`},
		{`8920.59`, `Supplier#000003967`, `ROMANIA`, `173966`, `Manufacturer#2`, `eHoAXe62SY9`, `29-194-731-3944`, `aters. express, pending instructions sleep. brave, r`},
		{`8913.96`, `Supplier#000004603`, `UNITED KINGDOM`, `137063`, `Manufacturer#2`, `OUzlvMUr7n,utLxmPNeYKSf3T24OXskxB5`, `33-789-255-7342`, ` haggle slyly above the furiously regular pinto beans. even `},
		{`8877.82`, `Supplier#000007967`, `FRANCE`, `167966`, `Manufacturer#5`, `A3pi1BARM4nx6R,qrwFoRPU`, `16-442-147-9345`, `ously foxes. express, ironic requests im`},
		{`8862.24`, `Supplier#000003323`, `ROMANIA`, `73322`, `Manufacturer#3`, `W9 lYcsC9FwBqk3ItL`, `29-736-951-3710`, `ly pending ideas sleep about the furiously unu`},
		{`8841.59`, `Supplier#000005750`, `ROMANIA`, `100729`, `Manufacturer#5`, `Erx3lAgu0g62iaHF9x50uMH4EgeN9hEG`, `29-344-502-5481`, `gainst the pinto beans. fluffily unusual dependencies affix slyly even deposits.`},
		{`8781.71`, `Supplier#000003121`, `ROMANIA`, `13120`, `Manufacturer#5`, `wNqTogx238ZYCamFb,50v,bj 4IbNFW9Bvw1xP`, `29-707-291-5144`, `s wake quickly ironic ideas`},
		{`8754.24`, `Supplier#000009407`, `UNITED KINGDOM`, `179406`, `Manufacturer#4`, `CHRCbkaWcf5B`, `33-903-970-9604`, `e ironic requests. carefully even foxes above the furious`},
		{`8691.06`, `Supplier#000004429`, `UNITED KINGDOM`, `126892`, `Manufacturer#2`, `k,BQms5UhoAF1B2Asi,fLib`, `33-964-337-5038`, `efully express deposits kindle after the deposits. final `},
		{`8655.99`, `Supplier#000006330`, `RUSSIA`, `193810`, `Manufacturer#2`, `UozlaENr0ytKe2w6CeIEWFWn iO3S8Rae7Ou`, `32-561-198-3705`, `symptotes use about the express dolphins. requests use after the express platelets. final, ex`},
		{`8638.36`, `Supplier#000002920`, `RUSSIA`, `75398`, `Manufacturer#1`, `Je2a8bszf3L`, `32-122-621-7549`, `ly quickly ironic requests. even requests whithout t`},
		{`8638.36`, `Supplier#000002920`, `RUSSIA`, `170402`, `Manufacturer#3`, `Je2a8bszf3L`, `32-122-621-7549`, `ly quickly ironic requests. even requests whithout t`},
		{`8607.69`, `Supplier#000006003`, `UNITED KINGDOM`, `76002`, `Manufacturer#2`, `EH9wADcEiuenM0NR08zDwMidw,52Y2RyILEiA`, `33-416-807-5206`, `ar, pending accounts. pending depende`},
		{`8569.52`, `Supplier#000005936`, `RUSSIA`, `5935`, `Manufacturer#5`, `jXaNZ6vwnEWJ2ksLZJpjtgt0bY2a3AU`, `32-644-251-7916`, `. regular foxes nag carefully atop the regular, silent deposits. quickly regular packages `},
		{`8564.12`, `Supplier#000000033`, `GERMANY`, `110032`, `Manufacturer#1`, `gfeKpYw3400L0SDywXA6Ya1Qmq1w6YB9f3R`, `17-138-897-9374`, `n sauternes along the regular asymptotes are regularly along the `},
		{`8553.82`, `Supplier#000003979`, `ROMANIA`, `143978`, `Manufacturer#4`, `BfmVhCAnCMY3jzpjUMy4CNWs9 HzpdQR7INJU`, `29-124-646-4897`, `ic requests wake against the blithely unusual accounts. fluffily r`},
		{`8517.23`, `Supplier#000009529`, `RUSSIA`, `37025`, `Manufacturer#5`, `e44R8o7JAIS9iMcr`, `32-565-297-8775`, `ove the even courts. furiously special platelets `},
		{`8517.23`, `Supplier#000009529`, `RUSSIA`, `59528`, `Manufacturer#2`, `e44R8o7JAIS9iMcr`, `32-565-297-8775`, `ove the even courts. furiously special platelets `},
		{`8503.7`, `Supplier#000006830`, `RUSSIA`, `44325`, `Manufacturer#4`, `BC4WFCYRUZyaIgchU 4S`, `32-147-878-5069`, `pades cajole. furious packages among the carefully express excuses boost furiously across th`},
		{`8457.09`, `Supplier#000009456`, `UNITED KINGDOM`, `19455`, `Manufacturer#1`, `7SBhZs8gP1cJjT0Qf433YBk`, `33-858-440-4349`, `cing requests along the furiously unusual deposits promise among the furiously unus`},
		{`8441.4`, `Supplier#000003817`, `FRANCE`, `141302`, `Manufacturer#2`, `hU3fz3xL78`, `16-339-356-5115`, `ely even ideas. ideas wake slyly furiously unusual instructions. pinto beans sleep ag`},
		{`8432.89`, `Supplier#000003990`, `RUSSIA`, `191470`, `Manufacturer#1`, `wehBBp1RQbfxAYDASS75MsywmsKHRVdkrvNe6m`, `32-839-509-9301`, `ep furiously. packages should have to haggle slyly across the deposits. furiously regu`},
		{`8431.4`, `Supplier#000002675`, `ROMANIA`, `5174`, `Manufacturer#1`, `HJFStOu9R5NGPOegKhgbzBdyvrG2yh8w`, `29-474-643-1443`, `ithely express pinto beans. blithely even foxes haggle. furiously regular theodol`},
		{`8407.04`, `Supplier#000005406`, `RUSSIA`, `162889`, `Manufacturer#4`, `j7 gYF5RW8DC5UrjKC`, `32-626-152-4621`, `r the blithely regular packages. slyly ironic theodoli`},
		{`8386.08`, `Supplier#000008518`, `FRANCE`, `36014`, `Manufacturer#3`, `2jqzqqAVe9crMVGP,n9nTsQXulNLTUYoJjEDcqWV`, `16-618-780-7481`, `blithely bold pains are carefully platelets. finally regular pinto beans sleep carefully special`},
		{`8376.52`, `Supplier#000005306`, `UNITED KINGDOM`, `190267`, `Manufacturer#5`, `9t8Y8 QqSIsoADPt6NLdk,TP5zyRx41oBUlgoGc9`, `33-632-514-7931`, `ly final accounts sleep special, regular requests. furiously regular`},
		{`8348.74`, `Supplier#000008851`, `FRANCE`, `66344`, `Manufacturer#4`, `nWxi7GwEbjhw1`, `16-796-240-2472`, ` boldly final deposits. regular, even instructions detect slyly. fluffily unusual pinto bea`},
		{`8338.58`, `Supplier#000007269`, `FRANCE`, `17268`, `Manufacturer#4`, `ZwhJSwABUoiB04,3`, `16-267-277-4365`, `iously final accounts. even pinto beans cajole slyly regular`},
		{`8328.46`, `Supplier#000001744`, `ROMANIA`, `69237`, `Manufacturer#5`, `oLo3fV64q2,FKHa3p,qHnS7Yzv,ps8`, `29-330-728-5873`, `ep carefully-- even, careful packages are slyly along t`},
		{`8307.93`, `Supplier#000003142`, `GERMANY`, `18139`, `Manufacturer#1`, `dqblvV8dCNAorGlJ`, `17-595-447-6026`, `olites wake furiously regular decoys. final requests nod `},
		{`8231.61`, `Supplier#000009558`, `RUSSIA`, `192000`, `Manufacturer#2`, `mcdgen,yT1iJDHDS5fV`, `32-762-137-5858`, ` foxes according to the furi`},
		{`8152.61`, `Supplier#000002731`, `ROMANIA`, `15227`, `Manufacturer#4`, ` nluXJCuY1tu`, `29-805-463-2030`, ` special requests. even, regular warhorses affix among the final gr`},
		{`8109.09`, `Supplier#000009186`, `FRANCE`, `99185`, `Manufacturer#1`, `wgfosrVPexl9pEXWywaqlBMDYYf`, `16-668-570-1402`, `tions haggle slyly about the sil`},
		{`8102.62`, `Supplier#000003347`, `UNITED KINGDOM`, `18344`, `Manufacturer#5`, `m CtXS2S16i`, `33-454-274-8532`, `egrate with the slyly bold instructions. special foxes haggle silently among the`},
		{`8046.07`, `Supplier#000008780`, `FRANCE`, `191222`, `Manufacturer#3`, `AczzuE0UK9osj ,Lx0Jmh`, `16-473-215-6395`, `onic platelets cajole after the regular instructions. permanently bold excuses`},
		{`8042.09`, `Supplier#000003245`, `RUSSIA`, `135705`, `Manufacturer#4`, `Dh8Ikg39onrbOL4DyTfGw8a9oKUX3d9Y`, `32-836-132-8872`, `osits. packages cajole slyly. furiously regular deposits cajole slyly. q`},
		{`8042.09`, `Supplier#000003245`, `RUSSIA`, `150729`, `Manufacturer#1`, `Dh8Ikg39onrbOL4DyTfGw8a9oKUX3d9Y`, `32-836-132-8872`, `osits. packages cajole slyly. furiously regular deposits cajole slyly. q`},
		{`7992.4`, `Supplier#000006108`, `FRANCE`, `118574`, `Manufacturer#1`, `8tBydnTDwUqfBfFV4l3`, `16-974-998-8937`, ` ironic ideas? fluffily even instructions wake. blithel`},
		{`7980.65`, `Supplier#000001288`, `FRANCE`, `13784`, `Manufacturer#4`, `zE,7HgVPrCn`, `16-646-464-8247`, `ully bold courts. escapades nag slyly. furiously fluffy theodo`},
		{`7950.37`, `Supplier#000008101`, `GERMANY`, `33094`, `Manufacturer#5`, `kkYvL6IuvojJgTNG IKkaXQDYgx8ILohj`, `17-627-663-8014`, `arefully unusual requests x-ray above the quickly final deposits. `},
		{`7937.93`, `Supplier#000009012`, `ROMANIA`, `83995`, `Manufacturer#2`, `iUiTziH,Ek3i4lwSgunXMgrcTzwdb`, `29-250-925-9690`, `to the blithely ironic deposits nag sly`},
		{`7914.45`, `Supplier#000001013`, `RUSSIA`, `125988`, `Manufacturer#2`, `riRcntps4KEDtYScjpMIWeYF6mNnR`, `32-194-698-3365`, ` busily bold packages are dolphi`},
		{`7912.91`, `Supplier#000004211`, `GERMANY`, `159180`, `Manufacturer#5`, `2wQRVovHrm3,v03IKzfTd,1PYsFXQFFOG`, `17-266-947-7315`, `ay furiously regular platelets. cou`},
		{`7912.91`, `Supplier#000004211`, `GERMANY`, `184210`, `Manufacturer#4`, `2wQRVovHrm3,v03IKzfTd,1PYsFXQFFOG`, `17-266-947-7315`, `ay furiously regular platelets. cou`},
		{`7894.56`, `Supplier#000007981`, `GERMANY`, `85472`, `Manufacturer#4`, `NSJ96vMROAbeXP`, `17-963-404-3760`, `ic platelets affix after the furiously`},
		{`7887.08`, `Supplier#000009792`, `GERMANY`, `164759`, `Manufacturer#3`, `Y28ITVeYriT3kIGdV2K8fSZ V2UqT5H1Otz`, `17-988-938-4296`, `ckly around the carefully fluffy theodolites. slyly ironic pack`},
		{`7871.5`, `Supplier#000007206`, `RUSSIA`, `104695`, `Manufacturer#1`, `3w fNCnrVmvJjE95sgWZzvW`, `32-432-452-7731`, `ironic requests. furiously final theodolites cajole. final, express packages sleep. quickly reg`},
		{`7852.45`, `Supplier#000005864`, `RUSSIA`, `8363`, `Manufacturer#4`, `WCNfBPZeSXh3h,c`, `32-454-883-3821`, `usly unusual pinto beans. brave ideas sleep carefully quickly ironi`},
		{`7850.66`, `Supplier#000001518`, `UNITED KINGDOM`, `86501`, `Manufacturer#1`, `ONda3YJiHKJOC`, `33-730-383-3892`, `ifts haggle fluffily pending pai`},
		{`7843.52`, `Supplier#000006683`, `FRANCE`, `11680`, `Manufacturer#4`, `2Z0JGkiv01Y00oCFwUGfviIbhzCdy`, `16-464-517-8943`, ` express, final pinto beans x-ray slyly asymptotes. unusual, unusual`},
	},
	3: {
		{`2456423`, `406181.0111`, `1995-03-05 00:00:00 +0000 +0000`, `0`},
		{`3459808`, `405838.69889999996`, `1995-03-04 00:00:00 +0000 +0000`, `0`},
		{`492164`, `390324.061`, `1995-02-19 00:00:00 +0000 +0000`, `0`},
		{`1188320`, `384537.9359`, `1995-03-09 00:00:00 +0000 +0000`, `0`},
		{`2435712`, `378673.05580000003`, `1995-02-26 00:00:00 +0000 +0000`, `0`},
		{`4878020`, `378376.7952`, `1995-03-12 00:00:00 +0000 +0000`, `0`},
		{`5521732`, `375153.9215`, `1995-03-13 00:00:00 +0000 +0000`, `0`},
		{`2628192`, `373133.30939999997`, `1995-02-22 00:00:00 +0000 +0000`, `0`},
		{`993600`, `371407.4595`, `1995-03-05 00:00:00 +0000 +0000`, `0`},
		{`2300070`, `367371.1452000001`, `1995-03-13 00:00:00 +0000 +0000`, `0`},
	},
	4: {
		{`1-URGENT`, `10594`},
		{`2-HIGH`, `10476`},
		{`3-MEDIUM`, `10410`},
		{`4-NOT SPECIFIED`, `10556`},
		{`5-LOW`, `10487`},
	},
	5: {
		{`INDONESIA`, `5.5502041169699945e+07`},
		{`VIETNAM`, `5.529508699669996e+07`},
		{`CHINA`, `5.372449425659997e+07`},
		{`INDIA`, `5.203551200020005e+07`},
		{`JAPAN`, `4.5410175695400015e+07`},
	},
	6: {
		{`1.2314107822829871e+08`},
	},
	7: {
		{`FRANCE`, `GERMANY`, `1995`, `5.463973273359995e+07`},
		{`FRANCE`, `GERMANY`, `1996`, `5.463308330759997e+07`},
		{`GERMANY`, `FRANCE`, `1995`, `5.253174666969997e+07`},
		{`GERMANY`, `FRANCE`, `1996`, `5.252054902239985e+07`},
	},
	8: {
		{`1995`, `0.03443589040665483`},
		{`1996`, `0.04148552129353034`},
	},
	9: {
		{`ALGERIA`, `1998`, `2.713690018030001e+07`},
		{`ALGERIA`, `1997`, `4.861183349620003e+07`},
		{`ALGERIA`, `1996`, `4.828548267819995e+07`},
		{`ALGERIA`, `1995`, `4.440227359990003e+07`},
		{`ALGERIA`, `1994`, `4.869400806679992e+07`},
		{`ALGERIA`, `1993`, `4.604420778380005e+07`},
		{`ALGERIA`, `1992`, `4.5636849488100074e+07`},
		{`ARGENTINA`, `1998`, `2.8341663784800023e+07`},
		{`ARGENTINA`, `1997`, `4.714396411760004e+07`},
		{`ARGENTINA`, `1996`, `4.525527860210005e+07`},
		{`ARGENTINA`, `1995`, `4.5631769205400005e+07`},
		{`ARGENTINA`, `1994`, `4.826885635470004e+07`},
		{`ARGENTINA`, `1993`, `4.860559361619997e+07`},
		{`ARGENTINA`, `1992`, `4.6654240748700015e+07`},
		{`BRAZIL`, `1998`, `2.652773639600002e+07`},
		{`BRAZIL`, `1997`, `4.564066076770004e+07`},
		{`BRAZIL`, `1996`, `4.509064716300003e+07`},
		{`BRAZIL`, `1995`, `4.401588851319997e+07`},
		{`BRAZIL`, `1994`, `4.485421889319998e+07`},
		{`BRAZIL`, `1993`, `4.576660373789999e+07`},
		{`BRAZIL`, `1992`, `4.5280216802700125e+07`},
		{`CANADA`, `1998`, `2.6828985394399986e+07`},
		{`CANADA`, `1997`, `4.484995431859992e+07`},
		{`CANADA`, `1996`, `4.630793611079995e+07`},
		{`CANADA`, `1995`, `4.731199304410007e+07`},
		{`CANADA`, `1994`, `4.669149195960006e+07`},
		{`CANADA`, `1993`, `4.663479111210005e+07`},
		{`CANADA`, `1992`, `4.587384968819996e+07`},
		{`CHINA`, `1998`, `2.7510180165700015e+07`},
		{`CHINA`, `1997`, `4.612386540969995e+07`},
		{`CHINA`, `1996`, `4.95328070600999e+07`},
		{`CHINA`, `1995`, `4.6734651483800106e+07`},
		{`CHINA`, `1994`, `4.639789660970005e+07`},
		{`CHINA`, `1993`, `4.963467394630013e+07`},
		{`CHINA`, `1992`, `4.6949457642600045e+07`},
		{`EGYPT`, `1998`, `2.840149179680002e+07`},
		{`EGYPT`, `1997`, `4.76748576783001e+07`},
		{`EGYPT`, `1996`, `4.774572754500001e+07`},
		{`EGYPT`, `1995`, `4.5897160678300045e+07`},
		{`EGYPT`, `1994`, `4.719489522800005e+07`},
		{`EGYPT`, `1993`, `4.913362764709991e+07`},
		{`EGYPT`, `1992`, `4.7000574502699964e+07`},
		{`ETHIOPIA`, `1998`, `2.5135046137700006e+07`},
		{`ETHIOPIA`, `1997`, `4.3010596083800085e+07`},
		{`ETHIOPIA`, `1996`, `4.363628719220001e+07`},
		{`ETHIOPIA`, `1995`, `4.357575733430003e+07`},
		{`ETHIOPIA`, `1994`, `4.159720852830006e+07`},
		{`ETHIOPIA`, `1993`, `4.262280416159995e+07`},
		{`ETHIOPIA`, `1992`, `4.438573568130001e+07`},
		{`FRANCE`, `1998`, `2.6210392280399982e+07`},
		{`FRANCE`, `1997`, `4.2392969473099984e+07`},
		{`FRANCE`, `1996`, `4.330631797490005e+07`},
		{`FRANCE`, `1995`, `4.637740843279991e+07`},
		{`FRANCE`, `1994`, `4.3447352992200114e+07`},
		{`FRANCE`, `1993`, `4.3729961063900046e+07`},
		{`FRANCE`, `1992`, `4.4052308429000005e+07`},
		{`GERMANY`, `1998`, `2.599125710710004e+07`},
		{`GERMANY`, `1997`, `4.3968355807900004e+07`},
		{`GERMANY`, `1996`, `4.588207480490002e+07`},
		{`GERMANY`, `1995`, `4.33143383077e+07`},
		{`GERMANY`, `1994`, `4.461699543689998e+07`},
		{`GERMANY`, `1993`, `4.5126645911299974e+07`},
		{`GERMANY`, `1992`, `4.436114121070009e+07`},
		{`INDIA`, `1998`, `2.9626417237899978e+07`},
		{`INDIA`, `1997`, `5.1386111344799966e+07`},
		{`INDIA`, `1996`, `4.757101851220004e+07`},
		{`INDIA`, `1995`, `4.934406228290009e+07`},
		{`INDIA`, `1994`, `5.01069524261e+07`},
		{`INDIA`, `1993`, `4.8112766698700055e+07`},
		{`INDIA`, `1992`, `4.791430312339994e+07`},
		{`INDONESIA`, `1998`, `2.773490967630001e+07`},
		{`INDONESIA`, `1997`, `4.459381298630008e+07`},
		{`INDONESIA`, `1996`, `4.474672980779995e+07`},
		{`INDONESIA`, `1995`, `4.559362269929995e+07`},
		{`INDONESIA`, `1994`, `4.598848387720004e+07`},
		{`INDONESIA`, `1993`, `4.614796378949999e+07`},
		{`INDONESIA`, `1992`, `4.518577706879992e+07`},
		{`IRAN`, `1998`, `2.666160893009998e+07`},
		{`IRAN`, `1997`, `4.50191141696001e+07`},
		{`IRAN`, `1996`, `4.589139709920007e+07`},
		{`IRAN`, `1995`, `4.441428523480002e+07`},
		{`IRAN`, `1994`, `4.369636047950001e+07`},
		{`IRAN`, `1993`, `4.536277580940003e+07`},
		{`IRAN`, `1992`, `4.305233841429999e+07`},
		{`IRAQ`, `1998`, `3.1188498191400014e+07`},
		{`IRAQ`, `1997`, `4.858530752219989e+07`},
		{`IRAQ`, `1996`, `5.003659384040011e+07`},
		{`IRAQ`, `1995`, `4.877480172749997e+07`},
		{`IRAQ`, `1994`, `4.879584723100009e+07`},
		{`IRAQ`, `1993`, `4.743569150820013e+07`},
		{`IRAQ`, `1992`, `4.75623556571e+07`},
		{`JAPAN`, `1998`, `2.4694102172000006e+07`},
		{`JAPAN`, `1997`, `4.2377052345399976e+07`},
		{`JAPAN`, `1996`, `4.0267778909400046e+07`},
		{`JAPAN`, `1995`, `4.0925317464999996e+07`},
		{`JAPAN`, `1994`, `4.115951830580001e+07`},
		{`JAPAN`, `1993`, `3.958907427710004e+07`},
		{`JAPAN`, `1992`, `3.9113493905199975e+07`},
		{`JORDAN`, `1998`, `2.3489867789299987e+07`},
		{`JORDAN`, `1997`, `4.1615962661900036e+07`},
		{`JORDAN`, `1996`, `4.186085546840005e+07`},
		{`JORDAN`, `1995`, `3.9931672090799905e+07`},
		{`JORDAN`, `1994`, `4.070755546379999e+07`},
		{`JORDAN`, `1993`, `3.906040546580013e+07`},
		{`JORDAN`, `1992`, `4.1657604268400066e+07`},
		{`KENYA`, `1998`, `2.556633743030004e+07`},
		{`KENYA`, `1997`, `4.310884790240001e+07`},
		{`KENYA`, `1996`, `4.348295354299991e+07`},
		{`KENYA`, `1995`, `4.2517988981400006e+07`},
		{`KENYA`, `1994`, `4.3612479452299885e+07`},
		{`KENYA`, `1993`, `4.27240387571e+07`},
		{`KENYA`, `1992`, `4.3217106206800014e+07`},
		{`MOROCCO`, `1998`, `2.4915496875600014e+07`},
		{`MOROCCO`, `1997`, `4.269838285499999e+07`},
		{`MOROCCO`, `1996`, `4.298611350490005e+07`},
		{`MOROCCO`, `1995`, `4.231608915929998e+07`},
		{`MOROCCO`, `1994`, `4.345860460290005e+07`},
		{`MOROCCO`, `1993`, `4.267228806989992e+07`},
		{`MOROCCO`, `1992`, `4.2800781641499996e+07`},
		{`MOZAMBIQUE`, `1998`, `2.827987603010002e+07`},
		{`MOZAMBIQUE`, `1997`, `5.1159216229800016e+07`},
		{`MOZAMBIQUE`, `1996`, `4.807252506449996e+07`},
		{`MOZAMBIQUE`, `1995`, `4.890520060070001e+07`},
		{`MOZAMBIQUE`, `1994`, `4.609207628049998e+07`},
		{`MOZAMBIQUE`, `1993`, `4.855592626689999e+07`},
		{`MOZAMBIQUE`, `1992`, `4.780907511919999e+07`},
		{`PERU`, `1998`, `2.671396626780001e+07`},
		{`PERU`, `1997`, `4.832400860109999e+07`},
		{`PERU`, `1996`, `5.03100088629001e+07`},
		{`PERU`, `1995`, `4.964708096289996e+07`},
		{`PERU`, `1994`, `4.6420910277300104e+07`},
		{`PERU`, `1993`, `5.1536906248699956e+07`},
		{`PERU`, `1992`, `4.77116653137e+07`},
		{`ROMANIA`, `1998`, `2.7271993100999996e+07`},
		{`ROMANIA`, `1997`, `4.506305919529996e+07`},
		{`ROMANIA`, `1996`, `4.749233503229998e+07`},
		{`ROMANIA`, `1995`, `4.571063629090002e+07`},
		{`ROMANIA`, `1994`, `4.608804110660001e+07`},
		{`ROMANIA`, `1993`, `4.75150925612999e+07`},
		{`ROMANIA`, `1992`, `4.411143980440001e+07`},
		{`RUSSIA`, `1998`, `2.7935323727099977e+07`},
		{`RUSSIA`, `1997`, `4.822234729239991e+07`},
		{`RUSSIA`, `1996`, `4.75535594932001e+07`},
		{`RUSSIA`, `1995`, `4.675599009759991e+07`},
		{`RUSSIA`, `1994`, `4.800051561909993e+07`},
		{`RUSSIA`, `1993`, `4.8569624508199945e+07`},
		{`RUSSIA`, `1992`, `4.767283153289995e+07`},
		{`SAUDI ARABIA`, `1998`, `2.7113516842399996e+07`},
		{`SAUDI ARABIA`, `1997`, `4.669046896490007e+07`},
		{`SAUDI ARABIA`, `1996`, `4.777578266699988e+07`},
		{`SAUDI ARABIA`, `1995`, `4.665710782869998e+07`},
		{`SAUDI ARABIA`, `1994`, `4.8181672809999846e+07`},
		{`SAUDI ARABIA`, `1993`, `4.569255644379993e+07`},
		{`SAUDI ARABIA`, `1992`, `4.892491327169991e+07`},
		{`UNITED KINGDOM`, `1998`, `2.6366682878599998e+07`},
		{`UNITED KINGDOM`, `1997`, `4.451813018510009e+07`},
		{`UNITED KINGDOM`, `1996`, `4.553972961660003e+07`},
		{`UNITED KINGDOM`, `1995`, `4.6845879338999934e+07`},
		{`UNITED KINGDOM`, `1994`, `4.308160957369999e+07`},
		{`UNITED KINGDOM`, `1993`, `4.477014675549997e+07`},
		{`UNITED KINGDOM`, `1992`, `4.41234025484e+07`},
		{`UNITED STATES`, `1998`, `2.7826593682499964e+07`},
		{`UNITED STATES`, `1997`, `4.663857236479991e+07`},
		{`UNITED STATES`, `1996`, `4.668828054740005e+07`},
		{`UNITED STATES`, `1995`, `4.8951591615600005e+07`},
		{`UNITED STATES`, `1994`, `4.509909205979993e+07`},
		{`UNITED STATES`, `1993`, `4.618160052780006e+07`},
		{`UNITED STATES`, `1992`, `4.61682140901e+07`},
		{`VIETNAM`, `1998`, `2.728193100110004e+07`},
		{`VIETNAM`, `1997`, `4.873591417959998e+07`},
		{`VIETNAM`, `1996`, `4.782459590400004e+07`},
		{`VIETNAM`, `1995`, `4.823513580160001e+07`},
		{`VIETNAM`, `1994`, `4.772925633240001e+07`},
		{`VIETNAM`, `1993`, `4.5352676867199965e+07`},
		{`VIETNAM`, `1992`, `4.7846355648499995e+07`},
	},
	10: {
		{`57040`, `Customer#000057040`, `734235.2455000001`, `632.87`, `JAPAN`, `Eioyzjf4pp`, `22-895-641-3466`, `sits. slyly regular requests sleep alongside of the regular inst`},
		{`143347`, `Customer#000143347`, `721002.6947999999`, `2557.47`, `EGYPT`, `1aReFYv,Kw4`, `14-742-935-3718`, `ggle carefully enticing requests. final deposits use bold, bold pinto beans. ironic, idle re`},
		{`60838`, `Customer#000060838`, `679127.3077000001`, `2454.77`, `BRAZIL`, `64EaJ5vMAHWJlBOxJklpNc2RJiWE`, `12-913-494-9813`, ` need to boost against the slyly regular account`},
		{`101998`, `Customer#000101998`, `637029.5667`, `3790.89`, `UNITED KINGDOM`, `01c9CILnNtfOQYmZj`, `33-593-865-6378`, `ress foxes wake slyly after the bold excuses. ironic platelets are furiously carefully bold theodolites`},
		{`125341`, `Customer#000125341`, `633508.0860000001`, `4983.51`, `GERMANY`, `S29ODD6bceU8QSuuEJznkNaK`, `17-582-695-5962`, `arefully even depths. blithely even excuses sleep furiously. foxes use except the dependencies. ca`},
		{`25501`, `Customer#000025501`, `620269.7849000001`, `7725.04`, `ETHIOPIA`, `  W556MXuoiaYCCZamJI,Rn0B4ACUGdkQ8DZ`, `15-874-808-6793`, `he pending instructions wake carefully at the pinto beans. regular, final instructions along the slyly fina`},
		{`115831`, `Customer#000115831`, `596423.8672000001`, `5098.1`, `FRANCE`, `rFeBbEEyk dl ne7zV5fDrmiq1oK09wV7pxqCgIc`, `16-715-386-3788`, `l somas sleep. furiously final deposits wake blithely regular pinto b`},
		{`84223`, `Customer#000084223`, `594998.0238999999`, `528.65`, `UNITED KINGDOM`, `nAVZCs6BaWap rrM27N 2qBnzc5WBauxbA`, `33-442-824-8191`, ` slyly final deposits haggle regular, pending dependencies. pending escapades wake `},
		{`54289`, `Customer#000054289`, `585603.3918000001`, `5583.02`, `IRAN`, `vXCxoCsU0Bad5JQI ,oobkZ`, `20-834-292-4707`, `ely special foxes are quickly finally ironic p`},
		{`39922`, `Customer#000039922`, `584878.1133999999`, `7321.11`, `GERMANY`, `Zgy4s50l2GKN4pLDPBU8m342gIw6R`, `17-147-757-8036`, `y final requests. furiously final foxes cajole blithely special platelets. f`},
		{`6226`, `Customer#000006226`, `576783.7606`, `2230.09`, `UNITED KINGDOM`, `8gPu8,NPGkfyQQ0hcIYUGPIBWc,ybP5g,`, `33-657-701-3391`, `ending platelets along the express deposits cajole carefully final `},
		{`922`, `Customer#000000922`, `576767.5333`, `3869.25`, `GERMANY`, `Az9RFaut7NkPnc5zSD2PwHgVwr4jRzq`, `17-945-916-9648`, `luffily fluffy deposits. packages c`},
		{`147946`, `Customer#000147946`, `576455.132`, `2030.13`, `ALGERIA`, `iANyZHjqhyy7Ajah0pTrYyhJ`, `10-886-956-3143`, `ithely ironic deposits haggle blithely ironic requests. quickly regu`},
		{`115640`, `Customer#000115640`, `569341.1932999999`, `6436.1`, `ARGENTINA`, `Vtgfia9qI 7EpHgecU1X`, `11-411-543-4901`, `ost slyly along the patterns; pinto be`},
		{`73606`, `Customer#000073606`, `568656.8578`, `1785.67`, `JAPAN`, `xuR0Tro5yChDfOCrjkd2ol`, `22-437-653-6966`, `he furiously regular ideas. slowly`},
		{`110246`, `Customer#000110246`, `566842.9815`, `7763.35`, `VIETNAM`, `7KzflgX MDOq7sOkI`, `31-943-426-9837`, `egular deposits serve blithely above the fl`},
		{`142549`, `Customer#000142549`, `563537.2368000001`, `5085.99`, `INDONESIA`, `ChqEoK43OysjdHbtKCp6dKqjNyvvi9`, `19-955-562-2398`, `sleep pending courts. ironic deposits against the carefully unusual platelets cajole carefully express accounts.`},
		{`146149`, `Customer#000146149`, `557254.9865`, `1791.55`, `ROMANIA`, `s87fvzFQpU`, `29-744-164-6487`, ` of the slyly silent accounts. quickly final accounts across the `},
		{`52528`, `Customer#000052528`, `556397.3508999998`, `551.79`, `ARGENTINA`, `NFztyTOR10UOJ`, `11-208-192-3205`, ` deposits hinder. blithely pending asymptotes breach slyly regular re`},
		{`23431`, `Customer#000023431`, `554269.536`, `3381.86`, `ROMANIA`, `HgiV0phqhaIa9aydNoIlb`, `29-915-458-2654`, `nusual, even instructions: furiously stealthy n`},
	},
	// Query 11 returns 1048 rows, so we verify only the number of rows returned.
	12: {
		{`MAIL`, `6202`, `9324`},
		{`SHIP`, `6200`, `9262`},
	},
	13: {
		{`0`, `50005`},
		{`9`, `6641`},
		{`10`, `6532`},
		{`11`, `6014`},
		{`8`, `5937`},
		{`12`, `5639`},
		{`13`, `5024`},
		{`19`, `4793`},
		{`7`, `4687`},
		{`17`, `4587`},
		{`18`, `4529`},
		{`20`, `4516`},
		{`15`, `4505`},
		{`14`, `4446`},
		{`16`, `4273`},
		{`21`, `4190`},
		{`22`, `3623`},
		{`6`, `3265`},
		{`23`, `3225`},
		{`24`, `2742`},
		{`25`, `2086`},
		{`5`, `1948`},
		{`26`, `1612`},
		{`27`, `1179`},
		{`4`, `1007`},
		{`28`, `893`},
		{`29`, `593`},
		{`3`, `415`},
		{`30`, `376`},
		{`31`, `226`},
		{`32`, `148`},
		{`2`, `134`},
		{`33`, `75`},
		{`34`, `50`},
		{`35`, `37`},
		{`1`, `17`},
		{`36`, `14`},
		{`38`, `5`},
		{`37`, `5`},
		{`40`, `4`},
		{`41`, `2`},
		{`39`, `1`},
	},
	14: {
		{`16.380778626395557`},
	},
	15: {
		{`8449`, `Supplier#000008449`, `Wp34zim9qYFbVctdW`, `20-469-856-8873`, `1.7726272086999996e+06`},
	},
	// Query 16 returns 18314 rows, so we verify only the number of rows returned.
	17: {
		{`348406.05428571376`},
	},
	18: {
		{`Customer#000128120`, `128120`, `4722021`, `1994-04-07 00:00:00 +0000 +0000`, `544089.09`, `323`},
		{`Customer#000144617`, `144617`, `3043270`, `1997-02-12 00:00:00 +0000 +0000`, `530604.44`, `317`},
		{`Customer#000013940`, `13940`, `2232932`, `1997-04-13 00:00:00 +0000 +0000`, `522720.61`, `304`},
		{`Customer#000066790`, `66790`, `2199712`, `1996-09-30 00:00:00 +0000 +0000`, `515531.82`, `327`},
		{`Customer#000046435`, `46435`, `4745607`, `1997-07-03 00:00:00 +0000 +0000`, `508047.99`, `309`},
		{`Customer#000015272`, `15272`, `3883783`, `1993-07-28 00:00:00 +0000 +0000`, `500241.33`, `302`},
		{`Customer#000146608`, `146608`, `3342468`, `1994-06-12 00:00:00 +0000 +0000`, `499794.58`, `303`},
		{`Customer#000096103`, `96103`, `5984582`, `1992-03-16 00:00:00 +0000 +0000`, `494398.79`, `312`},
		{`Customer#000024341`, `24341`, `1474818`, `1992-11-15 00:00:00 +0000 +0000`, `491348.26`, `302`},
		{`Customer#000137446`, `137446`, `5489475`, `1997-05-23 00:00:00 +0000 +0000`, `487763.25`, `311`},
		{`Customer#000107590`, `107590`, `4267751`, `1994-11-04 00:00:00 +0000 +0000`, `485141.38`, `301`},
		{`Customer#000050008`, `50008`, `2366755`, `1996-12-09 00:00:00 +0000 +0000`, `483891.26`, `302`},
		{`Customer#000015619`, `15619`, `3767271`, `1996-08-07 00:00:00 +0000 +0000`, `480083.96`, `318`},
		{`Customer#000077260`, `77260`, `1436544`, `1992-09-12 00:00:00 +0000 +0000`, `479499.43`, `307`},
		{`Customer#000109379`, `109379`, `5746311`, `1996-10-10 00:00:00 +0000 +0000`, `478064.11`, `302`},
		{`Customer#000054602`, `54602`, `5832321`, `1997-02-09 00:00:00 +0000 +0000`, `471220.08`, `307`},
		{`Customer#000105995`, `105995`, `2096705`, `1994-07-03 00:00:00 +0000 +0000`, `469692.58`, `307`},
		{`Customer#000148885`, `148885`, `2942469`, `1992-05-31 00:00:00 +0000 +0000`, `469630.44`, `313`},
		{`Customer#000114586`, `114586`, `551136`, `1993-05-19 00:00:00 +0000 +0000`, `469605.59`, `308`},
		{`Customer#000105260`, `105260`, `5296167`, `1996-09-06 00:00:00 +0000 +0000`, `469360.57`, `303`},
		{`Customer#000147197`, `147197`, `1263015`, `1997-02-02 00:00:00 +0000 +0000`, `467149.67`, `320`},
		{`Customer#000064483`, `64483`, `2745894`, `1996-07-04 00:00:00 +0000 +0000`, `466991.35`, `304`},
		{`Customer#000136573`, `136573`, `2761378`, `1996-05-31 00:00:00 +0000 +0000`, `461282.73`, `301`},
		{`Customer#000016384`, `16384`, `502886`, `1994-04-12 00:00:00 +0000 +0000`, `458378.92`, `312`},
		{`Customer#000117919`, `117919`, `2869152`, `1996-06-20 00:00:00 +0000 +0000`, `456815.92`, `317`},
		{`Customer#000012251`, `12251`, `735366`, `1993-11-24 00:00:00 +0000 +0000`, `455107.26`, `309`},
		{`Customer#000120098`, `120098`, `1971680`, `1995-06-14 00:00:00 +0000 +0000`, `453451.23`, `308`},
		{`Customer#000066098`, `66098`, `5007490`, `1992-08-07 00:00:00 +0000 +0000`, `453436.16`, `304`},
		{`Customer#000117076`, `117076`, `4290656`, `1997-02-05 00:00:00 +0000 +0000`, `449545.85`, `301`},
		{`Customer#000129379`, `129379`, `4720454`, `1997-06-07 00:00:00 +0000 +0000`, `448665.79`, `303`},
		{`Customer#000126865`, `126865`, `4702759`, `1994-11-07 00:00:00 +0000 +0000`, `447606.65`, `320`},
		{`Customer#000088876`, `88876`, `983201`, `1993-12-30 00:00:00 +0000 +0000`, `446717.46`, `304`},
		{`Customer#000036619`, `36619`, `4806726`, `1995-01-17 00:00:00 +0000 +0000`, `446704.09`, `328`},
		{`Customer#000141823`, `141823`, `2806245`, `1996-12-29 00:00:00 +0000 +0000`, `446269.12`, `310`},
		{`Customer#000053029`, `53029`, `2662214`, `1993-08-13 00:00:00 +0000 +0000`, `446144.49`, `302`},
		{`Customer#000018188`, `18188`, `3037414`, `1995-01-25 00:00:00 +0000 +0000`, `443807.22`, `308`},
		{`Customer#000066533`, `66533`, `29158`, `1995-10-21 00:00:00 +0000 +0000`, `443576.5`, `305`},
		{`Customer#000037729`, `37729`, `4134341`, `1995-06-29 00:00:00 +0000 +0000`, `441082.97`, `309`},
		{`Customer#000003566`, `3566`, `2329187`, `1998-01-04 00:00:00 +0000 +0000`, `439803.36`, `304`},
		{`Customer#000045538`, `45538`, `4527553`, `1994-05-22 00:00:00 +0000 +0000`, `436275.31`, `305`},
		{`Customer#000081581`, `81581`, `4739650`, `1995-11-04 00:00:00 +0000 +0000`, `435405.9`, `305`},
		{`Customer#000119989`, `119989`, `1544643`, `1997-09-20 00:00:00 +0000 +0000`, `434568.25`, `320`},
		{`Customer#000003680`, `3680`, `3861123`, `1998-07-03 00:00:00 +0000 +0000`, `433525.97`, `301`},
		{`Customer#000113131`, `113131`, `967334`, `1995-12-15 00:00:00 +0000 +0000`, `432957.75`, `301`},
		{`Customer#000141098`, `141098`, `565574`, `1995-09-24 00:00:00 +0000 +0000`, `430986.69`, `301`},
		{`Customer#000093392`, `93392`, `5200102`, `1997-01-22 00:00:00 +0000 +0000`, `425487.51`, `304`},
		{`Customer#000015631`, `15631`, `1845057`, `1994-05-12 00:00:00 +0000 +0000`, `419879.59`, `302`},
		{`Customer#000112987`, `112987`, `4439686`, `1996-09-17 00:00:00 +0000 +0000`, `418161.49`, `305`},
		{`Customer#000012599`, `12599`, `4259524`, `1998-02-12 00:00:00 +0000 +0000`, `415200.61`, `304`},
		{`Customer#000105410`, `105410`, `4478371`, `1996-03-05 00:00:00 +0000 +0000`, `412754.51`, `302`},
		{`Customer#000149842`, `149842`, `5156581`, `1994-05-30 00:00:00 +0000 +0000`, `411329.35`, `302`},
		{`Customer#000010129`, `10129`, `5849444`, `1994-03-21 00:00:00 +0000 +0000`, `409129.85`, `309`},
		{`Customer#000069904`, `69904`, `1742403`, `1996-10-19 00:00:00 +0000 +0000`, `408513`, `305`},
		{`Customer#000017746`, `17746`, `6882`, `1997-04-09 00:00:00 +0000 +0000`, `408446.93`, `303`},
		{`Customer#000013072`, `13072`, `1481925`, `1998-03-15 00:00:00 +0000 +0000`, `399195.47`, `301`},
		{`Customer#000082441`, `82441`, `857959`, `1994-02-07 00:00:00 +0000 +0000`, `382579.74`, `305`},
		{`Customer#000088703`, `88703`, `2995076`, `1994-01-30 00:00:00 +0000 +0000`, `363812.12`, `302`},
	},
	19: {
		{`3.0838430578e+06`},
	},
	20: {
		{`Supplier#000000020`, `iybAE,RmTymrZVYaFZva2SH,j`},
		{`Supplier#000000091`, `YV45D7TkfdQanOOZ7q9QxkyGUapU1oOWU6q3`},
		{`Supplier#000000205`, `rF uV8d0JNEk`},
		{`Supplier#000000285`, `Br7e1nnt1yxrw6ImgpJ7YdhFDjuBf`},
		{`Supplier#000000287`, `7a9SP7qW5Yku5PvSg`},
		{`Supplier#000000354`, `w8fOo5W,aS`},
		{`Supplier#000000378`, `FfbhyCxWvcPrO8ltp9`},
		{`Supplier#000000402`, `i9Sw4DoyMhzhKXCH9By,AYSgmD`},
		{`Supplier#000000530`, `0qwCMwobKY OcmLyfRXlagA8ukENJv,`},
		{`Supplier#000000555`, `TfB,a5bfl3Ah 3Z 74GqnNs6zKVGM`},
		{`Supplier#000000640`, `mvvtlQKsTOsJj5Ihk7,cq`},
		{`Supplier#000000729`, `pqck2ppy758TQpZCUAjPvlU55K3QjfL7Bi`},
		{`Supplier#000000736`, `l6i2nMwVuovfKnuVgaSGK2rDy65DlAFLegiL7`},
		{`Supplier#000000761`, `zlSLelQUj2XrvTTFnv7WAcYZGvvMTx882d4`},
		{`Supplier#000000887`, `urEaTejH5POADP2ARrf`},
		{`Supplier#000000935`, `ij98czM 2KzWe7dDTOxB8sq0UfCdvrX`},
		{`Supplier#000000975`, `,AC e,tBpNwKb5xMUzeohxlRn, hdZJo73gFQF8y`},
		{`Supplier#000001263`, `rQWr6nf8ZhB2TAiIDIvo5Io`},
		{`Supplier#000001367`, `42YSkFcAXMMcucsqeEefOE4HeCC`},
		{`Supplier#000001426`, `bPOCc086oFm8sLtS,fGrH`},
		{`Supplier#000001446`, `lch9HMNU1R7a0LIybsUodVknk6`},
		{`Supplier#000001500`, `wDmF5xLxtQch9ctVu,`},
		{`Supplier#000001602`, `uKNWIeafaM644`},
		{`Supplier#000001626`, `UhxNRzUu1dtFmp0`},
		{`Supplier#000001682`, `pXTkGxrTQVyH1Rr`},
		{`Supplier#000001700`, `7hMlCof1Y5zLFg`},
		{`Supplier#000001726`, `TeRY7TtTH24sEword7yAaSkjx8`},
		{`Supplier#000001730`, `Rc8e,1Pybn r6zo0VJIEiD0UD vhk`},
		{`Supplier#000001746`, `qWsendlOekQG1aW4uq06uQaCm51se8lirv7 hBRd`},
		{`Supplier#000001806`, `M934fuZSnLW`},
		{`Supplier#000001855`, `MWk6EAeozXb`},
		{`Supplier#000001931`, `FpJbMU2h6ZR2eBv8I9NIxF`},
		{`Supplier#000002022`, ` dwebGX7Id2pc25YvY33`},
		{`Supplier#000002036`, `20ytTtVObjKUUI2WCB0A`},
		{`Supplier#000002096`, `kuxseyLtq QPLXxm9ZUrnB6Kkh92JtK5cQzzXNU `},
		{`Supplier#000002117`, `MRtkgKolHJ9Wh X9J,urANHKDzvjr`},
		{`Supplier#000002204`, `uYmlr46C06udCqanj0KiRsoTQakZsEyssL`},
		{`Supplier#000002218`, `nODZw5q4dx kp0K5`},
		{`Supplier#000002243`, `nSOEV3JeOU79`},
		{`Supplier#000002245`, `hz2qWXWVjOyKhqPYMoEwz6zFkrTaDM`},
		{`Supplier#000002282`, `ES21K9dxoW1I1TzWCj7ekdlNwSWnv1Z  6mQ,BKn`},
		{`Supplier#000002303`, `nCoWfpB6YOymbgOht7ltfklpkHl`},
		{`Supplier#000002331`, `WRh2w5WFvRg7Z0S1AvSvHCL`},
		{`Supplier#000002373`, `RzHSxOTQmElCjxIBiVA52Z JB58rJhPRylR`},
		{`Supplier#000002419`, `qydBQd14I5l5mVXa4fYY`},
		{`Supplier#000002571`, `JZUugz04c iJFLrlGsz9O N,W 1rVHNIReyq`},
		{`Supplier#000002585`, `CsPoKpw2QuTY4AV1NkWuttneIa4SN`},
		{`Supplier#000002629`, `0Bw,q5Zp8su9XrzoCngZ3cAEXZwZ`},
		{`Supplier#000002721`, `HVdFAN2JHMQSpKm`},
		{`Supplier#000002730`, `lIFxR4fzm31C6,muzJwl84z`},
		{`Supplier#000002775`, `yDclaDaBD4ihH`},
		{`Supplier#000002799`, `lwr, 6L3gdfc79PQut,4XO6nQsTJY63cAyYO`},
		{`Supplier#000002934`, `m,trBENywSArwg3DhB`},
		{`Supplier#000002941`, `Naddba 8YTEKekZyP0`},
		{`Supplier#000003028`, `jouzgX0WZjhNMWLaH4fy`},
		{`Supplier#000003095`, `HxON3jJhUi3zjt,r mTD`},
		{`Supplier#000003143`, `hdolgh608uTkHh7t6qfSqkifKaiFjnCH`},
		{`Supplier#000003185`, `hMa535Cbf2mj1Nw4OWOKWVrsK0VdDkJURrdjSIJe`},
		{`Supplier#000003189`, `DWdPxt7 RnkZv6VOByR0em`},
		{`Supplier#000003201`, `E87yws6I,t0qNs4QW7UzExKiJnJDZWue`},
		{`Supplier#000003213`, `pxrRP4irQ1VoyfQ,dTf3`},
		{`Supplier#000003275`, `9xO4nyJ2QJcX6vGf`},
		{`Supplier#000003288`, `EDdfNt7E5Uc,xLTupoIgYL4yY7ujh,`},
		{`Supplier#000003314`, `jnisU8MzqO4iUB3zsPcrysMw3DDUojS4q7LD`},
		{`Supplier#000003373`, `iy8VM48ynpc3N2OsBwAvhYakO2us9R1bi`},
		{`Supplier#000003421`, `Sh3dt9W5oeofFWovnFhrg,`},
		{`Supplier#000003422`, `DJoCEapUeBXoV1iYiCcPFQvzsTv2ZI960`},
		{`Supplier#000003441`, `zvFJIzS,oUuShHjpcX`},
		{`Supplier#000003590`, `sy79CMLxqb,Cbo`},
		{`Supplier#000003607`, `lNqFHQYjwSAkf`},
		{`Supplier#000003625`, `qY588W0Yk5iaUy1RXTgNrEKrMAjBYHcKs`},
		{`Supplier#000003723`, `jZEp0OEythCLcS OmJSrFtxJ66bMlzSp`},
		{`Supplier#000003849`, `KgbZEaRk,6Q3mWvwh6uptrs1KRUHg 0`},
		{`Supplier#000003894`, `vvGC rameLOk`},
		{`Supplier#000003941`, `Pmb05mQfBMS618O7WKqZJ 9vyv`},
		{`Supplier#000004059`, `umEYZSq9RJ2WEzdsv9meU8rmqwzVLRgiZwC`},
		{`Supplier#000004207`, `tF64pwiOM4IkWjN3mS,e06WuAjLx`},
		{`Supplier#000004236`, `dl,HPtJmGipxYsSqn9wmqkuWjst,mCeJ8O6T`},
		{`Supplier#000004278`, `bBddbpBxIVp Di9`},
		{`Supplier#000004281`, `1OwPHh Pgiyeus,iZS5eA23JDOipwk`},
		{`Supplier#000004304`, `hQCAz59k,HLlp2CKUrcBIL`},
		{`Supplier#000004346`, `S3076LEOwo`},
		{`Supplier#000004406`, `Ah0ZaLu6VwufPWUz,7kbXgYZhauEaHqGIg`},
		{`Supplier#000004430`, `yvSsKNSTL5HLXBET4luOsPNLxKzAMk`},
		{`Supplier#000004527`, `p pVXCnxgcklWF6A1o3OHY3qW6`},
		{`Supplier#000004655`, `67NqBc4 t3PG3F8aO IsqWNq4kGaPowYL`},
		{`Supplier#000004851`, `Rj,x6IgLT7kBL99nqp`},
		{`Supplier#000004871`, `,phpt6AWEnUS8t4Avb50rFfdg7O9c6nU8xxv8eC5`},
		{`Supplier#000004884`, `42Z1uLye9nsn6aTGBNd dI8 x`},
		{`Supplier#000004975`, `GPq5PMKY6Wy`},
		{`Supplier#000005076`, `Xl7h9ifgvIHmqxFLgWfHK4Gjav BkP`},
		{`Supplier#000005195`, `Woi3b2ZaicPh ZSfu1EfXhE`},
		{`Supplier#000005256`, `Onc3t57VAMchm,pmoVLaU8bONni9NsuaM PzMMFz`},
		{`Supplier#000005257`, `f9g8SEHB7obMj3QXAjXS2vfYY22`},
		{`Supplier#000005300`, `gXG28YqpxU`},
		{`Supplier#000005323`, `tMCkdqbDoyNo8vMIkzjBqYexoRAuv,T6 qzcu`},
		{`Supplier#000005386`, `Ub6AAfHpWLWP`},
		{`Supplier#000005426`, `9Dz2OVT1q sb4BK71ljQ1XjPBYRPvO`},
		{`Supplier#000005465`, `63cYZenZBRZ613Q1FaoG0,smnC5zl9`},
		{`Supplier#000005484`, `saFdOR qW7AFY,3asPqiiAa11Mo22pCoN0BtPrKo`},
		{`Supplier#000005505`, `d2sbjG43KwMPX`},
		{`Supplier#000005506`, `On f5ypzoWgB`},
		{`Supplier#000005631`, `14TVrjlzo2SJEBYCDgpMwTlvwSqC`},
		{`Supplier#000005642`, `ZwKxAv3V40tW E8P7Qwu,zlu,kPsL`},
		{`Supplier#000005686`, `f2RBKec2T1NIi7yS M`},
		{`Supplier#000005730`, `5rkb0PSews HvxkL8JaD41UpnSF2cg8H1`},
		{`Supplier#000005736`, `2dq XTYhtYWSfp`},
		{`Supplier#000005737`, `dmEWcS32C3kx,d,B95 OmYn48`},
		{`Supplier#000005797`, `,o,OebwRbSDmVl9gN9fpWPCiqB UogvlSR`},
		{`Supplier#000005875`, `lK,sYiGzB94hSyHy9xvSZFbVQNCZe2LXZuGbS`},
		{`Supplier#000005974`, `REhR5jE,lLusQXvf54SwYySgsSSVFhu`},
		{`Supplier#000006059`, `4m0cv8MwJ9yX2vlwI Z`},
		{`Supplier#000006065`, `UiI2Cy3W4Tu5sLk LuvXLRy6KihlGv`},
		{`Supplier#000006093`, `KJNUg1odUT2wtCS2s6PrH3D6fd`},
		{`Supplier#000006099`, `aZilwQKYDTVPoK`},
		{`Supplier#000006109`, `rY5gbfh3dKHnylcQUTPGCwnbe`},
		{`Supplier#000006217`, `RVN23SYT9jenUeaWGXUd`},
		{`Supplier#000006297`, `73VRDOO56GUCyvc40oYJ`},
		{`Supplier#000006435`, `xIgE69XszYbnO4Eon7cHHO8y`},
		{`Supplier#000006463`, `7 wkdj2EO49iotley2kmIM ADpLSszGV3RNWj`},
		{`Supplier#000006478`, `bQYPnj9lpmW3U`},
		{`Supplier#000006521`, `b9 2zjHzxR`},
		{`Supplier#000006642`, `N,CUclSqRLJcS8zQ`},
		{`Supplier#000006659`, `iTLsnvD8D2GzWNUv kRInwRjk5rDeEmfup1`},
		{`Supplier#000006669`, `NQ4Yryj624p7K53`},
		{`Supplier#000006748`, `rC,2rEn8gKDIS5Q0dJEoiF`},
		{`Supplier#000006761`, `n4jhxGMqB5prD1HhpLvwrWStOLlla`},
		{`Supplier#000006808`, `HGd2Xo 9nEcHJhZvXjXxWKIpApT`},
		{`Supplier#000006858`, `fnlINT885vBBhsWwTGiZ0o22thwGY16h GHJj21`},
		{`Supplier#000006946`, `To6Slo0GJTqcIvD`},
		{`Supplier#000006949`, `mLxYUJhsGcLtKe ,GFirNu183AvT`},
		{`Supplier#000007072`, `2tRyX9M1a 4Rcm57s779F1ANG9jlpK`},
		{`Supplier#000007098`, `G3j8g0KC4OcbAu2OVoPHrXQWMCUdjq8wgCHOExu`},
		{`Supplier#000007132`, `xonvn0KAQIL3p8kYk HC1FSSDSUSTC`},
		{`Supplier#000007135`, `ls DoKV7V5ulfQy9V`},
		{`Supplier#000007147`, `Xzb16kC63wmLVYexUEgB0hXFvHkjT5iPpq`},
		{`Supplier#000007160`, `TqDGBULB3cTqIT6FKDvm9BS4e4v,zwYiQPb`},
		{`Supplier#000007169`, `tEc95D2moN9S84nd55O,dlnW`},
		{`Supplier#000007278`, `I2ae3rS7KVF8GVHtB`},
		{`Supplier#000007365`, `51xhROLvQMJ05DndtZWt`},
		{`Supplier#000007398`, `V8eE6oZ00OFNU,`},
		{`Supplier#000007402`, `4UVv58ery1rjmqSR5`},
		{`Supplier#000007448`, `yhhpWiJi7EJ6Q5VCaQ`},
		{`Supplier#000007458`, `BYuucapYkptZl6fnd2QaDyZmI9gR1Ih16e`},
		{`Supplier#000007477`, `9m9j0wfhWzCvVHxkU,PpAxwSH0h`},
		{`Supplier#000007509`, `q8,V6LJRoHJjHcOuSG7aLTMg`},
		{`Supplier#000007561`, `rMcFg2530VC`},
		{`Supplier#000007616`, `R IovIqzDi3,QHnaqZk1xS4hGAgelhP4yj`},
		{`Supplier#000007760`, `JsPE18PvcdFTK`},
		{`Supplier#000007801`, `69fi,U1r6enUb `},
		{`Supplier#000007865`, `5cDGCS,T6N`},
		{`Supplier#000007885`, `u3sicchh5ZpyTUpN1cJKNcAoabIWgY`},
		{`Supplier#000007926`, `ErzCF80K9Uy`},
		{`Supplier#000007998`, `LnASFBfYRFOo9d6d,asBvVq9Lo2P`},
		{`Supplier#000008090`, `eonbJZvoDFYBNUinYfp6yERIg`},
		{`Supplier#000008224`, `TWxt9f,LVER`},
		{`Supplier#000008231`, `IK7eGw Yj90sTdpsP,vcqWxLB`},
		{`Supplier#000008243`, `2AyePMkDqmzVzjGTizXthFLo8h EiudCMxOmIIG`},
		{`Supplier#000008323`, `75I18sZmASwm POeheRMdj9tmpyeQ,BfCXN5BIAb`},
		{`Supplier#000008366`, `h778cEj14BuW9OEKlvPTWq4iwASR6EBBXN7zeS8`},
		{`Supplier#000008532`, `Uc29q4,5xVdDOF87UZrxhr4xWS0ihEUXuh`},
		{`Supplier#000008595`, `MH0iB73GQ3z UW3O DbCbqmc`},
		{`Supplier#000008610`, `SgVgP90vP452sUNTgzL9zKwXHXAzV6tV`},
		{`Supplier#000008683`, `gLuGcugfpJSeGQARnaHNCaWnGaqsNnjyl20`},
		{`Supplier#000008705`, `aE,trRNdPx,4yinTD9O3DebDIp`},
		{`Supplier#000008742`, `HmPlQEzKCPEcTUL14,kKq`},
		{`Supplier#000008841`, `I 85Lu1sekbg2xrSIzm0`},
		{`Supplier#000008872`, `8D 45GgxJO2OwwYP9S4AaXJKvDwPfLM`},
		{`Supplier#000008879`, `rDSA,D9oPM,65NMWEFrmGKAu`},
		{`Supplier#000008967`, `2kwEHyMG 7FwozNImAUE6mH0hYtqYculJM`},
		{`Supplier#000008972`, `w2vF6 D5YZO3visPXsqVfLADTK`},
		{`Supplier#000009032`, `qK,trB6Sdy4Dz1BRUFNy`},
		{`Supplier#000009043`, `57OPvKH4qyXIZ7IzYeCaw11a5N1Ki9f1WWmVQ,`},
		{`Supplier#000009278`, `RqYTzgxj93CLX 0mcYfCENOefD`},
		{`Supplier#000009326`, `XmiC,uy36B9,fb0zhcjaagiXQutg`},
		{`Supplier#000009430`, `igRqmneFt `},
		{`Supplier#000009549`, `h3RVchUf8MzY46IzbZ0ng09`},
		{`Supplier#000009601`, `51m637bO,Rw5DnHWFUvLacRx9`},
		{`Supplier#000009709`, `rRnCbHYgDgl9PZYnyWKVYSUW0vKg`},
		{`Supplier#000009753`, `wLhVEcRmd7PkJF4FBnGK7Z`},
		{`Supplier#000009799`, ` 4wNjXGa4OKWl`},
		{`Supplier#000009811`, `E3iuyq7UnZxU7oPZIe2Gu6`},
		{`Supplier#000009812`, `APFRMy3lCbgFga53n5t9DxzFPQPgnjrGt32`},
		{`Supplier#000009846`, `57sNwJJ3PtBDu,hMPP5QvpcOcSNRXn3PypJJrh`},
		{`Supplier#000009899`, `7XdpAHrzr1t,UQFZE`},
		{`Supplier#000009974`, `7wJ,J5DKcxSU4Kp1cQLpbcAvB5AsvKT`},
	},
	21: {
		{`Supplier#000002829`, `20`},
		{`Supplier#000005808`, `18`},
		{`Supplier#000000262`, `17`},
		{`Supplier#000000496`, `17`},
		{`Supplier#000002160`, `17`},
		{`Supplier#000002301`, `17`},
		{`Supplier#000002540`, `17`},
		{`Supplier#000003063`, `17`},
		{`Supplier#000005178`, `17`},
		{`Supplier#000008331`, `17`},
		{`Supplier#000002005`, `16`},
		{`Supplier#000002095`, `16`},
		{`Supplier#000005799`, `16`},
		{`Supplier#000005842`, `16`},
		{`Supplier#000006450`, `16`},
		{`Supplier#000006939`, `16`},
		{`Supplier#000009200`, `16`},
		{`Supplier#000009727`, `16`},
		{`Supplier#000000486`, `15`},
		{`Supplier#000000565`, `15`},
		{`Supplier#000001046`, `15`},
		{`Supplier#000001047`, `15`},
		{`Supplier#000001161`, `15`},
		{`Supplier#000001336`, `15`},
		{`Supplier#000001435`, `15`},
		{`Supplier#000003075`, `15`},
		{`Supplier#000003335`, `15`},
		{`Supplier#000005649`, `15`},
		{`Supplier#000006027`, `15`},
		{`Supplier#000006795`, `15`},
		{`Supplier#000006800`, `15`},
		{`Supplier#000006824`, `15`},
		{`Supplier#000007131`, `15`},
		{`Supplier#000007382`, `15`},
		{`Supplier#000008913`, `15`},
		{`Supplier#000009787`, `15`},
		{`Supplier#000000633`, `14`},
		{`Supplier#000001960`, `14`},
		{`Supplier#000002323`, `14`},
		{`Supplier#000002490`, `14`},
		{`Supplier#000002993`, `14`},
		{`Supplier#000003101`, `14`},
		{`Supplier#000004489`, `14`},
		{`Supplier#000005435`, `14`},
		{`Supplier#000005583`, `14`},
		{`Supplier#000005774`, `14`},
		{`Supplier#000007579`, `14`},
		{`Supplier#000008180`, `14`},
		{`Supplier#000008695`, `14`},
		{`Supplier#000009224`, `14`},
		{`Supplier#000000357`, `13`},
		{`Supplier#000000436`, `13`},
		{`Supplier#000000610`, `13`},
		{`Supplier#000000788`, `13`},
		{`Supplier#000000889`, `13`},
		{`Supplier#000001062`, `13`},
		{`Supplier#000001498`, `13`},
		{`Supplier#000002056`, `13`},
		{`Supplier#000002312`, `13`},
		{`Supplier#000002344`, `13`},
		{`Supplier#000002596`, `13`},
		{`Supplier#000002615`, `13`},
		{`Supplier#000002978`, `13`},
		{`Supplier#000003048`, `13`},
		{`Supplier#000003234`, `13`},
		{`Supplier#000003727`, `13`},
		{`Supplier#000003806`, `13`},
		{`Supplier#000004472`, `13`},
		{`Supplier#000005236`, `13`},
		{`Supplier#000005906`, `13`},
		{`Supplier#000006241`, `13`},
		{`Supplier#000006326`, `13`},
		{`Supplier#000006384`, `13`},
		{`Supplier#000006394`, `13`},
		{`Supplier#000006624`, `13`},
		{`Supplier#000006629`, `13`},
		{`Supplier#000006682`, `13`},
		{`Supplier#000006737`, `13`},
		{`Supplier#000006825`, `13`},
		{`Supplier#000007021`, `13`},
		{`Supplier#000007417`, `13`},
		{`Supplier#000007497`, `13`},
		{`Supplier#000007602`, `13`},
		{`Supplier#000008134`, `13`},
		{`Supplier#000008234`, `13`},
		{`Supplier#000009435`, `13`},
		{`Supplier#000009436`, `13`},
		{`Supplier#000009564`, `13`},
		{`Supplier#000009896`, `13`},
		{`Supplier#000000379`, `12`},
		{`Supplier#000000673`, `12`},
		{`Supplier#000000762`, `12`},
		{`Supplier#000000811`, `12`},
		{`Supplier#000000821`, `12`},
		{`Supplier#000001337`, `12`},
		{`Supplier#000001916`, `12`},
		{`Supplier#000001925`, `12`},
		{`Supplier#000002039`, `12`},
		{`Supplier#000002357`, `12`},
		{`Supplier#000002483`, `12`},
	},
	22: {
		{`13`, `888`, `6.737713990000005e+06`},
		{`17`, `861`, `6.460573720000007e+06`},
		{`18`, `964`, `7.236687400000006e+06`},
		{`23`, `892`, `6.701457950000001e+06`},
		{`29`, `948`, `7.158866629999988e+06`},
		{`30`, `909`, `6.80843613e+06`},
		{`31`, `922`, `6.80667018e+06`},
	},
}
