// Copyright 2019 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

// Copyright 2016 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in licenses/BSD-golang.txt.

// This code originated in Go's sync package.

package syncutil

import (
	"math/rand"
	"reflect"
	"runtime"
	"sync"
	"sync/atomic"
	"testing"
	"testing/quick"

	"github.com/stretchr/testify/require"
)

type intMapInterface mapInterface[int64, int64]

type mapOp string

const (
	opLoad          = mapOp("Load")
	opStore         = mapOp("Store")
	opLoadOrStore   = mapOp("LoadOrStore")
	opLoadAndDelete = mapOp("LoadAndDelete")
	opDelete        = mapOp("Delete")
)

var mapOps = [...]mapOp{opLoad, opStore, opLoadOrStore, opLoadAndDelete, opDelete}

// mapCall is a quick.Generator for calls on mapInterface.
type mapCall struct {
	op mapOp
	k  int64
	v  *int64
}

func (c mapCall) apply(m intMapInterface) (*int64, bool) {
	switch c.op {
	case opLoad:
		return m.Load(c.k)
	case opStore:
		m.Store(c.k, c.v)
		return nil, false
	case opLoadOrStore:
		return m.LoadOrStore(c.k, c.v)
	case opLoadAndDelete:
		return m.LoadAndDelete(c.k)
	case opDelete:
		m.Delete(c.k)
		return nil, false
	default:
		panic("invalid mapOp")
	}
}

type mapResult struct {
	value *int64
	ok    bool
}

func randValue(r *rand.Rand) *int64 {
	i := new(int64)
	*i = r.Int63()
	return i
}

func (mapCall) Generate(r *rand.Rand, size int) reflect.Value {
	c := mapCall{op: mapOps[rand.Intn(len(mapOps))], k: r.Int63()}
	switch c.op {
	case opStore, opLoadOrStore:
		c.v = randValue(r)
	}
	return reflect.ValueOf(c)
}

func applyCalls(m intMapInterface, calls []mapCall) (results []mapResult, final map[int64]*int64) {
	for _, c := range calls {
		v, ok := c.apply(m)
		results = append(results, mapResult{v, ok})
	}

	final = make(map[int64]*int64)
	m.Range(func(k int64, v *int64) bool {
		final[k] = v
		return true
	})

	return results, final
}

func applyMap(calls []mapCall) ([]mapResult, map[int64]*int64) {
	return applyCalls(new(Map[int64, int64]), calls)
}

func applyRWMutexMap(calls []mapCall) ([]mapResult, map[int64]*int64) {
	return applyCalls(new(RWMutexMap[int64, int64]), calls)
}

func applyDeepCopyMap(calls []mapCall) ([]mapResult, map[int64]*int64) {
	return applyCalls(new(DeepCopyMap[int64, int64]), calls)
}

func TestMapMatchesRWMutex(t *testing.T) {
	if err := quick.CheckEqual(applyMap, applyRWMutexMap, nil); err != nil {
		t.Error(err)
	}
}

func TestMapMatchesDeepCopy(t *testing.T) {
	if err := quick.CheckEqual(applyMap, applyDeepCopyMap, nil); err != nil {
		t.Error(err)
	}
}

func TestConcurrentRange(t *testing.T) {
	const mapSize = 1 << 10

	m := new(Map[int64, int64])
	for n := int64(1); n <= mapSize; n++ {
		v := new(int64)
		*v = n
		m.Store(n, v)
	}

	done := make(chan struct{})
	var wg sync.WaitGroup
	defer func() {
		close(done)
		wg.Wait()
	}()
	for g := int64(runtime.GOMAXPROCS(0)); g > 0; g-- {
		r := rand.New(rand.NewSource(g))
		wg.Add(1)
		go func(g int64) {
			defer wg.Done()
			for i := int64(0); ; i++ {
				select {
				case <-done:
					return
				default:
				}
				for n := int64(1); n < mapSize; n++ {
					if r.Int63n(mapSize) == 0 {
						v := new(int64)
						*v = n * i * g
						m.Store(n, v)
					} else {
						m.Load(n)
					}
				}
			}
		}(g)
	}

	iters := 1 << 10
	if testing.Short() {
		iters = 16
	}
	for n := iters; n > 0; n-- {
		seen := make(map[int64]bool, mapSize)

		m.Range(func(ki int64, vi *int64) bool {
			k, v := ki, *(vi)
			if v%k != 0 {
				t.Fatalf("while Storing multiples of %v, Range saw value %v", k, v)
			}
			if seen[k] {
				t.Fatalf("Range visited key %v twice", k)
			}
			seen[k] = true
			return true
		})

		if len(seen) != mapSize {
			t.Fatalf("Range visited %v elements of %v-element Map", len(seen), mapSize)
		}
	}
}

func TestIssue40999(t *testing.T) {
	var m Map[int64, int64]
	k := int64(0)
	v := new(int64)

	// Since the miss-counting in missLocked (via Delete)
	// compares the miss count with len(m.dirty),
	// add an initial entry to bias len(m.dirty) above the miss count.
	m.Store(0, v)

	var finalized uint32

	// Set finalizers that count for collected keys. A non-zero count
	// indicates that keys have not been leaked.
	for atomic.LoadUint32(&finalized) == 0 {
		k++
		v2 := new(int64)
		runtime.SetFinalizer(v2, func(*int64) {
			atomic.AddUint32(&finalized, 1)
		})
		m.Store(k, v2)
		m.Delete(k)
		runtime.GC()
	}
}

func TestMapRangeNestedCall(t *testing.T) { // Issue 46399
	var m Map[int64, int64]
	for i := 0; i < 3; i++ {
		m.Store(int64(i), new(int64))
	}
	m.Range(func(key int64, value *int64) bool {
		m.Range(func(key int64, value *int64) bool {
			// We should be able to load the key offered in the Range callback,
			// because there are no concurrent Delete involved in this tested map.
			if v, ok := m.Load(key); !ok || !reflect.DeepEqual(v, value) {
				t.Fatalf("Nested Range loads unexpected value, got %+v want %+v", v, value)
			}

			// We didn't keep 42 and a value into the map before, if somehow we loaded
			// a value from such a key, meaning there must be an internal bug regarding
			// nested range in the Map.
			if _, loaded := m.LoadOrStore(42, new(int64)); loaded {
				t.Fatalf("Nested Range loads unexpected value, want store a new value")
			}

			// Try to Store then LoadAndDelete the corresponding value with the key
			// 42 to the Map. In this case, the key 42 and associated value should be
			// removed from the Map. Therefore any future range won't observe key 42
			// as we checked in above.
			val := new(int64)
			m.Store(42, val)
			if v, loaded := m.LoadAndDelete(42); !loaded || !reflect.DeepEqual(v, val) {
				t.Fatalf("Nested Range loads unexpected value, got %v, want %v", v, val)
			}
			return true
		})

		// Remove key from Map on-the-fly.
		m.Delete(key)
		return true
	})

	// After a Range of Delete, all keys should be removed and any
	// further Range won't invoke the callback. Hence length remains 0.
	length := 0
	m.Range(func(key int64, value *int64) bool {
		length++
		return true
	})

	if length != 0 {
		t.Fatalf("Unexpected syncutil.Map size, got %v want %v", length, 0)
	}
}

// TestMapCheckPtr tests that the checkptr checks do not fire under race when
// using Map with values larger than the size of a pointer, despite the unsafe
// cast performed in (*entry[V]).expunged.
func TestMapCheckPtr(t *testing.T) {
	// largeStruct is a type that is larger than the size of a pointer, which
	// would be necessary to trigger the checkptr checks in (*entry[V]).expunged,
	// had the expunged pointer not been given a large size.
	type largeStruct [16]int64

	var m Map[int64, largeStruct]
	keys := make([]int64, 256)
	for i := range keys {
		keys[i] = int64(i)
	}
	v := new(largeStruct)

	// Perform a sequence of random operations on the map to try to trigger a
	// checkptr assertion failure.
	for i := 0; i < 10_000; i++ {
		k := keys[rand.Intn(len(keys))]
		op := mapOps[rand.Intn(len(mapOps))]
		switch op {
		case opLoad:
			m.Load(k)
		case opStore:
			m.Store(k, v)
		case opLoadOrStore:
			m.LoadOrStore(k, v)
		case opLoadAndDelete:
			m.LoadAndDelete(k)
		case opDelete:
			m.Delete(k)
		default:
			panic("invalid mapOp")
		}
	}
}

// TestMapStoreNilValue tests that storing a nil value in a Map is not allowed.
func TestMapStoreNilValue(t *testing.T) {
	var m Map[int64, struct{}]
	require.Panics(t, func() { m.Store(1, nil) })
	require.Panics(t, func() { m.LoadOrStore(1, nil) })
	require.NotPanics(t, func() { m.Store(1, &struct{}{}) })
	require.NotPanics(t, func() { m.LoadOrStore(1, &struct{}{}) })
}
