// Copyright 2020 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package log

import "github.com/cockroachdb/cockroach/pkg/util/syncutil"

type loggerRegistry struct {
	mu struct {
		syncutil.Mutex
		loggers []*loggerT
	}
}

// clear erases the registry.
func (r *loggerRegistry) clear() {
	r.mu.Lock()
	r.mu.loggers = nil
	r.mu.Unlock()
}

// put adds a logger into the registry.
func (r *loggerRegistry) put(l *loggerT) {
	r.mu.Lock()
	r.mu.loggers = append(r.mu.loggers, l)
	r.mu.Unlock()
}

// del removes one logger from the registry.
func (r *loggerRegistry) del(l *loggerT) {
	// Make the registry forget about this logger. This avoids
	// stacking many secondary loggers together when there are
	// subsequent tests starting servers in the same package.
	r.mu.Lock()
	defer r.mu.Unlock()
	for i, thatLogger := range r.mu.loggers {
		if thatLogger != l {
			continue
		}
		r.mu.loggers = append(r.mu.loggers[:i], r.mu.loggers[i+1:]...)
		return
	}
}

type sinkInfoRegistry struct {
	mu struct {
		syncutil.Mutex
		sinkInfos []*sinkInfo
	}
}

// clear erases the registry.
func (r *sinkInfoRegistry) clear() {
	r.mu.Lock()
	r.mu.sinkInfos = nil
	r.mu.Unlock()
}

// iter iterates over all the sinks infos and stops at the first error
// encountered.
func (r *sinkInfoRegistry) iter(fn func(l *sinkInfo) error) error {
	r.mu.Lock()
	defer r.mu.Unlock()
	for _, l := range r.mu.sinkInfos {
		if err := fn(l); err != nil {
			return err
		}
	}
	return nil
}

// iterate iterates over all the file sinks and stops at the first
// error encountered.
func (r *sinkInfoRegistry) iterFileSinks(fn func(l *fileSink) error) error {
	return r.iter(func(si *sinkInfo) error {
		if fs, ok := si.sink.(*fileSink); ok {
			return fn(fs)
		}

		if bs, ok := si.sink.(*bufferedSink); ok {
			if fs, ok := bs.child.(*fileSink); ok {
				return fn(fs)
			}
		}
		return nil
	})
}

// iterBufferedSinks iterates over all the buffered sinks and stops at the first
// error encountered.
func (r *sinkInfoRegistry) iterBufferedSinks(fn func(bs *bufferedSink) error) error {
	return r.iter(func(si *sinkInfo) error {
		if bs, ok := si.sink.(*bufferedSink); ok {
			if err := fn(bs); err != nil {
				return err
			}
		}
		return nil
	})
}

// iterHttpSink iterates over all the http sinks and stops at the first error
// encountered.
func (r *sinkInfoRegistry) iterHTTPSinks(fn func(hs *httpSink) error) error {
	return r.iter(func(si *sinkInfo) error {
		if hs, ok := si.sink.(*httpSink); ok {
			return fn(hs)
		}
		// Many times we buffer HTTP sinks, so be sure to check that case as well.
		if bs, isBuffered := si.sink.(*bufferedSink); isBuffered {
			if ch, ok := bs.child.(*httpSink); ok {
				return fn(ch)
			}
		}
		return nil
	})
}

// put adds a sinkInfo into the registry.
func (r *sinkInfoRegistry) put(l *sinkInfo) {
	r.mu.Lock()
	r.mu.sinkInfos = append(r.mu.sinkInfos, l)
	r.mu.Unlock()
}

// del removes one sinkInfo from the registry.
func (r *sinkInfoRegistry) del(l *sinkInfo) {
	r.mu.Lock()
	defer r.mu.Unlock()
	for i, thatSink := range r.mu.sinkInfos {
		if thatSink != l {
			continue
		}
		r.mu.sinkInfos = append(r.mu.sinkInfos[:i], r.mu.sinkInfos[i+1:]...)
		return
	}
}
