// Copyright 2024 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

// DO NOT EDIT THIS FILE.
// This component will be replaced with an import from a shared CRDB UI library.

import { SystemIcons } from "@cockroachlabs/icons";
import { Icon } from "@cockroachlabs/ui-components";
import classNames from "classnames";
import * as React from "react";
import { Link } from "react-router-dom";

import "./button.scss";
export type ButtonActionType = "button" | "submit" | "reset";
type ButtonIconPosition = "left" | "right";

// exported for use in type signature of <SubmitButton> component
export type ButtonCategory =
  | "primary"
  | "secondary"
  | "tertiary"
  | "flat"
  | "danger"
  | "no-style"
  | "action-menu"
  | "action-menu-item"
  | "icon-container";

export interface ButtonProps {
  category: ButtonCategory;
  actionType?: ButtonActionType;
  actionIcon?: keyof typeof SystemIcons;
  id?: string;
  className?: string;
  ariaLabel?: string;
  text?: string;
  href?: string; // external link
  to?: string; // internal link
  download?: string;
  small?: boolean;
  icon?: React.ReactNode;
  iconPosition?: ButtonIconPosition;
  disabled?: boolean;
  fluid?: boolean;
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  onClick?: (...params: any) => any;
  title?: string;
}

function generateClassNames(props: ButtonProps) {
  return classNames(
    `crl-button crl-button--${props.category} ${props.className || ""}`,
    {
      "crl-button--small": props.small,
      "crl-button--icon": props.icon && props.text,
      "crl-button--icon-only": props.icon && !props.text,
      "crl-button--icon-left":
        props.text &&
        props.icon &&
        (props.iconPosition === "left" || !props.iconPosition),
      "crl-button--icon-right":
        props.text && props.icon && props.iconPosition === "right",
      "crl-button--fluid": props.fluid,
    },
  );
}

function renderButtonContent(props: ButtonProps) {
  if (props.category === "action-menu") {
    return <Icon iconName={props.actionIcon || "Ellipsis"} size="small" />;
  }
  return (
    <>
      {props.icon && (props.iconPosition === "left" || !props.iconPosition)
        ? props.icon
        : null}
      {props.text}
      {props.icon && props.iconPosition === "right" ? props.icon : null}
    </>
  );
}

export default function Button(props: ButtonProps) {
  if (props.href) {
    return (
      <a
        id={props.id}
        aria-label={props.ariaLabel}
        target="_blank" // opens in new tab
        rel="noopener noreferrer" // prevents "tabnabbing"
        href={props.href}
        onClick={props.onClick}
        download={props.download}
        className={generateClassNames(props)}
        title={props.title}
      >
        {renderButtonContent(props)}
      </a>
    );
  }

  // If props.to has a value but props.disabled is true,
  // let's render a <button disabled=true/>, since
  // links in <Link> (and <a>) tags can't be disabled.
  if (props.to && !props.disabled) {
    return (
      <Link
        id={props.id}
        to={props.to}
        className={generateClassNames(props)}
        onClick={props.onClick}
        aria-label={props.ariaLabel}
        title={props.title}
      >
        {renderButtonContent(props)}
      </Link>
    );
  }

  return (
    <button
      type={props.actionType ? props.actionType : "button"}
      id={props.id}
      className={generateClassNames(props)}
      onClick={props.onClick}
      disabled={props.disabled}
      aria-label={props.ariaLabel}
      title={props.title}
    >
      {renderButtonContent(props)}
    </button>
  );
}
