// Copyright 2022 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package builtins

import (
	"fmt"

	"github.com/cockroachdb/errors"
	"github.com/lib/pq/oid"
)

// builtinOidsArray is an *ordered* list of function overloads. It specifies
// the fixed OID for each overload via the overload's position in the slice
// literal. The oid for a given signature must not change, which means that
// the ordering of this array must not change. The signature for a given index
// (oid) should not change except cosmetically (add a new overload instead).
// To remove a builtin, replace its signature with the empty string.
// To add a builtin, add signatures to the bottom of the slice.
// Values in this slice are function name concatenated to overload.Signature(true).
var builtinOidsArray = []string{
	// There is no 0 function oid.
	0:    `INVALID`,
	1:    `array_agg(arg1: bool) -> bool[]`,
	2:    `array_agg(arg1: box2d) -> box2d[]`,
	3:    `array_agg(arg1: int) -> int[]`,
	4:    `array_agg(arg1: float) -> float[]`,
	5:    `array_agg(arg1: decimal) -> decimal[]`,
	6:    `array_agg(arg1: date) -> date[]`,
	7:    `array_agg(arg1: timestamp) -> timestamp[]`,
	8:    `array_agg(arg1: interval) -> interval[]`,
	9:    `array_agg(arg1: geography) -> geography[]`,
	10:   `array_agg(arg1: geometry) -> geometry[]`,
	11:   `array_agg(arg1: string) -> string[]`,
	12:   `array_agg(arg1: bytes) -> bytes[]`,
	13:   `array_agg(arg1: timestamptz) -> timestamptz[]`,
	14:   `array_agg(arg1: oid) -> oid[]`,
	15:   `array_agg(arg1: uuid) -> uuid[]`,
	16:   `array_agg(arg1: inet) -> inet[]`,
	17:   `array_agg(arg1: time) -> time[]`,
	18:   `array_agg(arg1: timetz) -> timetz[]`,
	19:   `array_agg(arg1: jsonb) -> jsonb[]`,
	20:   `array_agg(arg1: varbit) -> varbit[]`,
	21:   `array_agg(arg1: anyenum) -> anyenum[]`,
	22:   `array_agg(arg1: tuple) -> tuple[]`,
	23:   `avg(arg1: int) -> decimal`,
	24:   `avg(arg1: float) -> float`,
	25:   `avg(arg1: decimal) -> decimal`,
	26:   `avg(arg1: interval) -> interval`,
	27:   `bit_and(arg1: int) -> int`,
	28:   `bit_and(arg1: varbit) -> varbit`,
	29:   `bit_or(arg1: int) -> int`,
	30:   `bit_or(arg1: varbit) -> varbit`,
	31:   `bool_and(arg1: bool) -> bool`,
	32:   `bool_or(arg1: bool) -> bool`,
	33:   `concat_agg(arg1: string) -> string`,
	34:   `concat_agg(arg1: bytes) -> bytes`,
	35:   `corr(arg1: float, arg2: float) -> float`,
	36:   `corr(arg1: int, arg2: int) -> float`,
	37:   `corr(arg1: decimal, arg2: decimal) -> float`,
	38:   `corr(arg1: float, arg2: int) -> float`,
	39:   `corr(arg1: float, arg2: decimal) -> float`,
	40:   `corr(arg1: int, arg2: float) -> float`,
	41:   `corr(arg1: int, arg2: decimal) -> float`,
	42:   `corr(arg1: decimal, arg2: float) -> float`,
	43:   `corr(arg1: decimal, arg2: int) -> float`,
	44:   `covar_pop(arg1: float, arg2: float) -> float`,
	45:   `covar_pop(arg1: int, arg2: int) -> float`,
	46:   `covar_pop(arg1: decimal, arg2: decimal) -> float`,
	47:   `covar_pop(arg1: float, arg2: int) -> float`,
	48:   `covar_pop(arg1: float, arg2: decimal) -> float`,
	49:   `covar_pop(arg1: int, arg2: float) -> float`,
	50:   `covar_pop(arg1: int, arg2: decimal) -> float`,
	51:   `covar_pop(arg1: decimal, arg2: float) -> float`,
	52:   `covar_pop(arg1: decimal, arg2: int) -> float`,
	53:   `final_covar_pop(arg1: decimal[]) -> float`,
	54:   `final_regr_sxx(arg1: decimal[]) -> float`,
	55:   `final_regr_sxy(arg1: decimal[]) -> float`,
	56:   `final_regr_syy(arg1: decimal[]) -> float`,
	57:   `final_regr_avgx(arg1: decimal[]) -> float`,
	58:   `final_regr_avgy(arg1: decimal[]) -> float`,
	59:   `final_regr_intercept(arg1: decimal[]) -> float`,
	60:   `final_regr_r2(arg1: decimal[]) -> float`,
	61:   `final_regr_slope(arg1: decimal[]) -> float`,
	62:   `final_corr(arg1: decimal[]) -> float`,
	63:   `final_covar_samp(arg1: decimal[]) -> float`,
	64:   `final_sqrdiff(arg1: decimal, arg2: decimal, arg3: int) -> decimal`,
	65:   `final_sqrdiff(arg1: float, arg2: float, arg3: int) -> float`,
	66:   `transition_regression_aggregate(arg1: float, arg2: float) -> decimal[]`,
	67:   `transition_regression_aggregate(arg1: int, arg2: int) -> decimal[]`,
	68:   `transition_regression_aggregate(arg1: decimal, arg2: decimal) -> decimal[]`,
	69:   `transition_regression_aggregate(arg1: float, arg2: int) -> decimal[]`,
	70:   `transition_regression_aggregate(arg1: float, arg2: decimal) -> decimal[]`,
	71:   `transition_regression_aggregate(arg1: int, arg2: float) -> decimal[]`,
	72:   `transition_regression_aggregate(arg1: int, arg2: decimal) -> decimal[]`,
	73:   `transition_regression_aggregate(arg1: decimal, arg2: float) -> decimal[]`,
	74:   `transition_regression_aggregate(arg1: decimal, arg2: int) -> decimal[]`,
	75:   `covar_samp(arg1: float, arg2: float) -> float`,
	76:   `covar_samp(arg1: int, arg2: int) -> float`,
	77:   `covar_samp(arg1: decimal, arg2: decimal) -> float`,
	78:   `covar_samp(arg1: float, arg2: int) -> float`,
	79:   `covar_samp(arg1: float, arg2: decimal) -> float`,
	80:   `covar_samp(arg1: int, arg2: float) -> float`,
	81:   `covar_samp(arg1: int, arg2: decimal) -> float`,
	82:   `covar_samp(arg1: decimal, arg2: float) -> float`,
	83:   `covar_samp(arg1: decimal, arg2: int) -> float`,
	84:   `regr_avgx(arg1: float, arg2: float) -> float`,
	85:   `regr_avgx(arg1: int, arg2: int) -> float`,
	86:   `regr_avgx(arg1: decimal, arg2: decimal) -> float`,
	87:   `regr_avgx(arg1: float, arg2: int) -> float`,
	88:   `regr_avgx(arg1: float, arg2: decimal) -> float`,
	89:   `regr_avgx(arg1: int, arg2: float) -> float`,
	90:   `regr_avgx(arg1: int, arg2: decimal) -> float`,
	91:   `regr_avgx(arg1: decimal, arg2: float) -> float`,
	92:   `regr_avgx(arg1: decimal, arg2: int) -> float`,
	93:   `regr_avgy(arg1: float, arg2: float) -> float`,
	94:   `regr_avgy(arg1: int, arg2: int) -> float`,
	95:   `regr_avgy(arg1: decimal, arg2: decimal) -> float`,
	96:   `regr_avgy(arg1: float, arg2: int) -> float`,
	97:   `regr_avgy(arg1: float, arg2: decimal) -> float`,
	98:   `regr_avgy(arg1: int, arg2: float) -> float`,
	99:   `regr_avgy(arg1: int, arg2: decimal) -> float`,
	100:  `regr_avgy(arg1: decimal, arg2: float) -> float`,
	101:  `regr_avgy(arg1: decimal, arg2: int) -> float`,
	102:  `regr_intercept(arg1: float, arg2: float) -> float`,
	103:  `regr_intercept(arg1: int, arg2: int) -> float`,
	104:  `regr_intercept(arg1: decimal, arg2: decimal) -> float`,
	105:  `regr_intercept(arg1: float, arg2: int) -> float`,
	106:  `regr_intercept(arg1: float, arg2: decimal) -> float`,
	107:  `regr_intercept(arg1: int, arg2: float) -> float`,
	108:  `regr_intercept(arg1: int, arg2: decimal) -> float`,
	109:  `regr_intercept(arg1: decimal, arg2: float) -> float`,
	110:  `regr_intercept(arg1: decimal, arg2: int) -> float`,
	111:  `regr_r2(arg1: float, arg2: float) -> float`,
	112:  `regr_r2(arg1: int, arg2: int) -> float`,
	113:  `regr_r2(arg1: decimal, arg2: decimal) -> float`,
	114:  `regr_r2(arg1: float, arg2: int) -> float`,
	115:  `regr_r2(arg1: float, arg2: decimal) -> float`,
	116:  `regr_r2(arg1: int, arg2: float) -> float`,
	117:  `regr_r2(arg1: int, arg2: decimal) -> float`,
	118:  `regr_r2(arg1: decimal, arg2: float) -> float`,
	119:  `regr_r2(arg1: decimal, arg2: int) -> float`,
	120:  `regr_slope(arg1: float, arg2: float) -> float`,
	121:  `regr_slope(arg1: int, arg2: int) -> float`,
	122:  `regr_slope(arg1: decimal, arg2: decimal) -> float`,
	123:  `regr_slope(arg1: float, arg2: int) -> float`,
	124:  `regr_slope(arg1: float, arg2: decimal) -> float`,
	125:  `regr_slope(arg1: int, arg2: float) -> float`,
	126:  `regr_slope(arg1: int, arg2: decimal) -> float`,
	127:  `regr_slope(arg1: decimal, arg2: float) -> float`,
	128:  `regr_slope(arg1: decimal, arg2: int) -> float`,
	129:  `regr_sxx(arg1: float, arg2: float) -> float`,
	130:  `regr_sxx(arg1: int, arg2: int) -> float`,
	131:  `regr_sxx(arg1: decimal, arg2: decimal) -> float`,
	132:  `regr_sxx(arg1: float, arg2: int) -> float`,
	133:  `regr_sxx(arg1: float, arg2: decimal) -> float`,
	134:  `regr_sxx(arg1: int, arg2: float) -> float`,
	135:  `regr_sxx(arg1: int, arg2: decimal) -> float`,
	136:  `regr_sxx(arg1: decimal, arg2: float) -> float`,
	137:  `regr_sxx(arg1: decimal, arg2: int) -> float`,
	138:  `regr_sxy(arg1: float, arg2: float) -> float`,
	139:  `regr_sxy(arg1: int, arg2: int) -> float`,
	140:  `regr_sxy(arg1: decimal, arg2: decimal) -> float`,
	141:  `regr_sxy(arg1: float, arg2: int) -> float`,
	142:  `regr_sxy(arg1: float, arg2: decimal) -> float`,
	143:  `regr_sxy(arg1: int, arg2: float) -> float`,
	144:  `regr_sxy(arg1: int, arg2: decimal) -> float`,
	145:  `regr_sxy(arg1: decimal, arg2: float) -> float`,
	146:  `regr_sxy(arg1: decimal, arg2: int) -> float`,
	147:  `regr_syy(arg1: float, arg2: float) -> float`,
	148:  `regr_syy(arg1: int, arg2: int) -> float`,
	149:  `regr_syy(arg1: decimal, arg2: decimal) -> float`,
	150:  `regr_syy(arg1: float, arg2: int) -> float`,
	151:  `regr_syy(arg1: float, arg2: decimal) -> float`,
	152:  `regr_syy(arg1: int, arg2: float) -> float`,
	153:  `regr_syy(arg1: int, arg2: decimal) -> float`,
	154:  `regr_syy(arg1: decimal, arg2: float) -> float`,
	155:  `regr_syy(arg1: decimal, arg2: int) -> float`,
	156:  `regr_count(arg1: float, arg2: float) -> int`,
	157:  `regr_count(arg1: int, arg2: int) -> int`,
	158:  `regr_count(arg1: decimal, arg2: decimal) -> int`,
	159:  `regr_count(arg1: float, arg2: int) -> int`,
	160:  `regr_count(arg1: float, arg2: decimal) -> int`,
	161:  `regr_count(arg1: int, arg2: float) -> int`,
	162:  `regr_count(arg1: int, arg2: decimal) -> int`,
	163:  `regr_count(arg1: decimal, arg2: float) -> int`,
	164:  `regr_count(arg1: decimal, arg2: int) -> int`,
	165:  `count(arg1: anyelement) -> int`,
	166:  `count_rows() -> int`,
	167:  `every(arg1: bool) -> bool`,
	168:  `max(arg1: collatedstring{*}) -> anyelement`,
	169:  `max(arg1: anyenum) -> anyelement`,
	170:  `max(arg1: bool) -> anyelement`,
	171:  `max(arg1: box2d) -> anyelement`,
	172:  `max(arg1: int) -> anyelement`,
	173:  `max(arg1: float) -> anyelement`,
	174:  `max(arg1: decimal) -> anyelement`,
	175:  `max(arg1: date) -> anyelement`,
	176:  `max(arg1: timestamp) -> anyelement`,
	177:  `max(arg1: interval) -> anyelement`,
	178:  `max(arg1: geography) -> anyelement`,
	179:  `max(arg1: geometry) -> anyelement`,
	180:  `max(arg1: string) -> anyelement`,
	181:  `max(arg1: bytes) -> anyelement`,
	182:  `max(arg1: timestamptz) -> anyelement`,
	183:  `max(arg1: oid) -> anyelement`,
	184:  `max(arg1: uuid) -> anyelement`,
	185:  `max(arg1: inet) -> anyelement`,
	186:  `max(arg1: time) -> anyelement`,
	187:  `max(arg1: timetz) -> anyelement`,
	188:  `max(arg1: jsonb) -> anyelement`,
	189:  `max(arg1: varbit) -> anyelement`,
	190:  `min(arg1: collatedstring{*}) -> anyelement`,
	191:  `min(arg1: anyenum) -> anyelement`,
	192:  `min(arg1: bool) -> anyelement`,
	193:  `min(arg1: box2d) -> anyelement`,
	194:  `min(arg1: int) -> anyelement`,
	195:  `min(arg1: float) -> anyelement`,
	196:  `min(arg1: decimal) -> anyelement`,
	197:  `min(arg1: date) -> anyelement`,
	198:  `min(arg1: timestamp) -> anyelement`,
	199:  `min(arg1: interval) -> anyelement`,
	200:  `min(arg1: geography) -> anyelement`,
	201:  `min(arg1: geometry) -> anyelement`,
	202:  `min(arg1: string) -> anyelement`,
	203:  `min(arg1: bytes) -> anyelement`,
	204:  `min(arg1: timestamptz) -> anyelement`,
	205:  `min(arg1: oid) -> anyelement`,
	206:  `min(arg1: uuid) -> anyelement`,
	207:  `min(arg1: inet) -> anyelement`,
	208:  `min(arg1: time) -> anyelement`,
	209:  `min(arg1: timetz) -> anyelement`,
	210:  `min(arg1: jsonb) -> anyelement`,
	211:  `min(arg1: varbit) -> anyelement`,
	212:  `string_agg(arg1: string, arg2: string) -> string`,
	213:  `string_agg(arg1: bytes, arg2: bytes) -> bytes`,
	214:  `sum_int(arg1: int) -> int`,
	215:  `sum(arg1: int) -> decimal`,
	216:  `sum(arg1: float) -> float`,
	217:  `sum(arg1: decimal) -> decimal`,
	218:  `sum(arg1: interval) -> interval`,
	219:  `sqrdiff(arg1: int) -> decimal`,
	220:  `sqrdiff(arg1: decimal) -> decimal`,
	221:  `sqrdiff(arg1: float) -> float`,
	222:  `final_variance(arg1: decimal, arg2: decimal, arg3: int) -> decimal`,
	223:  `final_variance(arg1: float, arg2: float, arg3: int) -> float`,
	224:  `final_var_pop(arg1: decimal, arg2: decimal, arg3: int) -> decimal`,
	225:  `final_var_pop(arg1: float, arg2: float, arg3: int) -> float`,
	226:  `final_stddev(arg1: decimal, arg2: decimal, arg3: int) -> decimal`,
	227:  `final_stddev(arg1: float, arg2: float, arg3: int) -> float`,
	228:  `final_stddev_pop(arg1: decimal, arg2: decimal, arg3: int) -> decimal`,
	229:  `final_stddev_pop(arg1: float, arg2: float, arg3: int) -> float`,
	230:  `variance(arg1: int) -> decimal`,
	231:  `variance(arg1: decimal) -> decimal`,
	232:  `variance(arg1: float) -> float`,
	233:  `var_samp(arg1: int) -> decimal`,
	234:  `var_samp(arg1: decimal) -> decimal`,
	235:  `var_samp(arg1: float) -> float`,
	236:  `var_pop(arg1: int) -> decimal`,
	237:  `var_pop(arg1: decimal) -> decimal`,
	238:  `var_pop(arg1: float) -> float`,
	239:  `stddev(arg1: int) -> decimal`,
	240:  `stddev(arg1: decimal) -> decimal`,
	241:  `stddev(arg1: float) -> float`,
	242:  `stddev_samp(arg1: int) -> decimal`,
	243:  `stddev_samp(arg1: decimal) -> decimal`,
	244:  `stddev_samp(arg1: float) -> float`,
	245:  `stddev_pop(arg1: int) -> decimal`,
	246:  `stddev_pop(arg1: decimal) -> decimal`,
	247:  `stddev_pop(arg1: float) -> float`,
	248:  `xor_agg(arg1: bytes) -> bytes`,
	249:  `xor_agg(arg1: int) -> int`,
	250:  `json_agg(arg1: anyelement) -> jsonb`,
	251:  `jsonb_agg(arg1: anyelement) -> jsonb`,
	252:  `json_object_agg(arg1: string, arg2: anyelement) -> jsonb`,
	253:  `jsonb_object_agg(arg1: string, arg2: anyelement) -> jsonb`,
	254:  `st_makeline(arg1: geometry) -> geometry`,
	255:  `st_extent(arg1: geometry) -> box2d`,
	256:  `st_union(arg1: geometry) -> geometry`,
	257:  `st_memunion(arg1: geometry) -> geometry`,
	258:  `st_collect(arg1: geometry) -> geometry`,
	259:  `st_memcollect(arg1: geometry) -> geometry`,
	260:  `any_not_null(arg1: anyelement) -> anyelement`,
	261:  `percentile_disc(arg1: float) -> anyelement`,
	262:  `percentile_disc(arg1: float[]) -> anyelement`,
	263:  `percentile_disc_impl(arg1: float, arg2: bool) -> bool`,
	264:  `percentile_disc_impl(arg1: float, arg2: box2d) -> box2d`,
	265:  `percentile_disc_impl(arg1: float, arg2: int) -> int`,
	266:  `percentile_disc_impl(arg1: float, arg2: float) -> float`,
	267:  `percentile_disc_impl(arg1: float, arg2: decimal) -> decimal`,
	268:  `percentile_disc_impl(arg1: float, arg2: date) -> date`,
	269:  `percentile_disc_impl(arg1: float, arg2: timestamp) -> timestamp`,
	270:  `percentile_disc_impl(arg1: float, arg2: interval) -> interval`,
	271:  `percentile_disc_impl(arg1: float, arg2: geography) -> geography`,
	272:  `percentile_disc_impl(arg1: float, arg2: geometry) -> geometry`,
	273:  `percentile_disc_impl(arg1: float, arg2: string) -> string`,
	274:  `percentile_disc_impl(arg1: float, arg2: bytes) -> bytes`,
	275:  `percentile_disc_impl(arg1: float, arg2: timestamptz) -> timestamptz`,
	276:  `percentile_disc_impl(arg1: float, arg2: oid) -> oid`,
	277:  `percentile_disc_impl(arg1: float, arg2: uuid) -> uuid`,
	278:  `percentile_disc_impl(arg1: float, arg2: inet) -> inet`,
	279:  `percentile_disc_impl(arg1: float, arg2: time) -> time`,
	280:  `percentile_disc_impl(arg1: float, arg2: timetz) -> timetz`,
	281:  `percentile_disc_impl(arg1: float, arg2: jsonb) -> jsonb`,
	282:  `percentile_disc_impl(arg1: float, arg2: varbit) -> varbit`,
	283:  `percentile_disc_impl(arg1: float[], arg2: bool) -> bool[]`,
	284:  `percentile_disc_impl(arg1: float[], arg2: box2d) -> box2d[]`,
	285:  `percentile_disc_impl(arg1: float[], arg2: int) -> int[]`,
	286:  `percentile_disc_impl(arg1: float[], arg2: float) -> float[]`,
	287:  `percentile_disc_impl(arg1: float[], arg2: decimal) -> decimal[]`,
	288:  `percentile_disc_impl(arg1: float[], arg2: date) -> date[]`,
	289:  `percentile_disc_impl(arg1: float[], arg2: timestamp) -> timestamp[]`,
	290:  `percentile_disc_impl(arg1: float[], arg2: interval) -> interval[]`,
	291:  `percentile_disc_impl(arg1: float[], arg2: geography) -> geography[]`,
	292:  `percentile_disc_impl(arg1: float[], arg2: geometry) -> geometry[]`,
	293:  `percentile_disc_impl(arg1: float[], arg2: string) -> string[]`,
	294:  `percentile_disc_impl(arg1: float[], arg2: bytes) -> bytes[]`,
	295:  `percentile_disc_impl(arg1: float[], arg2: timestamptz) -> timestamptz[]`,
	296:  `percentile_disc_impl(arg1: float[], arg2: oid) -> oid[]`,
	297:  `percentile_disc_impl(arg1: float[], arg2: uuid) -> uuid[]`,
	298:  `percentile_disc_impl(arg1: float[], arg2: inet) -> inet[]`,
	299:  `percentile_disc_impl(arg1: float[], arg2: time) -> time[]`,
	300:  `percentile_disc_impl(arg1: float[], arg2: timetz) -> timetz[]`,
	301:  `percentile_disc_impl(arg1: float[], arg2: jsonb) -> jsonb[]`,
	302:  `percentile_disc_impl(arg1: float[], arg2: varbit) -> varbit[]`,
	303:  `percentile_cont(arg1: float) -> float`,
	304:  `percentile_cont(arg1: float) -> interval`,
	305:  `percentile_cont(arg1: float[]) -> float[]`,
	306:  `percentile_cont(arg1: float[]) -> interval[]`,
	307:  `percentile_cont_impl(arg1: float, arg2: float) -> float`,
	308:  `percentile_cont_impl(arg1: float, arg2: interval) -> interval`,
	309:  `percentile_cont_impl(arg1: float[], arg2: float) -> float[]`,
	310:  `percentile_cont_impl(arg1: float[], arg2: interval) -> interval[]`,
	311:  `format(string, anyelement...) -> string`,
	312:  `array_to_json(array: anyelement[]) -> jsonb`,
	313:  `array_to_json(array: anyelement[], pretty_bool: bool) -> jsonb`,
	314:  `aclexplode(aclitems: string[]) -> tuple{oid AS grantor, oid AS grantee, string AS privilege_type, bool AS is_grantable}`,
	315:  `crdb_internal.scan(start_key: bytes, end_key: bytes) -> tuple{bytes AS key, bytes AS value, string AS ts}`,
	316:  `crdb_internal.scan(span: bytes[]) -> tuple{bytes AS key, bytes AS value, string AS ts}`,
	317:  `generate_series(start: int, end: int) -> int`,
	318:  `generate_series(start: int, end: int, step: int) -> int`,
	319:  `generate_series(start: timestamp, end: timestamp, step: interval) -> timestamp`,
	320:  `generate_series(start: timestamptz, end: timestamptz, step: interval) -> timestamptz`,
	321:  `crdb_internal.testing_callback(name: string) -> int`,
	322:  `pg_get_keywords() -> tuple{string AS word, string AS catcode, string AS catdesc}`,
	323:  `pg_options_to_table(options: string[]) -> tuple{string AS option_name, string AS option_value}`,
	324:  `regexp_split_to_table(string: string, pattern: string) -> string`,
	325:  `regexp_split_to_table(string: string, pattern: string, flags: string) -> string`,
	326:  `unnest(input: anyelement[]) -> anyelement`,
	327:  `unnest(anyelement[], anyelement[], anyelement[]...) -> tuple`,
	328:  `information_schema._pg_expandarray(input: anyelement[]) -> anyelement`,
	329:  `crdb_internal.unary_table() -> tuple`,
	330:  `generate_subscripts(array: anyelement[]) -> int`,
	331:  `generate_subscripts(array: anyelement[], dim: int) -> int`,
	332:  `generate_subscripts(array: anyelement[], dim: int, reverse: bool) -> int`,
	333:  `json_array_elements(input: jsonb) -> jsonb`,
	334:  `jsonb_array_elements(input: jsonb) -> jsonb`,
	335:  `json_array_elements_text(input: jsonb) -> string`,
	336:  `jsonb_array_elements_text(input: jsonb) -> string`,
	337:  `json_object_keys(input: jsonb) -> string`,
	338:  `jsonb_object_keys(input: jsonb) -> string`,
	339:  `json_each(input: jsonb) -> tuple{string AS key, jsonb AS value}`,
	340:  `jsonb_each(input: jsonb) -> tuple{string AS key, jsonb AS value}`,
	341:  `json_each_text(input: jsonb) -> tuple{string AS key, string AS value}`,
	342:  `jsonb_each_text(input: jsonb) -> tuple{string AS key, string AS value}`,
	343:  `json_populate_record(base: anyelement, from_json: jsonb) -> anyelement`,
	344:  `jsonb_populate_record(base: anyelement, from_json: jsonb) -> anyelement`,
	345:  `json_populate_recordset(base: anyelement, from_json: jsonb) -> anyelement`,
	346:  `jsonb_populate_recordset(base: anyelement, from_json: jsonb) -> anyelement`,
	347:  `crdb_internal.check_consistency(stats_only: bool, start_key: bytes, end_key: bytes) -> tuple{int AS range_id, bytes AS start_key, string AS start_key_pretty, string AS status, string AS detail, interval AS duration}`,
	348:  `crdb_internal.list_sql_keys_in_range(range_id: int) -> tuple{string AS key, string AS value, string AS ts}`,
	349:  `crdb_internal.payloads_for_span(span_id: int) -> tuple{string AS payload_type, jsonb AS payload_jsonb}`,
	350:  `crdb_internal.payloads_for_trace(trace_id: int) -> tuple{int AS span_id, string AS payload_type, jsonb AS payload_jsonb}`,
	351:  `crdb_internal.show_create_all_schemas(database_name: string) -> string`,
	352:  `crdb_internal.show_create_all_tables(database_name: string) -> string`,
	353:  `crdb_internal.show_create_all_types(database_name: string) -> string`,
	354:  `crdb_internal.decode_plan_gist(gist: string) -> string`,
	355:  `crdb_internal.decode_external_plan_gist(gist: string) -> string`,
	356:  `crdb_internal.probe_ranges(timeout: interval, probe_type: unknown_enum) -> tuple{int AS range_id, string AS error, int AS end_to_end_latency_ms, string AS verbose_trace}`,
	357:  `postgis_addbbox(geometry: geometry) -> geometry`,
	358:  `postgis_dropbbox(geometry: geometry) -> geometry`,
	359:  `postgis_hasbbox(geometry: geometry) -> bool`,
	360:  `postgis_getbbox(geometry: geometry) -> box2d`,
	361:  `postgis_extensions_upgrade() -> string`,
	362:  `postgis_full_version() -> string`,
	363:  `postgis_geos_version() -> string`,
	364:  `postgis_libxml_version() -> string`,
	365:  `postgis_lib_build_date() -> string`,
	366:  `postgis_lib_version() -> string`,
	367:  `postgis_liblwgeom_version() -> string`,
	368:  `postgis_proj_version() -> string`,
	369:  `postgis_scripts_build_date() -> string`,
	370:  `postgis_scripts_installed() -> string`,
	371:  `postgis_scripts_released() -> string`,
	372:  `postgis_version() -> string`,
	373:  `postgis_wagyu_version() -> string`,
	374:  `st_s2covering(geometry: geometry) -> geometry`,
	375:  `st_s2covering(geometry: geometry, settings: string) -> geometry`,
	376:  `st_s2covering(geography: geography) -> geography`,
	377:  `st_s2covering(geography: geography, settings: string) -> geography`,
	378:  `st_geometryfromtext(val: string) -> geometry`,
	379:  `st_geometryfromtext(str: string, srid: int) -> geometry`,
	380:  `st_geomfromewkt(val: string) -> geometry`,
	381:  `st_wkbtosql(val: bytes) -> geometry`,
	382:  `st_wkttosql(val: string) -> geometry`,
	383:  `st_geomfromwkb(val: bytes) -> geometry`,
	384:  `st_geomfromwkb(bytes: bytes, srid: int) -> geometry`,
	385:  `st_geomfromewkb(val: bytes) -> geometry`,
	386:  `st_geomfromgeojson(val: string) -> geometry`,
	387:  `st_geomfromgeojson(val: jsonb) -> geometry`,
	388:  `st_makepoint(x: float, y: float) -> geometry`,
	389:  `st_makepoint(x: float, y: float, z: float) -> geometry`,
	390:  `st_makepoint(x: float, y: float, z: float, m: float) -> geometry`,
	391:  `st_makepointm(x: float, y: float, m: float) -> geometry`,
	392:  `st_makepolygon(geometry: geometry) -> geometry`,
	393:  `st_makepolygon(outer: geometry, interior: anyelement[]) -> geometry`,
	394:  `st_polygon(geometry: geometry, srid: int) -> geometry`,
	395:  `st_geomcollfromtext(val: string) -> geometry`,
	396:  `st_geomcollfromtext(str: string, srid: int) -> geometry`,
	397:  `st_geomcollfromwkb(val: bytes) -> geometry`,
	398:  `st_geomcollfromwkb(wkb: bytes, srid: int) -> geometry`,
	399:  `st_linefromtext(val: string) -> geometry`,
	400:  `st_linefromtext(str: string, srid: int) -> geometry`,
	401:  `st_linefromwkb(val: bytes) -> geometry`,
	402:  `st_linefromwkb(wkb: bytes, srid: int) -> geometry`,
	403:  `st_linestringfromtext(val: string) -> geometry`,
	404:  `st_linestringfromtext(str: string, srid: int) -> geometry`,
	405:  `st_linestringfromwkb(val: bytes) -> geometry`,
	406:  `st_linestringfromwkb(wkb: bytes, srid: int) -> geometry`,
	407:  `st_mlinefromtext(val: string) -> geometry`,
	408:  `st_mlinefromtext(str: string, srid: int) -> geometry`,
	409:  `st_mlinefromwkb(val: bytes) -> geometry`,
	410:  `st_mlinefromwkb(wkb: bytes, srid: int) -> geometry`,
	411:  `st_mpointfromtext(val: string) -> geometry`,
	412:  `st_mpointfromtext(str: string, srid: int) -> geometry`,
	413:  `st_mpointfromwkb(val: bytes) -> geometry`,
	414:  `st_mpointfromwkb(wkb: bytes, srid: int) -> geometry`,
	415:  `st_mpolyfromtext(val: string) -> geometry`,
	416:  `st_mpolyfromtext(str: string, srid: int) -> geometry`,
	417:  `st_mpolyfromwkb(val: bytes) -> geometry`,
	418:  `st_mpolyfromwkb(wkb: bytes, srid: int) -> geometry`,
	419:  `st_multilinefromtext(val: string) -> geometry`,
	420:  `st_multilinefromtext(str: string, srid: int) -> geometry`,
	421:  `st_multilinefromwkb(val: bytes) -> geometry`,
	422:  `st_multilinefromwkb(wkb: bytes, srid: int) -> geometry`,
	423:  `st_multilinestringfromtext(val: string) -> geometry`,
	424:  `st_multilinestringfromtext(str: string, srid: int) -> geometry`,
	425:  `st_multilinestringfromwkb(val: bytes) -> geometry`,
	426:  `st_multilinestringfromwkb(wkb: bytes, srid: int) -> geometry`,
	427:  `st_multipointfromtext(val: string) -> geometry`,
	428:  `st_multipointfromtext(str: string, srid: int) -> geometry`,
	429:  `st_multipointfromwkb(val: bytes) -> geometry`,
	430:  `st_multipointfromwkb(wkb: bytes, srid: int) -> geometry`,
	431:  `st_multipolyfromtext(val: string) -> geometry`,
	432:  `st_multipolyfromtext(str: string, srid: int) -> geometry`,
	433:  `st_multipolyfromwkb(val: bytes) -> geometry`,
	434:  `st_multipolyfromwkb(wkb: bytes, srid: int) -> geometry`,
	435:  `st_multipolygonfromtext(val: string) -> geometry`,
	436:  `st_multipolygonfromtext(str: string, srid: int) -> geometry`,
	437:  `st_multipolygonfromwkb(val: bytes) -> geometry`,
	438:  `st_multipolygonfromwkb(wkb: bytes, srid: int) -> geometry`,
	439:  `st_pointfromtext(val: string) -> geometry`,
	440:  `st_pointfromtext(str: string, srid: int) -> geometry`,
	441:  `st_pointfromwkb(val: bytes) -> geometry`,
	442:  `st_pointfromwkb(wkb: bytes, srid: int) -> geometry`,
	443:  `st_polyfromtext(val: string) -> geometry`,
	444:  `st_polyfromtext(str: string, srid: int) -> geometry`,
	445:  `st_polyfromwkb(val: bytes) -> geometry`,
	446:  `st_polyfromwkb(wkb: bytes, srid: int) -> geometry`,
	447:  `st_polygonfromtext(val: string) -> geometry`,
	448:  `st_polygonfromtext(str: string, srid: int) -> geometry`,
	449:  `st_polygonfromwkb(val: bytes) -> geometry`,
	450:  `st_polygonfromwkb(wkb: bytes, srid: int) -> geometry`,
	451:  `st_geographyfromtext(val: string) -> geography`,
	452:  `st_geographyfromtext(str: string, srid: int) -> geography`,
	453:  `st_geogfromewkt(val: string) -> geography`,
	454:  `st_geogfromwkb(val: bytes) -> geography`,
	455:  `st_geogfromwkb(bytes: bytes, srid: int) -> geography`,
	456:  `st_geogfromewkb(val: bytes) -> geography`,
	457:  `st_geogfromgeojson(val: string) -> geography`,
	458:  `st_geogfromgeojson(val: jsonb) -> geography`,
	459:  `st_point(x: float, y: float) -> geometry`,
	460:  `st_pointfromgeohash(geohash: string, precision: int) -> geometry`,
	461:  `st_pointfromgeohash(geohash: string) -> geometry`,
	462:  `st_geomfromgeohash(geohash: string, precision: int) -> geometry`,
	463:  `st_geomfromgeohash(geohash: string) -> geometry`,
	464:  `st_box2dfromgeohash(geohash: string, precision: int) -> box2d`,
	465:  `st_box2dfromgeohash(geohash: string) -> box2d`,
	466:  `json_to_record(input: jsonb) -> tuple`,
	467:  `json_to_recordset(input: jsonb) -> tuple`,
	468:  `jsonb_to_record(input: jsonb) -> tuple`,
	469:  `jsonb_to_recordset(input: jsonb) -> tuple`,
	470:  `crdb_internal.to_json_as_changefeed_with_flags(tuple, string...) -> bytes`,
	474:  `st_asbinary(geometry: geometry) -> bytes`,
	475:  `st_asbinary(geography: geography) -> bytes`,
	476:  `st_asbinary(geometry: geometry, xdr_or_ndr: string) -> bytes`,
	477:  `st_asbinary(geography: geography, xdr_or_ndr: string) -> bytes`,
	478:  `st_asewkb(geometry: geometry) -> bytes`,
	479:  `st_asewkb(geography: geography) -> bytes`,
	480:  `st_ashexwkb(geometry: geometry) -> string`,
	481:  `st_ashexwkb(geography: geography) -> string`,
	482:  `st_ashexewkb(geometry: geometry) -> string`,
	483:  `st_ashexewkb(geography: geography) -> string`,
	484:  `st_ashexewkb(geometry: geometry, xdr_or_ndr: string) -> string`,
	485:  `st_ashexewkb(geography: geography, xdr_or_ndr: string) -> string`,
	486:  `st_astwkb(geometry: geometry, precision_xy: int) -> bytes`,
	487:  `st_astwkb(geometry: geometry, precision_xy: int, precision_z: int) -> bytes`,
	488:  `st_astwkb(geometry: geometry, precision_xy: int, precision_z: int, precision_m: int) -> bytes`,
	491:  `st_geohash(geometry: geometry) -> string`,
	492:  `st_geohash(geometry: geometry, precision: int) -> string`,
	493:  `st_geohash(geography: geography) -> string`,
	494:  `st_geohash(geography: geography, precision: int) -> string`,
	505:  `st_project(geography: geography, distance: float, azimuth: float) -> geography`,
	506:  `st_ndims(geometry: geometry) -> int`,
	507:  `st_dimension(geometry: geometry) -> int`,
	508:  `st_startpoint(geometry: geometry) -> geometry`,
	509:  `st_summary(geometry: geometry) -> string`,
	510:  `st_summary(geography: geography) -> string`,
	511:  `st_endpoint(geometry: geometry) -> geometry`,
	512:  `st_generatepoints(geometry: geometry, npoints: int4) -> geometry`,
	513:  `st_generatepoints(geometry: geometry, npoints: int4, seed: int4) -> geometry`,
	514:  `st_numpoints(geometry: geometry) -> int`,
	515:  `st_hasarc(geometry: geometry) -> bool`,
	516:  `st_npoints(geometry: geometry) -> int`,
	517:  `st_points(geometry: geometry) -> geometry`,
	518:  `st_exteriorring(geometry: geometry) -> geometry`,
	519:  `st_interiorringn(geometry: geometry, n: int) -> geometry`,
	520:  `st_pointn(geometry: geometry, n: int) -> geometry`,
	521:  `st_geometryn(geometry: geometry, n: int) -> geometry`,
	522:  `st_minimumclearance(geometry: geometry) -> float`,
	523:  `st_minimumclearanceline(geometry: geometry) -> geometry`,
	524:  `st_numinteriorrings(geometry: geometry) -> int`,
	525:  `st_nrings(geometry: geometry) -> int`,
	526:  `st_force2d(geometry: geometry) -> geometry`,
	527:  `st_force3dz(geometry: geometry) -> geometry`,
	528:  `st_force3dz(geometry: geometry, defaultZ: float) -> geometry`,
	529:  `st_force3dm(geometry: geometry) -> geometry`,
	530:  `st_force3dm(geometry: geometry, defaultM: float) -> geometry`,
	531:  `st_force4d(geometry: geometry) -> geometry`,
	532:  `st_force4d(geometry: geometry, defaultZ: float) -> geometry`,
	533:  `st_force4d(geometry: geometry, defaultZ: float, defaultM: float) -> geometry`,
	534:  `st_forcepolygoncw(geometry: geometry) -> geometry`,
	535:  `st_forcepolygonccw(geometry: geometry) -> geometry`,
	536:  `st_ispolygoncw(geometry: geometry) -> bool`,
	537:  `st_ispolygonccw(geometry: geometry) -> bool`,
	538:  `st_numgeometries(geometry: geometry) -> int`,
	539:  `st_x(geometry: geometry) -> float`,
	540:  `st_xmin(geometry: geometry) -> float`,
	541:  `st_xmin(box2d: box2d) -> float`,
	542:  `st_xmax(geometry: geometry) -> float`,
	543:  `st_xmax(box2d: box2d) -> float`,
	544:  `st_y(geometry: geometry) -> float`,
	545:  `st_ymin(geometry: geometry) -> float`,
	546:  `st_ymin(box2d: box2d) -> float`,
	547:  `st_ymax(geometry: geometry) -> float`,
	548:  `st_ymax(box2d: box2d) -> float`,
	549:  `st_z(geometry: geometry) -> float`,
	550:  `st_m(geometry: geometry) -> float`,
	551:  `st_zmflag(geometry: geometry) -> int2`,
	555:  `st_area2d(geometry: geometry) -> float`,
	559:  `st_length2d(geometry: geometry) -> float`,
	560:  `st_perimeter(geography: geography) -> float`,
	561:  `st_perimeter(geography: geography, use_spheroid: bool) -> float`,
	562:  `st_perimeter(geometry: geometry) -> float`,
	563:  `st_perimeter2d(geometry: geometry) -> float`,
	564:  `st_srid(geography: geography) -> int`,
	565:  `st_srid(geometry: geometry) -> int`,
	566:  `geometrytype(geometry: geometry) -> string`,
	567:  `st_geometrytype(geometry: geometry) -> string`,
	568:  `st_addmeasure(geometry: geometry, start: float, end: float) -> geometry`,
	569:  `st_lineinterpolatepoint(geometry: geometry, fraction: float) -> geometry`,
	570:  `st_lineinterpolatepoints(geometry: geometry, fraction: float) -> geometry`,
	571:  `st_lineinterpolatepoints(geometry: geometry, fraction: float, repeat: bool) -> geometry`,
	572:  `st_multi(geometry: geometry) -> geometry`,
	573:  `st_collectionextract(geometry: geometry, type: int) -> geometry`,
	574:  `st_collectionhomogenize(geometry: geometry) -> geometry`,
	575:  `st_forcecollection(geometry: geometry) -> geometry`,
	576:  `st_linefrommultipoint(geometry: geometry) -> geometry`,
	577:  `st_linemerge(geometry: geometry) -> geometry`,
	578:  `st_shiftlongitude(geometry: geometry) -> geometry`,
	579:  `st_isclosed(geometry: geometry) -> bool`,
	580:  `st_iscollection(geometry: geometry) -> bool`,
	581:  `st_isempty(geometry: geometry) -> bool`,
	582:  `st_isring(geometry: geometry) -> bool`,
	583:  `st_issimple(geometry: geometry) -> bool`,
	584:  `st_azimuth(geometry_a: geometry, geometry_b: geometry) -> float`,
	585:  `st_azimuth(geography_a: geography, geography_b: geography) -> float`,
	589:  `st_distancesphere(geometry_a: geometry, geometry_b: geometry) -> float`,
	590:  `st_distancespheroid(geometry_a: geometry, geometry_b: geometry) -> float`,
	591:  `st_frechetdistance(geometry_a: geometry, geometry_b: geometry) -> float`,
	592:  `st_frechetdistance(geometry_a: geometry, geometry_b: geometry, densify_frac: float) -> float`,
	593:  `st_hausdorffdistance(geometry_a: geometry, geometry_b: geometry) -> float`,
	594:  `st_hausdorffdistance(geometry_a: geometry, geometry_b: geometry, densify_frac: float) -> float`,
	595:  `st_maxdistance(geometry_a: geometry, geometry_b: geometry) -> float`,
	596:  `st_longestline(geometry_a: geometry, geometry_b: geometry) -> geometry`,
	597:  `st_shortestline(geometry_a: geometry, geometry_b: geometry) -> geometry`,
	602:  `st_contains(geometry_a: geometry, geometry_b: geometry) -> bool`,
	603:  `st_containsproperly(geometry_a: geometry, geometry_b: geometry) -> bool`,
	604:  `st_crosses(geometry_a: geometry, geometry_b: geometry) -> bool`,
	605:  `st_disjoint(geometry_a: geometry, geometry_b: geometry) -> bool`,
	606:  `st_dfullywithin(geometry_a: geometry, geometry_b: geometry, distance: float) -> bool`,
	610:  `st_equals(geometry_a: geometry, geometry_b: geometry) -> bool`,
	611:  `st_orderingequals(geometry_a: geometry, geometry_b: geometry) -> bool`,
	612:  `st_normalize(geometry: geometry) -> geometry`,
	615:  `st_overlaps(geometry_a: geometry, geometry_b: geometry) -> bool`,
	616:  `st_touches(geometry_a: geometry, geometry_b: geometry) -> bool`,
	617:  `st_within(geometry_a: geometry, geometry_b: geometry) -> bool`,
	618:  `st_relate(geometry_a: geometry, geometry_b: geometry) -> string`,
	619:  `st_relate(geometry_a: geometry, geometry_b: geometry, pattern: string) -> bool`,
	620:  `st_relate(geometry_a: geometry, geometry_b: geometry, bnr: int) -> string`,
	621:  `st_relatematch(intersection_matrix: string, pattern: string) -> bool`,
	622:  `st_isvalid(geometry: geometry) -> bool`,
	623:  `st_isvalid(geometry: geometry, flags: int) -> bool`,
	624:  `st_isvalidreason(geometry: geometry) -> string`,
	625:  `st_isvalidreason(geometry: geometry, flags: int) -> string`,
	626:  `st_isvalidtrajectory(geometry: geometry) -> bool`,
	627:  `st_makevalid(geometry: geometry) -> geometry`,
	628:  `st_boundary(geometry: geometry) -> geometry`,
	632:  `st_clipbybox2d(geometry: geometry, box2d: box2d) -> geometry`,
	633:  `st_convexhull(geometry: geometry) -> geometry`,
	634:  `st_difference(geometry_a: geometry, geometry_b: geometry) -> geometry`,
	635:  `st_pointonsurface(geometry: geometry) -> geometry`,
	638:  `st_sharedpaths(geometry_a: geometry, geometry_b: geometry) -> geometry`,
	639:  `st_closestpoint(geometry_a: geometry, geometry_b: geometry) -> geometry`,
	640:  `st_symdifference(geometry_a: geometry, geometry_b: geometry) -> geometry`,
	641:  `st_simplify(geometry: geometry, tolerance: float) -> geometry`,
	642:  `st_simplify(geometry: geometry, tolerance: float, preserve_collapsed: bool) -> geometry`,
	643:  `st_simplifypreservetopology(geometry: geometry, tolerance: float) -> geometry`,
	644:  `st_setsrid(geometry: geometry, srid: int) -> geometry`,
	645:  `st_setsrid(geography: geography, srid: int) -> geography`,
	646:  `st_transform(geometry: geometry, srid: int) -> geometry`,
	647:  `st_transform(geometry: geometry, to_proj_text: string) -> geometry`,
	648:  `st_transform(geometry: geometry, from_proj_text: string, to_proj_text: string) -> geometry`,
	649:  `st_transform(geometry: geometry, from_proj_text: string, srid: int) -> geometry`,
	650:  `st_translate(g: geometry, delta_x: float, delta_y: float) -> geometry`,
	651:  `st_translate(g: geometry, delta_x: float, delta_y: float, delta_z: float) -> geometry`,
	652:  `st_affine(geometry: geometry, a: float, b: float, d: float, e: float, x_off: float, y_off: float) -> geometry`,
	653:  `st_affine(geometry: geometry, a: float, b: float, c: float, d: float, e: float, f: float, g: float, h: float, i: float, x_off: float, y_off: float, z_off: float) -> geometry`,
	654:  `st_scale(geometry: geometry, x_factor: float, y_factor: float) -> geometry`,
	655:  `st_scale(g: geometry, factor: geometry) -> geometry`,
	656:  `st_scale(g: geometry, factor: geometry, origin: geometry) -> geometry`,
	657:  `st_rotate(g: geometry, angle_radians: float) -> geometry`,
	658:  `st_rotate(g: geometry, angle_radians: float, origin_x: float, origin_y: float) -> geometry`,
	659:  `st_rotate(g: geometry, angle_radians: float, origin_point: geometry) -> geometry`,
	660:  `st_rotatex(g: geometry, angle_radians: float) -> geometry`,
	661:  `st_rotatey(g: geometry, angle_radians: float) -> geometry`,
	662:  `st_rotatez(g: geometry, angle_radians: float) -> geometry`,
	663:  `st_addpoint(line_string: geometry, point: geometry, index: int) -> geometry`,
	664:  `st_addpoint(line_string: geometry, point: geometry) -> geometry`,
	665:  `st_setpoint(line_string: geometry, index: int, point: geometry) -> geometry`,
	666:  `st_removepoint(line_string: geometry, index: int) -> geometry`,
	667:  `st_removerepeatedpoints(geometry: geometry, tolerance: float) -> geometry`,
	668:  `st_removerepeatedpoints(geometry: geometry) -> geometry`,
	669:  `st_reverse(geometry: geometry) -> geometry`,
	670:  `st_segmentize(geography: geography, max_segment_length_meters: float) -> geography`,
	671:  `st_segmentize(geometry: geometry, max_segment_length: float) -> geometry`,
	672:  `st_snaptogrid(geometry: geometry, size: float) -> geometry`,
	673:  `st_snaptogrid(geometry: geometry, size_x: float, size_y: float) -> geometry`,
	674:  `st_snaptogrid(geometry: geometry, origin_x: float, origin_y: float, size_x: float, size_y: float) -> geometry`,
	675:  `st_snaptogrid(geometry: geometry, origin: geometry, size_x: float, size_y: float, size_z: float, size_m: float) -> geometry`,
	676:  `st_snap(input: geometry, target: geometry, tolerance: float) -> geometry`,
	685:  `st_envelope(geometry: geometry) -> geometry`,
	686:  `st_envelope(box2d: box2d) -> geometry`,
	687:  `st_makeenvelope(xmin: float, ymin: float, xmax: float, ymax: float, srid: int) -> geometry`,
	688:  `st_makeenvelope(xmin: float, ymin: float, xmax: float, ymax: float) -> geometry`,
	689:  `st_flipcoordinates(geometry: geometry) -> geometry`,
	690:  `st_swapordinates(geometry: geometry, swap_ordinate_string: string) -> geometry`,
	691:  `st_angle(point1: geometry, point2: geometry, point3: geometry, point4: geometry) -> float`,
	692:  `st_angle(point1: geometry, point2: geometry, point3: geometry) -> float`,
	693:  `st_angle(line1: geometry, line2: geometry) -> float`,
	694:  `st_asencodedpolyline(geometry: geometry) -> string`,
	695:  `st_asencodedpolyline(geometry: geometry, precision: int4) -> string`,
	696:  `st_linefromencodedpolyline(encoded_polyline: string) -> geometry`,
	697:  `st_linefromencodedpolyline(encoded_polyline: string, precision: int4) -> geometry`,
	698:  `st_unaryunion(geometry: geometry) -> geometry`,
	699:  `st_node(geometry: geometry) -> geometry`,
	700:  `st_subdivide(geometry: geometry) -> geometry`,
	701:  `st_subdivide(geometry: geometry, max_vertices: int4) -> geometry`,
	702:  `st_makebox2d(geometry_a: geometry, geometry_b: geometry) -> box2d`,
	703:  `st_combinebbox(box2d: box2d, geometry: geometry) -> box2d`,
	704:  `st_expand(box2d: box2d, delta: float) -> box2d`,
	705:  `st_expand(box2d: box2d, delta_x: float, delta_y: float) -> box2d`,
	706:  `st_expand(geometry: geometry, delta: float) -> geometry`,
	707:  `st_expand(geometry: geometry, delta_x: float, delta_y: float) -> geometry`,
	708:  `st_estimatedextent(schema_name: string, table_name: string, geocolumn_name: string, parent_only: bool) -> box2d`,
	709:  `st_estimatedextent(schema_name: string, table_name: string, geocolumn_name: string) -> box2d`,
	710:  `st_estimatedextent(table_name: string, geocolumn_name: string) -> box2d`,
	711:  `addgeometrycolumn(table_name: string, column_name: string, srid: int, type: string, dimension: int, use_typmod: bool) -> string`,
	712:  `addgeometrycolumn(schema_name: string, table_name: string, column_name: string, srid: int, type: string, dimension: int, use_typmod: bool) -> string`,
	713:  `addgeometrycolumn(catalog_name: string, schema_name: string, table_name: string, column_name: string, srid: int, type: string, dimension: int, use_typmod: bool) -> string`,
	714:  `addgeometrycolumn(table_name: string, column_name: string, srid: int, type: string, dimension: int) -> string`,
	715:  `addgeometrycolumn(schema_name: string, table_name: string, column_name: string, srid: int, type: string, dimension: int) -> string`,
	716:  `addgeometrycolumn(catalog_name: string, schema_name: string, table_name: string, column_name: string, srid: int, type: string, dimension: int) -> string`,
	720:  `st_dfullywithinexclusive(geometry_a: geometry, geometry_b: geometry, distance: float) -> bool`,
	721:  `st_pointinsidecircle(geometry: geometry, x_coord: float, y_coord: float, radius: float) -> bool`,
	722:  `st_memsize(geometry: geometry) -> int`,
	723:  `st_linelocatepoint(line: geometry, point: geometry) -> float`,
	724:  `st_minimumboundingradius(geometry: geometry) -> tuple{geometry AS center, float AS radius}`,
	725:  `st_minimumboundingcircle(geometry: geometry) -> geometry`,
	726:  `st_minimumboundingcircle(geometry: geometry,  num_segs: int) -> geometry`,
	727:  `st_transscale(geometry: geometry, delta_x: float, delta_y: float, x_factor: float, y_factor: float) -> geometry`,
	728:  `st_voronoipolygons(geometry: geometry) -> geometry`,
	729:  `st_voronoipolygons(geometry: geometry, tolerance: float) -> geometry`,
	730:  `st_voronoipolygons(geometry: geometry, tolerance: float, extend_to: geometry) -> geometry`,
	731:  `st_voronoilines(geometry: geometry) -> geometry`,
	732:  `st_voronoilines(geometry: geometry, tolerance: float) -> geometry`,
	733:  `st_voronoilines(geometry: geometry, tolerance: float, extend_to: geometry) -> geometry`,
	734:  `st_orientedenvelope(geometry: geometry) -> geometry`,
	735:  `st_linesubstring(linestring: geometry, start_fraction: float, end_fraction: float) -> geometry`,
	736:  `st_linesubstring(linestring: geometry, start_fraction: decimal, end_fraction: decimal) -> geometry`,
	737:  `st_linecrossingdirection(linestring_a: geometry, linestring_b: geometry) -> int`,
	738:  `abs(val: float) -> float`,
	739:  `abs(val: decimal) -> decimal`,
	740:  `abs(val: int) -> int`,
	741:  `acos(val: float) -> float`,
	742:  `acosd(val: float) -> float`,
	743:  `acosh(val: float) -> float`,
	744:  `asin(val: float) -> float`,
	745:  `asind(val: float) -> float`,
	746:  `asinh(val: float) -> float`,
	747:  `atan(val: float) -> float`,
	748:  `atand(val: float) -> float`,
	749:  `atanh(val: float) -> float`,
	750:  `atan2(x: float, y: float) -> float`,
	751:  `atan2d(x: float, y: float) -> float`,
	752:  `cbrt(val: float) -> float`,
	753:  `cbrt(val: decimal) -> decimal`,
	754:  `ceil(val: float) -> float`,
	755:  `ceil(val: decimal) -> decimal`,
	756:  `ceil(val: int) -> float`,
	757:  `ceiling(val: float) -> float`,
	758:  `ceiling(val: decimal) -> decimal`,
	759:  `ceiling(val: int) -> float`,
	760:  `cos(val: float) -> float`,
	761:  `cosd(val: float) -> float`,
	762:  `cosh(val: float) -> float`,
	763:  `cot(val: float) -> float`,
	764:  `cotd(val: float) -> float`,
	765:  `degrees(val: float) -> float`,
	766:  `div(x: float, y: float) -> float`,
	767:  `div(x: decimal, y: decimal) -> decimal`,
	768:  `div(x: int, y: int) -> int`,
	769:  `exp(val: float) -> float`,
	770:  `exp(val: decimal) -> decimal`,
	771:  `floor(val: float) -> float`,
	772:  `floor(val: decimal) -> decimal`,
	773:  `floor(val: int) -> float`,
	774:  `isnan(val: float) -> bool`,
	775:  `isnan(val: decimal) -> bool`,
	776:  `ln(val: float) -> float`,
	777:  `ln(val: decimal) -> decimal`,
	778:  `log(val: float) -> float`,
	779:  `log(b: float, x: float) -> float`,
	780:  `log(val: decimal) -> decimal`,
	781:  `log(b: decimal, x: decimal) -> decimal`,
	782:  `mod(x: float, y: float) -> float`,
	783:  `mod(x: decimal, y: decimal) -> decimal`,
	784:  `mod(x: int, y: int) -> int`,
	785:  `pi() -> float`,
	786:  `pow(x: float, y: float) -> float`,
	787:  `pow(x: decimal, y: decimal) -> decimal`,
	788:  `pow(x: int, y: int) -> int`,
	789:  `power(x: float, y: float) -> float`,
	790:  `power(x: decimal, y: decimal) -> decimal`,
	791:  `power(x: int, y: int) -> int`,
	792:  `radians(val: float) -> float`,
	793:  `round(val: float) -> float`,
	794:  `round(val: decimal) -> decimal`,
	795:  `round(input: float, decimal_accuracy: int) -> float`,
	796:  `round(input: decimal, decimal_accuracy: int) -> decimal`,
	797:  `sin(val: float) -> float`,
	798:  `sind(val: float) -> float`,
	799:  `sinh(val: float) -> float`,
	800:  `sign(val: float) -> float`,
	801:  `sign(val: decimal) -> decimal`,
	802:  `sign(val: int) -> int`,
	803:  `sqrt(val: float) -> float`,
	804:  `sqrt(val: decimal) -> decimal`,
	805:  `tan(val: float) -> float`,
	806:  `tand(val: float) -> float`,
	807:  `tanh(val: float) -> float`,
	808:  `trunc(val: float) -> float`,
	809:  `trunc(val: decimal) -> decimal`,
	810:  `trunc(val: decimal, scale: int) -> decimal`,
	811:  `width_bucket(operand: decimal, b1: decimal, b2: decimal, count: int) -> int`,
	812:  `width_bucket(operand: int, b1: int, b2: int, count: int) -> int`,
	813:  `width_bucket(operand: anyelement, thresholds: anyelement[]) -> int`,
	814:  `length(val: string) -> int`,
	815:  `length(val: bytes) -> int`,
	816:  `length(val: varbit) -> int`,
	817:  `char_length(val: string) -> int`,
	818:  `char_length(val: bytes) -> int`,
	819:  `character_length(val: string) -> int`,
	820:  `character_length(val: bytes) -> int`,
	821:  `bit_length(val: string) -> int`,
	822:  `bit_length(val: bytes) -> int`,
	823:  `bit_length(val: varbit) -> int`,
	824:  `octet_length(val: string) -> int`,
	825:  `octet_length(val: bytes) -> int`,
	826:  `octet_length(val: varbit) -> int`,
	827:  `lower(val: string) -> string`,
	828:  `unaccent(val: string) -> string`,
	829:  `upper(val: string) -> string`,
	830:  `prettify_statement(val: string) -> string`,
	831:  `prettify_statement(statement: string, line_width: int, align_mode: int, case_mode: int) -> string`,
	832:  `substr(input: string, start_pos: int) -> string`,
	833:  `substr(input: string, start_pos: int, length: int) -> string`,
	834:  `substr(input: string, regex: string) -> string`,
	835:  `substr(input: string, regex: string, escape_char: string) -> string`,
	836:  `substr(input: varbit, start_pos: int) -> varbit`,
	837:  `substr(input: varbit, start_pos: int, length: int) -> varbit`,
	838:  `substr(input: bytes, start_pos: int) -> bytes`,
	839:  `substr(input: bytes, start_pos: int, length: int) -> bytes`,
	840:  `substring(input: string, start_pos: int) -> string`,
	841:  `substring(input: string, start_pos: int, length: int) -> string`,
	842:  `substring(input: string, regex: string) -> string`,
	843:  `substring(input: string, regex: string, escape_char: string) -> string`,
	844:  `substring(input: varbit, start_pos: int) -> varbit`,
	845:  `substring(input: varbit, start_pos: int, length: int) -> varbit`,
	846:  `substring(input: bytes, start_pos: int) -> bytes`,
	847:  `substring(input: bytes, start_pos: int, length: int) -> bytes`,
	848:  `concat(anyelement...) -> string`,
	849:  `concat_ws(string...) -> string`,
	850:  `convert_from(str: bytes, enc: string) -> string`,
	851:  `convert_to(str: string, enc: string) -> bytes`,
	852:  `get_bit(bit_string: varbit, index: int) -> int`,
	853:  `get_bit(byte_string: bytes, index: int) -> int`,
	854:  `get_byte(byte_string: bytes, index: int) -> int`,
	855:  `set_bit(bit_string: varbit, index: int, to_set: int) -> varbit`,
	856:  `set_bit(byte_string: bytes, index: int, to_set: int) -> bytes`,
	857:  `set_byte(byte_string: bytes, index: int, to_set: int) -> bytes`,
	858:  `uuid_generate_v4() -> uuid`,
	859:  `uuid_nil() -> uuid`,
	860:  `uuid_ns_dns() -> uuid`,
	861:  `uuid_ns_url() -> uuid`,
	862:  `uuid_ns_oid() -> uuid`,
	863:  `uuid_ns_x500() -> uuid`,
	864:  `uuid_generate_v1() -> uuid`,
	865:  `uuid_generate_v1mc() -> uuid`,
	866:  `uuid_generate_v3(namespace: uuid, name: string) -> uuid`,
	867:  `uuid_generate_v5(namespace: uuid, name: string) -> uuid`,
	868:  `to_uuid(val: string) -> bytes`,
	869:  `from_uuid(val: bytes) -> string`,
	870:  `gen_random_ulid() -> uuid`,
	871:  `uuid_to_ulid(val: uuid) -> string`,
	872:  `ulid_to_uuid(val: string) -> uuid`,
	873:  `abbrev(val: inet) -> string`,
	874:  `broadcast(val: inet) -> inet`,
	875:  `family(val: inet) -> int`,
	876:  `host(val: inet) -> string`,
	877:  `hostmask(val: inet) -> inet`,
	878:  `masklen(val: inet) -> int`,
	879:  `netmask(val: inet) -> inet`,
	880:  `set_masklen(val: inet, prefixlen: int) -> inet`,
	881:  `text(inet: inet) -> string`,
	882:  `inet_same_family(val: inet, val: inet) -> bool`,
	883:  `inet_contained_by_or_equals(val: inet, container: inet) -> bool`,
	884:  `inet_contains_or_equals(container: inet, val: inet) -> bool`,
	885:  `inet(val: string) -> inet`,
	886:  `from_ip(val: bytes) -> string`,
	887:  `to_ip(val: string) -> bytes`,
	888:  `split_part(input: string, delimiter: string, return_index_pos: int) -> string`,
	889:  `repeat(input: string, repeat_counter: int) -> string`,
	890:  `encode(data: bytes, format: string) -> string`,
	891:  `decode(text: string, format: string) -> bytes`,
	892:  `compress(data: bytes, codec: string) -> bytes`,
	893:  `decompress(data: bytes, codec: string) -> bytes`,
	894:  `ascii(val: string) -> int`,
	895:  `chr(val: int) -> string`,
	896:  `md5(string...) -> string`,
	897:  `md5(bytes...) -> string`,
	898:  `sha1(string...) -> string`,
	899:  `sha1(bytes...) -> string`,
	900:  `sha224(string...) -> string`,
	901:  `sha224(bytes...) -> string`,
	902:  `sha256(string...) -> string`,
	903:  `sha256(bytes...) -> string`,
	904:  `sha384(string...) -> string`,
	905:  `sha384(bytes...) -> string`,
	906:  `sha512(string...) -> string`,
	907:  `sha512(bytes...) -> string`,
	908:  `fnv32(string...) -> int`,
	909:  `fnv32(bytes...) -> int`,
	910:  `fnv32a(string...) -> int`,
	911:  `fnv32a(bytes...) -> int`,
	912:  `fnv64(string...) -> int`,
	913:  `fnv64(bytes...) -> int`,
	914:  `fnv64a(string...) -> int`,
	915:  `fnv64a(bytes...) -> int`,
	916:  `crc32ieee(string...) -> int`,
	917:  `crc32ieee(bytes...) -> int`,
	918:  `crc32c(string...) -> int`,
	919:  `crc32c(bytes...) -> int`,
	920:  `to_hex(val: int) -> string`,
	921:  `to_hex(val: bytes) -> string`,
	922:  `to_hex(val: string) -> string`,
	923:  `to_english(val: int) -> string`,
	924:  `strpos(input: string, find: string) -> int`,
	925:  `strpos(input: varbit, find: varbit) -> int`,
	926:  `strpos(input: bytes, find: bytes) -> int`,
	927:  `overlay(input: string, overlay_val: string, start_pos: int) -> string`,
	928:  `overlay(input: string, overlay_val: string, start_pos: int, end_pos: int) -> string`,
	929:  `lpad(string: string, length: int) -> string`,
	930:  `lpad(string: string, length: int, fill: string) -> string`,
	931:  `rpad(string: string, length: int) -> string`,
	932:  `rpad(string: string, length: int, fill: string) -> string`,
	933:  `btrim(input: string, trim_chars: string) -> string`,
	934:  `btrim(val: string) -> string`,
	935:  `ltrim(input: string, trim_chars: string) -> string`,
	936:  `ltrim(val: string) -> string`,
	937:  `rtrim(input: string, trim_chars: string) -> string`,
	938:  `rtrim(val: string) -> string`,
	939:  `reverse(val: string) -> string`,
	940:  `replace(input: string, find: string, replace: string) -> string`,
	941:  `translate(input: string, find: string, replace: string) -> string`,
	942:  `regexp_extract(input: string, regex: string) -> string`,
	943:  `regexp_replace(input: string, regex: string, replace: string) -> string`,
	944:  `regexp_replace(input: string, regex: string, replace: string, flags: string) -> string`,
	945:  `regexp_split_to_array(string: string, pattern: string) -> string[]`,
	946:  `regexp_split_to_array(string: string, pattern: string, flags: string) -> string[]`,
	947:  `like_escape(unescaped: string, pattern: string, escape: string) -> bool`,
	948:  `not_like_escape(unescaped: string, pattern: string, escape: string) -> bool`,
	949:  `ilike_escape(unescaped: string, pattern: string, escape: string) -> bool`,
	950:  `not_ilike_escape(unescaped: string, pattern: string, escape: string) -> bool`,
	951:  `similar_escape(pattern: string) -> string`,
	952:  `similar_escape(pattern: string, escape: string) -> string`,
	953:  `similar_to_escape(pattern: string) -> string`,
	954:  `similar_to_escape(pattern: string, escape: string) -> string`,
	955:  `similar_to_escape(unescaped: string, pattern: string, escape: string) -> bool`,
	956:  `not_similar_to_escape(unescaped: string, pattern: string, escape: string) -> bool`,
	957:  `initcap(val: string) -> string`,
	958:  `quote_ident(val: string) -> string`,
	959:  `quote_literal(val: string) -> string`,
	960:  `quote_literal(val: anyelement) -> string`,
	961:  `quote_nullable(val: string) -> string`,
	962:  `quote_nullable(val: anyelement) -> string`,
	963:  `left(input: bytes, return_set: int) -> bytes`,
	964:  `left(input: string, return_set: int) -> string`,
	965:  `right(input: bytes, return_set: int) -> bytes`,
	966:  `right(input: string, return_set: int) -> string`,
	967:  `random() -> float`,
	968:  `unique_rowid() -> int`,
	969:  `unordered_unique_rowid() -> int`,
	970:  `nextval(sequence_name: string) -> int`,
	971:  `nextval(sequence_name: regclass) -> int`,
	972:  `currval(sequence_name: string) -> int`,
	973:  `currval(sequence_name: regclass) -> int`,
	974:  `lastval() -> int`,
	975:  `setval(sequence_name: string, value: int) -> int`,
	976:  `setval(sequence_name: regclass, value: int) -> int`,
	977:  `setval(sequence_name: string, value: int, is_called: bool) -> int`,
	978:  `setval(sequence_name: regclass, value: int, is_called: bool) -> int`,
	979:  `experimental_uuid_v4() -> bytes`,
	980:  `uuid_v4() -> bytes`,
	981:  `greatest(anyelement...) -> anyelement`,
	982:  `least(anyelement...) -> anyelement`,
	983:  `strftime(input: timestamp, extract_format: string) -> string`,
	984:  `strftime(input: date, extract_format: string) -> string`,
	985:  `strftime(input: timestamptz, extract_format: string) -> string`,
	986:  `experimental_strftime(input: timestamp, extract_format: string) -> string`,
	987:  `experimental_strftime(input: date, extract_format: string) -> string`,
	988:  `experimental_strftime(input: timestamptz, extract_format: string) -> string`,
	989:  `strptime(input: string, format: string) -> timestamptz`,
	990:  `experimental_strptime(input: string, format: string) -> timestamptz`,
	991:  `to_char(interval: interval) -> string`,
	992:  `to_char(timestamp: timestamp) -> string`,
	993:  `to_char(date: date) -> string`,
	994:  `to_char_with_style(interval: interval, style: string) -> string`,
	995:  `to_char_with_style(timestamp: timestamp, datestyle: string) -> string`,
	996:  `to_char_with_style(date: date, datestyle: string) -> string`,
	997:  `to_timestamp(timestamp: float) -> timestamptz`,
	998:  `age(val: timestamptz) -> interval`,
	999:  `age(end: timestamptz, begin: timestamptz) -> interval`,
	1000: `current_date() -> date`,
	1001: `now() -> timestamptz`,
	1002: `now() -> timestamp`,
	1003: `now() -> date`,
	1004: `current_time() -> timetz`,
	1005: `current_time() -> time`,
	1006: `current_time(precision: int) -> timetz`,
	1007: `current_time(precision: int) -> time`,
	1008: `current_timestamp(precision: int) -> timestamptz`,
	1009: `current_timestamp(precision: int) -> timestamp`,
	1010: `current_timestamp(precision: int) -> date`,
	1011: `current_timestamp() -> timestamptz`,
	1012: `current_timestamp() -> timestamp`,
	1013: `current_timestamp() -> date`,
	1014: `transaction_timestamp() -> timestamptz`,
	1015: `transaction_timestamp() -> timestamp`,
	1016: `transaction_timestamp() -> date`,
	1017: `localtimestamp(precision: int) -> timestamptz`,
	1018: `localtimestamp(precision: int) -> timestamp`,
	1019: `localtimestamp(precision: int) -> date`,
	1020: `localtimestamp() -> timestamptz`,
	1021: `localtimestamp() -> timestamp`,
	1022: `localtimestamp() -> date`,
	1023: `localtime() -> timetz`,
	1024: `localtime() -> time`,
	1025: `localtime(precision: int) -> timetz`,
	1026: `localtime(precision: int) -> time`,
	1027: `statement_timestamp() -> timestamptz`,
	1028: `statement_timestamp() -> timestamp`,
	1029: `follower_read_timestamp() -> timestamptz`,
	1030: `experimental_follower_read_timestamp() -> timestamptz`,
	1031: `with_min_timestamp(min_timestamp: timestamptz) -> timestamptz`,
	1032: `with_min_timestamp(min_timestamp: timestamptz, nearest_only: bool) -> timestamptz`,
	1033: `with_max_staleness(max_staleness: interval) -> timestamptz`,
	1034: `with_max_staleness(max_staleness: interval, nearest_only: bool) -> timestamptz`,
	1035: `cluster_logical_timestamp() -> decimal`,
	1036: `hlc_to_timestamp(hlc: decimal) -> timestamptz`,
	1037: `clock_timestamp() -> timestamptz`,
	1038: `clock_timestamp() -> timestamp`,
	1039: `timeofday() -> string`,
	1040: `extract(element: string, input: timestamp) -> float`,
	1041: `extract(element: string, input: interval) -> float`,
	1042: `extract(element: string, input: date) -> float`,
	1043: `extract(element: string, input: timestamptz) -> float`,
	1044: `extract(element: string, input: time) -> float`,
	1045: `extract(element: string, input: timetz) -> float`,
	1046: `date_part(element: string, input: timestamp) -> float`,
	1047: `date_part(element: string, input: interval) -> float`,
	1048: `date_part(element: string, input: date) -> float`,
	1049: `date_part(element: string, input: timestamptz) -> float`,
	1050: `date_part(element: string, input: time) -> float`,
	1051: `date_part(element: string, input: timetz) -> float`,
	1052: `extract_duration(element: string, input: interval) -> int`,
	1053: `date_trunc(element: string, input: timestamp) -> timestamp`,
	1054: `date_trunc(element: string, input: date) -> timestamptz`,
	1055: `date_trunc(element: string, input: time) -> interval`,
	1056: `date_trunc(element: string, input: timestamptz) -> timestamptz`,
	1057: `date_trunc(element: string, input: interval) -> interval`,
	1058: `row_to_json(row: tuple) -> jsonb`,
	1059: `timezone(timezone: string, timestamptz_string: string) -> timestamp`,
	1060: `timezone(timezone: string, timestamp: timestamp) -> timestamptz`,
	1061: `timezone(timezone: string, timestamptz: timestamptz) -> timestamp`,
	1062: `timezone(timezone: string, time: time) -> timetz`,
	1063: `timezone(timezone: string, timetz: timetz) -> timetz`,
	1064: `parse_timestamp(val: string) -> timestamp`,
	1065: `parse_timestamp(string: string, datestyle: string) -> timestamp`,
	1066: `parse_date(val: string) -> date`,
	1067: `parse_date(string: string, datestyle: string) -> date`,
	1068: `parse_time(val: string) -> time`,
	1069: `parse_time(string: string, timestyle: string) -> time`,
	1070: `parse_interval(val: string) -> interval`,
	1071: `parse_interval(string: string, style: string) -> interval`,
	1072: `parse_timetz(val: string) -> timetz`,
	1073: `parse_timetz(string: string, timestyle: string) -> timetz`,
	1074: `string_to_array(str: string, delimiter: string) -> string[]`,
	1075: `string_to_array(str: string, delimiter: string, null: string) -> string[]`,
	1076: `array_to_string(input: anyelement[], delim: string) -> string`,
	1077: `array_to_string(input: anyelement[], delimiter: string, null: string) -> string`,
	1078: `array_length(input: anyelement[], array_dimension: int) -> int`,
	1079: `cardinality(input: anyelement[]) -> int`,
	1080: `array_lower(input: anyelement[], array_dimension: int) -> int`,
	1081: `array_upper(input: anyelement[], array_dimension: int) -> int`,
	1082: `array_append(array: bool[], elem: bool) -> bool[]`,
	1083: `array_append(array: box2d[], elem: box2d) -> box2d[]`,
	1084: `array_append(array: int[], elem: int) -> int[]`,
	1085: `array_append(array: float[], elem: float) -> float[]`,
	1086: `array_append(array: decimal[], elem: decimal) -> decimal[]`,
	1087: `array_append(array: date[], elem: date) -> date[]`,
	1088: `array_append(array: timestamp[], elem: timestamp) -> timestamp[]`,
	1089: `array_append(array: interval[], elem: interval) -> interval[]`,
	1090: `array_append(array: geography[], elem: geography) -> geography[]`,
	1091: `array_append(array: geometry[], elem: geometry) -> geometry[]`,
	1092: `array_append(array: string[], elem: string) -> string[]`,
	1093: `array_append(array: bytes[], elem: bytes) -> bytes[]`,
	1094: `array_append(array: timestamptz[], elem: timestamptz) -> timestamptz[]`,
	1095: `array_append(array: oid[], elem: oid) -> oid[]`,
	1096: `array_append(array: uuid[], elem: uuid) -> uuid[]`,
	1097: `array_append(array: inet[], elem: inet) -> inet[]`,
	1098: `array_append(array: time[], elem: time) -> time[]`,
	1099: `array_append(array: timetz[], elem: timetz) -> timetz[]`,
	1100: `array_append(array: jsonb[], elem: jsonb) -> jsonb[]`,
	1101: `array_append(array: varbit[], elem: varbit) -> varbit[]`,
	1102: `array_append(array: anyenum[], elem: anyenum) -> anyenum[]`,
	1103: `array_append(array: tuple[], elem: tuple) -> tuple[]`,
	1104: `array_prepend(elem: bool, array: bool[]) -> bool[]`,
	1105: `array_prepend(elem: box2d, array: box2d[]) -> box2d[]`,
	1106: `array_prepend(elem: int, array: int[]) -> int[]`,
	1107: `array_prepend(elem: float, array: float[]) -> float[]`,
	1108: `array_prepend(elem: decimal, array: decimal[]) -> decimal[]`,
	1109: `array_prepend(elem: date, array: date[]) -> date[]`,
	1110: `array_prepend(elem: timestamp, array: timestamp[]) -> timestamp[]`,
	1111: `array_prepend(elem: interval, array: interval[]) -> interval[]`,
	1112: `array_prepend(elem: geography, array: geography[]) -> geography[]`,
	1113: `array_prepend(elem: geometry, array: geometry[]) -> geometry[]`,
	1114: `array_prepend(elem: string, array: string[]) -> string[]`,
	1115: `array_prepend(elem: bytes, array: bytes[]) -> bytes[]`,
	1116: `array_prepend(elem: timestamptz, array: timestamptz[]) -> timestamptz[]`,
	1117: `array_prepend(elem: oid, array: oid[]) -> oid[]`,
	1118: `array_prepend(elem: uuid, array: uuid[]) -> uuid[]`,
	1119: `array_prepend(elem: inet, array: inet[]) -> inet[]`,
	1120: `array_prepend(elem: time, array: time[]) -> time[]`,
	1121: `array_prepend(elem: timetz, array: timetz[]) -> timetz[]`,
	1122: `array_prepend(elem: jsonb, array: jsonb[]) -> jsonb[]`,
	1123: `array_prepend(elem: varbit, array: varbit[]) -> varbit[]`,
	1124: `array_prepend(elem: anyenum, array: anyenum[]) -> anyenum[]`,
	1125: `array_prepend(elem: tuple, array: tuple[]) -> tuple[]`,
	1126: `array_cat(left: bool[], right: bool[]) -> bool[]`,
	1127: `array_cat(left: box2d[], right: box2d[]) -> box2d[]`,
	1128: `array_cat(left: int[], right: int[]) -> int[]`,
	1129: `array_cat(left: float[], right: float[]) -> float[]`,
	1130: `array_cat(left: decimal[], right: decimal[]) -> decimal[]`,
	1131: `array_cat(left: date[], right: date[]) -> date[]`,
	1132: `array_cat(left: timestamp[], right: timestamp[]) -> timestamp[]`,
	1133: `array_cat(left: interval[], right: interval[]) -> interval[]`,
	1134: `array_cat(left: geography[], right: geography[]) -> geography[]`,
	1135: `array_cat(left: geometry[], right: geometry[]) -> geometry[]`,
	1136: `array_cat(left: string[], right: string[]) -> string[]`,
	1137: `array_cat(left: bytes[], right: bytes[]) -> bytes[]`,
	1138: `array_cat(left: timestamptz[], right: timestamptz[]) -> timestamptz[]`,
	1139: `array_cat(left: oid[], right: oid[]) -> oid[]`,
	1140: `array_cat(left: uuid[], right: uuid[]) -> uuid[]`,
	1141: `array_cat(left: inet[], right: inet[]) -> inet[]`,
	1142: `array_cat(left: time[], right: time[]) -> time[]`,
	1143: `array_cat(left: timetz[], right: timetz[]) -> timetz[]`,
	1144: `array_cat(left: jsonb[], right: jsonb[]) -> jsonb[]`,
	1145: `array_cat(left: varbit[], right: varbit[]) -> varbit[]`,
	1146: `array_cat(left: anyenum[], right: anyenum[]) -> anyenum[]`,
	1147: `array_cat(left: tuple[], right: tuple[]) -> tuple[]`,
	1148: `array_remove(array: bool[], elem: bool) -> anyelement`,
	1149: `array_remove(array: box2d[], elem: box2d) -> anyelement`,
	1150: `array_remove(array: int[], elem: int) -> anyelement`,
	1151: `array_remove(array: float[], elem: float) -> anyelement`,
	1152: `array_remove(array: decimal[], elem: decimal) -> anyelement`,
	1153: `array_remove(array: date[], elem: date) -> anyelement`,
	1154: `array_remove(array: timestamp[], elem: timestamp) -> anyelement`,
	1155: `array_remove(array: interval[], elem: interval) -> anyelement`,
	1156: `array_remove(array: geography[], elem: geography) -> anyelement`,
	1157: `array_remove(array: geometry[], elem: geometry) -> anyelement`,
	1158: `array_remove(array: string[], elem: string) -> anyelement`,
	1159: `array_remove(array: bytes[], elem: bytes) -> anyelement`,
	1160: `array_remove(array: timestamptz[], elem: timestamptz) -> anyelement`,
	1161: `array_remove(array: oid[], elem: oid) -> anyelement`,
	1162: `array_remove(array: uuid[], elem: uuid) -> anyelement`,
	1163: `array_remove(array: inet[], elem: inet) -> anyelement`,
	1164: `array_remove(array: time[], elem: time) -> anyelement`,
	1165: `array_remove(array: timetz[], elem: timetz) -> anyelement`,
	1166: `array_remove(array: jsonb[], elem: jsonb) -> anyelement`,
	1167: `array_remove(array: varbit[], elem: varbit) -> anyelement`,
	1168: `array_remove(array: anyenum[], elem: anyenum) -> anyelement`,
	1169: `array_remove(array: tuple[], elem: tuple) -> anyelement`,
	1170: `array_replace(array: bool[], toreplace: bool, replacewith: bool) -> anyelement`,
	1171: `array_replace(array: box2d[], toreplace: box2d, replacewith: box2d) -> anyelement`,
	1172: `array_replace(array: int[], toreplace: int, replacewith: int) -> anyelement`,
	1173: `array_replace(array: float[], toreplace: float, replacewith: float) -> anyelement`,
	1174: `array_replace(array: decimal[], toreplace: decimal, replacewith: decimal) -> anyelement`,
	1175: `array_replace(array: date[], toreplace: date, replacewith: date) -> anyelement`,
	1176: `array_replace(array: timestamp[], toreplace: timestamp, replacewith: timestamp) -> anyelement`,
	1177: `array_replace(array: interval[], toreplace: interval, replacewith: interval) -> anyelement`,
	1178: `array_replace(array: geography[], toreplace: geography, replacewith: geography) -> anyelement`,
	1179: `array_replace(array: geometry[], toreplace: geometry, replacewith: geometry) -> anyelement`,
	1180: `array_replace(array: string[], toreplace: string, replacewith: string) -> anyelement`,
	1181: `array_replace(array: bytes[], toreplace: bytes, replacewith: bytes) -> anyelement`,
	1182: `array_replace(array: timestamptz[], toreplace: timestamptz, replacewith: timestamptz) -> anyelement`,
	1183: `array_replace(array: oid[], toreplace: oid, replacewith: oid) -> anyelement`,
	1184: `array_replace(array: uuid[], toreplace: uuid, replacewith: uuid) -> anyelement`,
	1185: `array_replace(array: inet[], toreplace: inet, replacewith: inet) -> anyelement`,
	1186: `array_replace(array: time[], toreplace: time, replacewith: time) -> anyelement`,
	1187: `array_replace(array: timetz[], toreplace: timetz, replacewith: timetz) -> anyelement`,
	1188: `array_replace(array: jsonb[], toreplace: jsonb, replacewith: jsonb) -> anyelement`,
	1189: `array_replace(array: varbit[], toreplace: varbit, replacewith: varbit) -> anyelement`,
	1190: `array_replace(array: anyenum[], toreplace: anyenum, replacewith: anyenum) -> anyelement`,
	1191: `array_replace(array: tuple[], toreplace: tuple, replacewith: tuple) -> anyelement`,
	1192: `array_position(array: bool[], elem: bool) -> int`,
	1193: `array_position(array: box2d[], elem: box2d) -> int`,
	1194: `array_position(array: int[], elem: int) -> int`,
	1195: `array_position(array: float[], elem: float) -> int`,
	1196: `array_position(array: decimal[], elem: decimal) -> int`,
	1197: `array_position(array: date[], elem: date) -> int`,
	1198: `array_position(array: timestamp[], elem: timestamp) -> int`,
	1199: `array_position(array: interval[], elem: interval) -> int`,
	1200: `array_position(array: geography[], elem: geography) -> int`,
	1201: `array_position(array: geometry[], elem: geometry) -> int`,
	1202: `array_position(array: string[], elem: string) -> int`,
	1203: `array_position(array: bytes[], elem: bytes) -> int`,
	1204: `array_position(array: timestamptz[], elem: timestamptz) -> int`,
	1205: `array_position(array: oid[], elem: oid) -> int`,
	1206: `array_position(array: uuid[], elem: uuid) -> int`,
	1207: `array_position(array: inet[], elem: inet) -> int`,
	1208: `array_position(array: time[], elem: time) -> int`,
	1209: `array_position(array: timetz[], elem: timetz) -> int`,
	1210: `array_position(array: jsonb[], elem: jsonb) -> int`,
	1211: `array_position(array: varbit[], elem: varbit) -> int`,
	1212: `array_position(array: anyenum[], elem: anyenum) -> int`,
	1213: `array_position(array: tuple[], elem: tuple) -> int`,
	1214: `array_positions(array: bool[], elem: bool) -> int[]`,
	1215: `array_positions(array: box2d[], elem: box2d) -> int[]`,
	1216: `array_positions(array: int[], elem: int) -> int[]`,
	1217: `array_positions(array: float[], elem: float) -> int[]`,
	1218: `array_positions(array: decimal[], elem: decimal) -> int[]`,
	1219: `array_positions(array: date[], elem: date) -> int[]`,
	1220: `array_positions(array: timestamp[], elem: timestamp) -> int[]`,
	1221: `array_positions(array: interval[], elem: interval) -> int[]`,
	1222: `array_positions(array: geography[], elem: geography) -> int[]`,
	1223: `array_positions(array: geometry[], elem: geometry) -> int[]`,
	1224: `array_positions(array: string[], elem: string) -> int[]`,
	1225: `array_positions(array: bytes[], elem: bytes) -> int[]`,
	1226: `array_positions(array: timestamptz[], elem: timestamptz) -> int[]`,
	1227: `array_positions(array: oid[], elem: oid) -> int[]`,
	1228: `array_positions(array: uuid[], elem: uuid) -> int[]`,
	1229: `array_positions(array: inet[], elem: inet) -> int[]`,
	1230: `array_positions(array: time[], elem: time) -> int[]`,
	1231: `array_positions(array: timetz[], elem: timetz) -> int[]`,
	1232: `array_positions(array: jsonb[], elem: jsonb) -> int[]`,
	1233: `array_positions(array: varbit[], elem: varbit) -> int[]`,
	1234: `array_positions(array: anyenum[], elem: anyenum) -> int[]`,
	1235: `array_positions(array: tuple[], elem: tuple) -> int[]`,
	1236: `soundex(source: string) -> string`,
	1237: `difference(source: string, target: string) -> int`,
	1238: `levenshtein(source: string, target: string) -> int`,
	1239: `levenshtein(source: string, target: string, ins_cost: int, del_cost: int, sub_cost: int) -> int`,
	1240: `json_remove_path(val: jsonb, path: string[]) -> jsonb`,
	1241: `json_extract_path(jsonb, string...) -> jsonb`,
	1242: `jsonb_extract_path(jsonb, string...) -> jsonb`,
	1243: `json_extract_path_text(jsonb, string...) -> string`,
	1244: `jsonb_extract_path_text(jsonb, string...) -> string`,
	1245: `json_set(val: jsonb, path: string[], to: jsonb) -> jsonb`,
	1246: `json_set(val: jsonb, path: string[], to: jsonb, create_missing: bool) -> jsonb`,
	1247: `jsonb_set(val: jsonb, path: string[], to: jsonb) -> jsonb`,
	1248: `jsonb_set(val: jsonb, path: string[], to: jsonb, create_missing: bool) -> jsonb`,
	1249: `jsonb_insert(target: jsonb, path: string[], new_val: jsonb) -> jsonb`,
	1250: `jsonb_insert(target: jsonb, path: string[], new_val: jsonb, insert_after: bool) -> jsonb`,
	1251: `jsonb_pretty(val: jsonb) -> string`,
	1252: `json_typeof(val: jsonb) -> string`,
	1253: `jsonb_typeof(val: jsonb) -> string`,
	1254: `to_json(val: anyelement) -> jsonb`,
	1255: `to_jsonb(val: anyelement) -> jsonb`,
	1256: `json_build_array(anyelement...) -> jsonb`,
	1257: `jsonb_build_array(anyelement...) -> jsonb`,
	1258: `json_build_object(anyelement...) -> jsonb`,
	1259: `jsonb_build_object(anyelement...) -> jsonb`,
	1260: `json_object(texts: string[]) -> jsonb`,
	1261: `json_object(keys: string[], values: string[]) -> jsonb`,
	1262: `jsonb_object(texts: string[]) -> jsonb`,
	1263: `jsonb_object(keys: string[], values: string[]) -> jsonb`,
	1264: `json_strip_nulls(from_json: jsonb) -> jsonb`,
	1265: `jsonb_strip_nulls(from_json: jsonb) -> jsonb`,
	1266: `json_array_length(json: jsonb) -> int`,
	1267: `jsonb_array_length(json: jsonb) -> int`,
	1268: `jsonb_exists_any(json: jsonb, array: string[]) -> bool`,
	1269: `json_valid(string: string) -> bool`,
	1270: `crdb_internal.pb_to_json(pbname: string, data: bytes) -> jsonb`,
	1271: `crdb_internal.pb_to_json(pbname: string, data: bytes, emit_defaults: bool) -> jsonb`,
	1272: `crdb_internal.pb_to_json(pbname: string, data: bytes, emit_defaults: bool, emit_redacted: bool) -> jsonb`,
	1273: `crdb_internal.json_to_pb(pbname: string, json: jsonb) -> bytes`,
	1274: `crdb_internal.read_file(uri: string) -> bytes`,
	1275: `crdb_internal.write_file(data: bytes, uri: string) -> int`,
	1276: `crdb_internal.datums_to_bytes(anyelement...) -> bytes`,
	1277: `crdb_internal.merge_statement_stats(input: jsonb[]) -> jsonb`,
	1278: `crdb_internal.merge_transaction_stats(input: jsonb[]) -> jsonb`,
	1279: `crdb_internal.merge_stats_metadata(input: jsonb[]) -> jsonb`,
	1280: `enum_first(val: anyenum) -> anyelement`,
	1281: `enum_last(val: anyenum) -> anyelement`,
	1282: `enum_range(val: anyenum) -> anyelement`,
	1283: `enum_range(lower: anyenum, upper: anyenum) -> anyelement`,
	1284: `version() -> string`,
	1285: `current_database() -> string`,
	1286: `current_schema() -> string`,
	1287: `current_schemas(include_pg_catalog: bool) -> string[]`,
	1288: `current_user() -> string`,
	1289: `session_user() -> string`,
	1290: `crdb_internal.trace_id() -> int`,
	1291: `crdb_internal.set_trace_verbose(trace_id: int, verbosity: bool) -> bool`,
	1292: `crdb_internal.locality_value(key: string) -> string`,
	1293: `crdb_internal.cluster_setting_encoded_default(setting: string) -> string`,
	1294: `crdb_internal.decode_cluster_setting(setting: string, value: string) -> string`,
	1295: `crdb_internal.node_executable_version() -> string`,
	1296: `crdb_internal.active_version() -> jsonb`,
	1297: `crdb_internal.is_at_least_version(version: string) -> bool`,
	1298: `crdb_internal.approximate_timestamp(timestamp: decimal) -> timestamp`,
	1299: `crdb_internal.cluster_id() -> uuid`,
	1300: `crdb_internal.node_id() -> int`,
	1301: `crdb_internal.cluster_name() -> string`,
	1302: `crdb_internal.create_tenant(id: int) -> int`,
	1303: `crdb_internal.create_join_token() -> string`,
	1304: `crdb_internal.destroy_tenant(id: int) -> int`,
	1305: `crdb_internal.destroy_tenant(id: int, synchronous: bool) -> int`,
	1306: `crdb_internal.unsafe_clear_gossip_info(key: string) -> bool`,
	1307: `crdb_internal.encode_key(table_id: int, index_id: int, row_tuple: anyelement) -> bytes`,
	1308: `crdb_internal.force_error(errorCode: string, msg: string) -> int`,
	1309: `crdb_internal.notice(msg: string) -> int`,
	1310: `crdb_internal.notice(severity: string, msg: string) -> int`,
	1311: `crdb_internal.force_assertion_error(msg: string) -> int`,
	1312: `crdb_internal.void_func() -> void`,
	1313: `crdb_internal.force_panic(msg: string) -> int`,
	1314: `crdb_internal.force_log_fatal(msg: string) -> int`,
	1315: `crdb_internal.force_retry(val: interval) -> int`,
	1316: `crdb_internal.lease_holder(key: bytes) -> int`,
	1318: `crdb_internal.trim_tenant_prefix(key: bytes) -> bytes`,
	1319: `crdb_internal.trim_tenant_prefix(keys: bytes[]) -> bytes[]`,
	1320: `crdb_internal.tenant_span(tenant_id: int) -> bytes[]`,
	1321: `crdb_internal.table_span(table_id: int) -> bytes[]`,
	1322: `crdb_internal.index_span(table_id: int, index_id: int) -> bytes[]`,
	1323: `crdb_internal.pretty_key(raw_key: bytes, skip_fields: int) -> string`,
	1324: `crdb_internal.pretty_span(raw_key_start: bytes, raw_key_end: bytes, skip_fields: int) -> string`,
	1325: `crdb_internal.range_stats(key: bytes) -> jsonb`,
	1326: `crdb_internal.get_namespace_id(parent_id: int, name: string) -> int`,
	1327: `crdb_internal.get_namespace_id(parent_id: int, parent_schema_id: int, name: string) -> int`,
	1328: `crdb_internal.get_database_id(name: string) -> int`,
	1329: `crdb_internal.get_zone_config(namespace_id: int) -> bytes`,
	1330: `crdb_internal.set_vmodule(vmodule_string: string) -> int`,
	1331: `crdb_internal.get_vmodule() -> string`,
	1332: `crdb_internal.num_geo_inverted_index_entries(table_id: int, index_id: int, val: geography) -> int`,
	1333: `crdb_internal.num_geo_inverted_index_entries(table_id: int, index_id: int, val: geometry) -> int`,
	1334: `crdb_internal.num_inverted_index_entries(val: jsonb) -> int`,
	1335: `crdb_internal.num_inverted_index_entries(val: anyelement[]) -> int`,
	1336: `crdb_internal.num_inverted_index_entries(val: jsonb, version: int) -> int`,
	1337: `crdb_internal.num_inverted_index_entries(val: string, version: int) -> int`,
	1338: `crdb_internal.num_inverted_index_entries(val: anyelement[], version: int) -> int`,
	1341: `crdb_internal.assignment_cast(val: anyelement, type: anyelement) -> anyelement`,
	1342: `crdb_internal.round_decimal_values(val: decimal, scale: int) -> decimal`,
	1343: `crdb_internal.round_decimal_values(val: decimal[], scale: int) -> decimal[]`,
	1345: `crdb_internal.unsafe_upsert_descriptor(id: int, desc: bytes) -> bool`,
	1346: `crdb_internal.unsafe_upsert_descriptor(id: int, desc: bytes, force: bool) -> bool`,
	1347: `crdb_internal.unsafe_delete_descriptor(id: int) -> bool`,
	1348: `crdb_internal.unsafe_delete_descriptor(id: int, force: bool) -> bool`,
	1349: `crdb_internal.unsafe_upsert_namespace_entry(parent_id: int, parent_schema_id: int, name: string, desc_id: int, force: bool) -> bool`,
	1350: `crdb_internal.unsafe_upsert_namespace_entry(parent_id: int, parent_schema_id: int, name: string, desc_id: int) -> bool`,
	1351: `crdb_internal.unsafe_delete_namespace_entry(parent_id: int, parent_schema_id: int, name: string, desc_id: int) -> bool`,
	1352: `crdb_internal.unsafe_delete_namespace_entry(parent_id: int, parent_schema_id: int, name: string, desc_id: int, force: bool) -> bool`,
	1353: `crdb_internal.sql_liveness_is_alive(session_id: bytes) -> bool`,
	1355: `crdb_internal.update_tenant_resource_limits(tenant_id: int, available_tokens: float, refill_rate: float, max_burst_tokens: float, as_of: timestamp, as_of_consumed_tokens: float) -> int`,
	1356: `crdb_internal.compact_engine_span(node_id: int, store_id: int, start_key: bytes, end_key: bytes) -> bool`,
	1357: `crdb_internal.increment_feature_counter(feature: string) -> bool`,
	1358: `num_nulls(anyelement...) -> int`,
	1359: `num_nonnulls(anyelement...) -> int`,
	1360: `gateway_region() -> string`,
	1361: `default_to_database_primary_region(val: string) -> string`,
	1362: `rehome_row() -> string`,
	1363: `crdb_internal.validate_multi_region_zone_configs() -> bool`,
	1364: `crdb_internal.reset_multi_region_zone_configs_for_table(id: int) -> bool`,
	1365: `crdb_internal.reset_multi_region_zone_configs_for_database(id: int) -> bool`,
	1366: `crdb_internal.filter_multiregion_fields_from_zone_config_sql(val: string) -> string`,
	1367: `crdb_internal.reset_index_usage_stats() -> bool`,
	1368: `crdb_internal.reset_sql_stats() -> bool`,
	1369: `crdb_internal.force_delete_table_data(id: int) -> bool`,
	1370: `crdb_internal.serialize_session() -> bytes`,
	1371: `crdb_internal.deserialize_session(session: bytes) -> bool`,
	1372: `crdb_internal.create_session_revival_token() -> bytes`,
	1373: `crdb_internal.validate_session_revival_token(token: bytes) -> bool`,
	1374: `crdb_internal.validate_ttl_scheduled_jobs() -> void`,
	1375: `crdb_internal.repair_ttl_table_scheduled_job(oid: oid) -> void`,
	1376: `crdb_internal.check_password_hash_format(password: bytes) -> string`,
	1377: `crdb_internal.schedule_sql_stats_compaction() -> bool`,
	1378: `crdb_internal.create_sql_schema_telemetry_job() -> int`,
	1379: `crdb_internal.revalidate_unique_constraints_in_all_tables() -> void`,
	1380: `crdb_internal.revalidate_unique_constraints_in_table(table_name: string) -> void`,
	1381: `crdb_internal.revalidate_unique_constraint(table_name: string, constraint_name: string) -> void`,
	1382: `crdb_internal.is_constraint_active(table_name: string, constraint_name: string) -> bool`,
	1383: `crdb_internal.kv_set_queue_active(queue_name: string, active: bool) -> bool`,
	1384: `crdb_internal.kv_set_queue_active(queue_name: string, active: bool, store_id: int) -> bool`,
	1385: `crdb_internal.kv_enqueue_replica(range_id: int, queue_name: string, skip_should_queue: bool) -> bool`,
	1386: `crdb_internal.kv_enqueue_replica(range_id: int, queue_name: string, skip_should_queue: bool, should_return_trace: bool) -> string`,
	1387: `crdb_internal.kv_enqueue_replica(range_id: int, queue_name: string, skip_should_queue: bool, store_id: int) -> bool`,
	1388: `crdb_internal.request_statement_bundle(stmtFingerprint: string, samplingProbability: float, minExecutionLatency: interval, expiresAfter: interval) -> bool`,
	1389: `crdb_internal.set_compaction_concurrency(node_id: int, store_id: int, compaction_concurrency: int) -> bool`,
	1390: `overlaps(s1: timestamp, e1: timestamp, s1: timestamp, e2: timestamp) -> bool`,
	1391: `overlaps(s1: timestamp, e1: interval, s1: timestamp, e2: interval) -> bool`,
	1392: `overlaps(s1: timetz, e1: timetz, s1: timetz, e2: timetz) -> bool`,
	1393: `overlaps(s1: timetz, e1: interval, s1: timetz, e2: interval) -> bool`,
	1394: `overlaps(s1: time, e1: time, s1: time, e2: time) -> bool`,
	1395: `overlaps(s1: time, e1: interval, s1: time, e2: interval) -> bool`,
	1396: `overlaps(s1: timestamptz, e1: timestamptz, s1: timestamptz, e2: timestamptz) -> bool`,
	1397: `overlaps(s1: timestamptz, e1: interval, s1: timestamptz, e2: interval) -> bool`,
	1398: `overlaps(s1: date, e1: date, s1: date, e2: date) -> bool`,
	1399: `overlaps(s1: date, e1: interval, s1: date, e2: interval) -> bool`,
	1400: `pg_backend_pid() -> int`,
	1401: `pg_encoding_to_char(encoding_id: int) -> string`,
	1402: `getdatabaseencoding() -> string`,
	1403: `pg_get_expr(pg_node_tree: string, relation_oid: oid) -> string`,
	1404: `pg_get_expr(pg_node_tree: string, relation_oid: oid, pretty_bool: bool) -> string`,
	1405: `pg_get_constraintdef(constraint_oid: oid, pretty_bool: bool) -> string`,
	1406: `pg_get_constraintdef(constraint_oid: oid) -> string`,
	1407: `pg_get_partkeydef(oid: oid) -> string`,
	1408: `pg_get_function_result(func_oid: oid) -> string`,
	1409: `pg_get_function_identity_arguments(func_oid: oid) -> string`,
	1410: `pg_get_indexdef(index_oid: oid) -> string`,
	1411: `pg_get_indexdef(index_oid: oid, column_no: int, pretty_bool: bool) -> string`,
	1412: `pg_get_viewdef(view_oid: oid) -> string`,
	1413: `pg_get_viewdef(view_oid: oid, pretty_bool: bool) -> string`,
	1414: `pg_get_serial_sequence(table_name: string, column_name: string) -> string`,
	1415: `pg_my_temp_schema() -> oid`,
	1416: `pg_is_other_temp_schema(oid: oid) -> bool`,
	1417: `pg_typeof(val: anyelement) -> string`,
	1418: `pg_collation_for(str: anyelement) -> string`,
	1419: `pg_get_userbyid(role_oid: oid) -> string`,
	1420: `pg_sequence_parameters(sequence_oid: oid) -> tuple{int AS start_value, int AS minimum_value, int AS maxmimum_value, int AS increment, bool AS cycle_option, int AS cache_size, oid AS data_type}`,
	1421: `format_type(type_oid: oid, typemod: int) -> string`,
	1422: `col_description(table_oid: oid, column_number: int) -> string`,
	1423: `obj_description(object_oid: oid) -> string`,
	1424: `obj_description(object_oid: oid, catalog_name: string) -> string`,
	1425: `oid(int: int) -> oid`,
	1426: `shobj_description(object_oid: oid, catalog_name: string) -> string`,
	1427: `pg_try_advisory_lock(int: int) -> bool`,
	1428: `pg_advisory_unlock(key: int) -> bool`,
	1429: `pg_client_encoding() -> string`,
	1430: `pg_function_is_visible(oid: oid) -> bool`,
	1431: `pg_table_is_visible(oid: oid) -> bool`,
	1432: `pg_type_is_visible(oid: oid) -> bool`,
	1433: `pg_relation_is_updatable(reloid: oid, include_triggers: bool) -> int4`,
	1434: `pg_column_is_updatable(reloid: oid, attnum: int2, include_triggers: bool) -> bool`,
	1435: `pg_sleep(seconds: float) -> bool`,
	1436: `pg_is_in_recovery() -> bool`,
	1437: `pg_is_xlog_replay_paused() -> bool`,
	1438: `has_any_column_privilege(table: string, privilege: string) -> bool`,
	1439: `has_any_column_privilege(table: oid, privilege: string) -> bool`,
	1440: `has_any_column_privilege(user: string, table: string, privilege: string) -> bool`,
	1441: `has_any_column_privilege(user: string, table: oid, privilege: string) -> bool`,
	1442: `has_any_column_privilege(user: oid, table: string, privilege: string) -> bool`,
	1443: `has_any_column_privilege(user: oid, table: oid, privilege: string) -> bool`,
	1444: `has_column_privilege(table: string, column: string, privilege: string) -> bool`,
	1445: `has_column_privilege(table: string, column: int, privilege: string) -> bool`,
	1446: `has_column_privilege(table: oid, column: string, privilege: string) -> bool`,
	1447: `has_column_privilege(table: oid, column: int, privilege: string) -> bool`,
	1448: `has_column_privilege(user: string, table: string, column: string, privilege: string) -> bool`,
	1449: `has_column_privilege(user: string, table: string, column: int, privilege: string) -> bool`,
	1450: `has_column_privilege(user: string, table: oid, column: string, privilege: string) -> bool`,
	1451: `has_column_privilege(user: string, table: oid, column: int, privilege: string) -> bool`,
	1452: `has_column_privilege(user: oid, table: string, column: string, privilege: string) -> bool`,
	1453: `has_column_privilege(user: oid, table: string, column: int, privilege: string) -> bool`,
	1454: `has_column_privilege(user: oid, table: oid, column: string, privilege: string) -> bool`,
	1455: `has_column_privilege(user: oid, table: oid, column: int, privilege: string) -> bool`,
	1456: `has_database_privilege(database: string, privilege: string) -> bool`,
	1457: `has_database_privilege(database: oid, privilege: string) -> bool`,
	1458: `has_database_privilege(user: string, database: string, privilege: string) -> bool`,
	1459: `has_database_privilege(user: string, database: oid, privilege: string) -> bool`,
	1460: `has_database_privilege(user: oid, database: string, privilege: string) -> bool`,
	1461: `has_database_privilege(user: oid, database: oid, privilege: string) -> bool`,
	1462: `has_foreign_data_wrapper_privilege(fdw: string, privilege: string) -> bool`,
	1463: `has_foreign_data_wrapper_privilege(fdw: oid, privilege: string) -> bool`,
	1464: `has_foreign_data_wrapper_privilege(user: string, fdw: string, privilege: string) -> bool`,
	1465: `has_foreign_data_wrapper_privilege(user: string, fdw: oid, privilege: string) -> bool`,
	1466: `has_foreign_data_wrapper_privilege(user: oid, fdw: string, privilege: string) -> bool`,
	1467: `has_foreign_data_wrapper_privilege(user: oid, fdw: oid, privilege: string) -> bool`,
	1468: `has_function_privilege(function: string, privilege: string) -> bool`,
	1469: `has_function_privilege(function: oid, privilege: string) -> bool`,
	1470: `has_function_privilege(user: string, function: string, privilege: string) -> bool`,
	1471: `has_function_privilege(user: string, function: oid, privilege: string) -> bool`,
	1472: `has_function_privilege(user: oid, function: string, privilege: string) -> bool`,
	1473: `has_function_privilege(user: oid, function: oid, privilege: string) -> bool`,
	1474: `has_language_privilege(language: string, privilege: string) -> bool`,
	1475: `has_language_privilege(language: oid, privilege: string) -> bool`,
	1476: `has_language_privilege(user: string, language: string, privilege: string) -> bool`,
	1477: `has_language_privilege(user: string, language: oid, privilege: string) -> bool`,
	1478: `has_language_privilege(user: oid, language: string, privilege: string) -> bool`,
	1479: `has_language_privilege(user: oid, language: oid, privilege: string) -> bool`,
	1480: `has_schema_privilege(schema: string, privilege: string) -> bool`,
	1481: `has_schema_privilege(schema: oid, privilege: string) -> bool`,
	1482: `has_schema_privilege(user: string, schema: string, privilege: string) -> bool`,
	1483: `has_schema_privilege(user: string, schema: oid, privilege: string) -> bool`,
	1484: `has_schema_privilege(user: oid, schema: string, privilege: string) -> bool`,
	1485: `has_schema_privilege(user: oid, schema: oid, privilege: string) -> bool`,
	1486: `has_sequence_privilege(sequence: string, privilege: string) -> bool`,
	1487: `has_sequence_privilege(sequence: oid, privilege: string) -> bool`,
	1488: `has_sequence_privilege(user: string, sequence: string, privilege: string) -> bool`,
	1489: `has_sequence_privilege(user: string, sequence: oid, privilege: string) -> bool`,
	1490: `has_sequence_privilege(user: oid, sequence: string, privilege: string) -> bool`,
	1491: `has_sequence_privilege(user: oid, sequence: oid, privilege: string) -> bool`,
	1492: `has_server_privilege(server: string, privilege: string) -> bool`,
	1493: `has_server_privilege(server: oid, privilege: string) -> bool`,
	1494: `has_server_privilege(user: string, server: string, privilege: string) -> bool`,
	1495: `has_server_privilege(user: string, server: oid, privilege: string) -> bool`,
	1496: `has_server_privilege(user: oid, server: string, privilege: string) -> bool`,
	1497: `has_server_privilege(user: oid, server: oid, privilege: string) -> bool`,
	1498: `has_table_privilege(table: string, privilege: string) -> bool`,
	1499: `has_table_privilege(table: oid, privilege: string) -> bool`,
	1500: `has_table_privilege(user: string, table: string, privilege: string) -> bool`,
	1501: `has_table_privilege(user: string, table: oid, privilege: string) -> bool`,
	1502: `has_table_privilege(user: oid, table: string, privilege: string) -> bool`,
	1503: `has_table_privilege(user: oid, table: oid, privilege: string) -> bool`,
	1504: `has_tablespace_privilege(tablespace: string, privilege: string) -> bool`,
	1505: `has_tablespace_privilege(tablespace: oid, privilege: string) -> bool`,
	1506: `has_tablespace_privilege(user: string, tablespace: string, privilege: string) -> bool`,
	1507: `has_tablespace_privilege(user: string, tablespace: oid, privilege: string) -> bool`,
	1508: `has_tablespace_privilege(user: oid, tablespace: string, privilege: string) -> bool`,
	1509: `has_tablespace_privilege(user: oid, tablespace: oid, privilege: string) -> bool`,
	1510: `has_type_privilege(type: string, privilege: string) -> bool`,
	1511: `has_type_privilege(type: oid, privilege: string) -> bool`,
	1512: `has_type_privilege(user: string, type: string, privilege: string) -> bool`,
	1513: `has_type_privilege(user: string, type: oid, privilege: string) -> bool`,
	1514: `has_type_privilege(user: oid, type: string, privilege: string) -> bool`,
	1515: `has_type_privilege(user: oid, type: oid, privilege: string) -> bool`,
	1516: `pg_has_role(role: string, privilege: string) -> bool`,
	1517: `pg_has_role(role: oid, privilege: string) -> bool`,
	1518: `pg_has_role(user: string, role: string, privilege: string) -> bool`,
	1519: `pg_has_role(user: string, role: oid, privilege: string) -> bool`,
	1520: `pg_has_role(user: oid, role: string, privilege: string) -> bool`,
	1521: `pg_has_role(user: oid, role: oid, privilege: string) -> bool`,
	1522: `current_setting(setting_name: string) -> string`,
	1523: `current_setting(setting_name: string, missing_ok: bool) -> string`,
	1524: `set_config(setting_name: string, new_value: string, is_local: bool) -> string`,
	1525: `inet_client_addr() -> inet`,
	1526: `inet_client_port() -> int`,
	1527: `inet_server_addr() -> inet`,
	1528: `inet_server_port() -> int`,
	1529: `pg_column_size(anyelement...) -> int`,
	1530: `information_schema._pg_truetypid(pg_attribute: tuple, pg_type: tuple) -> oid`,
	1531: `information_schema._pg_truetypmod(pg_attribute: tuple, pg_type: tuple) -> int4`,
	1532: `information_schema._pg_char_max_length(typid: oid, typmod: int4) -> int`,
	1533: `information_schema._pg_index_position(oid: oid, col: int2) -> int`,
	1534: `information_schema._pg_numeric_precision(typid: oid, typmod: int4) -> int`,
	1535: `information_schema._pg_numeric_precision_radix(typid: oid, typmod: int4) -> int`,
	1536: `information_schema._pg_numeric_scale(typid: oid, typmod: int4) -> int`,
	1537: `crypt(password: string, salt: string) -> string`,
	1538: `digest(data: string, type: string) -> bytes`,
	1539: `digest(data: bytes, type: string) -> bytes`,
	1540: `gen_random_uuid() -> uuid`,
	1541: `gen_salt(type: string) -> string`,
	1542: `gen_salt(type: string, iter_count: int) -> string`,
	1543: `hmac(data: string, key: string, type: string) -> bytes`,
	1544: `hmac(data: bytes, key: bytes, type: string) -> bytes`,
	1545: `crdb_internal.complete_stream_ingestion_job(job_id: int, cutover_ts: timestamptz) -> int`,
	1546: `crdb_internal.stream_ingestion_stats_json(job_id: int) -> jsonb`,
	1547: `crdb_internal.stream_ingestion_stats_pb(job_id: int) -> bytes`,
	1549: `crdb_internal.replication_stream_progress(stream_id: int, frontier_ts: string) -> bytes`,
	1550: `crdb_internal.stream_partition(stream_id: int, partition_spec: bytes) -> bytes`,
	1551: `crdb_internal.replication_stream_spec(stream_id: int) -> bytes`,
	1552: `crdb_internal.complete_replication_stream(stream_id: int, successful_ingestion: bool) -> int`,
	1553: `similarity(left: string, right: string) -> float`,
	1554: `show_trgm(input: string) -> string[]`,
	1555: `row_number() -> int`,
	1556: `rank() -> int`,
	1557: `dense_rank() -> int`,
	1558: `percent_rank() -> float`,
	1559: `cume_dist() -> float`,
	1560: `ntile(n: int) -> int`,
	1561: `lag(val: bool) -> bool`,
	1562: `lag(val: box2d) -> box2d`,
	1563: `lag(val: int) -> int`,
	1564: `lag(val: float) -> float`,
	1565: `lag(val: decimal) -> decimal`,
	1566: `lag(val: date) -> date`,
	1567: `lag(val: timestamp) -> timestamp`,
	1568: `lag(val: interval) -> interval`,
	1569: `lag(val: geography) -> geography`,
	1570: `lag(val: geometry) -> geometry`,
	1571: `lag(val: string) -> string`,
	1572: `lag(val: bytes) -> bytes`,
	1573: `lag(val: timestamptz) -> timestamptz`,
	1574: `lag(val: oid) -> oid`,
	1575: `lag(val: uuid) -> uuid`,
	1576: `lag(val: inet) -> inet`,
	1577: `lag(val: time) -> time`,
	1578: `lag(val: timetz) -> timetz`,
	1579: `lag(val: jsonb) -> jsonb`,
	1580: `lag(val: varbit) -> varbit`,
	1581: `lag(val: bool, n: int) -> bool`,
	1582: `lag(val: box2d, n: int) -> box2d`,
	1583: `lag(val: int, n: int) -> int`,
	1584: `lag(val: float, n: int) -> float`,
	1585: `lag(val: decimal, n: int) -> decimal`,
	1586: `lag(val: date, n: int) -> date`,
	1587: `lag(val: timestamp, n: int) -> timestamp`,
	1588: `lag(val: interval, n: int) -> interval`,
	1589: `lag(val: geography, n: int) -> geography`,
	1590: `lag(val: geometry, n: int) -> geometry`,
	1591: `lag(val: string, n: int) -> string`,
	1592: `lag(val: bytes, n: int) -> bytes`,
	1593: `lag(val: timestamptz, n: int) -> timestamptz`,
	1594: `lag(val: oid, n: int) -> oid`,
	1595: `lag(val: uuid, n: int) -> uuid`,
	1596: `lag(val: inet, n: int) -> inet`,
	1597: `lag(val: time, n: int) -> time`,
	1598: `lag(val: timetz, n: int) -> timetz`,
	1599: `lag(val: jsonb, n: int) -> jsonb`,
	1600: `lag(val: varbit, n: int) -> varbit`,
	1601: `lag(val: bool, n: int, default: bool) -> bool`,
	1602: `lag(val: box2d, n: int, default: box2d) -> box2d`,
	1603: `lag(val: int, n: int, default: int) -> int`,
	1604: `lag(val: float, n: int, default: float) -> float`,
	1605: `lag(val: decimal, n: int, default: decimal) -> decimal`,
	1606: `lag(val: date, n: int, default: date) -> date`,
	1607: `lag(val: timestamp, n: int, default: timestamp) -> timestamp`,
	1608: `lag(val: interval, n: int, default: interval) -> interval`,
	1609: `lag(val: geography, n: int, default: geography) -> geography`,
	1610: `lag(val: geometry, n: int, default: geometry) -> geometry`,
	1611: `lag(val: string, n: int, default: string) -> string`,
	1612: `lag(val: bytes, n: int, default: bytes) -> bytes`,
	1613: `lag(val: timestamptz, n: int, default: timestamptz) -> timestamptz`,
	1614: `lag(val: oid, n: int, default: oid) -> oid`,
	1615: `lag(val: uuid, n: int, default: uuid) -> uuid`,
	1616: `lag(val: inet, n: int, default: inet) -> inet`,
	1617: `lag(val: time, n: int, default: time) -> time`,
	1618: `lag(val: timetz, n: int, default: timetz) -> timetz`,
	1619: `lag(val: jsonb, n: int, default: jsonb) -> jsonb`,
	1620: `lag(val: varbit, n: int, default: varbit) -> varbit`,
	1621: `lead(val: bool) -> bool`,
	1622: `lead(val: box2d) -> box2d`,
	1623: `lead(val: int) -> int`,
	1624: `lead(val: float) -> float`,
	1625: `lead(val: decimal) -> decimal`,
	1626: `lead(val: date) -> date`,
	1627: `lead(val: timestamp) -> timestamp`,
	1628: `lead(val: interval) -> interval`,
	1629: `lead(val: geography) -> geography`,
	1630: `lead(val: geometry) -> geometry`,
	1631: `lead(val: string) -> string`,
	1632: `lead(val: bytes) -> bytes`,
	1633: `lead(val: timestamptz) -> timestamptz`,
	1634: `lead(val: oid) -> oid`,
	1635: `lead(val: uuid) -> uuid`,
	1636: `lead(val: inet) -> inet`,
	1637: `lead(val: time) -> time`,
	1638: `lead(val: timetz) -> timetz`,
	1639: `lead(val: jsonb) -> jsonb`,
	1640: `lead(val: varbit) -> varbit`,
	1641: `lead(val: bool, n: int) -> bool`,
	1642: `lead(val: box2d, n: int) -> box2d`,
	1643: `lead(val: int, n: int) -> int`,
	1644: `lead(val: float, n: int) -> float`,
	1645: `lead(val: decimal, n: int) -> decimal`,
	1646: `lead(val: date, n: int) -> date`,
	1647: `lead(val: timestamp, n: int) -> timestamp`,
	1648: `lead(val: interval, n: int) -> interval`,
	1649: `lead(val: geography, n: int) -> geography`,
	1650: `lead(val: geometry, n: int) -> geometry`,
	1651: `lead(val: string, n: int) -> string`,
	1652: `lead(val: bytes, n: int) -> bytes`,
	1653: `lead(val: timestamptz, n: int) -> timestamptz`,
	1654: `lead(val: oid, n: int) -> oid`,
	1655: `lead(val: uuid, n: int) -> uuid`,
	1656: `lead(val: inet, n: int) -> inet`,
	1657: `lead(val: time, n: int) -> time`,
	1658: `lead(val: timetz, n: int) -> timetz`,
	1659: `lead(val: jsonb, n: int) -> jsonb`,
	1660: `lead(val: varbit, n: int) -> varbit`,
	1661: `lead(val: bool, n: int, default: bool) -> bool`,
	1662: `lead(val: box2d, n: int, default: box2d) -> box2d`,
	1663: `lead(val: int, n: int, default: int) -> int`,
	1664: `lead(val: float, n: int, default: float) -> float`,
	1665: `lead(val: decimal, n: int, default: decimal) -> decimal`,
	1666: `lead(val: date, n: int, default: date) -> date`,
	1667: `lead(val: timestamp, n: int, default: timestamp) -> timestamp`,
	1668: `lead(val: interval, n: int, default: interval) -> interval`,
	1669: `lead(val: geography, n: int, default: geography) -> geography`,
	1670: `lead(val: geometry, n: int, default: geometry) -> geometry`,
	1671: `lead(val: string, n: int, default: string) -> string`,
	1672: `lead(val: bytes, n: int, default: bytes) -> bytes`,
	1673: `lead(val: timestamptz, n: int, default: timestamptz) -> timestamptz`,
	1674: `lead(val: oid, n: int, default: oid) -> oid`,
	1675: `lead(val: uuid, n: int, default: uuid) -> uuid`,
	1676: `lead(val: inet, n: int, default: inet) -> inet`,
	1677: `lead(val: time, n: int, default: time) -> time`,
	1678: `lead(val: timetz, n: int, default: timetz) -> timetz`,
	1679: `lead(val: jsonb, n: int, default: jsonb) -> jsonb`,
	1680: `lead(val: varbit, n: int, default: varbit) -> varbit`,
	1681: `first_value(val: bool) -> bool`,
	1682: `first_value(val: box2d) -> box2d`,
	1683: `first_value(val: int) -> int`,
	1684: `first_value(val: float) -> float`,
	1685: `first_value(val: decimal) -> decimal`,
	1686: `first_value(val: date) -> date`,
	1687: `first_value(val: timestamp) -> timestamp`,
	1688: `first_value(val: interval) -> interval`,
	1689: `first_value(val: geography) -> geography`,
	1690: `first_value(val: geometry) -> geometry`,
	1691: `first_value(val: string) -> string`,
	1692: `first_value(val: bytes) -> bytes`,
	1693: `first_value(val: timestamptz) -> timestamptz`,
	1694: `first_value(val: oid) -> oid`,
	1695: `first_value(val: uuid) -> uuid`,
	1696: `first_value(val: inet) -> inet`,
	1697: `first_value(val: time) -> time`,
	1698: `first_value(val: timetz) -> timetz`,
	1699: `first_value(val: jsonb) -> jsonb`,
	1700: `first_value(val: varbit) -> varbit`,
	1701: `last_value(val: bool) -> bool`,
	1702: `last_value(val: box2d) -> box2d`,
	1703: `last_value(val: int) -> int`,
	1704: `last_value(val: float) -> float`,
	1705: `last_value(val: decimal) -> decimal`,
	1706: `last_value(val: date) -> date`,
	1707: `last_value(val: timestamp) -> timestamp`,
	1708: `last_value(val: interval) -> interval`,
	1709: `last_value(val: geography) -> geography`,
	1710: `last_value(val: geometry) -> geometry`,
	1711: `last_value(val: string) -> string`,
	1712: `last_value(val: bytes) -> bytes`,
	1713: `last_value(val: timestamptz) -> timestamptz`,
	1714: `last_value(val: oid) -> oid`,
	1715: `last_value(val: uuid) -> uuid`,
	1716: `last_value(val: inet) -> inet`,
	1717: `last_value(val: time) -> time`,
	1718: `last_value(val: timetz) -> timetz`,
	1719: `last_value(val: jsonb) -> jsonb`,
	1720: `last_value(val: varbit) -> varbit`,
	1721: `nth_value(val: bool, n: int) -> bool`,
	1722: `nth_value(val: box2d, n: int) -> box2d`,
	1723: `nth_value(val: int, n: int) -> int`,
	1724: `nth_value(val: float, n: int) -> float`,
	1725: `nth_value(val: decimal, n: int) -> decimal`,
	1726: `nth_value(val: date, n: int) -> date`,
	1727: `nth_value(val: timestamp, n: int) -> timestamp`,
	1728: `nth_value(val: interval, n: int) -> interval`,
	1729: `nth_value(val: geography, n: int) -> geography`,
	1730: `nth_value(val: geometry, n: int) -> geometry`,
	1731: `nth_value(val: string, n: int) -> string`,
	1732: `nth_value(val: bytes, n: int) -> bytes`,
	1733: `nth_value(val: timestamptz, n: int) -> timestamptz`,
	1734: `nth_value(val: oid, n: int) -> oid`,
	1735: `nth_value(val: uuid, n: int) -> uuid`,
	1736: `nth_value(val: inet, n: int) -> inet`,
	1737: `nth_value(val: time, n: int) -> time`,
	1738: `nth_value(val: timetz, n: int) -> timetz`,
	1739: `nth_value(val: jsonb, n: int) -> jsonb`,
	1740: `nth_value(val: varbit, n: int) -> varbit`,
	1741: `geomfromewkt(val: string) -> geometry`,
	1742: `geomfromewkb(val: bytes) -> geometry`,
	1743: `st_coorddim(geometry: geometry) -> int`,
	1744: `st_geogfromtext(val: string) -> geography`,
	1745: `st_geogfromtext(str: string, srid: int) -> geography`,
	1746: `st_geomfromtext(val: string) -> geometry`,
	1747: `st_geomfromtext(str: string, srid: int) -> geometry`,
	1748: `st_numinteriorring(geometry: geometry) -> int`,
	1749: `st_symmetricdifference(geometry_a: geometry, geometry_b: geometry) -> geometry`,
	1750: `st_force3d(geometry: geometry) -> geometry`,
	1751: `st_force3d(geometry: geometry, defaultZ: float) -> geometry`,
	1752: `_st_touches(geometry_a: geometry, geometry_b: geometry) -> bool`,
	1753: `_st_containsproperly(geometry_a: geometry, geometry_b: geometry) -> bool`,
	1754: `_st_equals(geometry_a: geometry, geometry_b: geometry) -> bool`,
	1755: `_st_within(geometry_a: geometry, geometry_b: geometry) -> bool`,
	1756: `_st_dwithin(geometry_a: geometry, geometry_b: geometry, distance: float) -> bool`,
	1757: `_st_dwithin(geography_a: geography, geography_b: geography, distance: float) -> bool`,
	1758: `_st_dwithin(geography_a: geography, geography_b: geography, distance: float, use_spheroid: bool) -> bool`,
	1759: `_st_intersects(geometry_a: geometry, geometry_b: geometry) -> bool`,
	1760: `_st_intersects(geography_a: geography, geography_b: geography) -> bool`,
	1761: `_st_dfullywithinexclusive(geometry_a: geometry, geometry_b: geometry, distance: float) -> bool`,
	1762: `_st_dfullywithin(geometry_a: geometry, geometry_b: geometry, distance: float) -> bool`,
	1763: `_st_coveredby(geometry_a: geometry, geometry_b: geometry) -> bool`,
	1764: `_st_coveredby(geography_a: geography, geography_b: geography) -> bool`,
	1765: `_st_contains(geometry_a: geometry, geometry_b: geometry) -> bool`,
	1766: `_st_crosses(geometry_a: geometry, geometry_b: geometry) -> bool`,
	1767: `_st_overlaps(geometry_a: geometry, geometry_b: geometry) -> bool`,
	1768: `_st_dwithinexclusive(geometry_a: geometry, geometry_b: geometry, distance: float) -> bool`,
	1769: `_st_dwithinexclusive(geography_a: geography, geography_b: geography, distance: float) -> bool`,
	1770: `_st_dwithinexclusive(geography_a: geography, geography_b: geography, distance: float, use_spheroid: bool) -> bool`,
	1771: `_st_covers(geometry_a: geometry, geometry_b: geometry) -> bool`,
	1772: `_st_covers(geography_a: geography, geography_b: geography) -> bool`,
	1773: `st_area(geography: geography) -> float`,
	1774: `st_area(geography: geography, use_spheroid: bool) -> float`,
	1775: `st_area(geometry: geometry) -> float`,
	1776: `st_area(geometry_str: string) -> float`,
	1777: `st_asewkt(geometry: geometry) -> string`,
	1778: `st_asewkt(geometry: geometry, max_decimal_digits: int) -> string`,
	1779: `st_asewkt(geography: geography) -> string`,
	1780: `st_asewkt(geography: geography, max_decimal_digits: int) -> string`,
	1781: `st_asewkt(geometry_str: string) -> string`,
	1782: `st_asewkt(geometry_str: string, max_decimal_digits: int) -> string`,
	1783: `st_asgeojson(row: tuple) -> string`,
	1784: `st_asgeojson(row: tuple, geo_column: string) -> string`,
	1785: `st_asgeojson(row: tuple, geo_column: string, max_decimal_digits: int) -> string`,
	1786: `st_asgeojson(row: tuple, geo_column: string, max_decimal_digits: int, pretty: bool) -> string`,
	1787: `st_asgeojson(geometry: geometry) -> string`,
	1788: `st_asgeojson(geometry: geometry, max_decimal_digits: int) -> string`,
	1789: `st_asgeojson(geometry: geometry, max_decimal_digits: int, options: int) -> string`,
	1790: `st_asgeojson(geography: geography) -> string`,
	1791: `st_asgeojson(geography: geography, max_decimal_digits: int) -> string`,
	1792: `st_asgeojson(geography: geography, max_decimal_digits: int, options: int) -> string`,
	1793: `st_asgeojson(geometry_str: string) -> string`,
	1794: `st_asgeojson(geometry_str: string, max_decimal_digits: int) -> string`,
	1795: `st_asgeojson(geometry_str: string, max_decimal_digits: int, options: int) -> string`,
	1796: `st_askml(geometry: geometry) -> string`,
	1797: `st_askml(geography: geography) -> string`,
	1798: `st_askml(geometry_str: string) -> string`,
	1799: `st_astext(geometry: geometry) -> string`,
	1800: `st_astext(geometry: geometry, max_decimal_digits: int) -> string`,
	1801: `st_astext(geography: geography) -> string`,
	1802: `st_astext(geography: geography, max_decimal_digits: int) -> string`,
	1803: `st_astext(geometry_str: string) -> string`,
	1804: `st_astext(geometry_str: string, max_decimal_digits: int) -> string`,
	1805: `st_buffer(geometry: geometry, distance: int) -> geometry`,
	1806: `st_buffer(geometry: geometry, distance: float) -> geometry`,
	1807: `st_buffer(geometry: geometry, distance: decimal) -> geometry`,
	1808: `st_buffer(geometry: geometry, distance: float, quad_segs: int) -> geometry`,
	1809: `st_buffer(geometry: geometry, distance: float, buffer_style_params: string) -> geometry`,
	1810: `st_buffer(geography: geography, distance: float) -> geography`,
	1811: `st_buffer(geography: geography, distance: float, quad_segs: int) -> geography`,
	1812: `st_buffer(geography: geography, distance: float, buffer_style_params: string) -> geography`,
	1813: `st_buffer(geometry_str: string, distance: int) -> geometry`,
	1814: `st_buffer(geometry_str: string, distance: float) -> geometry`,
	1815: `st_buffer(geometry_str: string, distance: decimal) -> geometry`,
	1816: `st_buffer(geometry_str: string, distance: float, quad_segs: int) -> geometry`,
	1817: `st_buffer(geometry_str: string, distance: float, buffer_style_params: string) -> geometry`,
	1818: `st_centroid(geography: geography) -> geography`,
	1819: `st_centroid(geography: geography, use_spheroid: bool) -> geography`,
	1820: `st_centroid(geometry: geometry) -> geometry`,
	1821: `st_centroid(geometry_str: string) -> geometry`,
	1822: `st_coveredby(geometry_a: geometry, geometry_b: geometry) -> bool`,
	1823: `st_coveredby(geography_a: geography, geography_b: geography) -> bool`,
	1824: `st_coveredby(geometry_a_str: string, geometry_b_str: string) -> bool`,
	1825: `st_covers(geometry_a: geometry, geometry_b: geometry) -> bool`,
	1826: `st_covers(geography_a: geography, geography_b: geography) -> bool`,
	1827: `st_covers(geometry_a_str: string, geometry_b_str: string) -> bool`,
	1828: `st_distance(geometry_a: geometry, geometry_b: geometry) -> float`,
	1829: `st_distance(geography_a: geography, geography_b: geography) -> float`,
	1830: `st_distance(geography_a: geography, geography_b: geography, use_spheroid: bool) -> float`,
	1831: `st_distance(geometry_a_str: string, geometry_b_str: string) -> float`,
	1832: `st_dwithin(geometry_a: geometry, geometry_b: geometry, distance: float) -> bool`,
	1833: `st_dwithin(geography_a: geography, geography_b: geography, distance: float) -> bool`,
	1834: `st_dwithin(geography_a: geography, geography_b: geography, distance: float, use_spheroid: bool) -> bool`,
	1835: `st_dwithin(geometry_a_str: string, geometry_b_str: string, distance: float) -> bool`,
	1836: `st_intersects(geometry_a: geometry, geometry_b: geometry) -> bool`,
	1837: `st_intersects(geography_a: geography, geography_b: geography) -> bool`,
	1838: `st_intersects(geometry_a_str: string, geometry_b_str: string) -> bool`,
	1839: `st_intersection(geometry_a: geometry, geometry_b: geometry) -> geometry`,
	1840: `st_intersection(geography_a: geography, geography_b: geography) -> geography`,
	1841: `st_intersection(geometry_a_str: string, geometry_b_str: string) -> geometry`,
	1842: `st_length(geography: geography) -> float`,
	1843: `st_length(geography: geography, use_spheroid: bool) -> float`,
	1844: `st_length(geometry: geometry) -> float`,
	1845: `st_length(geometry_str: string) -> float`,
	1846: `st_dwithinexclusive(geometry_a: geometry, geometry_b: geometry, distance: float) -> bool`,
	1847: `st_dwithinexclusive(geography_a: geography, geography_b: geography, distance: float) -> bool`,
	1848: `st_dwithinexclusive(geography_a: geography, geography_b: geography, distance: float, use_spheroid: bool) -> bool`,
	1849: `st_dwithinexclusive(geometry_a_str: string, geometry_b_str: string, distance: float) -> bool`,
	1850: `regrolesend(regrole: regrole) -> bytes`,
	1851: `regrolerecv(input: anyelement) -> regrole`,
	1852: `regroleout(regrole: regrole) -> bytes`,
	1853: `regrolein(input: anyelement) -> regrole`,
	1854: `anyelement_send(anyelement: anyelement) -> bytes`,
	1855: `anyelement_recv(input: anyelement) -> anyelement`,
	1856: `anyelement_out(anyelement: anyelement) -> bytes`,
	1857: `anyelement_in(input: anyelement) -> anyelement`,
	1858: `jsonb_send(jsonb: jsonb) -> bytes`,
	1859: `jsonb_recv(input: anyelement) -> jsonb`,
	1860: `jsonb_out(jsonb: jsonb) -> bytes`,
	1861: `jsonb_in(input: anyelement) -> jsonb`,
	1862: `timestamptz_send(timestamptz: timestamptz) -> bytes`,
	1863: `timestamptz_recv(input: anyelement) -> timestamptz`,
	1864: `timestamptz_out(timestamptz: timestamptz) -> bytes`,
	1865: `timestamptz_in(input: anyelement) -> timestamptz`,
	1866: `geography_send(geography: geography) -> bytes`,
	1867: `geography_recv(input: anyelement) -> geography`,
	1868: `geography_out(geography: geography) -> bytes`,
	1869: `geography_in(input: anyelement) -> geography`,
	1870: `bpcharsend(bpchar: bpchar) -> bytes`,
	1871: `bpcharrecv(input: anyelement) -> bpchar`,
	1872: `bpcharout(bpchar: bpchar) -> bytes`,
	1873: `bpcharin(input: anyelement) -> bpchar`,
	1874: `geometry_send(geometry: geometry) -> bytes`,
	1875: `geometry_recv(input: anyelement) -> geometry`,
	1876: `geometry_out(geometry: geometry) -> bytes`,
	1877: `geometry_in(input: anyelement) -> geometry`,
	1878: `int2vectorsend(int2vector: int2vector) -> bytes`,
	1879: `int2vectorrecv(input: anyelement) -> int2vector`,
	1880: `int2vectorout(int2vector: int2vector) -> bytes`,
	1881: `int2vectorin(input: anyelement) -> int2vector`,
	1882: `textsend(string: string) -> bytes`,
	1883: `textrecv(input: anyelement) -> string`,
	1884: `textout(string: string) -> bytes`,
	1885: `textin(input: anyelement) -> string`,
	1886: `varcharsend(varchar: varchar) -> bytes`,
	1887: `varcharrecv(input: anyelement) -> varchar`,
	1888: `varcharout(varchar: varchar) -> bytes`,
	1889: `varcharin(input: anyelement) -> varchar`,
	1890: `regclasssend(regclass: regclass) -> bytes`,
	1891: `regclassrecv(input: anyelement) -> regclass`,
	1892: `regclassout(regclass: regclass) -> bytes`,
	1893: `regclassin(input: anyelement) -> regclass`,
	1894: `unknownsend(unknown: unknown) -> bytes`,
	1895: `unknownrecv(input: anyelement) -> unknown`,
	1896: `unknownout(unknown: unknown) -> bytes`,
	1897: `unknownin(input: anyelement) -> unknown`,
	1898: `int8send(int: int) -> bytes`,
	1899: `int8recv(input: anyelement) -> int`,
	1900: `int8out(int: int) -> bytes`,
	1901: `int8in(input: anyelement) -> int`,
	1902: `oidsend(oid: oid) -> bytes`,
	1903: `oidrecv(input: anyelement) -> oid`,
	1904: `oidout(oid: oid) -> bytes`,
	1905: `oidin(input: anyelement) -> oid`,
	1906: `timetz_send(timetz: timetz) -> bytes`,
	1907: `timetz_recv(input: anyelement) -> timetz`,
	1908: `timetz_out(timetz: timetz) -> bytes`,
	1909: `timetz_in(input: anyelement) -> timetz`,
	1910: `float4send(float4: float4) -> bytes`,
	1911: `float4recv(input: anyelement) -> float4`,
	1912: `float4out(float4: float4) -> bytes`,
	1913: `float4in(input: anyelement) -> float4`,
	1914: `regtypesend(regtype: regtype) -> bytes`,
	1915: `regtyperecv(input: anyelement) -> regtype`,
	1916: `regtypeout(regtype: regtype) -> bytes`,
	1917: `regtypein(input: anyelement) -> regtype`,
	1918: `byteasend(bytes: bytes) -> bytes`,
	1919: `bytearecv(input: anyelement) -> bytes`,
	1920: `byteaout(bytes: bytes) -> bytes`,
	1921: `byteain(input: anyelement) -> bytes`,
	1922: `regproceduresend(regprocedure: regprocedure) -> bytes`,
	1923: `regprocedurerecv(input: anyelement) -> regprocedure`,
	1924: `regprocedureout(regprocedure: regprocedure) -> bytes`,
	1925: `regprocedurein(input: anyelement) -> regprocedure`,
	1926: `charsend("char": "char") -> bytes`,
	1927: `charrecv(input: anyelement) -> "char"`,
	1928: `charout("char": "char") -> bytes`,
	1929: `charin(input: anyelement) -> "char"`,
	1930: `record_send(tuple: tuple) -> bytes`,
	1931: `record_recv(input: anyelement) -> anyelement`,
	1932: `record_out(tuple: tuple) -> bytes`,
	1933: `record_in(input: anyelement) -> anyelement`,
	1934: `regprocsend(regproc: regproc) -> bytes`,
	1935: `regprocrecv(input: anyelement) -> regproc`,
	1936: `regprocout(regproc: regproc) -> bytes`,
	1937: `regprocin(input: anyelement) -> regproc`,
	1938: `varbit_send(varbit: varbit) -> bytes`,
	1939: `varbit_recv(input: anyelement) -> varbit`,
	1940: `varbit_out(varbit: varbit) -> bytes`,
	1941: `varbit_in(input: anyelement) -> varbit`,
	1942: `bit_send(bit: bit) -> bytes`,
	1943: `bit_recv(input: anyelement) -> bit`,
	1944: `bit_out(bit: bit) -> bytes`,
	1945: `bit_in(input: anyelement) -> bit`,
	1946: `boolsend(bool: bool) -> bytes`,
	1947: `boolrecv(input: anyelement) -> bool`,
	1948: `boolout(bool: bool) -> bytes`,
	1949: `boolin(input: anyelement) -> bool`,
	1950: `inetsend(inet: inet) -> bytes`,
	1951: `inetrecv(input: anyelement) -> inet`,
	1952: `inetout(inet: inet) -> bytes`,
	1953: `inetin(input: anyelement) -> inet`,
	1954: `date_send(date: date) -> bytes`,
	1955: `date_recv(input: anyelement) -> date`,
	1956: `date_out(date: date) -> bytes`,
	1957: `date_in(input: anyelement) -> date`,
	1958: `uuid_send(uuid: uuid) -> bytes`,
	1959: `uuid_recv(input: anyelement) -> uuid`,
	1960: `uuid_out(uuid: uuid) -> bytes`,
	1961: `uuid_in(input: anyelement) -> uuid`,
	1962: `oidvectorsend(oidvector: oidvector) -> bytes`,
	1963: `oidvectorrecv(input: anyelement) -> oidvector`,
	1964: `oidvectorout(oidvector: oidvector) -> bytes`,
	1965: `oidvectorin(input: anyelement) -> oidvector`,
	1966: `numeric_send(decimal: decimal) -> bytes`,
	1967: `numeric_recv(input: anyelement) -> decimal`,
	1968: `numeric_out(decimal: decimal) -> bytes`,
	1969: `numeric_in(input: anyelement) -> decimal`,
	1970: `box2d_send(box2d: box2d) -> bytes`,
	1971: `box2d_recv(input: anyelement) -> box2d`,
	1972: `box2d_out(box2d: box2d) -> bytes`,
	1973: `box2d_in(input: anyelement) -> box2d`,
	1974: `float8send(float: float) -> bytes`,
	1975: `float8recv(input: anyelement) -> float`,
	1976: `float8out(float: float) -> bytes`,
	1977: `float8in(input: anyelement) -> float`,
	1978: `int4send(int4: int4) -> bytes`,
	1979: `int4recv(input: anyelement) -> int4`,
	1980: `int4out(int4: int4) -> bytes`,
	1981: `int4in(input: anyelement) -> int4`,
	1982: `namesend(name: name) -> bytes`,
	1983: `namerecv(input: anyelement) -> name`,
	1984: `nameout(name: name) -> bytes`,
	1985: `namein(input: anyelement) -> name`,
	1986: `regnamespacesend(regnamespace: regnamespace) -> bytes`,
	1987: `regnamespacerecv(input: anyelement) -> regnamespace`,
	1988: `regnamespaceout(regnamespace: regnamespace) -> bytes`,
	1989: `regnamespacein(input: anyelement) -> regnamespace`,
	1990: `timestamp_send(timestamp: timestamp) -> bytes`,
	1991: `timestamp_recv(input: anyelement) -> timestamp`,
	1992: `timestamp_out(timestamp: timestamp) -> bytes`,
	1993: `timestamp_in(input: anyelement) -> timestamp`,
	1994: `voidsend(void: void) -> bytes`,
	1995: `voidrecv(input: anyelement) -> void`,
	1996: `voidout(void: void) -> bytes`,
	1997: `voidin(input: anyelement) -> void`,
	1998: `int2send(int2: int2) -> bytes`,
	1999: `int2recv(input: anyelement) -> int2`,
	2000: `int2out(int2: int2) -> bytes`,
	2001: `int2in(input: anyelement) -> int2`,
	2002: `interval_send(interval: interval) -> bytes`,
	2003: `interval_recv(input: anyelement) -> interval`,
	2004: `interval_out(interval: interval) -> bytes`,
	2005: `interval_in(input: anyelement) -> interval`,
	2006: `time_send(time: time) -> bytes`,
	2007: `time_recv(input: anyelement) -> time`,
	2008: `time_out(time: time) -> bytes`,
	2009: `time_in(input: anyelement) -> time`,
	2010: `array_send(anyelement[]: anyelement[]) -> bytes`,
	2011: `array_recv(input: anyelement) -> anyelement[]`,
	2012: `array_out(anyelement[]: anyelement[]) -> bytes`,
	2013: `array_in(input: anyelement) -> anyelement[]`,
	2014: `anyarray_send(anyelement[]: anyelement[]) -> bytes`,
	2015: `anyarray_recv(input: anyelement) -> anyelement[]`,
	2016: `anyarray_out(anyelement[]: anyelement[]) -> bytes`,
	2017: `anyarray_in(input: anyelement) -> anyelement[]`,
	2018: `enum_send(anyenum: anyenum) -> bytes`,
	2019: `enum_recv(input: anyelement) -> anyenum`,
	2020: `enum_out(anyenum: anyenum) -> bytes`,
	2021: `enum_in(input: anyelement) -> anyenum`,
	2022: `crdb_internal.create_regclass(oid: oid, name: string) -> regclass`,
	2023: `to_regclass(text: string) -> regtype`,
	2024: `crdb_internal.create_regnamespace(oid: oid, name: string) -> regnamespace`,
	2025: `to_regnamespace(text: string) -> regtype`,
	2026: `crdb_internal.create_regproc(oid: oid, name: string) -> regproc`,
	2027: `to_regproc(text: string) -> regtype`,
	2028: `crdb_internal.create_regprocedure(oid: oid, name: string) -> regprocedure`,
	2029: `to_regprocedure(text: string) -> regtype`,
	2030: `crdb_internal.create_regrole(oid: oid, name: string) -> regrole`,
	2031: `to_regrole(text: string) -> regtype`,
	2032: `crdb_internal.create_regtype(oid: oid, name: string) -> regtype`,
	2033: `to_regtype(text: string) -> regtype`,
	2035: `pg_get_functiondef(func_oid: oid) -> string`,
	2036: `crdb_internal.create_tenant(id: int, name: string) -> int`,
	2038: `crdb_internal.redact_descriptor(descriptor: bytes) -> bytes`,
	2039: `crdb_internal.descriptor_with_post_deserialization_changes(descriptor: bytes) -> bytes`,
	2040: `to_char(timestamp: timestamp, format: string) -> string`,
	2041: `to_char(timestamptz: timestamptz, format: string) -> string`,
	2042: `to_char(interval: interval, format: string) -> string`,
	2043: `crdb_internal.create_tenant(name: string) -> int`,
	2044: `crdb_internal.start_replication_stream(tenant_name: string) -> bytes`,
	2045: `pg_blocking_pids() -> int[]`,
	2046: `crdb_internal.fingerprint(span: bytes[], start_time: timestamptz, all_revisions: bool) -> int`,
	2047: `tsquerysend(tsquery: tsquery) -> bytes`,
	2048: `tsvectorsend(tsvector: tsvector) -> bytes`,
	2049: `tsqueryout(tsquery: tsquery) -> bytes`,
	2050: `tsvectorout(tsvector: tsvector) -> bytes`,
	2051: `tsqueryrecv(input: anyelement) -> tsquery`,
	2052: `tsvectorrecv(input: anyelement) -> tsvector`,
	2053: `tsqueryin(input: anyelement) -> tsquery`,
	2054: `tsvectorin(input: anyelement) -> tsvector`,
	2055: `crdb_internal.hide_sql_constants(val: string) -> string`,
	2056: `crdb_internal.hide_sql_constants(val: string[]) -> string[]`,
	2057: ``, // formerly crdb_internal.unsafe_optimize_system_database() -> bool
	2058: `parse_ident(qualified_identifier: string) -> string[]`,
	2059: `parse_ident(qualified_identifier: string, strict: bool) -> string[]`,
	2060: `pg_get_function_arguments(func_oid: oid) -> string`,
	2061: `crdb_internal.job_payload_type(data: bytes) -> string`,
	2062: `crdb_internal.tenant_span(tenant_name: string) -> bytes[]`,
	2064: `crdb_internal.generate_test_objects(names: string, number: int) -> jsonb`,
	2065: `crdb_internal.generate_test_objects(names: string, counts: int[]) -> jsonb`,
	2066: `crdb_internal.generate_test_objects(parameters: jsonb) -> jsonb`,
	2067: `crdb_internal.gen_rand_ident(name_pattern: string, count: int) -> string`,
	2068: `crdb_internal.gen_rand_ident(name_pattern: string, count: int, parameters: jsonb) -> string`,
	2069: `crdb_internal.create_tenant(parameters: jsonb) -> int`,
	2070: `crdb_internal.num_inverted_index_entries(val: tsvector, version: int) -> int`,
	2072: `crdb_internal.upsert_dropped_relation_gc_ttl(desc_id: int, gc_ttl: interval) -> bool`,
	2073: `to_tsquery(config: string, text: string) -> tsquery`,
	2074: `to_tsvector(config: string, text: string) -> tsvector`,
	2075: `phraseto_tsquery(config: string, text: string) -> tsquery`,
	2076: `plainto_tsquery(config: string, text: string) -> tsquery`,
	2077: `ts_parse(parser_name: string, document: string) -> tuple{int AS tokid, string AS token}`,
	2078: `crdb_internal.humanize_bytes(val: int) -> string`,
	2079: `geography(jsonb: jsonb) -> geography`,
	2080: `geography(string: string) -> geography`,
	2081: `timetz(time: time) -> timetz`,
	2082: `timetz(timestamptz: timestamptz) -> timetz`,
	2083: `timetz(timetz: timetz) -> timetz`,
	2084: `timetz(string: string) -> timetz`,
	2085: `int8(bit: bit) -> int`,
	2086: `int8(interval: interval) -> int`,
	2087: `int8(decimal: decimal) -> int`,
	2088: `int8(date: date) -> int`,
	2089: `int8(oid: oid) -> int`,
	2090: `int8(bool: bool) -> int`,
	2091: `int8(int: int) -> int`,
	2092: `int8(timestamp: timestamp) -> int`,
	2093: `int8(timestamptz: timestamptz) -> int`,
	2094: `int8(float: float) -> int`,
	2095: `int8(jsonb: jsonb) -> int`,
	2096: `int8(string: string) -> int`,
	2097: `timestamp(date: date) -> timestamp`,
	2098: `timestamp(int: int) -> timestamp`,
	2099: `timestamp(timestamp: timestamp) -> timestamp`,
	2100: `timestamp(timestamptz: timestamptz) -> timestamp`,
	2101: `timestamp(string: string) -> timestamp`,
	2102: `bit(bit: bit) -> bit`,
	2103: `bit(int: int) -> bit`,
	2104: `bit(string: string) -> bit`,
	2105: `float8(interval: interval) -> float`,
	2106: `float8(decimal: decimal) -> float`,
	2107: `float8(date: date) -> float`,
	2108: `float8(bool: bool) -> float`,
	2109: `float8(int: int) -> float`,
	2110: `float8(timestamp: timestamp) -> float`,
	2111: `float8(timestamptz: timestamptz) -> float`,
	2112: `float8(float: float) -> float`,
	2113: `float8(jsonb: jsonb) -> float`,
	2114: `float8(string: string) -> float`,
	2115: `tsquery(string: string) -> tsquery`,
	2116: `tsquery(tsquery: tsquery) -> tsquery`,
	2117: `inet(string: string) -> inet`,
	2118: `inet(inet: inet) -> inet`,
	2119: `char(uuid: uuid) -> "char"`,
	2120: `char(bit: bit) -> "char"`,
	2121: `char(box2d: box2d) -> "char"`,
	2122: `char(interval: interval) -> "char"`,
	2123: `char(decimal: decimal) -> "char"`,
	2124: `char(tuple: tuple) -> "char"`,
	2125: `char(date: date) -> "char"`,
	2126: `char(oid: oid) -> "char"`,
	2127: `char(geometry: geometry) -> "char"`,
	2128: `char(time: time) -> "char"`,
	2129: `char(tsvector: tsvector) -> "char"`,
	2130: `char(bool: bool) -> "char"`,
	2131: `char(int: int) -> "char"`,
	2132: `char(timestamp: timestamp) -> "char"`,
	2133: `char(timestamptz: timestamptz) -> "char"`,
	2134: `char(tsquery: tsquery) -> "char"`,
	2135: `char(float: float) -> "char"`,
	2136: `char(geography: geography) -> "char"`,
	2137: `char(inet: inet) -> "char"`,
	2138: `char(timetz: timetz) -> "char"`,
	2139: `char(void: void) -> "char"`,
	2140: `char(bytes: bytes) -> "char"`,
	2141: `char(jsonb: jsonb) -> "char"`,
	2142: `char(string: string) -> "char"`,
	2143: `bytes(uuid: uuid) -> bytes`,
	2144: `bytes(geometry: geometry) -> bytes`,
	2145: `bytes(geography: geography) -> bytes`,
	2146: `bytes(string: string) -> bytes`,
	2147: `bytes(bytes: bytes) -> bytes`,
	2148: `int4(bit: bit) -> int4`,
	2149: `int4(interval: interval) -> int4`,
	2150: `int4(decimal: decimal) -> int4`,
	2151: `int4(date: date) -> int4`,
	2152: `int4(oid: oid) -> int4`,
	2153: `int4(bool: bool) -> int4`,
	2154: `int4(timestamp: timestamp) -> int4`,
	2155: `int4(timestamptz: timestamptz) -> int4`,
	2156: `int4(float: float) -> int4`,
	2157: `int4(jsonb: jsonb) -> int4`,
	2158: `int4(string: string) -> int4`,
	2159: `int4(int: int) -> int4`,
	2160: `geometry(box2d: box2d) -> geometry`,
	2161: `geometry(geometry: geometry) -> geometry`,
	2162: `geometry(geography: geography) -> geometry`,
	2163: `geometry(bytes: bytes) -> geometry`,
	2164: `geometry(jsonb: jsonb) -> geometry`,
	2165: `geometry(string: string) -> geometry`,
	2166: `float4(interval: interval) -> float4`,
	2167: `float4(decimal: decimal) -> float4`,
	2168: `float4(date: date) -> float4`,
	2169: `float4(bool: bool) -> float4`,
	2170: `float4(int: int) -> float4`,
	2171: `float4(timestamp: timestamp) -> float4`,
	2172: `float4(timestamptz: timestamptz) -> float4`,
	2173: `float4(jsonb: jsonb) -> float4`,
	2174: `float4(string: string) -> float4`,
	2175: `float4(float: float) -> float4`,
	2176: `regprocedure(oid: oid) -> regprocedure`,
	2177: `regprocedure(int: int) -> regprocedure`,
	2178: `regprocedure(string: string) -> regprocedure`,
	2179: `regtype(oid: oid) -> regtype`,
	2180: `regtype(int: int) -> regtype`,
	2181: `regtype(string: string) -> regtype`,
	2182: `text(uuid: uuid) -> string`,
	2183: `text(bit: bit) -> string`,
	2184: `text(box2d: box2d) -> string`,
	2185: `text(interval: interval) -> string`,
	2186: `text(decimal: decimal) -> string`,
	2187: `text(tuple: tuple) -> string`,
	2188: `text(date: date) -> string`,
	2189: `text(oid: oid) -> string`,
	2190: `text(geometry: geometry) -> string`,
	2191: `text(time: time) -> string`,
	2192: `text(tsvector: tsvector) -> string`,
	2193: `text(bool: bool) -> string`,
	2194: `text(int: int) -> string`,
	2195: `text(timestamp: timestamp) -> string`,
	2196: `text(timestamptz: timestamptz) -> string`,
	2197: `text(tsquery: tsquery) -> string`,
	2198: `text(float: float) -> string`,
	2199: `text(geography: geography) -> string`,
	2201: `text(timetz: timetz) -> string`,
	2202: `text(void: void) -> string`,
	2203: `text(bytes: bytes) -> string`,
	2204: `text(jsonb: jsonb) -> string`,
	2205: `text(string: string) -> string`,
	2206: `varchar(uuid: uuid) -> varchar`,
	2207: `varchar(bit: bit) -> varchar`,
	2208: `varchar(box2d: box2d) -> varchar`,
	2209: `varchar(interval: interval) -> varchar`,
	2210: `varchar(decimal: decimal) -> varchar`,
	2211: `varchar(tuple: tuple) -> varchar`,
	2212: `varchar(date: date) -> varchar`,
	2213: `varchar(oid: oid) -> varchar`,
	2214: `varchar(geometry: geometry) -> varchar`,
	2215: `varchar(time: time) -> varchar`,
	2216: `varchar(tsvector: tsvector) -> varchar`,
	2217: `varchar(bool: bool) -> varchar`,
	2218: `varchar(int: int) -> varchar`,
	2219: `varchar(timestamp: timestamp) -> varchar`,
	2220: `varchar(timestamptz: timestamptz) -> varchar`,
	2221: `varchar(tsquery: tsquery) -> varchar`,
	2222: `varchar(float: float) -> varchar`,
	2223: `varchar(geography: geography) -> varchar`,
	2224: `varchar(inet: inet) -> varchar`,
	2225: `varchar(timetz: timetz) -> varchar`,
	2226: `varchar(void: void) -> varchar`,
	2227: `varchar(bytes: bytes) -> varchar`,
	2228: `varchar(jsonb: jsonb) -> varchar`,
	2229: `varchar(string: string) -> varchar`,
	2230: `void(string: string) -> void`,
	2231: `void(void: void) -> void`,
	2232: `regnamespace(oid: oid) -> regnamespace`,
	2233: `regnamespace(int: int) -> regnamespace`,
	2234: `regnamespace(string: string) -> regnamespace`,
	2235: `regclass(oid: oid) -> regclass`,
	2236: `regclass(int: int) -> regclass`,
	2237: `regclass(string: string) -> regclass`,
	2238: `tsvector(string: string) -> tsvector`,
	2239: `tsvector(tsvector: tsvector) -> tsvector`,
	2240: `record(string: string) -> anyelement`,
	2241: `record(tuple: tuple) -> anyelement`,
	2242: `varbit(bit: bit) -> varbit`,
	2243: `varbit(string: string) -> varbit`,
	2244: `timestamptz(date: date) -> timestamptz`,
	2245: `timestamptz(int: int) -> timestamptz`,
	2246: `timestamptz(timestamp: timestamp) -> timestamptz`,
	2247: `timestamptz(timestamptz: timestamptz) -> timestamptz`,
	2248: `timestamptz(string: string) -> timestamptz`,
	2249: `regrole(oid: oid) -> regrole`,
	2250: `regrole(int: int) -> regrole`,
	2251: `regrole(string: string) -> regrole`,
	2252: `jsonb(geometry: geometry) -> jsonb`,
	2253: `jsonb(geography: geography) -> jsonb`,
	2254: `jsonb(string: string) -> jsonb`,
	2255: `jsonb(jsonb: jsonb) -> jsonb`,
	2256: `oid(int: int) -> oid`,
	2257: `oid(string: string) -> oid`,
	2258: `oid(oid: oid) -> oid`,
	2259: `name(uuid: uuid) -> name`,
	2260: `name(bit: bit) -> name`,
	2261: `name(box2d: box2d) -> name`,
	2262: `name(interval: interval) -> name`,
	2263: `name(decimal: decimal) -> name`,
	2264: `name(tuple: tuple) -> name`,
	2265: `name(date: date) -> name`,
	2266: `name(oid: oid) -> name`,
	2267: `name(geometry: geometry) -> name`,
	2268: `name(time: time) -> name`,
	2269: `name(tsvector: tsvector) -> name`,
	2270: `name(bool: bool) -> name`,
	2271: `name(int: int) -> name`,
	2272: `name(timestamp: timestamp) -> name`,
	2273: `name(timestamptz: timestamptz) -> name`,
	2274: `name(tsquery: tsquery) -> name`,
	2275: `name(float: float) -> name`,
	2276: `name(geography: geography) -> name`,
	2277: `name(inet: inet) -> name`,
	2278: `name(timetz: timetz) -> name`,
	2279: `name(void: void) -> name`,
	2280: `name(bytes: bytes) -> name`,
	2281: `name(jsonb: jsonb) -> name`,
	2282: `name(string: string) -> name`,
	2283: `time(interval: interval) -> time`,
	2284: `time(time: time) -> time`,
	2285: `time(timestamp: timestamp) -> time`,
	2286: `time(timestamptz: timestamptz) -> time`,
	2287: `time(timetz: timetz) -> time`,
	2288: `time(string: string) -> time`,
	2289: `bool(decimal: decimal) -> bool`,
	2290: `bool(int: int) -> bool`,
	2291: `bool(float: float) -> bool`,
	2292: `bool(jsonb: jsonb) -> bool`,
	2293: `bool(string: string) -> bool`,
	2294: `bool(bool: bool) -> bool`,
	2295: `uuid(bytes: bytes) -> uuid`,
	2296: `uuid(string: string) -> uuid`,
	2297: `uuid(uuid: uuid) -> uuid`,
	2298: `int2(bit: bit) -> int2`,
	2299: `int2(interval: interval) -> int2`,
	2300: `int2(decimal: decimal) -> int2`,
	2301: `int2(date: date) -> int2`,
	2302: `int2(oid: oid) -> int2`,
	2303: `int2(bool: bool) -> int2`,
	2304: `int2(int: int) -> int2`,
	2305: `int2(timestamp: timestamp) -> int2`,
	2306: `int2(timestamptz: timestamptz) -> int2`,
	2307: `int2(float: float) -> int2`,
	2308: `int2(jsonb: jsonb) -> int2`,
	2309: `int2(string: string) -> int2`,
	2310: `interval(interval: interval) -> interval`,
	2311: `interval(decimal: decimal) -> interval`,
	2312: `interval(time: time) -> interval`,
	2313: `interval(int: int) -> interval`,
	2314: `interval(float: float) -> interval`,
	2315: `interval(string: string) -> interval`,
	2316: `regproc(oid: oid) -> regproc`,
	2317: `regproc(int: int) -> regproc`,
	2318: `regproc(string: string) -> regproc`,
	2319: `date(int: int) -> date`,
	2320: `date(timestamp: timestamp) -> date`,
	2321: `date(timestamptz: timestamptz) -> date`,
	2322: `date(string: string) -> date`,
	2323: `date(date: date) -> date`,
	2324: `bpchar(uuid: uuid) -> bpchar`,
	2325: `bpchar(bit: bit) -> bpchar`,
	2326: `bpchar(box2d: box2d) -> bpchar`,
	2327: `bpchar(interval: interval) -> bpchar`,
	2328: `bpchar(decimal: decimal) -> bpchar`,
	2329: `bpchar(tuple: tuple) -> bpchar`,
	2330: `bpchar(date: date) -> bpchar`,
	2331: `bpchar(oid: oid) -> bpchar`,
	2332: `bpchar(geometry: geometry) -> bpchar`,
	2333: `bpchar(time: time) -> bpchar`,
	2334: `bpchar(tsvector: tsvector) -> bpchar`,
	2335: `bpchar(bool: bool) -> bpchar`,
	2336: `bpchar(int: int) -> bpchar`,
	2337: `bpchar(timestamp: timestamp) -> bpchar`,
	2338: `bpchar(timestamptz: timestamptz) -> bpchar`,
	2339: `bpchar(tsquery: tsquery) -> bpchar`,
	2340: `bpchar(float: float) -> bpchar`,
	2341: `bpchar(geography: geography) -> bpchar`,
	2342: `bpchar(inet: inet) -> bpchar`,
	2343: `bpchar(timetz: timetz) -> bpchar`,
	2344: `bpchar(void: void) -> bpchar`,
	2345: `bpchar(bytes: bytes) -> bpchar`,
	2346: `bpchar(jsonb: jsonb) -> bpchar`,
	2347: `bpchar(string: string) -> bpchar`,
	2348: `numeric(interval: interval) -> decimal`,
	2349: `numeric(decimal: decimal) -> decimal`,
	2350: `numeric(date: date) -> decimal`,
	2351: `numeric(bool: bool) -> decimal`,
	2352: `numeric(int: int) -> decimal`,
	2353: `numeric(timestamp: timestamp) -> decimal`,
	2354: `numeric(timestamptz: timestamptz) -> decimal`,
	2355: `numeric(float: float) -> decimal`,
	2356: `numeric(jsonb: jsonb) -> decimal`,
	2357: `numeric(string: string) -> decimal`,
	2358: `box2d(geometry: geometry) -> box2d`,
	2359: `box2d(string: string) -> box2d`,
	2360: `box2d(box2d: box2d) -> box2d`,
	2361: `geography(geometry: geometry) -> geography`,
	2362: `geography(geography: geography) -> geography`,
	2363: `geography(bytes: bytes) -> geography`,
	2364: `crdb_internal.tenant_span_stats() -> tuple{int AS database_id, int AS table_id, int AS range_count, int AS approximate_disk_bytes, int AS live_bytes, int AS total_bytes, float AS live_percentage}`,
	2365: `crdb_internal.tenant_span_stats(database_id: int) -> tuple{int AS database_id, int AS table_id, int AS range_count, int AS approximate_disk_bytes, int AS live_bytes, int AS total_bytes, float AS live_percentage}`,
	2366: `crdb_internal.tenant_span_stats(database_id: int, table_id: int) -> tuple{int AS database_id, int AS table_id, int AS range_count, int AS approximate_disk_bytes, int AS live_bytes, int AS total_bytes, float AS live_percentage}`,
	2367: `pg_advisory_unlock(key1: int4, key2: int4) -> bool`,
	2368: `pg_advisory_unlock_shared(key: int) -> bool`,
	2369: `pg_advisory_unlock_shared(key1: int4, key2: int4) -> bool`,
	2370: `pg_advisory_unlock_all() -> void`,
	2371: `crdb_internal.redactable_sql_constants(val: string) -> string`,
	2372: `crdb_internal.redactable_sql_constants(val: string[]) -> string[]`,
	2373: `crdb_internal.redact(val: string) -> string`,
	2374: `crdb_internal.redact(val: string[]) -> string[]`,
	2375: `array_cat_agg(arg1: bool[]) -> bool[]`,
	2376: `array_cat_agg(arg1: box2d[]) -> box2d[]`,
	2377: `array_cat_agg(arg1: int[]) -> int[]`,
	2378: `array_cat_agg(arg1: float[]) -> float[]`,
	2379: `array_cat_agg(arg1: decimal[]) -> decimal[]`,
	2380: `array_cat_agg(arg1: date[]) -> date[]`,
	2381: `array_cat_agg(arg1: timestamp[]) -> timestamp[]`,
	2382: `array_cat_agg(arg1: interval[]) -> interval[]`,
	2383: `array_cat_agg(arg1: geography[]) -> geography[]`,
	2384: `array_cat_agg(arg1: geometry[]) -> geometry[]`,
	2385: `array_cat_agg(arg1: string[]) -> string[]`,
	2386: `array_cat_agg(arg1: bytes[]) -> bytes[]`,
	2387: `array_cat_agg(arg1: timestamptz[]) -> timestamptz[]`,
	2388: `array_cat_agg(arg1: oid[]) -> oid[]`,
	2389: `array_cat_agg(arg1: uuid[]) -> uuid[]`,
	2390: `array_cat_agg(arg1: inet[]) -> inet[]`,
	2391: `array_cat_agg(arg1: time[]) -> time[]`,
	2392: `array_cat_agg(arg1: timetz[]) -> timetz[]`,
	2393: `array_cat_agg(arg1: jsonb[]) -> jsonb[]`,
	2394: `array_cat_agg(arg1: varbit[]) -> varbit[]`,
	2395: `array_cat_agg(arg1: anyenum[]) -> anyenum[]`,
	2396: `array_cat_agg(arg1: tuple[]) -> tuple[]`,
	2397: `crdb_internal.update_tenant_resource_limits(tenant_name: string, available_tokens: float, refill_rate: float, max_burst_tokens: float) -> int`,
	2398: `to_tsquery(text: string) -> tsquery`,
	2399: `to_tsvector(text: string) -> tsvector`,
	2400: `phraseto_tsquery(text: string) -> tsquery`,
	2401: `plainto_tsquery(text: string) -> tsquery`,
	2402: `ts_rank(weights: float[], vector: tsvector, query: tsquery, normalization: int) -> float4`,
	2403: `ts_rank(vector: tsvector, query: tsquery, normalization: int) -> float4`,
	2404: `ts_rank(vector: tsvector, query: tsquery) -> float4`,
	2405: `ts_rank(weights: float[], vector: tsvector, query: tsquery) -> float4`,
	2406: `crdb_internal.fingerprint(span: bytes[], stripped: bool) -> int`,
	2407: `crdb_internal.tenant_span() -> bytes[]`,
	2408: `crdb_internal.job_execution_details(job_id: int) -> jsonb`,
	2409: `st_bdpolyfromtext(str: string, srid: int) -> geometry`,
	2410: `crdb_internal.pretty_value(raw_value: bytes) -> string`,
	2411: `to_char(date: date, format: string) -> string`,
	2412: `crdb_internal.unsafe_lock_replica(range_id: int, lock: bool) -> bool`,
	2413: `crdb_internal.fingerprint(span: bytes[], start_time: decimal, all_revisions: bool) -> int`,
	2414: `crdb_internal.tenant_span_stats(spans: tuple[]) -> tuple{bytes AS start_key, bytes AS end_key, jsonb AS stats}`,
	2415: `pg_lsnrecv(input: anyelement) -> pg_lsn`,
	2416: `pg_lsnout(pg_lsn: pg_lsn) -> bytes`,
	2417: `pg_lsnin(input: anyelement) -> pg_lsn`,
	2418: `pg_lsn(string: string) -> pg_lsn`,
	2419: `pg_lsn(pg_lsn: pg_lsn) -> pg_lsn`,
	2420: `varchar(pg_lsn: pg_lsn) -> varchar`,
	2421: `text(pg_lsn: pg_lsn) -> string`,
	2422: `bpchar(pg_lsn: pg_lsn) -> bpchar`,
	2423: `name(pg_lsn: pg_lsn) -> name`,
	2424: `char(pg_lsn: pg_lsn) -> "char"`,
	2425: `max(arg1: pg_lsn) -> anyelement`,
	2426: `percentile_disc_impl(arg1: float, arg2: pg_lsn) -> pg_lsn`,
	2427: `percentile_disc_impl(arg1: float[], arg2: pg_lsn) -> pg_lsn[]`,
	2428: `min(arg1: pg_lsn) -> anyelement`,
	2429: `array_cat_agg(arg1: pg_lsn[]) -> pg_lsn[]`,
	2430: `array_agg(arg1: pg_lsn) -> pg_lsn[]`,
	2431: `array_prepend(elem: pg_lsn, array: pg_lsn[]) -> pg_lsn[]`,
	2432: `array_remove(array: pg_lsn[], elem: pg_lsn) -> anyelement`,
	2433: `array_positions(array: pg_lsn[], elem: pg_lsn) -> int[]`,
	2434: `array_cat(left: pg_lsn[], right: pg_lsn[]) -> pg_lsn[]`,
	2435: `array_position(array: pg_lsn[], elem: pg_lsn) -> int`,
	2436: `array_replace(array: pg_lsn[], toreplace: pg_lsn, replacewith: pg_lsn) -> anyelement`,
	2437: `array_append(array: pg_lsn[], elem: pg_lsn) -> pg_lsn[]`,
	2438: `first_value(val: pg_lsn) -> pg_lsn`,
	2439: `nth_value(val: pg_lsn, n: int) -> pg_lsn`,
	2440: `lag(val: pg_lsn) -> pg_lsn`,
	2441: `lag(val: pg_lsn, n: int) -> pg_lsn`,
	2442: `lag(val: pg_lsn, n: int, default: pg_lsn) -> pg_lsn`,
	2443: `lead(val: pg_lsn) -> pg_lsn`,
	2444: `lead(val: pg_lsn, n: int) -> pg_lsn`,
	2445: `lead(val: pg_lsn, n: int, default: pg_lsn) -> pg_lsn`,
	2446: `last_value(val: pg_lsn) -> pg_lsn`,
	2447: `pg_lsnsend(pg_lsn: pg_lsn) -> bytes`,
	2448: `st_asmvtgeom(geometry: geometry, bbox: box2d, extent: int, buffer: int, clip: bool) -> geometry`,
	2449: `st_asmvtgeom(geometry: geometry, bbox: box2d, extent: int, buffer: int) -> geometry`,
	2450: `st_asmvtgeom(geometry: geometry, bbox: box2d, extent: int) -> geometry`,
	2451: `st_asmvtgeom(geometry: geometry, bbox: box2d) -> geometry`,
	2452: `crdb_internal.repaired_descriptor(descriptor: bytes, valid_descriptor_ids: int[], valid_job_ids: int[], valid_roles: string[]) -> bytes`,
	2453: `crdb_internal.reset_activity_tables() -> bool`,
	2454: `crdb_internal.sstable_metrics(node_id: int, store_id: int, start_key: bytes, end_key: bytes) -> tuple{int AS node_id, int AS store_id, int AS level, int AS file_num, int AS approximate_span_bytes, jsonb AS metrics}`,
	2455: `crdb_internal.repair_catalog_corruption(descriptor_id: int, corruption: string) -> bool`,
	2456: `crdb_internal.merge_aggregated_stmt_metadata(input: jsonb[]) -> jsonb`,
	2457: `crdb_internal.request_job_execution_details(jobID: int) -> bool`,
	2458: `pg_sequence_last_value(sequence_oid: oid) -> int`,
	2459: `nameconcatoid(name: string, oid: oid) -> name`,
	2460: `pg_get_function_arg_default(func_oid: oid, arg_num: int4) -> string`,
	2461: `crdb_internal.plpgsql_raise(severity: string, message: string, detail: string, hint: string, code: string) -> int`,
	2462: `workload_index_recs() -> string`,
	2463: `workload_index_recs(timestamptz: timestamptz) -> string`,
	2466: `crdb_internal.setup_span_configs_stream(tenant_name: string) -> bytes`,
	2467: `crdb_internal.request_statement_bundle(stmtFingerprint: string, planGist: string, samplingProbability: float, minExecutionLatency: interval, expiresAfter: interval) -> bool`,
	2468: `crdb_internal.request_statement_bundle(stmtFingerprint: string, planGist: string, antiPlanGist: bool, samplingProbability: float, minExecutionLatency: interval, expiresAfter: interval) -> bool`,
	2469: `crdb_internal.is_system_table_key(raw_key: bytes) -> bool`,
	2470: `crdb_internal.scan_storage_internal_keys(node_id: int, store_id: int, start_key: bytes, end_key: bytes) -> tuple{int AS level, int AS node_id, int AS store_id, int AS snapshot_pinned_keys, int AS snapshot_pinned_keys_bytes, int AS point_key_delete_is_latest_count, int AS point_key_delete_count, int AS point_key_set_is_latest_count, int AS point_key_set_count, int AS range_delete_count, int AS range_key_set_count, int AS range_key_delete_count}`,
	2471: `crdb_internal.scan_storage_internal_keys(node_id: int, store_id: int, start_key: bytes, end_key: bytes, mb_per_second: int4) -> tuple{int AS level, int AS node_id, int AS store_id, int AS snapshot_pinned_keys, int AS snapshot_pinned_keys_bytes, int AS point_key_delete_is_latest_count, int AS point_key_delete_count, int AS point_key_set_is_latest_count, int AS point_key_set_count, int AS range_delete_count, int AS range_key_set_count, int AS range_key_delete_count}`,
	2472: `bitmask_or(a: varbit, b: varbit) -> varbit`,
	2473: `bitmask_or(a: string, b: string) -> varbit`,
	2474: `bitmask_or(a: varbit, b: string) -> varbit`,
	2475: `bitmask_or(a: string, b: varbit) -> varbit`,
	2476: `bitmask_and(a: varbit, b: varbit) -> varbit`,
	2477: `bitmask_and(a: string, b: string) -> varbit`,
	2478: `bitmask_and(a: varbit, b: string) -> varbit`,
	2479: `bitmask_and(a: string, b: varbit) -> varbit`,
	2480: `bitmask_xor(a: varbit, b: varbit) -> varbit`,
	2481: `bitmask_xor(a: string, b: string) -> varbit`,
	2482: `bitmask_xor(a: varbit, b: string) -> varbit`,
	2483: `bitmask_xor(a: string, b: varbit) -> varbit`,
	2484: `oidvectortypes(vector: oidvector) -> string`,
	2485: `encrypt(data: bytes, key: bytes, type: string) -> bytes`,
	2486: `encrypt_iv(data: bytes, key: bytes, iv: bytes, type: string) -> bytes`,
	2487: `decrypt(data: bytes, key: bytes, type: string) -> bytes`,
	2488: `decrypt_iv(data: bytes, key: bytes, iv: bytes, type: string) -> bytes`,
	2489: `gen_random_bytes(count: int) -> bytes`,
	2490: `make_timestamp(year: int, month: int, day: int, hour: int, min: int, sec: float) -> timestamp`,
	2491: `make_timestamptz(year: int, month: int, day: int, hour: int, min: int, sec: float) -> timestamptz`,
	2492: `make_timestamptz(year: int, month: int, day: int, hour: int, min: int, sec: float, timezone: string) -> timestamptz`,
	2493: `date_trunc(element: string, input: timestamptz, timezone: string) -> timestamptz`,
	2494: `make_date(year: int, month: int, day: int) -> date`,
	2495: `crdb_internal.plpgsql_gen_cursor_name(name: refcursor) -> refcursor`,
	2496: `crdb_internal.plpgsql_close(name: refcursor) -> int`,
	2497: `crdb_internal.plpgsql_fetch(name: refcursor, direction: int, count: int, resultTypes: anyelement) -> anyelement`,
	2498: `refcursorin(input: anyelement) -> refcursor`,
	2499: `refcursorout(refcursor: refcursor) -> bytes`,
	2500: `refcursorsend(refcursor: refcursor) -> bytes`,
	2501: `refcursorrecv(input: anyelement) -> refcursor`,
	2502: `refcursor(refcursor: refcursor) -> refcursor`,
	2503: `refcursor(string: string) -> refcursor`,
	2504: `bpchar(refcursor: refcursor) -> bpchar`,
	2505: `char(refcursor: refcursor) -> "char"`,
	2506: `name(refcursor: refcursor) -> name`,
	2507: `text(refcursor: refcursor) -> string`,
	2508: `varchar(refcursor: refcursor) -> varchar`,
	2509: `crdb_internal.unsafe_revert_tenant_to_timestamp(tenant_name: string, ts: decimal) -> decimal`,
	2510: `crdb_internal.reset_insights_tables() -> bool`,
	2511: `merge_stats_metadata(arg1: jsonb) -> jsonb`,
	2512: `merge_statement_stats(arg1: jsonb) -> jsonb`,
	2513: `merge_transaction_stats(arg1: jsonb) -> jsonb`,
	2514: `crdb_internal.start_replication_stream(tenant_name: string, spec: bytes) -> bytes`,
	2515: `crdb_internal.privilege_name(internal_key: string) -> string`,
	2516: `crdb_internal.privilege_name(internal_key: string[]) -> string[]`,
	2517: `jsonb_array_to_string_array(input: jsonb) -> string[]`,
	2518: `array_cat_agg(arg1: refcursor[]) -> refcursor[]`,
	2519: `array_agg(arg1: refcursor) -> refcursor[]`,
	2520: `array_prepend(elem: refcursor, array: refcursor[]) -> refcursor[]`,
	2521: `array_remove(array: refcursor[], elem: refcursor) -> anyelement`,
	2522: `array_positions(array: refcursor[], elem: refcursor) -> int[]`,
	2523: `array_cat(left: refcursor[], right: refcursor[]) -> refcursor[]`,
	2524: `array_position(array: refcursor[], elem: refcursor) -> int`,
	2525: `array_replace(array: refcursor[], toreplace: refcursor, replacewith: refcursor) -> anyelement`,
	2526: `array_append(array: refcursor[], elem: refcursor) -> refcursor[]`,
	2527: `first_value(val: refcursor) -> refcursor`,
	2528: `nth_value(val: refcursor, n: int) -> refcursor`,
	2529: `lag(val: refcursor) -> refcursor`,
	2530: `lag(val: refcursor, n: int) -> refcursor`,
	2531: `lag(val: refcursor, n: int, default: refcursor) -> refcursor`,
	2532: `lead(val: refcursor) -> refcursor`,
	2533: `lead(val: refcursor, n: int) -> refcursor`,
	2534: `lead(val: refcursor, n: int, default: refcursor) -> refcursor`,
	2535: `last_value(val: refcursor) -> refcursor`,
	2536: `percentile_disc_impl(arg1: float, arg2: refcursor) -> refcursor`,
	2537: `percentile_disc_impl(arg1: float[], arg2: refcursor) -> refcursor[]`,
	2538: `information_schema._pg_char_octet_length(typid: oid, typmod: int4) -> int`,
	2539: `pg_encoding_max_length(encoding: int) -> int`,
	2540: `information_schema._pg_datetime_precision(typid: oid, typmod: int4) -> int`,
	2541: `information_schema._pg_interval_type(typid: oid, typmod: int4) -> string`,
	2542: `crdb_internal.release_series(version: string) -> string`,
	2543: `crdb_internal.fips_ready() -> bool`,
	2544: `st_tileenvelope(tileZoom: int4, tileX: int4, tileY: int4) -> geometry`,
	2545: `st_tileenvelope(tileZoom: int4, tileX: int4, tileY: int4, bounds: geometry) -> geometry`,
	2546: `st_tileenvelope(tileZoom: int4, tileX: int4, tileY: int4, bounds: geometry, margin: float) -> geometry`,
	2547: `array_position(array: bool[], elem: bool, start: int) -> int`,
	2548: `array_position(array: box2d[], elem: box2d, start: int) -> int`,
	2549: `array_position(array: int[], elem: int, start: int) -> int`,
	2550: `array_position(array: float[], elem: float, start: int) -> int`,
	2551: `array_position(array: decimal[], elem: decimal, start: int) -> int`,
	2552: `array_position(array: date[], elem: date, start: int) -> int`,
	2553: `array_position(array: timestamp[], elem: timestamp, start: int) -> int`,
	2554: `array_position(array: interval[], elem: interval, start: int) -> int`,
	2555: `array_position(array: geography[], elem: geography, start: int) -> int`,
	2556: `array_position(array: geometry[], elem: geometry, start: int) -> int`,
	2557: `array_position(array: string[], elem: string, start: int) -> int`,
	2558: `array_position(array: bytes[], elem: bytes, start: int) -> int`,
	2559: `array_position(array: timestamptz[], elem: timestamptz, start: int) -> int`,
	2560: `array_position(array: oid[], elem: oid, start: int) -> int`,
	2561: `array_position(array: uuid[], elem: uuid, start: int) -> int`,
	2562: `array_position(array: inet[], elem: inet, start: int) -> int`,
	2563: `array_position(array: time[], elem: time, start: int) -> int`,
	2564: `array_position(array: timetz[], elem: timetz, start: int) -> int`,
	2565: `array_position(array: jsonb[], elem: jsonb, start: int) -> int`,
	2566: `array_position(array: varbit[], elem: varbit, start: int) -> int`,
	2567: `array_position(array: anyenum[], elem: anyenum, start: int) -> int`,
	2568: `array_position(array: tuple[], elem: tuple, start: int) -> int`,
	2569: `array_position(array: pg_lsn[], elem: pg_lsn, start: int) -> int`,
	2570: `array_position(array: refcursor[], elem: refcursor, start: int) -> int`,
	2571: `bit_count(val: bytes) -> int`,
	2572: `bit_count(val: varbit) -> int`,
	2573: `metaphone(source: string, max_output_length: int) -> string`,
	2574: `array_agg(arg1: bool[]) -> bool[][]`,
	2575: `array_agg(arg1: box2d[]) -> box2d[][]`,
	2576: `array_agg(arg1: int[]) -> int[][]`,
	2577: `array_agg(arg1: float[]) -> float[][]`,
	2578: `array_agg(arg1: decimal[]) -> decimal[][]`,
	2579: `array_agg(arg1: date[]) -> date[][]`,
	2580: `array_agg(arg1: timestamp[]) -> timestamp[][]`,
	2581: `array_agg(arg1: interval[]) -> interval[][]`,
	2582: `array_agg(arg1: geography[]) -> geography[][]`,
	2583: `array_agg(arg1: geometry[]) -> geometry[][]`,
	2584: `array_agg(arg1: string[]) -> string[][]`,
	2585: `array_agg(arg1: bytes[]) -> bytes[][]`,
	2586: `array_agg(arg1: timestamptz[]) -> timestamptz[][]`,
	2587: `array_agg(arg1: oid[]) -> oid[][]`,
	2588: `array_agg(arg1: uuid[]) -> uuid[][]`,
	2589: `array_agg(arg1: inet[]) -> inet[][]`,
	2590: `array_agg(arg1: time[]) -> time[][]`,
	2591: `array_agg(arg1: timetz[]) -> timetz[][]`,
	2592: `array_agg(arg1: jsonb[]) -> jsonb[][]`,
	2593: `array_agg(arg1: varbit[]) -> varbit[][]`,
	2594: `array_agg(arg1: anyenum[]) -> anyenum[][]`,
	2595: `array_agg(arg1: pg_lsn[]) -> pg_lsn[][]`,
	2596: `array_agg(arg1: refcursor[]) -> refcursor[][]`,
	2597: `array_agg(arg1: tuple[]) -> tuple[][]`,
	2598: `setseed(seed: float) -> void`,
	2599: `crdb_internal.execute_internally(query: string) -> string`,
	2600: `crdb_internal.execute_internally(query: string, session_bound: bool) -> string`,
	2601: `crdb_internal.execute_internally(query: string, overrides: string) -> string`,
	2602: `crdb_internal.execute_internally(query: string, session_bound: bool, overrides: string) -> string`,
	2603: `crdb_internal.execute_internally(query: string, overrides: string, use_session_txn: bool) -> string`,
	2604: `crdb_internal.execute_internally(query: string, session_bound: bool, overrides: string, use_session_txn: bool) -> string`,
	2605: `merge_aggregated_stmt_metadata(arg1: jsonb) -> jsonb`,
	2606: `crdb_internal.protect_mvcc_history(timestamp: decimal, expiration_window: interval, description: string) -> int`,
	2607: `crdb_internal.extend_mvcc_history_protection(job_id: int) -> void`,
	2608: `crdb_internal.force_panic(msg: string, mode: string) -> int`,
	2609: `crdb_internal.request_statement_bundle(stmtFingerprint: string, samplingProbability: float, minExecutionLatency: interval, expiresAfter: interval, redacted: bool) -> bool`,
	2610: `crdb_internal.request_statement_bundle(stmtFingerprint: string, planGist: string, samplingProbability: float, minExecutionLatency: interval, expiresAfter: interval, redacted: bool) -> bool`,
	2611: `crdb_internal.request_statement_bundle(stmtFingerprint: string, planGist: string, antiPlanGist: bool, samplingProbability: float, minExecutionLatency: interval, expiresAfter: interval, redacted: bool) -> bool`,
	2612: `crdb_internal.index_span(tenant_id: int, table_id: int, index_id: int) -> bytes[]`,
	2613: `crdb_internal.split_at(key: bytes, ttl: interval) -> void`,
	2614: `crdb_internal.scatter(key: bytes) -> void`,
	2615: `crdb_internal.scatter(key: bytes, end_key: bytes) -> void`,
	2616: `crdb_internal.plan_logical_replication(req: bytes) -> bytes`,
	2617: `crdb_internal.start_replication_stream_for_tables(req: bytes) -> bytes`,
	2618: `crdb_internal.logical_replication_inject_failures(stream: int, proc: int, percent: int) -> void`,
	2619: `vectorsend(vector: vector) -> bytes`,
	2620: `vectorrecv(input: anyelement) -> vector`,
	2621: `vectorout(vector: vector) -> bytes`,
	2622: `vectorin(input: anyelement) -> vector`,
	2623: `char(vector: vector) -> "char"`,
	2624: `name(vector: vector) -> name`,
	2625: `text(vector: vector) -> string`,
	2626: `varchar(vector: vector) -> varchar`,
	2627: `bpchar(vector: vector) -> bpchar`,
	2628: `vector(string: string) -> vector`,
	2629: `vector(vector: vector) -> vector`,
	2630: `cosine_distance(v1: vector, v2: vector) -> float`,
	2631: `l1_distance(v1: vector, v2: vector) -> float`,
	2632: `l2_distance(v1: vector, v2: vector) -> float`,
	2633: `inner_product(v1: vector, v2: vector) -> float`,
	2634: `vector_dims(vector: vector) -> int`,
	2635: `vector_norm(vector: vector) -> float`,
	2636: `crdb_internal.log(msg: string) -> void`,
	2637: `crdb_internal.start_logical_replication_job(conn_str: string, table_names: string[]) -> int`,
	2638: `crdb_internal.plan_logical_replication(req: bytes) -> bytes`,
	2639: `crdb_internal.start_replication_stream_for_tables(req: bytes) -> bytes`,
	2640: `crdb_internal.clear_query_plan_cache() -> void`,
	2641: `crdb_internal.clear_table_stats_cache() -> void`,
	2642: `crdb_internal.get_fully_qualified_table_name(table_descriptor_id: int) -> string`,
	2643: `crdb_internal.type_is_indexable(oid: oid) -> bool`,
	2644: `crdb_internal.range_stats_with_errors(key: bytes) -> jsonb`,
	2645: `crdb_internal.lease_holder_with_errors(key: bytes) -> jsonb`,
	2646: `crdb_internal.pretty_key(raw_key: bytes) -> string`,
}

var builtinOidsBySignature map[string]oid.Oid

func signatureMustHaveHardcodedOID(sig string) oid.Oid {
	maybeInitializeBuiltinMap()
	oid, ok := builtinOidsBySignature[sig]
	if !ok {
		panic(fmt.Sprintf("Missing an entry for %s in builtins.builtinOidsArray. "+
			"Update the signature there if it's changed, or if it's new, add the following entry to the bottom of the list:"+
			"%d: `%s`", sig, nextUnusedOid(), sig))
	}
	return oid
}

func oidsMustBeUnique() error {
	sigsByOid := make(map[oid.Oid][]string, len(builtinOidsBySignature))
	for n, oid := range builtinOidsBySignature {
		sigsByOid[oid] = append(sigsByOid[oid], n)
	}
	for oid, sigs := range sigsByOid {
		if len(sigs) > 1 {
			return errors.Newf("Multiple signatures have oid %d: %v", oid, sigs)
		}
	}
	return nil
}

// nextUnusedOid is used to construct the error message when a signature is missing
// from the map. Note that it doesn't check for collisions elsewhere.
func nextUnusedOid() oid.Oid {
	var max oid.Oid
	for _, oid := range builtinOidsBySignature {
		if oid > max {
			max = oid
		}
	}
	return max + 1
}

func maybeInitializeBuiltinMap() {
	if builtinOidsBySignature != nil {
		return
	}
	builtinOidsBySignature = make(map[string]oid.Oid, len(builtinOidsArray))
	for i, val := range builtinOidsArray {
		if i == 0 || val == "" {
			continue
		}
		builtinOidsBySignature[val] = oid.Oid(i)
	}
	if err := oidsMustBeUnique(); err != nil {
		panic(err)
	}
}

func init() {
	maybeInitializeBuiltinMap()
}
