// Code generated by execgen; DO NOT EDIT.

// Copyright 2018 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package colexec

import (
	"bytes"
	"math"
	"time"

	"github.com/cockroachdb/apd/v3"
	"github.com/cockroachdb/cockroach/pkg/col/coldata"
	"github.com/cockroachdb/cockroach/pkg/col/coldataext"
	"github.com/cockroachdb/cockroach/pkg/col/typeconv"
	"github.com/cockroachdb/cockroach/pkg/sql/colexecerror"
	"github.com/cockroachdb/cockroach/pkg/sql/sem/tree"
	"github.com/cockroachdb/cockroach/pkg/sql/types"
	"github.com/cockroachdb/cockroach/pkg/util/duration"
	"github.com/cockroachdb/cockroach/pkg/util/json"
	"github.com/cockroachdb/errors"
)

// partitioner is a simple implementation of sorted distinct that's useful for
// other operators that need to partition an arbitrarily-sized Vec.
type partitioner interface {
	// partition partitions the input colVec of size n, writing true to the
	// outputCol for every value that differs from the previous one.
	partition(colVec *coldata.Vec, outputCol []bool, n int)

	// partitionWithOrder is like partition, except it performs the partitioning
	// on the input Vec as if it were ordered via the input order vector, which is
	// a selection vector. The output is written in absolute order, however. For
	// example, with an input vector [a,b,b] and an order vector [1,2,0], which
	// implies a reordered input vector [b,b,a], the resultant outputCol would be
	// [true, false, true], indicating a distinct value at the 0th and 2nd
	// elements.
	partitionWithOrder(colVec *coldata.Vec, order []int, outputCol []bool, n int)
}

// newPartitioner returns a new partitioner on type t.
func newPartitioner(t *types.T, nullsAreDistinct bool) partitioner {
	switch typeconv.TypeFamilyToCanonicalTypeFamily(t.Family()) {
	case types.BoolFamily:
		switch t.Width() {
		case -1:
		default:
			return partitionerBool{nullsAreDistinct: nullsAreDistinct}
		}
	case types.BytesFamily:
		switch t.Width() {
		case -1:
		default:
			return partitionerBytes{nullsAreDistinct: nullsAreDistinct}
		}
	case types.DecimalFamily:
		switch t.Width() {
		case -1:
		default:
			return partitionerDecimal{nullsAreDistinct: nullsAreDistinct}
		}
	case types.IntFamily:
		switch t.Width() {
		case 16:
			return partitionerInt16{nullsAreDistinct: nullsAreDistinct}
		case 32:
			return partitionerInt32{nullsAreDistinct: nullsAreDistinct}
		case -1:
		default:
			return partitionerInt64{nullsAreDistinct: nullsAreDistinct}
		}
	case types.FloatFamily:
		switch t.Width() {
		case -1:
		default:
			return partitionerFloat64{nullsAreDistinct: nullsAreDistinct}
		}
	case types.TimestampTZFamily:
		switch t.Width() {
		case -1:
		default:
			return partitionerTimestamp{nullsAreDistinct: nullsAreDistinct}
		}
	case types.IntervalFamily:
		switch t.Width() {
		case -1:
		default:
			return partitionerInterval{nullsAreDistinct: nullsAreDistinct}
		}
	case types.JsonFamily:
		switch t.Width() {
		case -1:
		default:
			return partitionerJSON{nullsAreDistinct: nullsAreDistinct}
		}
	case typeconv.DatumVecCanonicalTypeFamily:
		switch t.Width() {
		case -1:
		default:
			return partitionerDatum{nullsAreDistinct: nullsAreDistinct}
		}
	}
	colexecerror.InternalError(errors.AssertionFailedf("unsupported type %s", t))
	// This code is unreachable, but the compiler cannot infer that.
	return nil
}

// partitionerBool partitions an arbitrary-length colVec by running a distinct
// operation over it. It writes the same format to outputCol that sorted
// distinct does: true for every row that differs from the previous row in the
// input column.
type partitionerBool struct {
	nullsAreDistinct bool
}

func (p partitionerBool) partitionWithOrder(
	colVec *coldata.Vec, order []int, outputCol []bool, n int,
) {
	var lastVal bool
	var lastValNull bool
	var nulls *coldata.Nulls
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Bool()
	// Eliminate bounds checks.
	_ = outputCol[n-1]
	_ = order[n-1]
	outputCol[0] = true
	if nulls != nil {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var (
					__retval_lastVal     bool
					__retval_lastValNull bool
				)
				{
					var nullsAreDistinct bool = p.nullsAreDistinct
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								if !v && lastVal {
									cmpResult = -1
								} else if v && !lastVal {
									cmpResult = 1
								} else {
									cmpResult = 0
								}

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var __retval_0 bool
				{
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						if !v && lastVal {
							cmpResult = -1
						} else if v && !lastVal {
							cmpResult = 1
						} else {
							cmpResult = 0
						}

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

func (p partitionerBool) partition(colVec *coldata.Vec, outputCol []bool, n int) {
	var (
		lastVal     bool
		lastValNull bool
		nulls       *coldata.Nulls
	)
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Bool()
	_ = col.Get(n - 1)
	_ = outputCol[n-1]
	outputCol[0] = true
	if nulls != nil {
		for idx := 0; idx < n; idx++ {
			{
				var (
					__retval_lastVal     bool
					__retval_lastValNull bool
				)
				{
					var (
						checkIdx         int  = idx
						outputIdx        int  = idx
						nullsAreDistinct bool = p.nullsAreDistinct
					)
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						//gcassert:bce
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								if !v && lastVal {
									cmpResult = -1
								} else if v && !lastVal {
									cmpResult = 1
								} else {
									cmpResult = 0
								}

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for idx := 0; idx < n; idx++ {
			{
				var __retval_0 bool
				{
					var (
						checkIdx  int = idx
						outputIdx int = idx
					)
					//gcassert:bce
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						if !v && lastVal {
							cmpResult = -1
						} else if v && !lastVal {
							cmpResult = 1
						} else {
							cmpResult = 0
						}

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

// partitionerBytes partitions an arbitrary-length colVec by running a distinct
// operation over it. It writes the same format to outputCol that sorted
// distinct does: true for every row that differs from the previous row in the
// input column.
type partitionerBytes struct {
	nullsAreDistinct bool
}

func (p partitionerBytes) partitionWithOrder(
	colVec *coldata.Vec, order []int, outputCol []bool, n int,
) {
	var lastVal []byte
	var lastValNull bool
	var nulls *coldata.Nulls
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Bytes()
	// Eliminate bounds checks.
	_ = outputCol[n-1]
	_ = order[n-1]
	outputCol[0] = true
	if nulls != nil {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var (
					__retval_lastVal     []byte
					__retval_lastValNull bool
				)
				{
					var nullsAreDistinct bool = p.nullsAreDistinct
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int
								cmpResult = bytes.Compare(v, lastVal)
								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var __retval_0 []byte
				{
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int
						cmpResult = bytes.Compare(v, lastVal)
						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

func (p partitionerBytes) partition(colVec *coldata.Vec, outputCol []bool, n int) {
	var (
		lastVal     []byte
		lastValNull bool
		nulls       *coldata.Nulls
	)
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Bytes()
	_ = outputCol[n-1]
	outputCol[0] = true
	if nulls != nil {
		for idx := 0; idx < n; idx++ {
			{
				var (
					__retval_lastVal     []byte
					__retval_lastValNull bool
				)
				{
					var (
						checkIdx         int  = idx
						outputIdx        int  = idx
						nullsAreDistinct bool = p.nullsAreDistinct
					)
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int
								cmpResult = bytes.Compare(v, lastVal)
								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for idx := 0; idx < n; idx++ {
			{
				var __retval_0 []byte
				{
					var (
						checkIdx  int = idx
						outputIdx int = idx
					)
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int
						cmpResult = bytes.Compare(v, lastVal)
						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

// partitionerDecimal partitions an arbitrary-length colVec by running a distinct
// operation over it. It writes the same format to outputCol that sorted
// distinct does: true for every row that differs from the previous row in the
// input column.
type partitionerDecimal struct {
	nullsAreDistinct bool
}

func (p partitionerDecimal) partitionWithOrder(
	colVec *coldata.Vec, order []int, outputCol []bool, n int,
) {
	var lastVal apd.Decimal
	var lastValNull bool
	var nulls *coldata.Nulls
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Decimal()
	// Eliminate bounds checks.
	_ = outputCol[n-1]
	_ = order[n-1]
	outputCol[0] = true
	if nulls != nil {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var (
					__retval_lastVal     apd.Decimal
					__retval_lastValNull bool
				)
				{
					var nullsAreDistinct bool = p.nullsAreDistinct
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int
								cmpResult = tree.CompareDecimals(&v, &lastVal)
								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var __retval_0 apd.Decimal
				{
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int
						cmpResult = tree.CompareDecimals(&v, &lastVal)
						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

func (p partitionerDecimal) partition(colVec *coldata.Vec, outputCol []bool, n int) {
	var (
		lastVal     apd.Decimal
		lastValNull bool
		nulls       *coldata.Nulls
	)
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Decimal()
	_ = col.Get(n - 1)
	_ = outputCol[n-1]
	outputCol[0] = true
	if nulls != nil {
		for idx := 0; idx < n; idx++ {
			{
				var (
					__retval_lastVal     apd.Decimal
					__retval_lastValNull bool
				)
				{
					var (
						checkIdx         int  = idx
						outputIdx        int  = idx
						nullsAreDistinct bool = p.nullsAreDistinct
					)
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						//gcassert:bce
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int
								cmpResult = tree.CompareDecimals(&v, &lastVal)
								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for idx := 0; idx < n; idx++ {
			{
				var __retval_0 apd.Decimal
				{
					var (
						checkIdx  int = idx
						outputIdx int = idx
					)
					//gcassert:bce
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int
						cmpResult = tree.CompareDecimals(&v, &lastVal)
						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

// partitionerInt16 partitions an arbitrary-length colVec by running a distinct
// operation over it. It writes the same format to outputCol that sorted
// distinct does: true for every row that differs from the previous row in the
// input column.
type partitionerInt16 struct {
	nullsAreDistinct bool
}

func (p partitionerInt16) partitionWithOrder(
	colVec *coldata.Vec, order []int, outputCol []bool, n int,
) {
	var lastVal int16
	var lastValNull bool
	var nulls *coldata.Nulls
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Int16()
	// Eliminate bounds checks.
	_ = outputCol[n-1]
	_ = order[n-1]
	outputCol[0] = true
	if nulls != nil {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var (
					__retval_lastVal     int16
					__retval_lastValNull bool
				)
				{
					var nullsAreDistinct bool = p.nullsAreDistinct
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								{
									a, b := int64(v), int64(lastVal)
									if a < b {
										cmpResult = -1
									} else if a > b {
										cmpResult = 1
									} else {
										cmpResult = 0
									}
								}

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var __retval_0 int16
				{
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						{
							a, b := int64(v), int64(lastVal)
							if a < b {
								cmpResult = -1
							} else if a > b {
								cmpResult = 1
							} else {
								cmpResult = 0
							}
						}

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

func (p partitionerInt16) partition(colVec *coldata.Vec, outputCol []bool, n int) {
	var (
		lastVal     int16
		lastValNull bool
		nulls       *coldata.Nulls
	)
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Int16()
	_ = col.Get(n - 1)
	_ = outputCol[n-1]
	outputCol[0] = true
	if nulls != nil {
		for idx := 0; idx < n; idx++ {
			{
				var (
					__retval_lastVal     int16
					__retval_lastValNull bool
				)
				{
					var (
						checkIdx         int  = idx
						outputIdx        int  = idx
						nullsAreDistinct bool = p.nullsAreDistinct
					)
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						//gcassert:bce
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								{
									a, b := int64(v), int64(lastVal)
									if a < b {
										cmpResult = -1
									} else if a > b {
										cmpResult = 1
									} else {
										cmpResult = 0
									}
								}

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for idx := 0; idx < n; idx++ {
			{
				var __retval_0 int16
				{
					var (
						checkIdx  int = idx
						outputIdx int = idx
					)
					//gcassert:bce
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						{
							a, b := int64(v), int64(lastVal)
							if a < b {
								cmpResult = -1
							} else if a > b {
								cmpResult = 1
							} else {
								cmpResult = 0
							}
						}

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

// partitionerInt32 partitions an arbitrary-length colVec by running a distinct
// operation over it. It writes the same format to outputCol that sorted
// distinct does: true for every row that differs from the previous row in the
// input column.
type partitionerInt32 struct {
	nullsAreDistinct bool
}

func (p partitionerInt32) partitionWithOrder(
	colVec *coldata.Vec, order []int, outputCol []bool, n int,
) {
	var lastVal int32
	var lastValNull bool
	var nulls *coldata.Nulls
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Int32()
	// Eliminate bounds checks.
	_ = outputCol[n-1]
	_ = order[n-1]
	outputCol[0] = true
	if nulls != nil {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var (
					__retval_lastVal     int32
					__retval_lastValNull bool
				)
				{
					var nullsAreDistinct bool = p.nullsAreDistinct
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								{
									a, b := int64(v), int64(lastVal)
									if a < b {
										cmpResult = -1
									} else if a > b {
										cmpResult = 1
									} else {
										cmpResult = 0
									}
								}

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var __retval_0 int32
				{
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						{
							a, b := int64(v), int64(lastVal)
							if a < b {
								cmpResult = -1
							} else if a > b {
								cmpResult = 1
							} else {
								cmpResult = 0
							}
						}

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

func (p partitionerInt32) partition(colVec *coldata.Vec, outputCol []bool, n int) {
	var (
		lastVal     int32
		lastValNull bool
		nulls       *coldata.Nulls
	)
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Int32()
	_ = col.Get(n - 1)
	_ = outputCol[n-1]
	outputCol[0] = true
	if nulls != nil {
		for idx := 0; idx < n; idx++ {
			{
				var (
					__retval_lastVal     int32
					__retval_lastValNull bool
				)
				{
					var (
						checkIdx         int  = idx
						outputIdx        int  = idx
						nullsAreDistinct bool = p.nullsAreDistinct
					)
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						//gcassert:bce
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								{
									a, b := int64(v), int64(lastVal)
									if a < b {
										cmpResult = -1
									} else if a > b {
										cmpResult = 1
									} else {
										cmpResult = 0
									}
								}

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for idx := 0; idx < n; idx++ {
			{
				var __retval_0 int32
				{
					var (
						checkIdx  int = idx
						outputIdx int = idx
					)
					//gcassert:bce
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						{
							a, b := int64(v), int64(lastVal)
							if a < b {
								cmpResult = -1
							} else if a > b {
								cmpResult = 1
							} else {
								cmpResult = 0
							}
						}

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

// partitionerInt64 partitions an arbitrary-length colVec by running a distinct
// operation over it. It writes the same format to outputCol that sorted
// distinct does: true for every row that differs from the previous row in the
// input column.
type partitionerInt64 struct {
	nullsAreDistinct bool
}

func (p partitionerInt64) partitionWithOrder(
	colVec *coldata.Vec, order []int, outputCol []bool, n int,
) {
	var lastVal int64
	var lastValNull bool
	var nulls *coldata.Nulls
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Int64()
	// Eliminate bounds checks.
	_ = outputCol[n-1]
	_ = order[n-1]
	outputCol[0] = true
	if nulls != nil {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var (
					__retval_lastVal     int64
					__retval_lastValNull bool
				)
				{
					var nullsAreDistinct bool = p.nullsAreDistinct
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								{
									a, b := int64(v), int64(lastVal)
									if a < b {
										cmpResult = -1
									} else if a > b {
										cmpResult = 1
									} else {
										cmpResult = 0
									}
								}

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var __retval_0 int64
				{
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						{
							a, b := int64(v), int64(lastVal)
							if a < b {
								cmpResult = -1
							} else if a > b {
								cmpResult = 1
							} else {
								cmpResult = 0
							}
						}

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

func (p partitionerInt64) partition(colVec *coldata.Vec, outputCol []bool, n int) {
	var (
		lastVal     int64
		lastValNull bool
		nulls       *coldata.Nulls
	)
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Int64()
	_ = col.Get(n - 1)
	_ = outputCol[n-1]
	outputCol[0] = true
	if nulls != nil {
		for idx := 0; idx < n; idx++ {
			{
				var (
					__retval_lastVal     int64
					__retval_lastValNull bool
				)
				{
					var (
						checkIdx         int  = idx
						outputIdx        int  = idx
						nullsAreDistinct bool = p.nullsAreDistinct
					)
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						//gcassert:bce
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								{
									a, b := int64(v), int64(lastVal)
									if a < b {
										cmpResult = -1
									} else if a > b {
										cmpResult = 1
									} else {
										cmpResult = 0
									}
								}

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for idx := 0; idx < n; idx++ {
			{
				var __retval_0 int64
				{
					var (
						checkIdx  int = idx
						outputIdx int = idx
					)
					//gcassert:bce
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						{
							a, b := int64(v), int64(lastVal)
							if a < b {
								cmpResult = -1
							} else if a > b {
								cmpResult = 1
							} else {
								cmpResult = 0
							}
						}

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

// partitionerFloat64 partitions an arbitrary-length colVec by running a distinct
// operation over it. It writes the same format to outputCol that sorted
// distinct does: true for every row that differs from the previous row in the
// input column.
type partitionerFloat64 struct {
	nullsAreDistinct bool
}

func (p partitionerFloat64) partitionWithOrder(
	colVec *coldata.Vec, order []int, outputCol []bool, n int,
) {
	var lastVal float64
	var lastValNull bool
	var nulls *coldata.Nulls
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Float64()
	// Eliminate bounds checks.
	_ = outputCol[n-1]
	_ = order[n-1]
	outputCol[0] = true
	if nulls != nil {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var (
					__retval_lastVal     float64
					__retval_lastValNull bool
				)
				{
					var nullsAreDistinct bool = p.nullsAreDistinct
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								{
									a, b := float64(v), float64(lastVal)
									if a < b {
										cmpResult = -1
									} else if a > b {
										cmpResult = 1
									} else if a == b {
										cmpResult = 0
									} else if math.IsNaN(a) {
										if math.IsNaN(b) {
											cmpResult = 0
										} else {
											cmpResult = -1
										}
									} else {
										cmpResult = 1
									}
								}

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var __retval_0 float64
				{
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						{
							a, b := float64(v), float64(lastVal)
							if a < b {
								cmpResult = -1
							} else if a > b {
								cmpResult = 1
							} else if a == b {
								cmpResult = 0
							} else if math.IsNaN(a) {
								if math.IsNaN(b) {
									cmpResult = 0
								} else {
									cmpResult = -1
								}
							} else {
								cmpResult = 1
							}
						}

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

func (p partitionerFloat64) partition(colVec *coldata.Vec, outputCol []bool, n int) {
	var (
		lastVal     float64
		lastValNull bool
		nulls       *coldata.Nulls
	)
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Float64()
	_ = col.Get(n - 1)
	_ = outputCol[n-1]
	outputCol[0] = true
	if nulls != nil {
		for idx := 0; idx < n; idx++ {
			{
				var (
					__retval_lastVal     float64
					__retval_lastValNull bool
				)
				{
					var (
						checkIdx         int  = idx
						outputIdx        int  = idx
						nullsAreDistinct bool = p.nullsAreDistinct
					)
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						//gcassert:bce
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								{
									a, b := float64(v), float64(lastVal)
									if a < b {
										cmpResult = -1
									} else if a > b {
										cmpResult = 1
									} else if a == b {
										cmpResult = 0
									} else if math.IsNaN(a) {
										if math.IsNaN(b) {
											cmpResult = 0
										} else {
											cmpResult = -1
										}
									} else {
										cmpResult = 1
									}
								}

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for idx := 0; idx < n; idx++ {
			{
				var __retval_0 float64
				{
					var (
						checkIdx  int = idx
						outputIdx int = idx
					)
					//gcassert:bce
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						{
							a, b := float64(v), float64(lastVal)
							if a < b {
								cmpResult = -1
							} else if a > b {
								cmpResult = 1
							} else if a == b {
								cmpResult = 0
							} else if math.IsNaN(a) {
								if math.IsNaN(b) {
									cmpResult = 0
								} else {
									cmpResult = -1
								}
							} else {
								cmpResult = 1
							}
						}

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

// partitionerTimestamp partitions an arbitrary-length colVec by running a distinct
// operation over it. It writes the same format to outputCol that sorted
// distinct does: true for every row that differs from the previous row in the
// input column.
type partitionerTimestamp struct {
	nullsAreDistinct bool
}

func (p partitionerTimestamp) partitionWithOrder(
	colVec *coldata.Vec, order []int, outputCol []bool, n int,
) {
	var lastVal time.Time
	var lastValNull bool
	var nulls *coldata.Nulls
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Timestamp()
	// Eliminate bounds checks.
	_ = outputCol[n-1]
	_ = order[n-1]
	outputCol[0] = true
	if nulls != nil {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var (
					__retval_lastVal     time.Time
					__retval_lastValNull bool
				)
				{
					var nullsAreDistinct bool = p.nullsAreDistinct
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								if v.Before(lastVal) {
									cmpResult = -1
								} else if lastVal.Before(v) {
									cmpResult = 1
								} else {
									cmpResult = 0
								}
								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var __retval_0 time.Time
				{
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						if v.Before(lastVal) {
							cmpResult = -1
						} else if lastVal.Before(v) {
							cmpResult = 1
						} else {
							cmpResult = 0
						}
						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

func (p partitionerTimestamp) partition(colVec *coldata.Vec, outputCol []bool, n int) {
	var (
		lastVal     time.Time
		lastValNull bool
		nulls       *coldata.Nulls
	)
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Timestamp()
	_ = col.Get(n - 1)
	_ = outputCol[n-1]
	outputCol[0] = true
	if nulls != nil {
		for idx := 0; idx < n; idx++ {
			{
				var (
					__retval_lastVal     time.Time
					__retval_lastValNull bool
				)
				{
					var (
						checkIdx         int  = idx
						outputIdx        int  = idx
						nullsAreDistinct bool = p.nullsAreDistinct
					)
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						//gcassert:bce
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								if v.Before(lastVal) {
									cmpResult = -1
								} else if lastVal.Before(v) {
									cmpResult = 1
								} else {
									cmpResult = 0
								}
								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for idx := 0; idx < n; idx++ {
			{
				var __retval_0 time.Time
				{
					var (
						checkIdx  int = idx
						outputIdx int = idx
					)
					//gcassert:bce
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						if v.Before(lastVal) {
							cmpResult = -1
						} else if lastVal.Before(v) {
							cmpResult = 1
						} else {
							cmpResult = 0
						}
						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

// partitionerInterval partitions an arbitrary-length colVec by running a distinct
// operation over it. It writes the same format to outputCol that sorted
// distinct does: true for every row that differs from the previous row in the
// input column.
type partitionerInterval struct {
	nullsAreDistinct bool
}

func (p partitionerInterval) partitionWithOrder(
	colVec *coldata.Vec, order []int, outputCol []bool, n int,
) {
	var lastVal duration.Duration
	var lastValNull bool
	var nulls *coldata.Nulls
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Interval()
	// Eliminate bounds checks.
	_ = outputCol[n-1]
	_ = order[n-1]
	outputCol[0] = true
	if nulls != nil {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var (
					__retval_lastVal     duration.Duration
					__retval_lastValNull bool
				)
				{
					var nullsAreDistinct bool = p.nullsAreDistinct
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int
								cmpResult = v.Compare(lastVal)
								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var __retval_0 duration.Duration
				{
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int
						cmpResult = v.Compare(lastVal)
						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

func (p partitionerInterval) partition(colVec *coldata.Vec, outputCol []bool, n int) {
	var (
		lastVal     duration.Duration
		lastValNull bool
		nulls       *coldata.Nulls
	)
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Interval()
	_ = col.Get(n - 1)
	_ = outputCol[n-1]
	outputCol[0] = true
	if nulls != nil {
		for idx := 0; idx < n; idx++ {
			{
				var (
					__retval_lastVal     duration.Duration
					__retval_lastValNull bool
				)
				{
					var (
						checkIdx         int  = idx
						outputIdx        int  = idx
						nullsAreDistinct bool = p.nullsAreDistinct
					)
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						//gcassert:bce
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int
								cmpResult = v.Compare(lastVal)
								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for idx := 0; idx < n; idx++ {
			{
				var __retval_0 duration.Duration
				{
					var (
						checkIdx  int = idx
						outputIdx int = idx
					)
					//gcassert:bce
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int
						cmpResult = v.Compare(lastVal)
						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

// partitionerJSON partitions an arbitrary-length colVec by running a distinct
// operation over it. It writes the same format to outputCol that sorted
// distinct does: true for every row that differs from the previous row in the
// input column.
type partitionerJSON struct {
	nullsAreDistinct bool
}

func (p partitionerJSON) partitionWithOrder(
	colVec *coldata.Vec, order []int, outputCol []bool, n int,
) {
	var lastVal json.JSON
	var lastValNull bool
	var nulls *coldata.Nulls
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.JSON()
	// Eliminate bounds checks.
	_ = outputCol[n-1]
	_ = order[n-1]
	outputCol[0] = true
	if nulls != nil {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var (
					__retval_lastVal     json.JSON
					__retval_lastValNull bool
				)
				{
					var nullsAreDistinct bool = p.nullsAreDistinct
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								var err error
								cmpResult, err = v.Compare(lastVal)
								if err != nil {
									colexecerror.ExpectedError(err)
								}

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var __retval_0 json.JSON
				{
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						var err error
						cmpResult, err = v.Compare(lastVal)
						if err != nil {
							colexecerror.ExpectedError(err)
						}

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

func (p partitionerJSON) partition(colVec *coldata.Vec, outputCol []bool, n int) {
	var (
		lastVal     json.JSON
		lastValNull bool
		nulls       *coldata.Nulls
	)
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.JSON()
	_ = outputCol[n-1]
	outputCol[0] = true
	if nulls != nil {
		for idx := 0; idx < n; idx++ {
			{
				var (
					__retval_lastVal     json.JSON
					__retval_lastValNull bool
				)
				{
					var (
						checkIdx         int  = idx
						outputIdx        int  = idx
						nullsAreDistinct bool = p.nullsAreDistinct
					)
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								var err error
								cmpResult, err = v.Compare(lastVal)
								if err != nil {
									colexecerror.ExpectedError(err)
								}

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for idx := 0; idx < n; idx++ {
			{
				var __retval_0 json.JSON
				{
					var (
						checkIdx  int = idx
						outputIdx int = idx
					)
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						var err error
						cmpResult, err = v.Compare(lastVal)
						if err != nil {
							colexecerror.ExpectedError(err)
						}

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

// partitionerDatum partitions an arbitrary-length colVec by running a distinct
// operation over it. It writes the same format to outputCol that sorted
// distinct does: true for every row that differs from the previous row in the
// input column.
type partitionerDatum struct {
	nullsAreDistinct bool
}

func (p partitionerDatum) partitionWithOrder(
	colVec *coldata.Vec, order []int, outputCol []bool, n int,
) {
	var lastVal interface{}
	var lastValNull bool
	var nulls *coldata.Nulls
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Datum()
	// Eliminate bounds checks.
	_ = outputCol[n-1]
	_ = order[n-1]
	outputCol[0] = true
	if nulls != nil {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var (
					__retval_lastVal     interface{}
					__retval_lastValNull bool
				)
				{
					var nullsAreDistinct bool = p.nullsAreDistinct
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								cmpResult = coldataext.CompareDatum(v, col, lastVal)

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for outputIdx := 0; outputIdx < n; outputIdx++ {
			//gcassert:bce
			checkIdx := order[outputIdx]
			{
				var __retval_0 interface{}
				{
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						cmpResult = coldataext.CompareDatum(v, col, lastVal)

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}

func (p partitionerDatum) partition(colVec *coldata.Vec, outputCol []bool, n int) {
	var (
		lastVal     interface{}
		lastValNull bool
		nulls       *coldata.Nulls
	)
	if colVec.MaybeHasNulls() {
		nulls = colVec.Nulls()
	}

	col := colVec.Datum()
	_ = outputCol[n-1]
	outputCol[0] = true
	if nulls != nil {
		for idx := 0; idx < n; idx++ {
			{
				var (
					__retval_lastVal     interface{}
					__retval_lastValNull bool
				)
				{
					var (
						checkIdx         int  = idx
						outputIdx        int  = idx
						nullsAreDistinct bool = p.nullsAreDistinct
					)
					null := nulls.NullAt(checkIdx)
					if null {
						if !lastValNull || nullsAreDistinct {
							// The current value is null, and either the previous one is not
							// (meaning they are definitely distinct) or we treat nulls as
							// distinct values.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						}
					} else {
						v := col.Get(checkIdx)
						if lastValNull {
							// The previous value was null while the current is not.
							_ = true
							//gcassert:bce
							outputCol[outputIdx] = true
						} else {
							// Neither value is null, so we must compare.
							var unique bool

							{
								var cmpResult int

								cmpResult = coldataext.CompareDatum(v, col, lastVal)

								unique = cmpResult != 0
							}

							//gcassert:bce
							outputCol[outputIdx] = outputCol[outputIdx] || unique
						}
						lastVal = v
					}
					{
						__retval_lastVal = lastVal
						__retval_lastValNull = null
					}
				}
				lastVal, lastValNull = __retval_lastVal, __retval_lastValNull
			}
		}
	} else {
		for idx := 0; idx < n; idx++ {
			{
				var __retval_0 interface{}
				{
					var (
						checkIdx  int = idx
						outputIdx int = idx
					)
					v := col.Get(checkIdx)
					var unique bool

					{
						var cmpResult int

						cmpResult = coldataext.CompareDatum(v, col, lastVal)

						unique = cmpResult != 0
					}

					//gcassert:bce
					outputCol[outputIdx] = outputCol[outputIdx] || unique
					{
						__retval_0 = v
					}
				}
				lastVal = __retval_0
			}
		}
	}
}
