// Code generated by execgen; DO NOT EDIT.
// Copyright 2022 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

// Copyright 2022 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package colexec

import "math/bits"

// This file is copied from the Go standard library's sort
// implementation, found in https://golang.org/src/sort/sort.go. The only
// modifications are to template each function into each sort_* struct, so
// that the sort methods can be called without incurring any interface overhead
// for the Swap and Less methods.

type sortedHint int // hint for pdqsort when choosing the pivot

const (
	unknownHint sortedHint = iota
	increasingHint
	decreasingHint
)

// xorshift paper: https://www.jstatsoft.org/article/view/v008i14/xorshift.pdf
type xorshift uint64

func (r *xorshift) Next() uint64 {
	*r ^= *r << 13
	*r ^= *r >> 17
	*r ^= *r << 5
	return uint64(*r)
}

func nextPowerOfTwo(length int) uint {
	shift := uint(bits.Len(uint(length)))
	return uint(1 << shift)
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortBoolAscWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortBoolAscWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortBoolAscWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortBoolAscWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortBoolAscWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortBoolAscWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortBoolAscWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortBoolAscWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortBoolAscWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortBoolAscWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortBoolAscWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortBoolAscWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortBoolAscWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortBytesAscWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortBytesAscWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortBytesAscWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortBytesAscWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortBytesAscWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortBytesAscWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortBytesAscWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortBytesAscWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortBytesAscWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortBytesAscWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortBytesAscWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortBytesAscWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortBytesAscWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortDecimalAscWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortDecimalAscWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortDecimalAscWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortDecimalAscWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortDecimalAscWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortDecimalAscWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortDecimalAscWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortDecimalAscWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortDecimalAscWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortDecimalAscWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortDecimalAscWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortDecimalAscWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortDecimalAscWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortInt16AscWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortInt16AscWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortInt16AscWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortInt16AscWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortInt16AscWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortInt16AscWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortInt16AscWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortInt16AscWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortInt16AscWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortInt16AscWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortInt16AscWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortInt16AscWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortInt16AscWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortInt32AscWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortInt32AscWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortInt32AscWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortInt32AscWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortInt32AscWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortInt32AscWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortInt32AscWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortInt32AscWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortInt32AscWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortInt32AscWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortInt32AscWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortInt32AscWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortInt32AscWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortInt64AscWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortInt64AscWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortInt64AscWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortInt64AscWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortInt64AscWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortInt64AscWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortInt64AscWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortInt64AscWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortInt64AscWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortInt64AscWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortInt64AscWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortInt64AscWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortInt64AscWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortFloat64AscWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortFloat64AscWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortFloat64AscWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortFloat64AscWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortFloat64AscWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortFloat64AscWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortFloat64AscWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortFloat64AscWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortFloat64AscWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortFloat64AscWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortFloat64AscWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortFloat64AscWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortFloat64AscWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortTimestampAscWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortTimestampAscWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortTimestampAscWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortTimestampAscWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortTimestampAscWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortTimestampAscWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortTimestampAscWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortTimestampAscWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortTimestampAscWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortTimestampAscWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortTimestampAscWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortTimestampAscWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortTimestampAscWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortIntervalAscWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortIntervalAscWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortIntervalAscWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortIntervalAscWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortIntervalAscWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortIntervalAscWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortIntervalAscWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortIntervalAscWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortIntervalAscWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortIntervalAscWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortIntervalAscWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortIntervalAscWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortIntervalAscWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortJSONAscWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortJSONAscWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortJSONAscWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortJSONAscWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortJSONAscWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortJSONAscWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortJSONAscWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortJSONAscWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortJSONAscWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortJSONAscWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortJSONAscWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortJSONAscWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortJSONAscWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortDatumAscWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortDatumAscWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortDatumAscWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortDatumAscWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortDatumAscWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortDatumAscWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortDatumAscWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortDatumAscWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortDatumAscWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortDatumAscWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortDatumAscWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortDatumAscWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortDatumAscWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortBoolDescWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortBoolDescWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortBoolDescWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortBoolDescWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortBoolDescWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortBoolDescWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortBoolDescWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortBoolDescWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortBoolDescWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortBoolDescWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortBoolDescWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortBoolDescWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortBoolDescWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortBytesDescWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortBytesDescWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortBytesDescWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortBytesDescWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortBytesDescWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortBytesDescWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortBytesDescWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortBytesDescWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortBytesDescWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortBytesDescWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortBytesDescWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortBytesDescWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortBytesDescWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortDecimalDescWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortDecimalDescWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortDecimalDescWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortDecimalDescWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortDecimalDescWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortDecimalDescWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortDecimalDescWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortDecimalDescWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortDecimalDescWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortDecimalDescWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortDecimalDescWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortDecimalDescWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortDecimalDescWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortInt16DescWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortInt16DescWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortInt16DescWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortInt16DescWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortInt16DescWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortInt16DescWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortInt16DescWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortInt16DescWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortInt16DescWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortInt16DescWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortInt16DescWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortInt16DescWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortInt16DescWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortInt32DescWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortInt32DescWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortInt32DescWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortInt32DescWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortInt32DescWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortInt32DescWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortInt32DescWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortInt32DescWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortInt32DescWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortInt32DescWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortInt32DescWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortInt32DescWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortInt32DescWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortInt64DescWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortInt64DescWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortInt64DescWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortInt64DescWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortInt64DescWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortInt64DescWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortInt64DescWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortInt64DescWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortInt64DescWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortInt64DescWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortInt64DescWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortInt64DescWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortInt64DescWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortFloat64DescWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortFloat64DescWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortFloat64DescWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortFloat64DescWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortFloat64DescWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortFloat64DescWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortFloat64DescWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortFloat64DescWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortFloat64DescWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortFloat64DescWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortFloat64DescWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortFloat64DescWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortFloat64DescWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortTimestampDescWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortTimestampDescWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortTimestampDescWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortTimestampDescWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortTimestampDescWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortTimestampDescWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortTimestampDescWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortTimestampDescWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortTimestampDescWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortTimestampDescWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortTimestampDescWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortTimestampDescWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortTimestampDescWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortIntervalDescWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortIntervalDescWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortIntervalDescWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortIntervalDescWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortIntervalDescWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortIntervalDescWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortIntervalDescWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortIntervalDescWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortIntervalDescWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortIntervalDescWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortIntervalDescWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortIntervalDescWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortIntervalDescWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortJSONDescWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortJSONDescWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortJSONDescWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortJSONDescWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortJSONDescWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortJSONDescWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortJSONDescWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortJSONDescWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortJSONDescWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortJSONDescWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortJSONDescWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortJSONDescWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortJSONDescWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortDatumDescWithNullsOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortDatumDescWithNullsOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortDatumDescWithNullsOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortDatumDescWithNullsOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortDatumDescWithNullsOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortDatumDescWithNullsOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortDatumDescWithNullsOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortDatumDescWithNullsOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortDatumDescWithNullsOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortDatumDescWithNullsOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortDatumDescWithNullsOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortDatumDescWithNullsOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortDatumDescWithNullsOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortBoolAscOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortBoolAscOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortBoolAscOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortBoolAscOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortBoolAscOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortBoolAscOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortBoolAscOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortBoolAscOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortBoolAscOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortBoolAscOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortBoolAscOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortBoolAscOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortBoolAscOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortBytesAscOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortBytesAscOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortBytesAscOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortBytesAscOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortBytesAscOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortBytesAscOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortBytesAscOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortBytesAscOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortBytesAscOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortBytesAscOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortBytesAscOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortBytesAscOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortBytesAscOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortDecimalAscOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortDecimalAscOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortDecimalAscOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortDecimalAscOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortDecimalAscOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortDecimalAscOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortDecimalAscOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortDecimalAscOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortDecimalAscOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortDecimalAscOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortDecimalAscOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortDecimalAscOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortDecimalAscOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortInt16AscOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortInt16AscOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortInt16AscOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortInt16AscOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortInt16AscOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortInt16AscOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortInt16AscOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortInt16AscOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortInt16AscOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortInt16AscOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortInt16AscOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortInt16AscOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortInt16AscOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortInt32AscOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortInt32AscOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortInt32AscOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortInt32AscOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortInt32AscOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortInt32AscOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortInt32AscOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortInt32AscOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortInt32AscOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortInt32AscOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortInt32AscOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortInt32AscOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortInt32AscOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortInt64AscOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortInt64AscOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortInt64AscOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortInt64AscOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortInt64AscOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortInt64AscOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortInt64AscOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortInt64AscOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortInt64AscOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortInt64AscOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortInt64AscOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortInt64AscOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortInt64AscOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortFloat64AscOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortFloat64AscOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortFloat64AscOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortFloat64AscOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortFloat64AscOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortFloat64AscOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortFloat64AscOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortFloat64AscOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortFloat64AscOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortFloat64AscOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortFloat64AscOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortFloat64AscOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortFloat64AscOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortTimestampAscOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortTimestampAscOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortTimestampAscOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortTimestampAscOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortTimestampAscOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortTimestampAscOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortTimestampAscOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortTimestampAscOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortTimestampAscOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortTimestampAscOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortTimestampAscOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortTimestampAscOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortTimestampAscOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortIntervalAscOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortIntervalAscOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortIntervalAscOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortIntervalAscOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortIntervalAscOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortIntervalAscOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortIntervalAscOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortIntervalAscOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortIntervalAscOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortIntervalAscOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortIntervalAscOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortIntervalAscOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortIntervalAscOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortJSONAscOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortJSONAscOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortJSONAscOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortJSONAscOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortJSONAscOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortJSONAscOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortJSONAscOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortJSONAscOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortJSONAscOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortJSONAscOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortJSONAscOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortJSONAscOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortJSONAscOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortDatumAscOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortDatumAscOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortDatumAscOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortDatumAscOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortDatumAscOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortDatumAscOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortDatumAscOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortDatumAscOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortDatumAscOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortDatumAscOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortDatumAscOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortDatumAscOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortDatumAscOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortBoolDescOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortBoolDescOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortBoolDescOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortBoolDescOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortBoolDescOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortBoolDescOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortBoolDescOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortBoolDescOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortBoolDescOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortBoolDescOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortBoolDescOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortBoolDescOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortBoolDescOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortBytesDescOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortBytesDescOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortBytesDescOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortBytesDescOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortBytesDescOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortBytesDescOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortBytesDescOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortBytesDescOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortBytesDescOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortBytesDescOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortBytesDescOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortBytesDescOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortBytesDescOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortDecimalDescOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortDecimalDescOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortDecimalDescOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortDecimalDescOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortDecimalDescOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortDecimalDescOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortDecimalDescOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortDecimalDescOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortDecimalDescOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortDecimalDescOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortDecimalDescOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortDecimalDescOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortDecimalDescOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortInt16DescOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortInt16DescOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortInt16DescOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortInt16DescOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortInt16DescOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortInt16DescOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortInt16DescOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortInt16DescOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortInt16DescOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortInt16DescOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortInt16DescOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortInt16DescOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortInt16DescOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortInt32DescOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortInt32DescOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortInt32DescOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortInt32DescOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortInt32DescOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortInt32DescOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortInt32DescOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortInt32DescOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortInt32DescOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortInt32DescOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortInt32DescOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortInt32DescOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortInt32DescOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortInt64DescOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortInt64DescOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortInt64DescOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortInt64DescOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortInt64DescOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortInt64DescOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortInt64DescOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortInt64DescOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortInt64DescOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortInt64DescOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortInt64DescOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortInt64DescOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortInt64DescOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortFloat64DescOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortFloat64DescOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortFloat64DescOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortFloat64DescOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortFloat64DescOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortFloat64DescOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortFloat64DescOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortFloat64DescOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortFloat64DescOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortFloat64DescOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortFloat64DescOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortFloat64DescOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortFloat64DescOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortTimestampDescOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortTimestampDescOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortTimestampDescOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortTimestampDescOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortTimestampDescOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortTimestampDescOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortTimestampDescOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortTimestampDescOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortTimestampDescOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortTimestampDescOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortTimestampDescOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortTimestampDescOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortTimestampDescOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortIntervalDescOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortIntervalDescOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortIntervalDescOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortIntervalDescOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortIntervalDescOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortIntervalDescOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortIntervalDescOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortIntervalDescOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortIntervalDescOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortIntervalDescOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortIntervalDescOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortIntervalDescOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortIntervalDescOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortJSONDescOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortJSONDescOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortJSONDescOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortJSONDescOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortJSONDescOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortJSONDescOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortJSONDescOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortJSONDescOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortJSONDescOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortJSONDescOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortJSONDescOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortJSONDescOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortJSONDescOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}

// insertionSort sorts data[a:b] using insertion sort.
func (p *sortDatumDescOp) insertionSort(a, b int) {
	for i := a + 1; i < b; i++ {
		for j := i; j > a && p.Less(j, j-1); j-- {
			p.Swap(j, j-1)
		}
	}
}

// siftDown implements the heap property on data[lo:hi].
// first is an offset into the array where the root of the heap lies.
func (p *sortDatumDescOp) siftDown(lo, hi, first int) {
	root := lo
	for {
		child := 2*root + 1
		if child >= hi {
			break
		}
		if child+1 < hi && p.Less(first+child, first+child+1) {
			child++
		}
		if !p.Less(first+root, first+child) {
			return
		}
		p.Swap(first+root, first+child)
		root = child
	}
}

func (p *sortDatumDescOp) heapSort(a, b int) {
	first := a
	lo := 0
	hi := b - a

	// Build heap with greatest element at top.
	for i := (hi - 1) / 2; i >= 0; i-- {
		p.cancelChecker.Check()
		p.siftDown(i, hi, first)
	}

	// Pop elements, largest first, into end of data.
	for i := hi - 1; i >= 0; i-- {
		p.Swap(first, first+i)
		p.siftDown(lo, i, first)
	}
}

// pdqsort sorts data[a:b].
// The algorithm based on pattern-defeating quicksort(pdqsort), but without the optimizations from BlockQuicksort.
// pdqsort paper: https://arxiv.org/pdf/2106.05123.pdf
// C++ implementation: https://github.com/orlp/pdqsort
// Rust implementation: https://docs.rs/pdqsort/latest/pdqsort/
// limit is the number of allowed bad (very unbalanced) pivots before falling back to heapsort.
func (p *sortDatumDescOp) pdqsort(a, b, limit int) {
	const maxInsertion = 12

	var (
		wasBalanced    = true // whether the last partitioning was reasonably balanced
		wasPartitioned = true // whether the slice was already partitioned
	)

	for {
		length := b - a

		if length <= maxInsertion {
			p.insertionSort(a, b)
			return
		}

		// Fall back to heapsort if too many bad choices were made.
		if limit == 0 {
			p.heapSort(a, b)
			return
		}

		// If the last partitioning was imbalanced, we need to breaking patterns.
		if !wasBalanced {
			p.breakPatterns(a, b)
			limit--
		}

		p.cancelChecker.Check()
		pivot, hint := p.choosePivot(a, b)
		if hint == decreasingHint {
			p.reverseRange(a, b)
			// The chosen pivot was pivot-a elements after the start of the array.
			// After reversing it is pivot-a elements before the end of the array.
			// The idea came from Rust's implementation.
			pivot = (b - 1) - (pivot - a)
			hint = increasingHint
		}

		// The slice is likely already sorted.
		if wasBalanced && wasPartitioned && hint == increasingHint {
			if p.partialInsertionSort(a, b) {
				return
			}
		}

		// Probably the slice contains many duplicate elements, partition the slice into
		// elements equal to and elements greater than the pivot.
		if a > 0 && !p.Less(a-1, pivot) {
			mid := p.partitionEqual(a, b, pivot)
			a = mid
			continue
		}

		mid, alreadyPartitioned := p.partition(a, b, pivot)
		wasPartitioned = alreadyPartitioned

		leftLen, rightLen := mid-a, b-mid
		balanceThreshold := length / 8
		if leftLen < rightLen {
			wasBalanced = leftLen >= balanceThreshold
			p.pdqsort(a, mid, limit)
			a = mid + 1
		} else {
			wasBalanced = rightLen >= balanceThreshold
			p.pdqsort(mid+1, b, limit)
			b = mid
		}
	}
}

// partition does one quicksort partition.
// Let p = data[pivot]
// Moves elements in data[a:b] around, so that data[i]<p and data[j]>=p for i<newpivot and j>newpivot.
// On return, data[newpivot] = p
func (p *sortDatumDescOp) partition(
	a, b, pivot int,
) (newpivot int, alreadyPartitioned bool) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for i <= j && p.Less(i, a) {
		i++
	}
	for i <= j && !p.Less(j, a) {
		j--
	}
	if i > j {
		p.Swap(j, a)
		return j, true
	}
	p.Swap(i, j)
	i++
	j--

	for {
		for i <= j && p.Less(i, a) {
			i++
		}
		for i <= j && !p.Less(j, a) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	p.Swap(j, a)
	return j, false
}

// partitionEqual partitions data[a:b] into elements equal to data[pivot] followed by elements greater than data[pivot].
// It assumed that data[a:b] does not contain elements smaller than the data[pivot].
func (p *sortDatumDescOp) partitionEqual(a, b, pivot int) (newpivot int) {
	p.Swap(a, pivot)
	i, j := a+1, b-1 // i and j are inclusive of the elements remaining to be partitioned

	for {
		for i <= j && !p.Less(a, i) {
			i++
		}
		for i <= j && p.Less(a, j) {
			j--
		}
		if i > j {
			break
		}
		p.Swap(i, j)
		i++
		j--
	}
	return i
}

// partialInsertionSort partially sorts a slice, returns true if the slice is sorted at the end.
func (p *sortDatumDescOp) partialInsertionSort(a, b int) bool {
	const (
		maxSteps         = 5  // maximum number of adjacent out-of-order pairs that will get shifted
		shortestShifting = 50 // don't shift any elements on short arrays
	)
	i := a + 1
	for j := 0; j < maxSteps; j++ {
		for i < b && !p.Less(i, i-1) {
			i++
		}

		if i == b {
			return true
		}

		if b-a < shortestShifting {
			return false
		}

		p.Swap(i, i-1)

		// Shift the smaller one to the left.
		if i-a >= 2 {
			for j := i - 1; j >= 1; j-- {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
		// Shift the greater one to the right.
		if b-i >= 2 {
			for j := i + 1; j < b; j++ {
				if !p.Less(j, j-1) {
					break
				}
				p.Swap(j, j-1)
			}
		}
	}
	return false
}

// breakPatterns scatters some elements around in an attempt to break some patterns
// that might cause imbalanced partitions in quicksort.
func (p *sortDatumDescOp) breakPatterns(a, b int) {
	length := b - a
	if length >= 8 {
		random := xorshift(length)
		modulus := nextPowerOfTwo(length)

		for idx := a + (length/4)*2 - 1; idx <= a+(length/4)*2+1; idx++ {
			other := int(uint(random.Next()) & (modulus - 1))
			if other >= length {
				other -= length
			}
			p.Swap(idx, a+other)
		}
	}
}

// choosePivot chooses a pivot in data[a:b].
//
// [0,8): chooses a static pivot.
// [8,shortestNinther): uses the simple median-of-three method.
// [shortestNinther,∞): uses the Tukey ninther method.
func (p *sortDatumDescOp) choosePivot(a, b int) (pivot int, hint sortedHint) {
	const (
		shortestNinther = 50
		maxSwaps        = 4 * 3
	)

	l := b - a

	var (
		swaps int
		i     = a + l/4*1
		j     = a + l/4*2
		k     = a + l/4*3
	)

	if l >= 8 {
		if l >= shortestNinther {
			// Tukey ninther method, the idea came from Rust's implementation.
			i = p.medianAdjacent(i, &swaps)
			j = p.medianAdjacent(j, &swaps)
			k = p.medianAdjacent(k, &swaps)
		}
		// Find the median among i, j, k and stores it into j.
		j = p.median(i, j, k, &swaps)
	}

	switch swaps {
	case 0:
		return j, increasingHint
	case maxSwaps:
		return j, decreasingHint
	default:
		return j, unknownHint
	}
}

// order2 returns x,y where data[x] <= data[y], where x,y=a,b or x,y=b,a.
func (p *sortDatumDescOp) order2(a, b int, swaps *int) (int, int) {
	if p.Less(b, a) {
		*swaps++
		return b, a
	}
	return a, b
}

// median returns x where data[x] is the median of data[a],data[b],data[c], where x is a, b, or c.
func (p *sortDatumDescOp) median(a, b, c int, swaps *int) int {
	a, b = p.order2(a, b, swaps)
	b, _ = p.order2(b, c, swaps)
	_, b = p.order2(a, b, swaps)
	return b
}

// medianAdjacent finds the median of data[a - 1], data[a], data[a + 1] and stores the index into a.
func (p *sortDatumDescOp) medianAdjacent(a int, swaps *int) int {
	return p.median(a-1, a, a+1, swaps)
}

func (p *sortDatumDescOp) reverseRange(a, b int) {
	i := a
	j := b - 1
	for i < j {
		p.Swap(i, j)
		i++
		j--
	}
}
