// Code generated by execgen; DO NOT EDIT.
// Copyright 2020 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package colexec

import (
	"github.com/cockroachdb/cockroach/pkg/col/coldata"
	"github.com/cockroachdb/cockroach/pkg/sql/colexec/colexecutils"
	"github.com/cockroachdb/cockroach/pkg/sql/colexecop"
	"github.com/cockroachdb/cockroach/pkg/sql/colmem"
	"github.com/cockroachdb/cockroach/pkg/sql/sem/tree"
	"github.com/cockroachdb/cockroach/pkg/sql/types"
)

type isNullProjBase struct {
	colexecop.OneInputHelper
	allocator *colmem.Allocator
	colIdx    int
	outputIdx int
	negate    bool
}

// NewIsNullProjOp returns a new isNullProjOp.
// - negate indicates whether a negative version is used (we either have IS NOT
// NULL or IS DISTINCT FROM).
// - isTupleNull indicates whether special "is tuple null" version is needed
// (we either have IS NULL or IS NOT NULL with tuple type as the input vector).
func NewIsNullProjOp(
	allocator *colmem.Allocator,
	input colexecop.Operator,
	colIdx, outputIdx int,
	negate bool,
	isTupleNull bool,
) colexecop.Operator {
	input = colexecutils.NewVectorTypeEnforcer(allocator, input, types.Bool, outputIdx)
	base := isNullProjBase{
		OneInputHelper: colexecop.MakeOneInputHelper(input),
		allocator:      allocator,
		colIdx:         colIdx,
		outputIdx:      outputIdx,
		negate:         negate,
	}
	if isTupleNull {
		return &isTupleNullProjOp{isNullProjBase: base}
	}
	return &isNullProjOp{isNullProjBase: base}
}

// isNullProjOp is an Operator that projects into outputIdx Vec whether
// the corresponding value in colIdx Vec is NULL (i.e. it performs IS NULL
// check). If negate is true, it does the opposite - it performs IS NOT NULL
// check. Tuples require special attention, so we have a separate struct for
// them.
type isNullProjOp struct {
	isNullProjBase
}

var _ colexecop.Operator = &isNullProjOp{}

func (o *isNullProjOp) Next() coldata.Batch {
	batch := o.Input.Next()
	n := batch.Length()
	if n == 0 {
		return coldata.ZeroBatch
	}
	vec := batch.ColVec(o.colIdx)
	nulls := vec.Nulls()
	projVec := batch.ColVec(o.outputIdx)
	projCol := projVec.Bool()
	if nulls.MaybeHasNulls() {
		if sel := batch.Selection(); sel != nil {
			sel = sel[:n]
			for _, i := range sel {
				projCol[i] = nulls.NullAt(i) != o.negate

			}
		} else {
			projCol = projCol[:n]
			for i := range projCol {
				projCol[i] = nulls.NullAt(i) != o.negate

			}
		}
	} else {
		if sel := batch.Selection(); sel != nil {
			sel = sel[:n]
			for _, i := range sel {
				// There are no NULLs, so we don't need to check each index for nullity.
				projCol[i] = o.negate

			}
		} else {
			projCol = projCol[:n]
			for i := range projCol {
				// There are no NULLs, so we don't need to check each index for nullity.
				projCol[i] = o.negate

			}
		}
	}
	return batch
}

// isTupleNullProjOp is an Operator that projects into outputIdx Vec whether
// the corresponding value in colIdx Vec is NULL (i.e. it performs IS NULL
// check). If negate is true, it does the opposite - it performs IS NOT NULL
// check. Tuples require special attention, so we have a separate struct for
// them.
type isTupleNullProjOp struct {
	isNullProjBase
}

var _ colexecop.Operator = &isTupleNullProjOp{}

func (o *isTupleNullProjOp) Next() coldata.Batch {
	batch := o.Input.Next()
	n := batch.Length()
	if n == 0 {
		return coldata.ZeroBatch
	}
	vec := batch.ColVec(o.colIdx)
	nulls := vec.Nulls()
	datums := vec.Datum()
	projVec := batch.ColVec(o.outputIdx)
	projCol := projVec.Bool()
	if nulls.MaybeHasNulls() {
		if sel := batch.Selection(); sel != nil {
			sel = sel[:n]
			for _, i := range sel {
				if nulls.NullAt(i) {
					projCol[i] = !o.negate
				} else {
					projCol[i] = isTupleNull(datums.Get(i).(tree.Datum), o.negate)
				}

			}
		} else {
			projCol = projCol[:n]
			for i := range projCol {
				if nulls.NullAt(i) {
					projCol[i] = !o.negate
				} else {
					projCol[i] = isTupleNull(datums.Get(i).(tree.Datum), o.negate)
				}

			}
		}
	} else {
		if sel := batch.Selection(); sel != nil {
			sel = sel[:n]
			for _, i := range sel {
				if nulls.NullAt(i) {
					projCol[i] = !o.negate
				} else {
					projCol[i] = isTupleNull(datums.Get(i).(tree.Datum), o.negate)
				}

			}
		} else {
			projCol = projCol[:n]
			for i := range projCol {
				if nulls.NullAt(i) {
					projCol[i] = !o.negate
				} else {
					projCol[i] = isTupleNull(datums.Get(i).(tree.Datum), o.negate)
				}

			}
		}
	}
	return batch
}

type isNullSelBase struct {
	colexecop.OneInputHelper
	colIdx int
	negate bool
}

// NewIsNullSelOp returns a new isNullSelOp.
// - negate indicates whether a negative version is used (we either have IS NOT
// NULL or IS DISTINCT FROM).
// - isTupleNull indicates whether special "is tuple null" version is needed
// (we either have IS NULL or IS NOT NULL with tuple type as the input vector).
func NewIsNullSelOp(
	input colexecop.Operator, colIdx int, negate bool, isTupleNull bool,
) colexecop.Operator {
	base := isNullSelBase{
		OneInputHelper: colexecop.MakeOneInputHelper(input),
		colIdx:         colIdx,
		negate:         negate,
	}
	if isTupleNull {
		return &isTupleNullSelOp{isNullSelBase: base}
	}
	return &isNullSelOp{isNullSelBase: base}
}

// isNullSelOp is an Operator that selects all the tuples that have a NULL
// value in colIdx Vec. If negate is true, then it does the opposite -
// selecting all the tuples that have a non-NULL value in colIdx Vec.
type isNullSelOp struct {
	isNullSelBase
}

var _ colexecop.Operator = &isNullSelOp{}

func (o *isNullSelOp) Next() coldata.Batch {
	for {
		batch := o.Input.Next()
		n := batch.Length()
		if n == 0 {
			return batch
		}
		var idx int
		vec := batch.ColVec(o.colIdx)
		nulls := vec.Nulls()
		if nulls.MaybeHasNulls() {
			// There might be NULLs in the Vec, so we'll need to iterate over all
			// tuples.
			if sel := batch.Selection(); sel != nil {
				sel = sel[:n]
				for _, i := range sel {
					if nulls.NullAt(i) != o.negate {
						sel[idx] = i
						idx++
					}

				}
			} else {
				batch.SetSelection(true)
				sel := batch.Selection()[:n]
				for i := range sel {
					if nulls.NullAt(i) != o.negate {
						sel[idx] = i
						idx++
					}

				}
			}
			if idx > 0 {
				batch.SetLength(idx)
				return batch
			}
		} else {
			// There are no NULLs, so we don't need to check each index for nullity.
			if o.negate {
				// o.negate is true, so we select all tuples, i.e. we don't need to
				// modify the batch and can just return it.
				return batch
			}
			// o.negate is false, so we omit all tuples from this batch and move onto
			// the next one.
		}
	}
}

// isTupleNullSelOp is an Operator that selects all the tuples that have a NULL
// value in colIdx Vec. If negate is true, then it does the opposite -
// selecting all the tuples that have a non-NULL value in colIdx Vec.
type isTupleNullSelOp struct {
	isNullSelBase
}

var _ colexecop.Operator = &isTupleNullSelOp{}

func (o *isTupleNullSelOp) Next() coldata.Batch {
	for {
		batch := o.Input.Next()
		n := batch.Length()
		if n == 0 {
			return batch
		}
		var idx int
		vec := batch.ColVec(o.colIdx)
		nulls := vec.Nulls()
		datums := vec.Datum()
		// There might be NULLs in the Vec, so we'll need to iterate over all
		// tuples.
		if sel := batch.Selection(); sel != nil {
			sel = sel[:n]
			for _, i := range sel {
				var selectTuple bool
				if nulls.NullAt(i) {
					selectTuple = !o.negate
				} else {
					selectTuple = isTupleNull(datums.Get(i).(tree.Datum), o.negate)
				}
				if selectTuple {
					sel[idx] = i
					idx++
				}

			}
		} else {
			batch.SetSelection(true)
			sel := batch.Selection()[:n]
			for i := range sel {
				var selectTuple bool
				if nulls.NullAt(i) {
					selectTuple = !o.negate
				} else {
					selectTuple = isTupleNull(datums.Get(i).(tree.Datum), o.negate)
				}
				if selectTuple {
					sel[idx] = i
					idx++
				}

			}
		}
		if idx > 0 {
			batch.SetLength(idx)
			return batch
		}
	}
}

// isTupleNull returns the evaluation of IS [NOT] NULL predicate on datum
// (which is either a tree.DNull or tree.DTuple). negate differentiates between
// IS NULL and IS NOT NULL.
func isTupleNull(datum tree.Datum, negate bool) bool {
	if datum == tree.DNull {
		return !negate
	}
	// A tuple IS NULL if all elements are NULL.
	// A tuple IS NOT NULL if all elements are not NULL.
	for _, tupleDatum := range datum.(*tree.DTuple).D {
		datumIsNull := tupleDatum == tree.DNull
		if datumIsNull == negate {
			return false
		}
	}
	return true
}
