// Code generated by execgen; DO NOT EDIT.
// Copyright 2021 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package colexecwindow

import (
	"context"

	"github.com/cockroachdb/cockroach/pkg/sql/colexec/colexecutils"
	"github.com/cockroachdb/cockroach/pkg/sql/colexecerror"
	"github.com/cockroachdb/cockroach/pkg/sql/execinfrapb"
	"github.com/cockroachdb/cockroach/pkg/sql/sem/eval"
	"github.com/cockroachdb/cockroach/pkg/sql/sem/tree"
	"github.com/cockroachdb/cockroach/pkg/sql/types"
	"github.com/cockroachdb/errors"
)

// windowFramer is used to calculate the window frame for each row in a
// partition.
type windowFramer interface {
	// getColsToStore is called upon initialization of the windowFramer operators
	// in order to add to the list of columns to store in the SpillingBuffer owned
	// by the parent operator.
	// getColsToStore adds column indices to the given list when the windowFramer
	// operator needs access to values in a column for each partition (for
	// example, the peer groups column). If a column to be stored is already
	// present in the list, a duplicate entry will not be created.
	getColsToStore(oldColsToStore []int) (colsToStore []int)

	// startPartition is called before each partition is processed. It initializes
	// the windowFramer operators with the size of the partition and stored
	// columns. It also resets any fields used for window frame calculation. It
	// must be called before calling next.
	startPartition(ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer)

	// next should be called for each row in the partition. It advances to the
	// next row and then calculates the window frame for that row. (windowFramer
	// keeps track internally of the index of the current row starting from zero).
	// next should not be called beyond the end of the partition - otherwise,
	// undefined behavior will result.
	next(ctx context.Context)

	// frameFirstIdx returns the index of the first row in the window frame for
	// the current row. If no such row exists, frameFirstIdx returns -1.
	frameFirstIdx() int

	// frameLastIdx returns the index of the last row in the window frame for
	// the current row. If no such row exists, frameLastIdx returns -1. Note that
	// this index is inclusive so that operators that need the last index don't
	// need to have logic to detect when the frame is empty (say, because of the
	// exclusion clause).
	frameLastIdx() int

	// frameNthIdx returns the index of the nth row (starting from one to mirror
	// the nth_value window function) in the window frame for the current row. If
	// no such row exists, frameNthIdx returns -1.
	frameNthIdx(n int) int

	// frameIntervals returns a series of intervals that describes the set of all
	// rows that are part of the frame for the current row. Note that there are at
	// most three intervals - this case can occur when EXCLUDE TIES is used.
	// frameIntervals is used to compute aggregate functions over a window. The
	// returned intervals cannot be modified.
	frameIntervals() []windowInterval

	// slidingWindowIntervals returns a pair of interval sets that describes the
	// rows that should be added to the current aggregation, and those which
	// should be removed from the current aggregation. It is used to implement the
	// sliding window optimization for aggregate window functions. toAdd specifies
	// the rows that should be accumulated in the current aggregation, and
	// toRemove specifies those which should be removed.
	slidingWindowIntervals() (toAdd, toRemove []windowInterval)

	// close should always be called upon closing of the parent operator. It
	// releases all references to enable garbage collection.
	close()
}

func newWindowFramer(
	evalCtx *eval.Context,
	frame *execinfrapb.WindowerSpec_Frame,
	ordering *execinfrapb.Ordering,
	inputTypes []*types.T,
	peersColIdx int,
) windowFramer {
	startBound := frame.Bounds.Start
	endBound := frame.Bounds.End
	exclude := frame.Exclusion != execinfrapb.WindowerSpec_Frame_NO_EXCLUSION
	switch frame.Mode {
	case execinfrapb.WindowerSpec_Frame_ROWS:
		switch startBound.BoundType {
		case execinfrapb.WindowerSpec_Frame_UNBOUNDED_PRECEDING:
			switch endBound.BoundType {
			case execinfrapb.WindowerSpec_Frame_OFFSET_PRECEDING:
				switch exclude {
				case false:
					op := &windowFramerRowsUnboundedPrecedingOffsetPreceding{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRowsUnboundedPrecedingOffsetPrecedingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_CURRENT_ROW:
				switch exclude {
				case false:
					op := &windowFramerRowsUnboundedPrecedingCurrentRow{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRowsUnboundedPrecedingCurrentRowExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRowsUnboundedPrecedingOffsetFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRowsUnboundedPrecedingOffsetFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_UNBOUNDED_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRowsUnboundedPrecedingUnboundedFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRowsUnboundedPrecedingUnboundedFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			}
		case execinfrapb.WindowerSpec_Frame_OFFSET_PRECEDING:
			switch endBound.BoundType {
			case execinfrapb.WindowerSpec_Frame_OFFSET_PRECEDING:
				switch exclude {
				case false:
					op := &windowFramerRowsOffsetPrecedingOffsetPreceding{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRowsOffsetPrecedingOffsetPrecedingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_CURRENT_ROW:
				switch exclude {
				case false:
					op := &windowFramerRowsOffsetPrecedingCurrentRow{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRowsOffsetPrecedingCurrentRowExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRowsOffsetPrecedingOffsetFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRowsOffsetPrecedingOffsetFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_UNBOUNDED_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRowsOffsetPrecedingUnboundedFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRowsOffsetPrecedingUnboundedFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			}
		case execinfrapb.WindowerSpec_Frame_CURRENT_ROW:
			switch endBound.BoundType {
			case execinfrapb.WindowerSpec_Frame_CURRENT_ROW:
				switch exclude {
				case false:
					op := &windowFramerRowsCurrentRowCurrentRow{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRowsCurrentRowCurrentRowExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRowsCurrentRowOffsetFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRowsCurrentRowOffsetFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_UNBOUNDED_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRowsCurrentRowUnboundedFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRowsCurrentRowUnboundedFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			}
		case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
			switch endBound.BoundType {
			case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRowsOffsetFollowingOffsetFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRowsOffsetFollowingOffsetFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_UNBOUNDED_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRowsOffsetFollowingUnboundedFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRowsOffsetFollowingUnboundedFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			}
		}
	case execinfrapb.WindowerSpec_Frame_GROUPS:
		switch startBound.BoundType {
		case execinfrapb.WindowerSpec_Frame_UNBOUNDED_PRECEDING:
			switch endBound.BoundType {
			case execinfrapb.WindowerSpec_Frame_OFFSET_PRECEDING:
				switch exclude {
				case false:
					op := &windowFramerGroupsUnboundedPrecedingOffsetPreceding{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerGroupsUnboundedPrecedingOffsetPrecedingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_CURRENT_ROW:
				switch exclude {
				case false:
					op := &windowFramerGroupsUnboundedPrecedingCurrentRow{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerGroupsUnboundedPrecedingCurrentRowExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerGroupsUnboundedPrecedingOffsetFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerGroupsUnboundedPrecedingOffsetFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_UNBOUNDED_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerGroupsUnboundedPrecedingUnboundedFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerGroupsUnboundedPrecedingUnboundedFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			}
		case execinfrapb.WindowerSpec_Frame_OFFSET_PRECEDING:
			switch endBound.BoundType {
			case execinfrapb.WindowerSpec_Frame_OFFSET_PRECEDING:
				switch exclude {
				case false:
					op := &windowFramerGroupsOffsetPrecedingOffsetPreceding{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerGroupsOffsetPrecedingOffsetPrecedingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_CURRENT_ROW:
				switch exclude {
				case false:
					op := &windowFramerGroupsOffsetPrecedingCurrentRow{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerGroupsOffsetPrecedingCurrentRowExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerGroupsOffsetPrecedingOffsetFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerGroupsOffsetPrecedingOffsetFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_UNBOUNDED_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerGroupsOffsetPrecedingUnboundedFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerGroupsOffsetPrecedingUnboundedFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			}
		case execinfrapb.WindowerSpec_Frame_CURRENT_ROW:
			switch endBound.BoundType {
			case execinfrapb.WindowerSpec_Frame_CURRENT_ROW:
				switch exclude {
				case false:
					op := &windowFramerGroupsCurrentRowCurrentRow{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerGroupsCurrentRowCurrentRowExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerGroupsCurrentRowOffsetFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerGroupsCurrentRowOffsetFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_UNBOUNDED_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerGroupsCurrentRowUnboundedFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerGroupsCurrentRowUnboundedFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			}
		case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
			switch endBound.BoundType {
			case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerGroupsOffsetFollowingOffsetFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerGroupsOffsetFollowingOffsetFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_UNBOUNDED_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerGroupsOffsetFollowingUnboundedFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerGroupsOffsetFollowingUnboundedFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			}
		}
	case execinfrapb.WindowerSpec_Frame_RANGE:
		switch startBound.BoundType {
		case execinfrapb.WindowerSpec_Frame_UNBOUNDED_PRECEDING:
			switch endBound.BoundType {
			case execinfrapb.WindowerSpec_Frame_OFFSET_PRECEDING:
				switch exclude {
				case false:
					op := &windowFramerRangeUnboundedPrecedingOffsetPreceding{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRangeUnboundedPrecedingOffsetPrecedingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_CURRENT_ROW:
				switch exclude {
				case false:
					op := &windowFramerRangeUnboundedPrecedingCurrentRow{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRangeUnboundedPrecedingCurrentRowExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRangeUnboundedPrecedingOffsetFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRangeUnboundedPrecedingOffsetFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_UNBOUNDED_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRangeUnboundedPrecedingUnboundedFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRangeUnboundedPrecedingUnboundedFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			}
		case execinfrapb.WindowerSpec_Frame_OFFSET_PRECEDING:
			switch endBound.BoundType {
			case execinfrapb.WindowerSpec_Frame_OFFSET_PRECEDING:
				switch exclude {
				case false:
					op := &windowFramerRangeOffsetPrecedingOffsetPreceding{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRangeOffsetPrecedingOffsetPrecedingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_CURRENT_ROW:
				switch exclude {
				case false:
					op := &windowFramerRangeOffsetPrecedingCurrentRow{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRangeOffsetPrecedingCurrentRowExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRangeOffsetPrecedingOffsetFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRangeOffsetPrecedingOffsetFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_UNBOUNDED_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRangeOffsetPrecedingUnboundedFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRangeOffsetPrecedingUnboundedFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			}
		case execinfrapb.WindowerSpec_Frame_CURRENT_ROW:
			switch endBound.BoundType {
			case execinfrapb.WindowerSpec_Frame_CURRENT_ROW:
				switch exclude {
				case false:
					op := &windowFramerRangeCurrentRowCurrentRow{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRangeCurrentRowCurrentRowExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRangeCurrentRowOffsetFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRangeCurrentRowOffsetFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_UNBOUNDED_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRangeCurrentRowUnboundedFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRangeCurrentRowUnboundedFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			}
		case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
			switch endBound.BoundType {
			case execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRangeOffsetFollowingOffsetFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRangeOffsetFollowingOffsetFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			case execinfrapb.WindowerSpec_Frame_UNBOUNDED_FOLLOWING:
				switch exclude {
				case false:
					op := &windowFramerRangeOffsetFollowingUnboundedFollowing{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				case true:
					op := &windowFramerRangeOffsetFollowingUnboundedFollowingExclude{
						windowFramerBase: windowFramerBase{
							peersColIdx: peersColIdx,
							ordColIdx:   tree.NoColumnIdx,
							exclusion:   frame.Exclusion,
						},
					}
					op.handleOffsets(evalCtx, frame, ordering, inputTypes)
					return op
				}
			}
		}
	}
	colexecerror.InternalError(errors.AssertionFailedf("unsupported window framer type"))
	return nil
}

// windowFramerBase extracts common fields and methods for window framer
// operators. Window framer operators keep track of the start and end bounds of
// the window frame for the current row. The operators use the window frame for
// the previous row as a starting point for calculating the window frame for the
// current row. This takes advantage of the fact that window frame bounds are
// always non-decreasing from row to row.
//
// Consider the case of ROWS BETWEEN UNBOUNDED PRECEDING AND 3 FOLLOWING. The
// start of the frame will always be the first row, and the end will be three
// rows after the current one, which will always advance as the current row is
// advanced. Similar logic holds for the other framing modes and bounds.
// Excluded rows have the same non-decreasing property, and so are handled
// similarly.
type windowFramerBase struct {
	// storedCols stores the columns for which any row in the partition may be
	// accessed during evaluation of the window frame. The window framer operators
	// are not responsible for handling the SpillingBuffer other than to release
	// the reference after close is called.
	storedCols *colexecutils.SpillingBuffer

	partitionSize int // The total number of tuples in the partition.
	currentRow    int // The index of the current row.

	// currentGroup is the index of the current peer group, used by GROUPS mode.
	// It is only used when one or both bounds is of type OFFSET PRECEDING, to
	// ensure that the frame boundaries aren't advanced until the current row is
	// at least OFFSET groups beyond the start of the partition.
	currentGroup int

	startIdx int // Inclusive start of the window frame ignoring exclusion.
	endIdx   int // Exclusive end of the window frame ignoring exclusion.

	excludeStartIdx int // Inclusive start of rows specified by EXCLUDE clause.
	excludeEndIdx   int // Exclusive end of rows specified by EXCLUDE clause

	peersColIdx int // Indicates the beginning of each peer group. Can be unset.
	ordColIdx   int // The single ordering column for RANGE mode with offset.

	// startOffset and endOffset store the integer offsets for ROWS and GROUPS
	// modes with OFFSET PRECEDING or OFFSET FOLLOWING.
	startOffset, endOffset int

	// startHandler and endHandler are used to calculate the start and end indexes
	// in RANGE mode with OFFSET PRECEDING or OFFSET FOLLOWING.
	startHandler, endHandler rangeOffsetHandler

	// datumAlloc is used to decode the offsets in RANGE mode. It is initialized
	// lazily.
	datumAlloc *tree.DatumAlloc

	exclusion execinfrapb.WindowerSpec_Frame_Exclusion

	// intervals is a small (at most length 3) slice that is used during
	// aggregation computation.
	intervals       []windowInterval
	intervalsAreSet bool

	// prevIntervals, toAdd, and toRemove are used to calculate the intervals
	// for calculating aggregate window functions using the sliding window
	// optimization.
	prevIntervals, toAdd, toRemove []windowInterval
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (b *windowFramerBase) frameFirstIdx() (idx int) {
	if b.startIdx >= b.endIdx {
		// The window frame is empty, so there is no start index.
		return -1
	}
	return b.startIdx
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (b *windowFramerBase) frameLastIdx() (idx int) {
	if b.startIdx >= b.endIdx {
		// The window frame is empty, so there is no end index.
		return -1
	}
	return b.endIdx - 1
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (b *windowFramerBase) frameNthIdx(n int) (idx int) {
	if b.startIdx >= b.endIdx {
		// The window frame is empty, so there is no nth index.
		return -1
	}
	// Subtract from n to make it a zero-based index.
	n = n - 1
	idx = b.startIdx + n
	if idx < 0 || idx >= b.endIdx {
		// The requested index is out of range for this window frame.
		return -1
	}
	return idx
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (b *windowFramerBase) frameIntervals() []windowInterval {
	if b.intervalsAreSet {
		return b.intervals
	}
	b.intervalsAreSet = true
	b.intervals = b.intervals[:0]
	if b.startIdx >= b.endIdx {
		return b.intervals
	}
	b.intervals = append(b.intervals, windowInterval{start: b.startIdx, end: b.endIdx})
	return b.intervals
}

// getColsToStore appends to the given slice of column indices whatever columns
// to which the window framer will need access. getColsToStore also remaps the
// corresponding fields in the window framer to refer to ordinal positions
// within the colsToStore slice rather than within the input batches.
func (b *windowFramerBase) getColsToStore(oldColsToStore []int) (colsToStore []int) {
	colsToStore = oldColsToStore
	storeCol := func(colIdx int) int {
		for i := range colsToStore {
			if colsToStore[i] == colIdx {
				// The column is already present in colsToStore. Do not store any column
				// more than once.
				return i
			}
		}
		colsToStore = append(colsToStore, colIdx)
		return len(colsToStore) - 1
	}
	if b.peersColIdx != tree.NoColumnIdx {
		b.peersColIdx = storeCol(b.peersColIdx)
	}
	if b.ordColIdx != tree.NoColumnIdx {
		b.ordColIdx = storeCol(b.ordColIdx)
	}
	return colsToStore
}

// startPartition is called before each partition is processed. It initializes
// the windowFramer operators with the size of the partition and stored
// columns. It also resets any fields used for window frame calculation.
func (b *windowFramerBase) startPartition(
	partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	// Initialize currentRow to -1 to ensure that the first call to next advances
	// currentRow to zero.
	b.currentRow = -1
	b.partitionSize = partitionSize
	b.storedCols = storedCols
	b.startIdx = 0
	b.endIdx = 0
	b.intervals = b.intervals[:0]
	b.prevIntervals = b.prevIntervals[:0]
	b.toAdd = b.toAdd[:0]
	b.toRemove = b.toRemove[:0]
}

// incrementPeerGroup increments the given index by 'groups' peer groups,
// returning the resulting index. If the given offset is greater than the
// remaining number of groups, the returned index will be equal to the size of
// the partition.
func (b *windowFramerBase) incrementPeerGroup(ctx context.Context, index, groups int) int {
	if groups <= 0 {
		return index
	}
	if b.peersColIdx == tree.NoColumnIdx || index >= b.partitionSize {
		// With no ORDER BY clause, the entire partition is a single peer group.
		return b.partitionSize
	}
	// We have to iterate to the beginning of the next peer group.
	index++
	for {
		if index >= b.partitionSize {
			return b.partitionSize
		}
		vec, vecIdx, n := b.storedCols.GetVecWithTuple(ctx, b.peersColIdx, index)
		peersCol := vec.Bool()
		_, _ = peersCol[vecIdx], peersCol[n-1]
		for ; vecIdx < n; vecIdx++ {
			//gcassert:bce
			if peersCol[vecIdx] {
				// We have reached the start of the next peer group (the end of the
				// current one).
				groups--
				if groups <= 0 {
					return index
				}
			}
			index++
		}
	}
}

// isFirstPeer returns true if the row at the given index into the current
// partition is the first in its peer group.
func (b *windowFramerBase) isFirstPeer(ctx context.Context, idx int) bool {
	if idx == 0 {
		// The first row in the partition is always the first in its peer group.
		return true
	}
	if b.peersColIdx == tree.NoColumnIdx {
		// All rows are peers, so only the first is the first in its peer group.
		return false
	}
	vec, vecIdx, _ := b.storedCols.GetVecWithTuple(ctx, b.peersColIdx, idx)
	return vec.Bool()[vecIdx]
}

// handleOffsets populates the offset fields of the window framer operator, if
// one or both bounds are OFFSET PRECEDING or OFFSET FOLLOWING.
func (b *windowFramerBase) handleOffsets(
	evalCtx *eval.Context,
	frame *execinfrapb.WindowerSpec_Frame,
	ordering *execinfrapb.Ordering,
	inputTypes []*types.T,
) {
	startBound, endBound := &frame.Bounds.Start, frame.Bounds.End
	startHasOffset := startBound.BoundType == execinfrapb.WindowerSpec_Frame_OFFSET_PRECEDING ||
		startBound.BoundType == execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING
	endHasOffset := endBound.BoundType == execinfrapb.WindowerSpec_Frame_OFFSET_PRECEDING ||
		endBound.BoundType == execinfrapb.WindowerSpec_Frame_OFFSET_FOLLOWING
	if !(startHasOffset || endHasOffset) {
		return
	}

	if frame.Mode != execinfrapb.WindowerSpec_Frame_RANGE {
		// For ROWS or GROUPS mode, the offset is of type int64 and the ordering
		// column(s) is not stored.
		if startHasOffset {
			b.startOffset = int(startBound.IntOffset)
		}
		if endHasOffset {
			b.endOffset = int(endBound.IntOffset)
		}
		return
	}

	// For RANGE mode with an offset, there is a single ordering column. The
	// offset type depends on the ordering column type. In addition, the ordering
	// column must be stored.
	if len(ordering.Columns) != 1 {
		colexecerror.InternalError(
			errors.AssertionFailedf("expected exactly one ordering column for RANGE mode with offset"))
	}
	// Only initialize the DatumAlloc field when we know we will need it.
	b.datumAlloc = &tree.DatumAlloc{}
	b.ordColIdx = int(ordering.Columns[0].ColIdx)
	ordColType := inputTypes[b.ordColIdx]
	ordColAsc := ordering.Columns[0].Direction == execinfrapb.Ordering_Column_ASC
	if startHasOffset {
		b.startHandler = newRangeOffsetHandler(
			evalCtx, b.datumAlloc, startBound, ordColType, ordColAsc, true, /* isStart */
		)
	}
	if endHasOffset {
		b.endHandler = newRangeOffsetHandler(
			evalCtx, b.datumAlloc, endBound, ordColType, ordColAsc, false, /* isStart */
		)
	}
}

// handleExcludeForNext updates the start and end indices for rows excluded by
// the window frame's exclusion clause.
func (b *windowFramerBase) handleExcludeForNext(ctx context.Context, currRowIsGroupStart bool) {
	if b.excludeCurrRow() {
		b.excludeStartIdx = b.currentRow
		b.excludeEndIdx = b.excludeStartIdx + 1
	} else {
		// Note that EXCLUDE GROUP and EXCLUDE TIES are handled in the same way here
		// because they have the same start and end bounds. EXCLUDE TIES does not
		// actually exclude the current row, but that is handled later (by
		// handleExcludeForFirstIdx, handleExcludeForLastIdx, etc.).
		if currRowIsGroupStart {
			// Only update the exclude indices upon entering a new peer group.
			b.excludeStartIdx = b.excludeEndIdx
			b.excludeEndIdx = b.incrementPeerGroup(ctx, b.excludeEndIdx, 1 /* groups */)
		}
	}
}

// handleExcludeForFirstIdx adjusts the given 'first index' to account for the
// exclusion clause.
func (b *windowFramerBase) handleExcludeForFirstIdx(idx int) int {
	if idx == -1 {
		return idx
	}
	// Note that EXCLUDE GROUP and EXCLUDE CURRENT ROW are handled the same way
	// here because they can both be described simply by the interval
	// [excludeStartIdx, excludeEndIdx).
	if b.excludeStartIdx <= idx && b.excludeEndIdx > idx {
		if b.excludeTies() && b.currentRow >= b.startIdx && b.currentRow < b.endIdx {
			return b.currentRow
		}
		if b.excludeEndIdx >= b.endIdx {
			// All rows are excluded.
			return -1
		}
		return b.excludeEndIdx
	}
	return idx
}

// handleExcludeForLastIdx adjusts the given 'last index' to account for the
// exclusion clause.
func (b *windowFramerBase) handleExcludeForLastIdx(idx int) int {
	if idx == -1 {
		return idx
	}
	// Note that EXCLUDE GROUP and EXCLUDE CURRENT ROW are handled the same way
	// here because they can both be described simply by the interval
	// [excludeStartIdx, excludeEndIdx).
	if b.excludeStartIdx <= idx && b.excludeEndIdx > idx {
		if b.excludeTies() && b.currentRow >= b.startIdx && b.currentRow < b.endIdx {
			return b.currentRow
		}
		if b.excludeStartIdx <= b.startIdx {
			// All rows are excluded.
			return -1
		}
		return b.excludeStartIdx - 1
	}
	return idx
}

// handleExcludeForNthIdx adjusts the given 'nth index' to account for the
// exclusion clause.
func (b *windowFramerBase) handleExcludeForNthIdx(idx int) int {
	if idx == -1 {
		return idx
	}
	// Retrieve the rows that are actually excluded - those that are within
	// [startIdx, endIdx) in addition to being specified by the EXCLUDE clause.
	// Note that EXCLUDE GROUP and EXCLUDE CURRENT ROW are handled the same way
	// here because they can both be described simply by the interval
	// [excludeStartIdx, excludeEndIdx).
	excludedRowsStart := b.excludeStartIdx
	if excludedRowsStart < b.startIdx {
		excludedRowsStart = b.startIdx
	}
	excludedRowsEnd := b.excludeEndIdx
	if excludedRowsEnd > b.endIdx {
		excludedRowsEnd = b.endIdx
	}
	if excludedRowsStart < excludedRowsEnd && idx >= excludedRowsStart {
		if b.excludeTies() && b.currentRow >= b.startIdx && b.currentRow < b.endIdx {
			if idx == excludedRowsStart {
				return b.currentRow
			}
			idx--
		}
		idx += excludedRowsEnd - excludedRowsStart
		if idx >= b.endIdx {
			// The nth index doesn't exist.
			return -1
		}
	}
	return idx
}

func (b *windowFramerBase) excludeCurrRow() bool {
	return b.exclusion == execinfrapb.WindowerSpec_Frame_EXCLUDE_CURRENT_ROW
}

func (b *windowFramerBase) excludeTies() bool {
	return b.exclusion == execinfrapb.WindowerSpec_Frame_EXCLUDE_TIES
}

// getSlidingWindowIntervals is a helper function used to calculate the sets of
// rows that are a part of the current window frame, but not the previous one,
// and rows that were a part of the previous window frame, but not the current
// one. getSlidingWindowIntervals expects the intervals stored in currIntervals
// and prevIntervals to be non-overlapping and increasing, and guarantees the
// same invariants for the output intervals.
func getSlidingWindowIntervals(
	currIntervals, prevIntervals, toAdd, toRemove []windowInterval,
) ([]windowInterval, []windowInterval) {
	toAdd, toRemove = toAdd[:0], toRemove[:0]
	var prevIdx, currIdx int
	var prev, curr windowInterval
	setPrev, setCurr := true, true
	for {
		// We need to find the set difference currIntervals \ prevIntervals (toAdd)
		// and the set difference prevIntervals \ currIntervals (toRemove). To do
		// this, take advantage of the fact that both sets of intervals are in
		// ascending order, similar to merging sorted lists. Maintain indices into
		// each list, and iterate whichever index has the 'smaller' interval
		// (e.g. whichever ends first). The portions of the intervals that overlap
		// are ignored, while those that don't are added to one of the 'toAdd' and
		// 'toRemove' sets.
		if prevIdx >= len(prevIntervals) {
			// None of the remaining intervals in the current frame were part of the
			// previous frame.
			if !setCurr {
				// The remaining interval stored in curr still hasn't been handled.
				toAdd = append(toAdd, curr)
				currIdx++
			}
			if currIdx < len(currIntervals) {
				toAdd = append(toAdd, currIntervals[currIdx:]...)
			}
			break
		}
		if currIdx >= len(currIntervals) {
			// None of the remaining intervals in the previous frame are part of the
			// current frame.
			if !setPrev {
				// The remaining interval stored in prev still hasn't been handled.
				toRemove = append(toRemove, prev)
				prevIdx++
			}
			if prevIdx < len(prevIntervals) {
				toRemove = append(toRemove, prevIntervals[prevIdx:]...)
			}
			break
		}
		if setPrev {
			prev = prevIntervals[prevIdx]
			setPrev = false
		}
		if setCurr {
			curr = currIntervals[currIdx]
			setCurr = false
		}
		if prev == curr {
			// This interval has not changed from the previous frame.
			prevIdx++
			currIdx++
			setPrev, setCurr = true, true
			continue
		}
		if prev.start >= curr.end {
			// The intervals do not overlap, and the curr interval did not exist in
			// the previous window frame.
			toAdd = append(toAdd, curr)
			currIdx++
			setCurr = true
			continue
		}
		if curr.start >= prev.end {
			// The intervals do not overlap, and the prev interval existed in the
			// previous window frame, but not the current one.
			toRemove = append(toRemove, prev)
			prevIdx++
			setPrev = true
			continue
		}
		// The intervals overlap but are not equal.
		if curr.start < prev.start {
			// curr starts before prev. Add the prefix of curr to 'toAdd'. Advance the
			// start of curr to the start of prev to reflect that the prefix has
			// already been processed.
			toAdd = append(toAdd, windowInterval{start: curr.start, end: prev.start})
			curr.start = prev.start
		} else if prev.start < curr.start {
			// prev starts before curr. Add the prefix of prev to 'toRemove'. Advance
			// the start of prev to the start of curr to reflect that the prefix has
			// already been processed.
			toRemove = append(toRemove, windowInterval{start: prev.start, end: curr.start})
			prev.start = curr.start
		}
		if curr.end > prev.end {
			// prev ends before curr. Set the start of curr to the end of prev to
			// indicate that prev has been processed.
			curr.start = prev.end
			prevIdx++
			setPrev = true
		} else if prev.end > curr.end {
			// curr ends before prev. Set the start of prev to the end of curr to
			// indicate that curr has been processed.
			prev.start = curr.end
			currIdx++
			setCurr = true
		} else {
			// prev and curr end at the same index. The prefix of whichever one starts
			// first has already been handled.
			prevIdx++
			currIdx++
			setPrev, setCurr = true, true
		}
	}
	return toAdd, toRemove
}

type windowFramerRowsUnboundedPrecedingOffsetPreceding struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsUnboundedPrecedingOffsetPreceding{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsUnboundedPrecedingOffsetPreceding) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsUnboundedPrecedingOffsetPreceding) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	f.endIdx = f.currentRow - f.endOffset + 1
	if f.endIdx < 0 {
		f.endIdx = 0
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsUnboundedPrecedingOffsetPreceding) close() {
	*f = windowFramerRowsUnboundedPrecedingOffsetPreceding{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsUnboundedPrecedingOffsetPreceding) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRowsUnboundedPrecedingOffsetPrecedingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsUnboundedPrecedingOffsetPrecedingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsUnboundedPrecedingOffsetPrecedingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsUnboundedPrecedingOffsetPrecedingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	f.endIdx = f.currentRow - f.endOffset + 1
	if f.endIdx < 0 {
		f.endIdx = 0
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsUnboundedPrecedingOffsetPrecedingExclude) close() {
	*f = windowFramerRowsUnboundedPrecedingOffsetPrecedingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsUnboundedPrecedingOffsetPrecedingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRowsUnboundedPrecedingOffsetPrecedingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRowsUnboundedPrecedingOffsetPrecedingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRowsUnboundedPrecedingOffsetPrecedingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRowsUnboundedPrecedingOffsetPrecedingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRowsUnboundedPrecedingCurrentRow struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsUnboundedPrecedingCurrentRow{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsUnboundedPrecedingCurrentRow) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsUnboundedPrecedingCurrentRow) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	f.endIdx = f.currentRow + 1
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsUnboundedPrecedingCurrentRow) close() {
	*f = windowFramerRowsUnboundedPrecedingCurrentRow{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsUnboundedPrecedingCurrentRow) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRowsUnboundedPrecedingCurrentRowExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsUnboundedPrecedingCurrentRowExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsUnboundedPrecedingCurrentRowExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsUnboundedPrecedingCurrentRowExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	f.endIdx = f.currentRow + 1
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsUnboundedPrecedingCurrentRowExclude) close() {
	*f = windowFramerRowsUnboundedPrecedingCurrentRowExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsUnboundedPrecedingCurrentRowExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRowsUnboundedPrecedingCurrentRowExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRowsUnboundedPrecedingCurrentRowExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRowsUnboundedPrecedingCurrentRowExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRowsUnboundedPrecedingCurrentRowExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRowsUnboundedPrecedingOffsetFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsUnboundedPrecedingOffsetFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsUnboundedPrecedingOffsetFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsUnboundedPrecedingOffsetFollowing) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	f.endIdx = f.currentRow + f.endOffset + 1
	if f.endIdx > f.partitionSize || f.endOffset >= f.partitionSize {
		// The second part of the condition protects us from an integer
		// overflow when offset is very large.
		f.endIdx = f.partitionSize
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsUnboundedPrecedingOffsetFollowing) close() {
	*f = windowFramerRowsUnboundedPrecedingOffsetFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsUnboundedPrecedingOffsetFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRowsUnboundedPrecedingOffsetFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsUnboundedPrecedingOffsetFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsUnboundedPrecedingOffsetFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsUnboundedPrecedingOffsetFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	f.endIdx = f.currentRow + f.endOffset + 1
	if f.endIdx > f.partitionSize || f.endOffset >= f.partitionSize {
		// The second part of the condition protects us from an integer
		// overflow when offset is very large.
		f.endIdx = f.partitionSize
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsUnboundedPrecedingOffsetFollowingExclude) close() {
	*f = windowFramerRowsUnboundedPrecedingOffsetFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsUnboundedPrecedingOffsetFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRowsUnboundedPrecedingOffsetFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRowsUnboundedPrecedingOffsetFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRowsUnboundedPrecedingOffsetFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRowsUnboundedPrecedingOffsetFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRowsUnboundedPrecedingUnboundedFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsUnboundedPrecedingUnboundedFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsUnboundedPrecedingUnboundedFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsUnboundedPrecedingUnboundedFollowing) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	f.endIdx = f.partitionSize
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsUnboundedPrecedingUnboundedFollowing) close() {
	*f = windowFramerRowsUnboundedPrecedingUnboundedFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsUnboundedPrecedingUnboundedFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRowsUnboundedPrecedingUnboundedFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsUnboundedPrecedingUnboundedFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsUnboundedPrecedingUnboundedFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsUnboundedPrecedingUnboundedFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	f.endIdx = f.partitionSize
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsUnboundedPrecedingUnboundedFollowingExclude) close() {
	*f = windowFramerRowsUnboundedPrecedingUnboundedFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsUnboundedPrecedingUnboundedFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRowsUnboundedPrecedingUnboundedFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRowsUnboundedPrecedingUnboundedFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRowsUnboundedPrecedingUnboundedFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRowsUnboundedPrecedingUnboundedFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRowsOffsetPrecedingOffsetPreceding struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsOffsetPrecedingOffsetPreceding{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsOffsetPrecedingOffsetPreceding) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsOffsetPrecedingOffsetPreceding) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = f.currentRow - f.startOffset
	if f.startIdx < 0 {
		f.startIdx = 0
	}

	// Handle the end bound.
	f.endIdx = f.currentRow - f.endOffset + 1
	if f.endIdx < 0 {
		f.endIdx = 0
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsOffsetPrecedingOffsetPreceding) close() {
	*f = windowFramerRowsOffsetPrecedingOffsetPreceding{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsOffsetPrecedingOffsetPreceding) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRowsOffsetPrecedingOffsetPrecedingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsOffsetPrecedingOffsetPrecedingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsOffsetPrecedingOffsetPrecedingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsOffsetPrecedingOffsetPrecedingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = f.currentRow - f.startOffset
	if f.startIdx < 0 {
		f.startIdx = 0
	}

	// Handle the end bound.
	f.endIdx = f.currentRow - f.endOffset + 1
	if f.endIdx < 0 {
		f.endIdx = 0
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsOffsetPrecedingOffsetPrecedingExclude) close() {
	*f = windowFramerRowsOffsetPrecedingOffsetPrecedingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsOffsetPrecedingOffsetPrecedingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRowsOffsetPrecedingOffsetPrecedingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRowsOffsetPrecedingOffsetPrecedingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRowsOffsetPrecedingOffsetPrecedingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRowsOffsetPrecedingOffsetPrecedingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRowsOffsetPrecedingCurrentRow struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsOffsetPrecedingCurrentRow{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsOffsetPrecedingCurrentRow) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsOffsetPrecedingCurrentRow) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = f.currentRow - f.startOffset
	if f.startIdx < 0 {
		f.startIdx = 0
	}

	// Handle the end bound.
	f.endIdx = f.currentRow + 1
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsOffsetPrecedingCurrentRow) close() {
	*f = windowFramerRowsOffsetPrecedingCurrentRow{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsOffsetPrecedingCurrentRow) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRowsOffsetPrecedingCurrentRowExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsOffsetPrecedingCurrentRowExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsOffsetPrecedingCurrentRowExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsOffsetPrecedingCurrentRowExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = f.currentRow - f.startOffset
	if f.startIdx < 0 {
		f.startIdx = 0
	}

	// Handle the end bound.
	f.endIdx = f.currentRow + 1
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsOffsetPrecedingCurrentRowExclude) close() {
	*f = windowFramerRowsOffsetPrecedingCurrentRowExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsOffsetPrecedingCurrentRowExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRowsOffsetPrecedingCurrentRowExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRowsOffsetPrecedingCurrentRowExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRowsOffsetPrecedingCurrentRowExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRowsOffsetPrecedingCurrentRowExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRowsOffsetPrecedingOffsetFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsOffsetPrecedingOffsetFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsOffsetPrecedingOffsetFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsOffsetPrecedingOffsetFollowing) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = f.currentRow - f.startOffset
	if f.startIdx < 0 {
		f.startIdx = 0
	}

	// Handle the end bound.
	f.endIdx = f.currentRow + f.endOffset + 1
	if f.endIdx > f.partitionSize || f.endOffset >= f.partitionSize {
		// The second part of the condition protects us from an integer
		// overflow when offset is very large.
		f.endIdx = f.partitionSize
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsOffsetPrecedingOffsetFollowing) close() {
	*f = windowFramerRowsOffsetPrecedingOffsetFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsOffsetPrecedingOffsetFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRowsOffsetPrecedingOffsetFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsOffsetPrecedingOffsetFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsOffsetPrecedingOffsetFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsOffsetPrecedingOffsetFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = f.currentRow - f.startOffset
	if f.startIdx < 0 {
		f.startIdx = 0
	}

	// Handle the end bound.
	f.endIdx = f.currentRow + f.endOffset + 1
	if f.endIdx > f.partitionSize || f.endOffset >= f.partitionSize {
		// The second part of the condition protects us from an integer
		// overflow when offset is very large.
		f.endIdx = f.partitionSize
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsOffsetPrecedingOffsetFollowingExclude) close() {
	*f = windowFramerRowsOffsetPrecedingOffsetFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsOffsetPrecedingOffsetFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRowsOffsetPrecedingOffsetFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRowsOffsetPrecedingOffsetFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRowsOffsetPrecedingOffsetFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRowsOffsetPrecedingOffsetFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRowsOffsetPrecedingUnboundedFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsOffsetPrecedingUnboundedFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsOffsetPrecedingUnboundedFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsOffsetPrecedingUnboundedFollowing) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = f.currentRow - f.startOffset
	if f.startIdx < 0 {
		f.startIdx = 0
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsOffsetPrecedingUnboundedFollowing) close() {
	*f = windowFramerRowsOffsetPrecedingUnboundedFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsOffsetPrecedingUnboundedFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRowsOffsetPrecedingUnboundedFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsOffsetPrecedingUnboundedFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsOffsetPrecedingUnboundedFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsOffsetPrecedingUnboundedFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = f.currentRow - f.startOffset
	if f.startIdx < 0 {
		f.startIdx = 0
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsOffsetPrecedingUnboundedFollowingExclude) close() {
	*f = windowFramerRowsOffsetPrecedingUnboundedFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsOffsetPrecedingUnboundedFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRowsOffsetPrecedingUnboundedFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRowsOffsetPrecedingUnboundedFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRowsOffsetPrecedingUnboundedFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRowsOffsetPrecedingUnboundedFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRowsCurrentRowCurrentRow struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsCurrentRowCurrentRow{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsCurrentRowCurrentRow) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsCurrentRowCurrentRow) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = f.currentRow

	// Handle the end bound.
	f.endIdx = f.currentRow + 1
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsCurrentRowCurrentRow) close() {
	*f = windowFramerRowsCurrentRowCurrentRow{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsCurrentRowCurrentRow) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRowsCurrentRowCurrentRowExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsCurrentRowCurrentRowExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsCurrentRowCurrentRowExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsCurrentRowCurrentRowExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = f.currentRow

	// Handle the end bound.
	f.endIdx = f.currentRow + 1
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsCurrentRowCurrentRowExclude) close() {
	*f = windowFramerRowsCurrentRowCurrentRowExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsCurrentRowCurrentRowExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRowsCurrentRowCurrentRowExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRowsCurrentRowCurrentRowExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRowsCurrentRowCurrentRowExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRowsCurrentRowCurrentRowExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRowsCurrentRowOffsetFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsCurrentRowOffsetFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsCurrentRowOffsetFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsCurrentRowOffsetFollowing) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = f.currentRow

	// Handle the end bound.
	f.endIdx = f.currentRow + f.endOffset + 1
	if f.endIdx > f.partitionSize || f.endOffset >= f.partitionSize {
		// The second part of the condition protects us from an integer
		// overflow when offset is very large.
		f.endIdx = f.partitionSize
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsCurrentRowOffsetFollowing) close() {
	*f = windowFramerRowsCurrentRowOffsetFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsCurrentRowOffsetFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRowsCurrentRowOffsetFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsCurrentRowOffsetFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsCurrentRowOffsetFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsCurrentRowOffsetFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = f.currentRow

	// Handle the end bound.
	f.endIdx = f.currentRow + f.endOffset + 1
	if f.endIdx > f.partitionSize || f.endOffset >= f.partitionSize {
		// The second part of the condition protects us from an integer
		// overflow when offset is very large.
		f.endIdx = f.partitionSize
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsCurrentRowOffsetFollowingExclude) close() {
	*f = windowFramerRowsCurrentRowOffsetFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsCurrentRowOffsetFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRowsCurrentRowOffsetFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRowsCurrentRowOffsetFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRowsCurrentRowOffsetFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRowsCurrentRowOffsetFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRowsCurrentRowUnboundedFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsCurrentRowUnboundedFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsCurrentRowUnboundedFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsCurrentRowUnboundedFollowing) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = f.currentRow

	// Handle the end bound.
	f.endIdx = f.partitionSize
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsCurrentRowUnboundedFollowing) close() {
	*f = windowFramerRowsCurrentRowUnboundedFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsCurrentRowUnboundedFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRowsCurrentRowUnboundedFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsCurrentRowUnboundedFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsCurrentRowUnboundedFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsCurrentRowUnboundedFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = f.currentRow

	// Handle the end bound.
	f.endIdx = f.partitionSize
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsCurrentRowUnboundedFollowingExclude) close() {
	*f = windowFramerRowsCurrentRowUnboundedFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsCurrentRowUnboundedFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRowsCurrentRowUnboundedFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRowsCurrentRowUnboundedFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRowsCurrentRowUnboundedFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRowsCurrentRowUnboundedFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRowsOffsetFollowingOffsetFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsOffsetFollowingOffsetFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsOffsetFollowingOffsetFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsOffsetFollowingOffsetFollowing) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = f.currentRow + f.startOffset
	if f.startIdx > f.partitionSize || f.startOffset >= f.partitionSize {
		// The second part of the condition protects us from an integer
		// overflow when offset is very large.
		f.startIdx = f.partitionSize
	}

	// Handle the end bound.
	f.endIdx = f.currentRow + f.endOffset + 1
	if f.endIdx > f.partitionSize || f.endOffset >= f.partitionSize {
		// The second part of the condition protects us from an integer
		// overflow when offset is very large.
		f.endIdx = f.partitionSize
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsOffsetFollowingOffsetFollowing) close() {
	*f = windowFramerRowsOffsetFollowingOffsetFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsOffsetFollowingOffsetFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRowsOffsetFollowingOffsetFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsOffsetFollowingOffsetFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsOffsetFollowingOffsetFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsOffsetFollowingOffsetFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = f.currentRow + f.startOffset
	if f.startIdx > f.partitionSize || f.startOffset >= f.partitionSize {
		// The second part of the condition protects us from an integer
		// overflow when offset is very large.
		f.startIdx = f.partitionSize
	}

	// Handle the end bound.
	f.endIdx = f.currentRow + f.endOffset + 1
	if f.endIdx > f.partitionSize || f.endOffset >= f.partitionSize {
		// The second part of the condition protects us from an integer
		// overflow when offset is very large.
		f.endIdx = f.partitionSize
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsOffsetFollowingOffsetFollowingExclude) close() {
	*f = windowFramerRowsOffsetFollowingOffsetFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsOffsetFollowingOffsetFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRowsOffsetFollowingOffsetFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRowsOffsetFollowingOffsetFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRowsOffsetFollowingOffsetFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRowsOffsetFollowingOffsetFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRowsOffsetFollowingUnboundedFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsOffsetFollowingUnboundedFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsOffsetFollowingUnboundedFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsOffsetFollowingUnboundedFollowing) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = f.currentRow + f.startOffset
	if f.startIdx > f.partitionSize || f.startOffset >= f.partitionSize {
		// The second part of the condition protects us from an integer
		// overflow when offset is very large.
		f.startIdx = f.partitionSize
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsOffsetFollowingUnboundedFollowing) close() {
	*f = windowFramerRowsOffsetFollowingUnboundedFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsOffsetFollowingUnboundedFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRowsOffsetFollowingUnboundedFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRowsOffsetFollowingUnboundedFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRowsOffsetFollowingUnboundedFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRowsOffsetFollowingUnboundedFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = f.currentRow + f.startOffset
	if f.startIdx > f.partitionSize || f.startOffset >= f.partitionSize {
		// The second part of the condition protects us from an integer
		// overflow when offset is very large.
		f.startIdx = f.partitionSize
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRowsOffsetFollowingUnboundedFollowingExclude) close() {
	*f = windowFramerRowsOffsetFollowingUnboundedFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRowsOffsetFollowingUnboundedFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRowsOffsetFollowingUnboundedFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRowsOffsetFollowingUnboundedFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRowsOffsetFollowingUnboundedFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRowsOffsetFollowingUnboundedFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerGroupsUnboundedPrecedingOffsetPreceding struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsUnboundedPrecedingOffsetPreceding{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsUnboundedPrecedingOffsetPreceding) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.currentGroup = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsUnboundedPrecedingOffsetPreceding) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	if currRowIsGroupStart && f.currentGroup >= f.endOffset {
		// Advance the end index to the start of the next peer group.
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	if currRowIsGroupStart {
		f.currentGroup++
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsUnboundedPrecedingOffsetPreceding) close() {
	*f = windowFramerGroupsUnboundedPrecedingOffsetPreceding{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsUnboundedPrecedingOffsetPreceding) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerGroupsUnboundedPrecedingOffsetPrecedingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsUnboundedPrecedingOffsetPrecedingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsUnboundedPrecedingOffsetPrecedingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.currentGroup = 0
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsUnboundedPrecedingOffsetPrecedingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	if currRowIsGroupStart && f.currentGroup >= f.endOffset {
		// Advance the end index to the start of the next peer group.
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	if currRowIsGroupStart {
		f.currentGroup++
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsUnboundedPrecedingOffsetPrecedingExclude) close() {
	*f = windowFramerGroupsUnboundedPrecedingOffsetPrecedingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsUnboundedPrecedingOffsetPrecedingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerGroupsUnboundedPrecedingOffsetPrecedingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerGroupsUnboundedPrecedingOffsetPrecedingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerGroupsUnboundedPrecedingOffsetPrecedingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerGroupsUnboundedPrecedingOffsetPrecedingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerGroupsUnboundedPrecedingCurrentRow struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsUnboundedPrecedingCurrentRow{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsUnboundedPrecedingCurrentRow) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsUnboundedPrecedingCurrentRow) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsUnboundedPrecedingCurrentRow) close() {
	*f = windowFramerGroupsUnboundedPrecedingCurrentRow{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsUnboundedPrecedingCurrentRow) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerGroupsUnboundedPrecedingCurrentRowExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsUnboundedPrecedingCurrentRowExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsUnboundedPrecedingCurrentRowExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsUnboundedPrecedingCurrentRowExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsUnboundedPrecedingCurrentRowExclude) close() {
	*f = windowFramerGroupsUnboundedPrecedingCurrentRowExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsUnboundedPrecedingCurrentRowExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerGroupsUnboundedPrecedingCurrentRowExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerGroupsUnboundedPrecedingCurrentRowExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerGroupsUnboundedPrecedingCurrentRowExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerGroupsUnboundedPrecedingCurrentRowExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerGroupsUnboundedPrecedingOffsetFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsUnboundedPrecedingOffsetFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsUnboundedPrecedingOffsetFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	// f.endIdx must be advanced by 'endOffset' peer groups.
	f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, f.endOffset)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsUnboundedPrecedingOffsetFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	if currRowIsGroupStart {
		// The end index will already have been advanced by f.startOffset peer
		// groups, so we only need to advance to the next adjacent peers group
		// whenever the currentRow pointer enters a new peers group.
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsUnboundedPrecedingOffsetFollowing) close() {
	*f = windowFramerGroupsUnboundedPrecedingOffsetFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsUnboundedPrecedingOffsetFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerGroupsUnboundedPrecedingOffsetFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsUnboundedPrecedingOffsetFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsUnboundedPrecedingOffsetFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	// f.endIdx must be advanced by 'endOffset' peer groups.
	f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, f.endOffset)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsUnboundedPrecedingOffsetFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	if currRowIsGroupStart {
		// The end index will already have been advanced by f.startOffset peer
		// groups, so we only need to advance to the next adjacent peers group
		// whenever the currentRow pointer enters a new peers group.
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsUnboundedPrecedingOffsetFollowingExclude) close() {
	*f = windowFramerGroupsUnboundedPrecedingOffsetFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsUnboundedPrecedingOffsetFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerGroupsUnboundedPrecedingOffsetFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerGroupsUnboundedPrecedingOffsetFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerGroupsUnboundedPrecedingOffsetFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerGroupsUnboundedPrecedingOffsetFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerGroupsUnboundedPrecedingUnboundedFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsUnboundedPrecedingUnboundedFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsUnboundedPrecedingUnboundedFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsUnboundedPrecedingUnboundedFollowing) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	f.endIdx = f.partitionSize
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsUnboundedPrecedingUnboundedFollowing) close() {
	*f = windowFramerGroupsUnboundedPrecedingUnboundedFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsUnboundedPrecedingUnboundedFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerGroupsUnboundedPrecedingUnboundedFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsUnboundedPrecedingUnboundedFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsUnboundedPrecedingUnboundedFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsUnboundedPrecedingUnboundedFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	f.endIdx = f.partitionSize
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsUnboundedPrecedingUnboundedFollowingExclude) close() {
	*f = windowFramerGroupsUnboundedPrecedingUnboundedFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsUnboundedPrecedingUnboundedFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerGroupsUnboundedPrecedingUnboundedFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerGroupsUnboundedPrecedingUnboundedFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerGroupsUnboundedPrecedingUnboundedFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerGroupsUnboundedPrecedingUnboundedFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerGroupsOffsetPrecedingOffsetPreceding struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsOffsetPrecedingOffsetPreceding{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsOffsetPrecedingOffsetPreceding) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.currentGroup = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsOffsetPrecedingOffsetPreceding) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart && f.currentGroup > f.startOffset {
		// Advance the start index to the start of the next peer group.
		f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, 1 /* groups */)
	}

	// Handle the end bound.
	if currRowIsGroupStart && f.currentGroup >= f.endOffset {
		// Advance the end index to the start of the next peer group.
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	if currRowIsGroupStart {
		f.currentGroup++
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsOffsetPrecedingOffsetPreceding) close() {
	*f = windowFramerGroupsOffsetPrecedingOffsetPreceding{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsOffsetPrecedingOffsetPreceding) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerGroupsOffsetPrecedingOffsetPrecedingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsOffsetPrecedingOffsetPrecedingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsOffsetPrecedingOffsetPrecedingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.currentGroup = 0
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsOffsetPrecedingOffsetPrecedingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart && f.currentGroup > f.startOffset {
		// Advance the start index to the start of the next peer group.
		f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, 1 /* groups */)
	}

	// Handle the end bound.
	if currRowIsGroupStart && f.currentGroup >= f.endOffset {
		// Advance the end index to the start of the next peer group.
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	if currRowIsGroupStart {
		f.currentGroup++
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsOffsetPrecedingOffsetPrecedingExclude) close() {
	*f = windowFramerGroupsOffsetPrecedingOffsetPrecedingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsOffsetPrecedingOffsetPrecedingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerGroupsOffsetPrecedingOffsetPrecedingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerGroupsOffsetPrecedingOffsetPrecedingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerGroupsOffsetPrecedingOffsetPrecedingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerGroupsOffsetPrecedingOffsetPrecedingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerGroupsOffsetPrecedingCurrentRow struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsOffsetPrecedingCurrentRow{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsOffsetPrecedingCurrentRow) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.currentGroup = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsOffsetPrecedingCurrentRow) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart && f.currentGroup > f.startOffset {
		// Advance the start index to the start of the next peer group.
		f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, 1 /* groups */)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	if currRowIsGroupStart {
		f.currentGroup++
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsOffsetPrecedingCurrentRow) close() {
	*f = windowFramerGroupsOffsetPrecedingCurrentRow{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsOffsetPrecedingCurrentRow) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerGroupsOffsetPrecedingCurrentRowExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsOffsetPrecedingCurrentRowExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsOffsetPrecedingCurrentRowExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.currentGroup = 0
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsOffsetPrecedingCurrentRowExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart && f.currentGroup > f.startOffset {
		// Advance the start index to the start of the next peer group.
		f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, 1 /* groups */)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	if currRowIsGroupStart {
		f.currentGroup++
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsOffsetPrecedingCurrentRowExclude) close() {
	*f = windowFramerGroupsOffsetPrecedingCurrentRowExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsOffsetPrecedingCurrentRowExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerGroupsOffsetPrecedingCurrentRowExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerGroupsOffsetPrecedingCurrentRowExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerGroupsOffsetPrecedingCurrentRowExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerGroupsOffsetPrecedingCurrentRowExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerGroupsOffsetPrecedingOffsetFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsOffsetPrecedingOffsetFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsOffsetPrecedingOffsetFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.currentGroup = 0
	// f.endIdx must be advanced by 'endOffset' peer groups.
	f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, f.endOffset)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsOffsetPrecedingOffsetFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart && f.currentGroup > f.startOffset {
		// Advance the start index to the start of the next peer group.
		f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, 1 /* groups */)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		// The end index will already have been advanced by f.startOffset peer
		// groups, so we only need to advance to the next adjacent peers group
		// whenever the currentRow pointer enters a new peers group.
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	if currRowIsGroupStart {
		f.currentGroup++
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsOffsetPrecedingOffsetFollowing) close() {
	*f = windowFramerGroupsOffsetPrecedingOffsetFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsOffsetPrecedingOffsetFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerGroupsOffsetPrecedingOffsetFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsOffsetPrecedingOffsetFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsOffsetPrecedingOffsetFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.currentGroup = 0
	// f.endIdx must be advanced by 'endOffset' peer groups.
	f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, f.endOffset)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsOffsetPrecedingOffsetFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart && f.currentGroup > f.startOffset {
		// Advance the start index to the start of the next peer group.
		f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, 1 /* groups */)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		// The end index will already have been advanced by f.startOffset peer
		// groups, so we only need to advance to the next adjacent peers group
		// whenever the currentRow pointer enters a new peers group.
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	if currRowIsGroupStart {
		f.currentGroup++
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsOffsetPrecedingOffsetFollowingExclude) close() {
	*f = windowFramerGroupsOffsetPrecedingOffsetFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsOffsetPrecedingOffsetFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerGroupsOffsetPrecedingOffsetFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerGroupsOffsetPrecedingOffsetFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerGroupsOffsetPrecedingOffsetFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerGroupsOffsetPrecedingOffsetFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerGroupsOffsetPrecedingUnboundedFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsOffsetPrecedingUnboundedFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsOffsetPrecedingUnboundedFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.currentGroup = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsOffsetPrecedingUnboundedFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart && f.currentGroup > f.startOffset {
		// Advance the start index to the start of the next peer group.
		f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, 1 /* groups */)
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	if currRowIsGroupStart {
		f.currentGroup++
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsOffsetPrecedingUnboundedFollowing) close() {
	*f = windowFramerGroupsOffsetPrecedingUnboundedFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsOffsetPrecedingUnboundedFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerGroupsOffsetPrecedingUnboundedFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsOffsetPrecedingUnboundedFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsOffsetPrecedingUnboundedFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.currentGroup = 0
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsOffsetPrecedingUnboundedFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart && f.currentGroup > f.startOffset {
		// Advance the start index to the start of the next peer group.
		f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, 1 /* groups */)
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	if currRowIsGroupStart {
		f.currentGroup++
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsOffsetPrecedingUnboundedFollowingExclude) close() {
	*f = windowFramerGroupsOffsetPrecedingUnboundedFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsOffsetPrecedingUnboundedFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerGroupsOffsetPrecedingUnboundedFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerGroupsOffsetPrecedingUnboundedFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerGroupsOffsetPrecedingUnboundedFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerGroupsOffsetPrecedingUnboundedFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerGroupsCurrentRowCurrentRow struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsCurrentRowCurrentRow{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsCurrentRowCurrentRow) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsCurrentRowCurrentRow) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.currentRow
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsCurrentRowCurrentRow) close() {
	*f = windowFramerGroupsCurrentRowCurrentRow{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsCurrentRowCurrentRow) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerGroupsCurrentRowCurrentRowExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsCurrentRowCurrentRowExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsCurrentRowCurrentRowExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsCurrentRowCurrentRowExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.currentRow
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsCurrentRowCurrentRowExclude) close() {
	*f = windowFramerGroupsCurrentRowCurrentRowExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsCurrentRowCurrentRowExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerGroupsCurrentRowCurrentRowExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerGroupsCurrentRowCurrentRowExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerGroupsCurrentRowCurrentRowExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerGroupsCurrentRowCurrentRowExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerGroupsCurrentRowOffsetFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsCurrentRowOffsetFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsCurrentRowOffsetFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	// f.endIdx must be advanced by 'endOffset' peer groups.
	f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, f.endOffset)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsCurrentRowOffsetFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.currentRow
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		// The end index will already have been advanced by f.startOffset peer
		// groups, so we only need to advance to the next adjacent peers group
		// whenever the currentRow pointer enters a new peers group.
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsCurrentRowOffsetFollowing) close() {
	*f = windowFramerGroupsCurrentRowOffsetFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsCurrentRowOffsetFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerGroupsCurrentRowOffsetFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsCurrentRowOffsetFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsCurrentRowOffsetFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	// f.endIdx must be advanced by 'endOffset' peer groups.
	f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, f.endOffset)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsCurrentRowOffsetFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.currentRow
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		// The end index will already have been advanced by f.startOffset peer
		// groups, so we only need to advance to the next adjacent peers group
		// whenever the currentRow pointer enters a new peers group.
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsCurrentRowOffsetFollowingExclude) close() {
	*f = windowFramerGroupsCurrentRowOffsetFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsCurrentRowOffsetFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerGroupsCurrentRowOffsetFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerGroupsCurrentRowOffsetFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerGroupsCurrentRowOffsetFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerGroupsCurrentRowOffsetFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerGroupsCurrentRowUnboundedFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsCurrentRowUnboundedFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsCurrentRowUnboundedFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsCurrentRowUnboundedFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.currentRow
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsCurrentRowUnboundedFollowing) close() {
	*f = windowFramerGroupsCurrentRowUnboundedFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsCurrentRowUnboundedFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerGroupsCurrentRowUnboundedFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsCurrentRowUnboundedFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsCurrentRowUnboundedFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsCurrentRowUnboundedFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.currentRow
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsCurrentRowUnboundedFollowingExclude) close() {
	*f = windowFramerGroupsCurrentRowUnboundedFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsCurrentRowUnboundedFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerGroupsCurrentRowUnboundedFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerGroupsCurrentRowUnboundedFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerGroupsCurrentRowUnboundedFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerGroupsCurrentRowUnboundedFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerGroupsOffsetFollowingOffsetFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsOffsetFollowingOffsetFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsOffsetFollowingOffsetFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	// f.startIdx must be advanced by 'startOffset' peer groups.
	f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, f.startOffset)
	// f.endIdx must be advanced by 'endOffset' peer groups.
	f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, f.endOffset)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsOffsetFollowingOffsetFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart && f.currentRow > 0 {
		// The start index will already have been advanced by f.startOffset peer
		// groups, so we only need to advance to the next adjacent peers group
		// whenever the currentRow pointer enters a new peers group.
		f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, 1 /* groups */)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		// The end index will already have been advanced by f.startOffset peer
		// groups, so we only need to advance to the next adjacent peers group
		// whenever the currentRow pointer enters a new peers group.
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsOffsetFollowingOffsetFollowing) close() {
	*f = windowFramerGroupsOffsetFollowingOffsetFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsOffsetFollowingOffsetFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerGroupsOffsetFollowingOffsetFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsOffsetFollowingOffsetFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsOffsetFollowingOffsetFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	// f.startIdx must be advanced by 'startOffset' peer groups.
	f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, f.startOffset)
	// f.endIdx must be advanced by 'endOffset' peer groups.
	f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, f.endOffset)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsOffsetFollowingOffsetFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart && f.currentRow > 0 {
		// The start index will already have been advanced by f.startOffset peer
		// groups, so we only need to advance to the next adjacent peers group
		// whenever the currentRow pointer enters a new peers group.
		f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, 1 /* groups */)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		// The end index will already have been advanced by f.startOffset peer
		// groups, so we only need to advance to the next adjacent peers group
		// whenever the currentRow pointer enters a new peers group.
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsOffsetFollowingOffsetFollowingExclude) close() {
	*f = windowFramerGroupsOffsetFollowingOffsetFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsOffsetFollowingOffsetFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerGroupsOffsetFollowingOffsetFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerGroupsOffsetFollowingOffsetFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerGroupsOffsetFollowingOffsetFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerGroupsOffsetFollowingOffsetFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerGroupsOffsetFollowingUnboundedFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsOffsetFollowingUnboundedFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsOffsetFollowingUnboundedFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	// f.startIdx must be advanced by 'startOffset' peer groups.
	f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, f.startOffset)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsOffsetFollowingUnboundedFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart && f.currentRow > 0 {
		// The start index will already have been advanced by f.startOffset peer
		// groups, so we only need to advance to the next adjacent peers group
		// whenever the currentRow pointer enters a new peers group.
		f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, 1 /* groups */)
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsOffsetFollowingUnboundedFollowing) close() {
	*f = windowFramerGroupsOffsetFollowingUnboundedFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsOffsetFollowingUnboundedFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerGroupsOffsetFollowingUnboundedFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerGroupsOffsetFollowingUnboundedFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerGroupsOffsetFollowingUnboundedFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	// f.startIdx must be advanced by 'startOffset' peer groups.
	f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, f.startOffset)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerGroupsOffsetFollowingUnboundedFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart && f.currentRow > 0 {
		// The start index will already have been advanced by f.startOffset peer
		// groups, so we only need to advance to the next adjacent peers group
		// whenever the currentRow pointer enters a new peers group.
		f.startIdx = f.incrementPeerGroup(ctx, f.startIdx, 1 /* groups */)
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerGroupsOffsetFollowingUnboundedFollowingExclude) close() {
	*f = windowFramerGroupsOffsetFollowingUnboundedFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerGroupsOffsetFollowingUnboundedFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerGroupsOffsetFollowingUnboundedFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerGroupsOffsetFollowingUnboundedFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerGroupsOffsetFollowingUnboundedFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerGroupsOffsetFollowingUnboundedFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRangeUnboundedPrecedingOffsetPreceding struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeUnboundedPrecedingOffsetPreceding{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeUnboundedPrecedingOffsetPreceding) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.endHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeUnboundedPrecedingOffsetPreceding) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.endHandler.getIdx(ctx, f.currentRow, f.endIdx)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeUnboundedPrecedingOffsetPreceding) close() {
	f.endHandler.close()
	*f = windowFramerRangeUnboundedPrecedingOffsetPreceding{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeUnboundedPrecedingOffsetPreceding) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRangeUnboundedPrecedingOffsetPrecedingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeUnboundedPrecedingOffsetPrecedingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeUnboundedPrecedingOffsetPrecedingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.endHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeUnboundedPrecedingOffsetPrecedingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.endHandler.getIdx(ctx, f.currentRow, f.endIdx)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeUnboundedPrecedingOffsetPrecedingExclude) close() {
	f.endHandler.close()
	*f = windowFramerRangeUnboundedPrecedingOffsetPrecedingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeUnboundedPrecedingOffsetPrecedingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRangeUnboundedPrecedingOffsetPrecedingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRangeUnboundedPrecedingOffsetPrecedingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRangeUnboundedPrecedingOffsetPrecedingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRangeUnboundedPrecedingOffsetPrecedingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRangeUnboundedPrecedingCurrentRow struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeUnboundedPrecedingCurrentRow{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeUnboundedPrecedingCurrentRow) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeUnboundedPrecedingCurrentRow) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeUnboundedPrecedingCurrentRow) close() {
	*f = windowFramerRangeUnboundedPrecedingCurrentRow{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeUnboundedPrecedingCurrentRow) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRangeUnboundedPrecedingCurrentRowExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeUnboundedPrecedingCurrentRowExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeUnboundedPrecedingCurrentRowExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeUnboundedPrecedingCurrentRowExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeUnboundedPrecedingCurrentRowExclude) close() {
	*f = windowFramerRangeUnboundedPrecedingCurrentRowExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeUnboundedPrecedingCurrentRowExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRangeUnboundedPrecedingCurrentRowExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRangeUnboundedPrecedingCurrentRowExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRangeUnboundedPrecedingCurrentRowExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRangeUnboundedPrecedingCurrentRowExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRangeUnboundedPrecedingOffsetFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeUnboundedPrecedingOffsetFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeUnboundedPrecedingOffsetFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.endHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeUnboundedPrecedingOffsetFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.endHandler.getIdx(ctx, f.currentRow, f.endIdx)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeUnboundedPrecedingOffsetFollowing) close() {
	f.endHandler.close()
	*f = windowFramerRangeUnboundedPrecedingOffsetFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeUnboundedPrecedingOffsetFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRangeUnboundedPrecedingOffsetFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeUnboundedPrecedingOffsetFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeUnboundedPrecedingOffsetFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.endHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeUnboundedPrecedingOffsetFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.endHandler.getIdx(ctx, f.currentRow, f.endIdx)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeUnboundedPrecedingOffsetFollowingExclude) close() {
	f.endHandler.close()
	*f = windowFramerRangeUnboundedPrecedingOffsetFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeUnboundedPrecedingOffsetFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRangeUnboundedPrecedingOffsetFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRangeUnboundedPrecedingOffsetFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRangeUnboundedPrecedingOffsetFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRangeUnboundedPrecedingOffsetFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRangeUnboundedPrecedingUnboundedFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeUnboundedPrecedingUnboundedFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeUnboundedPrecedingUnboundedFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeUnboundedPrecedingUnboundedFollowing) next(ctx context.Context) {
	f.currentRow++
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	f.endIdx = f.partitionSize
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeUnboundedPrecedingUnboundedFollowing) close() {
	*f = windowFramerRangeUnboundedPrecedingUnboundedFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeUnboundedPrecedingUnboundedFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRangeUnboundedPrecedingUnboundedFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeUnboundedPrecedingUnboundedFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeUnboundedPrecedingUnboundedFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeUnboundedPrecedingUnboundedFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	f.startIdx = 0

	// Handle the end bound.
	f.endIdx = f.partitionSize
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeUnboundedPrecedingUnboundedFollowingExclude) close() {
	*f = windowFramerRangeUnboundedPrecedingUnboundedFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeUnboundedPrecedingUnboundedFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRangeUnboundedPrecedingUnboundedFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRangeUnboundedPrecedingUnboundedFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRangeUnboundedPrecedingUnboundedFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRangeUnboundedPrecedingUnboundedFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRangeOffsetPrecedingOffsetPreceding struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeOffsetPrecedingOffsetPreceding{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeOffsetPrecedingOffsetPreceding) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.startHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.endHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeOffsetPrecedingOffsetPreceding) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.startHandler.getIdx(ctx, f.currentRow, f.startIdx)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.endHandler.getIdx(ctx, f.currentRow, f.endIdx)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeOffsetPrecedingOffsetPreceding) close() {
	f.startHandler.close()
	f.endHandler.close()
	*f = windowFramerRangeOffsetPrecedingOffsetPreceding{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeOffsetPrecedingOffsetPreceding) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRangeOffsetPrecedingOffsetPrecedingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeOffsetPrecedingOffsetPrecedingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeOffsetPrecedingOffsetPrecedingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.startHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.endHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeOffsetPrecedingOffsetPrecedingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.startHandler.getIdx(ctx, f.currentRow, f.startIdx)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.endHandler.getIdx(ctx, f.currentRow, f.endIdx)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeOffsetPrecedingOffsetPrecedingExclude) close() {
	f.startHandler.close()
	f.endHandler.close()
	*f = windowFramerRangeOffsetPrecedingOffsetPrecedingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeOffsetPrecedingOffsetPrecedingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRangeOffsetPrecedingOffsetPrecedingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRangeOffsetPrecedingOffsetPrecedingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRangeOffsetPrecedingOffsetPrecedingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRangeOffsetPrecedingOffsetPrecedingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRangeOffsetPrecedingCurrentRow struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeOffsetPrecedingCurrentRow{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeOffsetPrecedingCurrentRow) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.startHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeOffsetPrecedingCurrentRow) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.startHandler.getIdx(ctx, f.currentRow, f.startIdx)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeOffsetPrecedingCurrentRow) close() {
	f.startHandler.close()
	*f = windowFramerRangeOffsetPrecedingCurrentRow{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeOffsetPrecedingCurrentRow) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRangeOffsetPrecedingCurrentRowExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeOffsetPrecedingCurrentRowExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeOffsetPrecedingCurrentRowExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.startHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeOffsetPrecedingCurrentRowExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.startHandler.getIdx(ctx, f.currentRow, f.startIdx)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeOffsetPrecedingCurrentRowExclude) close() {
	f.startHandler.close()
	*f = windowFramerRangeOffsetPrecedingCurrentRowExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeOffsetPrecedingCurrentRowExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRangeOffsetPrecedingCurrentRowExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRangeOffsetPrecedingCurrentRowExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRangeOffsetPrecedingCurrentRowExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRangeOffsetPrecedingCurrentRowExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRangeOffsetPrecedingOffsetFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeOffsetPrecedingOffsetFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeOffsetPrecedingOffsetFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.startHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.endHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeOffsetPrecedingOffsetFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.startHandler.getIdx(ctx, f.currentRow, f.startIdx)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.endHandler.getIdx(ctx, f.currentRow, f.endIdx)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeOffsetPrecedingOffsetFollowing) close() {
	f.startHandler.close()
	f.endHandler.close()
	*f = windowFramerRangeOffsetPrecedingOffsetFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeOffsetPrecedingOffsetFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRangeOffsetPrecedingOffsetFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeOffsetPrecedingOffsetFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeOffsetPrecedingOffsetFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.startHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.endHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeOffsetPrecedingOffsetFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.startHandler.getIdx(ctx, f.currentRow, f.startIdx)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.endHandler.getIdx(ctx, f.currentRow, f.endIdx)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeOffsetPrecedingOffsetFollowingExclude) close() {
	f.startHandler.close()
	f.endHandler.close()
	*f = windowFramerRangeOffsetPrecedingOffsetFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeOffsetPrecedingOffsetFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRangeOffsetPrecedingOffsetFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRangeOffsetPrecedingOffsetFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRangeOffsetPrecedingOffsetFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRangeOffsetPrecedingOffsetFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRangeOffsetPrecedingUnboundedFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeOffsetPrecedingUnboundedFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeOffsetPrecedingUnboundedFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.startHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeOffsetPrecedingUnboundedFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.startHandler.getIdx(ctx, f.currentRow, f.startIdx)
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeOffsetPrecedingUnboundedFollowing) close() {
	f.startHandler.close()
	*f = windowFramerRangeOffsetPrecedingUnboundedFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeOffsetPrecedingUnboundedFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRangeOffsetPrecedingUnboundedFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeOffsetPrecedingUnboundedFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeOffsetPrecedingUnboundedFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.startHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeOffsetPrecedingUnboundedFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.startHandler.getIdx(ctx, f.currentRow, f.startIdx)
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeOffsetPrecedingUnboundedFollowingExclude) close() {
	f.startHandler.close()
	*f = windowFramerRangeOffsetPrecedingUnboundedFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeOffsetPrecedingUnboundedFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRangeOffsetPrecedingUnboundedFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRangeOffsetPrecedingUnboundedFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRangeOffsetPrecedingUnboundedFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRangeOffsetPrecedingUnboundedFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRangeCurrentRowCurrentRow struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeCurrentRowCurrentRow{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeCurrentRowCurrentRow) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeCurrentRowCurrentRow) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.currentRow
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeCurrentRowCurrentRow) close() {
	*f = windowFramerRangeCurrentRowCurrentRow{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeCurrentRowCurrentRow) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRangeCurrentRowCurrentRowExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeCurrentRowCurrentRowExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeCurrentRowCurrentRowExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeCurrentRowCurrentRowExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.currentRow
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.incrementPeerGroup(ctx, f.endIdx, 1 /* groups */)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeCurrentRowCurrentRowExclude) close() {
	*f = windowFramerRangeCurrentRowCurrentRowExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeCurrentRowCurrentRowExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRangeCurrentRowCurrentRowExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRangeCurrentRowCurrentRowExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRangeCurrentRowCurrentRowExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRangeCurrentRowCurrentRowExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRangeCurrentRowOffsetFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeCurrentRowOffsetFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeCurrentRowOffsetFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.endHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeCurrentRowOffsetFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.currentRow
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.endHandler.getIdx(ctx, f.currentRow, f.endIdx)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeCurrentRowOffsetFollowing) close() {
	f.endHandler.close()
	*f = windowFramerRangeCurrentRowOffsetFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeCurrentRowOffsetFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRangeCurrentRowOffsetFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeCurrentRowOffsetFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeCurrentRowOffsetFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.endHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeCurrentRowOffsetFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.currentRow
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.endHandler.getIdx(ctx, f.currentRow, f.endIdx)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeCurrentRowOffsetFollowingExclude) close() {
	f.endHandler.close()
	*f = windowFramerRangeCurrentRowOffsetFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeCurrentRowOffsetFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRangeCurrentRowOffsetFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRangeCurrentRowOffsetFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRangeCurrentRowOffsetFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRangeCurrentRowOffsetFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRangeCurrentRowUnboundedFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeCurrentRowUnboundedFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeCurrentRowUnboundedFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeCurrentRowUnboundedFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.currentRow
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeCurrentRowUnboundedFollowing) close() {
	*f = windowFramerRangeCurrentRowUnboundedFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeCurrentRowUnboundedFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRangeCurrentRowUnboundedFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeCurrentRowUnboundedFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeCurrentRowUnboundedFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeCurrentRowUnboundedFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.currentRow
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeCurrentRowUnboundedFollowingExclude) close() {
	*f = windowFramerRangeCurrentRowUnboundedFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeCurrentRowUnboundedFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRangeCurrentRowUnboundedFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRangeCurrentRowUnboundedFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRangeCurrentRowUnboundedFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRangeCurrentRowUnboundedFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRangeOffsetFollowingOffsetFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeOffsetFollowingOffsetFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeOffsetFollowingOffsetFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.startHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.endHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeOffsetFollowingOffsetFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.startHandler.getIdx(ctx, f.currentRow, f.startIdx)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.endHandler.getIdx(ctx, f.currentRow, f.endIdx)
	}
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeOffsetFollowingOffsetFollowing) close() {
	f.startHandler.close()
	f.endHandler.close()
	*f = windowFramerRangeOffsetFollowingOffsetFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeOffsetFollowingOffsetFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRangeOffsetFollowingOffsetFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeOffsetFollowingOffsetFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeOffsetFollowingOffsetFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.startHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.endHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeOffsetFollowingOffsetFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.startHandler.getIdx(ctx, f.currentRow, f.startIdx)
	}

	// Handle the end bound.
	if currRowIsGroupStart {
		f.endIdx = f.endHandler.getIdx(ctx, f.currentRow, f.endIdx)
	}
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeOffsetFollowingOffsetFollowingExclude) close() {
	f.startHandler.close()
	f.endHandler.close()
	*f = windowFramerRangeOffsetFollowingOffsetFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeOffsetFollowingOffsetFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRangeOffsetFollowingOffsetFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRangeOffsetFollowingOffsetFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRangeOffsetFollowingOffsetFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRangeOffsetFollowingOffsetFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}

type windowFramerRangeOffsetFollowingUnboundedFollowing struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeOffsetFollowingUnboundedFollowing{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeOffsetFollowingUnboundedFollowing) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.startHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeOffsetFollowingUnboundedFollowing) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.startHandler.getIdx(ctx, f.currentRow, f.startIdx)
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeOffsetFollowingUnboundedFollowing) close() {
	f.startHandler.close()
	*f = windowFramerRangeOffsetFollowingUnboundedFollowing{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeOffsetFollowingUnboundedFollowing) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

type windowFramerRangeOffsetFollowingUnboundedFollowingExclude struct {
	windowFramerBase
}

var _ windowFramer = &windowFramerRangeOffsetFollowingUnboundedFollowingExclude{}

// startPartition prepares the window framer to begin iterating through a new
// partition. It must be called before calling next.
func (f *windowFramerRangeOffsetFollowingUnboundedFollowingExclude) startPartition(
	ctx context.Context, partitionSize int, storedCols *colexecutils.SpillingBuffer,
) {
	f.windowFramerBase.startPartition(partitionSize, storedCols)
	f.startHandler.startPartition(storedCols, f.peersColIdx, f.ordColIdx)
	f.excludeStartIdx = 0
	f.excludeEndIdx = 0
}

// next is called for each row in the partition. It advances to the next row and
// then calculates the window frame for the current row. next should not be
// called beyond the end of the partition, or undefined behavior may result.
func (f *windowFramerRangeOffsetFollowingUnboundedFollowingExclude) next(ctx context.Context) {
	f.currentRow++
	currRowIsGroupStart := f.isFirstPeer(ctx, f.currentRow)
	// Handle the start bound.
	if currRowIsGroupStart {
		f.startIdx = f.startHandler.getIdx(ctx, f.currentRow, f.startIdx)
	}

	// Handle the end bound.
	f.endIdx = f.partitionSize
	// Handle exclusion clause.
	f.handleExcludeForNext(ctx, currRowIsGroupStart)
	f.prevIntervals = append(f.prevIntervals[:0], f.intervals...)
	f.intervalsAreSet = false
}

func (f *windowFramerRangeOffsetFollowingUnboundedFollowingExclude) close() {
	f.startHandler.close()
	*f = windowFramerRangeOffsetFollowingUnboundedFollowingExclude{}
}

// slidingWindowIntervals returns a pair of interval sets that describes the
// rows that should be added to the current aggregation, and those which
// should be removed from the current aggregation. It is used to implement the
// sliding window optimization for aggregate window functions.
func (f *windowFramerRangeOffsetFollowingUnboundedFollowingExclude) slidingWindowIntervals() (toAdd, toRemove []windowInterval) {
	f.toAdd, f.toRemove = f.toAdd[:0], f.toRemove[:0]
	f.frameIntervals()
	f.toAdd, f.toRemove = getSlidingWindowIntervals(f.intervals, f.prevIntervals, f.toAdd, f.toRemove)
	return f.toAdd, f.toRemove
}

// frameFirstIdx returns the index of the first row in the window frame for
// the current row. If no such row exists, frameFirstIdx returns -1.
func (f *windowFramerRangeOffsetFollowingUnboundedFollowingExclude) frameFirstIdx() (idx int) {
	idx = f.windowFramerBase.frameFirstIdx()
	return f.handleExcludeForFirstIdx(idx)
}

// frameLastIdx returns the index of the last row in the window frame for
// the current row. If no such row exists, frameLastIdx returns -1.
func (f *windowFramerRangeOffsetFollowingUnboundedFollowingExclude) frameLastIdx() (idx int) {
	idx = f.windowFramerBase.frameLastIdx()
	return f.handleExcludeForLastIdx(idx)
}

// frameNthIdx returns the index of the nth row (starting from one) in the
// window frame for the current row. If no such row exists, frameNthIdx
// returns -1.
func (f *windowFramerRangeOffsetFollowingUnboundedFollowingExclude) frameNthIdx(n int) (idx int) {
	idx = f.windowFramerBase.frameNthIdx(n)
	return f.handleExcludeForNthIdx(idx)
}

// frameIntervals returns a series of intervals that describes the set of all
// rows that are part of the frame for the current row. Note that there are at
// most three intervals - this case can occur when EXCLUDE TIES is used.
// frameIntervals is used to compute aggregate functions over a window.
func (f *windowFramerRangeOffsetFollowingUnboundedFollowingExclude) frameIntervals() []windowInterval {
	if f.startIdx >= f.endIdx {
		f.intervals = f.intervals[:0]
		return f.intervals
	}
	if f.excludeStartIdx >= f.endIdx || f.excludeEndIdx <= f.startIdx {
		// No rows excluded.
		return f.windowFramerBase.frameIntervals()
	}
	f.intervals = f.intervals[:0]
	if f.excludeStartIdx > f.startIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.startIdx, end: f.excludeStartIdx})
	}
	if f.excludeTies() && f.currentRow >= f.startIdx && f.currentRow < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.currentRow, end: f.currentRow + 1})
	}
	if f.excludeEndIdx < f.endIdx {
		f.intervals = append(f.intervals, windowInterval{start: f.excludeEndIdx, end: f.endIdx})
	}
	return f.intervals
}
