// Code generated by execgen; DO NOT EDIT.
// Copyright 2020 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package colconv

import (
	"sync"

	"github.com/cockroachdb/cockroach/pkg/col/coldata"
	"github.com/cockroachdb/cockroach/pkg/col/typeconv"
	"github.com/cockroachdb/cockroach/pkg/sql/colexecerror"
	"github.com/cockroachdb/cockroach/pkg/sql/execinfra/execreleasable"
	"github.com/cockroachdb/cockroach/pkg/sql/sem/tree"
	"github.com/cockroachdb/cockroach/pkg/sql/types"
	"github.com/cockroachdb/cockroach/pkg/util/json"
	"github.com/cockroachdb/cockroach/pkg/util/timeutil/pgdate"
	"github.com/cockroachdb/cockroach/pkg/util/uuid"
	"github.com/lib/pq/oid"
)

// Workaround for bazel auto-generated code. goimports does not automatically
// pick up the right packages when run within the bazel sandbox.
var (
	_ colexecerror.StorageError
	_ json.JSON
	_ pgdate.Date
	_ = typeconv.DatumVecCanonicalTypeFamily
	_ uuid.UUID
)

// VecToDatumConverter is a helper struct that converts vectors from batches to
// their datum representations.
// TODO(yuzefovich): the result of converting the vectors to datums is usually
// put into rowenc.EncDatumRow, so it might make sense to look into creating
// a converter that would store EncDatumRows directly. I prototyped such
// converter, but it showed worse performance both in the microbenchmarks and
// some of the TPCH queries. I think the main reason for the slowdown is that
// the amount of memory allocated increases just because EncDatums take more
// space than Datums. Another thing is that allocating whole W vectors, one
// vector at a time, in VecToDatumConverter is noticeably faster that
// allocating N rows of W length, one row at a time (meaning that
// O(batch width) vs O(batch length) comparison). We could probably play around
// with allocating a big flat []EncDatum slice in which datums from the same
// column are contiguous and then populate the output row in the materializer
// by choosing appropriate elements, but I'm not sure whether it would be more
// performant.
type VecToDatumConverter struct {
	convertedVecs    []tree.Datums
	vecIdxsToConvert []int
	da               tree.DatumAlloc
}

var _ execreleasable.Releasable = &VecToDatumConverter{}

var vecToDatumConverterPool = sync.Pool{
	New: func() interface{} {
		return &VecToDatumConverter{}
	},
}

// getNewVecToDatumConverter returns a new VecToDatumConverter that is able to
// handle batchWidth number of columns. willRelease indicates whether the caller
// will call Release() on the converter.
func getNewVecToDatumConverter(batchWidth int, willRelease bool) *VecToDatumConverter {
	var c *VecToDatumConverter
	// Having willRelease knob (i.e. not defaulting to using the pool all the
	// time) is justified by the following scenario: there is constant workload
	// running against the cluster (e.g. TPCC) that has the same "access
	// pattern", so all converters used by that workload can be reused very
	// effectively - the width of the batches are the same, etc. However, when a
	// random query comes in and picks up a converter from the sync.Pool, it'll
	// use it once and discard it afterwards. It is likely that for this random
	// query it's better to allocate a fresh converter and not touch the ones in
	// the pool.
	if willRelease {
		c = vecToDatumConverterPool.Get().(*VecToDatumConverter)
	} else {
		c = &VecToDatumConverter{}
	}
	if cap(c.convertedVecs) < batchWidth {
		c.convertedVecs = make([]tree.Datums, batchWidth)
	} else {
		c.convertedVecs = c.convertedVecs[:batchWidth]
	}
	return c
}

// NewVecToDatumConverter creates a new VecToDatumConverter.
//   - batchWidth determines the width of the batches that it will be converting.
//   - vecIdxsToConvert determines which vectors need to be converted.
//   - willRelease indicates whether the caller intends to call Release() on the
//     converter.
func NewVecToDatumConverter(
	batchWidth int, vecIdxsToConvert []int, willRelease bool,
) *VecToDatumConverter {
	c := getNewVecToDatumConverter(batchWidth, willRelease)
	c.vecIdxsToConvert = vecIdxsToConvert
	return c
}

// NewAllVecToDatumConverter is like NewVecToDatumConverter except all of the
// vectors in the batch will be converted.
// NOTE: it is assumed that the caller will Release the returned converter.
func NewAllVecToDatumConverter(batchWidth int) *VecToDatumConverter {
	c := getNewVecToDatumConverter(batchWidth, true /* willRelease */)
	if cap(c.vecIdxsToConvert) < batchWidth {
		c.vecIdxsToConvert = make([]int, batchWidth)
	} else {
		c.vecIdxsToConvert = c.vecIdxsToConvert[:batchWidth]
	}
	for i := 0; i < batchWidth; i++ {
		c.vecIdxsToConvert[i] = i
	}
	return c
}

// Release is part of the execinfra.Releasable interface.
func (c *VecToDatumConverter) Release() {
	// Deeply reset the converted vectors so that we don't hold onto the old
	// datums.
	for _, vec := range c.convertedVecs {
		for i := range vec {
			//gcassert:bce
			vec[i] = nil
		}
	}
	*c = VecToDatumConverter{
		convertedVecs: c.convertedVecs[:0],
		// This slice is of integers, so there is no need to reset it deeply.
		vecIdxsToConvert: c.vecIdxsToConvert[:0],
	}
	vecToDatumConverterPool.Put(c)
}

// ConvertBatchAndDeselect converts the selected vectors from the batch while
// performing a deselection step. It doesn't account for the memory used by the
// newly created tree.Datums, so it is up to the caller to do the memory
// accounting.
// NOTE: converted columns are "dense" in regards to the selection vector - if
// there was a selection vector on the batch, only elements that were selected
// are converted, so in order to access the tuple at position tupleIdx, use
// GetDatumColumn(colIdx)[tupleIdx] and *NOT*
// GetDatumColumn(colIdx)[sel[tupleIdx]].
func (c *VecToDatumConverter) ConvertBatchAndDeselect(batch coldata.Batch) {
	if len(c.vecIdxsToConvert) == 0 {
		// No vectors were selected for conversion, so there is nothing to do.
		return
	}
	batchLength := batch.Length()
	if batchLength == 0 {
		return
	}
	// Ensure that convertedVecs are of sufficient length.
	for _, vecIdx := range c.vecIdxsToConvert {
		if cap(c.convertedVecs[vecIdx]) < batchLength {
			c.convertedVecs[vecIdx] = make([]tree.Datum, batchLength)
		} else {
			c.convertedVecs[vecIdx] = c.convertedVecs[vecIdx][:batchLength]
		}
	}
	if c.da.DefaultAllocSize < batchLength {
		// Adjust the datum alloc according to the length of the batch since
		// this batch is the longest we've seen so far.
		c.da.DefaultAllocSize = batchLength
	}
	vecs := batch.ColVecs()
	c.da.ResetTypeAllocSizes()
	for _, vecIdx := range c.vecIdxsToConvert {
		// Provide the datum allocator with hints about the number of
		// allocations for each type.
		c.da.AddTypeAllocSize(batchLength, vecs[vecIdx].Type().Family())
	}
	sel := batch.Selection()
	for _, vecIdx := range c.vecIdxsToConvert {
		ColVecToDatumAndDeselect(
			c.convertedVecs[vecIdx], vecs[vecIdx], batchLength, sel, &c.da,
		)
	}
}

// ConvertBatch converts the selected vectors from the batch *without*
// performing a deselection step. It doesn't account for the memory used by the
// newly created tree.Datums, so it is up to the caller to do the memory
// accounting.
// NOTE: converted columns are "sparse" in regards to the selection vector - if
// there was a selection vector, only elements that were selected are
// converted, but the results are put at position sel[tupleIdx], so use
// GetDatumColumn(colIdx)[sel[tupleIdx]] and *NOT*
// GetDatumColumn(colIdx)[tupleIdx].
func (c *VecToDatumConverter) ConvertBatch(batch coldata.Batch) {
	if c == nil {
		// If the converter is nil, then it wasn't allocated because there are
		// no vectors to convert, so exit early.
		return
	}
	c.ConvertVecs(batch.ColVecs(), batch.Length(), batch.Selection())
}

// ConvertVecs converts the selected vectors from vecs *without* performing a
// deselection step. It doesn't account for the memory used by the newly
// created tree.Datums, so it is up to the caller to do the memory accounting.
// Note that this method is equivalent to ConvertBatch with the only difference
// being the fact that it takes in a "disassembled" batch and not coldata.Batch.
// Consider whether you should be using ConvertBatch instead.
func (c *VecToDatumConverter) ConvertVecs(vecs []*coldata.Vec, inputLen int, sel []int) {
	if len(c.vecIdxsToConvert) == 0 || inputLen == 0 {
		// No vectors were selected for conversion or there are no tuples to
		// convert, so there is nothing to do.
		return
	}
	// Ensure that convertedVecs are of sufficient length.
	requiredLength := inputLen
	if sel != nil {
		// When sel is non-nil, it might be something like sel = [1023], so we
		// need to allocate up to the largest index mentioned in sel. Here, we
		// rely on the fact that selection vectors are increasing sequences.
		requiredLength = sel[inputLen-1] + 1
	}
	for _, vecIdx := range c.vecIdxsToConvert {
		if cap(c.convertedVecs[vecIdx]) < requiredLength {
			c.convertedVecs[vecIdx] = make([]tree.Datum, requiredLength)
		} else {
			c.convertedVecs[vecIdx] = c.convertedVecs[vecIdx][:requiredLength]
		}
	}
	if c.da.DefaultAllocSize < requiredLength {
		// Adjust the datum alloc according to the length of the batch since
		// this batch is the longest we've seen so far.
		c.da.DefaultAllocSize = requiredLength
	}
	c.da.ResetTypeAllocSizes()
	for _, vecIdx := range c.vecIdxsToConvert {
		// Provide the datum allocator with hints about the number of
		// allocations for each type.
		c.da.AddTypeAllocSize(inputLen, vecs[vecIdx].Type().Family())
	}
	for _, vecIdx := range c.vecIdxsToConvert {
		ColVecToDatum(
			c.convertedVecs[vecIdx], vecs[vecIdx], inputLen, sel, &c.da,
		)
	}
}

// GetDatumColumn returns the converted column of tree.Datum of the vector on
// position colIdx from the last converted batch.
func (c *VecToDatumConverter) GetDatumColumn(colIdx int) tree.Datums {
	return c.convertedVecs[colIdx]
}

// ColVecToDatumAndDeselect converts a vector of coldata-represented values in
// col into tree.Datum representation while performing a deselection step.
// length specifies the number of values to be converted and sel is an optional
// selection vector. It doesn't account for the memory used by the newly
// created tree.Datums, so it is up to the caller to do the memory accounting.
func ColVecToDatumAndDeselect(
	converted []tree.Datum, col *coldata.Vec, length int, sel []int, da *tree.DatumAlloc,
) {
	if length == 0 {
		return
	}
	if sel == nil {
		ColVecToDatum(converted, col, length, sel, da)
		return
	}
	if col.MaybeHasNulls() {
		nulls := col.Nulls()
		{
			_ = converted[length-1]
			_ = sel[length-1]
			var idx, destIdx, srcIdx int
			switch ct := col.Type(); ct.Family() {
			case types.StringFamily:
				// Note that there is no need for a copy since casting to a string will
				// do that.
				bytes := col.Bytes()
				if ct.Oid() == oid.T_name {
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						v := da.NewDName(tree.DString(bytes.Get(srcIdx)))
						//gcassert:bce
						converted[destIdx] = v
					}
					goto vecToDatum_true_true_true_return_0
				}
				for idx = 0; idx < length; idx++ {
					{
						destIdx = idx
					}
					{
						//gcassert:bce
						srcIdx = sel[idx]
					}
					if nulls.NullAt(srcIdx) {
						//gcassert:bce
						converted[destIdx] = tree.DNull
						continue
					}
					v := da.NewDString(tree.DString(bytes.Get(srcIdx)))
					//gcassert:bce
					converted[destIdx] = v
				}
			case types.BoolFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Bool()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := tree.MakeDBool(tree.DBool(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.IntFamily:
				switch ct.Width() {
				case 16:
					typedCol := col.Int16()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDInt(tree.DInt(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				case 32:
					typedCol := col.Int32()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDInt(tree.DInt(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				case -1:
				default:
					typedCol := col.Int64()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDInt(tree.DInt(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.FloatFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Float64()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDFloat(tree.DFloat(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.DecimalFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Decimal()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDDecimal(tree.DDecimal{Decimal: v})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.DateFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Int64()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDDate(tree.DDate{Date: pgdate.MakeCompatibleDateFromDisk(v)})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.BytesFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Bytes()
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						v := typedCol.Get(srcIdx)
						// Note that there is no need for a copy since DBytes uses a string
						// as underlying storage, which will perform the copy for us.
						_converted := da.NewDBytes(tree.DBytes(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.EncodedKeyFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Bytes()
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						v := typedCol.Get(srcIdx)
						// Note that there is no need for a copy since DEncodedKey uses a string
						// as underlying storage, which will perform the copy for us.
						_converted := da.NewDEncodedKey(tree.DEncodedKey(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.JsonFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.JSON()
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						v := typedCol.Get(srcIdx)

						// The following operation deliberately copies the input JSON
						// bytes, since FromEncoding is lazy and keeps a handle on the bytes
						// it is passed in.
						_bytes, _err := json.EncodeJSON(nil, v)
						if _err != nil {
							colexecerror.ExpectedError(_err)
						}
						var _j json.JSON
						_j, _err = json.FromEncoding(_bytes)
						if _err != nil {
							colexecerror.ExpectedError(_err)
						}
						_converted := da.NewDJSON(tree.DJSON{JSON: _j})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.UuidFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Bytes()
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						v := typedCol.Get(srcIdx)
						// Note that there is no need for a copy because uuid.FromBytes
						// will perform a copy.
						id, err := uuid.FromBytes(v)
						if err != nil {
							colexecerror.InternalError(err)
						}
						_converted := da.NewDUuid(tree.DUuid{UUID: id})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.TimestampFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Timestamp()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDTimestamp(tree.DTimestamp{Time: v})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.TimestampTZFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Timestamp()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDTimestampTZ(tree.DTimestampTZ{Time: v})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.IntervalFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Interval()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDInterval(tree.DInterval{Duration: v})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.EnumFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Bytes()
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						v := typedCol.Get(srcIdx)
						e, err := tree.MakeDEnumFromPhysicalRepresentation(ct, v)
						if err != nil {
							colexecerror.InternalError(err)
						}
						_converted := da.NewDEnum(e)
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case typeconv.DatumVecCanonicalTypeFamily:
			default:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Datum()
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						v := typedCol.Get(srcIdx)
						_converted := v.(tree.Datum)
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			}
		vecToDatum_true_true_true_return_0:
		}
	} else {
		{
			_ = converted[length-1]
			_ = sel[length-1]
			var idx, destIdx, srcIdx int
			switch ct := col.Type(); ct.Family() {
			case types.StringFamily:
				// Note that there is no need for a copy since casting to a string will
				// do that.
				bytes := col.Bytes()
				if ct.Oid() == oid.T_name {
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						v := da.NewDName(tree.DString(bytes.Get(srcIdx)))
						//gcassert:bce
						converted[destIdx] = v
					}
					goto vecToDatum_false_true_true_return_1
				}
				for idx = 0; idx < length; idx++ {
					{
						destIdx = idx
					}
					{
						//gcassert:bce
						srcIdx = sel[idx]
					}
					v := da.NewDString(tree.DString(bytes.Get(srcIdx)))
					//gcassert:bce
					converted[destIdx] = v
				}
			case types.BoolFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Bool()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := tree.MakeDBool(tree.DBool(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.IntFamily:
				switch ct.Width() {
				case 16:
					typedCol := col.Int16()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDInt(tree.DInt(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				case 32:
					typedCol := col.Int32()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDInt(tree.DInt(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				case -1:
				default:
					typedCol := col.Int64()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDInt(tree.DInt(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.FloatFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Float64()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDFloat(tree.DFloat(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.DecimalFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Decimal()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDDecimal(tree.DDecimal{Decimal: v})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.DateFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Int64()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDDate(tree.DDate{Date: pgdate.MakeCompatibleDateFromDisk(v)})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.BytesFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Bytes()
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						v := typedCol.Get(srcIdx)
						// Note that there is no need for a copy since DBytes uses a string
						// as underlying storage, which will perform the copy for us.
						_converted := da.NewDBytes(tree.DBytes(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.EncodedKeyFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Bytes()
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						v := typedCol.Get(srcIdx)
						// Note that there is no need for a copy since DEncodedKey uses a string
						// as underlying storage, which will perform the copy for us.
						_converted := da.NewDEncodedKey(tree.DEncodedKey(v))
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.JsonFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.JSON()
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						v := typedCol.Get(srcIdx)

						// The following operation deliberately copies the input JSON
						// bytes, since FromEncoding is lazy and keeps a handle on the bytes
						// it is passed in.
						_bytes, _err := json.EncodeJSON(nil, v)
						if _err != nil {
							colexecerror.ExpectedError(_err)
						}
						var _j json.JSON
						_j, _err = json.FromEncoding(_bytes)
						if _err != nil {
							colexecerror.ExpectedError(_err)
						}
						_converted := da.NewDJSON(tree.DJSON{JSON: _j})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.UuidFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Bytes()
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						v := typedCol.Get(srcIdx)
						// Note that there is no need for a copy because uuid.FromBytes
						// will perform a copy.
						id, err := uuid.FromBytes(v)
						if err != nil {
							colexecerror.InternalError(err)
						}
						_converted := da.NewDUuid(tree.DUuid{UUID: id})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.TimestampFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Timestamp()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDTimestamp(tree.DTimestamp{Time: v})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.TimestampTZFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Timestamp()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDTimestampTZ(tree.DTimestampTZ{Time: v})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.IntervalFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Interval()
					_ = true
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						_ = true
						v := typedCol.Get(srcIdx)
						_converted := da.NewDInterval(tree.DInterval{Duration: v})
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case types.EnumFamily:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Bytes()
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						v := typedCol.Get(srcIdx)
						e, err := tree.MakeDEnumFromPhysicalRepresentation(ct, v)
						if err != nil {
							colexecerror.InternalError(err)
						}
						_converted := da.NewDEnum(e)
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			case typeconv.DatumVecCanonicalTypeFamily:
			default:
				switch ct.Width() {
				case -1:
				default:
					typedCol := col.Datum()
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						v := typedCol.Get(srcIdx)
						_converted := v.(tree.Datum)
						//gcassert:bce
						converted[destIdx] = _converted
					}
				}
			}
		vecToDatum_false_true_true_return_1:
		}
	}
}

// ColVecToDatum converts a vector of coldata-represented values in col into
// tree.Datum representation *without* performing a deselection step. It
// doesn't account for the memory used by the newly created tree.Datums, so it
// is up to the caller to do the memory accounting.
func ColVecToDatum(
	converted []tree.Datum, col *coldata.Vec, length int, sel []int, da *tree.DatumAlloc,
) {
	if length == 0 {
		return
	}
	if col.MaybeHasNulls() {
		nulls := col.Nulls()
		if sel != nil {
			{
				_ = sel[length-1]
				var idx, destIdx, srcIdx int
				switch ct := col.Type(); ct.Family() {
				case types.StringFamily:
					// Note that there is no need for a copy since casting to a string will
					// do that.
					bytes := col.Bytes()
					if ct.Oid() == oid.T_name {
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							v := da.NewDName(tree.DString(bytes.Get(srcIdx)))
							converted[destIdx] = v
						}
						goto vecToDatum_true_true_false_return_2
					}
					for idx = 0; idx < length; idx++ {
						{
							//gcassert:bce
							destIdx = sel[idx]
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						if nulls.NullAt(srcIdx) {
							converted[destIdx] = tree.DNull
							continue
						}
						v := da.NewDString(tree.DString(bytes.Get(srcIdx)))
						converted[destIdx] = v
					}
				case types.BoolFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bool()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := tree.MakeDBool(tree.DBool(v))
							converted[destIdx] = _converted
						}
					}
				case types.IntFamily:
					switch ct.Width() {
					case 16:
						typedCol := col.Int16()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInt(tree.DInt(v))
							converted[destIdx] = _converted
						}
					case 32:
						typedCol := col.Int32()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInt(tree.DInt(v))
							converted[destIdx] = _converted
						}
					case -1:
					default:
						typedCol := col.Int64()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInt(tree.DInt(v))
							converted[destIdx] = _converted
						}
					}
				case types.FloatFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Float64()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDFloat(tree.DFloat(v))
							converted[destIdx] = _converted
						}
					}
				case types.DecimalFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Decimal()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDDecimal(tree.DDecimal{Decimal: v})
							converted[destIdx] = _converted
						}
					}
				case types.DateFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Int64()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDDate(tree.DDate{Date: pgdate.MakeCompatibleDateFromDisk(v)})
							converted[destIdx] = _converted
						}
					}
				case types.BytesFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							v := typedCol.Get(srcIdx)
							// Note that there is no need for a copy since DBytes uses a string
							// as underlying storage, which will perform the copy for us.
							_converted := da.NewDBytes(tree.DBytes(v))
							converted[destIdx] = _converted
						}
					}
				case types.EncodedKeyFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							v := typedCol.Get(srcIdx)
							// Note that there is no need for a copy since DEncodedKey uses a string
							// as underlying storage, which will perform the copy for us.
							_converted := da.NewDEncodedKey(tree.DEncodedKey(v))
							converted[destIdx] = _converted
						}
					}
				case types.JsonFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.JSON()
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							v := typedCol.Get(srcIdx)

							// The following operation deliberately copies the input JSON
							// bytes, since FromEncoding is lazy and keeps a handle on the bytes
							// it is passed in.
							_bytes, _err := json.EncodeJSON(nil, v)
							if _err != nil {
								colexecerror.ExpectedError(_err)
							}
							var _j json.JSON
							_j, _err = json.FromEncoding(_bytes)
							if _err != nil {
								colexecerror.ExpectedError(_err)
							}
							_converted := da.NewDJSON(tree.DJSON{JSON: _j})
							converted[destIdx] = _converted
						}
					}
				case types.UuidFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							v := typedCol.Get(srcIdx)
							// Note that there is no need for a copy because uuid.FromBytes
							// will perform a copy.
							id, err := uuid.FromBytes(v)
							if err != nil {
								colexecerror.InternalError(err)
							}
							_converted := da.NewDUuid(tree.DUuid{UUID: id})
							converted[destIdx] = _converted
						}
					}
				case types.TimestampFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Timestamp()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDTimestamp(tree.DTimestamp{Time: v})
							converted[destIdx] = _converted
						}
					}
				case types.TimestampTZFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Timestamp()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDTimestampTZ(tree.DTimestampTZ{Time: v})
							converted[destIdx] = _converted
						}
					}
				case types.IntervalFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Interval()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInterval(tree.DInterval{Duration: v})
							converted[destIdx] = _converted
						}
					}
				case types.EnumFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							v := typedCol.Get(srcIdx)
							e, err := tree.MakeDEnumFromPhysicalRepresentation(ct, v)
							if err != nil {
								colexecerror.InternalError(err)
							}
							_converted := da.NewDEnum(e)
							converted[destIdx] = _converted
						}
					}
				case typeconv.DatumVecCanonicalTypeFamily:
				default:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Datum()
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							if nulls.NullAt(srcIdx) {
								converted[destIdx] = tree.DNull
								continue
							}
							v := typedCol.Get(srcIdx)
							_converted := v.(tree.Datum)
							converted[destIdx] = _converted
						}
					}
				}
			vecToDatum_true_true_false_return_2:
			}
		} else {
			{
				_ = converted[length-1]
				var idx, destIdx, srcIdx int
				switch ct := col.Type(); ct.Family() {
				case types.StringFamily:
					// Note that there is no need for a copy since casting to a string will
					// do that.
					bytes := col.Bytes()
					if ct.Oid() == oid.T_name {
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							v := da.NewDName(tree.DString(bytes.Get(srcIdx)))
							//gcassert:bce
							converted[destIdx] = v
						}
						goto vecToDatum_true_false_false_return_3
					}
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							srcIdx = idx
						}
						if nulls.NullAt(srcIdx) {
							//gcassert:bce
							converted[destIdx] = tree.DNull
							continue
						}
						v := da.NewDString(tree.DString(bytes.Get(srcIdx)))
						//gcassert:bce
						converted[destIdx] = v
					}
				case types.BoolFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bool()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := tree.MakeDBool(tree.DBool(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.IntFamily:
					switch ct.Width() {
					case 16:
						typedCol := col.Int16()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInt(tree.DInt(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					case 32:
						typedCol := col.Int32()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInt(tree.DInt(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					case -1:
					default:
						typedCol := col.Int64()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInt(tree.DInt(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.FloatFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Float64()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDFloat(tree.DFloat(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.DecimalFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Decimal()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDDecimal(tree.DDecimal{Decimal: v})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.DateFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Int64()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDDate(tree.DDate{Date: pgdate.MakeCompatibleDateFromDisk(v)})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.BytesFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							v := typedCol.Get(srcIdx)
							// Note that there is no need for a copy since DBytes uses a string
							// as underlying storage, which will perform the copy for us.
							_converted := da.NewDBytes(tree.DBytes(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.EncodedKeyFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							v := typedCol.Get(srcIdx)
							// Note that there is no need for a copy since DEncodedKey uses a string
							// as underlying storage, which will perform the copy for us.
							_converted := da.NewDEncodedKey(tree.DEncodedKey(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.JsonFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.JSON()
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							v := typedCol.Get(srcIdx)

							// The following operation deliberately copies the input JSON
							// bytes, since FromEncoding is lazy and keeps a handle on the bytes
							// it is passed in.
							_bytes, _err := json.EncodeJSON(nil, v)
							if _err != nil {
								colexecerror.ExpectedError(_err)
							}
							var _j json.JSON
							_j, _err = json.FromEncoding(_bytes)
							if _err != nil {
								colexecerror.ExpectedError(_err)
							}
							_converted := da.NewDJSON(tree.DJSON{JSON: _j})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.UuidFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							v := typedCol.Get(srcIdx)
							// Note that there is no need for a copy because uuid.FromBytes
							// will perform a copy.
							id, err := uuid.FromBytes(v)
							if err != nil {
								colexecerror.InternalError(err)
							}
							_converted := da.NewDUuid(tree.DUuid{UUID: id})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.TimestampFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Timestamp()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDTimestamp(tree.DTimestamp{Time: v})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.TimestampTZFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Timestamp()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDTimestampTZ(tree.DTimestampTZ{Time: v})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.IntervalFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Interval()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInterval(tree.DInterval{Duration: v})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.EnumFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							v := typedCol.Get(srcIdx)
							e, err := tree.MakeDEnumFromPhysicalRepresentation(ct, v)
							if err != nil {
								colexecerror.InternalError(err)
							}
							_converted := da.NewDEnum(e)
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case typeconv.DatumVecCanonicalTypeFamily:
				default:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Datum()
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							if nulls.NullAt(srcIdx) {
								//gcassert:bce
								converted[destIdx] = tree.DNull
								continue
							}
							v := typedCol.Get(srcIdx)
							_converted := v.(tree.Datum)
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				}
			vecToDatum_true_false_false_return_3:
			}
		}
	} else {
		if sel != nil {
			{
				_ = sel[length-1]
				var idx, destIdx, srcIdx int
				switch ct := col.Type(); ct.Family() {
				case types.StringFamily:
					// Note that there is no need for a copy since casting to a string will
					// do that.
					bytes := col.Bytes()
					if ct.Oid() == oid.T_name {
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							v := da.NewDName(tree.DString(bytes.Get(srcIdx)))
							converted[destIdx] = v
						}
						goto vecToDatum_false_true_false_return_4
					}
					for idx = 0; idx < length; idx++ {
						{
							//gcassert:bce
							destIdx = sel[idx]
						}
						{
							//gcassert:bce
							srcIdx = sel[idx]
						}
						v := da.NewDString(tree.DString(bytes.Get(srcIdx)))
						converted[destIdx] = v
					}
				case types.BoolFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bool()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := tree.MakeDBool(tree.DBool(v))
							converted[destIdx] = _converted
						}
					}
				case types.IntFamily:
					switch ct.Width() {
					case 16:
						typedCol := col.Int16()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInt(tree.DInt(v))
							converted[destIdx] = _converted
						}
					case 32:
						typedCol := col.Int32()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInt(tree.DInt(v))
							converted[destIdx] = _converted
						}
					case -1:
					default:
						typedCol := col.Int64()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInt(tree.DInt(v))
							converted[destIdx] = _converted
						}
					}
				case types.FloatFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Float64()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDFloat(tree.DFloat(v))
							converted[destIdx] = _converted
						}
					}
				case types.DecimalFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Decimal()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDDecimal(tree.DDecimal{Decimal: v})
							converted[destIdx] = _converted
						}
					}
				case types.DateFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Int64()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDDate(tree.DDate{Date: pgdate.MakeCompatibleDateFromDisk(v)})
							converted[destIdx] = _converted
						}
					}
				case types.BytesFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							v := typedCol.Get(srcIdx)
							// Note that there is no need for a copy since DBytes uses a string
							// as underlying storage, which will perform the copy for us.
							_converted := da.NewDBytes(tree.DBytes(v))
							converted[destIdx] = _converted
						}
					}
				case types.EncodedKeyFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							v := typedCol.Get(srcIdx)
							// Note that there is no need for a copy since DEncodedKey uses a string
							// as underlying storage, which will perform the copy for us.
							_converted := da.NewDEncodedKey(tree.DEncodedKey(v))
							converted[destIdx] = _converted
						}
					}
				case types.JsonFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.JSON()
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							v := typedCol.Get(srcIdx)

							// The following operation deliberately copies the input JSON
							// bytes, since FromEncoding is lazy and keeps a handle on the bytes
							// it is passed in.
							_bytes, _err := json.EncodeJSON(nil, v)
							if _err != nil {
								colexecerror.ExpectedError(_err)
							}
							var _j json.JSON
							_j, _err = json.FromEncoding(_bytes)
							if _err != nil {
								colexecerror.ExpectedError(_err)
							}
							_converted := da.NewDJSON(tree.DJSON{JSON: _j})
							converted[destIdx] = _converted
						}
					}
				case types.UuidFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							v := typedCol.Get(srcIdx)
							// Note that there is no need for a copy because uuid.FromBytes
							// will perform a copy.
							id, err := uuid.FromBytes(v)
							if err != nil {
								colexecerror.InternalError(err)
							}
							_converted := da.NewDUuid(tree.DUuid{UUID: id})
							converted[destIdx] = _converted
						}
					}
				case types.TimestampFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Timestamp()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDTimestamp(tree.DTimestamp{Time: v})
							converted[destIdx] = _converted
						}
					}
				case types.TimestampTZFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Timestamp()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDTimestampTZ(tree.DTimestampTZ{Time: v})
							converted[destIdx] = _converted
						}
					}
				case types.IntervalFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Interval()
						_ = true
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							_ = true
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInterval(tree.DInterval{Duration: v})
							converted[destIdx] = _converted
						}
					}
				case types.EnumFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							v := typedCol.Get(srcIdx)
							e, err := tree.MakeDEnumFromPhysicalRepresentation(ct, v)
							if err != nil {
								colexecerror.InternalError(err)
							}
							_converted := da.NewDEnum(e)
							converted[destIdx] = _converted
						}
					}
				case typeconv.DatumVecCanonicalTypeFamily:
				default:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Datum()
						for idx = 0; idx < length; idx++ {
							{
								//gcassert:bce
								destIdx = sel[idx]
							}
							{
								//gcassert:bce
								srcIdx = sel[idx]
							}
							v := typedCol.Get(srcIdx)
							_converted := v.(tree.Datum)
							converted[destIdx] = _converted
						}
					}
				}
			vecToDatum_false_true_false_return_4:
			}
		} else {
			{
				_ = converted[length-1]
				var idx, destIdx, srcIdx int
				switch ct := col.Type(); ct.Family() {
				case types.StringFamily:
					// Note that there is no need for a copy since casting to a string will
					// do that.
					bytes := col.Bytes()
					if ct.Oid() == oid.T_name {
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							v := da.NewDName(tree.DString(bytes.Get(srcIdx)))
							//gcassert:bce
							converted[destIdx] = v
						}
						goto vecToDatum_false_false_false_return_5
					}
					for idx = 0; idx < length; idx++ {
						{
							destIdx = idx
						}
						{
							srcIdx = idx
						}
						v := da.NewDString(tree.DString(bytes.Get(srcIdx)))
						//gcassert:bce
						converted[destIdx] = v
					}
				case types.BoolFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bool()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := tree.MakeDBool(tree.DBool(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.IntFamily:
					switch ct.Width() {
					case 16:
						typedCol := col.Int16()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInt(tree.DInt(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					case 32:
						typedCol := col.Int32()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInt(tree.DInt(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					case -1:
					default:
						typedCol := col.Int64()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInt(tree.DInt(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.FloatFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Float64()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDFloat(tree.DFloat(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.DecimalFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Decimal()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDDecimal(tree.DDecimal{Decimal: v})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.DateFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Int64()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDDate(tree.DDate{Date: pgdate.MakeCompatibleDateFromDisk(v)})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.BytesFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							v := typedCol.Get(srcIdx)
							// Note that there is no need for a copy since DBytes uses a string
							// as underlying storage, which will perform the copy for us.
							_converted := da.NewDBytes(tree.DBytes(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.EncodedKeyFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							v := typedCol.Get(srcIdx)
							// Note that there is no need for a copy since DEncodedKey uses a string
							// as underlying storage, which will perform the copy for us.
							_converted := da.NewDEncodedKey(tree.DEncodedKey(v))
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.JsonFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.JSON()
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							v := typedCol.Get(srcIdx)

							// The following operation deliberately copies the input JSON
							// bytes, since FromEncoding is lazy and keeps a handle on the bytes
							// it is passed in.
							_bytes, _err := json.EncodeJSON(nil, v)
							if _err != nil {
								colexecerror.ExpectedError(_err)
							}
							var _j json.JSON
							_j, _err = json.FromEncoding(_bytes)
							if _err != nil {
								colexecerror.ExpectedError(_err)
							}
							_converted := da.NewDJSON(tree.DJSON{JSON: _j})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.UuidFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							v := typedCol.Get(srcIdx)
							// Note that there is no need for a copy because uuid.FromBytes
							// will perform a copy.
							id, err := uuid.FromBytes(v)
							if err != nil {
								colexecerror.InternalError(err)
							}
							_converted := da.NewDUuid(tree.DUuid{UUID: id})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.TimestampFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Timestamp()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDTimestamp(tree.DTimestamp{Time: v})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.TimestampTZFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Timestamp()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDTimestampTZ(tree.DTimestampTZ{Time: v})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.IntervalFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Interval()
						_ = true
						_ = typedCol.Get(length - 1)
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							_ = true
							//gcassert:bce
							v := typedCol.Get(srcIdx)
							_converted := da.NewDInterval(tree.DInterval{Duration: v})
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case types.EnumFamily:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Bytes()
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							v := typedCol.Get(srcIdx)
							e, err := tree.MakeDEnumFromPhysicalRepresentation(ct, v)
							if err != nil {
								colexecerror.InternalError(err)
							}
							_converted := da.NewDEnum(e)
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				case typeconv.DatumVecCanonicalTypeFamily:
				default:
					switch ct.Width() {
					case -1:
					default:
						typedCol := col.Datum()
						for idx = 0; idx < length; idx++ {
							{
								destIdx = idx
							}
							{
								srcIdx = idx
							}
							v := typedCol.Get(srcIdx)
							_converted := v.(tree.Datum)
							//gcassert:bce
							converted[destIdx] = _converted
						}
					}
				}
			vecToDatum_false_false_false_return_5:
			}
		}
	}
}

// This template function is a small helper that updates destIdx based on
// whether we want the deselection behavior.
// execgen:inline
const _ = "template_setDestIdx"

// execgen:inline
const _ = "template_setSrcIdx"

// vecToDatum converts the columnar data in col to the corresponding
// tree.Datum representation that is assigned to converted. length determines
// how many columnar values need to be converted and sel is an optional
// selection vector.
// NOTE: if sel is non-nil, it might perform the deselection
// step meaning (that it will densely populate converted with only values that
// are selected according to sel) based on deselect value.
// Note: len(converted) must be of sufficient length.
// execgen:inline
const _ = "template_vecToDatum"

// vecToDatum converts the columnar data in col to the corresponding
// tree.Datum representation that is assigned to converted. length determines
// how many columnar values need to be converted and sel is an optional
// selection vector.
// NOTE: if sel is non-nil, it might perform the deselection
// step meaning (that it will densely populate converted with only values that
// are selected according to sel) based on deselect value.
// Note: len(converted) must be of sufficient length.
// execgen:inline
const _ = "inlined_vecToDatum_true_true_true"

// vecToDatum converts the columnar data in col to the corresponding
// tree.Datum representation that is assigned to converted. length determines
// how many columnar values need to be converted and sel is an optional
// selection vector.
// NOTE: if sel is non-nil, it might perform the deselection
// step meaning (that it will densely populate converted with only values that
// are selected according to sel) based on deselect value.
// Note: len(converted) must be of sufficient length.
// execgen:inline
const _ = "inlined_vecToDatum_false_true_true"

// vecToDatum converts the columnar data in col to the corresponding
// tree.Datum representation that is assigned to converted. length determines
// how many columnar values need to be converted and sel is an optional
// selection vector.
// NOTE: if sel is non-nil, it might perform the deselection
// step meaning (that it will densely populate converted with only values that
// are selected according to sel) based on deselect value.
// Note: len(converted) must be of sufficient length.
// execgen:inline
const _ = "inlined_vecToDatum_true_true_false"

// vecToDatum converts the columnar data in col to the corresponding
// tree.Datum representation that is assigned to converted. length determines
// how many columnar values need to be converted and sel is an optional
// selection vector.
// NOTE: if sel is non-nil, it might perform the deselection
// step meaning (that it will densely populate converted with only values that
// are selected according to sel) based on deselect value.
// Note: len(converted) must be of sufficient length.
// execgen:inline
const _ = "inlined_vecToDatum_true_false_false"

// vecToDatum converts the columnar data in col to the corresponding
// tree.Datum representation that is assigned to converted. length determines
// how many columnar values need to be converted and sel is an optional
// selection vector.
// NOTE: if sel is non-nil, it might perform the deselection
// step meaning (that it will densely populate converted with only values that
// are selected according to sel) based on deselect value.
// Note: len(converted) must be of sufficient length.
// execgen:inline
const _ = "inlined_vecToDatum_false_true_false"

// vecToDatum converts the columnar data in col to the corresponding
// tree.Datum representation that is assigned to converted. length determines
// how many columnar values need to be converted and sel is an optional
// selection vector.
// NOTE: if sel is non-nil, it might perform the deselection
// step meaning (that it will densely populate converted with only values that
// are selected according to sel) based on deselect value.
// Note: len(converted) must be of sufficient length.
// execgen:inline
const _ = "inlined_vecToDatum_false_false_false"

// This template function is a small helper that updates destIdx based on
// whether we want the deselection behavior.
// execgen:inline
const _ = "inlined_setDestIdx_true_true"

// execgen:inline
const _ = "inlined_setSrcIdx_true"

// This template function is a small helper that updates destIdx based on
// whether we want the deselection behavior.
// execgen:inline
const _ = "inlined_setDestIdx_true_false"

// This template function is a small helper that updates destIdx based on
// whether we want the deselection behavior.
// execgen:inline
const _ = "inlined_setDestIdx_false_false"

// execgen:inline
const _ = "inlined_setSrcIdx_false"
