// Copyright 2015 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package systemschema

import (
	"context"
	"math"
	"sort"
	"time"

	"github.com/cockroachdb/cockroach/pkg/clusterversion"
	"github.com/cockroachdb/cockroach/pkg/keys"
	"github.com/cockroachdb/cockroach/pkg/security/username"
	"github.com/cockroachdb/cockroach/pkg/sql/catalog"
	"github.com/cockroachdb/cockroach/pkg/sql/catalog/catenumpb"
	"github.com/cockroachdb/cockroach/pkg/sql/catalog/catpb"
	"github.com/cockroachdb/cockroach/pkg/sql/catalog/catprivilege"
	"github.com/cockroachdb/cockroach/pkg/sql/catalog/dbdesc"
	"github.com/cockroachdb/cockroach/pkg/sql/catalog/descpb"
	"github.com/cockroachdb/cockroach/pkg/sql/catalog/tabledesc"
	"github.com/cockroachdb/cockroach/pkg/sql/privilege"
	"github.com/cockroachdb/cockroach/pkg/sql/sem/catconstants"
	"github.com/cockroachdb/cockroach/pkg/sql/sem/tree"
	"github.com/cockroachdb/cockroach/pkg/sql/types"
	"github.com/cockroachdb/cockroach/pkg/util/log"
)

// sql CREATE commands and full schema for each system table.
// These strings are *not* used at runtime, but are checked by the
// `TestSystemTableLiterals` test that compares the table generated by
// evaluating the `CREATE TABLE` statement to the descriptor literal that is
// actually used at runtime.

// These system tables are part of the system config.
const (
	NamespaceTableSchema = `
CREATE TABLE system.namespace (
  "parentID" INT8,
  "parentSchemaID" INT8,
  name       STRING,
  id         INT8,
  CONSTRAINT "primary" PRIMARY KEY ("parentID", "parentSchemaID", name)
);`

	DescriptorTableSchema = `
CREATE TABLE system.descriptor (
  id         INT8,
  descriptor BYTES,
  CONSTRAINT "primary" PRIMARY KEY (id)
);`

	// UsersTableSchema represents the system.users table.
	UsersTableSchema = `
CREATE TABLE system.users (
  username         STRING NOT NULL,
  "hashedPassword" BYTES NULL,
  "isRole"         BOOL NOT NULL DEFAULT false,
  user_id          OID NOT NULL,
  CONSTRAINT "primary" PRIMARY KEY (username),
  UNIQUE INDEX users_user_id_idx (user_id ASC),
  FAMILY "primary" (username, user_id)
);`
	RoleOptionsTableSchema = `
CREATE TABLE system.role_options (
	username STRING NOT NULL,
	option STRING NOT NULL,
	value STRING,
	user_id OID NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (username, option),
	INDEX users_user_id_idx (user_id ASC),
	FAMILY "primary" (username, option, value, user_id)
)`

	// Zone settings per DB/Table.
	ZonesTableSchema = `
CREATE TABLE system.zones (
  id     INT8,
  config BYTES,
  CONSTRAINT "primary" PRIMARY KEY (id)
);`

	SettingsTableSchema = `
CREATE TABLE system.settings (
	-- the internal key for the setting. The column is called 'name' for
	-- historical reasons.
	name              STRING    NOT NULL,
	value             STRING    NOT NULL,
	"lastUpdated"     TIMESTAMP NOT NULL DEFAULT now(),
	"valueType"       STRING,
	CONSTRAINT "primary" PRIMARY KEY (name),
	FAMILY (name, value, "lastUpdated", "valueType")
);`

	DescIDSequenceSchema = `
CREATE SEQUENCE system.descriptor_id_seq;`

	TenantIDSequenceSchema = `
CREATE SEQUENCE system.tenant_id_seq;`

	// Note: the "active" column is deprecated.
	TenantsTableSchema = `
CREATE TABLE system.tenants (
	id           INT8 NOT NULL,
	active       BOOL NOT NULL DEFAULT true NOT VISIBLE,
	info         BYTES,
	name         STRING,
	data_state   INT,
	service_mode INT,
	CONSTRAINT "primary" PRIMARY KEY (id),
	FAMILY "primary" (id, active, info, name, data_state, service_mode),
	UNIQUE INDEX tenants_name_idx (name ASC),
	INDEX tenants_service_mode_idx (service_mode ASC)
);`

	// RoleIDSequenceSchema starts at 100 so we have reserved IDs for special
	// roles such as root and admin.
	RoleIDSequenceSchema = `
CREATE SEQUENCE system.role_id_seq START 100 MINVALUE 100 MAXVALUE 2147483647;`

	indexUsageComputeExpr           = `(statistics->'statistics':::STRING)->'indexes':::STRING`
	executionCountComputeExpr       = `((statistics->'statistics':::STRING)->'cnt':::STRING)::INT8`
	serviceLatencyComputeExpr       = `(((statistics->'statistics':::STRING)->'svcLat':::STRING)->'mean':::STRING)::FLOAT8`
	cpuSqlNanosComputeExpr          = `(((statistics->'execution_statistics':::STRING)->'cpuSQLNanos':::STRING)->'mean':::STRING)::FLOAT8`
	contentionTimeComputeExpr       = `(((statistics->'execution_statistics':::STRING)->'contentionTime':::STRING)->'mean':::STRING)::FLOAT8`
	totalEstimatedExecutionTimeExpr = `((statistics->'statistics':::STRING)->>'cnt':::STRING)::FLOAT8 * (((statistics->'statistics':::STRING)->'svcLat':::STRING)->>'mean':::STRING)::FLOAT8`
	p99LatencyComputeExpr           = `(((statistics->'statistics':::STRING)->'latencyInfo':::STRING)->'p99':::STRING)::FLOAT8`
)

var indexUsageComputeExprStr = indexUsageComputeExpr
var executionCountComputeExprStr = executionCountComputeExpr
var serviceLatencyComputeExprStr = serviceLatencyComputeExpr
var cpuSqlNanosComputeExprStr = cpuSqlNanosComputeExpr
var contentionTimeComputeExprStr = contentionTimeComputeExpr
var totalEstimatedExecutionTimeExprStr = totalEstimatedExecutionTimeExpr
var p99LatencyComputeExprStr = p99LatencyComputeExpr

// These system tables are not part of the system config.
const (
	// Note: the column "nodeID" refers to the SQL instance ID. It is named
	// "nodeID" for historical reasons.
	LeaseTableSchema = `CREATE TABLE system.lease (
  "descID"     INT8,
  version      INT8,
  "nodeID"     INT8,
  expiration   TIMESTAMP,
  crdb_region  BYTES NOT NULL,
  CONSTRAINT   "primary" PRIMARY KEY (crdb_region, "descID", version, expiration, "nodeID"),
  FAMILY       "primary" ("descID", version, "nodeID", expiration, crdb_region)
);`

	// LeaseTableSchema_V24_1 is the new session based leasing table format.
	LeaseTableSchema_V24_1 = `CREATE TABLE system.lease (
  desc_id          INT8,
  version          INT8,
  sql_instance_id  INT8 NOT NULL,
  session_id       BYTES,
  crdb_region      BYTES NOT NULL,
  CONSTRAINT       "primary" PRIMARY KEY (crdb_region, desc_id, version, session_id),
  FAMILY           "primary" (desc_id, version, sql_instance_id, session_id, crdb_region)
) WITH (exclude_data_from_backup = true);`

	// system.eventlog contains notable events from the cluster.
	//
	// This data is also exported to the Observability Service. This table might
	// go away in the future.
	//
	// The "reportingID" column is the SQL instance ID of the
	// server that reported the event. For node events, this
	// value is also equal to the node ID.
	//
	// Note: the column "targetID" was deprecated in v21.1 and
	// is not populated any more as of v22.2 (its value remains zero).
	// TODO(knz): Implement a migration to remove it.
	EventLogTableSchema = `
CREATE TABLE system.eventlog (
  timestamp     TIMESTAMP  NOT NULL,
  "eventType"   STRING     NOT NULL,
  "targetID"    INT8       NOT NULL,
  "reportingID" INT8       NOT NULL,
  info          STRING,
  "uniqueID"    BYTES      DEFAULT uuid_v4(),
  CONSTRAINT "primary" PRIMARY KEY (timestamp, "uniqueID")
);`

	// rangelog is currently envisioned as a wide table; many different event
	// types can be recorded to the table.
	RangeEventTableSchema = `
CREATE TABLE system.rangelog (
  timestamp      TIMESTAMP  NOT NULL,
  "rangeID"      INT8       NOT NULL,
  "storeID"      INT8       NOT NULL,
  "eventType"    STRING     NOT NULL,
  "otherRangeID" INT8,
  info           STRING,
  "uniqueID"     INT8       DEFAULT unique_rowid(),
  CONSTRAINT "primary" PRIMARY KEY (timestamp, "uniqueID")
);`

	UITableSchema = `
CREATE TABLE system.ui (
	key           STRING,
	value         BYTES,
	"lastUpdated" TIMESTAMP NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (key)
);`

	// JobsRunStatsIdxPredicate is the predicate in jobs_run_stats_idx in JobsTable.
	JobsRunStatsIdxPredicate = `status IN ('running':::STRING, 'reverting':::STRING, 'pending':::STRING, 'pause-requested':::STRING, 'cancel-requested':::STRING)`

	// Note: this schema is changed in a migration (a progress column is added in
	// a separate family).
	// NB: main column family uses old, pre created_by_type/created_by_id columns, named.
	// This is done to minimize migration work required.
	JobsTableSchema = `
CREATE TABLE system.jobs (
	id                INT8      DEFAULT unique_rowid(),
	status            STRING    NOT NULL,
	created           TIMESTAMP NOT NULL DEFAULT now(),
	dropped_payload   BYTES NOT VISIBLE,
	dropped_progress  BYTES NOT VISIBLE,
	created_by_type   STRING,
	created_by_id     INT,
	claim_session_id  BYTES,
	claim_instance_id INT8,
	num_runs          INT8,
	last_run          TIMESTAMP,
	job_type          STRING,
	owner             STRING,
	description       STRING,
	error_msg         STRING,
	finished          TIMESTAMPTZ,
	CONSTRAINT "primary" PRIMARY KEY (id),
	INDEX (status, created),
	INDEX (created_by_type, created_by_id) STORING (status),
	INDEX jobs_run_stats_idx (
    claim_session_id,
    status,
    created
  ) STORING(last_run, num_runs, claim_instance_id)
    WHERE ` + JobsRunStatsIdxPredicate + `,
  INDEX jobs_job_type_idx (job_type),
	FAMILY fam_0_id_status_created_payload (id, status, created, dropped_payload, created_by_type, created_by_id, job_type, owner, description, error_msg, finished),
	FAMILY progress (dropped_progress),
	FAMILY claim (claim_session_id, claim_instance_id, num_runs, last_run)
);`

	// JobProgressTableSchema is a table that contains one row per job reflecting
	// that job's current progress for human consumption (e.g. in SHOW or the UI).
	// This table includes the time the progress was reported in the PK, so to
	// update the current progress for a job, one would DELETE the old row and
	// INSERT a new one, rather than UPDATEing the existing row. This is done to
	// ensure revisions of the progress are not revisions to one "row" in CRDB,
	// and thus the ranges can split between revisions to progress even though
	// they cannot split between revisions to a SQL row. This is the same pattern
	// used by job_info.
	JobProgressTableSchema = `
CREATE TABLE system.job_progress (
	job_id INT8 NOT NULL,
	written TIMESTAMPTZ NOT NULL DEFAULT now(), -- when this progress was reported.
	fraction FLOAT, -- the fraction of some known work that is complete (e.g. half done with backfill)
	resolved DECIMAL, -- a timestamp up to which processing is complete (aka "highwater"/resolvedTS/etc). Typically used by perpetual jobs instead of fraction.
	--
	FAMILY "primary" ("job_id", "written", "fraction", "resolved"),
	CONSTRAINT "primary" PRIMARY KEY (job_id ASC, written DESC)
)`

	// JobProgressHistoryTableSchema is identical to job_progress, but is allowed
	// to contain multiple rows for the same job_id. This table is used to track
	// the progress of a job over time, whereas job_progress only contains the
	// most recent progress update to allow more efficient lookups of the current
	// progress in joins for SHOW JOBS. This table generally should only queried
	// for a specific job at a time. History is not stored indefinitely and may be
	// pruned based on retention limits.
	JobProgressHistoryTableSchema = `
	CREATE TABLE system.job_progress_history (
		job_id INT8 NOT NULL,
		written TIMESTAMPTZ NOT NULL DEFAULT now(),
		fraction FLOAT,
		resolved DECIMAL,
		--
		FAMILY "primary" ("job_id", "written", "fraction", "resolved"),
		CONSTRAINT "primary" PRIMARY KEY (job_id ASC, written DESC)
	)`

	// JobStatusTableSchema is the table that contains the current, user-visible
	// status of a job, which is typically its current phase of execution such as
	// such as "backfilling" or "waiting for GC". This table contains one row per
	// job; a change to the "current status" for a job replaces this row with a
	// new row rather than updating it in place (see the comment on job_progress).
	// Note: jobs which wish to just make some fact or condition visible to a user
	// that may be of note but which is not a change in its status, such as if it
	// is retrying or is catching up but still in its "backfilling" phase, may
	// wish to write that as a message in job_message rather than changing their
	// status in this table; jobs should avoid frequent mutations to status.
	JobStatusTableSchema = `
	CREATE TABLE system.job_status (
		job_id INT8 NOT NULL,
		written TIMESTAMPTZ NOT NULL DEFAULT now(), 
		status STRING NOT NULL, -- the human-readable status message e.g. "reverting" or "backfilling".
		--
		FAMILY "primary" ("job_id", "written", "status"),
		CONSTRAINT "primary" PRIMARY KEY (job_id, written DESC)
	)`

	// JobMessageTableSchema is the table that contains a history of messages a
	// job wishes to be visible to a human inspecting that specific job. Unlike
	// the status table which only retains one string "status" for any given job
	// at one time, this table table retains multiple messages per job, both as it
	// contains historical messages and can contain messages of different kinds at
	// the same time, thus it is intended to be queried for a specific job at a
	// time rather than for all jobs at once. Messages can indicate what kind of
	// message is being reported ("warning", "status_change", "state transition",
	// etc). Messages may be dropped based on rate and/or retention limits. Jobs
	// may be more liberal in emitting messages here than they are in changing
	// their status in job_status, as this table is not joined in the same txn
	// when listing all jobs.
	JobMessageTableSchema = `
	CREATE TABLE system.job_message (
		job_id INT8 NOT NULL,
		written TIMESTAMPTZ NOT NULL DEFAULT now(), 
		kind STRING, -- the type of message, e.g. "warning", "status_change", etc
		message STRING NOT NULL, -- the human-readable status message e.g. "reverting" or "backfilling".
		--
		FAMILY "primary" ("job_id", "written", "kind", "message"),
		CONSTRAINT "primary" PRIMARY KEY (job_id ASC, written DESC, kind ASC)
	)`

	// web_sessions are used to track authenticated user actions over stateless
	// connections, such as the cookie-based authentication used by the Admin
	// UI.
	// Design outlined in /docs/RFCS/web_session_login.rfc
	WebSessionsTableSchema = `
CREATE TABLE system.web_sessions (
	id             INT8       NOT NULL DEFAULT unique_rowid(),
	"hashedSecret" BYTES      NOT NULL,
	username       STRING     NOT NULL,
	"createdAt"    TIMESTAMP  NOT NULL DEFAULT now(),
	"expiresAt"    TIMESTAMP  NOT NULL,
	"revokedAt"    TIMESTAMP,
	"lastUsedAt"   TIMESTAMP  NOT NULL DEFAULT now(),
	"auditInfo"    STRING,
	user_id        OID        NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (id),
	INDEX ("expiresAt"),
	INDEX ("createdAt"),
  INDEX ("revokedAt"),
  INDEX ("lastUsedAt"),
	FAMILY "fam_0_id_hashedSecret_username_createdAt_expiresAt_revokedAt_lastUsedAt_auditInfo" (id, "hashedSecret", username, "createdAt", "expiresAt", "revokedAt", "lastUsedAt", "auditInfo", user_id)
);`

	// table_statistics is used to track statistics collected about individual
	// columns or groups of columns from every table in the database. Each row
	// contains the number of distinct values of the column group, the number of
	// null values, the average size of the column(s), and (optionally) a
	// histogram if there is only one column in columnIDs.
	//
	// Design outlined in /docs/RFCS/20170908_sql_optimizer_statistics.md
	// Note: avgSize is a newer statistic than the RFC above. It contains the
	// average size of the column group in bytes.
	TableStatisticsTableSchema = `
CREATE TABLE system.table_statistics (
	"tableID"            INT8       NOT NULL,
	"statisticID"        INT8       NOT NULL DEFAULT unique_rowid(),
	name                 STRING,
	"columnIDs"          INT8[]     NOT NULL,
	"createdAt"          TIMESTAMP  NOT NULL DEFAULT now(),
	"rowCount"           INT8       NOT NULL,
	"distinctCount"      INT8       NOT NULL,
	"nullCount"          INT8       NOT NULL,
	histogram            BYTES,
	"avgSize"            INT8       NOT NULL DEFAULT 0,
	"partialPredicate"   STRING,
	"fullStatisticID"    INT8,
	CONSTRAINT "primary" PRIMARY KEY ("tableID", "statisticID"),
	FAMILY "fam_0_tableID_statisticID_name_columnIDs_createdAt_rowCount_distinctCount_nullCount_histogram" ("tableID", "statisticID", name, "columnIDs", "createdAt", "rowCount", "distinctCount", "nullCount", histogram, "avgSize", "partialPredicate", "fullStatisticID")
);`

	// locations are used to map a locality specified by a node to geographic
	// latitude, longitude coordinates, specified as degrees.
	LocationsTableSchema = `
CREATE TABLE system.locations (
  "localityKey"   STRING,
  "localityValue" STRING,
  latitude        DECIMAL(18,15) NOT NULL,
  longitude       DECIMAL(18,15) NOT NULL,
  CONSTRAINT "primary" PRIMARY KEY ("localityKey", "localityValue"),
  FAMILY ("localityKey", "localityValue", latitude, longitude)
);`

	// role_members stores relationships between roles (role->role and role->user).
	RoleMembersTableSchema = `
CREATE TABLE system.role_members (
  "role"    STRING NOT NULL,
  "member"  STRING NOT NULL,
  "isAdmin" BOOL NOT NULL,
  role_id   OID NOT NULL,
  member_id OID NOT NULL,
  CONSTRAINT "primary" PRIMARY KEY ("role", "member"),
  INDEX ("role"),
  INDEX ("member"),
  INDEX (role_id),
  INDEX (member_id),
  UNIQUE INDEX (role_id, member_id)
);`

	// comments stores comments(database, table, column...).
	CommentsTableSchema = `
CREATE TABLE system.comments (
   type      INT NOT NULL,    -- type of object, to distinguish between db, table, column and others
   object_id INT NOT NULL,    -- object ID, this will be usually db/table desc ID
   sub_id    INT NOT NULL,    -- sub ID for column or indexes inside table, 0 for pure table
   comment   STRING NOT NULL, -- the comment
   CONSTRAINT "primary" PRIMARY KEY (type, object_id, sub_id)
);`

	// reports_meta stores reports metadata
	ReportsMetaTableSchema = `
CREATE TABLE system.reports_meta (
	id INT8 NOT NULL, "generated" TIMESTAMPTZ NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (id ASC),
	FAMILY "primary" (id, "generated")
);`

	// replication_constraint_stats stores replication constraint statistics
	ReplicationConstraintStatsTableSchema = `
CREATE TABLE system.replication_constraint_stats (
	zone_id
		INT8 NOT NULL,
	subzone_id
		INT8 NOT NULL,
	type
		STRING NOT NULL,
	config
		STRING NOT NULL,
	report_id
		INT8 NOT NULL,
	violation_start
		TIMESTAMPTZ NULL,
	violating_ranges
		INT8 NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (zone_id ASC, subzone_id ASC, type ASC, config ASC),
	FAMILY "primary" (zone_id, subzone_id, type, config, report_id, violation_start, violating_ranges)
) WITH (exclude_data_from_backup = true);`

	// replication_critical_localities stores replication critical localities
	ReplicationCriticalLocalitiesTableSchema = `
CREATE TABLE system.replication_critical_localities (
	zone_id
		INT8 NOT NULL,
	subzone_id
		INT8 NOT NULL,
	locality
		STRING NOT NULL,
	report_id
		INT8 NOT NULL,
	at_risk_ranges
		INT8 NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (zone_id ASC, subzone_id ASC, locality ASC),
	FAMILY "primary" (zone_id, subzone_id, locality, report_id, at_risk_ranges)
);`

	// replication_stats stores replication statistics
	ReplicationStatsTableSchema = `
CREATE TABLE system.replication_stats (
	zone_id
		INT8 NOT NULL,
	subzone_id
		INT8 NOT NULL,
	report_id
		INT8 NOT NULL,
	total_ranges
		INT8 NOT NULL,
	unavailable_ranges
		INT8 NOT NULL,
	under_replicated_ranges
		INT8 NOT NULL,
	over_replicated_ranges
		INT8 NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (zone_id, subzone_id),
	FAMILY "primary" (
		zone_id,
		subzone_id,
		report_id,
		total_ranges,
		unavailable_ranges,
		under_replicated_ranges,
		over_replicated_ranges
	)
) WITH (exclude_data_from_backup = true);`

	// protected_ts_meta stores a single row of metadata for the protectedts
	// subsystem.
	ProtectedTimestampsMetaTableSchema = `
CREATE TABLE system.protected_ts_meta (
   singleton   BOOL NOT NULL DEFAULT (true),
   version     INT8 NOT NULL,
   num_records INT8 NOT NULL,
   num_spans   INT8 NOT NULL,
   total_bytes INT8 NOT NULL,
   CONSTRAINT check_singleton  CHECK (singleton),
   CONSTRAINT "primary" PRIMARY KEY (singleton),
	 FAMILY "primary" (singleton, version, num_records, num_spans, total_bytes)
);`

	ProtectedTimestampsRecordsTableSchema = `
CREATE TABLE system.protected_ts_records (
   id        UUID NOT NULL,
   ts        DECIMAL NOT NULL,
   meta_type STRING NOT NULL,
   meta      BYTES,
   num_spans INT8 NOT NULL, -- num spans is important to know how to decode spans
   spans     BYTES NOT NULL,
   verified  BOOL NOT NULL DEFAULT (false),
   target    BYTES,         -- target is an encoded protobuf that specifies what the pts record will protect
   CONSTRAINT "primary" PRIMARY KEY (id),
	 FAMILY "primary" (id, ts, meta_type, meta, num_spans, spans, verified, target)
);`

	StatementBundleChunksTableSchema = `
CREATE TABLE system.statement_bundle_chunks (
   id          INT8 DEFAULT unique_rowid(),
	 description STRING,
	 data        BYTES NOT NULL,
	 CONSTRAINT "primary" PRIMARY KEY (id),
	 FAMILY "primary" (id, description, data)
);`

	StatementDiagnosticsRequestsTableSchema = `
CREATE TABLE system.statement_diagnostics_requests(
	id INT8 DEFAULT unique_rowid() NOT NULL,
	completed BOOL NOT NULL DEFAULT FALSE,
	statement_fingerprint STRING NOT NULL,
	statement_diagnostics_id INT8,
	requested_at TIMESTAMPTZ NOT NULL,
	min_execution_latency INTERVAL NULL,
	expires_at TIMESTAMPTZ NULL,
	sampling_probability FLOAT NULL,
	plan_gist STRING NULL,
	anti_plan_gist BOOL NULL,
	redacted BOOL NOT NULL DEFAULT FALSE,
	CONSTRAINT "primary" PRIMARY KEY (id),
	CONSTRAINT check_sampling_probability CHECK (sampling_probability BETWEEN 0.0 AND 1.0),
	INDEX completed_idx (completed, id) STORING (statement_fingerprint, min_execution_latency, expires_at, sampling_probability, plan_gist, anti_plan_gist, redacted),
	FAMILY "primary" (id, completed, statement_fingerprint, statement_diagnostics_id, requested_at, min_execution_latency, expires_at, sampling_probability, plan_gist, anti_plan_gist, redacted)
);`

	StatementDiagnosticsTableSchema = `
create table system.statement_diagnostics(
  id INT8 DEFAULT unique_rowid() NOT NULL,
  statement_fingerprint STRING NOT NULL,
  statement STRING NOT NULL,
  collected_at TIMESTAMPTZ NOT NULL,
  trace JSONB,
  bundle_chunks INT ARRAY,
	error STRING,
	CONSTRAINT "primary" PRIMARY KEY (id),

	FAMILY "primary" (id, statement_fingerprint, statement, collected_at, trace, bundle_chunks, error)
);`

	ScheduledJobsTableSchema = `
CREATE TABLE system.scheduled_jobs (
    schedule_id      INT DEFAULT unique_rowid() NOT NULL,
    schedule_name    STRING NOT NULL,
    created          TIMESTAMPTZ NOT NULL DEFAULT now(),
    owner            STRING NOT NULL,
    next_run         TIMESTAMPTZ,
    schedule_state   BYTES,
    schedule_expr    STRING,
    schedule_details BYTES,
    executor_type    STRING NOT NULL,
    execution_args   BYTES NOT NULL,

    CONSTRAINT "primary" PRIMARY KEY (schedule_id),
    INDEX "next_run_idx" (next_run),

 	 FAMILY sched (schedule_id, next_run, schedule_state),
 	 FAMILY other (
       schedule_name, created, owner, schedule_expr, 
       schedule_details, executor_type, execution_args 
    )
)`

	SqllivenessTableSchema = `
CREATE TABLE system.sqlliveness (
    session_id           BYTES NOT NULL,
    expiration           DECIMAL NOT NULL,
    crdb_region          BYTES NOT NULL,
    CONSTRAINT "primary" PRIMARY KEY (crdb_region, session_id),
    FAMILY "primary" (crdb_region, session_id, expiration)
)`

	// system.migrations stores completion records for upgrades performed by the
	// upgrade package. Only completed upgrades have a row in this table.
	MigrationsTableSchema = `
CREATE TABLE system.migrations (
    major        INT8 NOT NULL,
    minor        INT8 NOT NULL,
    patch        INT8 NOT NULL,
    internal     INT8 NOT NULL,
    completed_at TIMESTAMPTZ NOT NULL,
 	 FAMILY "primary" (major, minor, patch, internal, completed_at),
    CONSTRAINT "primary" PRIMARY KEY (major, minor, patch, internal)
)`

	JoinTokensTableSchema = `
CREATE TABLE system.join_tokens (
    id           UUID NOT NULL,
    secret       BYTES NOT NULL,
    expiration   TIMESTAMPTZ NOT NULL,
    CONSTRAINT "primary" PRIMARY KEY (id),
 	 FAMILY "primary" (id, secret, expiration)
)`

	// TODO(azhng): Currently we choose number of bucket for hash-sharding to be
	//  8 for both statement statistics table and transaction statistics table.
	//  This is an arbitrary choice for now. After persisted SQL Stats is fully
	//  implemented, we need to revisit this choice and retune the configuration.
	StatementStatisticsTableSchema = `
CREATE TABLE system.statement_statistics (
    aggregated_ts              TIMESTAMPTZ NOT NULL,
    fingerprint_id             BYTES NOT NULL,
    transaction_fingerprint_id BYTES NOT NULL,
    plan_hash                  BYTES NOT NULL,
    app_name                   STRING NOT NULL,
    node_id                    INT8 NOT NULL,

    agg_interval INTERVAL NOT NULL,
    metadata   JSONB NOT NULL,
    statistics JSONB NOT NULL,
    plan JSONB NOT NULL,

    crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_plan_hash_transaction_fingerprint_id_shard_8 INT4 NOT VISIBLE NOT NULL AS (
      mod(fnv32(crdb_internal.datums_to_bytes(aggregated_ts, app_name, fingerprint_id, node_id, plan_hash, transaction_fingerprint_id)), 8:::INT8)
    ) STORED,

    index_recommendations STRING[] NOT NULL DEFAULT (array[]::STRING[]),
    indexes_usage JSONB AS (` + indexUsageComputeExpr + `) VIRTUAL,
    execution_count INT8 AS (` + executionCountComputeExpr + `) STORED,
    service_latency FLOAT AS (` + serviceLatencyComputeExpr + `) STORED,
    cpu_sql_nanos FLOAT AS (` + cpuSqlNanosComputeExpr + `) STORED,
    contention_time FLOAT AS (` + contentionTimeComputeExpr + `) STORED,
    total_estimated_execution_time FLOAT AS (` + totalEstimatedExecutionTimeExpr + `) STORED,
    p99_latency FLOAT8 AS (` + p99LatencyComputeExpr + `) STORED,

    CONSTRAINT "primary" PRIMARY KEY (aggregated_ts, fingerprint_id, transaction_fingerprint_id, plan_hash, app_name, node_id)
      USING HASH WITH (bucket_count=8),
    INDEX "fingerprint_stats_idx" (fingerprint_id, transaction_fingerprint_id),
    INVERTED INDEX "indexes_usage_idx" (indexes_usage),
    INDEX "execution_count_idx" (aggregated_ts, app_name, execution_count DESC) WHERE app_name NOT LIKE '$ internal%',
    INDEX "service_latency_idx" (aggregated_ts, app_name, service_latency DESC) WHERE app_name NOT LIKE '$ internal%',
    INDEX "cpu_sql_nanos_idx" (aggregated_ts, app_name, cpu_sql_nanos DESC) WHERE app_name NOT LIKE '$ internal%',
    INDEX "contention_time_idx" (aggregated_ts, app_name, contention_time DESC) WHERE app_name NOT LIKE '$ internal%',
    INDEX "total_estimated_execution_time_idx" (aggregated_ts, app_name, total_estimated_execution_time DESC) WHERE app_name NOT LIKE '$ internal%',
    INDEX "p99_latency_idx" (aggregated_ts, app_name, p99_latency DESC) WHERE app_name NOT LIKE '$ internal%',
		FAMILY "primary" (
			crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_plan_hash_transaction_fingerprint_id_shard_8,
			aggregated_ts,
			fingerprint_id,
			transaction_fingerprint_id,
			plan_hash,
			app_name,
			node_id,
			agg_interval,
			metadata,
			statistics,
			plan,
			index_recommendations,
			execution_count,
			service_latency,
			cpu_sql_nanos,
			contention_time,
			total_estimated_execution_time,
			p99_latency
		)
)
`

	TransactionStatisticsTableSchema = `
CREATE TABLE system.transaction_statistics (
    aggregated_ts  TIMESTAMPTZ NOT NULL,
    fingerprint_id BYTES NOT NULL,
    app_name       STRING NOT NULL,
    node_id        INT8 NOT NULL,

    agg_interval INTERVAL NOT NULL,
    metadata   JSONB NOT NULL,
    statistics JSONB NOT NULL,

    crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_shard_8 INT4 NOT VISIBLE NOT NULL AS (
      mod(fnv32("crdb_internal.datums_to_bytes"(aggregated_ts, app_name, fingerprint_id, node_id)), 8:::INT8
    )) STORED,
    execution_count INT8 AS (` + executionCountComputeExpr + `) STORED,
    service_latency FLOAT AS (` + serviceLatencyComputeExpr + `) STORED,
    cpu_sql_nanos FLOAT AS (` + cpuSqlNanosComputeExpr + `) STORED,
    contention_time FLOAT AS (` + contentionTimeComputeExpr + `) STORED,
    total_estimated_execution_time FLOAT AS (` + totalEstimatedExecutionTimeExpr + `) STORED,
    p99_latency FLOAT8 AS (` + p99LatencyComputeExpr + `) STORED,

    CONSTRAINT "primary" PRIMARY KEY (aggregated_ts, fingerprint_id, app_name, node_id)
      USING HASH WITH (bucket_count=8),
    INDEX "fingerprint_stats_idx" (fingerprint_id),
    INDEX "execution_count_idx" (aggregated_ts, app_name, execution_count DESC) WHERE app_name NOT LIKE '$ internal%',
    INDEX "service_latency_idx" (aggregated_ts, app_name, service_latency DESC) WHERE app_name NOT LIKE '$ internal%',
    INDEX "cpu_sql_nanos_idx" (aggregated_ts, app_name, cpu_sql_nanos DESC) WHERE app_name NOT LIKE '$ internal%',
    INDEX "contention_time_idx" (aggregated_ts, app_name, contention_time DESC) WHERE app_name NOT LIKE '$ internal%',
    INDEX "total_estimated_execution_time_idx" (aggregated_ts, app_name, total_estimated_execution_time DESC) WHERE app_name NOT LIKE '$ internal%',
    INDEX "p99_latency_idx" (aggregated_ts, app_name, p99_latency DESC) WHERE app_name NOT LIKE '$ internal%',
		FAMILY "primary" (
			crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_shard_8,
			aggregated_ts,
			fingerprint_id,
			app_name,
			node_id,
			agg_interval,
			metadata,
			statistics,
			execution_count,
			service_latency,
			cpu_sql_nanos,
			contention_time,
			total_estimated_execution_time,
			p99_latency
		)
);
`

	StatementActivityTableSchema = `
CREATE TABLE system.statement_activity
(
    aggregated_ts                    TIMESTAMPTZ NOT NULL,
    fingerprint_id                   BYTES       NOT NULL,
    transaction_fingerprint_id       BYTES       NOT NULL,
    plan_hash                        BYTES       NOT NULL,
    app_name                         STRING      NOT NULL,
    agg_interval                     INTERVAL    NOT NULL,
    metadata                         JSONB       NOT NULL,
    statistics                       JSONB       NOT NULL,
    plan                             JSONB       NOT NULL,
    index_recommendations            STRING[]    NOT NULL DEFAULT (ARRAY[]:::STRING[]),
    execution_count                  INT         NOT NULL,
    execution_total_seconds          FLOAT       NOT NULL,
    execution_total_cluster_seconds  FLOAT       NOT NULL,
    contention_time_avg_seconds      FLOAT       NOT NULL,
    cpu_sql_avg_nanos                FLOAT       NOT NULL,
    service_latency_avg_seconds      FLOAT       NOT NULL,
    service_latency_p99_seconds      FLOAT       NOT NULL,
    CONSTRAINT "primary" PRIMARY KEY (aggregated_ts, fingerprint_id, transaction_fingerprint_id, plan_hash, app_name),
    INDEX "fingerprint_id_idx" (fingerprint_id, transaction_fingerprint_id),
    INDEX "execution_count_idx" (aggregated_ts, execution_count DESC),
    INDEX "execution_total_seconds_idx" (aggregated_ts, execution_total_seconds DESC),
    INDEX "contention_time_avg_seconds_idx" (aggregated_ts, contention_time_avg_seconds DESC),
    INDEX "cpu_sql_avg_nanos_idx" (aggregated_ts, cpu_sql_avg_nanos DESC),
    INDEX "service_latency_avg_seconds_idx" (aggregated_ts, service_latency_avg_seconds DESC),
    INDEX "service_latency_p99_seconds_idx" (aggregated_ts, service_latency_p99_seconds DESC),
    FAMILY "primary" (
                      aggregated_ts,
                      fingerprint_id,
                      transaction_fingerprint_id,
                      plan_hash,
                      app_name,
                      agg_interval,
                      metadata,
                      statistics,
                      plan,
                      index_recommendations,
                      execution_count,
                      execution_total_seconds,
                      execution_total_cluster_seconds,
                      contention_time_avg_seconds,
                      cpu_sql_avg_nanos,
                      service_latency_avg_seconds,
                      service_latency_p99_seconds
        )
)
`

	TransactionActivityTableSchema = `
CREATE TABLE system.transaction_activity
(
    aggregated_ts                    TIMESTAMPTZ NOT NULL,
    fingerprint_id                   BYTES       NOT NULL,
    app_name                         STRING      NOT NULL,
    agg_interval                     INTERVAL    NOT NULL,
    metadata                         JSONB       NOT NULL,
    statistics                       JSONB       NOT NULL,
    query                            STRING      NOT NULL,
    execution_count                  INT         NOT NULL,
    execution_total_seconds          FLOAT       NOT NULL,
    execution_total_cluster_seconds  FLOAT       NOT NULL,
    contention_time_avg_seconds      FLOAT       NOT NULL,
    cpu_sql_avg_nanos                FLOAT       NOT NULL,
    service_latency_avg_seconds      FLOAT       NOT NULL,
    service_latency_p99_seconds      FLOAT       NOT NULL,
    CONSTRAINT "primary" PRIMARY KEY (aggregated_ts, fingerprint_id, app_name),
    INDEX "fingerprint_id_idx" (fingerprint_id),
    INDEX "execution_count_idx" (aggregated_ts, execution_count DESC),
    INDEX "execution_total_seconds_idx" (aggregated_ts, execution_total_seconds DESC),
    INDEX "contention_time_avg_seconds_idx" (aggregated_ts, contention_time_avg_seconds DESC),
    INDEX "cpu_sql_avg_nanos_idx" (aggregated_ts, cpu_sql_avg_nanos DESC),
    INDEX "service_latency_avg_seconds_idx" (aggregated_ts, service_latency_avg_seconds DESC),
    INDEX "service_latency_p99_seconds_idx" (aggregated_ts, service_latency_p99_seconds DESC),
    FAMILY "primary" (
                      aggregated_ts,
                      fingerprint_id,
                      app_name,
                      agg_interval,
                      metadata,
                      statistics,
                      query,
                      execution_count,
                      execution_total_seconds,
                      execution_total_cluster_seconds,
                      contention_time_avg_seconds,
                      cpu_sql_avg_nanos,
                      service_latency_avg_seconds,
                      service_latency_p99_seconds
        )
);
`

	DatabaseRoleSettingsTableSchema = `
CREATE TABLE system.database_role_settings (
    database_id  OID NOT NULL,
    role_name    STRING NOT NULL,
    settings     STRING[] NOT NULL,
    role_id      OID NOT NULL,
    CONSTRAINT "primary" PRIMARY KEY (database_id, role_name),
    UNIQUE INDEX (database_id, role_id) STORING (settings),
		FAMILY "primary" (
			database_id,
			role_name,
			settings,
			role_id
		)
);`

	TenantUsageTableSchema = `
CREATE TABLE system.tenant_usage (
  tenant_id INT NOT NULL,

  -- For each tenant, there is a special row with instance_id = 0 which contains
  -- per-tenant stat. Each SQL instance (pod) also has its own row with
  -- per-instance state.
  instance_id INT NOT NULL,

  -- next_instance_id identifies the next live instance ID, with the smallest ID
  -- larger than this instance_id (or 0 if there is no such ID).
  -- We are overlaying a circular linked list of all live instances, with
  -- instance 0 acting as a sentinel (always the head of the list).
  next_instance_id INT NOT NULL,

  -- Time when we last interacted with this row. For the per-tenant row, this
  -- is the time of the last update from any instance. For instance rows, this
  -- is the time of the last update from that particular instance.
  last_update TIMESTAMP NOT NULL,

  -- -------------------------------------------------------------------
  --  The following fields are used only for the per-tenant state, when
  --  instance_id = 0.
  -- -------------------------------------------------------------------

  -- Bucket configuration.
  ru_burst_limit FLOAT,
  ru_refill_rate FLOAT,

  -- Current amount of RUs in the bucket.
  ru_current FLOAT,

  -- Current sum of the shares values for all instances.
  current_share_sum FLOAT,

  -- Cumulative usage statistics, encoded as roachpb.TenantConsumption.
  total_consumption BYTES,

  -- -------------------------------------------------------------
  --  The following fields are used for per-instance state, when
  --  instance_id != 0.
  -- --------------------------------------------------------------

  -- The lease is a unique identifier for this instance, necessary because
  -- instance IDs can be reused.
  instance_lease BYTES,

  -- Last request sequence number. These numbers are provided by the
  -- instance and are monotonically increasing; used to detect duplicate
  -- requests and provide idempotency.
  instance_seq INT,

  -- Current shares value for this instance.
  instance_shares FLOAT,

  -- -------------------------------------------------------------
  --  The following fields are used only for the per-tenant state, when
  --  instance_id = 0.
  --  These were added later and need to be ordered last in order to
  --  be compatible with older versions of the table that are upgraded.
  -- --------------------------------------------------------------

  -- Computed rates of consumption for metrics used in tenant modeling, encoded
  -- as roachpb.TenantConsumptionRates.
  current_rates BYTES,

  -- In-progress computed rates of consumption for metrics used in tenant
  -- modeling, encoded as roachpb.TenantConsumptionRates.
  next_rates BYTES,

  FAMILY "primary" (
    tenant_id, instance_id, next_instance_id, last_update,
    ru_burst_limit, ru_refill_rate, ru_current, current_share_sum,
    total_consumption, instance_lease, instance_seq, instance_shares,
    current_rates, next_rates
  ),

  CONSTRAINT "primary" PRIMARY KEY (tenant_id, instance_id)
) WITH (exclude_data_from_backup = true)`

	SQLInstancesTableSchema = `
CREATE TABLE system.sql_instances (
    id             INT NOT NULL,
    addr           STRING,
    session_id     BYTES,
    locality       JSONB,
    sql_addr       STRING,
    crdb_region    BYTES NOT NULL,
    binary_version STRING,
    is_draining    BOOL NULL,
    CONSTRAINT "primary" PRIMARY KEY (crdb_region, id),
    FAMILY "primary" (
      id, 
      addr, 
      session_id, 
      locality, 
      sql_addr, 
      crdb_region, 
      binary_version, 
      is_draining
    )
)`

	SpanConfigurationsTableSchema = `
CREATE TABLE system.span_configurations (
    start_key    BYTES NOT NULL,
    end_key      BYTES NOT NULL,
    config        BYTES NOT NULL,
    CONSTRAINT "primary" PRIMARY KEY (start_key),
    CONSTRAINT check_bounds CHECK (start_key < end_key),
    FAMILY "primary" (start_key, end_key, config)
) WITH (exclude_data_from_backup = true)`

	TenantSettingsTableSchema = `
CREATE TABLE system.tenant_settings (
	-- A non-zero tenant_id indicates that this is a setting specific to that
	-- particular tenant. A zero tenant_id indicates an "all tenant" setting that
	-- applies to all tenants which don't a tenant-specific value for this
	-- setting.
	tenant_id    INT8 NOT NULL,
	-- the internal key for the setting. The column is called 'name' for
	-- historical reasons.
	name         STRING NOT NULL,
	value        STRING NOT NULL,
	last_updated TIMESTAMP NOT NULL DEFAULT now(),
	value_type   STRING NOT NULL,

	-- reason is unused for now.
	reason       STRING,
	CONSTRAINT "primary" PRIMARY KEY (tenant_id, name),
	FAMILY (tenant_id, name, value, last_updated, value_type, reason)
);`

	SpanCountTableSchema = `
CREATE TABLE system.span_count (
	singleton  BOOL DEFAULT TRUE,
	span_count INT NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (singleton),
	CONSTRAINT single_row CHECK (singleton),
	FAMILY "primary" (singleton, span_count)
);`

	SystemPrivilegeTableSchema = `
CREATE TABLE system.privileges (
	username STRING NOT NULL,
	path STRING NOT NULL,
	privileges STRING[] NOT NULL,
	grant_options STRING[] NOT NULL,
	user_id OID NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (username, path),
	UNIQUE INDEX (path, user_id) STORING (privileges, grant_options),
	UNIQUE INDEX (path, username) STORING (privileges, grant_options),
	FAMILY "primary" (username, path, privileges, grant_options, user_id)
);`

	SystemExternalConnectionsTableSchema = `
CREATE TABLE system.external_connections (
	connection_name STRING NOT NULL,
	created TIMESTAMP NOT NULL DEFAULT now(),
	updated TIMESTAMP NOT NULL DEFAULT now(),
	connection_type STRING NOT NULL,
	connection_details BYTES NOT NULL,
	owner STRING NOT NULL,
	owner_id OID NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (connection_name),
	FAMILY "primary" (connection_name, created, updated, connection_type, connection_details, owner, owner_id)
);`

	SystemTenantTasksSchema = `
CREATE TABLE system.tenant_tasks (
	tenant_id    INT8 NOT NULL,
	issuer       STRING NOT NULL,
	task_id      INT8 NOT NULL,
	created      TIMESTAMPTZ NOT NULL DEFAULT now():::TIMESTAMPTZ,
	payload_id   STRING NOT NULL,
	owner        STRING NOT NULL,
	owner_id     OID NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (tenant_id, issuer, task_id),
	FAMILY "primary" (tenant_id, issuer, task_id, created, payload_id, owner, owner_id)
);`

	SystemTaskPayloadsSchema = `
CREATE TABLE system.task_payloads (
	id           STRING NOT NULL,
	created      TIMESTAMPTZ NOT NULL DEFAULT now():::TIMESTAMPTZ,
	owner        STRING NOT NULL,
	owner_id     OID NOT NULL,
	min_version  STRING NOT NULL,
	description  STRING,
	type         STRING NOT NULL,
	value        BYTES NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (id),
	FAMILY "primary" (id, created, owner, owner_id, min_version, description, type, value)
);`

	// SystemJobInfoTableSchema is the schema for the system.job_info table, which
	// is used by jobs to store arbitrary "info".
	SystemJobInfoTableSchema = `
CREATE TABLE system.job_info (
	job_id INT8 NOT NULL,
	info_key STRING NOT NULL,
	-- written is in the PK for this table to give it mvcc-over-sql semantics, so
	-- that revisions to the value for the logical job/key pair are separate SQL
	-- rows. This is done because we do not allow KV ranges to split between the
	-- revisions of a single row, and these job info KVs may be both large
	-- and revised frequently; if we did not store revisions to them in separate
	-- SQL rows, we could easily cause over-sized ranges that cannot split.
	written TIMESTAMPTZ NOT NULL DEFAULT now():::TIMESTAMPTZ,
	value BYTES,
	CONSTRAINT "primary" PRIMARY KEY (job_id, info_key, written DESC),
	FAMILY "primary" (job_id, info_key, written, value)
);`

	// SpanStatsUniqueKeysTableSchema defines the schema to store
	// the set of unique keys of all samples in SpanStatsSamplesTableSchema.
	// The raw bytes of keys are stored in this table, so that the cost of storing
	// a large key is amortized by each bucket that references it.
	SpanStatsUniqueKeysTableSchema = `
CREATE TABLE system.span_stats_unique_keys (
    -- Every key has a unique id. We can't use the value of the key itself
	-- because we want the cost of storing the key to 
	-- amortize with repeated references. A UUID is 16 bytes,
	-- but a roachpb.Key can be arbitrarily large.
	id UUID DEFAULT gen_random_uuid(),
	
	-- key_bytes stores the raw bytes of a roachpb.Key.
	key_bytes BYTES,
  	CONSTRAINT "primary" PRIMARY KEY (id),
  	UNIQUE INDEX unique_keys_key_bytes_idx (key_bytes ASC),
	FAMILY "primary" (id, key_bytes)
);`

	// SpanStatsBucketsTableSchema defines the schema to store
	// keyvispb.SpanStats objects.
	SpanStatsBucketsTableSchema = `
CREATE TABLE system.span_stats_buckets (
    -- Every bucket has a unique id.
	id UUID DEFAULT gen_random_uuid(),
	
	-- The bucket belongs to sample_id
	sample_id UUID NOT NULL,
	
	-- The uuid of this bucket's span's start key.
	start_key_id UUID NOT NULL,
	
	-- The uuid of this bucket's span's start key.
	end_key_id UUID NOT NULL,
	
	-- The number of KV requests destined for this span. 
	requests INT NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (id),
	INDEX buckets_sample_id_idx (sample_id ASC),
	FAMILY "primary" (id, sample_id, start_key_id, end_key_id, requests)
);`

	// SpanStatsSamplesTableSchema defines the schema to store
	// a keyvispb.Sample.
	SpanStatsSamplesTableSchema = `
CREATE TABLE system.span_stats_samples (
    -- Every sample has a unique id.
	id UUID DEFAULT gen_random_uuid(),
	
	-- sample_time represents the time the sample ended.
	-- The sample's start time is therefore equal to sample_time - keyvissettings.SampleInterval.
	sample_time TIMESTAMP NOT NULL DEFAULT now(),
	CONSTRAINT "primary" PRIMARY KEY (id),
	UNIQUE INDEX samples_sample_time_idx (sample_time ASC),
	FAMILY "primary" (id, sample_time)
);`

	// SpanStatsTenantBoundariesTableSchema stores the boundaries that a tenant
	// wants KV to collect statistics for. `boundaries` is populated with the
	// keyvispb.UpdateBoundariesRequest proto.
	SpanStatsTenantBoundariesTableSchema = `
CREATE TABLE system.span_stats_tenant_boundaries (
	tenant_id    INT8 NOT NULL,
	boundaries	 BYTES NOT NULL,
	CONSTRAINT "primary" PRIMARY KEY (tenant_id),
	FAMILY "primary" (tenant_id, boundaries)
);`

	// RegionLivenessTableSchema stores the liveness for a region
	RegionLivenessTableSchema = `CREATE TABLE system.public.region_liveness (
    crdb_region BYTES NOT NULL,
    unavailable_at TIMESTAMP NULL,
    CONSTRAINT region_liveness_pkey PRIMARY KEY (crdb_region ASC),
	  FAMILY "primary" (crdb_region, unavailable_at)
) ;
`

	SystemMVCCStatisticsSchema = `
CREATE TABLE system.mvcc_statistics (
	created_at TIMESTAMPTZ NOT NULL DEFAULT now():::TIMESTAMPTZ,
	database_id INT8 NOT NULL,
	table_id INT8 NOT NULL,
	index_id INT8 NOT NULL,
	statistics JSONB NOT NULL,
	crdb_internal_created_at_database_id_index_id_table_id_shard_16 INT4 NOT VISIBLE NOT NULL AS (mod(fnv32(md5(crdb_internal.datums_to_bytes(created_at))), 16:::INT8)) VIRTUAL,
	CONSTRAINT mvcc_statistics_pkey PRIMARY KEY (created_at ASC, database_id ASC, table_id ASC, index_id ASC) USING HASH WITH (bucket_count=16),
	FAMILY "primary" (
	    created_at,
	    database_id,
	    table_id,
	    index_id,
	    statistics
	)
);`

	TxnExecutionStatsTableSchema = `
	CREATE TABLE system.transaction_execution_insights (
		transaction_id                     UUID NOT NULL,
		transaction_fingerprint_id         BYTES NOT NULL,
		query_summary                      STRING,
		implicit_txn                       BOOL,
		session_id                         STRING NOT NULL,
		start_time                         TIMESTAMPTZ,
		end_time                           TIMESTAMPTZ,
		user_name                          STRING,
		app_name                           STRING,
		user_priority                      STRING,
		retries                            INT8,
		last_retry_reason                  STRING,
		problems                           INT[], 
		causes                             INT[],
		stmt_execution_ids                 STRING[],
		cpu_sql_nanos                      INT8,
		last_error_code                    STRING,
		status                             INT,
		contention_time                    INTERVAL,
		contention_info                    JSONB,
		details                            JSONB,
		created                            TIMESTAMPTZ NOT NULL DEFAULT now(),
		crdb_internal_end_time_start_time_shard_16 INT4 NOT VISIBLE NOT NULL AS (mod(fnv32(md5(crdb_internal.datums_to_bytes(end_time, start_time))), 16:::INT8)) VIRTUAL,
		CONSTRAINT "primary" PRIMARY KEY (transaction_id),
		INDEX transaction_fingerprint_id_idx (transaction_fingerprint_id),
		INDEX time_range_idx (start_time DESC, end_time DESC) USING HASH WITH (bucket_count=16),
		FAMILY "primary" (
			transaction_id,
			transaction_fingerprint_id,
			query_summary,
			implicit_txn,
			session_id,
			start_time,
			end_time,
			user_name,
			app_name,
			user_priority,
			retries,
			last_retry_reason,
			problems,
			causes,
			stmt_execution_ids,
			cpu_sql_nanos,
			last_error_code,
			status,
			contention_time,
			contention_info,
			details,
			created
		)
	);`

	StatementExecutionStatsTableSchema = `
	CREATE TABLE system.statement_execution_insights (
		session_id                 STRING NOT NULL,
		transaction_id             UUID NOT NULL,
		transaction_fingerprint_id BYTES NOT NULL,
		statement_id               STRING NOT NULL,
		statement_fingerprint_id   BYTES NOT NULL,
		problem                    INT,
		causes                     INT[],
		query                      STRING,
		status                     INT,
		start_time                 TIMESTAMPTZ,
		end_time                   TIMESTAMPTZ,
		full_scan                  BOOL,
		user_name                  STRING,
		app_name                   STRING,
		user_priority              STRING,
		database_name              STRING,
		plan_gist                  STRING,
		retries                    INT8,
		last_retry_reason          STRING,
		execution_node_ids         INT[],
		index_recommendations      STRING[],
		implicit_txn               BOOL,
		cpu_sql_nanos              INT8,
		error_code                 STRING,
		contention_time            INTERVAL,
		contention_info            JSONB,
		details                    JSONB,
		created                    TIMESTAMPTZ NOT NULL DEFAULT now(),
		crdb_internal_end_time_start_time_shard_16 INT4 NOT VISIBLE NOT NULL AS (mod(fnv32(md5(crdb_internal.datums_to_bytes(end_time, start_time))), 16:::INT8)) VIRTUAL,
		CONSTRAINT "primary" PRIMARY KEY (statement_id, transaction_id),
		INDEX transaction_id_idx (transaction_id),
		INDEX transaction_fingerprint_id_idx (transaction_fingerprint_id, start_time DESC, end_time DESC),
		INDEX statement_fingerprint_id_idx (statement_fingerprint_id, start_time DESC, end_time DESC),
		INDEX time_range_idx (start_time DESC, end_time DESC) USING HASH,
		FAMILY "primary" (
			session_id,
			transaction_id,
			transaction_fingerprint_id,
			statement_id,
			statement_fingerprint_id,
			problem,
			causes,
			query,
			status,
			start_time,
			end_time,
			full_scan,
			user_name,
			app_name,
			user_priority,
			database_name,
			plan_gist,
			retries,
			last_retry_reason,
			execution_node_ids,
			index_recommendations,
			implicit_txn,
			cpu_sql_nanos,
			error_code,
			contention_time,
			contention_info,
			details,
			created
		)
	);`

	crdbInternalTableIdLastUpdatedShard = "mod(fnv32(md5(crdb_internal.datums_to_bytes(table_id, last_updated))), 16:::INT8)"
	TableMetadataTableSchema            = ` CREATE TABLE system.table_metadata (
	  db_id INT8 NOT NULL,
    table_id INT8 NOT NULL,
	  db_name STRING NOT NULL,
		schema_name STRING NOT NULL,
    table_name STRING NOT NULL,
    total_columns INT8 NOT NULL,
    total_indexes INT8 NOT NULL,
    store_ids INT[] NOT NULL,
    replication_size_bytes INT NOT NULL,
    total_ranges INT NOT NULL,
    total_live_data_bytes INT NOT NULL,
    total_data_bytes INT NOT NULL,
    perc_live_data FLOAT NOT NULL,
    last_update_error string,
    last_updated TIMESTAMPTZ NOT NULL DEFAULT now(),
    table_type string NOT NULL,
		details JSONB NOT NULL,
		crdb_internal_last_updated_table_id_shard_16 INT4 NOT VISIBLE NOT NULL AS (` + crdbInternalTableIdLastUpdatedShard + `) VIRTUAL,
    CONSTRAINT "primary" PRIMARY KEY (db_id, table_id),
		INDEX "replication_size_bytes_table_id_idx" (replication_size_bytes desc, table_id),
		INDEX "total_ranges_table_id_idx" (total_ranges desc, table_id),
		INDEX "total_columns_table_id_idx" (total_columns desc, table_id),
		INDEX "total_indexes_table_id_idx" (total_indexes desc, table_id),
		INDEX "perc_live_data_id_idx" (perc_live_data desc, table_id),
		INDEX "last_updated_idx" (last_updated desc, table_id) USING HASH,
		INVERTED INDEX db_name_gin (db_name gin_trgm_ops),
		INVERTED INDEX table_name_gin (table_name gin_trgm_ops),
		INVERTED INDEX schema_name_gin (schema_name gin_trgm_ops),
		INVERTED INDEX store_ids_gin (store_ids),
    FAMILY "primary" (
      db_id,
			table_id,
			db_name,
			schema_name,
			table_name,
			total_columns,
			total_indexes,
			store_ids,
			replication_size_bytes,
			total_ranges,
			total_live_data_bytes,
			total_data_bytes,
			perc_live_data,
			last_update_error,
			last_updated,
			table_type,
			details
    )
	);`

	PreparedTransactionsTableSchema = `
CREATE TABLE system.prepared_transactions (
  global_id        STRING       NOT NULL,
  transaction_id   UUID         NOT NULL,
  -- Null if the transaction does not have a transaction record.
  transaction_key  BYTES        NULL,
  prepared         TIMESTAMPTZ  NOT NULL DEFAULT now(),
  owner            STRING       NOT NULL,
  database         STRING       NOT NULL,
  -- Unused. Included in schema to support a future implementation of XA "heuristic completion".
  heuristic        STRING       NULL,
  CONSTRAINT "primary" PRIMARY KEY (global_id),
  FAMILY "primary" (global_id, transaction_id, transaction_key, prepared, owner, database, heuristic)
);`
)

func pk(name string) descpb.IndexDescriptor {
	return descpb.IndexDescriptor{
		Name:                tabledesc.LegacyPrimaryKeyIndexName,
		ID:                  1,
		Unique:              true,
		KeyColumnNames:      []string{name},
		KeyColumnDirections: singleASC,
		KeyColumnIDs:        singleID1,
	}
}

// Helpers used to make some of the descpb.TableDescriptor literals below more concise.
var (
	singleASC = []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC}
	singleID1 = []descpb.ColumnID{1}

	// The hash computation expression below is generated by running the CREATE
	// TABLE statements for both statement and transaction tables in a SQL shell.
	// If we are to change how we compute hash values in the future, we need to
	// modify these two expressions as well.
	sqlStmtHashComputeExpr   = `mod(fnv32(crdb_internal.datums_to_bytes(aggregated_ts, app_name, fingerprint_id, node_id, plan_hash, transaction_fingerprint_id)), 8:::INT8)`
	sqlTxnHashComputeExpr    = `mod(fnv32(crdb_internal.datums_to_bytes(aggregated_ts, app_name, fingerprint_id, node_id)), 8:::INT8)`
	timestampHashComputeExpr = `mod(fnv32(md5(crdb_internal.datums_to_bytes(created_at))), 16:::INT8)`
)

const (
	// SQLStatsHashShardBucketCount is the number of buckets used in the hash sharded
	// primary key in the sql stats tables. If we are to change the number of buckets
	// in the hash sharded primary key in the sql stats tables, this value needs to
	// be updated.
	SQLStatsHashShardBucketCount = 8

	// StmtStatsHashColumnName is the name of the hash column of
	// system.statement_statistics.
	StmtStatsHashColumnName = "crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_plan_hash_transaction_fingerprint_id_shard_8"

	// TxnStatsHashColumnName is the name of the hash column of
	// system.transaction_statistics.
	TxnStatsHashColumnName = "crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_shard_8"
)

// SystemDatabaseName is the name of the system database.
const SystemDatabaseName = catconstants.SystemDatabaseName

// SystemDatabaseSchemaBootstrapVersion is the system database schema version
// that should be used during bootstrap. It should be bumped up alongside any
// upgrade that has an associated migration (and the upgrade for the final
// release version).
//
// NB: Don't set this to clusterversion.Latest; use a specific version instead.
var SystemDatabaseSchemaBootstrapVersion = clusterversion.V25_1_JobsBackfill.Version()

// MakeSystemDatabaseDesc constructs a copy of the system database
// descriptor.
func MakeSystemDatabaseDesc() catalog.DatabaseDescriptor {
	priv := privilege.List{privilege.CONNECT}
	return dbdesc.NewBuilder(&descpb.DatabaseDescriptor{
		Name:    SystemDatabaseName,
		ID:      keys.SystemDatabaseID,
		Version: 1,
		// Assign max privileges to root user.
		Privileges: catpb.NewCustomSuperuserPrivilegeDescriptor(
			priv, username.NodeUserName()),
		SystemDatabaseSchemaVersion: &SystemDatabaseSchemaBootstrapVersion,
	}).BuildImmutableDatabase()
}

func systemTable(
	name catconstants.SystemTableName,
	id descpb.ID,
	columns []descpb.ColumnDescriptor,
	families []descpb.ColumnFamilyDescriptor,
	indexes ...descpb.IndexDescriptor,
) descpb.TableDescriptor {
	tbl := descpb.TableDescriptor{
		Name:                    string(name),
		ID:                      id,
		ParentID:                keys.SystemDatabaseID,
		UnexposedParentSchemaID: keys.SystemPublicSchemaID,
		Version:                 1,
		Columns:                 columns,
		Families:                families,
		PrimaryIndex:            indexes[0],
		Indexes:                 indexes[1:],
		FormatVersion:           descpb.InterleavedFormatVersion,
		NextMutationID:          1,
		NextConstraintID:        1,
	}
	for _, col := range columns {
		if tbl.NextColumnID <= col.ID {
			tbl.NextColumnID = col.ID + 1
		}
	}
	for _, fam := range families {
		if tbl.NextFamilyID <= fam.ID {
			tbl.NextFamilyID = fam.ID + 1
		}
	}
	tbl.NextIndexID = tbl.PrimaryIndex.ID + 1
	for i, idx := range indexes {
		if tbl.NextIndexID <= idx.ID {
			tbl.NextIndexID = idx.ID + 1
		}
		// Only assigned constraint IDs to unique non-primary indexes.
		if idx.Unique && i >= 1 {
			tbl.Indexes[i-1].ConstraintID = tbl.NextConstraintID
			tbl.NextConstraintID++
		}
	}

	// When creating tables normally, unique index constraint ids are
	// assigned before the primary index.
	tbl.PrimaryIndex.ConstraintID = tbl.NextConstraintID
	tbl.NextConstraintID++

	// Make sure all primary indexes have the correct encoding and version,
	// which also requires the stored columns to be set.
	tbl.PrimaryIndex.Version = descpb.PrimaryIndexWithStoredColumnsVersion
	tbl.PrimaryIndex.EncodingType = catenumpb.PrimaryIndexEncoding
	storedColumnIDs := make([]descpb.ColumnID, 0, len(tbl.Columns))
	storedColumnNames := make([]string, 0, len(tbl.Columns))
	keyColIDs := catalog.TableColSet{}
	for _, colID := range tbl.PrimaryIndex.KeyColumnIDs {
		keyColIDs.Add(colID)
	}
	for _, col := range tbl.Columns {
		if keyColIDs.Contains(col.ID) || col.Virtual {
			continue
		}
		storedColumnIDs = append(storedColumnIDs, col.ID)
		storedColumnNames = append(storedColumnNames, col.Name)
	}
	if len(storedColumnIDs) == 0 {
		storedColumnIDs = nil
		storedColumnNames = nil
	}
	tbl.PrimaryIndex.StoreColumnIDs = storedColumnIDs
	tbl.PrimaryIndex.StoreColumnNames = storedColumnNames

	return tbl
}

// SystemTable wraps the catalog descriptor with extra fields that are used to
// validate the descriptor.
type SystemTable struct {
	catalog.TableDescriptor

	// Schema is a CREATE TABLE string that would create the sytem database.
	Schema string
}

func makeSystemTable(
	createTableStmt string, tbl descpb.TableDescriptor, fns ...func(tbl *descpb.TableDescriptor),
) SystemTable {
	ctx := context.Background()
	{
		nameInfo := descpb.NameInfo{
			ParentID:       tbl.ParentID,
			ParentSchemaID: tbl.UnexposedParentSchemaID,
			Name:           tbl.Name,
		}
		privs := catprivilege.SystemSuperuserPrivileges(nameInfo)
		if privs == nil {
			log.Fatalf(ctx, "no superuser privileges found when building descriptor of system table %q", tbl.Name)
		}
		tbl.Privileges = catpb.NewCustomSuperuserPrivilegeDescriptor(privs, username.NodeUserName())
	}
	for _, fn := range fns {
		fn(&tbl)
	}
	b := tabledesc.NewBuilder(&tbl)
	return SystemTable{
		Schema:          createTableStmt,
		TableDescriptor: b.BuildImmutableTable(),
	}
}

var (
	// SystemDB is the descriptor for the system database.
	SystemDB = MakeSystemDatabaseDesc()
)

// MakeSystemTables produces a list of all tables in the system database.
func MakeSystemTables() []SystemTable {
	return []SystemTable{
		NamespaceTable,
		DescriptorTable,
		UsersTable,
		ZonesTable,
		SettingsTable,
		DescIDSequence,
		RoleIDSequence,
		TenantIDSequence,
		TenantsTable,
		LeaseTable(),
		EventLogTable,
		RangeEventTable,
		UITable,
		JobsTable,
		SystemJobInfoTable,
		WebSessionsTable,
		TableStatisticsTable,
		LocationsTable,
		RoleMembersTable,
		CommentsTable,
		ReportsMetaTable,
		ReplicationConstraintStatsTable,
		ReplicationCriticalLocalitiesTable,
		ReplicationStatsTable,
		ProtectedTimestampsMetaTable,
		ProtectedTimestampsRecordsTable,
		RoleOptionsTable,
		StatementBundleChunksTable,
		StatementDiagnosticsRequestsTable,
		StatementDiagnosticsTable,
		ScheduledJobsTable,
		SqllivenessTable(),
		MigrationsTable,
		JoinTokensTable,
		StatementStatisticsTable,
		TransactionStatisticsTable,
		DatabaseRoleSettingsTable,
		TenantUsageTable,
		SQLInstancesTable(),
		SpanConfigurationsTable,
		TenantSettingsTable,
		SpanCountTable,
		SystemPrivilegeTable,
		SystemExternalConnectionsTable,
		SpanStatsTenantBoundariesTable,
		SpanStatsUniqueKeysTable,
		SpanStatsBucketsTable,
		SpanStatsSamplesTable,
		SystemTaskPayloadsTable,
		SystemTenantTasksTable,
		StatementActivityTable,
		TransactionActivityTable,
		RegionLivenessTable,
		SystemMVCCStatisticsTable,
		StatementExecInsightsTable,
		TransactionExecInsightsTable,
		TableMetadata,
		SystemJobProgressTable,
		SystemJobProgressHistoryTable,
		SystemJobStatusTable,
		SystemJobMessageTable,
		PreparedTransactionsTable,
	}
}

// These system config descpb.TableDescriptor literals should match the descriptor
// that would be produced by evaluating one of the above `CREATE TABLE`
// statements. See the `TestSystemTableLiterals` which checks that they do
// indeed match, and has suggestions on writing and maintaining them.
var (

	// NamespaceTable is the descriptor for the namespace table. Note that this
	// table should only be written to via KV puts, not via the SQL layer. Some
	// code assumes that it only has KV entries for column family 4, not the
	// "sentinel" column family 0 which would be written by SQL.
	NamespaceTable = makeSystemTable(
		NamespaceTableSchema,
		systemTable(
			catconstants.NamespaceTableName,
			keys.NamespaceTableID,
			[]descpb.ColumnDescriptor{
				{Name: "parentID", ID: 1, Type: types.Int},
				{Name: "parentSchemaID", ID: 2, Type: types.Int},
				{Name: "name", ID: 3, Type: types.String},
				{Name: "id", ID: 4, Type: types.Int, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{Name: "primary", ID: 0, ColumnNames: []string{"parentID", "parentSchemaID", "name"}, ColumnIDs: []descpb.ColumnID{1, 2, 3}},
				{Name: "fam_4_id", ID: catconstants.NamespaceTableFamilyID, ColumnNames: []string{"id"}, ColumnIDs: []descpb.ColumnID{4}, DefaultColumnID: 4},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  catconstants.NamespaceTablePrimaryIndexID,
				Unique:              true,
				KeyColumnNames:      []string{"parentID", "parentSchemaID", "name"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2, 3},
			},
		))

	// DescriptorTable is the descriptor for the descriptor table.
	DescriptorTable = makeSystemTable(
		DescriptorTableSchema,
		systemTable(
			catconstants.DescriptorTableName,
			keys.DescriptorTableID,
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Int},
				{Name: "descriptor", ID: keys.DescriptorTableDescriptorColID, Type: types.Bytes, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				// The id of the first col fam is hardcoded in keys.MakeDescMetadataKey().
				{Name: "primary", ID: 0, ColumnNames: []string{"id"}, ColumnIDs: singleID1},
				{
					Name: "fam_2_descriptor", ID: keys.DescriptorTableDescriptorColFamID,
					ColumnNames:     []string{"descriptor"},
					ColumnIDs:       []descpb.ColumnID{keys.DescriptorTableDescriptorColID},
					DefaultColumnID: keys.DescriptorTableDescriptorColID,
				},
			},
			pk("id"),
		))

	falseBoolString = "false"
	trueBoolString  = "true"
	zeroIntString   = "0:::INT8"

	// UsersTable is the descriptor for the users table.
	UsersTable = makeSystemTable(
		UsersTableSchema,
		systemTable(
			catconstants.UsersTableName,
			keys.UsersTableID,
			[]descpb.ColumnDescriptor{
				{Name: "username", ID: 1, Type: types.String},
				{Name: "hashedPassword", ID: 2, Type: types.Bytes, Nullable: true},
				{Name: "isRole", ID: 3, Type: types.Bool, DefaultExpr: &falseBoolString},
				{Name: "user_id", ID: 4, Type: types.Oid},
			},
			[]descpb.ColumnFamilyDescriptor{
				{Name: "primary", ID: 0, ColumnNames: []string{"username", "user_id"}, ColumnIDs: []descpb.ColumnID{1, 4}, DefaultColumnID: 4},
				{Name: "fam_2_hashedPassword", ID: 2, ColumnNames: []string{"hashedPassword"}, ColumnIDs: []descpb.ColumnID{2}, DefaultColumnID: 2},
				{Name: "fam_3_isRole", ID: 3, ColumnNames: []string{"isRole"}, ColumnIDs: []descpb.ColumnID{3}, DefaultColumnID: 3},
			},
			pk("username"),
			descpb.IndexDescriptor{
				Name:                "users_user_id_idx",
				ID:                  2,
				Unique:              true,
				KeyColumnNames:      []string{"user_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{4},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		))

	// ZonesTable is the descriptor for the zones table.
	ZonesTable = makeSystemTable(
		ZonesTableSchema,
		systemTable(
			catconstants.ZonesTableName,
			keys.ZonesTableID,
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Int},
				{Name: "config", ID: keys.ZonesTableConfigColumnID, Type: types.Bytes, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{Name: "primary", ID: 0, ColumnNames: []string{"id"}, ColumnIDs: singleID1},
				{Name: "fam_2_config", ID: keys.ZonesTableConfigColFamID, ColumnNames: []string{"config"},
					ColumnIDs: []descpb.ColumnID{keys.ZonesTableConfigColumnID}, DefaultColumnID: keys.ZonesTableConfigColumnID},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  keys.ZonesTablePrimaryIndexID,
				Unique:              true,
				KeyColumnNames:      []string{"id"},
				KeyColumnDirections: singleASC,
				KeyColumnIDs:        []descpb.ColumnID{keys.ZonesTablePrimaryIndexID},
			},
		))

	// SettingsTable is the descriptor for the settings table.
	// It contains all cluster settings for which a value has been set.
	SettingsTable = makeSystemTable(
		SettingsTableSchema,
		systemTable(
			catconstants.SettingsTableName,
			keys.SettingsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "name", ID: 1, Type: types.String},
				{Name: "value", ID: 2, Type: types.String},
				{Name: "lastUpdated", ID: 3, Type: types.Timestamp, DefaultExpr: &nowString},
				{Name: "valueType", ID: 4, Type: types.String, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "fam_0_name_value_lastUpdated_valueType",
					ID:          0,
					ColumnNames: []string{"name", "value", "lastUpdated", "valueType"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4},
				},
			},
			pk("name"),
		))

	// DescIDSequence is the descriptor for the descriptor ID sequence.
	DescIDSequence = makeSystemTable(
		DescIDSequenceSchema,
		systemTable(
			catconstants.DescIDSequenceTableName,
			keys.DescIDSequenceID,
			[]descpb.ColumnDescriptor{
				{Name: tabledesc.SequenceColumnName, ID: tabledesc.SequenceColumnID, Type: types.Int},
			},
			[]descpb.ColumnFamilyDescriptor{{
				Name:            "primary",
				ID:              keys.SequenceColumnFamilyID,
				ColumnNames:     []string{tabledesc.SequenceColumnName},
				ColumnIDs:       []descpb.ColumnID{tabledesc.SequenceColumnID},
				DefaultColumnID: tabledesc.SequenceColumnID,
			}},
			descpb.IndexDescriptor{
				ID:                  keys.SequenceIndexID,
				Name:                tabledesc.LegacyPrimaryKeyIndexName,
				KeyColumnIDs:        []descpb.ColumnID{tabledesc.SequenceColumnID},
				KeyColumnNames:      []string{tabledesc.SequenceColumnName},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.SequenceOpts = &descpb.TableDescriptor_SequenceOpts{
				Increment: 1,
				MinValue:  1,
				MaxValue:  math.MaxInt64,
				Start:     1,
				CacheSize: 1,
			}
			tbl.NextColumnID = 0
			tbl.NextFamilyID = 0
			tbl.NextIndexID = 0
			tbl.NextMutationID = 0
			// Sequences never exposed their internal constraints,
			// so all IDs will be left at zero. CREATE SEQUENCE has
			// the same behaviour.
			tbl.NextConstraintID = 0
			tbl.PrimaryIndex.ConstraintID = 0
		},
	)

	// RoleIDSequence is the descriptor for the role id sequence.
	RoleIDSequence = makeSystemTable(
		RoleIDSequenceSchema,
		systemTable(
			catconstants.RoleIDSequenceName,
			keys.RoleIDSequenceID,
			[]descpb.ColumnDescriptor{
				{Name: tabledesc.SequenceColumnName, ID: tabledesc.SequenceColumnID, Type: types.Int},
			},
			[]descpb.ColumnFamilyDescriptor{{
				Name:            "primary",
				ID:              keys.SequenceColumnFamilyID,
				ColumnNames:     []string{tabledesc.SequenceColumnName},
				ColumnIDs:       []descpb.ColumnID{tabledesc.SequenceColumnID},
				DefaultColumnID: tabledesc.SequenceColumnID,
			}},
			descpb.IndexDescriptor{
				ID:                  keys.SequenceIndexID,
				Name:                tabledesc.LegacyPrimaryKeyIndexName,
				KeyColumnIDs:        []descpb.ColumnID{tabledesc.SequenceColumnID},
				KeyColumnNames:      []string{tabledesc.SequenceColumnName},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
			},
		),
		func(tbl *descpb.TableDescriptor) {
			opts := &descpb.TableDescriptor_SequenceOpts{
				Increment: 1,
				MinValue:  100,
				MaxValue:  math.MaxInt32,
				Start:     100,
				CacheSize: 1,
			}
			tbl.SequenceOpts = opts
			tbl.NextColumnID = 0
			tbl.NextFamilyID = 0
			tbl.NextIndexID = 0
			tbl.NextMutationID = 0
			// Sequences never exposed their internal constraints,
			// so all IDs will be left at zero. CREATE SEQUENCE has
			// the same behaviour.
			tbl.NextConstraintID = 0
			tbl.PrimaryIndex.ConstraintID = 0
		},
	)

	// TenantIDSequence is the descriptor for the descriptor ID sequence.
	TenantIDSequence = makeSystemTable(
		TenantIDSequenceSchema,
		systemTable(
			catconstants.TenantIDSequenceTableName,
			descpb.InvalidID, // dynamically assigned table ID
			[]descpb.ColumnDescriptor{
				{Name: tabledesc.SequenceColumnName, ID: tabledesc.SequenceColumnID, Type: types.Int},
			},
			[]descpb.ColumnFamilyDescriptor{{
				Name:            "primary",
				ID:              keys.SequenceColumnFamilyID,
				ColumnNames:     []string{tabledesc.SequenceColumnName},
				ColumnIDs:       []descpb.ColumnID{tabledesc.SequenceColumnID},
				DefaultColumnID: tabledesc.SequenceColumnID,
			}},
			descpb.IndexDescriptor{
				ID:                  keys.SequenceIndexID,
				Name:                tabledesc.LegacyPrimaryKeyIndexName,
				KeyColumnIDs:        []descpb.ColumnID{tabledesc.SequenceColumnID},
				KeyColumnNames:      []string{tabledesc.SequenceColumnName},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.SequenceOpts = &descpb.TableDescriptor_SequenceOpts{
				Increment: 1,
				MinValue:  1,
				MaxValue:  math.MaxInt64,
				Start:     1,
				CacheSize: 1,
			}
			tbl.NextColumnID = 0
			tbl.NextFamilyID = 0
			tbl.NextIndexID = 0
			tbl.NextMutationID = 0
			// Sequences never exposed their internal constraints,
			// so all IDs will be left at zero. CREATE SEQUENCE has
			// the same behaviour.
			tbl.NextConstraintID = 0
			tbl.PrimaryIndex.ConstraintID = 0
		},
	)

	TenantsTable = makeSystemTable(
		TenantsTableSchema,
		systemTable(
			catconstants.TenantsTableName,
			keys.TenantsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Int},
				{Name: "active", ID: 2, Type: types.Bool, DefaultExpr: &trueBoolString, Hidden: true},
				{Name: "info", ID: 3, Type: types.Bytes, Nullable: true},
				{Name: "name", ID: 4, Type: types.String, Nullable: true},
				{Name: "data_state", ID: 5, Type: types.Int, Nullable: true},
				{Name: "service_mode", ID: 6, Type: types.Int, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{{
				Name:        "primary",
				ID:          0,
				ColumnNames: []string{"id", "active", "info", "name", "data_state", "service_mode"},
				ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4, 5, 6},
			}},
			pk("id"),
			descpb.IndexDescriptor{
				Name:                "tenants_name_idx",
				ID:                  2,
				Unique:              true,
				KeyColumnNames:      []string{"name"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{4},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:                "tenants_service_mode_idx",
				ID:                  3,
				KeyColumnNames:      []string{"service_mode"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{6},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		))
)

// These system descpb.TableDescriptor literals should match the descriptor that
// would be produced by evaluating one of the above `CREATE TABLE` statements
// for system tables that are not system config tables. See the
// `TestSystemTableLiterals` which checks that they do indeed match, and has
// suggestions on writing and maintaining them.
var (
	LeaseTableTTL = time.Minute * 10
	// LeaseTable is the descriptor for the leases table with a session based
	// leasing table format.
	LeaseTable = func() SystemTable {
		return makeSystemTable(
			LeaseTableSchema_V24_1,
			systemTable(
				catconstants.LeaseTableName,
				keys.LeaseTableID,
				[]descpb.ColumnDescriptor{
					{Name: "desc_id", ID: 1, Type: types.Int},
					{Name: "version", ID: 2, Type: types.Int},
					{Name: "sql_instance_id", ID: 3, Type: types.Int},
					{Name: "session_id", ID: 4, Type: types.Bytes},
					{Name: "crdb_region", ID: 5, Type: types.Bytes},
				},
				[]descpb.ColumnFamilyDescriptor{
					{
						Name:            "primary",
						ID:              0,
						ColumnNames:     []string{"desc_id", "version", "sql_instance_id", "session_id", "crdb_region"},
						DefaultColumnID: 3,
						ColumnIDs:       []descpb.ColumnID{1, 2, 3, 4, 5},
					},
				},
				descpb.IndexDescriptor{
					Name:           "primary",
					ID:             3,
					Unique:         true,
					KeyColumnNames: []string{"crdb_region", "desc_id", "version", "session_id"},
					KeyColumnDirections: []catenumpb.IndexColumn_Direction{
						catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC,
					},
					KeyColumnIDs: []descpb.ColumnID{5, 1, 2, 4},
				},
			),
			func(tbl *descpb.TableDescriptor) {
				tbl.ExcludeDataFromBackup = true
			},
		)
	}

	// LeaseTable_V23_2 is the descriptor for the leases table with an expiry based
	// format
	LeaseTable_V23_2 = func() SystemTable {
		return makeSystemTable(
			LeaseTableSchema,
			systemTable(
				catconstants.LeaseTableName,
				keys.LeaseTableID,
				[]descpb.ColumnDescriptor{
					{Name: "descID", ID: 1, Type: types.Int},
					{Name: "version", ID: 2, Type: types.Int},
					{Name: "nodeID", ID: 3, Type: types.Int},
					{Name: "expiration", ID: 4, Type: types.Timestamp},
					{Name: "crdb_region", ID: 5, Type: types.Bytes},
				},
				[]descpb.ColumnFamilyDescriptor{
					{
						Name:        "primary",
						ID:          0,
						ColumnNames: []string{"descID", "version", "nodeID", "expiration", "crdb_region"},
						ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4, 5},
					},
				},
				descpb.IndexDescriptor{
					Name:           "primary",
					ID:             2,
					Unique:         true,
					KeyColumnNames: []string{"crdb_region", "descID", "version", "expiration", "nodeID"},
					KeyColumnDirections: []catenumpb.IndexColumn_Direction{
						catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC,
						catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC,
					},
					KeyColumnIDs: []descpb.ColumnID{5, 1, 2, 4, 3},
				},
			),
		)
	}
	V22_2_LeaseTable = func() SystemTable {
		return makeSystemTable(
			LeaseTableSchema,
			systemTable(
				catconstants.LeaseTableName,
				keys.LeaseTableID,
				[]descpb.ColumnDescriptor{
					{Name: "descID", ID: 1, Type: types.Int},
					{Name: "version", ID: 2, Type: types.Int},
					{Name: "nodeID", ID: 3, Type: types.Int},
					{Name: "expiration", ID: 4, Type: types.Timestamp},
				},
				[]descpb.ColumnFamilyDescriptor{
					{Name: "primary", ID: 0, ColumnNames: []string{"descID", "version", "nodeID", "expiration"}, ColumnIDs: []descpb.ColumnID{1, 2, 3, 4}},
				},
				descpb.IndexDescriptor{
					Name:                "primary",
					ID:                  1,
					Unique:              true,
					KeyColumnNames:      []string{"descID", "version", "expiration", "nodeID"},
					KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
					KeyColumnIDs:        []descpb.ColumnID{1, 2, 4, 3},
				},
			))
	}

	uuidV4String = "uuid_v4()"

	// EventLogTable is the descriptor for the event log table.
	EventLogTable = makeSystemTable(
		EventLogTableSchema,
		systemTable(
			catconstants.EventLogTableName,
			keys.EventLogTableID,
			[]descpb.ColumnDescriptor{
				{Name: "timestamp", ID: 1, Type: types.Timestamp},
				{Name: "eventType", ID: 2, Type: types.String},
				// TODO(knz): targetID and reportingID are deprecated and should
				// be removed after v21.1 is released. Their content is now
				// available inside the info payload, which is a JSON blob.
				{Name: "targetID", ID: 3, Type: types.Int},
				{Name: "reportingID", ID: 4, Type: types.Int},
				{Name: "info", ID: 5, Type: types.String, Nullable: true},
				{Name: "uniqueID", ID: 6, Type: types.Bytes, DefaultExpr: &uuidV4String},
			},
			[]descpb.ColumnFamilyDescriptor{
				{Name: "primary", ID: 0, ColumnNames: []string{"timestamp", "uniqueID"}, ColumnIDs: []descpb.ColumnID{1, 6}},
				{Name: "fam_2_eventType", ID: 2, ColumnNames: []string{"eventType"}, ColumnIDs: []descpb.ColumnID{2}, DefaultColumnID: 2},
				{Name: "fam_3_targetID", ID: 3, ColumnNames: []string{"targetID"}, ColumnIDs: []descpb.ColumnID{3}, DefaultColumnID: 3},
				{Name: "fam_4_reportingID", ID: 4, ColumnNames: []string{"reportingID"}, ColumnIDs: []descpb.ColumnID{4}, DefaultColumnID: 4},
				{Name: "fam_5_info", ID: 5, ColumnNames: []string{"info"}, ColumnIDs: []descpb.ColumnID{5}, DefaultColumnID: 5},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"timestamp", "uniqueID"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1, 6},
			},
		))

	uniqueRowIDString = "unique_rowid()"

	// RangeEventTable is the descriptor for the range log table.
	RangeEventTable = makeSystemTable(
		RangeEventTableSchema,
		systemTable(
			catconstants.RangeEventTableName,
			keys.RangeEventTableID,
			[]descpb.ColumnDescriptor{
				{Name: "timestamp", ID: 1, Type: types.Timestamp},
				{Name: "rangeID", ID: 2, Type: types.Int},
				{Name: "storeID", ID: 3, Type: types.Int},
				{Name: "eventType", ID: 4, Type: types.String},
				{Name: "otherRangeID", ID: 5, Type: types.Int, Nullable: true},
				{Name: "info", ID: 6, Type: types.String, Nullable: true},
				{Name: "uniqueID", ID: 7, Type: types.Int, DefaultExpr: &uniqueRowIDString},
			},
			[]descpb.ColumnFamilyDescriptor{
				{Name: "primary", ID: 0, ColumnNames: []string{"timestamp", "uniqueID"}, ColumnIDs: []descpb.ColumnID{1, 7}},
				{Name: "fam_2_rangeID", ID: 2, ColumnNames: []string{"rangeID"}, ColumnIDs: []descpb.ColumnID{2}, DefaultColumnID: 2},
				{Name: "fam_3_storeID", ID: 3, ColumnNames: []string{"storeID"}, ColumnIDs: []descpb.ColumnID{3}, DefaultColumnID: 3},
				{Name: "fam_4_eventType", ID: 4, ColumnNames: []string{"eventType"}, ColumnIDs: []descpb.ColumnID{4}, DefaultColumnID: 4},
				{Name: "fam_5_otherRangeID", ID: 5, ColumnNames: []string{"otherRangeID"}, ColumnIDs: []descpb.ColumnID{5}, DefaultColumnID: 5},
				{Name: "fam_6_info", ID: 6, ColumnNames: []string{"info"}, ColumnIDs: []descpb.ColumnID{6}, DefaultColumnID: 6},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"timestamp", "uniqueID"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1, 7},
			},
		))

	// UITable is the descriptor for the ui table.
	UITable = makeSystemTable(
		UITableSchema,
		systemTable(
			catconstants.UITableName,
			keys.UITableID,
			[]descpb.ColumnDescriptor{
				{Name: "key", ID: 1, Type: types.String},
				{Name: "value", ID: 2, Type: types.Bytes, Nullable: true},
				{Name: "lastUpdated", ID: 3, Type: types.Timestamp},
			},
			[]descpb.ColumnFamilyDescriptor{
				{Name: "primary", ID: 0, ColumnNames: []string{"key"}, ColumnIDs: singleID1},
				{Name: "fam_2_value", ID: 2, ColumnNames: []string{"value"}, ColumnIDs: []descpb.ColumnID{2}, DefaultColumnID: 2},
				{Name: "fam_3_lastUpdated", ID: 3, ColumnNames: []string{"lastUpdated"}, ColumnIDs: []descpb.ColumnID{3}, DefaultColumnID: 3},
			},
			pk("key"),
		))

	nowString = "now():::TIMESTAMP"

	// JobsTable is the descriptor for the jobs table.
	JobsTable = makeSystemTable(
		JobsTableSchema,
		systemTable(
			catconstants.JobsTableName,
			keys.JobsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Int, DefaultExpr: &uniqueRowIDString},
				{Name: "status", ID: 2, Type: types.String},
				{Name: "created", ID: 3, Type: types.Timestamp, DefaultExpr: &nowString},
				{Name: "dropped_payload", ID: 4, Type: types.Bytes, Nullable: true, Hidden: true},
				{Name: "dropped_progress", ID: 5, Type: types.Bytes, Nullable: true, Hidden: true},
				{Name: "created_by_type", ID: 6, Type: types.String, Nullable: true},
				{Name: "created_by_id", ID: 7, Type: types.Int, Nullable: true},
				{Name: "claim_session_id", ID: 8, Type: types.Bytes, Nullable: true},
				{Name: "claim_instance_id", ID: 9, Type: types.Int, Nullable: true},
				{Name: "num_runs", ID: 10, Type: types.Int, Nullable: true},
				{Name: "last_run", ID: 11, Type: types.Timestamp, Nullable: true},
				{Name: "job_type", ID: 12, Type: types.String, Nullable: true},
				{Name: "owner", ID: 13, Type: types.String, Nullable: true},
				{Name: "description", ID: 14, Type: types.String, Nullable: true},
				{Name: "error_msg", ID: 15, Type: types.String, Nullable: true},
				{Name: "finished", ID: 16, Type: types.TimestampTZ, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					// NB: We are using family name that existed prior to adding created_by_type,
					// created_by_id, and job_type columns. This is done to minimize and simplify migration work
					// that needed to be done.
					Name:        "fam_0_id_status_created_payload",
					ID:          0,
					ColumnNames: []string{"id", "status", "created", "dropped_payload", "created_by_type", "created_by_id", "job_type", "owner", "description", "error_msg", "finished"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4, 6, 7, 12, 13, 14, 15, 16},
				},
				{
					// NB: We are using family name that existed prior to adding created_by_type,
					// created_by_id, and job_type columns. This is done to minimize and simplify migration work
					// that needed to be done.
					Name:            "progress",
					ID:              1,
					ColumnNames:     []string{"dropped_progress"},
					ColumnIDs:       []descpb.ColumnID{5},
					DefaultColumnID: 5,
				},
				{
					Name:        "claim",
					ID:          2,
					ColumnNames: []string{"claim_session_id", "claim_instance_id", "num_runs", "last_run"},
					ColumnIDs:   []descpb.ColumnID{8, 9, 10, 11},
				},
			},
			pk("id"),
			descpb.IndexDescriptor{
				Name:                "jobs_status_created_idx",
				ID:                  2,
				Unique:              false,
				KeyColumnNames:      []string{"status", "created"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{2, 3},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:                "jobs_created_by_type_created_by_id_idx",
				ID:                  3,
				Unique:              false,
				KeyColumnNames:      []string{"created_by_type", "created_by_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{6, 7},
				StoreColumnIDs:      []descpb.ColumnID{2},
				StoreColumnNames:    []string{"status"},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:                "jobs_run_stats_idx",
				ID:                  4,
				Unique:              false,
				KeyColumnNames:      []string{"claim_session_id", "status", "created"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{8, 2, 3},
				StoreColumnNames:    []string{"last_run", "num_runs", "claim_instance_id"},
				StoreColumnIDs:      []descpb.ColumnID{11, 10, 9},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
				Predicate:           JobsRunStatsIdxPredicate,
			},
			descpb.IndexDescriptor{
				Name:                "jobs_job_type_idx",
				ID:                  5,
				Unique:              false,
				KeyColumnNames:      []string{"job_type"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{12},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		))

	// WebSessions table to authenticate sessions over stateless connections.
	WebSessionsTable = makeSystemTable(
		WebSessionsTableSchema,
		systemTable(
			catconstants.WebSessionsTableName,
			keys.WebSessionsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Int, DefaultExpr: &uniqueRowIDString},
				{Name: "hashedSecret", ID: 2, Type: types.Bytes},
				{Name: "username", ID: 3, Type: types.String},
				{Name: "createdAt", ID: 4, Type: types.Timestamp, DefaultExpr: &nowString},
				{Name: "expiresAt", ID: 5, Type: types.Timestamp},
				{Name: "revokedAt", ID: 6, Type: types.Timestamp, Nullable: true},
				{Name: "lastUsedAt", ID: 7, Type: types.Timestamp, DefaultExpr: &nowString},
				{Name: "auditInfo", ID: 8, Type: types.String, Nullable: true},
				{Name: "user_id", ID: 9, Type: types.Oid},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "fam_0_id_hashedSecret_username_createdAt_expiresAt_revokedAt_lastUsedAt_auditInfo",
					ID:   0,
					ColumnNames: []string{
						"id",
						"hashedSecret",
						"username",
						"createdAt",
						"expiresAt",
						"revokedAt",
						"lastUsedAt",
						"auditInfo",
						"user_id",
					},
					ColumnIDs: []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7, 8, 9},
				},
			},
			pk("id"),
			descpb.IndexDescriptor{
				Name:                "web_sessions_expiresAt_idx",
				ID:                  2,
				Unique:              false,
				KeyColumnNames:      []string{"expiresAt"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{5},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:                "web_sessions_createdAt_idx",
				ID:                  3,
				Unique:              false,
				KeyColumnNames:      []string{"createdAt"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{4},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:                "web_sessions_revokedAt_idx",
				ID:                  4,
				Unique:              false,
				KeyColumnNames:      []string{"revokedAt"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{6},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:                "web_sessions_lastUsedAt_idx",
				ID:                  5,
				Unique:              false,
				KeyColumnNames:      []string{"lastUsedAt"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{7},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		))

	// TableStatistics table to hold statistics about columns and column groups.
	TableStatisticsTable = makeSystemTable(
		TableStatisticsTableSchema,
		systemTable(
			catconstants.TableStatisticsTableName,
			keys.TableStatisticsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "tableID", ID: 1, Type: types.Int},
				{Name: "statisticID", ID: 2, Type: types.Int, DefaultExpr: &uniqueRowIDString},
				{Name: "name", ID: 3, Type: types.String, Nullable: true},
				{Name: "columnIDs", ID: 4, Type: types.IntArray},
				{Name: "createdAt", ID: 5, Type: types.Timestamp, DefaultExpr: &nowString},
				{Name: "rowCount", ID: 6, Type: types.Int},
				{Name: "distinctCount", ID: 7, Type: types.Int},
				{Name: "nullCount", ID: 8, Type: types.Int},
				{Name: "histogram", ID: 9, Type: types.Bytes, Nullable: true},
				{Name: "avgSize", ID: 10, Type: types.Int, DefaultExpr: &zeroIntString},
				{Name: "partialPredicate", ID: 11, Type: types.String, Nullable: true},
				{Name: "fullStatisticID", ID: 12, Type: types.Int, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "fam_0_tableID_statisticID_name_columnIDs_createdAt_rowCount_distinctCount_nullCount_histogram",
					ID:   0,
					ColumnNames: []string{
						"tableID",
						"statisticID",
						"name",
						"columnIDs",
						"createdAt",
						"rowCount",
						"distinctCount",
						"nullCount",
						"histogram",
						"avgSize",
						"partialPredicate",
						"fullStatisticID",
					},
					ColumnIDs: []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12},
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"tableID", "statisticID"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2},
			},
		))

	latLonDecimal = types.MakeDecimal(18, 15)

	// LocationsTable is the descriptor for the locations table.
	LocationsTable = makeSystemTable(
		LocationsTableSchema,
		systemTable(
			catconstants.LocationsTableName,
			keys.LocationsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "localityKey", ID: 1, Type: types.String},
				{Name: "localityValue", ID: 2, Type: types.String},
				{Name: "latitude", ID: 3, Type: latLonDecimal},
				{Name: "longitude", ID: 4, Type: latLonDecimal},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "fam_0_localityKey_localityValue_latitude_longitude",
					ID:          0,
					ColumnNames: []string{"localityKey", "localityValue", "latitude", "longitude"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4},
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"localityKey", "localityValue"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2},
			},
		))

	// RoleMembersTable is the descriptor for the role_members table.
	RoleMembersTable = makeSystemTable(
		RoleMembersTableSchema,
		systemTable(
			catconstants.RoleMembersTableName,
			keys.RoleMembersTableID,
			[]descpb.ColumnDescriptor{
				{Name: "role", ID: 1, Type: types.String},
				{Name: "member", ID: 2, Type: types.String},
				{Name: "isAdmin", ID: 3, Type: types.Bool},
				{Name: "role_id", ID: 4, Type: types.Oid},
				{Name: "member_id", ID: 5, Type: types.Oid},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ID:          0,
					ColumnNames: []string{"role", "member"},
					ColumnIDs:   []descpb.ColumnID{1, 2},
				},
				{
					Name:            "fam_3_isAdmin",
					ID:              3,
					ColumnNames:     []string{"isAdmin"},
					ColumnIDs:       []descpb.ColumnID{3},
					DefaultColumnID: 3,
				},
				{
					Name:            "fam_4_role_id",
					ID:              4,
					ColumnNames:     []string{"role_id"},
					ColumnIDs:       []descpb.ColumnID{4},
					DefaultColumnID: 4,
				},
				{
					Name:            "fam_5_member_id",
					ID:              5,
					ColumnNames:     []string{"member_id"},
					ColumnIDs:       []descpb.ColumnID{5},
					DefaultColumnID: 5,
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"role", "member"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2},
			},
			descpb.IndexDescriptor{
				Name:                "role_members_role_idx",
				ID:                  2,
				Unique:              false,
				KeyColumnNames:      []string{"role"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1},
				KeySuffixColumnIDs:  []descpb.ColumnID{2},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:                "role_members_member_idx",
				ID:                  3,
				Unique:              false,
				KeyColumnNames:      []string{"member"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{2},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:                "role_members_role_id_idx",
				ID:                  4,
				Unique:              false,
				KeyColumnNames:      []string{"role_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{4},
				KeySuffixColumnIDs:  []descpb.ColumnID{1, 2},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:                "role_members_member_id_idx",
				ID:                  5,
				Unique:              false,
				KeyColumnNames:      []string{"member_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{5},
				KeySuffixColumnIDs:  []descpb.ColumnID{1, 2},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:                "role_members_role_id_member_id_key",
				ID:                  6,
				Unique:              true,
				KeyColumnNames:      []string{"role_id", "member_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{4, 5},
				KeySuffixColumnIDs:  []descpb.ColumnID{1, 2},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		))

	// CommentsTable is the descriptor for the comments table.
	CommentsTable = makeSystemTable(
		CommentsTableSchema,
		systemTable(
			catconstants.CommentsTableName,
			keys.CommentsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "type", ID: 1, Type: types.Int},
				{Name: "object_id", ID: 2, Type: types.Int},
				{Name: "sub_id", ID: 3, Type: types.Int},
				{Name: "comment", ID: 4, Type: types.String},
			},
			[]descpb.ColumnFamilyDescriptor{
				{Name: "primary", ID: 0, ColumnNames: []string{"type", "object_id", "sub_id"}, ColumnIDs: []descpb.ColumnID{1, 2, 3}},
				{Name: "fam_4_comment", ID: keys.CommentsTableCommentColFamID, ColumnNames: []string{"comment"}, ColumnIDs: []descpb.ColumnID{4}, DefaultColumnID: 4},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  keys.CommentsTablePrimaryKeyIndexID,
				Unique:              true,
				KeyColumnNames:      []string{"type", "object_id", "sub_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2, 3},
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.Privileges.Version = catpb.Version23_2
			tbl.Privileges.Users = append(tbl.Privileges.Users, catpb.UserPrivileges{
				UserProto:  username.PublicRoleName().EncodeProto(),
				Privileges: privilege.List{privilege.SELECT}.ToBitField(),
			})
			sort.Slice(tbl.Privileges.Users, func(i, j int) bool {
				return tbl.Privileges.Users[i].User().LessThan(tbl.Privileges.Users[j].User())
			})
		},
	)

	ReportsMetaTable = makeSystemTable(
		ReportsMetaTableSchema,
		systemTable(
			catconstants.ReportsMetaTableName,
			keys.ReportsMetaTableID,
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Int},
				{Name: "generated", ID: 2, Type: types.TimestampTZ},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:            "primary",
					ID:              0,
					ColumnNames:     []string{"id", "generated"},
					ColumnIDs:       []descpb.ColumnID{1, 2},
					DefaultColumnID: 2,
				},
			},
			descpb.IndexDescriptor{
				Name:           "primary",
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1},
			},
		))

	ReplicationConstraintStatsTableTTL = time.Minute * 10
	// TODO(andrei): In 20.1 we should add a foreign key reference to the
	// reports_meta table. Until then, it would cost us having to create an index
	// on report_id.
	ReplicationConstraintStatsTable = makeSystemTable(
		ReplicationConstraintStatsTableSchema,
		systemTable(
			catconstants.ReplicationConstraintStatsTableName,
			keys.ReplicationConstraintStatsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "zone_id", ID: 1, Type: types.Int},
				{Name: "subzone_id", ID: 2, Type: types.Int},
				{Name: "type", ID: 3, Type: types.String},
				{Name: "config", ID: 4, Type: types.String},
				{Name: "report_id", ID: 5, Type: types.Int},
				{Name: "violation_start", ID: 6, Type: types.TimestampTZ, Nullable: true},
				{Name: "violating_ranges", ID: 7, Type: types.Int},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ID:   0,
					ColumnNames: []string{
						"zone_id",
						"subzone_id",
						"type",
						"config",
						"report_id",
						"violation_start",
						"violating_ranges",
					},
					ColumnIDs: []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7},
				},
			},
			descpb.IndexDescriptor{
				Name:           "primary",
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"zone_id", "subzone_id", "type", "config"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1, 2, 3, 4},
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.ExcludeDataFromBackup = true
		},
	)

	// TODO(andrei): In 20.1 we should add a foreign key reference to the
	// reports_meta table. Until then, it would cost us having to create an index
	// on report_id.
	ReplicationCriticalLocalitiesTable = makeSystemTable(
		ReplicationCriticalLocalitiesTableSchema,
		systemTable(
			catconstants.ReplicationCriticalLocalitiesTableName,
			keys.ReplicationCriticalLocalitiesTableID,
			[]descpb.ColumnDescriptor{
				{Name: "zone_id", ID: 1, Type: types.Int},
				{Name: "subzone_id", ID: 2, Type: types.Int},
				{Name: "locality", ID: 3, Type: types.String},
				{Name: "report_id", ID: 4, Type: types.Int},
				{Name: "at_risk_ranges", ID: 5, Type: types.Int},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ID:   0,
					ColumnNames: []string{
						"zone_id",
						"subzone_id",
						"locality",
						"report_id",
						"at_risk_ranges",
					},
					ColumnIDs: []descpb.ColumnID{1, 2, 3, 4, 5},
				},
			},
			descpb.IndexDescriptor{
				Name:           "primary",
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"zone_id", "subzone_id", "locality"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1, 2, 3},
			},
		))

	ReplicationStatsTableTTL = time.Minute * 10
	// TODO(andrei): In 20.1 we should add a foreign key reference to the
	// reports_meta table. Until then, it would cost us having to create an index
	// on report_id.
	ReplicationStatsTable = makeSystemTable(
		ReplicationStatsTableSchema,
		systemTable(
			"replication_stats",
			keys.ReplicationStatsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "zone_id", ID: 1, Type: types.Int},
				{Name: "subzone_id", ID: 2, Type: types.Int},
				{Name: "report_id", ID: 3, Type: types.Int},
				{Name: "total_ranges", ID: 4, Type: types.Int},
				{Name: "unavailable_ranges", ID: 5, Type: types.Int},
				{Name: "under_replicated_ranges", ID: 6, Type: types.Int},
				{Name: "over_replicated_ranges", ID: 7, Type: types.Int},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ID:   0,
					ColumnNames: []string{
						"zone_id",
						"subzone_id",
						"report_id",
						"total_ranges",
						"unavailable_ranges",
						"under_replicated_ranges",
						"over_replicated_ranges",
					},
					ColumnIDs: []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7},
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"zone_id", "subzone_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2},
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.ExcludeDataFromBackup = true
		},
	)

	ProtectedTimestampsMetaTable = makeSystemTable(
		ProtectedTimestampsMetaTableSchema,
		systemTable(
			catconstants.ProtectedTimestampsMetaTableName,
			keys.ProtectedTimestampsMetaTableID,
			[]descpb.ColumnDescriptor{
				{
					Name:        "singleton",
					ID:          1,
					Type:        types.Bool,
					DefaultExpr: &trueBoolString,
				},
				{Name: "version", ID: 2, Type: types.Int},
				{Name: "num_records", ID: 3, Type: types.Int},
				{Name: "num_spans", ID: 4, Type: types.Int},
				{Name: "total_bytes", ID: 5, Type: types.Int},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ColumnNames: []string{"singleton", "version", "num_records", "num_spans", "total_bytes"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4, 5},
				},
			},
			descpb.IndexDescriptor{
				Name:           "primary",
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"singleton"},
				KeyColumnIDs:   []descpb.ColumnID{1},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.Checks = []*descpb.TableDescriptor_CheckConstraint{{
				Name:         "check_singleton",
				Expr:         "singleton",
				ColumnIDs:    []descpb.ColumnID{1},
				ConstraintID: tbl.NextConstraintID,
			}}
			tbl.NextConstraintID++
		},
	)

	ProtectedTimestampsRecordsTable = makeSystemTable(
		ProtectedTimestampsRecordsTableSchema,
		systemTable(
			catconstants.ProtectedTimestampsRecordsTableName,
			keys.ProtectedTimestampsRecordsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Uuid},
				{Name: "ts", ID: 2, Type: types.Decimal},
				{Name: "meta_type", ID: 3, Type: types.String},
				{Name: "meta", ID: 4, Type: types.Bytes, Nullable: true},
				{Name: "num_spans", ID: 5, Type: types.Int},
				{Name: "spans", ID: 6, Type: types.Bytes},
				{Name: "verified", ID: 7, Type: types.Bool, DefaultExpr: &falseBoolString},
				// target will store an encoded protobuf indicating what the protected
				// timestamp record will protect. A record can protect either a cluster,
				// tenants or a schema objects (databases/tables).
				{Name: "target", ID: 8, Type: types.Bytes, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ColumnNames: []string{"id", "ts", "meta_type", "meta", "num_spans", "spans", "verified", "target"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7, 8},
				},
			},
			descpb.IndexDescriptor{
				Name:           "primary",
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"id"},
				KeyColumnIDs:   []descpb.ColumnID{1},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
			},
		))

	// RoleOptionsTable is the descriptor for the role_options table.
	RoleOptionsTable = makeSystemTable(
		RoleOptionsTableSchema,
		systemTable(
			catconstants.RoleOptionsTableName,
			keys.RoleOptionsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "username", ID: 1, Type: types.String},
				{Name: "option", ID: 2, Type: types.String},
				{Name: "value", ID: 3, Type: types.String, Nullable: true},
				{Name: "user_id", ID: 4, Type: types.Oid},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ColumnNames: []string{"username", "option", "value", "user_id"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4},
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"username", "option"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2},
			},
			descpb.IndexDescriptor{
				Name:                "users_user_id_idx",
				ID:                  2,
				Unique:              false,
				KeyColumnNames:      []string{"user_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{4},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
				KeySuffixColumnIDs:  []descpb.ColumnID{1, 2},
			},
		))

	StatementBundleChunksTable = makeSystemTable(
		StatementBundleChunksTableSchema,
		systemTable(
			catconstants.StatementBundleChunksTableName,
			keys.StatementBundleChunksTableID,
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Int, DefaultExpr: &uniqueRowIDString},
				{Name: "description", ID: 2, Type: types.String, Nullable: true},
				{Name: "data", ID: 3, Type: types.Bytes},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ColumnNames: []string{"id", "description", "data"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3},
				},
			},
			pk("id"),
		))

	// TODO(andrei): Add a foreign key reference to the statement_diagnostics table when
	// it no longer requires us to create an index on statement_diagnostics_id.
	StatementDiagnosticsRequestsTable = makeSystemTable(
		StatementDiagnosticsRequestsTableSchema,
		systemTable(
			catconstants.StatementDiagnosticsRequestsTableName,
			keys.StatementDiagnosticsRequestsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Int, DefaultExpr: &uniqueRowIDString, Nullable: false},
				{Name: "completed", ID: 2, Type: types.Bool, Nullable: false, DefaultExpr: &falseBoolString},
				{Name: "statement_fingerprint", ID: 3, Type: types.String, Nullable: false},
				{Name: "statement_diagnostics_id", ID: 4, Type: types.Int, Nullable: true},
				{Name: "requested_at", ID: 5, Type: types.TimestampTZ, Nullable: false},
				{Name: "min_execution_latency", ID: 6, Type: types.Interval, Nullable: true},
				{Name: "expires_at", ID: 7, Type: types.TimestampTZ, Nullable: true},
				{Name: "sampling_probability", ID: 8, Type: types.Float, Nullable: true},
				{Name: "plan_gist", ID: 9, Type: types.String, Nullable: true},
				{Name: "anti_plan_gist", ID: 10, Type: types.Bool, Nullable: true},
				{Name: "redacted", ID: 11, Type: types.Bool, Nullable: false, DefaultExpr: &falseBoolString},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ColumnNames: []string{"id", "completed", "statement_fingerprint", "statement_diagnostics_id", "requested_at", "min_execution_latency", "expires_at", "sampling_probability", "plan_gist", "anti_plan_gist", "redacted"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11},
				},
			},
			pk("id"),
			// Index for the polling query.
			descpb.IndexDescriptor{
				Name:                "completed_idx",
				ID:                  2,
				Unique:              false,
				KeyColumnNames:      []string{"completed", "id"},
				StoreColumnNames:    []string{"statement_fingerprint", "min_execution_latency", "expires_at", "sampling_probability", "plan_gist", "anti_plan_gist", "redacted"},
				KeyColumnIDs:        []descpb.ColumnID{2, 1},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				StoreColumnIDs:      []descpb.ColumnID{3, 6, 7, 8, 9, 10, 11},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.Checks = []*descpb.TableDescriptor_CheckConstraint{{
				Name:         "check_sampling_probability",
				Expr:         "sampling_probability BETWEEN 0.0:::FLOAT8 AND 1.0:::FLOAT8",
				ColumnIDs:    []descpb.ColumnID{8},
				ConstraintID: tbl.NextConstraintID,
			}}
			tbl.NextConstraintID++
		},
	)

	StatementDiagnosticsTable = makeSystemTable(
		StatementDiagnosticsTableSchema,
		systemTable(
			catconstants.StatementDiagnosticsTableName,
			keys.StatementDiagnosticsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Int, DefaultExpr: &uniqueRowIDString, Nullable: false},
				{Name: "statement_fingerprint", ID: 2, Type: types.String, Nullable: false},
				{Name: "statement", ID: 3, Type: types.String, Nullable: false},
				{Name: "collected_at", ID: 4, Type: types.TimestampTZ, Nullable: false},
				// TODO(radu): remove this column; it is no longer used.
				{Name: "trace", ID: 5, Type: types.Jsonb, Nullable: true},
				{Name: "bundle_chunks", ID: 6, Type: types.IntArray, Nullable: true},
				{Name: "error", ID: 7, Type: types.String, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ColumnNames: []string{"id", "statement_fingerprint", "statement",
						"collected_at", "trace", "bundle_chunks", "error"},
					ColumnIDs: []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7},
				},
			},
			pk("id"),
		))

	nowTZString = "now():::TIMESTAMPTZ"

	// ScheduledJobsTable is the descriptor for the scheduled jobs table.
	ScheduledJobsTable = makeSystemTable(
		ScheduledJobsTableSchema,
		systemTable(
			"scheduled_jobs",
			keys.ScheduledJobsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "schedule_id", ID: 1, Type: types.Int, DefaultExpr: &uniqueRowIDString, Nullable: false},
				{Name: "schedule_name", ID: 2, Type: types.String, Nullable: false},
				{Name: "created", ID: 3, Type: types.TimestampTZ, DefaultExpr: &nowTZString, Nullable: false},
				{Name: "owner", ID: 4, Type: types.String, Nullable: false},
				{Name: "next_run", ID: 5, Type: types.TimestampTZ, Nullable: true},
				{Name: "schedule_state", ID: 6, Type: types.Bytes, Nullable: true},
				{Name: "schedule_expr", ID: 7, Type: types.String, Nullable: true},
				{Name: "schedule_details", ID: 8, Type: types.Bytes, Nullable: true},
				{Name: "executor_type", ID: 9, Type: types.String, Nullable: false},
				{Name: "execution_args", ID: 10, Type: types.Bytes, Nullable: false},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "sched",
					ID:          0,
					ColumnNames: []string{"schedule_id", "next_run", "schedule_state"},
					ColumnIDs:   []descpb.ColumnID{1, 5, 6},
				},
				{
					Name: "other",
					ID:   1,
					ColumnNames: []string{
						"schedule_name", "created", "owner", "schedule_expr",
						"schedule_details", "executor_type", "execution_args",
					},
					ColumnIDs: []descpb.ColumnID{2, 3, 4, 7, 8, 9, 10},
				},
			},
			pk("schedule_id"),
			descpb.IndexDescriptor{
				Name:                "next_run_idx",
				ID:                  2,
				Unique:              false,
				KeyColumnNames:      []string{"next_run"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{5},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		))

	// SqllivenessTable is the descriptor for the sqlliveness table.
	//
	// TODO(jeffswenson): remove the function wrapper around the
	// SqllivenessTable descriptor. See TestSupportMultiRegion for context.
	SqllivenessTable = func() SystemTable {
		return makeSystemTable(
			SqllivenessTableSchema,
			systemTable(
				catconstants.SqllivenessTableName,
				keys.SqllivenessID,
				[]descpb.ColumnDescriptor{
					{Name: "session_id", ID: 1, Type: types.Bytes, Nullable: false},
					{Name: "expiration", ID: 2, Type: types.Decimal, Nullable: false},
					{Name: "crdb_region", ID: 3, Type: types.Bytes, Nullable: false},
				},
				[]descpb.ColumnFamilyDescriptor{
					{
						Name:            "primary",
						ID:              0,
						ColumnNames:     []string{"crdb_region", "session_id", "expiration"},
						ColumnIDs:       []descpb.ColumnID{3, 1, 2},
						DefaultColumnID: 2,
					},
				},
				descpb.IndexDescriptor{
					Name:                "primary",
					ID:                  2,
					Unique:              true,
					KeyColumnNames:      []string{"crdb_region", "session_id"},
					KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
					KeyColumnIDs:        []descpb.ColumnID{3, 1},
				},
			))
	}

	// MigrationsTable is the descriptor for the migrations table. It stores facts
	// about the completion state of long-running migrations. It is used to
	// prevent migrations from running again after they have been completed.
	MigrationsTable = makeSystemTable(
		MigrationsTableSchema,
		systemTable(
			catconstants.MigrationsTableName,
			keys.MigrationsID,
			[]descpb.ColumnDescriptor{
				{Name: "major", ID: 1, Type: types.Int, Nullable: false},
				{Name: "minor", ID: 2, Type: types.Int, Nullable: false},
				{Name: "patch", ID: 3, Type: types.Int, Nullable: false},
				{Name: "internal", ID: 4, Type: types.Int, Nullable: false},
				{Name: "completed_at", ID: 5, Type: types.TimestampTZ, Nullable: false},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:            "primary",
					ID:              0,
					ColumnNames:     []string{"major", "minor", "patch", "internal", "completed_at"},
					ColumnIDs:       []descpb.ColumnID{1, 2, 3, 4, 5},
					DefaultColumnID: 5,
				},
			},
			descpb.IndexDescriptor{
				Name:           tabledesc.LegacyPrimaryKeyIndexName,
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"major", "minor", "patch", "internal"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1, 2, 3, 4},
			},
		))

	// JoinTokensTable is the descriptor for the join tokens table.
	JoinTokensTable = makeSystemTable(
		JoinTokensTableSchema,
		systemTable(
			catconstants.JoinTokensTableName,
			keys.JoinTokensTableID,
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Uuid, Nullable: false},
				{Name: "secret", ID: 2, Type: types.Bytes, Nullable: false},
				{Name: "expiration", ID: 3, Type: types.TimestampTZ, Nullable: false},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:            "primary",
					ID:              0,
					ColumnNames:     []string{"id", "secret", "expiration"},
					ColumnIDs:       []descpb.ColumnID{1, 2, 3},
					DefaultColumnID: 0,
				},
			},
			descpb.IndexDescriptor{
				Name:           tabledesc.LegacyPrimaryKeyIndexName,
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1},
			},
		))

	defaultIndexRec = "ARRAY[]:::STRING[]"

	// StatementStatisticsTable is the descriptor for the SQL statement stats table.
	// It stores statistics for statement fingerprints.
	StatementStatisticsTable = makeSystemTable(
		StatementStatisticsTableSchema,
		systemTable(
			catconstants.StatementStatisticsTableName,
			keys.StatementStatisticsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "aggregated_ts", ID: 1, Type: types.TimestampTZ, Nullable: false},
				{Name: "fingerprint_id", ID: 2, Type: types.Bytes, Nullable: false},
				{Name: "transaction_fingerprint_id", ID: 3, Type: types.Bytes, Nullable: false},
				{Name: "plan_hash", ID: 4, Type: types.Bytes, Nullable: false},
				{Name: "app_name", ID: 5, Type: types.String, Nullable: false},
				{Name: "node_id", ID: 6, Type: types.Int, Nullable: false},
				{Name: "agg_interval", ID: 7, Type: types.Interval, Nullable: false},
				{Name: "metadata", ID: 8, Type: types.Jsonb, Nullable: false},
				{Name: "statistics", ID: 9, Type: types.Jsonb, Nullable: false},
				{Name: "plan", ID: 10, Type: types.Jsonb, Nullable: false},
				{
					Name:        "crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_plan_hash_transaction_fingerprint_id_shard_8",
					ID:          11,
					Type:        types.Int4,
					Nullable:    false,
					ComputeExpr: &sqlStmtHashComputeExpr,
					Hidden:      true,
				},
				{Name: "index_recommendations", ID: 12, Type: types.StringArray, Nullable: false, DefaultExpr: &defaultIndexRec},
				{Name: "indexes_usage", ID: 13, Type: types.Jsonb, Nullable: true, Virtual: true, ComputeExpr: &indexUsageComputeExprStr},
				{Name: "execution_count", ID: 14, Type: types.Int, Nullable: true, ComputeExpr: &executionCountComputeExprStr},
				{Name: "service_latency", ID: 15, Type: types.Float, Nullable: true, ComputeExpr: &serviceLatencyComputeExprStr},
				{Name: "cpu_sql_nanos", ID: 16, Type: types.Float, Nullable: true, ComputeExpr: &cpuSqlNanosComputeExprStr},
				{Name: "contention_time", ID: 17, Type: types.Float, Nullable: true, ComputeExpr: &contentionTimeComputeExprStr},
				{Name: "total_estimated_execution_time", ID: 18, Type: types.Float, Nullable: true, ComputeExpr: &totalEstimatedExecutionTimeExprStr},
				{Name: "p99_latency", ID: 19, Type: types.Float, Nullable: true, ComputeExpr: &p99LatencyComputeExprStr},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ID:   0,
					ColumnNames: []string{
						"crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_plan_hash_transaction_fingerprint_id_shard_8",
						"aggregated_ts", "fingerprint_id", "transaction_fingerprint_id", "plan_hash", "app_name", "node_id",
						"agg_interval", "metadata", "statistics", "plan", "index_recommendations", "execution_count",
						"service_latency", "cpu_sql_nanos", "contention_time", "total_estimated_execution_time",
						"p99_latency",
					},
					ColumnIDs:       []descpb.ColumnID{11, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 12, 14, 15, 16, 17, 18, 19},
					DefaultColumnID: 0,
				},
			},
			descpb.IndexDescriptor{
				Name:   tabledesc.LegacyPrimaryKeyIndexName,
				ID:     1,
				Unique: true,
				KeyColumnNames: []string{
					"crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_plan_hash_transaction_fingerprint_id_shard_8",
					"aggregated_ts",
					"fingerprint_id",
					"transaction_fingerprint_id",
					"plan_hash",
					"app_name",
					"node_id",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{11, 1, 2, 3, 4, 5, 6},
				Version:      descpb.StrictIndexColumnIDGuaranteesVersion,
				Sharded: catpb.ShardedDescriptor{
					IsSharded:    true,
					Name:         "crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_plan_hash_transaction_fingerprint_id_shard_8",
					ShardBuckets: 8,
					ColumnNames: []string{
						"aggregated_ts",
						"app_name",
						"fingerprint_id",
						"node_id",
						"plan_hash",
						"transaction_fingerprint_id",
					},
				},
			},
			descpb.IndexDescriptor{
				Name:   "fingerprint_stats_idx",
				ID:     2,
				Unique: false,
				KeyColumnNames: []string{
					"fingerprint_id",
					"transaction_fingerprint_id",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs:       []descpb.ColumnID{2, 3},
				KeySuffixColumnIDs: []descpb.ColumnID{11, 1, 4, 5, 6},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "indexes_usage_idx",
				ID:     3,
				Unique: false,
				KeyColumnNames: []string{
					"indexes_usage",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs:        []descpb.ColumnID{13},
				KeySuffixColumnIDs:  []descpb.ColumnID{11, 1, 2, 3, 4, 5, 6},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
				Type:                descpb.IndexDescriptor_INVERTED,
				InvertedColumnKinds: []catpb.InvertedIndexColumnKind{catpb.InvertedIndexColumnKind_DEFAULT},
			},
			descpb.IndexDescriptor{
				Name:   "execution_count_idx",
				ID:     4,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"app_name",
					"execution_count",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 5, 14},
				KeySuffixColumnIDs: []descpb.ColumnID{11, 2, 3, 4, 6},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Predicate:          "app_name NOT LIKE '$ internal%':::STRING",
			},
			descpb.IndexDescriptor{
				Name:   "service_latency_idx",
				ID:     5,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"app_name",
					"service_latency",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 5, 15},
				KeySuffixColumnIDs: []descpb.ColumnID{11, 2, 3, 4, 6},
				CompositeColumnIDs: []descpb.ColumnID{15},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Predicate:          "app_name NOT LIKE '$ internal%':::STRING",
			},
			descpb.IndexDescriptor{
				Name:   "cpu_sql_nanos_idx",
				ID:     6,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"app_name",
					"cpu_sql_nanos",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 5, 16},
				KeySuffixColumnIDs: []descpb.ColumnID{11, 2, 3, 4, 6},
				CompositeColumnIDs: []descpb.ColumnID{16},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Predicate:          "app_name NOT LIKE '$ internal%':::STRING",
			},
			descpb.IndexDescriptor{
				Name:   "contention_time_idx",
				ID:     7,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"app_name",
					"contention_time",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 5, 17},
				KeySuffixColumnIDs: []descpb.ColumnID{11, 2, 3, 4, 6},
				CompositeColumnIDs: []descpb.ColumnID{17},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Predicate:          "app_name NOT LIKE '$ internal%':::STRING",
			},
			descpb.IndexDescriptor{
				Name:   "total_estimated_execution_time_idx",
				ID:     8,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"app_name",
					"total_estimated_execution_time",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 5, 18},
				KeySuffixColumnIDs: []descpb.ColumnID{11, 2, 3, 4, 6},
				CompositeColumnIDs: []descpb.ColumnID{18},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Predicate:          "app_name NOT LIKE '$ internal%':::STRING",
			},
			descpb.IndexDescriptor{
				Name:   "p99_latency_idx",
				ID:     9,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"app_name",
					"p99_latency",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 5, 19},
				KeySuffixColumnIDs: []descpb.ColumnID{11, 2, 3, 4, 6},
				CompositeColumnIDs: []descpb.ColumnID{19},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Predicate:          "app_name NOT LIKE '$ internal%':::STRING",
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.Checks = []*descpb.TableDescriptor_CheckConstraint{{
				Expr:                  "crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_plan_hash_transaction_fingerprint_id_shard_8 IN (0:::INT8, 1:::INT8, 2:::INT8, 3:::INT8, 4:::INT8, 5:::INT8, 6:::INT8, 7:::INT8)",
				Name:                  "check_crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_plan_hash_transaction_fingerprint_id_shard_8",
				Validity:              descpb.ConstraintValidity_Validated,
				ColumnIDs:             []descpb.ColumnID{11},
				IsNonNullConstraint:   false,
				FromHashShardedColumn: true,
				ConstraintID:          tbl.NextConstraintID,
			}}
			tbl.NextConstraintID++
		},
	)

	// TransactionStatisticsTable is the descriptor for the SQL transaction stats
	// table. It stores statistics for transaction fingerprints.
	TransactionStatisticsTable = makeSystemTable(
		TransactionStatisticsTableSchema,
		systemTable(
			catconstants.TransactionStatisticsTableName,
			keys.TransactionStatisticsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "aggregated_ts", ID: 1, Type: types.TimestampTZ, Nullable: false},
				{Name: "fingerprint_id", ID: 2, Type: types.Bytes, Nullable: false},
				{Name: "app_name", ID: 3, Type: types.String, Nullable: false},
				{Name: "node_id", ID: 4, Type: types.Int, Nullable: false},
				{Name: "agg_interval", ID: 5, Type: types.Interval, Nullable: false},
				{Name: "metadata", ID: 6, Type: types.Jsonb, Nullable: false},
				{Name: "statistics", ID: 7, Type: types.Jsonb, Nullable: false},
				{
					Name:        "crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_shard_8",
					ID:          8,
					Type:        types.Int4,
					Nullable:    false,
					ComputeExpr: &sqlTxnHashComputeExpr,
					Hidden:      true,
				},
				{Name: "execution_count", ID: 9, Type: types.Int, Nullable: true, ComputeExpr: &executionCountComputeExprStr},
				{Name: "service_latency", ID: 10, Type: types.Float, Nullable: true, ComputeExpr: &serviceLatencyComputeExprStr},
				{Name: "cpu_sql_nanos", ID: 11, Type: types.Float, Nullable: true, ComputeExpr: &cpuSqlNanosComputeExprStr},
				{Name: "contention_time", ID: 12, Type: types.Float, Nullable: true, ComputeExpr: &contentionTimeComputeExprStr},
				{Name: "total_estimated_execution_time", ID: 13, Type: types.Float, Nullable: true, ComputeExpr: &totalEstimatedExecutionTimeExprStr},
				{Name: "p99_latency", ID: 14, Type: types.Float, Nullable: true, ComputeExpr: &p99LatencyComputeExprStr},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ID:   0,
					ColumnNames: []string{
						"crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_shard_8",
						"aggregated_ts", "fingerprint_id", "app_name", "node_id",
						"agg_interval", "metadata", "statistics", "execution_count", "service_latency", "cpu_sql_nanos",
						"contention_time", "total_estimated_execution_time", "p99_latency",
					},
					ColumnIDs:       []descpb.ColumnID{8, 1, 2, 3, 4, 5, 6, 7, 9, 10, 11, 12, 13, 14},
					DefaultColumnID: 0,
				},
			},
			descpb.IndexDescriptor{
				Name:   tabledesc.LegacyPrimaryKeyIndexName,
				ID:     1,
				Unique: true,
				KeyColumnNames: []string{
					"crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_shard_8",
					"aggregated_ts",
					"fingerprint_id",
					"app_name",
					"node_id",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{8, 1, 2, 3, 4},
				Version:      descpb.StrictIndexColumnIDGuaranteesVersion,
				Sharded: catpb.ShardedDescriptor{
					IsSharded:    true,
					Name:         "crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_shard_8",
					ShardBuckets: 8,
					ColumnNames: []string{
						"aggregated_ts",
						"app_name",
						"fingerprint_id",
						"node_id",
					},
				},
			},
			descpb.IndexDescriptor{
				Name:   "fingerprint_stats_idx",
				ID:     2,
				Unique: false,
				KeyColumnNames: []string{
					"fingerprint_id",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs:       []descpb.ColumnID{2},
				KeySuffixColumnIDs: []descpb.ColumnID{8, 1, 3, 4},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "execution_count_idx",
				ID:     3,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"app_name",
					"execution_count",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 3, 9},
				KeySuffixColumnIDs: []descpb.ColumnID{8, 2, 4},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Predicate:          "app_name NOT LIKE '$ internal%':::STRING",
			},
			descpb.IndexDescriptor{
				Name:   "service_latency_idx",
				ID:     4,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"app_name",
					"service_latency",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 3, 10},
				KeySuffixColumnIDs: []descpb.ColumnID{8, 2, 4},
				CompositeColumnIDs: []descpb.ColumnID{10},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Predicate:          "app_name NOT LIKE '$ internal%':::STRING",
			},
			descpb.IndexDescriptor{
				Name:   "cpu_sql_nanos_idx",
				ID:     5,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"app_name",
					"cpu_sql_nanos",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 3, 11},
				KeySuffixColumnIDs: []descpb.ColumnID{8, 2, 4},
				CompositeColumnIDs: []descpb.ColumnID{11},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Predicate:          "app_name NOT LIKE '$ internal%':::STRING",
			},
			descpb.IndexDescriptor{
				Name:   "contention_time_idx",
				ID:     6,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"app_name",
					"contention_time",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 3, 12},
				KeySuffixColumnIDs: []descpb.ColumnID{8, 2, 4},
				CompositeColumnIDs: []descpb.ColumnID{12},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Predicate:          "app_name NOT LIKE '$ internal%':::STRING",
			},
			descpb.IndexDescriptor{
				Name:   "total_estimated_execution_time_idx",
				ID:     7,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"app_name",
					"total_estimated_execution_time",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 3, 13},
				KeySuffixColumnIDs: []descpb.ColumnID{8, 2, 4},
				CompositeColumnIDs: []descpb.ColumnID{13},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Predicate:          "app_name NOT LIKE '$ internal%':::STRING",
			},
			descpb.IndexDescriptor{
				Name:   "p99_latency_idx",
				ID:     8,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"app_name",
					"p99_latency",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 3, 14},
				KeySuffixColumnIDs: []descpb.ColumnID{8, 2, 4},
				CompositeColumnIDs: []descpb.ColumnID{14},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Predicate:          "app_name NOT LIKE '$ internal%':::STRING",
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.Checks = []*descpb.TableDescriptor_CheckConstraint{{
				Expr:                  "crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_shard_8 IN (0:::INT8, 1:::INT8, 2:::INT8, 3:::INT8, 4:::INT8, 5:::INT8, 6:::INT8, 7:::INT8)",
				Name:                  "check_crdb_internal_aggregated_ts_app_name_fingerprint_id_node_id_shard_8",
				Validity:              descpb.ConstraintValidity_Validated,
				ColumnIDs:             []descpb.ColumnID{8},
				IsNonNullConstraint:   false,
				FromHashShardedColumn: true,
				ConstraintID:          tbl.NextConstraintID,
			}}
			tbl.NextConstraintID++
		},
	)

	// StatementActivityTable is the descriptor for the SQL statement stats table.
	// It stores statistics for statement fingerprints.
	StatementActivityTable = makeSystemTable(
		StatementActivityTableSchema,
		systemTable(
			catconstants.StatementActivityTableName,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "aggregated_ts", ID: 1, Type: types.TimestampTZ, Nullable: false},
				{Name: "fingerprint_id", ID: 2, Type: types.Bytes, Nullable: false},
				{Name: "transaction_fingerprint_id", ID: 3, Type: types.Bytes, Nullable: false},
				{Name: "plan_hash", ID: 4, Type: types.Bytes, Nullable: false},
				{Name: "app_name", ID: 5, Type: types.String, Nullable: false},
				{Name: "agg_interval", ID: 6, Type: types.Interval, Nullable: false},
				{Name: "metadata", ID: 7, Type: types.Jsonb, Nullable: false},
				{Name: "statistics", ID: 8, Type: types.Jsonb, Nullable: false},
				{Name: "plan", ID: 9, Type: types.Jsonb, Nullable: false},
				{Name: "index_recommendations", ID: 10, Type: types.StringArray, Nullable: false, DefaultExpr: &defaultIndexRec},
				{Name: "execution_count", ID: 11, Type: types.Int, Nullable: false},
				{Name: "execution_total_seconds", ID: 12, Type: types.Float, Nullable: false},
				{Name: "execution_total_cluster_seconds", ID: 13, Type: types.Float, Nullable: false},
				{Name: "contention_time_avg_seconds", ID: 14, Type: types.Float, Nullable: false},
				{Name: "cpu_sql_avg_nanos", ID: 15, Type: types.Float, Nullable: false},
				{Name: "service_latency_avg_seconds", ID: 16, Type: types.Float, Nullable: false},
				{Name: "service_latency_p99_seconds", ID: 17, Type: types.Float, Nullable: false},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ID:   0,
					ColumnNames: []string{
						"aggregated_ts", "fingerprint_id", "transaction_fingerprint_id",
						"plan_hash", "app_name", "agg_interval", "metadata",
						"statistics", "plan", "index_recommendations", "execution_count",
						"execution_total_seconds", "execution_total_cluster_seconds",
						"contention_time_avg_seconds", "cpu_sql_avg_nanos",
						"service_latency_avg_seconds", "service_latency_p99_seconds",
					},
					ColumnIDs:       []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17},
					DefaultColumnID: 0,
				},
			},
			descpb.IndexDescriptor{
				Name:   tabledesc.LegacyPrimaryKeyIndexName,
				ID:     1,
				Unique: true,
				KeyColumnNames: []string{
					"aggregated_ts",
					"fingerprint_id",
					"transaction_fingerprint_id",
					"plan_hash",
					"app_name",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1, 2, 3, 4, 5},
				Version:      descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "fingerprint_id_idx",
				ID:     2,
				Unique: false,
				KeyColumnNames: []string{
					"fingerprint_id",
					"transaction_fingerprint_id",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs:       []descpb.ColumnID{2, 3},
				KeySuffixColumnIDs: []descpb.ColumnID{1, 4, 5},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "execution_count_idx",
				ID:     3,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"execution_count",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 11},
				KeySuffixColumnIDs: []descpb.ColumnID{2, 3, 4, 5},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "execution_total_seconds_idx",
				ID:     4,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"execution_total_seconds",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 12},
				CompositeColumnIDs: []descpb.ColumnID{12},
				KeySuffixColumnIDs: []descpb.ColumnID{2, 3, 4, 5},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "contention_time_avg_seconds_idx",
				ID:     5,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"contention_time_avg_seconds",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 14},
				CompositeColumnIDs: []descpb.ColumnID{14},
				KeySuffixColumnIDs: []descpb.ColumnID{2, 3, 4, 5},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "cpu_sql_avg_nanos_idx",
				ID:     6,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"cpu_sql_avg_nanos",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 15},
				CompositeColumnIDs: []descpb.ColumnID{15},
				KeySuffixColumnIDs: []descpb.ColumnID{2, 3, 4, 5},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "service_latency_avg_seconds_idx",
				ID:     7,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"service_latency_avg_seconds",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 16},
				CompositeColumnIDs: []descpb.ColumnID{16},
				KeySuffixColumnIDs: []descpb.ColumnID{2, 3, 4, 5},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "service_latency_p99_seconds_idx",
				ID:     8,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"service_latency_p99_seconds",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 17},
				CompositeColumnIDs: []descpb.ColumnID{17},
				KeySuffixColumnIDs: []descpb.ColumnID{2, 3, 4, 5},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		),
	)

	// TransactionActivityTable is the descriptor for the SQL transaction stats
	// table. It stores statistics for transaction fingerprints.
	TransactionActivityTable = makeSystemTable(
		TransactionActivityTableSchema,
		systemTable(
			catconstants.TransactionActivityTableName,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "aggregated_ts", ID: 1, Type: types.TimestampTZ, Nullable: false},
				{Name: "fingerprint_id", ID: 2, Type: types.Bytes, Nullable: false},
				{Name: "app_name", ID: 3, Type: types.String, Nullable: false},
				{Name: "agg_interval", ID: 4, Type: types.Interval, Nullable: false},
				{Name: "metadata", ID: 5, Type: types.Jsonb, Nullable: false},
				{Name: "statistics", ID: 6, Type: types.Jsonb, Nullable: false},
				{Name: "query", ID: 7, Type: types.String, Nullable: false},
				{Name: "execution_count", ID: 8, Type: types.Int, Nullable: false},
				{Name: "execution_total_seconds", ID: 9, Type: types.Float, Nullable: false},
				{Name: "execution_total_cluster_seconds", ID: 10, Type: types.Float, Nullable: false},
				{Name: "contention_time_avg_seconds", ID: 11, Type: types.Float, Nullable: false},
				{Name: "cpu_sql_avg_nanos", ID: 12, Type: types.Float, Nullable: false},
				{Name: "service_latency_avg_seconds", ID: 13, Type: types.Float, Nullable: false},
				{Name: "service_latency_p99_seconds", ID: 14, Type: types.Float, Nullable: false},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ID:   0,
					ColumnNames: []string{
						"aggregated_ts", "fingerprint_id", "app_name",
						"agg_interval", "metadata", "statistics", "query", "execution_count",
						"execution_total_seconds", "execution_total_cluster_seconds",
						"contention_time_avg_seconds", "cpu_sql_avg_nanos",
						"service_latency_avg_seconds", "service_latency_p99_seconds",
					},
					ColumnIDs:       []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14},
					DefaultColumnID: 0,
				},
			},
			descpb.IndexDescriptor{
				Name:   tabledesc.LegacyPrimaryKeyIndexName,
				ID:     1,
				Unique: true,
				KeyColumnNames: []string{
					"aggregated_ts",
					"fingerprint_id",
					"app_name",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1, 2, 3},
				Version:      descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "fingerprint_id_idx",
				ID:     2,
				Unique: false,
				KeyColumnNames: []string{
					"fingerprint_id",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs:       []descpb.ColumnID{2},
				KeySuffixColumnIDs: []descpb.ColumnID{1, 3},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "execution_count_idx",
				ID:     3,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"execution_count",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 8},
				KeySuffixColumnIDs: []descpb.ColumnID{2, 3},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "execution_total_seconds_idx",
				ID:     4,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"execution_total_seconds",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 9},
				CompositeColumnIDs: []descpb.ColumnID{9},
				KeySuffixColumnIDs: []descpb.ColumnID{2, 3},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "contention_time_avg_seconds_idx",
				ID:     5,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"contention_time_avg_seconds",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 11},
				CompositeColumnIDs: []descpb.ColumnID{11},
				KeySuffixColumnIDs: []descpb.ColumnID{2, 3},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "cpu_sql_avg_nanos_idx",
				ID:     6,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"cpu_sql_avg_nanos",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 12},
				CompositeColumnIDs: []descpb.ColumnID{12},
				KeySuffixColumnIDs: []descpb.ColumnID{2, 3},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "service_latency_avg_seconds_idx",
				ID:     7,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"service_latency_avg_seconds",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 13},
				CompositeColumnIDs: []descpb.ColumnID{13},
				KeySuffixColumnIDs: []descpb.ColumnID{2, 3},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "service_latency_p99_seconds_idx",
				ID:     8,
				Unique: false,
				KeyColumnNames: []string{
					"aggregated_ts",
					"service_latency_p99_seconds",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{1, 14},
				CompositeColumnIDs: []descpb.ColumnID{14},
				KeySuffixColumnIDs: []descpb.ColumnID{2, 3},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		),
	)

	// DatabaseRoleSettingsTable holds default values for session variables
	// for each role and database combination. It is analogous to the
	// pg_db_role_setting table in Postgres. Note that roles do not currently
	// have stable OIDs associated with them, so this table continues the
	// convention of keying based on the role name (which is how privileges
	// work also).
	DatabaseRoleSettingsTable = makeSystemTable(
		DatabaseRoleSettingsTableSchema,
		systemTable(
			catconstants.DatabaseRoleSettingsTableName,
			keys.DatabaseRoleSettingsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "database_id", ID: 1, Type: types.Oid, Nullable: false},
				{Name: "role_name", ID: 2, Type: types.String, Nullable: false},
				{Name: "settings", ID: 3, Type: types.StringArray, Nullable: false},
				{Name: "role_id", ID: 4, Type: types.Oid, Nullable: false},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ID:          0,
					ColumnNames: []string{"database_id", "role_name", "settings", "role_id"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4},
				},
			},
			descpb.IndexDescriptor{
				Name:           tabledesc.LegacyPrimaryKeyIndexName,
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"database_id", "role_name"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1, 2},
			},
			descpb.IndexDescriptor{
				Name:                "database_role_settings_database_id_role_id_key",
				ID:                  2,
				Unique:              true,
				KeyColumnNames:      []string{"database_id", "role_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1, 4},
				KeySuffixColumnIDs:  []descpb.ColumnID{2},
				StoreColumnNames:    []string{"settings"},
				StoreColumnIDs:      []descpb.ColumnID{3},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		))

	// The tenant_usage table receives periodic updates from all SQL pods. We want
	// to limit the amount of clutter in the table.
	TenantUsageTableTTL = 2 * time.Hour
	// TenantUsageTable is the descriptor for the tenant_usage table. It is used
	// to coordinate throttling of tenant SQL pods and to track consumption.
	TenantUsageTable = makeSystemTable(
		TenantUsageTableSchema,
		systemTable(
			catconstants.TenantUsageTableName,
			keys.TenantUsageTableID,
			[]descpb.ColumnDescriptor{
				{Name: "tenant_id", ID: 1, Type: types.Int, Nullable: false},
				{Name: "instance_id", ID: 2, Type: types.Int, Nullable: false},
				{Name: "next_instance_id", ID: 3, Type: types.Int, Nullable: false},
				{Name: "last_update", ID: 4, Type: types.Timestamp, Nullable: false},
				{Name: "ru_burst_limit", ID: 5, Type: types.Float, Nullable: true},
				{Name: "ru_refill_rate", ID: 6, Type: types.Float, Nullable: true},
				{Name: "ru_current", ID: 7, Type: types.Float, Nullable: true},
				{Name: "current_share_sum", ID: 8, Type: types.Float, Nullable: true},
				{Name: "total_consumption", ID: 9, Type: types.Bytes, Nullable: true},
				{Name: "instance_lease", ID: 10, Type: types.Bytes, Nullable: true},
				{Name: "instance_seq", ID: 11, Type: types.Int, Nullable: true},
				{Name: "instance_shares", ID: 12, Type: types.Float, Nullable: true},
				{Name: "current_rates", ID: 13, Type: types.Bytes, Nullable: true},
				{Name: "next_rates", ID: 14, Type: types.Bytes, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ID:   0,
					ColumnNames: []string{
						"tenant_id", "instance_id", "next_instance_id", "last_update",
						"ru_burst_limit", "ru_refill_rate", "ru_current", "current_share_sum",
						"total_consumption",
						"instance_lease", "instance_seq", "instance_shares",
						"current_rates", "next_rates",
					},
					ColumnIDs:       []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14},
					DefaultColumnID: 0,
				},
			},
			descpb.IndexDescriptor{
				Name:           tabledesc.LegacyPrimaryKeyIndexName,
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"tenant_id", "instance_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1, 2},
				Version:      descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.ExcludeDataFromBackup = true
		},
	)

	// SQLInstancesTable is the descriptor for the sqlinstances table. It
	// stores information about all the SQL instances for a tenant and their
	// associated session, locality, and address information.
	//
	// TODO(jeffswenson): remove the function wrapper around the
	// SQLInstanceTable descriptor. See TestSupportMultiRegion for context.
	SQLInstancesTable = func() SystemTable {
		return makeSystemTable(
			SQLInstancesTableSchema,
			systemTable(
				catconstants.SQLInstancesTableName,
				keys.SQLInstancesTableID,
				[]descpb.ColumnDescriptor{
					{Name: "id", ID: 1, Type: types.Int, Nullable: false},
					{Name: "addr", ID: 2, Type: types.String, Nullable: true},
					{Name: "session_id", ID: 3, Type: types.Bytes, Nullable: true},
					{Name: "locality", ID: 4, Type: types.Jsonb, Nullable: true},
					{Name: "sql_addr", ID: 5, Type: types.String, Nullable: true},
					{Name: "crdb_region", ID: 6, Type: types.Bytes, Nullable: false},
					{Name: "binary_version", ID: 7, Type: types.String, Nullable: true},
					{Name: "is_draining", ID: 8, Type: types.Bool, Nullable: true},
				},
				[]descpb.ColumnFamilyDescriptor{
					{
						Name:            "primary",
						ID:              0,
						ColumnNames:     []string{"id", "addr", "session_id", "locality", "sql_addr", "crdb_region", "binary_version", "is_draining"},
						ColumnIDs:       []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7, 8},
						DefaultColumnID: 0,
					},
				},
				descpb.IndexDescriptor{
					Name:                "primary",
					ID:                  2,
					Unique:              true,
					KeyColumnNames:      []string{"crdb_region", "id"},
					KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
					KeyColumnIDs:        []descpb.ColumnID{6, 1},
				},
			))
	}

	// SpanConfigurationsTable is the descriptor for the system tenant's span
	// configurations table.
	SpanConfigurationsTable = makeSystemTable(
		SpanConfigurationsTableSchema,
		systemTable(
			catconstants.SpanConfigurationsTableName,
			keys.SpanConfigurationsTableID,
			[]descpb.ColumnDescriptor{
				{Name: "start_key", ID: 1, Type: types.Bytes},
				{Name: "end_key", ID: 2, Type: types.Bytes},
				{Name: "config", ID: 3, Type: types.Bytes},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ID:          0,
					ColumnNames: []string{"start_key", "end_key", "config"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3},
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  keys.SpanConfigurationsTablePrimaryKeyIndexID,
				Unique:              true,
				KeyColumnNames:      []string{"start_key"},
				KeyColumnDirections: singleASC,
				KeyColumnIDs:        singleID1,
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.Checks = []*descpb.TableDescriptor_CheckConstraint{{
				Name:         "check_bounds",
				Expr:         "start_key < end_key",
				ColumnIDs:    []descpb.ColumnID{1, 2},
				ConstraintID: tbl.NextConstraintID,
			}}
			tbl.NextConstraintID++
		},
		func(tbl *descpb.TableDescriptor) {
			tbl.ExcludeDataFromBackup = true
		},
	)

	// TenantSettingsTable is the descriptor for the tenant settings table.
	// It contains overrides for cluster settings for tenants.
	TenantSettingsTable = makeSystemTable(
		TenantSettingsTableSchema,
		systemTable(
			catconstants.TenantSettingsTableName,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "tenant_id", ID: 1, Type: types.Int},
				{Name: "name", ID: 2, Type: types.String},
				{Name: "value", ID: 3, Type: types.String},
				{Name: "last_updated", ID: 4, Type: types.Timestamp, DefaultExpr: &nowString},
				{Name: "value_type", ID: 5, Type: types.String},
				{Name: "reason", ID: 6, Type: types.String, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "fam_0_tenant_id_name_value_last_updated_value_type_reason",
					ID:          0,
					ColumnNames: []string{"tenant_id", "name", "value", "last_updated", "value_type", "reason"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4, 5, 6},
				},
			},
			descpb.IndexDescriptor{
				Name:           tabledesc.LegacyPrimaryKeyIndexName,
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"tenant_id", "name"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1, 2},
				Version:      descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		))

	// SpanCountTable is the descriptor for the split count table.
	SpanCountTable = makeSystemTable(
		SpanCountTableSchema,
		systemTable(
			catconstants.SpanCountTableName,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "singleton", ID: 1, Type: types.Bool, DefaultExpr: &trueBoolString},
				{Name: "span_count", ID: 2, Type: types.Int},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:            "primary",
					ID:              0,
					DefaultColumnID: 2,
					ColumnNames:     []string{"singleton", "span_count"},
					ColumnIDs:       []descpb.ColumnID{1, 2},
				},
			},
			pk("singleton"),
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.Checks = []*descpb.TableDescriptor_CheckConstraint{{
				Name:         "single_row",
				Expr:         "singleton",
				ColumnIDs:    []descpb.ColumnID{1},
				ConstraintID: tbl.NextConstraintID,
			}}
			tbl.NextConstraintID++
		},
	)

	SystemPrivilegeTable = makeSystemTable(
		SystemPrivilegeTableSchema,
		systemTable(
			catconstants.SystemPrivilegeTableName,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "username", ID: 1, Type: types.String},
				{Name: "path", ID: 2, Type: types.String},
				{Name: "privileges", ID: 3, Type: types.StringArray},
				{Name: "grant_options", ID: 4, Type: types.StringArray},
				{Name: "user_id", ID: 5, Type: types.Oid},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ID:          0,
					ColumnNames: []string{"username", "path", "privileges", "grant_options", "user_id"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4, 5},
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"username", "path"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2},
			},
			descpb.IndexDescriptor{
				Name:                "privileges_path_user_id_key",
				ID:                  2,
				Unique:              true,
				KeyColumnNames:      []string{"path", "user_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{2, 5},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				StoreColumnNames:    []string{"privileges", "grant_options"},
				StoreColumnIDs:      []descpb.ColumnID{3, 4},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:                "privileges_path_username_key",
				ID:                  3,
				Unique:              true,
				KeyColumnNames:      []string{"path", "username"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{2, 1},
				StoreColumnNames:    []string{"privileges", "grant_options"},
				StoreColumnIDs:      []descpb.ColumnID{3, 4},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		),
	)

	SystemExternalConnectionsTable = makeSystemTable(
		SystemExternalConnectionsTableSchema,
		systemTable(
			catconstants.SystemExternalConnectionsTableName,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "connection_name", ID: 1, Type: types.String},
				{Name: "created", ID: 2, Type: types.Timestamp, DefaultExpr: &nowString},
				{Name: "updated", ID: 3, Type: types.Timestamp, DefaultExpr: &nowString},
				{Name: "connection_type", ID: 4, Type: types.String},
				{Name: "connection_details", ID: 5, Type: types.Bytes},
				{Name: "owner", ID: 6, Type: types.String},
				{Name: "owner_id", ID: 7, Type: types.Oid},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ID:          0,
					ColumnNames: []string{"connection_name", "created", "updated", "connection_type", "connection_details", "owner", "owner_id"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7},
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"connection_name"},
				KeyColumnDirections: singleASC,
				KeyColumnIDs:        singleID1,
			},
		),
	)

	SystemTenantTasksTable = makeSystemTable(
		SystemTenantTasksSchema,
		systemTable(
			catconstants.TenantTasksTableName,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "tenant_id", ID: 1, Type: types.Int},
				{Name: "issuer", ID: 2, Type: types.String},
				{Name: "task_id", ID: 3, Type: types.Int},
				{Name: "created", ID: 4, Type: types.TimestampTZ, DefaultExpr: &nowTZString},
				{Name: "payload_id", ID: 5, Type: types.String},
				{Name: "owner", ID: 6, Type: types.String},
				{Name: "owner_id", ID: 7, Type: types.Oid},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ID:          0,
					ColumnNames: []string{"tenant_id", "issuer", "task_id", "created", "payload_id", "owner", "owner_id"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7},
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"tenant_id", "issuer", "task_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2, 3},
			}),
	)

	SystemTaskPayloadsTable = makeSystemTable(
		SystemTaskPayloadsSchema,
		systemTable(
			catconstants.TaskPayloadsTableName,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.String},
				{Name: "created", ID: 2, Type: types.TimestampTZ, DefaultExpr: &nowTZString},
				{Name: "owner", ID: 3, Type: types.String},
				{Name: "owner_id", ID: 4, Type: types.Oid},
				{Name: "min_version", ID: 5, Type: types.String},
				{Name: "description", ID: 6, Type: types.String, Nullable: true},
				{Name: "type", ID: 7, Type: types.String},
				{Name: "value", ID: 8, Type: types.Bytes},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ID:          0,
					ColumnNames: []string{"id", "created", "owner", "owner_id", "min_version", "description", "type", "value"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7, 8},
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"id"},
				KeyColumnDirections: singleASC,
				KeyColumnIDs:        singleID1,
			}),
	)

	// SystemJobProgressTable is described in comment on JobProgressTableSchema.
	SystemJobProgressTable = makeSystemTable(
		JobProgressTableSchema,
		systemTable(
			catconstants.JobsProgressTableName,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "job_id", ID: 1, Type: types.Int},
				{Name: "written", ID: 2, Type: types.TimestampTZ, DefaultExpr: &nowTZString},
				{Name: "fraction", ID: 3, Type: types.Float, Nullable: true},
				{Name: "resolved", ID: 4, Type: types.Decimal, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ID:          0,
					ColumnNames: []string{"job_id", "written", "fraction", "resolved"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4},
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"job_id", "written"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_DESC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2},
			}),
	)

	// SystemJobProgressHistoryTable is described in comment on JobProgressHistoryTableSchema.
	SystemJobProgressHistoryTable = makeSystemTable(
		JobProgressHistoryTableSchema,
		systemTable(
			catconstants.JobsProgressHistoryTableName,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "job_id", ID: 1, Type: types.Int},
				{Name: "written", ID: 2, Type: types.TimestampTZ, DefaultExpr: &nowTZString},
				{Name: "fraction", ID: 3, Type: types.Float, Nullable: true},
				{Name: "resolved", ID: 4, Type: types.Decimal, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ID:          0,
					ColumnNames: []string{"job_id", "written", "fraction", "resolved"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4},
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"job_id", "written"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_DESC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2},
			}),
	)

	// SystemJobStatusTable is described in comment on JobStatusTableSchema.
	SystemJobStatusTable = makeSystemTable(
		JobStatusTableSchema,
		systemTable(
			catconstants.JobsStatusTableName,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "job_id", ID: 1, Type: types.Int},
				{Name: "written", ID: 2, Type: types.TimestampTZ, DefaultExpr: &nowTZString},
				{Name: "status", ID: 3, Type: types.String},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:            "primary",
					ID:              0,
					ColumnNames:     []string{"job_id", "written", "status"},
					ColumnIDs:       []descpb.ColumnID{1, 2, 3},
					DefaultColumnID: 3,
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"job_id", "written"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_DESC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2},
			}),
	)

	SystemJobMessageTable = makeSystemTable(
		JobMessageTableSchema,
		systemTable(
			catconstants.JobsMessageTableName,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "job_id", ID: 1, Type: types.Int},
				{Name: "written", ID: 2, Type: types.TimestampTZ, DefaultExpr: &nowTZString},
				{Name: "kind", ID: 3, Type: types.String},
				{Name: "message", ID: 4, Type: types.String},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:            "primary",
					ID:              0,
					ColumnNames:     []string{"job_id", "written", "kind", "message"},
					ColumnIDs:       []descpb.ColumnID{1, 2, 3, 4},
					DefaultColumnID: 4,
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"job_id", "written", "kind"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_DESC, catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2, 3},
			}),
	)

	SystemJobInfoTable = makeSystemTable(
		SystemJobInfoTableSchema,
		systemTable(
			catconstants.SystemJobInfoTableName,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "job_id", ID: 1, Type: types.Int},
				{Name: "info_key", ID: 2, Type: types.String},
				{Name: "written", ID: 3, Type: types.TimestampTZ, DefaultExpr: &nowTZString},
				{Name: "value", ID: 4, Type: types.Bytes, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:            "primary",
					ID:              0,
					ColumnNames:     []string{"job_id", "info_key", "written", "value"},
					ColumnIDs:       []descpb.ColumnID{1, 2, 3, 4},
					DefaultColumnID: 4,
				},
			},
			descpb.IndexDescriptor{
				Name:                "primary",
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"job_id", "info_key", "written"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_ASC, catenumpb.IndexColumn_DESC},
				KeyColumnIDs:        []descpb.ColumnID{1, 2, 3},
			}),
	)

	genRandomUUIDString = "gen_random_uuid()"

	SpanStatsUniqueKeysTable = makeSystemTable(
		SpanStatsUniqueKeysTableSchema,
		systemTable(
			catconstants.SpanStatsUniqueKeys,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Uuid, DefaultExpr: &genRandomUUIDString},
				{Name: "key_bytes", ID: 2, Type: types.Bytes, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:            "primary",
					ID:              0,
					ColumnNames:     []string{"id", "key_bytes"},
					ColumnIDs:       []descpb.ColumnID{1, 2},
					DefaultColumnID: 2,
				},
			},
			descpb.IndexDescriptor{
				Name:           "primary",
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1},
			},
			descpb.IndexDescriptor{
				Name:                "unique_keys_key_bytes_idx",
				ID:                  2,
				Unique:              true,
				KeyColumnNames:      []string{"key_bytes"},
				KeyColumnDirections: singleASC,
				KeyColumnIDs:        []descpb.ColumnID{2},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		),
	)

	SpanStatsBucketsTable = makeSystemTable(
		SpanStatsBucketsTableSchema,
		systemTable(
			catconstants.SpanStatsBuckets,
			descpb.InvalidID, // dynamically assigned
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Uuid, DefaultExpr: &genRandomUUIDString},
				{Name: "sample_id", ID: 2, Type: types.Uuid},
				{Name: "start_key_id", ID: 3, Type: types.Uuid},
				{Name: "end_key_id", ID: 4, Type: types.Uuid},
				{Name: "requests", ID: 5, Type: types.Int},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ID:   0,
					ColumnNames: []string{"id", "sample_id", "start_key_id",
						"end_key_id", "requests"},
					ColumnIDs: []descpb.ColumnID{1, 2, 3, 4, 5},
				},
			},
			descpb.IndexDescriptor{
				Name:           "primary",
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1},
			},
			descpb.IndexDescriptor{
				Name:                "buckets_sample_id_idx",
				ID:                  2,
				Unique:              false,
				KeyColumnNames:      []string{"sample_id"},
				KeyColumnDirections: singleASC,
				KeyColumnIDs:        []descpb.ColumnID{2},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		),
	)

	SpanStatsSamplesTable = makeSystemTable(
		SpanStatsSamplesTableSchema,
		systemTable(
			catconstants.SpanStatsSamples, descpb.InvalidID,
			[]descpb.ColumnDescriptor{
				{Name: "id", ID: 1, Type: types.Uuid, DefaultExpr: &genRandomUUIDString},
				{Name: "sample_time", ID: 2, Type: types.Timestamp,
					DefaultExpr: &nowString},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:            "primary",
					ID:              0,
					ColumnNames:     []string{"id", "sample_time"},
					ColumnIDs:       []descpb.ColumnID{1, 2},
					DefaultColumnID: 2,
				},
			},
			descpb.IndexDescriptor{
				Name:           "primary",
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1},
			},
			descpb.IndexDescriptor{
				Name:                "samples_sample_time_idx",
				ID:                  2,
				Unique:              true,
				KeyColumnNames:      []string{"sample_time"},
				KeyColumnDirections: singleASC,
				KeyColumnIDs:        []descpb.ColumnID{2},
				KeySuffixColumnIDs:  []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
		),
	)

	SpanStatsTenantBoundariesTable = makeSystemTable(
		SpanStatsTenantBoundariesTableSchema,
		systemTable(
			catconstants.SpanStatsTenantBoundaries,
			descpb.InvalidID, // dynamically assigned table ID
			[]descpb.ColumnDescriptor{
				{Name: "tenant_id", ID: 1, Type: types.Int},
				{Name: "boundaries", ID: 2, Type: types.Bytes},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:            "primary",
					ID:              0,
					ColumnNames:     []string{"tenant_id", "boundaries"},
					ColumnIDs:       []descpb.ColumnID{1, 2},
					DefaultColumnID: 2,
				},
			},
			descpb.IndexDescriptor{
				Name:           "primary",
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"tenant_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1},
			},
		),
	)

	RegionLivenessTable = makeSystemTable(
		RegionLivenessTableSchema,
		systemTable(
			catconstants.RegionalLiveness,
			keys.RegionLivenessTableID,
			[]descpb.ColumnDescriptor{
				{Name: "crdb_region", ID: 1, Type: types.Bytes},
				{Name: "unavailable_at", ID: 2, Type: types.Timestamp, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:            "primary",
					ID:              0,
					ColumnNames:     []string{"crdb_region", "unavailable_at"},
					ColumnIDs:       []descpb.ColumnID{1, 2},
					DefaultColumnID: 2,
				},
			},
			descpb.IndexDescriptor{
				Name:           "region_liveness_pkey",
				ID:             1,
				Unique:         true,
				KeyColumnNames: []string{"crdb_region"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1},
			},
		),
	)

	SystemMVCCStatisticsTable = makeSystemTable(
		SystemMVCCStatisticsSchema,
		systemTable(
			catconstants.MVCCStatistics,
			descpb.InvalidID, // dynamically assigned table ID
			[]descpb.ColumnDescriptor{
				{Name: "created_at", ID: 1, Type: types.TimestampTZ, DefaultExpr: &nowTZString},
				{Name: "database_id", ID: 2, Type: types.Int},
				{Name: "table_id", ID: 3, Type: types.Int},
				{Name: "index_id", ID: 4, Type: types.Int},
				{Name: "statistics", ID: 5, Type: types.Jsonb},
				{
					Name:        "crdb_internal_created_at_database_id_index_id_table_id_shard_16",
					ID:          6,
					Type:        types.Int4,
					Nullable:    false,
					ComputeExpr: &timestampHashComputeExpr,
					Hidden:      true,
					Virtual:     true,
				},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ID:   0,
					ColumnNames: []string{
						"created_at",
						"database_id",
						"table_id",
						"index_id",
						"statistics",
					},
					ColumnIDs:       []descpb.ColumnID{1, 2, 3, 4, 5},
					DefaultColumnID: 5,
				},
			},
			descpb.IndexDescriptor{
				Name:   "mvcc_statistics_pkey",
				ID:     1,
				Unique: true,
				KeyColumnNames: []string{
					"crdb_internal_created_at_database_id_index_id_table_id_shard_16",
					"created_at",
					"database_id",
					"table_id",
					"index_id",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{6, 1, 2, 3, 4},
				Sharded: catpb.ShardedDescriptor{
					IsSharded:    true,
					Name:         "crdb_internal_created_at_database_id_index_id_table_id_shard_16",
					ShardBuckets: 16, // Cluster setting default.
					ColumnNames: []string{
						"created_at",
						"database_id",
						"index_id",
						"table_id",
					},
				},
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.Checks = []*descpb.TableDescriptor_CheckConstraint{{
				Expr:                  "crdb_internal_created_at_database_id_index_id_table_id_shard_16 IN (0:::INT8, 1:::INT8, 2:::INT8, 3:::INT8, 4:::INT8, 5:::INT8, 6:::INT8, 7:::INT8, 8:::INT8, 9:::INT8, 10:::INT8, 11:::INT8, 12:::INT8, 13:::INT8, 14:::INT8, 15:::INT8)",
				Name:                  "check_crdb_internal_created_at_database_id_index_id_table_id_shard_16",
				Validity:              descpb.ConstraintValidity_Validated,
				ColumnIDs:             []descpb.ColumnID{6},
				IsNonNullConstraint:   false,
				FromHashShardedColumn: true,
				ConstraintID:          tbl.NextConstraintID,
			}}
			tbl.NextConstraintID++
		},
	)

	timeRangeHashComputeExpr = `mod(fnv32(md5(crdb_internal.datums_to_bytes(end_time, start_time))), 16:::INT8)`

	TransactionExecInsightsTable = makeSystemTable(
		TxnExecutionStatsTableSchema,
		systemTable(
			catconstants.TxnExecInsightsTableName,
			descpb.InvalidID, // dynamically assigned table ID
			[]descpb.ColumnDescriptor{
				{Name: "transaction_id", ID: 1, Type: types.Uuid},
				{Name: "transaction_fingerprint_id", ID: 2, Type: types.Bytes},
				{Name: "query_summary", ID: 3, Type: types.String, Nullable: true},
				{Name: "implicit_txn", ID: 4, Type: types.Bool, Nullable: true},
				{Name: "session_id", ID: 5, Type: types.String},
				{Name: "start_time", ID: 6, Type: types.TimestampTZ, Nullable: true},
				{Name: "end_time", ID: 7, Type: types.TimestampTZ, Nullable: true},
				{Name: "user_name", ID: 8, Type: types.String, Nullable: true},
				{Name: "app_name", ID: 9, Type: types.String, Nullable: true},
				{Name: "user_priority", ID: 10, Type: types.String, Nullable: true},
				{Name: "retries", ID: 11, Type: types.Int, Nullable: true},
				{Name: "last_retry_reason", ID: 12, Type: types.String, Nullable: true},
				{Name: "problems", ID: 13, Type: types.IntArray, Nullable: true},
				{Name: "causes", ID: 14, Type: types.IntArray, Nullable: true},
				{Name: "stmt_execution_ids", ID: 15, Type: types.StringArray, Nullable: true},
				{Name: "cpu_sql_nanos", ID: 16, Type: types.Int, Nullable: true},
				{Name: "last_error_code", ID: 17, Type: types.String, Nullable: true},
				{Name: "status", ID: 18, Type: types.Int, Nullable: true},
				{Name: "contention_time", ID: 19, Type: types.Interval, Nullable: true},
				{Name: "contention_info", ID: 20, Type: types.Jsonb, Nullable: true},
				{Name: "details", ID: 21, Type: types.Jsonb, Nullable: true},
				{Name: "created", ID: 22, Type: types.TimestampTZ, DefaultExpr: &nowTZString},
				{
					Name:        "crdb_internal_end_time_start_time_shard_16",
					ID:          23,
					Type:        types.Int4,
					Nullable:    false,
					ComputeExpr: &timeRangeHashComputeExpr,
					Hidden:      true,
					Virtual:     true,
				},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ID:   0,
					ColumnNames: []string{
						"transaction_id",
						"transaction_fingerprint_id",
						"query_summary",
						"implicit_txn",
						"session_id",
						"start_time",
						"end_time",
						"user_name",
						"app_name",
						"user_priority",
						"retries",
						"last_retry_reason",
						"problems",
						"causes",
						"stmt_execution_ids",
						"cpu_sql_nanos",
						"last_error_code",
						"status",
						"contention_time",
						"contention_info",
						"details",
						"created",
					},
					ColumnIDs:       []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22},
					DefaultColumnID: 0,
				},
			},
			descpb.IndexDescriptor{
				Name:                tabledesc.LegacyPrimaryKeyIndexName,
				ID:                  1,
				Unique:              true,
				KeyColumnNames:      []string{"transaction_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				KeyColumnIDs:        []descpb.ColumnID{1},
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:           "transaction_fingerprint_id_idx",
				ID:             2,
				Unique:         false,
				KeyColumnNames: []string{"transaction_fingerprint_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs:       []descpb.ColumnID{2},
				KeySuffixColumnIDs: []descpb.ColumnID{1},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "time_range_idx",
				ID:     3,
				Unique: false,
				KeyColumnNames: []string{
					"crdb_internal_end_time_start_time_shard_16",
					"start_time",
					"end_time",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{23, 6, 7},
				KeySuffixColumnIDs: []descpb.ColumnID{1},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Sharded: catpb.ShardedDescriptor{
					IsSharded:    true,
					Name:         "crdb_internal_end_time_start_time_shard_16",
					ShardBuckets: 16,
					ColumnNames:  []string{"end_time", "start_time"},
				},
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.Checks = []*descpb.TableDescriptor_CheckConstraint{{
				Expr:                  "crdb_internal_end_time_start_time_shard_16 IN (0:::INT8, 1:::INT8, 2:::INT8, 3:::INT8, 4:::INT8, 5:::INT8, 6:::INT8, 7:::INT8, 8:::INT8, 9:::INT8, 10:::INT8, 11:::INT8, 12:::INT8, 13:::INT8, 14:::INT8, 15:::INT8)",
				Name:                  "check_crdb_internal_end_time_start_time_shard_16",
				Validity:              descpb.ConstraintValidity_Validated,
				ColumnIDs:             []descpb.ColumnID{23},
				IsNonNullConstraint:   false,
				FromHashShardedColumn: true,
				ConstraintID:          tbl.NextConstraintID,
			}}
			tbl.NextConstraintID++
		},
	)

	StatementExecInsightsTable = makeSystemTable(
		StatementExecutionStatsTableSchema,
		systemTable(
			catconstants.StmtExecInsightsTableName,
			descpb.InvalidID, // dynamically assigned table ID
			[]descpb.ColumnDescriptor{
				{Name: "session_id", ID: 1, Type: types.String},
				{Name: "transaction_id", ID: 2, Type: types.Uuid},
				{Name: "transaction_fingerprint_id", ID: 3, Type: types.Bytes},
				{Name: "statement_id", ID: 4, Type: types.String},
				{Name: "statement_fingerprint_id", ID: 5, Type: types.Bytes},
				{Name: "problem", ID: 6, Type: types.Int, Nullable: true},
				{Name: "causes", ID: 7, Type: types.IntArray, Nullable: true},
				{Name: "query", ID: 8, Type: types.String, Nullable: true},
				{Name: "status", ID: 9, Type: types.Int, Nullable: true},
				{Name: "start_time", ID: 10, Type: types.TimestampTZ, Nullable: true},
				{Name: "end_time", ID: 11, Type: types.TimestampTZ, Nullable: true},
				{Name: "full_scan", ID: 12, Type: types.Bool, Nullable: true},
				{Name: "user_name", ID: 13, Type: types.String, Nullable: true},
				{Name: "app_name", ID: 14, Type: types.String, Nullable: true},
				{Name: "user_priority", ID: 15, Type: types.String, Nullable: true},
				{Name: "database_name", ID: 16, Type: types.String, Nullable: true},
				{Name: "plan_gist", ID: 17, Type: types.String, Nullable: true},
				{Name: "retries", ID: 18, Type: types.Int, Nullable: true},
				{Name: "last_retry_reason", ID: 19, Type: types.String, Nullable: true},
				{Name: "execution_node_ids", ID: 20, Type: types.IntArray, Nullable: true},
				{Name: "index_recommendations", ID: 21, Type: types.StringArray, Nullable: true},
				{Name: "implicit_txn", ID: 22, Type: types.Bool, Nullable: true},
				{Name: "cpu_sql_nanos", ID: 23, Type: types.Int, Nullable: true},
				{Name: "error_code", ID: 24, Type: types.String, Nullable: true},
				{Name: "contention_time", ID: 25, Type: types.Interval,
					Nullable: true},
				{Name: "contention_info", ID: 26, Type: types.Jsonb,
					Nullable: true},
				{Name: "details", ID: 27, Type: types.Jsonb, Nullable: true},
				{Name: "created", ID: 28, Type: types.TimestampTZ,
					DefaultExpr: &nowTZString},
				{
					Name:        "crdb_internal_end_time_start_time_shard_16",
					ID:          29,
					Type:        types.Int4,
					Nullable:    false,
					ComputeExpr: &timeRangeHashComputeExpr,
					Hidden:      true,
					Virtual:     true,
				},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ID:   0,
					ColumnNames: []string{
						"session_id",
						"transaction_id",
						"transaction_fingerprint_id",
						"statement_id",
						"statement_fingerprint_id",
						"problem",
						"causes",
						"query",
						"status",
						"start_time",
						"end_time",
						"full_scan",
						"user_name",
						"app_name",
						"user_priority",
						"database_name",
						"plan_gist",
						"retries",
						"last_retry_reason",
						"execution_node_ids",
						"index_recommendations",
						"implicit_txn",
						"cpu_sql_nanos",
						"error_code",
						"contention_time",
						"contention_info",
						"details",
						"created",
					},
					ColumnIDs:       []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28},
					DefaultColumnID: 0,
				},
			},
			descpb.IndexDescriptor{
				Name:           tabledesc.LegacyPrimaryKeyIndexName,
				ID:             1,
				Unique:         true,
				Version:        descpb.StrictIndexColumnIDGuaranteesVersion,
				KeyColumnNames: []string{"statement_id", "transaction_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{4, 2},
			},
			descpb.IndexDescriptor{
				Name:           "transaction_id_idx",
				ID:             2,
				Unique:         false,
				KeyColumnNames: []string{"transaction_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs:       []descpb.ColumnID{2},
				KeySuffixColumnIDs: []descpb.ColumnID{4},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:           "transaction_fingerprint_id_idx",
				ID:             3,
				Unique:         false,
				KeyColumnNames: []string{"transaction_fingerprint_id", "start_time", "end_time"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{3, 10, 11},
				KeySuffixColumnIDs: []descpb.ColumnID{4, 2},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:           "statement_fingerprint_id_idx",
				ID:             4,
				Unique:         false,
				KeyColumnNames: []string{"statement_fingerprint_id", "start_time", "end_time"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{5, 10, 11},
				KeySuffixColumnIDs: []descpb.ColumnID{4, 2},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
			},
			descpb.IndexDescriptor{
				Name:   "time_range_idx",
				ID:     5,
				Unique: false,
				KeyColumnNames: []string{
					"crdb_internal_end_time_start_time_shard_16",
					"start_time",
					"end_time",
				},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
					catenumpb.IndexColumn_DESC,
				},
				KeyColumnIDs:       []descpb.ColumnID{29, 10, 11},
				KeySuffixColumnIDs: []descpb.ColumnID{4, 2},
				Version:            descpb.StrictIndexColumnIDGuaranteesVersion,
				Sharded: catpb.ShardedDescriptor{
					IsSharded:    true,
					Name:         "crdb_internal_end_time_start_time_shard_16",
					ShardBuckets: 16, // Cluster setting default.
					ColumnNames:  []string{"end_time", "start_time"},
				},
			},
		),
		func(tbl *descpb.TableDescriptor) {
			tbl.Checks = []*descpb.TableDescriptor_CheckConstraint{{
				Expr:                  "crdb_internal_end_time_start_time_shard_16 IN (0:::INT8, 1:::INT8, 2:::INT8, 3:::INT8, 4:::INT8, 5:::INT8, 6:::INT8, 7:::INT8, 8:::INT8, 9:::INT8, 10:::INT8, 11:::INT8, 12:::INT8, 13:::INT8, 14:::INT8, 15:::INT8)",
				Name:                  "check_crdb_internal_end_time_start_time_shard_16",
				Validity:              descpb.ConstraintValidity_Validated,
				ColumnIDs:             []descpb.ColumnID{29},
				IsNonNullConstraint:   false,
				FromHashShardedColumn: true,
				ConstraintID:          tbl.NextConstraintID,
			}}
			tbl.NextConstraintID++
		},
	)

	crdbInternalTableIdLastUpdatedShardStr = crdbInternalTableIdLastUpdatedShard

	TableMetadata = makeSystemTable(
		TableMetadataTableSchema,
		systemTable(catconstants.TableMetadata,
			descpb.InvalidID, // dynamically assigned table ID
			[]descpb.ColumnDescriptor{
				{Name: "db_id", ID: 1, Type: types.Int, Nullable: false},
				{Name: "table_id", ID: 2, Type: types.Int, Nullable: false},
				{Name: "db_name", ID: 3, Type: types.String, Nullable: false},
				{Name: "schema_name", ID: 4, Type: types.String, Nullable: false},
				{Name: "table_name", ID: 5, Type: types.String, Nullable: false},
				{Name: "total_columns", ID: 6, Type: types.Int, Nullable: false},
				{Name: "total_indexes", ID: 7, Type: types.Int, Nullable: false},
				{Name: "store_ids", ID: 8, Type: types.IntArray, Nullable: false},
				{Name: "replication_size_bytes", ID: 9, Type: types.Int, Nullable: false},
				{Name: "total_ranges", ID: 10, Type: types.Int, Nullable: false},
				{Name: "total_live_data_bytes", ID: 11, Type: types.Int, Nullable: false},
				{Name: "total_data_bytes", ID: 12, Type: types.Int, Nullable: false},
				{Name: "perc_live_data", ID: 13, Type: types.Float, Nullable: false},
				{Name: "last_update_error", ID: 14, Type: types.String, Nullable: true},
				{Name: "last_updated", ID: 15, Type: types.TimestampTZ, Nullable: false, DefaultExpr: &nowTZString},
				{Name: "table_type", ID: 16, Type: types.String, Nullable: false},
				{Name: "details", ID: 17, Type: types.Jsonb, Nullable: false},
				{
					Name:        "crdb_internal_last_updated_table_id_shard_16",
					ID:          18,
					Type:        types.Int4,
					Nullable:    false,
					ComputeExpr: &crdbInternalTableIdLastUpdatedShardStr,
					Hidden:      true,
					Virtual:     true,
				},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name: "primary",
					ID:   0,
					ColumnNames: []string{
						"db_id",
						"table_id",
						"db_name",
						"schema_name",
						"table_name",
						"total_columns",
						"total_indexes",
						"store_ids",
						"replication_size_bytes",
						"total_ranges",
						"total_live_data_bytes",
						"total_data_bytes",
						"perc_live_data",
						"last_update_error",
						"last_updated",
						"table_type",
						"details",
					},
					ColumnIDs: []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17},
				},
			},
			descpb.IndexDescriptor{
				Name:           tabledesc.LegacyPrimaryKeyIndexName,
				ID:             1,
				Unique:         true,
				Version:        descpb.StrictIndexColumnIDGuaranteesVersion,
				KeyColumnNames: []string{"db_id", "table_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs: []descpb.ColumnID{1, 2},
			},
			descpb.IndexDescriptor{
				Name:           "replication_size_bytes_table_id_idx",
				ID:             2,
				Unique:         false,
				Version:        descpb.StrictIndexColumnIDGuaranteesVersion,
				KeyColumnNames: []string{"replication_size_bytes", "table_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_DESC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs:       []descpb.ColumnID{9, 2},
				KeySuffixColumnIDs: []descpb.ColumnID{1},
			},
			descpb.IndexDescriptor{
				Name:           "total_ranges_table_id_idx",
				ID:             3,
				Unique:         false,
				Version:        descpb.StrictIndexColumnIDGuaranteesVersion,
				KeyColumnNames: []string{"total_ranges", "table_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_DESC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs:       []descpb.ColumnID{10, 2},
				KeySuffixColumnIDs: []descpb.ColumnID{1},
			},
			descpb.IndexDescriptor{
				Name:           "total_columns_table_id_idx",
				ID:             4,
				Unique:         false,
				Version:        descpb.StrictIndexColumnIDGuaranteesVersion,
				KeyColumnNames: []string{"total_columns", "table_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_DESC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs:       []descpb.ColumnID{6, 2},
				KeySuffixColumnIDs: []descpb.ColumnID{1},
			},
			descpb.IndexDescriptor{
				Name:           "total_indexes_table_id_idx",
				ID:             5,
				Unique:         false,
				Version:        descpb.StrictIndexColumnIDGuaranteesVersion,
				KeyColumnNames: []string{"total_indexes", "table_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_DESC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs:       []descpb.ColumnID{7, 2},
				KeySuffixColumnIDs: []descpb.ColumnID{1},
			},
			descpb.IndexDescriptor{
				Name:           "perc_live_data_id_idx",
				ID:             6,
				Unique:         false,
				Version:        descpb.StrictIndexColumnIDGuaranteesVersion,
				KeyColumnNames: []string{"perc_live_data", "table_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_DESC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs:       []descpb.ColumnID{13, 2},
				KeySuffixColumnIDs: []descpb.ColumnID{1},
				CompositeColumnIDs: []descpb.ColumnID{13},
			},
			descpb.IndexDescriptor{
				Name:           "last_updated_idx",
				ID:             7,
				Unique:         false,
				Version:        descpb.StrictIndexColumnIDGuaranteesVersion,
				KeyColumnNames: []string{"crdb_internal_last_updated_table_id_shard_16", "last_updated", "table_id"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{
					catenumpb.IndexColumn_ASC,
					catenumpb.IndexColumn_DESC,
					catenumpb.IndexColumn_ASC,
				},
				KeyColumnIDs:       []descpb.ColumnID{18, 15, 2},
				KeySuffixColumnIDs: []descpb.ColumnID{1},
				Sharded: catpb.ShardedDescriptor{
					IsSharded:    true,
					Name:         "crdb_internal_last_updated_table_id_shard_16",
					ShardBuckets: 16, // Cluster setting default.
					ColumnNames:  []string{"last_updated", "table_id"},
				},
			},
			descpb.IndexDescriptor{
				Name:                "db_name_gin",
				Type:                descpb.IndexDescriptor_INVERTED,
				ID:                  8,
				Unique:              false,
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
				KeyColumnNames:      []string{"db_name"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				InvertedColumnKinds: []catpb.InvertedIndexColumnKind{catpb.InvertedIndexColumnKind_TRIGRAM},
				KeyColumnIDs:        []descpb.ColumnID{3},
				KeySuffixColumnIDs:  []descpb.ColumnID{1, 2},
			},
			descpb.IndexDescriptor{
				Name:                "table_name_gin",
				Type:                descpb.IndexDescriptor_INVERTED,
				ID:                  9,
				Unique:              false,
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
				KeyColumnNames:      []string{"table_name"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				InvertedColumnKinds: []catpb.InvertedIndexColumnKind{catpb.InvertedIndexColumnKind_TRIGRAM},
				KeyColumnIDs:        []descpb.ColumnID{5},
				KeySuffixColumnIDs:  []descpb.ColumnID{1, 2},
			},
			descpb.IndexDescriptor{
				Name:                "schema_name_gin",
				Type:                descpb.IndexDescriptor_INVERTED,
				ID:                  10,
				Unique:              false,
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
				KeyColumnNames:      []string{"schema_name"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				InvertedColumnKinds: []catpb.InvertedIndexColumnKind{catpb.InvertedIndexColumnKind_TRIGRAM},
				KeyColumnIDs:        []descpb.ColumnID{4},
				KeySuffixColumnIDs:  []descpb.ColumnID{1, 2},
			},
			descpb.IndexDescriptor{
				Name:                "store_ids_gin",
				Type:                descpb.IndexDescriptor_INVERTED,
				ID:                  11,
				Unique:              false,
				Version:             descpb.StrictIndexColumnIDGuaranteesVersion,
				KeyColumnNames:      []string{"store_ids"},
				KeyColumnDirections: []catenumpb.IndexColumn_Direction{catenumpb.IndexColumn_ASC},
				InvertedColumnKinds: []catpb.InvertedIndexColumnKind{catpb.InvertedIndexColumnKind_DEFAULT},
				KeyColumnIDs:        []descpb.ColumnID{8},
				KeySuffixColumnIDs:  []descpb.ColumnID{1, 2},
			},
		), func(tbl *descpb.TableDescriptor) {
			tbl.Checks = []*descpb.TableDescriptor_CheckConstraint{{
				Expr:                  "crdb_internal_last_updated_table_id_shard_16 IN (0:::INT8, 1:::INT8, 2:::INT8, 3:::INT8, 4:::INT8, 5:::INT8, 6:::INT8, 7:::INT8, 8:::INT8, 9:::INT8, 10:::INT8, 11:::INT8, 12:::INT8, 13:::INT8, 14:::INT8, 15:::INT8)",
				Name:                  "check_crdb_internal_last_updated_table_id_shard_16",
				Validity:              descpb.ConstraintValidity_Validated,
				ColumnIDs:             []descpb.ColumnID{18},
				IsNonNullConstraint:   false,
				FromHashShardedColumn: true,
				ConstraintID:          tbl.NextConstraintID,
			}}
			tbl.NextConstraintID++
		},
	)

	PreparedTransactionsTable = makeSystemTable(
		PreparedTransactionsTableSchema,
		systemTable(
			catconstants.PreparedTransactionsTableName,
			descpb.InvalidID, // dynamically assigned table ID
			[]descpb.ColumnDescriptor{
				{Name: "global_id", ID: 1, Type: types.String},
				{Name: "transaction_id", ID: 2, Type: types.Uuid},
				{Name: "transaction_key", ID: 3, Type: types.Bytes, Nullable: true},
				{Name: "prepared", ID: 4, Type: types.TimestampTZ, DefaultExpr: &nowTZString},
				{Name: "owner", ID: 5, Type: types.String},
				{Name: "database", ID: 6, Type: types.String},
				{Name: "heuristic", ID: 7, Type: types.String, Nullable: true},
			},
			[]descpb.ColumnFamilyDescriptor{
				{
					Name:        "primary",
					ColumnNames: []string{"global_id", "transaction_id", "transaction_key", "prepared", "owner", "database", "heuristic"},
					ColumnIDs:   []descpb.ColumnID{1, 2, 3, 4, 5, 6, 7},
				},
			},
			pk("global_id"),
		),
	)
)

// SpanConfigurationsTableName represents system.span_configurations.
var SpanConfigurationsTableName = tree.NewTableNameWithSchema("system", catconstants.PublicSchemaName, tree.Name(catconstants.SpanConfigurationsTableName))
