// Copyright 2019 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package kvserverpb

import (
	"math"

	"github.com/cockroachdb/cockroach/pkg/storage/enginepb"
	"github.com/cockroachdb/cockroach/pkg/util/hlc"
)

var maxRaftCommandFooterSize = (&RaftCommandFooter{
	MaxLeaseIndex: math.MaxUint64,
	ClosedTimestamp: hlc.Timestamp{
		WallTime: math.MaxInt64,
		Logical:  math.MaxInt32,
	},
}).Size()

// MaxRaftCommandFooterSize returns the maximum possible size of an encoded
// RaftCommandFooter proto.
func MaxRaftCommandFooterSize() int {
	return maxRaftCommandFooterSize
}

// IsZero returns whether all fields are set to their zero value.
func (r ReplicatedEvalResult) IsZero() bool {
	return r == ReplicatedEvalResult{}
}

// IsTrivial implements apply.Command.
//
// Trivial commands can be combined in an apply.Batch for efficiency. We only
// allow this for commands that don't have any side effects except a few updates
// that are part of all commands (MVCCStats, WriteTimestamp, etc), as other side
// effects may not be correctly propagated between commands in the same batch.
//
// User writes are generally trivial, so nontrivial commands are the exception,
// notable examples being configuration changes, splits, merges, etc.
//
// At the time of writing it is possible that the current conditions are too
// strict but they are certainly sufficient.
func (r *ReplicatedEvalResult) IsTrivial() bool {
	// Check if there are any non-trivial State updates.
	if r.State != nil {
		stateAllowlist := *r.State
		// ReplicaState.Stats was previously non-nullable which caused nodes to
		// send a zero-value MVCCStats structure. If the proposal was generated by
		// an old node, we'll have decoded that zero-value structure setting
		// ReplicaState.Stats to a non-nil value which would trigger the "unhandled
		// field in ReplicatedEvalResult" assertion to fire if we didn't clear it.
		// TODO(ajwerner): eliminate this case that likely can no longer occur as of
		// at least 19.1.
		if stateAllowlist.Stats != nil && (*stateAllowlist.Stats == enginepb.MVCCStats{}) {
			stateAllowlist.Stats = nil
		}
		if stateAllowlist != (ReplicaState{}) {
			return false
		}
	}
	// Set allowlist to the value of r and clear the allowlisted fields.
	// If allowlist is zero-valued after clearing the allowlisted fields then
	// it is trivial.
	allowlist := *r
	allowlist.Delta = enginepb.MVCCStatsDelta{}
	allowlist.WriteTimestamp = hlc.Timestamp{}
	allowlist.PrevLeaseProposal = nil
	allowlist.IsProbe = false // probes are trivial, they always get refused in CheckForcedErr
	allowlist.State = nil
	// DoTimelyApplicationToAllReplicas is trivial since it can be combined in
	// an apply.Batch -- this is done by using the Command index in
	// apply.Batch.Stage to set the ForceFlushIndex. Different replicas can
	// combine different sets of Commands in an apply.Batch, but since the
	// Command index that specified DoTimelyApplicationToAllReplicas is the
	// same, the state machine will have the same state.
	allowlist.DoTimelyApplicationToAllReplicas = false
	return allowlist.IsZero()
}

// SetRaftTruncatedState puts the given RaftTruncatedState into this evaluation
// result. Since v25.1, it uses r.RaftTruncatedState field, and falls back to
// r.State.TruncatedState for backwards compatibility in mixed-version clusters.
//
// TODO(#97613): use r.RaftTruncatedState unconditionally when the ReplicaState
// field is removed.
func (r *ReplicatedEvalResult) SetRaftTruncatedState(ts *RaftTruncatedState, v25dot1 bool) {
	if v25dot1 {
		r.RaftTruncatedState = ts
	} else if r.State != nil {
		r.State.TruncatedState = ts
	} else {
		r.State = &ReplicaState{TruncatedState: ts}
	}
}

// GetRaftTruncatedState returns the RaftTruncatedState present in this
// evaluation result. Since v25.1, it can be present in one of the two places.
// See SetRaftTruncatedState.
func (r *ReplicatedEvalResult) GetRaftTruncatedState() *RaftTruncatedState {
	if ts := r.RaftTruncatedState; ts != nil {
		return ts
	} else if r.State == nil {
		return nil
	}
	return r.State.TruncatedState
}
