// Copyright 2019 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package jobs

import (
	"bytes"
	"context"
	"fmt"
	"strings"
	"time"

	"github.com/cockroachdb/cockroach/pkg/clusterversion"
	"github.com/cockroachdb/cockroach/pkg/jobs/jobspb"
	"github.com/cockroachdb/cockroach/pkg/sql/isql"
	"github.com/cockroachdb/cockroach/pkg/sql/sem/tree"
	"github.com/cockroachdb/cockroach/pkg/sql/sessiondata"
	"github.com/cockroachdb/cockroach/pkg/sql/sqlliveness"
	"github.com/cockroachdb/cockroach/pkg/util/hlc"
	"github.com/cockroachdb/cockroach/pkg/util/log"
	"github.com/cockroachdb/cockroach/pkg/util/protoutil"
	"github.com/cockroachdb/cockroach/pkg/util/timeutil"
	"github.com/cockroachdb/cockroach/pkg/util/tracing"
	"github.com/cockroachdb/errors"
)

// UpdateFn is the callback passed to Job.Update. It is called from the context
// of a transaction and is passed the current metadata for the job. The callback
// can modify metadata using the JobUpdater and the changes will be persisted
// within the same transaction.
//
// The function is free to modify contents of JobMetadata in place (but the
// changes will be ignored unless JobUpdater is used).
type UpdateFn func(txn isql.Txn, md JobMetadata, ju *JobUpdater) error

type Updater struct {
	j            *Job
	txn          isql.Txn
	txnDebugName string
}

func (j *Job) NoTxn() Updater {
	return Updater{j: j}
}

func (j *Job) DebugNameNoTxn(txnDebugName string) Updater {
	return Updater{j: j, txnDebugName: txnDebugName}
}

func (j *Job) WithTxn(txn isql.Txn) Updater {
	return Updater{j: j, txn: txn}
}

func (u Updater) update(ctx context.Context, updateFn UpdateFn) (retErr error) {
	if u.txn == nil {
		return u.j.registry.db.Txn(ctx, func(
			ctx context.Context, txn isql.Txn,
		) error {
			if u.txnDebugName != "" {
				txn.KV().SetDebugName(u.txnDebugName)
			}
			u.txn = txn
			return u.update(ctx, updateFn)
		})
	}
	ctx, sp := tracing.ChildSpan(ctx, "update-job")
	defer sp.Finish()

	var payload *jobspb.Payload
	var progress *jobspb.Progress
	var status Status
	j := u.j
	defer func() {
		if retErr != nil && !HasJobNotFoundError(retErr) {
			retErr = errors.Wrapf(retErr, "job %d", j.id)
			return
		}
		j.mu.Lock()
		defer j.mu.Unlock()
		if payload != nil {
			j.mu.payload = *payload
		}
		if progress != nil {
			j.mu.progress = *progress
		}
		if status != "" {
			j.mu.status = status
		}
	}()

	const loadJobQuery = `
WITH
  latestpayload AS (
    SELECT job_id, value
    FROM system.job_info AS payload
    WHERE info_key = 'legacy_payload' AND job_id = $1
    ORDER BY written DESC LIMIT 1
  ),
  latestprogress AS (
    SELECT job_id, value
    FROM system.job_info AS progress
    WHERE info_key = 'legacy_progress' AND job_id = $1
    ORDER BY written DESC LIMIT 1
  )
SELECT status, payload.value AS payload, progress.value AS progress,
       claim_session_id, COALESCE(last_run, created), COALESCE(num_runs, 0)
FROM system.jobs AS j
INNER JOIN latestpayload AS payload ON j.id = payload.job_id
LEFT JOIN latestprogress AS progress ON j.id = progress.job_id
WHERE id = $1
`
	row, err := u.txn.QueryRowEx(
		ctx, "select-job", u.txn.KV(),
		sessiondata.NodeUserSessionDataOverride,
		loadJobQuery, j.ID(),
	)
	if err != nil {
		return err
	}
	if row == nil {
		return &JobNotFoundError{jobID: j.ID()}
	}

	if status, err = unmarshalStatus(row[0]); err != nil {
		return err
	}
	if payload, err = UnmarshalPayload(row[1]); err != nil {
		return err
	}
	if progress, err = UnmarshalProgress(row[2]); err != nil {
		return err
	}
	beforeProgress := *progress
	beforePayload := *payload
	if j.session != nil {
		if row[3] == tree.DNull {
			return errors.Errorf(
				"with status %q: expected session %q but found NULL",
				status, j.session.ID())
		}
		storedSession := []byte(*row[3].(*tree.DBytes))
		if !bytes.Equal(storedSession, j.session.ID().UnsafeBytes()) {
			return errors.Errorf(
				"with status %q: expected session %q but found %q",
				status, j.session.ID(), sqlliveness.SessionID(storedSession))
		}
	} else {
		log.VInfof(ctx, 1, "job %d: update called with no session ID", j.ID())
	}

	lastRun, ok := row[4].(*tree.DTimestamp)
	if !ok {
		return errors.AssertionFailedf("expected timestamp last_run, but got %T", lastRun)
	}
	numRuns, ok := row[5].(*tree.DInt)
	if !ok {
		return errors.AssertionFailedf("expected int num_runs, but got %T", numRuns)
	}

	md := JobMetadata{
		ID:       j.ID(),
		Status:   status,
		Payload:  payload,
		Progress: progress,
	}

	var ju JobUpdater
	if err := updateFn(u.txn, md, &ju); err != nil {
		return err
	}

	// a job status is considered updated if:
	//  1. the status of the updated metadata is not empty
	//  2. the status of the updated metadata is not equal to old status
	//  3. the status of the updated metadata and the old status is running
	// #1 should be sufficient to determine whether a status change has happened
	// as the status field of ju.md is not empty only when JobMetadata.UpdateStatus is
	// called, and this is only called in places where a status change is happening.
	// Since this may not be in the case in the future we add condition #2. #3 is
	// required when a job starts because it may already have a "running" status.
	//
	if ju.md.Status != "" &&
		(ju.md.Status != status || (ju.md.Status == StatusRunning && status == StatusRunning)) {
		u.txn.KV().AddCommitTrigger(func(ctx context.Context) {
			p := ju.md.Payload
			// In some cases, ju.md.Payload may be nil, such as a cancel-requested status update.
			// In this case, payload is used.
			if p == nil {
				p = payload
			}
			// If run stats has been updated, use the updated run stats.
			LogStatusChangeStructured(ctx, md.ID, p.Type().String(), p, status, ju.md.Status)
		})
	}
	if j.registry.knobs.BeforeUpdate != nil {
		if err := j.registry.knobs.BeforeUpdate(md, ju.md); err != nil {
			return err
		}
	}

	if !ju.hasUpdates() {
		return nil
	}

	// Build a statement of the following form, depending on which properties
	// need updating:
	//
	//   UPDATE system.jobs
	//   SET
	//     [status = $2,]
	//     [payload = $y,]
	//     [progress = $z]
	//   WHERE
	//     id = $1

	var setters []string
	params := []interface{}{j.ID()} // $1 is always the job ID.
	addSetter := func(column string, value interface{}) {
		params = append(params, value)
		setters = append(setters, fmt.Sprintf("%s = $%d", column, len(params)))
	}

	if ju.md.Status != "" {
		addSetter("status", ju.md.Status)
	}

	var payloadBytes []byte
	if ju.md.Payload != nil {
		payload = ju.md.Payload
		var err error
		payloadBytes, err = protoutil.Marshal(payload)
		if err != nil {
			return err
		}
	}

	var progressBytes []byte
	if ju.md.Progress != nil {
		progress = ju.md.Progress
		progress.ModifiedMicros = timeutil.ToUnixMicros(u.now())
		var err error
		progressBytes, err = protoutil.Marshal(progress)
		if err != nil {
			return err
		}
	}

	if len(setters) != 0 {
		updateStmt := fmt.Sprintf(
			"UPDATE system.jobs SET %s WHERE id = $1",
			strings.Join(setters, ", "),
		)
		n, err := u.txn.ExecEx(
			ctx, "job-update", u.txn.KV(),
			sessiondata.NodeUserSessionDataOverride,
			updateStmt, params...,
		)
		if err != nil {
			return err
		}
		if n != 1 {
			return errors.Errorf(
				"expected exactly one row affected, but %d rows affected by job update", n,
			)
		}
	}

	v, err := u.txn.GetSystemSchemaVersion(ctx)
	if err != nil {
		return err
	}
	if v.AtLeast(clusterversion.V25_1_AddJobsTables.Version()) {
		if ju.md.Status != "" && ju.md.Status != status {
			if err := j.Messages().Record(ctx, u.txn, "state", string(ju.md.Status)); err != nil {
				return err
			}
			// If we are changing state, we should clear out "running status", unless
			// we are about to set it to something instead.
			if progress == nil || progress.RunningStatus == "" {
				if err := j.StatusStorage().Clear(ctx, u.txn); err != nil {
					return err
				}
			}
		}

		if progress != nil {
			var ts hlc.Timestamp
			if hwm := progress.GetHighWater(); hwm != nil {
				ts = *hwm
			}

			if err := j.ProgressStorage().Set(ctx, u.txn, float64(progress.GetFractionCompleted()), ts); err != nil {
				return err
			}

			if progress.RunningStatus != beforeProgress.RunningStatus {
				if err := j.StatusStorage().Set(ctx, u.txn, progress.RunningStatus); err != nil {
					return err
				}
			}

			if progress.TraceID != beforeProgress.TraceID {
				if err := j.Messages().Record(ctx, u.txn, "trace-id", fmt.Sprintf("%d", progress.TraceID)); err != nil {
					return err
				}
			}
		}
	}
	if v.AtLeast(clusterversion.V25_1_AddJobsColumns.Version()) {

		vals := []interface{}{j.ID()}

		var update strings.Builder

		if payloadBytes != nil {
			if beforePayload.Description != payload.Description {
				if update.Len() > 0 {
					update.WriteString(", ")
				}
				vals = append(vals, payload.Description)
				fmt.Fprintf(&update, "description = $%d", len(vals))
			}

			if beforePayload.UsernameProto.Decode() != payload.UsernameProto.Decode() {
				if update.Len() > 0 {
					update.WriteString(", ")
				}
				vals = append(vals, payload.UsernameProto.Decode().Normalized())
				fmt.Fprintf(&update, "owner = $%d", len(vals))
			}

			if beforePayload.Error != payload.Error {
				if update.Len() > 0 {
					update.WriteString(", ")
				}
				vals = append(vals, payload.Error)
				fmt.Fprintf(&update, "error_msg = $%d", len(vals))
			}

			if beforePayload.FinishedMicros != payload.FinishedMicros {
				if update.Len() > 0 {
					update.WriteString(", ")
				}
				vals = append(vals, time.UnixMicro(payload.FinishedMicros))
				fmt.Fprintf(&update, "finished = $%d", len(vals))
			}

		}
		if len(vals) > 1 {
			stmt := fmt.Sprintf("UPDATE system.jobs SET %s WHERE id = $1", update.String())
			if _, err := u.txn.ExecEx(
				ctx, "job-update-row", u.txn.KV(),
				sessiondata.NodeUserSessionDataOverride,
				stmt, vals...,
			); err != nil {
				return err
			}
		}
	}

	// Insert the job payload and progress into the system.jobs_info table.
	infoStorage := j.InfoStorage(u.txn)
	infoStorage.claimChecked = true
	if payloadBytes != nil {
		if err := infoStorage.WriteLegacyPayload(ctx, payloadBytes); err != nil {
			return err
		}
	}
	if progressBytes != nil {
		if err := infoStorage.WriteLegacyProgress(ctx, progressBytes); err != nil {
			return err
		}
	}

	return nil
}

// JobMetadata groups the job metadata values passed to UpdateFn.
type JobMetadata struct {
	ID       jobspb.JobID
	Status   Status
	Payload  *jobspb.Payload
	Progress *jobspb.Progress
}

// CheckRunningOrReverting returns an InvalidStatusError if md.Status is not
// StatusRunning or StatusReverting.
func (md *JobMetadata) CheckRunningOrReverting() error {
	if md.Status != StatusRunning && md.Status != StatusReverting {
		return &InvalidStatusError{md.ID, md.Status, "update progress on", md.Payload.Error}
	}
	return nil
}

// JobUpdater accumulates changes to job metadata that are to be persisted.
type JobUpdater struct {
	md JobMetadata
}

// UpdateStatus sets a new status (to be persisted).
func (ju *JobUpdater) UpdateStatus(status Status) {
	ju.md.Status = status
}

// UpdatePayload sets a new Payload (to be persisted).
//
// WARNING: the payload can be large (resulting in a large KV for each version);
// it shouldn't be updated frequently.
func (ju *JobUpdater) UpdatePayload(payload *jobspb.Payload) {
	ju.md.Payload = payload
}

// UpdateProgress sets a new Progress (to be persisted).
func (ju *JobUpdater) UpdateProgress(progress *jobspb.Progress) {
	ju.md.Progress = progress
}

func (ju *JobUpdater) hasUpdates() bool {
	return ju.md != JobMetadata{}
}

func (ju *JobUpdater) PauseRequested(
	ctx context.Context, txn isql.Txn, md JobMetadata, reason string,
) error {
	return ju.PauseRequestedWithFunc(ctx, txn, md, nil /* fn */, reason)
}

func (ju *JobUpdater) PauseRequestedWithFunc(
	ctx context.Context, txn isql.Txn, md JobMetadata, fn onPauseRequestFunc, reason string,
) error {
	if md.Status == StatusPauseRequested || md.Status == StatusPaused {
		return nil
	}
	if md.Status != StatusPending && md.Status != StatusRunning && md.Status != StatusReverting {
		return fmt.Errorf("job with status %s cannot be requested to be paused", md.Status)
	}
	if fn != nil {
		if err := fn(ctx, md, ju); err != nil {
			return err
		}
	}
	ju.UpdateStatus(StatusPauseRequested)
	md.Payload.PauseReason = reason
	ju.UpdatePayload(md.Payload)
	log.Infof(ctx, "job %d: pause requested recorded with reason %s", md.ID, reason)
	return nil
}

// Unpaused sets the status of the tracked job to running or reverting iff the
// job is currently paused. It does not directly resume the job.
func (ju *JobUpdater) Unpaused(_ context.Context, md JobMetadata) error {
	if md.Status == StatusRunning || md.Status == StatusReverting {
		// Already resumed - do nothing.
		return nil
	}
	if md.Status != StatusPaused {
		return fmt.Errorf("job with status %s cannot be resumed", md.Status)
	}
	// We use the absence of error to determine what state we should
	// resume into.
	if md.Payload.FinalResumeError == nil {
		ju.UpdateStatus(StatusRunning)
	} else {
		ju.UpdateStatus(StatusReverting)
	}
	return nil
}

func (ju *JobUpdater) CancelRequested(ctx context.Context, md JobMetadata) error {
	return ju.CancelRequestedWithReason(ctx, md, errJobCanceled)
}

func (ju *JobUpdater) CancelRequestedWithReason(
	ctx context.Context, md JobMetadata, reason error,
) error {
	if md.Payload.Noncancelable {
		return errors.Newf("job %d: not cancelable", md.ID)
	}
	if md.Status == StatusCancelRequested || md.Status == StatusCanceled {
		return nil
	}
	if md.Status != StatusPending && md.Status != StatusRunning && md.Status != StatusPaused {
		return fmt.Errorf("job with status %s cannot be requested to be canceled", md.Status)
	}
	if md.Status == StatusPaused && md.Payload.FinalResumeError != nil {
		decodedErr := errors.DecodeError(ctx, *md.Payload.FinalResumeError)
		return errors.Wrapf(decodedErr, "job %d is paused and has non-nil FinalResumeError "+
			"hence cannot be canceled and should be reverted", md.ID)
	}
	if !errors.Is(reason, errJobCanceled) {
		md.Payload.Error = reason.Error()
		ju.UpdatePayload(md.Payload)
	}
	ju.UpdateStatus(StatusCancelRequested)
	return nil
}

// Update is used to read the metadata for a job and potentially update it.
//
// The updateFn is called in the context of a transaction and is passed the
// current metadata for the job. It can choose to update parts of the metadata
// using the JobUpdater, causing them to be updated within the same transaction.
//
// Sample usage:
//
//	err := j.Update(ctx, func(_ *kv.Txn, md jobs.JobMetadata, ju *jobs.JobUpdater) error {
//	  if md.Status != StatusRunning {
//	    return errors.New("job no longer running")
//	  }
//	  ju.UpdateStatus(StatusPaused)
//	  // <modify md.Payload>
//	  ju.UpdatePayload(md.Payload)
//	}
//
// Note that there are various convenience wrappers (like FractionProgressed)
// defined in jobs.go.
func (u Updater) Update(ctx context.Context, updateFn UpdateFn) error {
	return u.update(ctx, updateFn)
}

func (u Updater) now() time.Time {
	return u.j.registry.clock.Now().GoTime()
}
