// Copyright 2021 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package geomfn

import (
	"testing"

	"github.com/cockroachdb/cockroach/pkg/geo"
)

var (
	benchLargePolygons = []geo.Geometry{
		geo.MustParseGeometry("MULTIPOLYGON (((96.863129102000073 -12.181735934999949, 96.869395379000082 -12.187920830999929, 96.845225457000026 -12.197686455999929, 96.830577019000032 -12.179945570999905, 96.823415561000047 -12.150974216999941, 96.82154381600003 -12.126560153999947, 96.828379754000082 -12.126560153999947, 96.828379754000082 -12.136325778999947, 96.829844597000033 -12.144463799999926, 96.83521569100003 -12.160577080999929, 96.83130944100003 -12.171482028999947, 96.836192254000082 -12.182061455999929, 96.847666863000029 -12.18718840899993, 96.863129102000073 -12.181735934999949)), ((96.902191602000073 -12.199965101999908, 96.892344597000033 -12.195896091999941, 96.889659050000034 -12.192803643999923, 96.888519727000073 -12.187432549999926, 96.90007571700005 -12.18678150799991, 96.909027540000068 -12.182061455999929, 96.914561394000032 -12.17351653399993, 96.916188998000052 -12.16179778399993, 96.921071811000047 -12.17351653399993, 96.920176629000082 -12.184991143999923, 96.913910352000073 -12.194512627999927, 96.902191602000073 -12.199965101999908)), ((105.704112175000034 -10.430840752999927, 105.714691602000073 -10.43718840899993, 105.712657097000033 -10.450941664999959, 105.711436394000032 -10.469903252999927, 105.70630944100003 -10.494235934999949, 105.70630944100003 -10.514255466999941, 105.697927280000044 -10.52898528399993, 105.699880405000044 -10.553399346999925, 105.69459069100003 -10.564873955999929, 105.678721550000034 -10.566013278999947, 105.67351321700005 -10.549086195999905, 105.66724694100003 -10.533298434999949, 105.664073113000029 -10.518487237999921, 105.649261915000068 -10.513116143999923, 105.62818444100003 -10.508965752999927, 105.604991082000026 -10.510023695999905, 105.584971550000034 -10.516371351999908, 105.581797722000033 -10.504815362999921, 105.592295769000032 -10.498467705999929, 105.599619988000029 -10.480564059999949, 105.594248894000032 -10.466729424999926, 105.594248894000032 -10.454034112999921, 105.607188347000033 -10.458428643999923, 105.627126498000052 -10.468926690999922, 105.654551629000025 -10.469903252999927, 105.66928144600007 -10.456231377999927, 105.68523196700005 -10.440362237999921, 105.704112175000034 -10.430840752999927)))"),
		geo.MustParseGeometry("MULTIPOLYGON (((33.78093509200005 34.976345119000086, 33.760427280000044 34.979681708000044, 33.717295769000089 34.977769273000035, 33.701508009000065 34.972886460000041, 33.702934611000103 34.987943014000038, 33.711461222000082 34.985565899000093, 33.715440308000098 34.997296448000085, 33.699730672000101 35.002722473000048, 33.696630086000084 35.008975322000069, 33.705311727000037 35.015228170000043, 33.702211141000078 35.022256165000059, 33.685002890000078 35.029284160000074, 33.679434599000103 35.033898764000057, 33.675649455000041 35.037035624000069, 33.674099162000061 35.046440734000029, 33.678853394000043 35.055794169000066, 33.694459675000076 35.058171285000029, 33.705311727000037 35.066749573000052, 33.714716838000072 35.066749573000052, 33.719367716000136 35.062770487000066, 33.711461222000082 35.04096303300004, 33.707585489000053 35.029284160000074, 33.718489217000126 35.032384745000016, 33.739676554000141 35.04721588200006, 33.766134888000067 35.031609599000021, 33.775539999000102 35.040187887000044, 33.786495402000071 35.038534240999994, 33.794350220000126 35.040187887000044, 33.798277628000108 35.052641907000023, 33.824012492000065 35.066749573000052, 33.834864542000048 35.063597311000095, 33.842719360000046 35.056620993000095, 33.853571411000075 35.058171285000029, 33.866903930000035 35.066749573000052, 33.871554810000077 35.073054098000071, 33.876929159000042 35.076826477000097, 33.871554810000077 35.085456442000051, 33.871554810000077 35.100235901000062, 33.876205689000074 35.118994446000059, 33.889434855000047 35.118994446000059, 33.891811971000095 35.110467834000033, 33.898839966000082 35.108814189000086, 33.903594198000121 35.099512431000065, 33.905867961000126 35.096360168999993, 33.905867961000126 35.090882466000025, 33.913619425000036 35.090882466000025, 33.921474244000137 35.080702210000041, 33.914446249000036 35.073054098000071, 33.908245077000061 35.07072865800005, 33.906505332000108 35.06910490500006, 33.898116496000114 35.061271872000063, 33.880133098000044 35.073054098000071, 33.874655396000037 35.067524719000076, 33.86762740100005 35.06049672400006, 33.855121704000112 35.053417053000132, 33.841169067000124 35.051091612999997, 33.834864542000048 35.056620993000095, 33.827113079000128 35.061271872000063, 33.813780558000076 35.055794169000066, 33.804375447000041 35.049541321000063, 33.799001099000066 35.038534240999994, 33.822358847000146 35.030059306000069, 33.830213664000041 35.023031311000082, 33.829386841000144 35.00117218000004, 33.840342244000112 34.993369039000115, 33.859049113000083 34.991818745000003, 33.859049113000083 34.974662171000048, 33.850470825000059 34.973008525000083, 33.838068481000107 34.963706767000062, 33.84581994600012 34.959727682000064, 33.864423462000047 34.962983297000065, 33.891840639430939 34.958139066829645, 33.883799675000034 34.949123440000051, 33.874522332000083 34.941229559000078, 33.862315300000034 34.937892971000053, 33.84742272200009 34.942450262000079, 33.81967207100007 34.964748440000051, 33.804209832000083 34.972601630000042, 33.781830274000072 34.976223049000055, 33.784945110000081 34.976212464000056, 33.788045694000118 34.976987610000052, 33.7927999260001 34.977762757000065, 33.794350220000126 34.977762757000065, 33.791146281000067 34.982413636000032, 33.786495402000071 34.984687399000066, 33.78256799400009 34.984687399000066, 33.777917115000037 34.984687399000066, 33.773989705000076 34.988666484000035, 33.766134888000067 34.990268454000059, 33.761484009000071 34.990268454000059, 33.759210246000066 34.988666484000035, 33.765411418000042 34.985565899000093, 33.769338826000137 34.983963929000069, 33.770889120000049 34.980088196000011, 33.775539999000102 34.980088196000011, 33.780190878000042 34.979313050000101, 33.78093509200005 34.976345119000086), (33.742792014000088 35.001232846999997, 33.746688555000105 35.002710845000067, 33.752063096000143 35.00432320700007, 33.752063096000143 35.014400470000041, 33.746151101000123 35.015206651000042, 33.741314015000114 35.013728653000058, 33.740239107000093 35.010100838000099, 33.738761108000062 35.00526375200009, 33.739701653000111 35.002576482000066, 33.742792014000088 35.001232846999997)))"),
		geo.MustParseGeometry("MULTIPOLYGON(((44.7661352950001 37.1419200650001,44.7525443930001 37.1131362920001,44.7526994220001 37.10331777,44.7606575930001 37.0856444290001,44.766548706 37.0789523320001,44.7736283770001 37.0763426720001,44.7811214600001 37.0744823200001,44.7883561610001 37.0702448530001,44.7922318930001 37.063733622,44.7972961830001 37.0484890750001,44.8018953860001 37.0435281380001,44.8111971440001 37.0419261680001,44.83093754 37.0468354290001,44.8406010340001 37.047197165,44.8475773520001 37.0440449010001,44.8589461670001 37.0341230270001,44.880753622 37.0249246220001,44.8873682050001 37.0159329230001,44.8857145590001 37.0050808720001,44.874759156 36.9940221150001,44.869178101 36.9673570760001,44.8748625080001 36.949657898,44.884060913 36.933353984,44.8888151450001 36.910926412,44.8834407960001 36.886845195,44.869643189 36.868965149,44.8341414800001 36.834031881,44.8233927820001 36.809330547,44.8312476 36.791915589,44.851194702 36.7812185670001,44.8772396240001 36.7764643360001,44.9082454830001 36.7778595990001,44.9227148840001 36.775895895,44.9352205810001 36.7671109010001,44.9444189860001 36.758274231,44.9547542720001 36.752331442,44.966433146 36.749334209,44.9794556070001 36.74902415,44.9961987710001 36.741686096,45.0109782310001 36.7255630500001,45.0352661540001 36.6895962530001,45.0459114990001 36.668047181,45.0443095300001 36.6493403120001,45.0248791910001 36.613166809,45.014182169 36.579008687,45.0132519940001 36.5579247030001,45.0102547610001 36.5560126760001,44.9978524170001 36.552085266,44.9934082440001 36.5496564740001,44.991392863 36.5337401330001,45.0059139400001 36.5187022910001,45.0095246660001 36.5161959100001,45.0253442790001 36.5052147420001,45.038160034 36.4940009560001,45.0390902100001 36.4798416140001,45.0454980870001 36.4716250610001,45.0540763750001 36.4649588010001,45.0620862230001 36.455346985,45.0668921310001 36.442376201,45.0685457760001 36.432506002,45.0722664790001 36.4234109500001,45.0837386480001 36.4128172810001,45.1108170980001 36.4025336710001,45.1399626060001 36.4043940230001,45.1959798580001 36.4224290970001,45.2213013100001 36.4209304820001,45.238871298 36.402998759,45.2496199960001 36.3779356900001,45.2544775800001 36.3547846480001,45.2570097250001 36.3116865030001,45.2639343670001 36.2942715460001,45.2826929120001 36.274892884,45.2585083420001 36.2622321580001,45.2641927490001 36.2501915490001,45.2827962650001 36.238409322,45.297369018 36.226730449,45.3020198980001 36.2075584920001,45.299539429 36.1614114380001,45.3046037190001 36.1403791300001,45.335609579 36.1077196250001,45.3483219800001 36.08751414,45.3441878660001 36.067463685,45.3318888760001 36.05159902,45.3195898850001 36.0310318000001,45.3136987710001 36.010412903,45.3202100020001 35.9942898560001,45.3202100020001 35.994134827,45.3203133540001 35.994031474,45.32052006 35.9939797980001,45.338090047 35.979303691,45.3595874430001 35.976874899,45.3818083090001 35.982972718,45.4015487070001 35.9939797980001,45.401858765 35.994031474,45.4019621180001 35.994083151,45.4021688230001 35.9942381800001,45.419945516 35.9982689420001,45.4531217850001 36.0116531370001,45.4792700600001 36.0120148720001,45.5014236760001 36.0054347770001,45.5399898680001 35.9939797980001,45.5769385170001 35.966229553,45.5898576260001 35.9596149710001,45.617039429 35.9554291800001,45.6468567300001 35.9332083130001,45.6932104900001 35.879878235,45.7187386480001 35.8283051560001,45.7310376390001 35.8151276660001,45.7490210370001 35.8109935510001,45.7867448320001 35.8202436330001,45.7979069420001 35.8183832810001,45.8141850180001 35.809339905,45.8347522380001 35.810528463,45.854079223 35.8183316040001,45.8783154700001 35.835229798,45.8884957280001 35.8348680630001,45.8987793380001 35.832180888,45.9091146240001 35.8313023890001,45.9412573650001 35.840345765,46.00492273 35.8379686490001,46.0249731860001 35.8432913210001,46.0443001710001 35.8522830200001,46.0604232180001 35.857243958,46.0769596760001 35.8569855750001,46.1077588300001 35.8474254360001,46.115045207 35.846443584,46.1198511150001 35.842877909,46.1248120520001 35.822465719,46.1291011960001 35.815541077,46.1351990150001 35.8100116990001,46.143363891 35.8045339960001,46.1635177000001 35.797506002,46.184963419 35.7981777960001,46.227441447 35.805722555,46.246406698 35.8114069630001,46.2635115960001 35.8211221320001,46.2813916420001 35.8283051560001,46.3029923910001 35.826548157,46.319632202 35.8168846640001,46.327073608 35.8035004690001,46.3250582280001 35.7878425090001,46.3133793540001 35.7715127570001,46.2975146900001 35.7598855590001,46.2671289470001 35.744124248,46.2534863690001 35.7277428190001,46.2378800860001 35.715547181,46.2170028080001 35.713583476,46.1777287190001 35.7153921510001,46.1276025800001 35.69368805,46.1070353600001 35.689812317,46.0370654710001 35.6915693160001,46.0164982510001 35.685729879,46.0037858480001 35.674257711,45.9958793540001 35.6583930460001,45.9923653560001 35.640874736,45.9923136800001 35.6243899540001,46.00202885 35.585942688,45.9992383220001 35.5721450810001,45.9686975500001 35.579741516,45.9638916420001 35.579741516,45.9593441160001 35.5787079880001,45.9680257570001 35.5585025030001,45.9792395430001 35.5394855750001,45.9831152750001 35.5352997840001,45.9845622150001 35.5310623170001,45.9832186280001 35.5269798790001,45.9792395430001 35.523000794,45.9713330480001 35.5180398560001,45.9660620530001 35.511476949,45.9635815840001 35.5034154260001,45.9639949950001 35.4941653440001,45.9770174560001 35.465329895,46.0409928790001 35.3815623980001,46.0962866620001 35.341151429,46.120161173 35.3185171510001,46.1273958740001 35.2890615840001,46.1252854410001 35.2849893410001,46.1200578200001 35.274902243,46.107655477 35.262758281,46.0987671310001 35.2500458790001,46.1014543050001 35.2342845660001,46.115406942 35.226326396,46.1541642660001 35.2266364550001,46.169253784 35.2162494920001,46.165481405 35.1899461870001,46.143828979 35.1596121220001,46.1303931070001 35.13139679,46.1510636800001 35.111604716,46.1431055100001 35.0995124310001,46.1323568120001 35.0947581990001,46.1199544680001 35.0927428180001,46.0945296630001 35.0858181760001,46.0766496170001 35.0896939090001,46.0665727140001 35.08881541,46.0397526450001 35.075534566,46.0250765380001 35.0642691040001,46.0092118730001 35.0609101360001,45.9792395430001 35.0715554820001,45.9663721110001 35.070728658,45.9560368240001 35.0742426560001,45.9349528400001 35.0854564420001,45.9201733810001 35.0896422330001,45.9131970620001 35.0873167930001,45.899399454 35.068609925,45.898365927 35.0639073690001,45.9011564540001 35.051608378,45.8990893960001 35.0469058230001,45.8930949300001 35.0443736780001,45.879038941 35.0432367960001,45.872476034 35.0414281210001,45.861882365 35.032798157,45.8570247800001 35.0212743120001,45.856818074 35.0080451460001,45.8597119550001 34.9944025680001,45.8597119550001 34.9942475380001,45.8689620360001 34.9683576460001,45.8663265380001 34.949547425,45.8578516030001 34.9313056440001,45.850203492 34.9068110150001,45.83501062 34.8901712040001,45.8080355230001 34.8978709920001,45.7962024270001 34.903063666,45.7782698980001 34.9108934530001,45.7551188560001 34.909963277,45.748504272 34.890739645,45.7504679770001 34.865366517,45.7454036870001 34.84097524,45.7177051190001 34.825110576,45.689231405 34.822630107,45.6765706790001 34.818289287,45.6667521560001 34.806920472,45.6635998940001 34.7949315390001,45.6659253340001 34.774726054,45.662824748 34.7632538860001,45.6350228270001 34.7369505830001,45.6273747150001 34.7208275350001,45.6422575280001 34.7068748990001,45.6505257570001 34.702999166,45.6595174560001 34.6972114060001,45.6668555100001 34.6899767050001,45.6703695060001 34.6818118290001,45.673470093 34.6711664840001,45.6783793540001 34.665120341,45.6834953200001 34.6602110800001,45.6876294350001 34.6532347620001,45.6907816980001 34.6343211880001,45.6912467860001 34.6155109660001,45.694450724 34.5985610970001,45.7056128340001 34.5855903120001,45.7085583910001 34.579079081,45.708300008 34.56838206,45.7056128340001 34.557736715,45.7015303960001 34.5512771610001,45.6955876050001 34.550243632,45.6735734460001 34.5566515100001,45.6048437910001 34.5611473600001,45.5739412850001 34.567296855,45.5396281330001 34.582231344,45.5006641030001 34.5916881310001,45.4965299880001 34.564247945,45.5048498950001 34.523785299,45.5034546310001 34.4941747030001,45.4987003990001 34.48624237,45.4930159910001 34.4790076700001,45.486504761 34.4727289840001,45.4792700600001 34.4672512820001,45.4267668050001 34.457510275,45.4170516350001 34.444358623,45.4479541420001 34.3620897430001,45.4607698980001 34.340463155,45.4792700600001 34.329146017,45.4992171640001 34.342401022,45.5215413820001 34.3425043740001,45.5422119550001 34.3316006470001,45.5575598550001 34.3119636030001,45.5630375570001 34.289277649,45.5610221760001 34.264602153,45.5476896570001 34.216491395,45.5427287190001 34.2074738570001,45.5286727300001 34.1885602830001,45.5266573490001 34.179310202,45.531463257 34.167915548,45.5388013100001 34.1592339070001,45.5445373950001 34.150629781,45.5445373950001 34.1394935100001,45.5343571370001 34.1282022100001,45.493946167 34.1006845090001,45.4792700600001 34.0876362110001,45.4548787840001 34.069859518,45.4436133220001 34.0476644900001,45.4351900640001 34.022136332,45.419945516 33.9943602500001,45.419945516 33.994256898,45.4197388110001 33.994256898,45.4114705810001 33.9871513880001,45.4017554120001 33.981622009,45.3803613690001 33.973560486,45.4010319420001 33.9494534300001,45.4237178950001 33.9387822470001,45.4497111410001 33.9373869840001,45.4792700600001 33.941185201,45.4817505290001 33.940875143,45.4839209390001 33.9400483200001,45.48578129 33.938653056,45.5747681070001 33.803932597,45.5882039790001 33.7915044150001,45.6036035560001 33.780729879,45.6167293700001 33.7686892700001,45.6239123940001 33.752075297,45.6282015390001 33.7318439740001,45.63512618 33.714170634,45.6451514080001 33.697918396,45.6587939860001 33.6816919970001,45.6739868570001 33.6690312710001,45.7211157640001 33.6407642620001,45.7246296230001 33.6330364130001,45.7279887290001 33.625648905,45.7277303470001 33.59009552,45.734551636 33.583119202,45.7520182700001 33.58676239,45.7684513760001 33.595909119,45.7975452060001 33.6193185420001,45.8157869870001 33.625803935,45.8643628330001 33.626424053,45.8859635820001 33.630919902,45.8820361740001 33.6001982630001,45.8997095140001 33.5851604210001,45.9208968500001 33.5729906210001,45.9279248450001 33.5508731080001,45.9156258550001 33.5358094280001,45.8681868900001 33.5115731820001,45.852580608 33.49436493,45.8697371830001 33.482117615,45.899399454 33.4763040170001,45.9296818450001 33.4795079550001,45.9481820070001 33.4942615770001,45.953142944 33.496587016,45.9581555590001 33.4973880010001,45.9632715250001 33.496561177,45.9644874470001 33.495991214,45.9682324620001 33.494235738,45.9713847250001 33.4927629600001,45.9744336350001 33.4907992550001,46.0080233150001 33.4556592810001,46.0193921310001 33.4383993530001,46.0274536540001 33.4195116170001,46.0305025640001 33.3992027800001,46.0310710050001 33.382511291,46.0362386480001 33.3679385380001,46.0697249760001 33.340860088,46.1095158290001 33.293705343,46.1413485110001 33.272104594,46.1553011480001 33.2602190150001,46.1641894940001 33.243346659,46.1643961990001 33.233063049,46.1583500570001 33.2138652550001,46.1576782640001 33.2056745410001,46.1621224360001 33.1963727830001,46.1736979570001 33.190223287,46.174318074 33.181179911,46.1671867260001 33.168338318,46.153440796 33.1542564910001,46.1264656990001 33.131777242,46.1053817140001 33.1183930470001,46.0891553140001 33.1156800340001,46.07241215 33.116765239,46.0503979900001 33.1151115930001,46.0302958570001 33.1056806440001,46.0293140050001 33.093510844,46.043266643 33.0834597780001,46.08719161 33.07922231,46.1060018310001 33.072504375,46.1204195560001 33.0616781620001,46.126155639 33.047983908,46.1194893800001 33.031008199,46.1046582440001 33.018063253,46.0880184330001 33.006720276,46.075719442 32.9944988000001,46.075719442 32.9943437700001,46.097010132 32.9543203740001,46.1555595290001 32.9483517460001,46.2734334720001 32.9594880170001,46.3796285400001 32.9317636110001,46.4793123780001 32.89179189,46.5073210050001 32.868020732,46.6006130590001 32.8225414760001,46.6044210210001 32.8206851200001,46.6503613690001 32.7893692020001,46.7158870850001 32.756012065,46.757331584 32.7161953740001,47.0583984790001 32.494477641,47.0586051840001 32.494477641,47.05865686 32.494400127,47.0905928960001 32.4745563760001,47.121030315 32.4610429890001,47.152552938 32.4551777140001,47.1877962650001 32.458381653,47.2054696050001 32.4640402220001,47.2512549240001 32.4854601040001,47.2653109130001 32.4846849570001,47.321948283 32.468226013,47.3433423260001 32.458691712,47.356106405 32.4461860150001,47.3673201910001 32.4307606,47.3841150310001 32.4125704960001,47.4108317470001 32.3952847300001,47.416412801 32.3879725140001,47.4179630940001 32.3764228310001,47.4143457440001 32.3698599250001,47.4087646890001 32.3643305460001,47.3999796960001 32.3457787080001,47.395948934 32.341980489,47.3957939050001 32.3370195520001,47.402460165 32.3235061650001,47.4070593670001 32.317770081,47.4368766680001 32.2936888630001,47.4415792240001 32.288185323,47.4453516030001 32.2826301070001,47.4630249430001 32.261830343,47.4694328200001 32.2561200970001,47.4791479900001 32.2520634980001,47.4845223390001 32.239351095,47.5040043540001 32.2267420460001,47.5092236730001 32.2137970990001,47.5076733800001 32.1998444630001,47.5023507080001 32.190129294,47.496407918 32.1813442990001,47.4924805100001 32.170492249,47.490413452 32.156539612,47.4913436280001 32.1486331180001,47.4981649170001 32.1441372680001,47.5138228760001 32.1403132120001,47.5266903080001 32.13338857,47.5347518310001 32.1231566370001,47.5436401770001 32.114578349,47.5587296960001 32.1129763800001,47.5780566820001 32.10615509,47.5802323560001 32.1034951960001,47.5950065520001 32.085432842,47.6177958580001 32.0418696090001,47.6331437580001 32.0269351200001,47.6680770260001 32.0124140430001,47.6779472250001 31.994689026,47.6779472250001 31.9945339970001,47.6826497800001 31.9769123340001,47.7184098720001 31.9223420210001,47.7252311610001 31.914332174,47.7433179120001 31.9042552690001,47.750966024 31.898209127,47.7565470790001 31.8891657510001,47.7617147220001 31.871647441,47.7652803950001 31.8641026820001,47.7813517660001 31.848858134,47.8207808840001 31.8235883590001,47.8344751380001 31.8057083130001,47.8372656660001 31.7844693,47.8313228760001 31.7618350220001,47.6789290770001 31.4078514610001,47.676443174 31.236517019,47.6729346110001 30.9946983850001,48.0015450440001 30.994646708,48.0122420660001 30.989065653,48.0154460040001 30.9762498990001,48.012035359 30.494573873,48.0135856530001 30.463878072,48.1194189860001 30.4508039350001,48.1305810960001 30.4474966430001,48.1409163820001 30.4419155890001,48.157659546 30.426309306,48.1704753010001 30.4066722610001,48.179467 30.3848648070001,48.1876835530001 30.3408364870001,48.1920760500001 30.3312763470001,48.2000858970001 30.3239382930001,48.2111963300001 30.3196491500001,48.222978556 30.318098857,48.235070841 30.318512268,48.2716577560001 30.3238349410001,48.2842668050001 30.3233181770001,48.2963590900001 30.3197525020001,48.305712525 30.3128278610001,48.3260213620001 30.283475647,48.3581641030001 30.2517979950001,48.3970247810001 30.220998841,48.4035876880001 30.212523906,48.4087036540001 30.2023436490001,48.4106156820001 30.191543275,48.4080318600001 30.180897929,48.4007454840001 30.1722679650001,48.3911336670001 30.1647748820001,48.3832788490001 30.1564549770001,48.38110844 30.1453445440001,48.383072143 30.1384199010001,48.3955778410001 30.1152171840001,48.4155249430001 30.0956318160001,48.4212093510001 30.0853482060001,48.4239482020001 30.0836428840001,48.4421899820001 30.033930156,48.4442570400001 30.0209076950001,48.4531453860001 30.0014256790001,48.4577962650001 29.9948627720001,48.4643074960001 29.989126689,48.4781567790001 29.979618225,48.4930395910001 29.9717634080001,48.5239937750001 29.964115295,48.531016472 29.9613304710001,48.5307723320001 29.9607608090001,48.5307723320001 29.9562035180001,48.5549422540001 29.9564883480001,48.559255405 29.9466006530001,48.5468856130001 29.9347191430001,48.520518425 29.928900458,48.411875847 29.938421942,48.3354598320001 29.961411851,48.2995711600001 29.9845238300001,48.26465905 29.9939639340001,48.2092391290001 30.0244815120001,48.165782097 30.037543036,48.1191512380001 30.0449893250001,48.0763452480001 30.0456403670001,48.038259311 30.036851304,47.9690047540001 30.0040550800001,47.961924675 30.0303408870001,47.9585067070001 30.0606143250001,47.9516707690001 30.088324286,47.9348250660001 30.1070824240001,47.9471134770001 30.0745303410001,47.9484969410001 30.0623233090001,47.9460555350001 30.0493838560001,47.9417423840001 30.0372582050001,47.9409285820001 30.0265160180001,47.9484969410001 30.0176455750001,47.9480086600001 29.994045315,47.7314323330001 30.088552145,47.674174845 30.0982156370001,47.4157926840001 30.0982156370001,47.3582251380001 30.0921178180001,47.1970980230001 30.0342402140001,47.1446981200001 30.0033377070001,47.1104883220001 29.9609113570001,47.0254289140001 29.7721373500001,46.988842 29.7126577760001,46.983674357 29.6982917280001,46.9798503010001 29.668061015,46.9773698320001 29.6579841110001,46.9578361410001 29.6204411830001,46.8831120200001 29.512514954,46.8534497480001 29.4445604460001,46.8385669350001 29.4249750780001,46.7744881590001 29.3635317990001,46.7118563230001 29.27139272,46.5614681480001 29.1241665090001,46.53243575 29.0957445270001,46.4886141360001 29.0875796520001,46.4448958740001 29.079414775,46.4273223200001 29.0761431030001,46.4011776130001 29.0712757370001,46.3574593500001 29.063136699,46.2521427820001 29.0714566040001,46.1759200440001 29.077476909,46.099697307 29.0835747280001,46.0236295980001 29.0895950320001,45.9473551840001 29.0956411740001,45.8449324950001 29.1037543740001,45.7424064530001 29.1118675750001,45.6398804120001 29.1200324500001,45.5374060470001 29.1281198120001,45.4349316810001 29.1362071740001,45.3324056400001 29.1443720500001,45.2299829510001 29.152459412,45.1274569090001 29.160624288,45.047823527 29.1669029750001,44.968293498 29.1732591760001,44.888711792 29.1795120240001,44.8089233810001 29.1858940640001,44.7174560950001 29.193102926,44.7108931880001 29.195273336,44.7046403410001 29.1974954220001,44.6982324620001 29.19971751,44.6918245850001 29.201836243,44.6148783780001 29.2560706590001,44.5197420650001 29.3233275350001,44.4242956950001 29.3905327360001,44.3289526770001 29.4577896130001,44.2336613360001 29.524994812,44.119146362 29.6058167520001,44.0046313880001 29.6866128540001,43.8900647380001 29.7673831180001,43.8240129630001 29.8139798590001,43.7754980880001 29.848205058,43.660983114 29.929026998,43.5463647870001 30.009797262,43.4317464600001 30.0905675260001,43.3172314860001 30.171337789,43.2290714930001 30.233504537,43.1409631750001 30.2956712850001,43.052906535 30.3577863570001,42.9647465410001 30.420004782,42.85901656 30.4945221960001,42.8589132080001 30.494573873,42.8588098550001 30.4946255490001,42.8587065020001 30.4946772260001,42.7834125790001 30.5510450320001,42.7639319260001 30.5656289670001,42.669157349 30.6366840620001,42.5744861250001 30.7076358030001,42.4797115480001 30.7786392220001,42.3951689050001 30.8415811160001,42.3107296140001 30.9045230100001,42.226290324 30.9674649050001,42.141644328 31.0304068000001,42.075395142 31.0798611450001,41.9862016200001 31.125284729,41.8985583900001 31.1699848430001,41.8107084550001 31.2146849580001,41.7230652260001 31.2593850710001,41.6354219970001 31.3040335080001,41.5475720620001 31.3487336230001,41.4598254800001 31.3934337360001,41.3721822510001 31.43813385,41.284539022 31.4828339640001,41.1966890870001 31.527585755,41.1090458580001 31.5722858680001,41.021299275 31.6169343060001,40.9335526940001 31.6616344200001,40.8459094650001 31.7063345340001,40.7581628820001 31.7510863240001,40.6703129470001 31.7956830860001,40.5826697190001 31.840434876,40.4798336180001 31.8928347780001,40.4241264240001 31.9205333450001,40.3702795820001 31.9384650680001,40.0293184820001 31.9943789670001,40.029111776 31.9944306440001,40.0290084230001 31.9944823200001,40.0288017170001 31.9944823200001,39.948599894 32.0061095170001,39.7509892170001 32.0348932910001,39.5532751870001 32.06372874,39.3555611570001 32.092564189,39.1577437740001 32.1213479620001,39.154953247 32.120572815,39.152162719 32.119745992,39.1492688400001 32.118919169,39.1463749600001 32.1181440230001,39.146168253 32.125843811,39.266470988 32.212866923,39.2919991450001 32.244518738,39.2711218670001 32.3119564820001,39.2563424070001 32.3426781210001,39.2357751870001 32.3528583780001,39.0463293870001 32.308494161,39.0362008060001 32.313351745,39.0287593990001 32.3283379110001,38.9799768480001 32.4721017460001,38.97863326 32.473729554,38.9782198490001 32.4749697880001,38.97863326 32.475693258,38.9799768480001 32.476054993,39.057181438 32.496596374,38.9900020750001 32.705575867,38.9427698160001 32.8523369340001,38.8971912030001 32.9943437700001,38.8625679930001 33.1007197060001,38.8210201420001 33.229032288,38.774511352 33.3716850790001,38.8850989180001 33.4271080530001,38.995686483 33.4824793500001,39.1062740480001 33.5379281610001,39.216861613 33.593325297,39.3274491780001 33.6487224320001,39.4380367430001 33.7040937300001,39.5486243080001 33.7594908650001,39.6592118740001 33.8148104860001,39.7696960850001 33.8702334600001,39.8803870030001 33.9256047570001,39.9908712160001 33.980976054,40.1014587810001 34.0363731890001,40.173110825 34.072282917,40.2120463460001 34.091796163,40.3226339110001 34.14716746,40.4332214760001 34.2025387570001,40.5438090410001 34.2579875700001,40.690466756 34.3314972940001,40.936033163 34.3860676070001,40.9652820230001 34.4018547570001,40.9880196530001 34.428519796,41.023986451 34.4941747030001,41.023986451 34.4943297330001,41.1956555580001 34.7684732060001,41.2042338460001 34.793122864,41.2065076090001 34.8193228150001,41.198032674 34.994040833,41.1923258570001 35.1589044370001,41.1915214440001 35.1821430460001,41.201339966 35.2430178840001,41.2430945230001 35.366524557,41.2518795170001 35.464089661,41.2610779210001 35.4941653440001,41.2611812740001 35.4941653440001,41.2612846270001 35.4941653440001,41.2612846270001 35.494320374,41.3084578340001 35.5522476250001,41.3422099210001 35.593694153,41.3580229090001 35.6239248660001,41.3635006100001 35.655240784,41.3592631430001 35.792751771,41.354508911 35.825566304,41.3436568610001 35.85765737,41.2663489180001 35.9942381800001,41.2663489180001 35.9943415320001,41.2662455650001 35.9942898560001,41.2406140540001 36.043020732,41.2366866450001 36.060332337,41.2365832930001 36.0770238240001,41.2688293870001 36.32796458,41.2769942620001 36.3547846480001,41.3652576090001 36.493897603,41.3652576090001 36.4940009560001,41.3653609620001 36.4940526330001,41.3653609620001 36.4941559860001,41.3854114170001 36.516376851,41.4148669840001 36.5273839320001,41.4797725830001 36.5361172490001,41.7899345300001 36.589292297,41.8173230390001 36.5997309370001,41.8437813730001 36.6178693640001,41.9785535070001 36.7336245730001,42.1784379480001 36.90531952,42.2815841070001 36.9939704390001,42.281894165 36.9940221150001,42.281894165 36.994125468,42.3458695880001 37.0429080200001,42.3768057240001 37.0620006190001,42.377185506 37.0622350060001,42.3768754480001 37.0767560830001,42.3711910400001 37.087944031,42.3636462810001 37.0981501270001,42.3572384030001 37.1099840290001,42.4018868410001 37.114143982,42.4591443280001 37.1293110150001,42.5452372640001 37.140886536,42.5612569580001 37.1466226200001,42.564667602 37.1520486450001,42.5769665940001 37.179230449,42.7023336190001 37.325345561,42.7061576740001 37.3332262170001,42.7073979090001 37.340150859,42.7094649650001 37.3471788540001,42.7156661380001 37.3552920530001,42.7221773680001 37.358909404,42.7715800370001 37.3749032600001,42.7804683840001 37.3754975380001,42.7924573160001 37.3743348190001,42.801138956 37.3690896610001,42.8054797770001 37.35185557,42.814058065 37.3468171190001,42.8967403570001 37.3249063110001,42.9370479730001 37.3201520800001,42.9796293540001 37.3318309540001,43.005157512 37.3472563680001,43.0435014240001 37.360252991,43.0837056890001 37.368831279,43.114814901 37.3711308800001,43.1319714760001 37.367255147,43.2633846430001 37.3106952930001,43.2703092850001 37.3086799110001,43.2786291910001 37.307749736,43.2875175380001 37.309144999,43.2968709720001 37.316741435,43.3056559650001 37.3199712120001,43.3241561280001 37.3222191370001,43.3361450610001 37.3202295940001,43.3625052150001 37.3039081520001,43.3760392660001 37.2955282600001,43.4167602950001 37.2791726680001,43.463269084 37.248683574,43.4795988370001 37.2433609010001,43.4924145910001 37.244756165,43.5171159260001 37.2523009240001,43.52951827 37.2539028930001,43.542437378 37.2523009240001,43.5506022550001 37.24873525,43.5688957120001 37.2377023320001,43.5942171630001 37.2294599410001,43.618298381 37.226979472,43.7208244230001 37.2326122030001,43.7469210210001 37.230648499,43.770640503 37.2258425900001,43.7805623780001 37.220364889,43.8020597740001 37.203570048,43.8093978270001 37.1996943150001,43.8220068770001 37.202381491,43.8399902760001 37.217109273,43.8936304120001 37.224912414,43.9241711830001 37.2530760700001,43.9537817790001 37.287466736,43.9903686930001 37.312503968,44.0355855710001 37.3182400520001,44.0690247360001 37.3137265880001,44.0880371500001 37.3111603800001,44.1844653730001 37.2791726680001,44.206892945 37.267519633,44.2231710210001 37.254057923,44.2350049230001 37.2367721560001,44.243531535 37.21382782,44.2485441480001 37.1919170130001,44.2496293540001 37.1794371550001,44.2483374430001 37.1696703090001,44.2401208900001 37.1579655970001,44.2301990160001 37.1543224080001,44.2187785240001 37.1526170860001,44.2059627690001 37.1466226200001,44.1892196040001 37.129207662,44.180641317 37.108924662,44.1800212000001 37.087608134,44.1874626060001 37.0669375610001,44.2277702230001 36.994125468,44.234281454 36.9836609910001,44.2432731530001 36.9777698780001,44.284821004 36.9691657520001,44.2972233480001 36.9699408980001,44.3066284590001 36.977227275,44.3158785400001 36.9939704390001,44.3159818930001 36.9939704390001,44.3161369220001 36.9940221150001,44.3161369220001 36.994125468,44.3316398520001 37.015467835,44.3351538490001 37.031151632,44.3431120200001 37.0424429330001,44.4280680740001 37.064767151,44.4544230550001 37.0763426720001,44.4792277430001 37.092000631,44.5034123130001 37.116624451,44.5393791100001 37.1436770630001,44.5782397870001 37.166414694,44.6102791750001 37.1783519490001,44.6289343670001 37.1790237430001,44.7337858480001 37.1672415170001,44.7538363030001 37.1591541550001,44.7661352950001 37.1419200650001)))"),
		geo.MustParseGeometry("MULTIPOLYGON(((-71.7574356759999 19.7101097680001,-71.7382706369999 19.7061221370001,-71.7203263009999 19.697455145,-71.7241104809999 19.7171898460001,-71.7418513659999 19.755804755,-71.7407934239999 19.7656924500001,-71.7407934239999 19.773179429,-71.7676488919999 19.7744815120001,-71.771066861 19.7851423200001,-71.7602432929999 19.8009300800001,-71.7444962229999 19.8172061220001,-71.7399796209999 19.82025788,-71.7348526679999 19.821275132,-71.7237035799999 19.820990302,-71.7169083319999 19.8236351580001,-71.7057999339999 19.8364932310001,-71.6998184889999 19.841498114,-71.6878962879999 19.8460147160001,-71.6790258449999 19.848089911,-71.6702774729999 19.851996161,-71.6588028639999 19.8619652360001,-71.6656388009999 19.8744977890001,-71.6667374339999 19.885687567,-71.6614477199999 19.8936221370001,-71.6485896479999 19.896714585,-71.6408585279999 19.8955752620001,-71.6288956369999 19.8904483090001,-71.6212458979999 19.8893089860001,-71.6179093089999 19.8914248720001,-71.6109513009999 19.900824286,-71.6073298819999 19.9028994810001,-71.5046280589999 19.91034577,-71.4785050119999 19.9068871110001,-71.452707486 19.8980980490001,-71.3572485019999 19.851263739,-71.3292130199999 19.845892645,-71.3026016919999 19.85569896,-71.2824600899999 19.8417829450001,-71.2680557929999 19.8355166690001,-71.2237442699999 19.8340518250001,-71.2120662099999 19.837103583,-71.2093399729999 19.8447940120001,-71.2088110019999 19.8549665390001,-71.2035212879999 19.8653832050001,-71.1917618479999 19.8696963560001,-71.1833389959999 19.8604190120001,-71.1722305979999 19.8340518250001,-71.1644180979999 19.8386091170001,-71.158518033 19.843491929,-71.1544083319999 19.8491071640001,-71.1517227859999 19.85569896,-71.1602270169999 19.8582217470001,-71.1620173819999 19.8594424500001,-71.1621801419999 19.8620466170001,-71.1653946609999 19.8688011740001,-71.1497289699999 19.863023179,-71.1365860669999 19.863714911,-71.1231990229999 19.866888739,-71.1070450509999 19.8688011740001,-71.0967911449999 19.876206773,-71.0782771479999 19.909125067,-71.0660701159999 19.9165713560001,-71.0508520169999 19.9202334660001,-71.0184220039999 19.935492255,-71.0008845689999 19.93768952,-70.9847305979999 19.9337425800001,-70.9693497389999 19.9251976580001,-70.9610489569999 19.9126651060001,-70.9661759109999 19.896714585,-70.9468888009999 19.8893089860001,-70.9394425119999 19.8893089860001,-70.9444473949999 19.904730536,-70.9468888009999 19.91034577,-70.8985082669999 19.9028994810001,-70.8842667309999 19.9042829450001,-70.8547257149999 19.910834052,-70.8432511059999 19.91034577,-70.8309626939999 19.90253327,-70.7869766919999 19.8528506530001,-70.7833145819999 19.8433291690001,-70.788644986 19.8340518250001,-70.7509659499999 19.8338076840001,-70.7365616529999 19.827093817,-70.6984757149999 19.79682038,-70.6622208319999 19.7793643250001,-70.6236873039999 19.766546942,-70.5790909499999 19.760199286,-70.5187068349999 19.7597516950001,-70.5136612619999 19.762640692,-70.5085343089999 19.7693545590001,-70.4964493479999 19.7772484400001,-70.4827774729999 19.783880927,-70.4727270169999 19.786851304,-70.4337052069999 19.7752139340001,-70.3727107409999 19.7141787780001,-70.3361710279999 19.6776390650001,-70.3195694649999 19.666449286,-70.3013402989999 19.6576602230001,-70.280832486 19.6518008480001,-70.2352188789999 19.6472028670001,-70.1985570949999 19.6346703150001,-70.1314184239999 19.6221377620001,-70.1164444649999 19.6223819030001,-70.1064346999999 19.625921942,-70.0975235669999 19.6332054710001,-70.0610245429999 19.670111395,-70.0421036449999 19.6784528670001,-69.9866837229999 19.6776390650001,-69.9595434239999 19.680121161,-69.9472550119999 19.678656317,-69.9320369129999 19.670152085,-69.8979386059999 19.635972398,-69.8871964179999 19.6127383480001,-69.8877660799999 19.5873477230001,-69.9041641919999 19.5323753930001,-69.8841446609999 19.524400132,-69.876535611 19.5066592470001,-69.8768204419999 19.4610049500001,-69.8743790359999 19.441148179,-69.867990689 19.4252790390001,-69.8013809889999 19.3308779970001,-69.7714737619999 19.3032494160001,-69.7676895819999 19.299750067,-69.7334692049999 19.2865664730001,-69.713205533 19.289740302,-69.6743871739999 19.3038597680001,-69.6543676419999 19.3070335960001,-69.6093643869999 19.30878327,-69.589426236 19.3128115910001,-69.5689591139999 19.3207054710001,-69.5394180979999 19.3394229190001,-69.5279841789999 19.3417829450001,-69.5176488919999 19.3396670590001,-69.4964493479999 19.329779364,-69.4833064439999 19.3275820980001,-69.4590958319999 19.3333194030001,-69.4485977859999 19.3343773460001,-69.4427791009999 19.334173895,-69.4373673169999 19.332953192,-69.4334203769999 19.3299014340001,-69.4317927729999 19.3241234400001,-69.4286189439999 19.3192406270001,-69.4217016269999 19.3211123720001,-69.4107152989999 19.3275820980001,-69.3526912099999 19.305121161,-69.3356013659999 19.293402411,-69.3242895169999 19.3150902360001,-69.3058975899999 19.3311221370001,-69.2604874339999 19.3548851580001,-69.231271939 19.3633487000001,-69.2243546209999 19.35179271,-69.2320857409999 19.3279483090001,-69.2468155589999 19.2996279970001,-69.226877408 19.292669989,-69.2043350899999 19.2927920590001,-69.182443814 19.2981631530001,-69.1642960279999 19.3070335960001,-69.1583552729999 19.28510163,-69.1713761059999 19.2678897160001,-69.1893204419999 19.2522647160001,-69.1983943349999 19.235052802,-69.202707486 19.2169457050001,-69.2143448559999 19.2006289730001,-69.2318009109999 19.188666083,-69.2536514959999 19.18353913,-69.3321427069999 19.197211005,-69.4181208979999 19.1909854190001,-69.4383845689999 19.196478583,-69.4754532539999 19.2140160180001,-69.6029353509999 19.228989976,-69.6172989569999 19.2252464860001,-69.6227921209999 19.211859442,-69.6260473299999 19.16429271,-69.6335343089999 19.1236839860001,-69.630441861 19.1084658870001,-69.6208796869999 19.1019554710001,-69.6084692049999 19.093451239,-69.5532934239999 19.1014671900001,-69.5342504549999 19.0947940120001,-69.5104874339999 19.102036851,-69.4634496739999 19.0847028670001,-69.4386287099999 19.0873477230001,-69.4447322259999 19.0923526060001,-69.4527481759999 19.096909898,-69.4623103509999 19.100287177,-69.4733780589999 19.1016299500001,-69.4733780589999 19.1084658870001,-69.4199926419999 19.1111514340001,-69.4044490229999 19.1084658870001,-69.3867081369999 19.0960960960001,-69.3729548819999 19.065375067,-69.3560684889999 19.053208726,-69.3389786449999 19.0488955750001,-69.3045955069999 19.044867255,-69.211740689 19.0209821640001,-69.1732478509999 19.0111351580001,-69.1574600899999 19.012193101,-69.1386612619999 19.0222842470001,-69.1448868479999 19.0296084660001,-69.1573380199999 19.0366885440001,-69.1574600899999 19.0463727890001,-69.1395157539999 19.051214911,-69.1238907539999 19.037787177,-69.1028539699999 19.0053571640001,-69.0892634759999 18.9978701840001,-69.0741267569999 18.9935570330001,-69.0611873039999 18.9881859400001,-69.0544327459999 18.977484442,-69.0424698559999 18.986761786,-69.0180557929999 18.9974632830001,-69.0065811839999 19.0053571640001,-69.0142309239999 19.013413804,-69.010568814 19.015366929,-69.0015763009999 19.0156110700001,-68.992990689 19.0184593770001,-68.9825333319999 19.0287946640001,-68.9793595039999 19.0328636740001,-68.975412564 19.033148505,-68.9395645819999 19.0313174500001,-68.9186091789999 19.02728913,-68.8998103509999 19.0209007830001,-68.883127408 19.012193101,-68.8812149729999 19.0094261740001,-68.881174283 19.005804755,-68.8803604809999 19.0020205750001,-68.8762914699999 18.9986026060001,-68.8558243479999 18.9917666690001,-68.8539932929999 18.9920921900001,-68.8182266919999 18.9849307310001,-68.7958878249999 18.984116929,-68.7868546209999 18.9810244810001,-68.7851456369999 18.9798851580001,-68.7769669259999 18.974351304,-68.7678116529999 18.9694684920001,-68.7499893869999 18.9680850280001,-68.7397354809999 18.9644229190001,-68.6235245429999 18.8618838560001,-68.5807999339999 18.8125674500001,-68.5204971999999 18.7688662780001,-68.4772843089999 18.7402204450001,-68.4649958979999 18.7350121110001,-68.458566861 18.7310244810001,-68.4134008449999 18.6896019550001,-68.3561905589999 18.6565615910001,-68.3286026679999 18.6165225280001,-68.334055142 18.5772972680001,-68.428049283 18.4412295590001,-68.4346817699999 18.435777085,-68.4412735669999 18.4290225280001,-68.4443253249999 18.4195010440001,-68.4424942699999 18.3847516950001,-68.4443253249999 18.3748233090001,-68.4556778639999 18.357855536,-68.4721573559999 18.349839585,-68.4928279289999 18.3474795590001,-68.5165909499999 18.3474795590001,-68.5349014959999 18.3533389340001,-68.574533658 18.3781598980001,-68.5889379549999 18.3822695980001,-68.6054581369999 18.3717308610001,-68.6028539699999 18.35565827,-68.5975235669999 18.336615302,-68.6060277989999 18.317084052,-68.6183162099999 18.301459052,-68.6260880199999 18.2833519550001,-68.6415095689999 18.2251651060001,-68.6470841139999 18.2148298200001,-68.6584366529999 18.2109235700001,-68.742990689 18.2045352230001,-68.7602432929999 18.2103539080001,-68.762521939 18.2469750020001,-68.7868546209999 18.295396226,-68.8201391269999 18.342474677,-68.8495987619999 18.3748233090001,-68.8809708319999 18.395493882,-68.9225154289999 18.412298895,-68.9643448559999 18.4179548200001,-69.0125626289999 18.399115302,-69.0854386059999 18.3959007830001,-69.1043595039999 18.399115302,-69.1481827459999 18.4132347680001,-69.1680395169999 18.4163272160001,-69.1852921209999 18.4216983090001,-69.2142227859999 18.4451358090001,-69.2363175119999 18.4505069030001,-69.2556046209999 18.4466820330001,-69.2752172519999 18.4403343770001,-69.2950333319999 18.4390322940001,-69.3150935539999 18.4505069030001,-69.3360082669999 18.432074286,-69.4659317699999 18.422552802,-69.5047501289999 18.4089216170001,-69.5252579419999 18.408433335,-69.5799861319999 18.4444847680001,-69.6021215489999 18.4557152360001,-69.6168106759999 18.4573428410001,-69.6231583319999 18.4479841170001,-69.622629361 18.4367536480001,-69.6208389959999 18.425482489,-69.6236059239999 18.4163272160001,-69.6325577459999 18.4135195980001,-69.6465958319999 18.4135602890001,-69.6594945949999 18.4159203150001,-69.6651505199999 18.4195010440001,-69.681955533 18.4467634140001,-69.6899307929999 18.454087632,-69.6973770819999 18.4563662780001,-69.8514705069999 18.4724795590001,-69.8793839179999 18.4712181660001,-69.8836563789999 18.4710147160001,-69.8998103509999 18.4664574240001,-69.931223111 18.4519717470001,-69.9571020169999 18.435288804,-69.9669083319999 18.4308128930001,-69.9935196609999 18.422552802,-70.0040177069999 18.4174665390001,-70.0165909499999 18.4113630230001,-70.0389298169999 18.3924828150001,-70.0557755199999 18.3672549500001,-70.0624080069999 18.3372256530001,-70.0676977199999 18.329087632,-70.0906876289999 18.3175316430001,-70.0959366529999 18.310248114,-70.1000056629999 18.2992210960001,-70.1337784499999 18.269273179,-70.1580704419999 18.242621161,-70.1717830069999 18.231431382,-70.1778865229999 18.2341983090001,-70.2021378249999 18.2335472680001,-70.2207738919999 18.2305362000001,-70.2331436839999 18.2239444030001,-70.250803189 18.23432038,-70.2696020169999 18.2381045590001,-70.3975723949999 18.2376162780001,-70.4190567699999 18.231146552,-70.4601944649999 18.208929755,-70.4835505849999 18.204087632,-70.5528458319999 18.2042503930001,-70.5689184239999 18.2103539080001,-70.5703018869999 18.21893952,-70.5630590489999 18.2256533870001,-70.5537003249999 18.2315127620001,-70.5490616529999 18.2376162780001,-70.5519099599999 18.2484398460001,-70.5587052069999 18.254380601,-70.5657445949999 18.2582054710001,-70.5733129549999 18.268011786,-70.5924373039999 18.2798526060001,-70.5967911449999 18.2891299500001,-70.5943497389999 18.299058335,-70.5884496739999 18.3036156270001,-70.5816544259999 18.307074286,-70.5764054029999 18.3139509140001,-70.5717667309999 18.3340518250001,-70.5711563789999 18.355169989,-70.5738826159999 18.3736839860001,-70.5794571609999 18.3856468770001,-70.6014298169999 18.411851304,-70.6146541009999 18.420843817,-70.6378067699999 18.429388739,-70.6575414699999 18.43378327,-70.6824845039999 18.4356143250001,-70.7038468089999 18.430121161,-70.7128800119999 18.4126651060001,-70.7094620429999 18.3917503930001,-70.7119848299999 18.3833682310001,-70.7416886059999 18.35541413,-70.7498673169999 18.3504092470001,-70.7606501939999 18.3474795590001,-70.7723282539999 18.34788646,-70.7931208979999 18.3540713560001,-70.802316861 18.3549258480001,-70.8100479809999 18.352036851,-70.8239639959999 18.3426781270001,-70.8333227199999 18.3406436220001,-70.846018033 18.3421898460001,-70.8504532539999 18.339585679,-70.8603409499999 18.3307152360001,-70.8631078769999 18.3269717470001,-70.8693741529999 18.3146019550001,-70.8705948559999 18.310248114,-70.8732804029999 18.3063011740001,-70.8867081369999 18.3020694030001,-70.8916723299999 18.2997093770001,-70.918283658 18.2695987000001,-70.9324845039999 18.2615420590001,-70.9562068349999 18.258734442,-70.9658096999999 18.262152411,-70.9720352859999 18.2702497420001,-70.9772843089999 18.280259507,-70.9837947259999 18.2891299500001,-70.9942520819999 18.296616929,-71.0048721999999 18.30101146,-71.0482478509999 18.3100039730001,-71.0683487619999 18.3103701840001,-71.0839737619999 18.304388739,-71.0902400379999 18.2891299500001,-71.0923559239999 18.2807070980001,-71.1012263659999 18.266099351,-71.1033422519999 18.2549502620001,-71.1014298169999 18.24213288,-71.0965470039999 18.2350934920001,-71.0899145169999 18.228705145,-71.082834439 18.217718817,-71.0665177069999 18.1633161480001,-71.0612686839999 18.153998114,-71.0611059239999 18.1420759140001,-71.0805557929999 18.1139183610001,-71.0971573559999 18.0737165390001,-71.1918025379999 17.9417992210001,-71.2001033189999 17.9191755230001,-71.2042537099999 17.911118882,-71.2137345039999 17.9048526060001,-71.2336319649999 17.894964911,-71.2459203769999 17.8850772160001,-71.254709439 17.8753115910001,-71.2615453769999 17.86322663,-71.2678116529999 17.8465843770001,-71.2752172519999 17.8490664730001,-71.2780655589999 17.850775458,-71.2820938789999 17.853989976,-71.2835994129999 17.8467471370001,-71.2851456369999 17.844183661,-71.2889298169999 17.841009833,-71.2820938789999 17.841009833,-71.2861221999999 17.8325869810001,-71.291330533 17.8261579450001,-71.2977188789999 17.821966864,-71.3056534499999 17.82050202,-71.3093155589999 17.8163923200001,-71.3158259759999 17.7964541690001,-71.3196508449999 17.7892113300001,-71.3318578769999 17.772406317,-71.365549283 17.678412177,-71.3758845689999 17.6601423200001,-71.4172257149999 17.6048037780001,-71.4272354809999 17.6092796900001,-71.435861783 17.6170921900001,-71.436350064 17.62128327,-71.446115689 17.629339911,-71.5148819649999 17.7377790390001,-71.5316055979999 17.755072333,-71.565256314 17.767523505,-71.6451309889999 17.7572695980001,-71.6793106759999 17.7652855490001,-71.6696671209999 17.772406317,-71.6377660799999 17.8040225280001,-71.6309301419999 17.8167992210001,-71.632394986 17.838324286,-71.6365860669999 17.853461005,-71.6520076159999 17.8819847680001,-71.6628311839999 17.8976911480001,-71.6649470689999 17.9034691430001,-71.6656388009999 17.9154727230001,-71.6588028639999 17.95331452,-71.6662491529999 17.9707705750001,-71.6838272779999 17.990179755,-71.7203263009999 18.0191104190001,-71.7762345039999 18.0392520200001,-71.7642712 18.0694924930001,-71.7606538499999 18.0868040970001,-71.7628759359999 18.1158462530001,-71.7622041419999 18.1324860640001,-71.7645812579999 18.143674011,-71.775329956 18.1723285930001,-71.7768285729999 18.1817853800001,-71.7740380459999 18.201474101,-71.7661315519999 18.220258484,-71.721708773 18.2929962370001,-71.7211117359998 18.2939738230001,-71.7093391519999 18.313250224,-71.7115095629999 18.3161441050001,-71.7207079679999 18.323973083,-71.7349706629999 18.3329389450001,-71.7887141529999 18.352214254,-71.8264379479999 18.376166281,-71.8345511479999 18.385080465,-71.8497440189999 18.4064486700001,-71.8581672779999 18.4135541790001,-71.8971829839999 18.4228817750001,-71.9121691489999 18.430736593,-71.9181636159999 18.4492367550001,-71.9143912359999 18.460217997,-71.9068981529999 18.464300436,-71.9015754799999 18.4697781380001,-71.90405595 18.4847901410001,-71.9105155029999 18.494272766,-72.0009492599999 18.5824585980001,-72.0098376059999 18.598814189,-71.9926293539999 18.6111390180001,-71.9554739999999 18.6186062620001,-71.880284791 18.605402934,-71.841579142 18.6179861450001,-71.8285566809999 18.631292827,-71.8079894609999 18.6646499640001,-71.7943468839999 18.679171041,-71.7917791329999 18.6806816980001,-71.7848976379999 18.6847302130001,-71.7580700279999 18.700513407,-71.7440657149999 18.7114429730001,-71.7320767829999 18.7300723270001,-71.7252554929999 18.7469705200001,-71.7203462329999 18.7654448450001,-71.7188476159999 18.7842292280001,-71.7269091399999 18.8235291550001,-71.7279426689999 18.8640693160001,-71.7334720469999 18.882569479,-71.7409134529999 18.8916128540001,-71.7638061119999 18.9115082810001,-71.7729528399999 18.9217143760001,-71.7871121829999 18.950033061,-71.796258911 18.9570610560001,-71.8196683349999 18.9578103640001,-71.8445246989999 18.949748841,-71.862456422 18.947087504,-71.8654019779999 18.9644249480001,-71.848142049 18.975483704,-71.7968273519999 18.9886095180001,-71.7837015379999 18.996309306,-71.7403450119999 19.0418879190001,-71.7104243569999 19.0816529340001,-71.6952831619999 19.094546204,-71.6613834229999 19.1175938930001,-71.6487743739999 19.1352930710001,-71.6436584069999 19.152475484,-71.6391108809999 19.212110088,-71.6517327169999 19.217923884,-71.7165221769999 19.247766825,-71.7495170769999 19.2797077230001,-71.7537292069999 19.283785299,-71.771557577 19.3075564570001,-71.7765185149999 19.3275035610001,-71.7699556079999 19.333808085,-71.742980509 19.3483808390001,-71.7334720469999 19.3554605110001,-71.720811321 19.3859496060001,-71.715281943 19.391659851,-71.7038614509999 19.4146300250001,-71.703189657 19.4592784630001,-71.7151269129999 19.537464905,-71.7430838629999 19.6000709030001,-71.747683065 19.622782695,-71.7460414089999 19.649017002,-71.7450475669999 19.6648989870001,-71.7486132409999 19.6825981650001,-71.7574356759999 19.7101097680001)),((-71.5244848299999 17.5455589860001,-71.5411677729999 17.55463288,-71.5413712229999 17.5761579450001,-71.5322159499999 17.601304429,-71.521066861 17.62128327,-71.5069473949999 17.6136742210001,-71.4925837879999 17.6036644550001,-71.467111783 17.580267645,-71.4825333319999 17.5756289730001,-71.507394986 17.551255601,-71.5244848299999 17.5455589860001)),((-68.6640518869999 18.1693382830001,-68.6123754549999 18.1708031270001,-68.5987849599999 18.166205145,-68.588693814 18.1533877620001,-68.5775040359999 18.133734442,-68.5701798169999 18.115708726,-68.571848111 18.107855536,-68.5870255199999 18.111070054,-68.6192927729999 18.1251488300001,-68.6360977859999 18.1283633480001,-68.6539200509999 18.1262881530001,-68.6848852199999 18.116888739,-68.7018936839999 18.1146914730001,-68.7370499339999 18.1236026060001,-68.7620743479999 18.1451683610001,-68.7781876289999 18.1720238300001,-68.7869360019999 18.1966820330001,-68.7720841139999 18.200873114,-68.7545466789999 18.1976585960001,-68.7371720039999 18.1895205750001,-68.7223201159999 18.1792666690001,-68.7107641269999 18.1753604190001,-68.6640518869999 18.1693382830001)))"),
	}
	benchLargePoints = []geo.Geometry{
		geo.MustParseGeometry("POINT (19.383354043259374 -29.13291298545164)"),
		geo.MustParseGeometry("POINT (15.547774209259558 45.487209288609961)"),
		geo.MustParseGeometry("POINT(44.6572635485688 33.7476416209197)"),
		geo.MustParseGeometry("POINT(-71.7069954585449 18.8759970016117)"),
	}
	benchSmallPolygons = []geo.Geometry{
		geo.MustParseGeometry("POLYGON EMPTY"),
		geo.MustParseGeometry("POLYGON((0 0, 1 0, 1 1, 0 1, 0 0))"),
		geo.MustParseGeometry("POLYGON((1 0, 0 1, 2 2, 1 0), (0.75 0.75, 1.25 0.75, 1.25 1.25, 0.75 1.25, 0.75 0.75))"),
		geo.MustParseGeometry("MULTIPOLYGON(((0 0, 1 0, 1 1, 0 1, 0 0)))"),
	}
	benchSmallPoints = []geo.Geometry{
		geo.MustParseGeometry("POINT EMPTY"),
		geo.MustParseGeometry("POINT(0 0)"),
		geo.MustParseGeometry("POINT(0.5 0.5)"),
		geo.MustParseGeometry("POINT(-1 -2)"),
		geo.MustParseGeometry("MULTIPOINT((0 0), (-1 -1), (0.5 0.25))"),
	}
)

var benchRes bool

func BenchmarkIntersectsLargePolygonsAndPoints(b *testing.B) {
	benchmarkIntersects(benchLargePoints, benchLargePolygons, b)
}

func BenchmarkIntersectsSmallPolygonsAndPoints(b *testing.B) {
	benchmarkIntersects(benchSmallPolygons, benchSmallPoints, b)
}

func benchmarkIntersects(geomListA []geo.Geometry, geomListB []geo.Geometry, b *testing.B) {
	var res bool
	var err error
	b.ResetTimer()
	for n := 0; n < b.N; n++ {
		for _, geomA := range geomListA {
			for _, geomB := range geomListB {
				res, err = Intersects(geomA, geomB)
				if err != nil {
					panic(err)
				}
			}
		}
	}
	benchRes = res
}
