// Copyright 2019 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package tests

import (
	"context"
	"fmt"
	"math"
	"os"
	"path/filepath"
	"strings"
	"time"

	"github.com/cockroachdb/cockroach/pkg/cmd/roachtest/cluster"
	"github.com/cockroachdb/cockroach/pkg/cmd/roachtest/option"
	"github.com/cockroachdb/cockroach/pkg/cmd/roachtest/registry"
	"github.com/cockroachdb/cockroach/pkg/cmd/roachtest/spec"
	"github.com/cockroachdb/cockroach/pkg/cmd/roachtest/test"
	"github.com/cockroachdb/cockroach/pkg/roachprod/install"
	"github.com/cockroachdb/cockroach/pkg/util/search"
	"github.com/cockroachdb/cockroach/pkg/workload/histogram"
	"github.com/cockroachdb/cockroach/pkg/workload/histogram/exporter"
	"github.com/cockroachdb/errors"
	"github.com/cockroachdb/ttycolor"
	"github.com/codahale/hdrhistogram"
)

// kvBenchKeyDistribution represents the distribution of keys generated by `workload`.
type kvBenchKeyDistribution int

const (
	sequential kvBenchKeyDistribution = iota
	random
	zipfian
)

type kvBenchSpec struct {
	Nodes int
	CPUs  int
	// Number of shards that the primary key `k` is sharded with. We manually pre-split
	// the `kv` table to create exactly as many splits as there are shards. This should,
	// in theory, split a sequential write load evenly across all the shards. Note that a
	// hash-sharded index combined with our load-based splitting mechanism should achieve
	// something similar to this, but that would take too long for it to be feasibly
	// relied upon for the purposes of this benchmark.
	NumShards       int
	KeyDistribution kvBenchKeyDistribution
	SecondaryIndex  bool

	EstimatedMaxThroughput int
	LatencyThresholdMs     float64
}

func registerKVBenchSpec(r registry.Registry, b kvBenchSpec) {
	nameParts := []string{
		"kv0bench",
		fmt.Sprintf("nodes=%d", b.Nodes),
		fmt.Sprintf("cpu=%d", b.CPUs),
	}
	if b.NumShards > 0 {
		nameParts = append(nameParts, fmt.Sprintf("shards=%d", b.NumShards))
	}
	opts := []spec.Option{spec.CPU(b.CPUs), spec.WorkloadNode(), spec.WorkloadNodeCPU(b.CPUs)}
	switch b.KeyDistribution {
	case sequential:
		nameParts = append(nameParts, "sequential")
	case random:
		nameParts = append(nameParts, "random")
	case zipfian:
		nameParts = append(nameParts, "zipfian")
	default:
		panic("unexpected")
	}

	if b.SecondaryIndex {
		nameParts = append(nameParts, "2nd_idx")
	}

	name := strings.Join(nameParts, "/")
	nodes := r.MakeClusterSpec(b.Nodes+1, opts...)
	r.Add(registry.TestSpec{
		Name: name,
		// These tests don't have pass/fail conditions so we don't want to run them
		// nightly. Currently they're only good for printing the results of a search
		// for --max-rate.
		// TODO(andrei): output something to roachperf and start running them
		// nightly.
		CompatibleClouds: registry.AllClouds,
		Suites:           registry.ManualOnly,
		Owner:            registry.OwnerKV,
		Benchmark:        true,
		Cluster:          nodes,
		Run: func(ctx context.Context, t test.Test, c cluster.Cluster) {
			runKVBench(ctx, t, c, b)
		},
	})
}

func registerKVBench(r registry.Registry) {
	specs := []kvBenchSpec{
		{
			Nodes:                  5,
			CPUs:                   8,
			KeyDistribution:        sequential,
			EstimatedMaxThroughput: 30000,
			LatencyThresholdMs:     10.0,
			SecondaryIndex:         false,
			NumShards:              10,
		},
		{
			Nodes:                  5,
			CPUs:                   8,
			KeyDistribution:        sequential,
			EstimatedMaxThroughput: 20000,
			LatencyThresholdMs:     10.0,
			SecondaryIndex:         false,
			NumShards:              0,
		},
		{
			Nodes:                  10,
			CPUs:                   8,
			KeyDistribution:        sequential,
			EstimatedMaxThroughput: 60000,
			LatencyThresholdMs:     10.0,
			SecondaryIndex:         false,
			NumShards:              20,
		},
		{
			Nodes:                  10,
			CPUs:                   8,
			KeyDistribution:        sequential,
			EstimatedMaxThroughput: 20000,
			LatencyThresholdMs:     10.0,
			SecondaryIndex:         false,
			NumShards:              0,
		},
		{
			Nodes:                  20,
			CPUs:                   8,
			KeyDistribution:        sequential,
			EstimatedMaxThroughput: 100000,
			LatencyThresholdMs:     10.0,
			SecondaryIndex:         false,
			NumShards:              80,
		},
		{
			Nodes:                  20,
			CPUs:                   8,
			KeyDistribution:        sequential,
			EstimatedMaxThroughput: 20000,
			LatencyThresholdMs:     10.0,
			SecondaryIndex:         false,
			NumShards:              0,
		},
		{
			Nodes:                  5,
			CPUs:                   8,
			KeyDistribution:        sequential,
			EstimatedMaxThroughput: 10000,
			LatencyThresholdMs:     10.0,
			SecondaryIndex:         true,
			NumShards:              10,
		},
		{
			Nodes:                  5,
			CPUs:                   8,
			KeyDistribution:        sequential,
			EstimatedMaxThroughput: 5000,
			LatencyThresholdMs:     10.0,
			SecondaryIndex:         true,
			NumShards:              0,
		},
	}

	for _, b := range specs {
		registerKVBenchSpec(r, b)
	}
}

// KVBench is a benchmarking tool that runs the `kv` workload against CockroachDB based on
// various configuration settings (see `kvBenchSpec`). The tool searches for the maximum
// throughput that can be sustained while maintaining an average latency below a certain
// threshold (as described in the configuration).
//
// This tool was primarily written with the objective of demonstrating the write
// performance characteristics of using hash sharded indexes, for sequential workloads
// which would've otherwise created a single-range hotspot.
func runKVBench(ctx context.Context, t test.Test, c cluster.Cluster, b kvBenchSpec) {
	const restartWait = 15 * time.Second
	// TODO(aayush): I do not have a good reasoning for why I chose this precision value.
	precision := int(math.Max(1.0, float64(b.EstimatedMaxThroughput/50)))
	initStepSize := 2 * precision
	// Search between 100 and 10000000 for the max throughput that can be maintained while
	// sustaining an avg latency of less than `LatencyThresholdMs`.
	// TODO(aayush): `avg` here is just an arbitrary statistic, should I make the
	// concerned statistic a config option?
	resultsDir, err := os.MkdirTemp("", "roachtest-kvbench")
	if err != nil {
		t.Fatal(errors.Wrapf(err, `failed to create temp results dir`))
	}
	s := search.NewLineSearcher(100 /* min */, 10000000 /* max */, b.EstimatedMaxThroughput, initStepSize, precision)
	searchPredicate := func(maxrate int) (bool, error) {
		m := c.NewMonitor(ctx, c.CRDBNodes())
		// Restart
		m.ExpectDeaths(int32(len(c.CRDBNodes())))
		// Wipe cluster before starting a new run because factors like load-based
		// splitting can significantly change the underlying layout of the table and
		// affect benchmark results.
		c.Wipe(ctx, c.CRDBNodes())
		c.Start(ctx, t.L(), option.NewStartOpts(option.NoBackupSchedule), install.MakeClusterSettings(), c.CRDBNodes())
		time.Sleep(restartWait)

		// We currently only support one loadGroup.
		resultChan := make(chan *kvBenchResult, 1)
		m.Go(func(ctx context.Context) error {
			db := c.Conn(ctx, t.L(), 1)

			var initCmd strings.Builder

			fmt.Fprintf(&initCmd, `./cockroach workload init kv --num-shards=%d`,
				b.NumShards)
			if b.SecondaryIndex {
				initCmd.WriteString(` --secondary-index`)
			}
			fmt.Fprintf(&initCmd, ` {pgurl%s}`, c.CRDBNodes())
			if err := c.RunE(ctx, option.WithNodes(c.WorkloadNode()), initCmd.String()); err != nil {
				return err
			}

			var splitCmd strings.Builder
			if b.NumShards > 0 {
				splitCmd.WriteString(`USE kv; ALTER TABLE kv SPLIT AT VALUES `)

				for i := 0; i < b.NumShards; i++ {
					if i != 0 {
						splitCmd.WriteString(`,`)
					}
					fmt.Fprintf(&splitCmd, `(%d)`, i)
				}
				splitCmd.WriteString(`;`)
			}
			if _, err := db.Exec(splitCmd.String()); err != nil {
				t.L().Printf(splitCmd.String())
				return err
			}

			workloadCmd := strings.Builder{}
			clusterHistPath := fmt.Sprintf("%s/kvbench/maxrate=%d/stats.json",
				t.PerfArtifactsDir(), maxrate)

			// The number of workers running on the loadGen node must be high enough to
			// fully saturate the loadGen node since the free variable here is the value
			// of the `--max-rate` flag passed to workload and not the level of
			// concurrency.
			//
			// To understand why this is needed, note that if this concurrency value were
			// too low, we would be searching indefinitely since we would never hit
			// throughput values that are greater than or equal to `maxrate`, and this
			// would lead to the `searchPredicate` always returning true as long as the
			// latency at this throughput ceiling was below the configured threshold.
			const loadConcurrency = 192
			const ramp = time.Second * 300
			const duration = time.Second * 300

			fmt.Fprintf(&workloadCmd,
				`./cockroach workload run kv --ramp=%fs --duration=%fs {pgurl%s} --read-percent=0`+
					` --concurrency=%d --histograms=%s --max-rate=%d --num-shards=%d`,
				ramp.Seconds(), duration.Seconds(), c.CRDBNodes(),
				b.CPUs*loadConcurrency, clusterHistPath, maxrate, b.NumShards)
			switch b.KeyDistribution {
			case sequential:
				workloadCmd.WriteString(` --sequential`)
			case zipfian:
				workloadCmd.WriteString(` --zipfian`)
			case random:
				workloadCmd.WriteString(` --random`)
			default:
				panic(`unexpected`)
			}

			err := c.RunE(ctx, option.WithNodes(c.WorkloadNode()), workloadCmd.String())
			if err != nil {
				return errors.Wrapf(err, `error running workload`)
			}

			localHistPath := filepath.Join(resultsDir, fmt.Sprintf(`kvbench-%d-stats.json`, maxrate))
			if err := c.Get(ctx, t.L(), clusterHistPath, localHistPath, c.WorkloadNode()); err != nil {
				t.Fatal(err)
			}

			snapshots, err := histogram.DecodeSnapshots(localHistPath)
			if err != nil {
				return errors.Wrapf(err, `failed to decode histogram snapshots`)
			}

			res := newResultFromSnapshots(maxrate, snapshots)
			resultChan <- res
			return nil
		})

		if err := m.WaitE(); err != nil {
			return false, err
		}
		close(resultChan)
		res := <-resultChan

		var color ttycolor.Code
		var msg string
		pass := res.latency() <= b.LatencyThresholdMs
		if pass {
			color = ttycolor.Green
			msg = "PASS"
		} else {
			color = ttycolor.Red
			msg = "FAIL"
		}
		ttycolor.Stdout(color)
		t.L().Printf(`--- SEARCH ITER %s: kv workload avg latency: %0.1fms (threshold: %0.1fms), avg throughput: %d`,
			msg, res.latency(), b.LatencyThresholdMs, res.throughput())
		ttycolor.Stdout(ttycolor.Reset)
		return pass, nil
	}
	if res, err := s.Search(searchPredicate); err != nil {
		t.Fatal(err)
	} else {
		ttycolor.Stdout(ttycolor.Green)
		t.L().Printf("-------\nMAX THROUGHPUT = %d\n--------\n\n", res)
		ttycolor.Stdout(ttycolor.Reset)
	}
}

type kvBenchResult struct {
	Cumulative map[string]*hdrhistogram.Histogram
	Elapsed    time.Duration
}

// TODO(aayush): The result related logic below is similar to `workload/tpcc/result.go`,
// so this could definitely be cleaner and better abstracted.
func newResultFromSnapshots(
	maxrate int, snapshots map[string][]exporter.SnapshotTick,
) *kvBenchResult {
	var start, end time.Time
	ret := make(map[string]*hdrhistogram.Histogram, len(snapshots))
	for n, snaps := range snapshots {
		var cur *hdrhistogram.Histogram
		for _, s := range snaps {
			h := hdrhistogram.Import(s.Hist)
			if cur == nil {
				cur = h
			} else {
				cur.Merge(h)
			}
			if start.IsZero() || s.Now.Before(start) {
				start = s.Now
			}
			if sEnd := s.Now.Add(s.Elapsed); end.IsZero() || sEnd.After(end) {
				end = sEnd
			}
		}
		ret[n] = cur
	}
	return &kvBenchResult{
		Cumulative: ret,
		Elapsed:    end.Sub(start),
	}
}

func (r kvBenchResult) latency() float64 {
	return time.Duration(r.Cumulative[`write`].Mean()).Seconds() * 1000
}

func (r kvBenchResult) throughput() int {
	// Currently the `kv` workload does not track histograms purely for throughput. We can
	// compute the average throughput here but not much more than that.
	return int(float64(r.Cumulative[`write`].TotalCount()) / r.Elapsed.Seconds())
}
