// Copyright 2018 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

// prereqs generates Make prerequisites for Go binaries. It works much like the
// traditional makedepend tool for C.
//
// Given the path to a Go package, prereqs will traverse the package's
// dependency graph to determine what files impact its compilation. It then
// outputs a Makefile that expresses these dependencies. For example:
//
//	$ prereqs ./pkg/cmd/foo
//	# Code generated by prereqs. DO NOT EDIT!
//
//	bin/foo: ./pkg/cmd/foo/foo.go ./some/dep.go ./some/other_dep.go
//
//	./pkg/cmd/foo/foo.go:
//	./some/dep.go:
//	./some/other_dep.go:
//
// The intended usage is automatic dependency generation from another Makefile:
//
//	bin/target:
//	 prereqs ./pkg/cmd/target > bin/target.d
//	 go build -o $@ ./pkg/cmd/target
//
//	include bin/target.d
//
// Notice that depended-upon files are mentioned not only in the prerequisites
// list but also as a rule with no prerequisite or recipe. This prevents Make
// from complaining if the prerequisite is deleted. See [0] for details on the
// approach.
//
// [0]: http://make.mad-scientist.net/papers/advanced-auto-dependency-generation/
package main

import (
	"flag"
	"fmt"
	"io"
	"os"
	"path/filepath"
	"sort"
	"strings"

	"golang.org/x/tools/go/packages"
)

func collectFiles(path string, includeTest bool, options testOptions) ([]string, error) {
	cwd, err := os.Getwd()
	if err != nil {
		return nil, err
	}

	// Symlinks in cwd confuse the relative path computation in collectFilesImpl.
	cwd, err = filepath.EvalSymlinks(cwd)
	if err != nil {
		return nil, err
	}

	seen := make(map[string]struct{})

	// Import the package.
	config := &packages.Config{
		Dir:   cwd,
		Mode:  packages.NeedName | packages.NeedFiles | packages.NeedDeps | packages.NeedImports,
		Tests: includeTest}
	// We turn this off because it's hard to set up test data that works with the
	// module system. Nevertheless, the tests are quite comprehensive.
	config.Env = append(os.Environ(), "GO111MODULE=off")
	if options.gopath != "" {
		config.Env = append(config.Env, "GOPATH="+options.gopath)
	}
	if options.fsOverlay != nil {
		config.Overlay = options.fsOverlay
	}
	p, err := packages.Load(config, path)
	if err != nil {
		return nil, fmt.Errorf("failed to import %s: %w", path, err)
	}

	var out []string
	var process func(pkg *packages.Package) error
	process = func(pkg *packages.Package) error {
		if len(pkg.Errors) > 0 {
			return fmt.Errorf("failed to import %s: %s", path, pkg.Errors)
		}
		// Skip packages we've seen before.
		if _, ok := seen[pkg.PkgPath]; ok {
			return nil
		}
		seen[pkg.PkgPath] = struct{}{}

		// Skip standard library packages.
		if isStdlibPackage(pkg.PkgPath) {
			return nil
		}
		sourceFileSets := [][]string{
			// Include all Go and Cgo source files.
			pkg.GoFiles, pkg.OtherFiles,
		}
		if len(pkg.GoFiles) > 0 {
			sourceFileSets = append(sourceFileSets, []string{
				// Include the package directory itself so that the target is considered
				// out-of-date if a new file is added to the directory.
				filepath.Dir(pkg.GoFiles[0]),
			})
		}

		// Collect files recursively from the package and its dependencies.
		for _, sourceFiles := range sourceFileSets {
			for _, sourceFile := range sourceFiles {
				if isFileAlwaysIgnored(sourceFile) || strings.HasPrefix(sourceFile, "zcgo_flags") {
					continue
				}
				f, err := filepath.Rel(cwd, sourceFile)
				if err != nil {
					return err
				}
				out = append(out, f)
			}
		}
		for _, imp := range pkg.Imports {
			if err := process(imp); err != nil {
				return err
			}
		}
		return nil
	}
	for _, pkg := range p {
		if err := process(pkg); err != nil {
			return nil, err
		}
	}

	return out, err
}

func isStdlibPackage(path string) bool {
	// Standard library packages never contain a dot; second- and third-party
	// packages nearly always do. Consider "github.com/cockroachdb/cockroach",
	// where the domain provides the dot, or "./pkg/sql", where the relative
	// import provides the dot.
	//
	// This logic is not foolproof, but it's the same logic used by goimports.
	return !strings.Contains(path, ".")
}

func isFileAlwaysIgnored(name string) bool {
	// build.Package.IgnoredGoFiles does not distinguish between Go files that are
	// always ignored and Go files that are temporarily ignored due to build tags.
	// Duplicate some logic from go/build [0] here so we can tell the difference.
	//
	// [0]: https://github.com/golang/go/blob/9ecf899b2/src/go/build/build.go#L1065-L1068
	return (strings.HasPrefix(name, "_") || strings.HasPrefix(name, ".")) && filepath.Ext(name) == ".go"
}

// See: https://www.cmcrossroads.com/article/gnu-make-escaping-walk-wild-side
var filenameEscaper = strings.NewReplacer(
	`[`, `\[`,
	`]`, `\]`,
	`*`, `\*`,
	`?`, `\?`,
	`~`, `\~`,
	`$`, `$$`,
	`#`, `\#`,
)

type testOptions struct {
	fsOverlay map[string][]byte
	gopath    string
}

func run(w io.Writer, path string, includeTest bool, binName string, options testOptions) error {
	files, err := collectFiles(path, includeTest, options)
	if err != nil {
		return err
	}

	for i := range files {
		files[i] = filenameEscaper.Replace(files[i])
	}

	absPath, err := filepath.Abs(path)
	if err != nil {
		return err
	}
	if binName == "" {
		binName = filepath.Base(absPath)
	}

	sort.Strings(files)

	fmt.Fprintln(w, "# Code generated by prereqs. DO NOT EDIT!")
	fmt.Fprintln(w)
	fmt.Fprintf(w, "bin/%s: %s\n", binName, strings.Join(files, " "))
	fmt.Fprintln(w)
	for _, f := range files {
		fmt.Fprintf(w, "%s:\n", f)
	}

	return nil
}

func main() {
	includeTest := flag.Bool("test", false, "include test dependencies")
	binName := flag.String("bin-name", "", "custom binary name (defaults to bin/<package name>)")
	flag.Usage = func() { fmt.Fprintf(os.Stderr, "usage: %s [-test] <package>\n", os.Args[0]) }
	flag.Parse()
	if flag.NArg() != 1 {
		flag.Usage()
		os.Exit(1)
	}

	if err := run(os.Stdout, flag.Arg(0), *includeTest, *binName, testOptions{}); err != nil {
		fmt.Fprintf(os.Stderr, "%s: %s\n", os.Args[0], err)
		os.Exit(1)
	}
}
